// Copyright (C) 1999-2000 Id Software, Inc.
//

#include "g_local.h"


/*
===============
G_DamageFeedback

Called just before a snapshot is sent to the given player.
Totals up all damage and generates both the player_state_t
damage values to that client for pain blends and kicks, and
global pain sound events for all clients.
===============
*/
void P_DamageFeedback( gentity_t *player ) {
	gclient_t	*client;
	float	count;
	vec3_t	angles;

	client = player->client;
	if ( client->ps.pm_type == PM_DEAD ) {
		return;
	}

	// total points of damage shot at the player this frame
	count = client->damage_blood + client->damage_armor;
	if ( count == 0 ) {
		return;		// didn't take any damage
	}

	if ( count > 255 ) {
		count = 255;
	}

	// send the information to the client

	// world damage (falling, slime, etc) uses a special code
	// to make the blend blob centered instead of positional
	if ( client->damage_fromWorld ) {
		client->ps.damagePitch = 255;
		client->ps.damageYaw = 255;

		client->damage_fromWorld = qfalse;
	} else {
		vectoangles( client->damage_from, angles );
		client->ps.damagePitch = angles[PITCH]/360.0 * 256;
		client->ps.damageYaw = angles[YAW]/360.0 * 256;
	}

	// play an apropriate pain sound
	if ( (level.time > player->pain_debounce_time) && !(player->flags & FL_GODMODE) ) {
		player->pain_debounce_time = level.time + 700;		
		G_AddEvent( player, EV_PAIN, player->health );
		client->ps.damageEvent++;
	}


	client->ps.damageCount = count;

	//
	// clear totals
	//
	client->damage_blood = 0;
	client->damage_armor = 0;
	client->damage_knockback = 0;
}



/*
=============
P_WorldEffects

Check for lava / slime contents and drowning
=============
*/
void P_WorldEffects( gentity_t *ent ) {
	qboolean	envirosuit;
	int			waterlevel;

	if ( ent->client->noclip ) {
		ent->client->airOutTime = level.time + 12000;	// don't need air
		return;
	}

	waterlevel = ent->waterlevel;

	envirosuit = ent->client->ps.powerups[PW_BATTLESUIT] > level.time;

	//
	// check for drowning
	//
	if ( waterlevel == 3 ) {
		// envirosuit give air
		if ( envirosuit ) {
			ent->client->airOutTime = level.time + 10000;
		}

		// if out of air, start drowning
		if ( ent->client->airOutTime < level.time) {
			// drown!
			ent->client->airOutTime += 5000;
			if ( ent->health > 0 ) {
				// take more damage the longer underwater
				ent->damage += 1;
				if (ent->damage > 15)
					ent->damage = 15;

				// play a gurp sound instead of a normal pain sound
				if (ent->health <= ent->damage) {
					G_Sound(ent, CHAN_VOICE, G_SoundIndex("*drown.wav"));
				} else if (rand()&1) {
					G_Sound(ent, CHAN_VOICE, G_SoundIndex("sound/player/gurp1.wav"));
				} else {
					G_Sound(ent, CHAN_VOICE, G_SoundIndex("sound/player/gurp2.wav"));
				}

				// don't play a normal pain sound
				ent->pain_debounce_time = level.time + 200;

				G_Damage (ent, NULL, NULL, NULL, NULL, 
					ent->damage, DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_WATER);
			}
		}
	} else {
		ent->client->airOutTime = level.time + 30000;
		ent->damage = 0;
	}

	//
	// check for sizzle damage (move to pmove?)
	//
	if (waterlevel && 
		(ent->watertype&(CONTENTS_LAVA|CONTENTS_SLIME)) ) {
		if (ent->health > 0
			&& ent->pain_debounce_time <= level.time	) {

			if ( envirosuit ) {
				G_AddEvent( ent, EV_POWERUP_BATTLESUIT, 0 );
			} else { // ## Hentai ## - these have been sped up and toned down, damage per second is roughly the same
				if (ent->watertype & CONTENTS_SLIME) {
					G_Damage (ent, NULL, NULL, NULL, NULL, 
						3 * waterlevel, 0, MOD_SLIME);
					ent->pain_debounce_time = level.time + 200;
				}
				if (ent->watertype & CONTENTS_LAVA) {
					G_Damage (ent, NULL, NULL, NULL, NULL, 
						5 * waterlevel, DAMAGE_NO_BLEEDING, MOD_LAVA);
					ent->pain_debounce_time = level.time + 100;
				}

				
				
			}
		}
	}
}



//==============================================================

/*
==============
ClientImpacts
==============
*/
void ClientImpacts( gentity_t *ent, pmove_t *pm ) {
	int		i, j;
	trace_t	trace;
	gentity_t	*other;

	memset( &trace, 0, sizeof( trace ) );
	for (i=0 ; i<pm->numtouch ; i++) {
		for (j=0 ; j<i ; j++) {
			if (pm->touchents[j] == pm->touchents[i] ) {
				break;
			}
		}
		if (j != i) {
			continue;	// duplicated
		}
		other = &g_entities[ pm->touchents[i] ];

		if ( ( ent->r.svFlags & SVF_BOT ) && ( ent->touch ) ) {
			ent->touch( ent, other, &trace );
		}

		if ( !other->touch ) {
			continue;
		}

		other->touch( other, ent, &trace );
	}

}

/*
============
G_TouchTriggers

Find all trigger entities that ent's current position touches.
Spectators will only interact with teleporters.
============
*/
void	G_TouchTriggers( gentity_t *ent ) {
	int			i, num;
	int			touch[MAX_GENTITIES];
	gentity_t	*hit;
	trace_t		trace;
	vec3_t		mins, maxs;
	static vec3_t	range = { 40, 40, 52 };

	if ( !ent->client ) {
		return;
	}

	// dead clients don't activate triggers!
	if ( ent->client->ps.stats[STAT_HEALTH] <= 0 ) {
		return;
	}

	VectorSubtract( ent->client->ps.origin, range, mins );
	VectorAdd( ent->client->ps.origin, range, maxs );

	num = trap_EntitiesInBox( mins, maxs, touch, MAX_GENTITIES );

	// can't use ent->absmin, because that has a one unit pad
	VectorAdd( ent->client->ps.origin, ent->r.mins, mins );
	VectorAdd( ent->client->ps.origin, ent->r.maxs, maxs );

	for ( i=0 ; i<num ; i++ ) {
		hit = &g_entities[touch[i]];

		if ( !hit->touch && !ent->touch ) {
			continue;
		}
		if ( !( hit->r.contents & CONTENTS_TRIGGER ) ) {
			continue;
		}

		// ignore most entities if a spectator
		if ( ent->client->sess.sessionTeam == TEAM_SPECTATOR ) {
			if ( hit->s.eType != ET_TELEPORT_TRIGGER &&
				// this is ugly but adding a new ET_? type will
				// most likely cause network incompatibilities
				hit->touch != Touch_DoorTrigger) {
				continue;
			}
		}

		// use seperate code for determining if an item is picked up
		// so you don't have to actually contact its bounding box
		if ( hit->s.eType == ET_ITEM ) {
			if ( !BG_PlayerTouchesItem( &ent->client->ps, &hit->s, level.time ) ) {
				continue;
			}
		} else {
			if ( !trap_EntityContact( mins, maxs, hit ) ) {
				continue;
			}
		}

		memset( &trace, 0, sizeof(trace) );

		if ( hit->touch ) {
			hit->touch (hit, ent, &trace);
		}

		if ( ( ent->r.svFlags & SVF_BOT ) && ( ent->touch ) ) {
			ent->touch( ent, hit, &trace );
		}
	}
}

/*
=================
SpectatorThink
=================
*/
void SpectatorThink( gentity_t *ent, usercmd_t *ucmd ) {
	pmove_t	pm;
	gclient_t	*client;

	client = ent->client;

	if ( client->sess.spectatorState != SPECTATOR_FOLLOW ) {
		client->ps.pm_type = PM_SPECTATOR;
		client->ps.speed = 400;	// faster than normal

		// set up for pmove
		memset (&pm, 0, sizeof(pm));
		pm.ps = &client->ps;
		pm.cmd = *ucmd;
		pm.tracemask = MASK_PLAYERSOLID & ~CONTENTS_BODY;	// spectators can fly through bodies
		pm.trace = trap_Trace;
		pm.pointcontents = trap_PointContents;

		// perform a pmove
		Pmove (&pm);

		// save results of pmove
		VectorCopy( client->ps.origin, ent->s.origin );

		G_TouchTriggers( ent );
		trap_UnlinkEntity( ent );
	}

	client->oldbuttons = client->buttons;
	client->buttons = ucmd->buttons;

	// attack button cycles through spectators
	if ( ( client->buttons & BUTTON_ATTACK ) && ! ( client->oldbuttons & BUTTON_ATTACK ) ) {
		Cmd_FollowCycle_f( ent, 1 );
	}
}



/*
=================
ClientInactivityTimer

Returns qfalse if the client is dropped
=================
*/
qboolean ClientInactivityTimer( gclient_t *client ) {
	if ( ! g_inactivity.integer ) {
		// give everyone some time, so if the operator sets g_inactivity during
		// gameplay, everyone isn't kicked
		client->inactivityTime = level.time + 60 * 1000;
		client->inactivityWarning = qfalse;
	} else if ( client->pers.cmd.forwardmove || 
		client->pers.cmd.rightmove || 
		client->pers.cmd.upmove ||
		(client->pers.cmd.buttons & BUTTON_ATTACK) ) {
		client->inactivityTime = level.time + g_inactivity.integer * 1000;
		client->inactivityWarning = qfalse;
	} else if ( !client->pers.localClient ) {
		if ( level.time > client->inactivityTime ) {
			trap_DropClient( client - level.clients, "Dropped due to inactivity" );
			return qfalse;
		}
		if ( level.time > client->inactivityTime - 10000 && !client->inactivityWarning ) {
			client->inactivityWarning = qtrue;
			trap_SendServerCommand( client - level.clients, "cp \"Ten seconds until inactivity drop!\n\"" );
		}
	}
	return qtrue;
}

/*
==================
ClientTimerActions

Actions that happen once a second
==================
*/
void ClientTimerActions( gentity_t *ent, int msec ) {
	gclient_t *client;

	client = ent->client;
	client->timeResidual += msec;

	while ( client->timeResidual >= 1000 ) {
		client->timeResidual -= 1000;

		// regenerate
		if ( client->ps.powerups[PW_BERZERK] ) {
			G_Damage(ent, ent, ent, vec3_origin, ent->s.origin, 1, DAMAGE_NO_BLEEDING | DAMAGE_NO_PROTECTION, MOD_BERZERK);
			client->ps.stats[STAT_BLEEDING] = 0;
			client->ps.stats[STAT_LEG_DAMAGE] = 0;
			client->ps.stats[STAT_HEAD_DAMAGE]++;
		}
		if ( client->ps.powerups[PW_REGEN] ) {
			if ( client->ps.stats[STAT_BLEEDING] > 0) 
			{
				client->ps.stats[STAT_BLEEDING] -= 25;
				if(client->ps.stats[STAT_BLEEDING] < 0)
				{
					client->ps.stats[STAT_HEALTH] -= client->ps.stats[STAT_BLEEDING] / 5;
					client->ps.stats[STAT_BLEEDING] = 0;
				}
			} 
			else
				client->ps.stats[STAT_HEALTH] += 10;

			if ( client->ps.stats[STAT_LEG_DAMAGE] > 0) 
			{
				client->ps.stats[STAT_LEG_DAMAGE] -= 20;
				if(client->ps.stats[STAT_LEG_DAMAGE] < 0)
				{
					client->ps.stats[STAT_HEALTH] -= client->ps.stats[STAT_LEG_DAMAGE] / 4;
					client->ps.stats[STAT_LEG_DAMAGE] = 0;
				}
			} 
			else 
				client->ps.stats[STAT_HEALTH] += 5;

			if ( client->ps.stats[STAT_HEAD_DAMAGE] > 0) 
			{
				client->ps.stats[STAT_LEG_DAMAGE] -= 10;
				if(client->ps.stats[STAT_LEG_DAMAGE] < 0)
				{
					client->ps.stats[STAT_HEALTH] -= client->ps.stats[STAT_LEG_DAMAGE] / 2;
					client->ps.stats[STAT_LEG_DAMAGE] = 0;
				}
			} 
			else 
				client->ps.stats[STAT_HEALTH] += 5;
			
			if ( client->ps.stats[STAT_HEALTH] > MAX_HEALTH ) 
				client->ps.stats[STAT_HEALTH] = MAX_HEALTH;
			
			G_AddEvent( ent, EV_POWERUP_REGEN, 0 );
		}
	}
}

/*
====================
ClientIntermissionThink
====================
*/
void ClientIntermissionThink( gclient_t *client ) {
	client->ps.eFlags &= ~EF_TALK;
	client->ps.eFlags &= ~EF_FIRING;

	// the level will exit when everyone wants to or after timeouts

	// swap and latch button actions
	client->oldbuttons = client->buttons;
	client->buttons = client->pers.cmd.buttons;
	if ( client->buttons & ( BUTTON_ATTACK | BUTTON_USE_HOLDABLE ) & ( client->oldbuttons ^ client->buttons ) ) {
		client->readyToExit ^= 1;
	}
}


/*
================
ClientEvents

Events will be passed on to the clients for presentation,
but any server game effects are handled here
================
*/
void ClientEvents( gentity_t *ent, int oldEventSequence ) {
	int		i;
	int		event;
	gclient_t *client;
	int		damage;
	vec3_t	dir;
//	vec3_t	origin, angles;
//	qboolean	fired;
//	gitem_t *item;
//	gentity_t *drop;

	client = ent->client;

	if ( oldEventSequence < client->ps.eventSequence - MAX_PS_EVENTS ) {
		oldEventSequence = client->ps.eventSequence - MAX_PS_EVENTS;
	}
	for ( i = oldEventSequence ; i < client->ps.eventSequence ; i++ ) {
		event = client->ps.events[ i & (MAX_PS_EVENTS-1) ];

		switch ( event ) {
		case EV_FALL_SHORT:
			if(ent->client->ps.stats[STAT_BLEEDING] <= 0 && ent->client->ps.stats[STAT_LEG_DAMAGE] <= 0)
				break;
		case EV_FALL_MEDIUM:
		case EV_FALL_FAR:
			if ( ent->s.eType != ET_PLAYER ) {
				break;		// not in the player model
			}
			damage = sqrt(client->ps.eventParms[ i & (MAX_PS_EVENTS-1) ]) - 5;			
			if(damage > 0)
			{
				VectorSet (dir, 0, 0, 1);
				ent->pain_debounce_time = level.time + 200;	// no normal pain sound
				if(ent->client->ps.stats[STAT_BLEEDING])						
					G_Damage (ent, ent, ent->enemy, NULL, NULL, (int) (damage + ((ent->client->ps.stats[STAT_BLEEDING] / 100.0 + ent->client->ps.stats[STAT_LEG_DAMAGE] * 1.0 / MAX_HEALTH) * damage)), DAMAGE_NO_ARMOR, MOD_BLEED);
				else
					G_Damage (ent, NULL, NULL, NULL, NULL, damage, DAMAGE_NO_BLEEDING, MOD_FALLING);
				if(ent->client->ps.stats[STAT_LEG_DAMAGE])
					G_Damage (ent, ent, NULL, NULL, NULL, (int) ((ent->client->ps.stats[STAT_BLEEDING] / 100.0 + ent->client->ps.stats[STAT_LEG_DAMAGE] * 1.0 / MAX_HEALTH) * damage), DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_FALLING);
			}
			break;
		case EV_FIRE_WEAPON:
			FireWeapon( ent );
			break;
		case EV_DROP_WEAPON:
			TossClientWeapon(ent, &bg_itemlist[client->ps.eventParms[ i & (MAX_PS_EVENTS-1) ]]);
			break;
		case EV_USE_BERZERK:
			if(ent->client->ps.powerups[PW_BERZERK] < level.time)
				ent->client->ps.powerups[PW_BERZERK] = level.time;
			ent->client->ps.powerups[PW_BERZERK] += 30000;
			break;
		default:
			break;
		}
	}

}

void BotTestSolid(vec3_t origin);

/*
==============
ClientThink

This will be called once for each client frame, which will
usually be a couple times for each server frame on fast clients.

If "g_syncronousClients 1" is set, this will be called exactly
once for each server frame, which makes for smooth demo recording.
==============
*/

void G_ExplodeMissile( gentity_t *ent );

float G_CalcMass(gitem_t *item, int quantity)
{
	int rval = item->mass;
	if(item->giType == IT_WEAPON)
	{
		rval += BG_FindAmmoForWeapon(item->giTag)->mass * quantity;
	}
	else if(item->giType == IT_AMMO)
	{
		rval *= quantity;
	}
	return rval;
}

void ClientThink_real( gentity_t *ent ) {
	gclient_t	*client;
	pmove_t		pm;
	vec3_t		oldOrigin;
	int			oldEventSequence;
	int			msec, i;
	float		mass, basemass, str;
	int oldLegs, oldTorso;
	usercmd_t	*ucmd;
	float jumpheight = 64.0;
	client = ent->client;
	if(!g_allowArmor.integer)
		client->ps.stats[STAT_ARMOR] = 0;
	
	// mark the time, so the connection sprite can be removed
	ucmd = &ent->client->pers.cmd;

	// sanity check the command time to prevent speedup cheating
	if ( ucmd->serverTime > level.time + 200 ) {
		ucmd->serverTime = level.time + 200;
	}
	if ( ucmd->serverTime < level.time - 1000 ) {
		ucmd->serverTime = level.time - 1000;
	} 

	msec = ucmd->serverTime - client->ps.commandTime;
	// following others may result in bad times, but we still want
	// to check for follow toggles
	if ( msec < 1 && client->sess.spectatorState != SPECTATOR_FOLLOW ) {
		return;
	}
	if ( msec > 200 ) {
		msec = 200;
	}

	//
	// check for exiting intermission
	//
	if ( level.intermissiontime ) {
		ClientIntermissionThink( client );
		return;
	}

	// spectators don't do much
	if ( client->sess.sessionTeam == TEAM_SPECTATOR ) {
		if ( client->sess.spectatorState == SPECTATOR_SCOREBOARD ) {
			return;
		}
		SpectatorThink( ent, ucmd );
		return;
	}

	// check for inactivity timer, but never drop the local client of a non-dedicated server
	if ( !ClientInactivityTimer( client ) ) {
		return;
	}

	// clear the rewards if time
	
	if(ent->s.eFlags & EF_DEAD)
	{
		client->ps.eFlags &= ~(EF_AWARD_IMPRESSIVE | EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
		ent->s.eFlags &= ~(EF_AWARD_IMPRESSIVE | EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
	}
	else if ( level.time > client->rewardTime ) {
		client->ps.eFlags &= ~(EF_AWARD_IMPRESSIVE | EF_AWARD_EXCELLENT | EF_AWARD_GAUNTLET );
	}

	if ( client->noclip ) {
		client->ps.pm_type = PM_NOCLIP;
	} else if ( client->ps.stats[STAT_HEALTH] <= 0 ) {
		client->ps.pm_type = PM_DEAD;
	} else {
		client->ps.pm_type = PM_NORMAL;
	}

	client->ps.gravity = g_gravity.value;
	if(ucmd->upmove > 0 && client->pers.playerclass >= 0 && client->pers.playerclass < p_classes)
		client->ps.gravity *= p_class[client->pers.playerclass].jumpgravity;

	// set speed
	client->ps.stats[STAT_JUMPSCALE] = 270; // to make sure a default gets set
	client->ps.stats[STAT_JUMP_PAUSE] = 250; // to make sure a default gets set
	if(client->ps.stats[STAT_LEG_DAMAGE] > 0 && client->ps.stats[STAT_HEALTH] > 0 && !client->ps.powerups[PW_FLIGHT])
	{		
		float limp = ((float) (client->ps.stats[STAT_HEALTH])) / ((float) (client->ps.stats[STAT_HEALTH] + client->ps.stats[STAT_LEG_DAMAGE]));
		if(limp < 0.25)
			limp = 0.25;
		client->ps.speed = g_speed.value * limp;
		
	}
	else
		client->ps.speed = g_speed.value;


	if(client->pers.playerclass >= 0 && client->pers.playerclass < p_classes)
	{
		int p = client->pers.playerclass;
		client->ps.stats[STAT_JUMP_PAUSE] = p_class[p].jumppause * 1000;
		if(client->ps.groundEntityNum == ENTITYNUM_NONE)
		{
			client->ps.speed *= p_class[p].jumpspeed;
		}
		else if(ucmd->upmove < 0)
		{
			client->ps.speed *= p_class[p].crouchspeed;
		}
		else
		{
			client->ps.speed *= p_class[p].speed;
		}
		if(client->ps.stats[STAT_LEG_DAMAGE] > 0 && client->ps.stats[STAT_HEALTH] > 0)
		{			
			client->ps.stats[STAT_JUMPSCALE] = 270 * sqrt(p_class[client->pers.playerclass].jumpfactor) * client->ps.stats[STAT_HEALTH] / (client->ps.stats[STAT_HEALTH] + client->ps.stats[STAT_LEG_DAMAGE] + 0.01);
		}
		else
			client->ps.stats[STAT_JUMPSCALE] = 270 * sqrt(p_class[client->pers.playerclass].jumpfactor);
	}
	else if(client->pers.playerclass < 0 || client->pers.playerclass > p_classes) // force classes, and this isn't a valid class.
	{	// turn into a spectator
		StopFollowing(ent); 
		G_SayTo(ent, ent, 2, 0, "Server", "Please choose a valid playerclass from the player model menu.");
		//client->ps.pers[PERS_PLAYERCLASS] = -1;
	}

	if(client->pers.playerclass >= 0 && client->pers.playerclass < MAX_PCLASSES)
		basemass = p_class[client->pers.playerclass].mass;
	else
		basemass = 200;
	
	mass = basemass + G_CalcMass(&bg_itemlist[client->ps.weapon], client->ps.ammo[AMMO_CURRENT]);
	for(i = AMMO_ITEM1; i < AMMO_ITEMS; i += 2)
	{
		mass += G_CalcMass(&bg_itemlist[client->ps.ammo[i]], client->ps.ammo[i + 1]);
	}

	client->ps.stats[STAT_MASS] = mass;
	str = client->ps.stats[STAT_STRENGTH];
	if(str != 0)
	{		
		str += (mass - basemass);
		client->ps.speed *= client->ps.stats[STAT_STRENGTH] / str;			
		client->ps.stats[STAT_JUMPSCALE] *= sqrt(client->ps.stats[STAT_STRENGTH] / str);
	}

	if ( client->ps.powerups[PW_HASTE] ) {
		client->ps.speed *= g_HasteFactor.value;
	}
	if ( client->ps.powerups[PW_BERZERK] ) {
		client->ps.speed *= 2;
	}

	// set up for pmove
	oldEventSequence = client->ps.eventSequence;

	memset (&pm, 0, sizeof(pm));

	// check for the hit-scan gauntlet, don't let the action
	// go through as an attack unless it actually hits something
	if ( (client->ps.ammo[AMMO_CURRENT] == 0 || bg_itemlist[client->ps.weapon].giTag <= WP_GAUNTLET || bg_itemlist[client->ps.weapon].giTag == WP_REDFLAG ||bg_itemlist[client->ps.weapon].giTag == WP_BLUEFLAG || bg_itemlist[client->ps.weapon].giType != IT_WEAPON) &&
		( ucmd->buttons & BUTTON_ATTACK ) && client->ps.weaponTime <= 0 ) {
		pm.gauntletHit = CheckMeleeAttack( ent );
	}
	else if(bg_itemlist[client->ps.weapon].giType == IT_WEAPON && bg_itemlist[client->ps.weapon].giTag == WP_SWORD)
	{
		pm.gauntletHit = CheckMeleeAttack( ent );
	}


	pm.ps = &client->ps;
	pm.cmd = *ucmd;
	if ( pm.ps->pm_type == PM_DEAD ) {
		pm.tracemask = MASK_PLAYERSOLID & ~CONTENTS_BODY;
	}
	else {
		pm.tracemask = MASK_PLAYERSOLID;
	}
	pm.trace = trap_Trace;
	pm.pointcontents = trap_PointContents;
	pm.debugLevel = g_debugMove.integer;
	pm.noFootsteps = ( g_dmflags.integer & DF_NO_FOOTSTEPS ) > 0;

	VectorCopy( client->ps.origin, oldOrigin );

	oldLegs = client->ps.legsAnim;
	oldTorso = client->ps.torsoAnim;
	// perform a pmove
	Pmove (&pm);
	
	// save results of pmove
	
	// check animation status
	if(oldLegs != client->ps.legsAnim)
	{
		client->legsAnimTime = level.time;
	}
	if(oldTorso != client->ps.torsoAnim)
	{
		client->torsoAnimTime = level.time;
	}
	
	if(client->anims)
	{
		int anim, frame, loopOffset;
		anim = client->ps.legsAnim & ~ANIM_TOGGLEBIT;
		loopOffset = client->anims[anim].firstFrame + (client->anims[anim].numFrames - 1) - client->anims[anim].loopFrames;
		
		frame = client->anims[anim].firstFrame + (level.time - client->legsAnimTime - client->anims[anim].initialLerp) / client->anims[anim].frameLerp;
		if(frame > loopOffset)
		{
			if(client->anims[anim].loopFrames)
			{
				frame -= loopOffset;
				frame %= client->anims[anim].loopFrames;
				frame += loopOffset;
			}
			else
				frame = loopOffset;
		}
		client->legsFrame = frame;

		anim = client->ps.torsoAnim & ~ANIM_TOGGLEBIT;
		loopOffset = client->anims[anim].firstFrame + (client->anims[anim].numFrames - 1) - client->anims[anim].loopFrames;
		frame = client->anims[anim].firstFrame + (level.time - client->torsoAnimTime - client->anims[anim].initialLerp) / client->anims[anim].frameLerp;
		if(frame > loopOffset)
		{
			if(client->anims[anim].loopFrames)
			{
				frame -= loopOffset;
				frame %= client->anims[anim].loopFrames;
				frame += loopOffset;
			}
			else
				frame = loopOffset;
		}
		client->torsoFrame = frame;
		//G_Printf("server thinks animation frames are L:%i T:%i\n", client->legsFrame, client->torsoFrame);
	}
	if ( ent->client->ps.eventSequence != oldEventSequence ) {
		ent->eventTime = level.time;
	}
	BG_PlayerStateToEntityState( &ent->client->ps, &ent->s, level.time, qtrue );
	if ( !( ent->client->ps.eFlags & EF_FIRING ) ) {
		client->fireHeld = qfalse;		// for grapple
	}

#if 1
	// use the precise origin for linking
	VectorCopy( ent->client->ps.origin, ent->r.currentOrigin );
#else
	// use the snapped origin for linking so it matches client predicted versions
	VectorCopy( ent->s.pos.trBase, ent->r.currentOrigin );
#endif

	VectorCopy (pm.mins, ent->r.mins);
	VectorCopy (pm.maxs, ent->r.maxs);
	
	ent->waterlevel = pm.waterlevel;
	ent->watertype = pm.watertype;

	// execute client events
	ClientEvents( ent, oldEventSequence );

	// link entity now, after any personal teleporters have been used
	trap_LinkEntity (ent);
	if ( !ent->client->noclip ) {
		G_TouchTriggers( ent );
	}

	//test for solid areas in the AAS file
	BotTestSolid(ent->r.currentOrigin);

	// touch other objects
	ClientImpacts( ent, &pm );

	// swap and latch button actions
	client->oldbuttons = client->buttons;
	client->buttons = ucmd->buttons;
	client->latched_buttons |= client->buttons & ~client->oldbuttons;

	// check for respawning
	if ( client->ps.stats[STAT_HEALTH] <= 0 ) {
		// wait for the attack button to be pressed
		if ( level.time > client->respawnTime ) {
			if(ent->s.eType != ET_INVISIBLE)
			{
				CopyToBodyQue (ent);
			}
			// forcerespawn is to prevent users from waiting out powerups
			if ( g_forcerespawn.integer > 0 && 
				( level.time - client->respawnTime ) > g_forcerespawn.integer * 1000 ) {
				respawn( ent );
				return;
			}
		
			// pressing attack or use is the normal respawn method
			if ( ucmd->buttons & ( BUTTON_ATTACK | BUTTON_USE_HOLDABLE ) ) {
				respawn( ent );
			}
		}
		return;
	}
	if(ent->onfire > level.time)
	{		
		if(ent->waterlevel > 1)
			ent->onfire = 0;
		if(ent->burntime < level.time)
		{
			G_Damage(ent, ent, ent->enemy, NULL, NULL, 1, DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_FLAMETHROWER);
			ent->burntime = level.time + 100;
			G_AddEvent(ent, EV_ONFIRE, 1); // make sure
			G_AddPredictableEvent(ent, EV_ONFIRE, 1);
		}
	}
	if(client->ps.stats[STAT_BLEEDING])
	{		
			
		if((level.time - client->bleedtime) >= 250)
		{
			client->bleed_damage += ((client->ps.stats[STAT_FATIGUE] + 100) / 100.0) * client->ps.stats[STAT_BLEEDING] * (level.time - client->bleedtime) / 100000.0;
			client->bleedtime = level.time;
		}
		if(client->bleed_damage >= 1)
		{
			//ent->pain_debounce_time = level.time + rand() % 5;
			if(client->bleed_damage > client->ps.stats[STAT_HEALTH] + 5)
				client->bleed_damage = client->ps.stats[STAT_HEALTH] + 5;			
			if(client->ps.stats[STAT_HEALTH] > 0)
			{
				if(ent->enemy)
					G_Damage(ent, NULL, ent->enemy, NULL, NULL, (int) client->bleed_damage, DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_BLEED);
				else
					G_Damage(ent, NULL, &g_entities[ENTITYNUM_WORLD], NULL, NULL, (int) client->bleed_damage, DAMAGE_NO_ARMOR | DAMAGE_NO_BLEEDING, MOD_BLEED);
				/*if(client->bleed_damage >= 2)
				{
					G_Printf("sent EV_BLOOD(%d)\n", (int) floor(client->bleed_damage));
					G_AddEvent(ent, EV_BLOOD, (int) floor(client->bleed_damage));
				}*/
			}
			else
			{
				//G_Printf("sent EV_BLOOD(16)\n");
				G_AddEvent(ent, EV_BLOOD, 16);
			}
			client->bleed_damage -= floor(client->bleed_damage);
			client->ps.stats[STAT_BLEEDING]--; // slowly heal
			if(client->ps.stats[STAT_HEALTH] > MAX_HEALTH || client->ps.stats[STAT_BLEEDING] < 0)
				client->ps.stats[STAT_BLEEDING] = 0;
		}
		
	}	
	//CrossProduct(client->muzzle, client->ps.velocity, oldOrigin);
	//G_Printf("v: %f %f %f m:%f %f %f c: %f %f\n", client->ps.velocity[0], client->ps.velocity[1], client->ps.velocity[2], client->muzzle[0], client->muzzle[1], client->muzzle[2], oldOrigin[PITCH], oldOrigin[YAW]);
	if(client->pers.playerclass >= 0 && client->pers.playerclass < p_classes)
	{
		//gentity_t *tent;
		//vec3_t org;
		client->waistheight = BG_PlayerAngles(client->pers.playerclass, client->legsFrame, client->torsoFrame, client->ps.viewangles, client->ps.velocity, client->muzzle, client->headpos);//G_PlayerAngles(client, client->muzzle, client->headpos);
		//VectorAdd(client->headpos, client->ps.origin, org);
		//tent = G_TempEntity(org, EV_MISSILE_HIT_ROBOT);
		//tent->s.weapon = WP_MACHINEGUN;
		//tent->s.otherEntityNum = ent->s.number;
		if((client->headpos[2] + p_class[client->pers.playerclass].headsize) > ent->r.maxs[2])
		{ // head's sticking up over the top of the model - gotta fix that
			ent->r.maxs[2] = client->headpos[2] + p_class[client->pers.playerclass].headsize;
		}
	}
	// perform once-a-second actions
	ClientTimerActions( ent, msec );
}

/*
==================
ClientThink

A new command has arrived from the client
==================
*/
void ClientThink( int clientNum ) {
	gentity_t *ent;

	ent = g_entities + clientNum;
	trap_GetUsercmd( clientNum, &ent->client->pers.cmd );

	// mark the time we got info, so we can display the
	// phone jack if they don't get any for a while
	ent->client->lastCmdTime = level.time;

	if ( !g_syncronousClients.integer ) {
		ClientThink_real( ent );
	}
}


void G_RunClient( gentity_t *ent ) {
	if ( !g_syncronousClients.integer ) {
		return;
	}
	ent->client->pers.cmd.serverTime = level.time;
	ClientThink_real( ent );
}


/*
==================
SpectatorClientEndFrame

==================
*/
void SpectatorClientEndFrame( gentity_t *ent ) {
	gclient_t	*cl;

	// if we are doing a chase cam or a remote view, grab the latest info
	if ( ent->client->sess.spectatorState == SPECTATOR_FOLLOW ) {
		int		clientNum;

		clientNum = ent->client->sess.spectatorClient;

		// team follow1 and team follow2 go to whatever clients are playing
		if ( clientNum == -1 ) {
			clientNum = level.follow1;
		} else if ( clientNum == -2 ) {
			clientNum = level.follow2;
		}
		if ( clientNum >= 0 ) {
			cl = &level.clients[ clientNum ];
			if ( cl->pers.connected == CON_CONNECTED && cl->sess.sessionTeam != TEAM_SPECTATOR ) {
				ent->client->ps = cl->ps;
				ent->client->ps.pm_flags |= PMF_FOLLOW;
				return;
			} else {
				// drop them to free spectators unless they are dedicated camera followers
				if ( ent->client->sess.spectatorClient >= 0 ) {
					ent->client->sess.spectatorState = SPECTATOR_FREE;
					ClientBegin( ent->client - level.clients );
				}
			}
		}
	}

	if ( ent->client->sess.spectatorState == SPECTATOR_SCOREBOARD ) {
		ent->client->ps.pm_flags |= PMF_SCOREBOARD;
	} else {
		ent->client->ps.pm_flags &= ~PMF_SCOREBOARD;
	}
}

/*
==============
ClientEndFrame

Called at the end of each server frame for each connected client
A fast client will have multiple ClientThink for each ClientEdFrame,
while a slow client may have multiple ClientEndFrame between ClientThink.
==============
*/
void ClientEndFrame( gentity_t *ent ) {
	int			i;
	clientPersistant_t	*pers;

	if ( ent->client->sess.sessionTeam == TEAM_SPECTATOR ) {
		SpectatorClientEndFrame( ent );
		return;
	}

	pers = &ent->client->pers;

	// turn off any expired powerups
	for ( i = 0 ; i < MAX_POWERUPS ; i++ ) {
		if ( ent->client->ps.powerups[ i ] < level.time ) {
			if(i == PW_BALL && ent->client->ps.powerups[ i ] > 0)
			{ // we just ran out of the ball
				G_Damage(ent, ent, ent, vec3_origin, vec3_origin, 40000, DAMAGE_NO_PROTECTION, MOD_PENALTY);
			}
			ent->client->ps.powerups[ i ] = 0;
		}
	}

	// save network bandwidth
#if 0
	if ( !g_syncronousClients->integer && ent->client->ps.pm_type == PM_NORMAL ) {
		// FIXME: this must change eventually for non-sync demo recording
		VectorClear( ent->client->ps.viewangles );
	}
#endif

	//
	// If the end of unit layout is displayed, don't give
	// the player any normal movement attributes
	//
	if ( level.intermissiontime ) {
		return;
	}

	// burn from lava, etc
	P_WorldEffects (ent);

	// apply all the damage taken this frame
	P_DamageFeedback (ent);

	// add the EF_CONNECTION flag if we haven't gotten commands recently
	if ( level.time - ent->client->lastCmdTime > 1000 ) {
		ent->s.eFlags |= EF_CONNECTION;
	} else {
		ent->s.eFlags &= ~EF_CONNECTION;
	}

	ent->health = ent->client->ps.stats[STAT_HEALTH];	// FIXME: get rid of ent->health...

	// set the latest infor
	BG_PlayerStateToEntityState( &ent->client->ps, &ent->s, level.time, qtrue );
}


