// Copyright (C) 1999-2000 Id Software, Inc.
//
// bg_misc.c -- both games misc functions, all completely stateless

#include "q_shared.h"
#include "bg_public.h"

/*QUAKED item_***** ( 0 0 0 ) (-16 -16 -16) (16 16 16) suspended
DO NOT USE THIS CLASS, IT JUST HOLDS GENERAL INFORMATION.
The suspended flag will allow items to hang in the air, otherwise they are dropped to the next surface.

If an item is the target of another entity, it will not spawn in until fired.

An item fires all of its targets when it is picked up.  If the toucher can't carry it, the targets won't be fired.

"notfree" if set to 1, don't spawn in free for all games
"notteam" if set to 1, don't spawn in team games
"notsingle" if set to 1, don't spawn in single player games
"wait"	override the default wait before respawning.  -1 = never respawn automatically, which can be used with targeted spawning.
"random" random number of plus or minus seconds varied from the respawn time
"count" override quantity or duration on most items.
*/


gitem_t	bg_itemlist[] = 
{
	{
		NULL,
		NULL,
		{ NULL,
		NULL,
		0, 0} ,
/* icon */		NULL,
/* pickup */	NULL,
		0,
		0,
		0,
		0,
		0,
/* precache */ "",
/* sounds */ ""
	},	// leave index 0 alone

	//
	// ARMOR
	//

/*QUAKED item_armor_combat (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_armor_combat", 
		"sound/misc/ar2_pkup.wav",
        { "models/powerups/armor/armor_yel.md3",
		0, 0, 0},
/* icon */		"icons/iconr_yellow",
/* pickup */	"Armor",
		50,
		IT_ARMOR,
		0, 0,
		30,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED item_armor_body (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_armor_body", 
		"sound/misc/ar3_pkup.wav",
        { "models/powerups/armor/armor_red.md3",
		0, 0, 0},
/* icon */		"icons/iconr_red",
/* pickup */	"Heavy Armor",
		100,
		IT_ARMOR,
		0, 0,
		50,
/* precache */ "",
/* sounds */ ""
	},

	//
	// health
	//
/*QUAKED item_health_small (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_health_small",
		"sound/items/s_health.wav",
        {"models/health/medikit.md3",
		0, //"models/powerups/health/small_sphere.md3", 
		0, 0 },
/* icon */		"icons/iconh_red",
/* pickup */	"Medikit",
		2,
		IT_HEALTH,
		0, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED item_health (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_health",
		"sound/items/n_health.wav",
        {"models/health/medikit.md3",
		0, 
		0, 0 },
/* icon */		"icons/iconh_red",
/* pickup */	"Medikit",
		5,
		IT_HEALTH,
		0, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED item_health_large (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_health_large",
		"sound/items/l_health.wav",
        {"models/health/medikit.md3",
		0, //"models/powerups/health/large_sphere.md3", 
		0, 0 },
/* icon */		"icons/iconh_red",
/* pickup */	"Medikit",
		10,
		IT_HEALTH,
		0, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED item_health_mega (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_health_mega",
		"sound/items/m_health.wav",
        {"models/health/medikit.md3",
		0, 
		0, 0 },
/* icon */		"icons/iconh_red",
/* pickup */	"Medikit",
		25,
		IT_HEALTH,
		0, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_gauntlet (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_none", 
		"",
        { 0,
		0, 0, 0},
/* icon */		"icons/iconw_gauntlet",
/* pickup */	"Fist",
		-1,
		IT_WEAPON,
		WP_NONE, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_gauntlet (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_sword", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/sword/sword.md3",
		0, 0, 0},
/* icon */		"icons/iconw_gauntlet",
/* pickup */	"Sword",
		-1,
		IT_WEAPON,
		WP_SWORD, 0,
		12,
/* precache */ "",
/* sounds */ ""
	},
	

	//
	// WEAPONS 
	//

/*QUAKED weapon_gauntlet (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_gauntlet", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/gauntlet/gauntlet.md3",
		0, 0, 0},
/* icon */		"icons/iconw_gauntlet",
/* pickup */	"Gauntlet",
		200,
		IT_WEAPON,
		WP_GAUNTLET, AMMO_CELLS,
		15,
/* precache */ "",
/* sounds */ ""
	},
	
/*QUAKED weapon_shotgun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_singleshotgun", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/single_shotgun/single_shotgun.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_shotgun",
/* pickup */	"Shotgun",
		6,
		IT_WEAPON,
		WP_SINGLE_SHOTGUN, AMMO_SHELLS,
		18,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_shotgun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_shotgun", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/shotgun/shotgun.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_shotgun",
/* pickup */	"Double Barreled Shotgun",
		6,
		IT_WEAPON,
		WP_SHOTGUN, AMMO_SHELLS,
		20,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_machinegun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_machinegun", 
		"sound/misc/w_pkup.wav",
		{ "models/weapons2/smg/smg.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_machinegun",
/* pickup */	"Machinegun",
		40,
		IT_WEAPON,
		WP_MACHINEGUN, AMMO_BULLETS,
		12,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_chaingun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_chaingun", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/chaingun/chaingun.md3",  
		0, 0, 0},
/* icon */		"icons/iconw_machinegun",
/* pickup */	"Chaingun",
		100,
		IT_WEAPON,
		WP_CHAINGUN, AMMO_BULLETS,
		45,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_grenadelauncher (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_grenadelauncher",
		"sound/misc/w_pkup.wav",
        { "models/weapons2/grenadel/grenadel.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_grenade",
/* pickup */	"Grenade Launcher",
		5,
		IT_WEAPON,
		WP_GRENADE_LAUNCHER, AMMO_GRENADES,
		10,
/* precache */ "",
/* sounds */ "sound/weapons/grenade/hgrenb1a.wav sound/weapons/grenade/hgrenb2a.wav"
	},

/*QUAKED weapon_rocketlauncher (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_rocketlauncher",
		"sound/misc/w_pkup.wav",
        { "models/weapons2/rocketl/rocketl.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_rocket",
/* pickup */	"Rocket Launcher",
		10,
		IT_WEAPON,
		WP_ROCKET_LAUNCHER, AMMO_ROCKETS,
		60,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_plasmagun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_plasmagun", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/plasma/plasma.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_plasma",
/* pickup */	"Plasma Gun",
		50,
		IT_WEAPON,
		WP_PLASMAGUN, AMMO_CELLS,
		40,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_railgun (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_railgun", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/railgun/railgun.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_railgun",
/* pickup */	"Railgun",
		10,
		IT_WEAPON,
		WP_RAILGUN, AMMO_SLUGS,
		30,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED weapon_flamethrower (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_flamethrower", //weapon_flamethrower
		"sound/misc/w_pkup.wav",
		{ "models/weapons2/grapple/grapple.md3", // we're using the grappling hook, sort of
		0, 0, 0},
/* icon */		"icons/iconw_flamethrower",            
/* pickup */	"Flamethrower",

		100,

		IT_WEAPON, 
		WP_FLAMETHROWER, AMMO_NAPALM,
		20,
/* precache */ "",
/* sounds */ ""
	},
/*QUAKED weapon_lightning (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"weapon_lightning", 
		"sound/misc/w_pkup.wav",
        { "models/weapons2/lightning/lightning.md3", 
		0, 0, 0},
/* icon */		"icons/iconw_lightning",
/* pickup */	"Lightning Gun",
		100,
		IT_WEAPON,
		WP_LIGHTNING, AMMO_CELLS,
		50,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED team_CTF_redflag (1 0 0) (-16 -16 -16) (16 16 16)
Only in CTF games
*/
	{
		"team_CTF_redflag",
		"sound/teamplay/flagtk_red.wav",
        { "models/flags/r_flag.md3",
		0, 0, 0 },
/* icon */		"icons/iconf_red1",
/* pickup */	"Red Flag",
		0,
		IT_WEAPON,
		WP_REDFLAG, AMMO_NONE,
		40,
/* precache */ "",
/* sounds */ "sound/teamplay/flagcap_red.wav sound/teamplay/flagtk_red.wav sound/teamplay/flagret_red.wav"
	},

/*QUAKED team_CTF_blueflag (0 0 1) (-16 -16 -16) (16 16 16)
Only in CTF games
*/
	{
		"team_CTF_blueflag",
		"sound/teamplay/flagtk_blu.wav",
        { "models/flags/b_flag.md3",
		0, 0, 0 },
/* icon */		"icons/iconf_blu1",
/* pickup */	"Blue Flag",
		0,
		IT_WEAPON,
		WP_BLUEFLAG, AMMO_NONE,
		40,
/* precache */ "",
/* sounds */ "sound/teamplay/flagcap_blu.wav sound/teamplay/flagtk_blu.wav sound/teamplay/flagret_blu.wav"
	},

	/*QUAKED team_CTF_ball (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"team_CTF_ball",
		"sound/misc/w_pkup.wav",
        { "models/powerups/instant/quad.md3", 
		0, 0, 0},
/* icon */		"menu/art/level_complete4", //red Q3 symbol
/* pickup */	"Ball",
		1,
		IT_WEAPON,
		WP_BALL, AMMO_NONE,
		5,
/* precache */ "",
/* sounds */ ""
	},

	//
	// AMMO ITEMS
	//

/*QUAKED ammo_shells (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_shells",
		"sound/misc/am_pkup.wav",
        { "models/weapons2/shells/s_shell.md3", 
		0, 0, 0},
/* icon */		"icons/icona_shotgun",
/* pickup */	"Shells",
		12,
		IT_AMMO,
		AMMO_SHELLS, 0,
		0.125,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED ammo_bullets (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_bullets",
		"sound/misc/am_pkup.wav",
        { "models/weapons2/shells/m_shell.md3", 
		0, 0, 0},
/* icon */		"icons/icona_machinegun",
/* pickup */	"Bullets",
		100,
		IT_AMMO,
		AMMO_BULLETS, 0,
		0.05,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED ammo_grenades (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_grenades",
		"sound/misc/am_pkup.wav",
		{ "models/ammo/grenade2.md3", 
		0, 0, 0},
/* icon */		"icons/icona_grenade",
/* pickup */	"Grenades",
		1,
		IT_AMMO,
		AMMO_GRENADES, 0,
		0.4,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED ammo_napalm (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_napalm",
		"sound/misc/am_pkup.wav",
		{ "models/ammo/grenade1.md3", 
		0, 0, 0},
/* icon */		"icons/icona_flamethrower",
/* pickup */	"Napalm",
		100,
		IT_AMMO,
		AMMO_NAPALM, 0,
		0.01,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED ammo_rockets (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_rockets",
		"sound/misc/am_pkup.wav",
        { "models/ammo/rocket2.md3", 
		0, 0, 0},
/* icon */		"icons/icona_rocket",
/* pickup */	"Rockets",
		5,
		IT_AMMO,
		AMMO_ROCKETS, 0,
		1,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED ammo_slugs (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_slugs",
		"sound/misc/am_pkup.wav",
        { "models/ammo/railgun.md3", 
		0, 0, 0},
/* icon */		"icons/icona_railgun",
/* pickup */	"Slugs",
		10,
		IT_AMMO,
		AMMO_SLUGS, 0,
		0.2,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED ammo_cells (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"ammo_cells", // generic ammo for Gauntlet, Lightning Gun, and Plasmagun
		"sound/misc/am_pkup.wav",
        { "models/powerups/ammo/bfgam.md3", // looks like BFG ammo
		0, 0, 0},
/* icon */		"icons/icona_bfg",
/* pickup */	"Power Cell",
		50,
		IT_AMMO,
		AMMO_CELLS, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

	//
	// HOLDABLE ITEMS
	//

/*QUAKED holdable_medkit (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"holdable_medkit", 
		"sound/items/holdable.wav",
        { 
		"models/powerups/holdable/medkit.md3", 
		"models/powerups/holdable/medkit_sphere.md3",
		0, 0},
/* icon */		"icons/medkit",
/* pickup */	"Berzerk",
		2,
		IT_HEALTH,
		PW_BERZERK, 0,
		10,
/* precache */ "",
/* sounds */ "sound/items/use_medkit.wav"
	},

	//
	// POWERUP ITEMS
	//

	{
		"item_berzerk", 
		"sound/items/quaddamage.wav",
        { "models/powerups/holdable/medkit.md3", 
        "models/powerups/instant/quad_ring.md3",
		0, 0 },
/* icon */		"icons/medkit",
/* pickup */	"Quad Damage",
		30,
		IT_POWERUP,
		PW_BERZERK, 0,
		0,
/* precache */ "",
/* sounds */ "sound/items/damage2.wav sound/items/damage3.wav"
	},

/*QUAKED item_quad (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_quad", 
		"sound/items/quaddamage.wav",
        { "models/powerups/instant/quad.md3", 
        "models/powerups/instant/quad_ring.md3",
		0, 0 },
/* icon */		"icons/quad",
/* pickup */	"Quad Damage",
		30,
		IT_POWERUP,
		PW_QUAD, 0,
		0,
/* precache */ "",
/* sounds */ "sound/items/damage2.wav sound/items/damage3.wav"
	},
/*QUAKED item_enviro (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_enviro",
		"sound/items/protect.wav",
        { "models/powerups/instant/enviro.md3", 
		"models/powerups/instant/enviro_ring.md3", 
		0, 0 },
/* icon */		"icons/envirosuit",
/* pickup */	"Battle Suit",
		30,
		IT_POWERUP,
		PW_BATTLESUIT, 0,
		0,
/* precache */ "",
/* sounds */ "sound/items/airout.wav sound/items/protect3.wav"
	},

/*QUAKED item_haste (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_haste",
		"sound/items/haste.wav",
        { "models/powerups/instant/haste.md3", 
		"models/powerups/instant/haste_ring.md3", 
		0, 0 },
/* icon */		"icons/haste",
/* pickup */	"Speed",
		30,
		IT_POWERUP,
		PW_HASTE, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED item_invis (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_invis",
		"sound/items/invisibility.wav",
        { "models/powerups/instant/invis.md3", 
		"models/powerups/instant/invis_ring.md3", 
		0, 0 },
/* icon */		"icons/invis",
/* pickup */	"Invisibility",
		30,
		IT_POWERUP,
		PW_INVIS, 0,
		0,
/* precache */ "",
/* sounds */ ""
	},

/*QUAKED item_regen (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_regen",
		"sound/items/regeneration.wav",
        { "models/powerups/instant/regen.md3", 
		"models/powerups/instant/regen_ring.md3", 
		0, 0 },
/* icon */		"icons/regen",
/* pickup */	"Regeneration",
		30,
		IT_POWERUP,
		PW_REGEN, 0,
		0,
/* precache */ "",
/* sounds */ "sound/items/regen.wav"
	},

/*QUAKED item_flight (.3 .3 1) (-16 -16 -16) (16 16 16) suspended
*/
	{
		"item_flight",
		"sound/items/flight.wav",
        { "models/powerups/instant/flight.md3", 
		"models/powerups/instant/flight_ring.md3", 
		0, 0 },
/* icon */		"icons/flight",
/* pickup */	"Flight",
		60,
		IT_POWERUP,
		PW_FLIGHT, 0,
		0,
/* precache */ "",
/* sounds */ "sound/items/flight.wav"
	},

	// end of list marker
	{NULL}
};

int		bg_numItems = sizeof(bg_itemlist) / sizeof(bg_itemlist[0]) - 1;


/*
==============
BG_FindItemForPowerup
==============
*/
gitem_t	*BG_FindItemForPowerup( powerup_t pw ) {
	int		i;
	for ( i = 0 ; i < bg_numItems ; i++ ) {
		if ( (bg_itemlist[i].giType == IT_POWERUP) && 
			bg_itemlist[i].giTag == pw ) {
			return &bg_itemlist[i];
		}
	}

	return NULL;
}


/*
===============
BG_FindItemForWeapon

===============
*/
gitem_t	*BG_FindItemForWeapon( weapon_t weapon ) {
	gitem_t	*it;
	
	for ( it = bg_itemlist + 1 ; it->classname ; it++) {
		if ( it->giType == IT_WEAPON && it->giTag == weapon ) {
			return it;
		}
	}

	Com_Error( ERR_DROP, "Couldn't find item for weapon %i", weapon);
	return NULL;
}

gitem_t	*BG_FindAmmoForWeapon( weapon_t weapon ) {
	gitem_t	*it, *w;
	w = BG_FindItemForWeapon(weapon);
	if(!w->giUses)
		return bg_itemlist;
	for ( it = bg_itemlist + 1 ; it->classname ; it++) 
	{
		if ( it->giType == IT_AMMO && it->giTag == w->giUses)
			return it;
	}
	return bg_itemlist;
	return NULL;
}

/*
===============
BG_FindItem

===============
*/
gitem_t	*BG_FindItem( const char *pickupName ) {
	gitem_t	*it;
	
	for ( it = bg_itemlist + 1 ; it->classname ; it++ ) {
		if ( !Q_stricmp( it->pickup_name, pickupName ) )
			return it;
	}

	return NULL;
}

/*
============
BG_PlayerTouchesItem

Items can be picked up without actually touching their physical bounds to make
grabbing them easier
============
*/
char *BG_GetCorpseModel(int index)
{
	char modelString[MAX_INFO_STRING];
	trap_GetConfigstring(CS_CORPSES, modelString, sizeof(modelString));
	return Info_ValueForKey(modelString, va("%d", index));
}

qboolean	BG_PlayerTouchesItem( playerState_t *ps, entityState_t *item, int atTime ) {
	vec3_t		origin;

	BG_EvaluateTrajectory( &item->pos, atTime, origin );
	
	// we are ignoring ducked differences here
	if ( ps->origin[0] - origin[0] > 44
		|| ps->origin[0] - origin[0] < -50
		|| ps->origin[1] - origin[1] > 36
		|| ps->origin[1] - origin[1] < -36
		|| ps->origin[2] - origin[2] > 36
		|| ps->origin[2] - origin[2] < -36 ) {
		return qfalse;
	}

	return qtrue;
}



/*
================
BG_CanItemBeGrabbed

Returns false if the item should not be picked up.
This needs to be the same for client side prediction and server use.
================
*/
qboolean	BG_CanItemBeGrabbed( const entityState_t *ent, const playerState_t *ps ) {
	gitem_t	*item;
	vec3_t org, dir;
	
	if ( ent->modelindex < 1 || ent->modelindex >= bg_numItems ) {
		Com_Error( ERR_DROP, "BG_CanItemBeGrabbed: index out of range" );
	}
	
	
	item = &bg_itemlist[ent->modelindex];
	if(item->giType == IT_POWERUP)
		return qtrue; // powerups are magic, they can ALWAYS be picked up.
	
	if(item->giType == IT_WEAPON && (item->giTag == WP_REDFLAG || item->giTag == WP_BLUEFLAG) && !(ent->eFlags & EF_DEAD))
	{ // check for CTF flags; this is the only time we can "pick up" an item if we already have one
		if(!ps->weapon)
		{
			if(item->giTag == WP_REDFLAG && ps->persistant[PERS_TEAM] == TEAM_RED)
				return qfalse; //can't pick up our own flag
			else if(item->giTag == WP_BLUEFLAG && ps->persistant[PERS_TEAM] == TEAM_BLUE)
				return qfalse; //can't pick up our own flag
			return qtrue;
		}
		else if(bg_itemlist[ps->weapon].giType == IT_WEAPON &&  bg_itemlist[ps->weapon].giTag >= WP_REDFLAG)
			return qtrue;
		else
			return qfalse;
	}
	if(ps->weapon) // can only pick something up if our hands are empty.
		return qfalse;
	
	if((ps->pm_flags & PMF_FIRE_HELD) || ps->weaponstate != WEAPON_READY )
		return qfalse; // we're busy
	
	if(ent->clientNum == ps->clientNum && (ent->eFlags & EF_DEAD))
		return qfalse; // ## Hentai ## - we just dropped this

	VectorSubtract(ent->origin, ps->origin, org);
	AngleVectors(ps->viewangles, dir, NULL, NULL);	
	
	if(!(ps->pm_flags & PMF_USE_ITEM_HELD))
		return qfalse; // have to hit 'use' button
	if(((ps->origin[2] - ent->origin[2] > 12) || (DotProduct(dir, org) < 0.25)) && !((ps->legsAnim & ~ANIM_TOGGLEBIT) == LEGS_IDLECR || (ps->legsAnim & ~(ANIM_TOGGLEBIT | ANIM_CLIMBWALLS)) == LEGS_WALKCR))
		return qfalse; // can only grab items below/behind us while crouched.
		
	return qtrue; // we're not holding anything else
}

//======================================================================

/*
================
BG_EvaluateTrajectory

================
*/
void BG_EvaluateTrajectory( const trajectory_t *tr, int atTime, vec3_t result ) {
	float		deltaTime;
	float		phase;

	switch( tr->trType ) {
	case TR_STATIONARY:
	case TR_INTERPOLATE:
		VectorCopy( tr->trBase, result );
		break;
	case TR_LINEAR:
		deltaTime = ( atTime - tr->trTime ) * 0.001;	// milliseconds to seconds
		VectorMA( tr->trBase, deltaTime, tr->trDelta, result );
		break;
	case TR_SINE:
		deltaTime = ( atTime - tr->trTime ) / (float) tr->trDuration;
		phase = sin( deltaTime * M_PI * 2 );
		VectorMA( tr->trBase, phase, tr->trDelta, result );
		break;
	case TR_LINEAR_STOP:
		if ( atTime > tr->trTime + tr->trDuration ) {
			atTime = tr->trTime + tr->trDuration;
		}
		deltaTime = ( atTime - tr->trTime ) * 0.001;	// milliseconds to seconds
		if ( deltaTime < 0 ) {
			deltaTime = 0;
		}
		VectorMA( tr->trBase, deltaTime, tr->trDelta, result );
		break;
	case TR_GRAVITY:
		deltaTime = ( atTime - tr->trTime ) * 0.001;	// milliseconds to seconds
		VectorMA( tr->trBase, deltaTime, tr->trDelta, result );
		result[2] -= 0.5 * DEFAULT_GRAVITY * deltaTime * deltaTime;		// FIXME: local gravity... (### Hentai ### - Done!)
		break;
	default:
		Com_Error( ERR_DROP, "BG_EvaluateTrajectory: unknown trType: %i", tr->trTime );
		break;
	}
}

/*
================
BG_EvaluateTrajectoryDelta

For determining velocity at a given time
================
*/
void BG_EvaluateTrajectoryDelta( const trajectory_t *tr, int atTime, vec3_t result ) {
	float	deltaTime;
	float	phase;

	switch( tr->trType ) {
	case TR_STATIONARY:
	case TR_INTERPOLATE:
		VectorClear( result );
		break;
	case TR_LINEAR:
		VectorCopy( tr->trDelta, result );
		break;
	case TR_SINE:
		deltaTime = ( atTime - tr->trTime ) / (float) tr->trDuration;
		phase = cos( deltaTime * M_PI * 2 );	// derivative of sin = cos
		phase *= 0.5;
		VectorScale( tr->trDelta, phase, result );
		break;
	case TR_LINEAR_STOP:
		if ( atTime > tr->trTime + tr->trDuration ) {
			VectorClear( result );
			return;
		}
		VectorCopy( tr->trDelta, result );
		break;
	case TR_GRAVITY:
		deltaTime = ( atTime - tr->trTime ) * 0.001;	// milliseconds to seconds
		VectorCopy( tr->trDelta, result );
		result[2] -= DEFAULT_GRAVITY * deltaTime;		// FIXME: local gravity... ( ### Hentai ### - Done!)
		break;
	default:
		Com_Error( ERR_DROP, "BG_EvaluateTrajectoryDelta: unknown trType: %i", tr->trTime );
		break;
	}
}

/*
===============
BG_AddPredictableEventToPlayerstate

Handles the sequence numbers
===============
*/
void BG_AddPredictableEventToPlayerstate( int newEvent, int eventParm, playerState_t *ps ) {
	ps->events[ps->eventSequence & (MAX_PS_EVENTS-1)] = newEvent;
	ps->eventParms[ps->eventSequence & (MAX_PS_EVENTS-1)] = eventParm;
	ps->eventSequence++;
}


/*
========================
BG_PlayerStateToEntityState

This is done after each set of usercmd_t on the server,
and after local prediction on the client
========================
*/
void BG_PlayerStateToEntityState( playerState_t *ps, entityState_t *s, int time, qboolean snap ) {
	int		i;

	s->number = ps->clientNum;
	if ( ps->pm_type == PM_INTERMISSION || ps->pm_type == PM_SPECTATOR || (ps->stats[STAT_HEALTH] <= GIB_HEALTH && ps->stats[STAT_HEAD_DAMAGE] < MAX_HEALTH)) 
		s->eType = ET_INVISIBLE;
	else 
		s->eType = ET_PLAYER;

	s->eFlags = ps->eFlags;
	if ( ps->stats[STAT_HEALTH] <= 0 ) 
	{
		s->eFlags |= EF_DEAD;
		if(ps->stats[STAT_HEAD_DAMAGE] > MAX_HEALTH)
			s->eFlags |= EF_AWARD_IMPRESSIVE;
		else if(ps->stats[STAT_HEALTH] < 0)
			s->eFlags &= ~EF_AWARD_IMPRESSIVE;
	}
	else 
		s->eFlags &= ~EF_DEAD;

	s->pos.trType = TR_INTERPOLATE;
	VectorCopy( ps->origin, s->pos.trBase );
	if ( snap ) {
		SnapVector( s->pos.trBase );
	}

	s->apos.trType = TR_INTERPOLATE;
	VectorCopy( ps->viewangles, s->apos.trBase );
	if ( snap ) {
		SnapVector( s->apos.trBase );
	}

	//s->time = ?;

	if(ps->powerups[PW_INVIS] > 0 )
		s->time2 = (ps->powerups[PW_INVIS] - time);	
	else
		s->time2 = (ps->ammo[AMMO_CURRENT] & 1023);

	// s->origin[0] = ?
	// s->origin[1] = ?
	// s->origin[2] = ?

	// s->origin2[0] = ?
	// s->origin2[1] = ?
	// s->origin2[2] = ?

	// s->angles[YAW] = ?
	// s->angles[PITCH] = ?
	// s->angles[ROLL] = ?
	
	s->angles2[YAW] = ps->movementDir;
	// s->angles2[PITCH] = ?
	// s->angles2[ROLL] = ?

	s->otherEntityNum = ps->ammo[AMMO_ITEM1] | ((ps->ammo[AMMO_ITEM1 + 1] & 255) << 8);
	s->otherEntityNum2 = ps->ammo[AMMO_ITEM2] | ((ps->ammo[AMMO_ITEM2 + 1] & 255) << 8);

	s->groundEntityNum = ps->groundEntityNum;

	s->constantLight = ps->ammo[AMMO_ITEM3] | ((ps->ammo[AMMO_ITEM3 + 1] & 255) << 8);
	s->loopSound = ps->ammo[AMMO_ITEM4] | ((ps->ammo[AMMO_ITEM4 + 1] & 255) << 8);

	s->modelindex = ps->ammo[AMMO_ITEM5] | ((ps->ammo[AMMO_ITEM5 + 1] & 255) << 8);
	s->modelindex2 = ps->ammo[AMMO_ITEM6] | ((ps->ammo[AMMO_ITEM6 + 1] & 255) << 8);

	s->clientNum = ps->clientNum;		// ET_PLAYER looks here instead of at number
										// so corpses can also reference the proper config
										// ### Hentai ### - FIXME: Is this needed anymore?

	s->frame = ps->ammo[AMMO_ITEM_BIG];
	s->solid = (ps->ammo[AMMO_ITEM_BIG + 1] & 1023); // last item is a weapon, so also send its ammo as 10 bits (since we could have more than 255)

	if ( ps->externalEvent ) {
		s->event = ps->externalEvent;
		s->eventParm = ps->externalEventParm;
	} else {
		int		seq;

		seq = (ps->eventSequence-1) & (MAX_PS_EVENTS-1);
		s->event = ps->events[ seq ] | ( ( ps->eventSequence & 3 ) << 8 );
		s->eventParm = ps->eventParms[ seq ];
	}

	
	s->powerups = 0;
	for ( i = 0 ; i < MAX_POWERUPS ; i++ ) {
		if ( ps->powerups[ i ] ) {
			s->powerups |= 1 << i;
		}
	}
	if(ps->stats[STAT_ARMOR] > 0)
		s->powerups |= 1 << PW_ARMOR;

	s->weapon = ps->weapon;

	s->legsAnim = ps->legsAnim;
	s->torsoAnim = ps->torsoAnim;



}

char *skipWhitespace(char *data)
{
	char *rval = data;
	while(*rval <= ' ')
	{
		if(rval[0] == '\0')
			return rval;
		rval++;
	}
	return rval;

}

int atov(char *string, vec3_t v)
{
	char *s = string;
	int k = 0;
	while(*s != '\n' && *s != '\0')
	{
		if((*s >= '0' && *s <= '9') || (*s == '-' || *s == '+' || *s == '.'))
		{
			v[k++] = atof(s);
			if(k == 3)
				return qtrue;
			while((*s >= '0' && *s <= '9') || (*s == '-' || *s == '+' || *s == '.'))
				s++;
		}
		s++;
	}
	return qfalse;
}

int				p_classes;
playerclass_t	p_class[MAX_PCLASSES];

char *animNames[] = {
	"BOTH_DEATH1",//0
	"BOTH_DEAD1",//1
	"BOTH_DEATH2",//2 
	"BOTH_DEAD2",//3
	"BOTH_DEATH3",//4
	"BOTH_DEAD3",//5

	"TORSO_GESTURE",//6

	"TORSO_ATTACK",//7
	"TORSO_ATTACK2",//8

	"TORSO_DROP",//9
	"TORSO_RAISE",//10

	"TORSO_STAND",//11
	"TORSO_STAND2",//12

	"LEGS_WALKCR",//13
	"LEGS_WALK",//14
	"LEGS_RUN",//15
	"LEGS_BACK",//16
	"LEGS_SWIM",//17

	"LEGS_JUMP",//18
	"LEGS_LAND",//19

	"LEGS_JUMPB",//20
	"LEGS_LANDB",//21

	"LEGS_IDLE",//22
	"LEGS_IDLECR",//23

	"LEGS_TURN",//24

};

int BG_PlayerClass(char *model)
{
	char test[16], tmodel[32];
	int i, bestfit = -1, slen;
	
	test[15] = 0;
	tmodel[31] = 0;
	for(slen = 0; slen < 15; slen++)
	{
		if(model[slen] == '/' || model[slen] <= ' ')
		{
			test[slen] = 0;
			break;
		}
		test[slen] = toupper(model[slen]);
		
	}
	for(slen = 0; slen < 31; slen++)
	{
		if(model[slen] <= ' ')
		{
			tmodel[slen] = 0;
			break;
		}
		tmodel[slen] = toupper(model[slen]);
		
	}
	
	for(i = 0; i < p_classes; i++)
	{
		if(!Q_strncmp(tmodel, p_class[i].name, 32))
		{ // test specific skins first
			return i;
		}		
		if(!Q_strncmp(test, p_class[i].name, 16))
		{ // then test generic model
			bestfit = i;
		}
	}
	if(p_classes < MAX_PCLASSES)
	{ // try loading a new class
		if(BG_ParseClass(model, &p_class[p_classes]))
		{
			if(strchr(p_class[p_classes].name, '/') || bestfit == -1)
			{
				return p_classes++;
			}
		}
	}
	return bestfit; // use a found precached, or default (-1)
}
char	data[65536];

qboolean BG_ParseAnimationFile( const char *filename, animation_t *animations ) {
	char		*text_p, *prev;
	int			len;
	int			i;
	char		*token;
	float		fps;
	int			skip;
	char		text[20000];
	fileHandle_t	f;

	memset( animations, 0, sizeof( animation_t ) * MAX_ANIMATIONS );

	// load the file
	len = trap_FS_FOpenFile( filename, &f, FS_READ );
	if ( len <= 0 ) {
		return qfalse;
	}
	if ( len >= ( sizeof( text ) - 1 ) ) {
		Com_Printf( "File %s too long\n", filename );
		return qfalse;
	}
	trap_FS_Read( text, len, f );
	text[len] = 0;
	trap_FS_FCloseFile( f );

	// parse the text
	text_p = text;
	skip = 0;	// quite the compiler warning

	// read optional parameters
	while ( 1 ) {
		prev = text_p;	// so we can unget
		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		if ( !Q_stricmp( token, "footsteps" ) ) {
			token = COM_Parse( &text_p );
			if ( !token ) {
				break;
			}
			continue;
		} else if ( !Q_stricmp( token, "headoffset" ) ) {
			for ( i = 0 ; i < 3 ; i++ ) {
				token = COM_Parse( &text_p );
				if ( !token ) {
					break;
				}
			}
			continue;
		} else if ( !Q_stricmp( token, "sex" ) ) {
			token = COM_Parse( &text_p );
			if ( !token ) {
				break;
			}
			continue;
		}

		// if it is a number, start parsing animations
		if ( token[0] >= '0' && token[0] <= '9' ) {
			text_p = prev;	// unget the token
			break;
		}

		Com_Printf( "unknown token '%s' is %s\n", token, filename );
	}

	// read information for each frame
	for ( i = 0 ; i < MAX_ANIMATIONS ; i++ ) {

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		animations[i].firstFrame = atoi( token );
		// leg only frames are adjusted to not count the upper body only frames
		if ( i == LEGS_WALKCR ) {
			skip = animations[LEGS_WALKCR].firstFrame - animations[TORSO_GESTURE].firstFrame;
		}
		if ( i >= LEGS_WALKCR ) {
			animations[i].firstFrame -= skip;
		}

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		animations[i].numFrames = atoi( token );

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		animations[i].loopFrames = atoi( token );

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		fps = atof( token );
		if ( fps == 0 ) {
			fps = 1;
		}
		animations[i].frameLerp = 1000 / fps;
		animations[i].initialLerp = 1000 / fps;
	}

	if ( i != MAX_ANIMATIONS ) {
		Com_Printf( "Error parsing animation file: %s", filename );
		return qfalse;
	}

	return qtrue;
}

qboolean BG_ParseClass(const char *skinname, playerclass_t *pclass)
{
	char classname[32], filename[64], *token = data;
	int i;
	fileHandle_t	classfile;
	
	Q_strncpyz(classname, skinname, sizeof(classname));
		
	Com_sprintf(filename, sizeof(filename), "models/players/%s.playerclass", classname);
	i = trap_FS_FOpenFile(filename, &classfile, FS_READ );
	if(classfile)
	{		
		trap_FS_Read(data, 64000, classfile);
		trap_FS_FCloseFile(classfile);
		if(i > 64000)
		{
			Com_Printf( "WARNING: Playerclass file %s is too large!\n", filename );
			i = 64000;
		}
		
	}
	else
	{ // couldn't find one for the skin, so find one for the model.
		char *slash;
		slash = strchr(classname, '/');

		if(slash)
		{
			slash[0] = '\0';
		}
		Com_sprintf(filename, sizeof(filename), "models/players/%s/default.playerclass", classname);
		i = trap_FS_FOpenFile(filename, &classfile, FS_READ );
		if(classfile)
		{
							
			if(i > 64000)
			{
				Com_Printf( "WARNING: Playerclass file %s is too large!\n", filename );
				i = 64000;
			}
			trap_FS_Read(data, i, classfile);
			trap_FS_FCloseFile(classfile);
			
			
		}
		else
		{
			return qfalse;
		}
	}
	Com_sprintf(data + i, 6, "\nEND\n");
	pclass->head[0] = 0;
	pclass->head[1] = 0;
	pclass->head[2] = 28;
	pclass->headsize = 6;
	pclass->waistsize = 1.0;

	pclass->gun[0] = 0;
	pclass->gun[1] = 8;
	pclass->gun[2] = 16;
	
	pclass->stand[0] = 0;
	pclass->walk[0] = -255;
	pclass->run[0] = -255;
	pclass->bleeds = qtrue;			
	pclass->wallclimber = qfalse;			
			
	pclass->mass = 200;
	pclass->strength = 200;			
	pclass->speed = 1;
	pclass->crouchspeed = 1;
	pclass->jumpspeed = 1;
	
	pclass->jumpfactor = 1;
	pclass->jumppause = 0.25;				
				
	pclass->jumpgravity = 1;
	
	pclass->head_dmgfactor = 2;
	pclass->torso_dmgfactor = 1;
	pclass->legs_dmgfactor = 1;
	pclass->name[31] = 0;
	for(i = 0; i < 31; i++)
	{
		if(classname[i] <= ' ')
		{
			pclass->name[i] = '\0';
			break;
		}
		pclass->name[i] = toupper(classname[i]);
			
	}
	while(!0)
	{
		token = skipWhitespace(token);
		if ( !token[0] ) {
			break;
		}
		else if ( !Q_strncmp( token, "speed:", 6 ) ) 
		{
			token = skipWhitespace(token + 6);
			pclass->speed = atof(token);			
		}
		else if ( !Q_strncmp( token, "cr_spd:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->crouchspeed = atof(token);
		}
		else if ( !Q_strncmp( token, "jspeed:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->jumpspeed = atof(token);
		}
		else if ( !Q_strncmp( token, "jpause:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->jumppause = atof(token);
			if(pclass->jumppause  <= 0)
				pclass->jumppause = 0.01;
		}
		else if ( !Q_strncmp( token, "jump_h:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->jumpfactor = atof(token);
		}
		else if ( !Q_strncmp( token, "jump_g:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->jumpgravity = atof(token);
		}
		else if ( !Q_strncmp( token, "mass:", 5 ) ) 
		{
			token = skipWhitespace(token + 5);
			pclass->mass = atoi(token);
		}

		else if ( !Q_strncmp( token, "carry:", 6 ) ) 
		{
			token = skipWhitespace(token + 6);
			pclass->strength = atoi(token);
		}
		
		else if ( !Q_strncmp( token, "h_head:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->head_dmgfactor = atof(token);
		}
		else if ( !Q_strncmp( token, "h_body:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->torso_dmgfactor = atof(token);
		}
		else if ( !Q_strncmp( token, "h_legs:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->legs_dmgfactor = atof(token);
		}
		else if ( !Q_strncmp( token, "w_head:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->headsize = atof(token);
		}
		else if ( !Q_strncmp( token, "wwaist:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			pclass->waistsize = atof(token);
		}
		else if ( !Q_strncmp( token, "bleed:", 6 ) ) 
		{
			token = skipWhitespace(token + 6);
			if(!Q_strncmp(token, "no", 2))
				pclass->bleeds = qfalse;
			else if(!Q_strncmp(token, "yes", 3))
				pclass->bleeds = qtrue;
		}
		else if ( !Q_strncmp( token, "sticky:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			if(!Q_strncmp(token, "no", 2))
				pclass->wallclimber = qfalse;
			else if(!Q_strncmp(token, "yes", 3))
				pclass->wallclimber = qtrue;
		}
		else if ( !Q_strncmp( token, "crouch:", 7 ) ) 
		{
			token = skipWhitespace(token + 7);
			atov(token, pclass->crouch );
		}
		else if ( !Q_strncmp( token, "stand:", 6 ) ) 
		{
			token = skipWhitespace(token + 6);
			atov(token, pclass->stand );
		}
		else if ( !Q_strncmp( token, "walk:", 5 ) ) 
		{
			token = skipWhitespace(token + 5);
			atov(token, pclass->walk );
		}
		else if ( !Q_strncmp( token, "run:", 4 ) ) 
		{
			token = skipWhitespace(token + 4);
			atov(token, pclass->run );
		}
		else if ( !Q_strncmp( token, "jump:", 5 ) ) 
		{
			token = skipWhitespace(token + 5);
			atov(token, pclass->jump );
		}
		else if ( !Q_strncmp( token, "head:", 5 ) ) 
		{
			token = skipWhitespace(token + 5);
			atov(token, pclass->head );
		}
		else if ( !Q_strncmp( token, "gun:", 4 ) ) 
		{
			token = skipWhitespace(token + 4);
			atov(token, pclass->gun );
			
		}

		else if ( !Q_strncmp( token, "END\n", 4 ) ) 
		{
			break;	
		}
		for(token; *token != '\n'; token++)
		{
			if(*token == '\0') 
				break;			
		}		
		token++;

	}	
	if(pclass->crouchspeed == -255)
		p_class->crouchspeed = p_class->speed;
	if(p_class->strength == -255)
		p_class->strength = p_class->mass;
	if(p_class->jumpspeed == -255)
		p_class->jumpspeed = p_class->speed;
	if(p_class->walk[0] == -255)
		VectorCopy(p_class->stand, p_class->walk);
	if(p_class->run[0] == -255)
		VectorCopy(p_class->walk, p_class->run);

	{ // now we need to break off the slash, to load up animation files
		char *slash;
		slash = strchr(classname, '/');

		if(slash)
		{
			slash[0] = '\0';
		}
		Com_sprintf(filename, sizeof(filename), "models/players/%s/animation.cfg", classname);
		if(!BG_ParseAnimationFile(filename, pclass->animations))
			return qfalse;
	}

	Com_sprintf(filename, sizeof(filename), "models/players/%s/lower.animation", classname);
	i = trap_FS_FOpenFile(filename, &classfile, FS_READ );
	if(classfile)
	{		
		trap_FS_Read(data, 64000, classfile);
		trap_FS_FCloseFile(classfile);
		if(i > 64000)
		{
			Com_Printf( "WARNING: Playerclass file %s is too large!\n", filename );
			i = 64000;
		}
		Com_sprintf(data + i, 6, "\nEND\n");

		while(!0)
		{
			token = skipWhitespace(token);
			if ( !token[0] ) {
				break;
			}
			else if ( !Q_strncmp( token, ":", 1 ) ) 
			{
				int frame;
				char *found;
				token = skipWhitespace(token + 1);
				frame = atoi(token);
				if(frame > 255)
				{
					Com_Printf( "WARNING: frame data for frame %d (> 255) found!\n", frame);
				}
				else
				{
					found = strstr(token, "tag_torso=");
					if(found)
					{
						found += strlen("tag_torso=");
						found = skipWhitespace(found);
						atov(found, pclass->tag_torso[frame]);
					}
				}
			}
			else if ( !Q_strncmp( token, "END\n", 4 ) ) 
			{
				break;	
			}
			
			
			for(token; *token != '\n'; token++)
			{
				if(*token == '\0') 
					break;			
			}		

			token++;
		}
		
	}

	Com_sprintf(filename, sizeof(filename), "models/players/%s/upper.animation", classname);
	i = trap_FS_FOpenFile(filename, &classfile, FS_READ );
	if(classfile)
	{		
		trap_FS_Read(data, 64000, classfile);
		trap_FS_FCloseFile(classfile);
		if(i > 64000)
		{
			Com_Printf( "WARNING: Playerclass file %s is too large!\n", filename );
			i = 64000;
		}
		Com_sprintf(data + i, 6, "\nEND\n");

		while(!0)
		{
			token = skipWhitespace(token);
			if ( !token[0] ) {
				break;
			}
			else if ( !Q_strncmp( token, ":", 1 ) ) 
			{
				int frame;
				char *found;
				token = skipWhitespace(token + 1);
				frame = atoi(token);				
				if(frame > 255)
				{
					Com_Printf( "WARNING: frame data for frame %d (> 255) found!\n", frame);
				}
				else
				{
					found = strstr(token, "tag_head=");
					if(found)
					{
						found += strlen("tag_head=");
						found = skipWhitespace(found);
						atov(found, pclass->tag_head[frame]);
					}
					found = strstr(token, "tag_weapon=");
					if(found)
					{
						found += strlen("tag_weapon=");
						found = skipWhitespace(found);
						atov(found, pclass->tag_weapon[frame]);
					}
				}
			}
			else if ( !Q_strncmp( token, "END\n", 4 ) ) 
			{
				break;	
			}
			
			
			for(token; *token != '\n'; token++)
			{
				if(*token == '\0') 
					break;			
			}		

			token++;
		}
	}
	
	return qtrue;

}


float BG_PlayerAngles(int playerclass, int legsFrame, int torsoFrame, vec3_t viewangles, vec3_t velocity, vec3_t muzzle, vec3_t headpoint ) 
{ // tihs was copied mostly from CG_PlayerAngles, an does basically the same thing, 
	// only with the playerclass "tag" info and without lerping

	vec3_t		legsAngles, torsoAngles, headAngles;
	vec3_t		waistpoint, neckpoint, gunpoint;
	vec3_t		forward, right, up;
	float		dest, *tempv;
	static	int	movementOffsets[8] = { 0, 22, 45, -22, 0, 22, -45, -22 };
	vec3_t		vel;
	float		speed;
	//int wallclimb;	
	if(playerclass < 0 || playerclass >= p_classes)
		return 0; // error?

	//wallclimb = legsAnim & ANIM_CLIMBWALLS;
	//legsAnim &= ~(ANIM_TOGGLEBIT | wallclimb);
	
//	if(legsAnim <= BOTH_DEAD3)
//	{
//		muzzle[0] = 443556;
//		return 0; // doesn't matter.(He's dead, Jim.)
//	}
	
	VectorCopy( viewangles, headAngles );
	headAngles[YAW] = AngleMod( headAngles[YAW] );
	VectorClear( legsAngles );
	VectorClear( torsoAngles );
	torsoAngles[YAW] = legsAngles[YAW] = headAngles[YAW];
	// --------- yaw -------------

	// allow yaw to drift a bit

//	if ( ( legsAnim & ~ANIM_TOGGLEBIT ) != LEGS_IDLE ) {
//		// if not standing still, always point all in the same direction
//
//	}

	// --------- pitch -------------

	// only show a fraction of the pitch angle in the torso
	if ( headAngles[PITCH] > 180 ) {
		dest = (-360 + headAngles[PITCH]) * 0.75;
	} else {
		dest = headAngles[PITCH] * 0.75;
	}
	torsoAngles[PITCH] = dest;

	// --------- roll -------------


	// lean towards the direction of travel
	
	VectorCopy( velocity, vel);
	speed = VectorNormalize( vel);
	if ( speed ) {
		vec3_t	axis[3];
		float	side;

		speed *= 0.05;

		AnglesToAxis( legsAngles, axis );
		side = speed * DotProduct( vel, axis[1] );
		legsAngles[ROLL] -= side;

		side = speed * DotProduct( vel, axis[0] );
		legsAngles[PITCH] += side;
	}

	
	/*if((playerclass < 0) || (playerclass >= p_classes))
	{
		tempv = temppoint;
		tempv[0] = 0;
		tempv[1] = 0;
		if(legsAnim == LEGS_IDLECR || legsAnim == LEGS_WALKCR)
			tempv[2] = -8;
		else
			tempv[2] = 8;
		
	}
	else
	{
		if(legsAnim == LEGS_IDLECR || legsAnim == LEGS_WALKCR)
			tempv = p_class[playerclass].crouch;
		// FIXME: These next three don't seem to always compute right.
		else if(legsAnim == LEGS_JUMP || legsAnim == LEGS_JUMPB)
			tempv = p_class[playerclass].jump;
		//else if(speed > 160)
		//	tempv = p_class[playerclass].run;
		//else if(speed)
		//	tempv = p_class[playerclass].walk;
		else
			tempv = p_class[playerclass].stand;
	}*/
	tempv = p_class[playerclass].tag_torso[legsFrame];
	AngleVectors(legsAngles, forward, right, up);

	waistpoint[0] = tempv[0] * forward[0] + tempv[1] * right[0] + tempv[2] * up[0];
	waistpoint[1] = tempv[0] * forward[1] + tempv[1] * right[1] + tempv[2] * up[1];
	waistpoint[2] = tempv[0] * forward[2] + tempv[1] * right[2] + tempv[2] * up[2];
	
	AngleVectors(torsoAngles, forward, right, up);

	/*if((playerclass < 0) || (playerclass >= p_classes))
		tempv[2] = 16;
	else
		tempv = p_class[playerclass].head;
	*/
	tempv = p_class[playerclass].tag_head[torsoFrame];

	neckpoint[0] = waistpoint[0] + tempv[0] * forward[0] - tempv[1] * right[0] + tempv[2] * up[0];
	neckpoint[1] = waistpoint[1] + tempv[0] * forward[1] - tempv[1] * right[1] + tempv[2] * up[1];
	neckpoint[2] = waistpoint[2] + tempv[0] * forward[2] - tempv[1] * right[2] + tempv[2] * up[2];
	
	/*if((playerclass < 0) || (playerclass >= p_classes))
	{
		tempv[0] = 0;
		tempv[1] = 12;
		tempv[2] = 4;
	}
	else
		tempv = p_class[playerclass].gun;
	*/
	tempv = p_class[playerclass].tag_weapon[torsoFrame];
	

	gunpoint[0] = waistpoint[0] + tempv[0] * forward[0] - tempv[1] * right[0] + tempv[2] * up[0];
	gunpoint[1] = waistpoint[1] + tempv[0] * forward[1] - tempv[1] * right[1] + tempv[2] * up[1];
	gunpoint[2] = waistpoint[2] + tempv[0] * forward[2] - tempv[1] * right[2] + tempv[2] * up[2];
	
	AngleVectors(headAngles, forward, right, up);
	
	if((playerclass < 0) || (playerclass >= p_classes))
	{ // head point gets set to the center of the model's head.
		headpoint[0] = neckpoint[0] + up[0] * 4;
		headpoint[1] = neckpoint[1] + up[1] * 4;
		headpoint[2] = neckpoint[2] + up[2] * 4;
	}
	else if(p_class[playerclass].wallclimber)
	{ // FIXME: This works because Orbb is the only wallclimber. Should make seperate neck/head "tags".
		headpoint[0] = neckpoint[0];
		headpoint[1] = neckpoint[1];
		headpoint[2] = neckpoint[2];
	}
	else
	{
		headpoint[0] = neckpoint[0] + up[0] * 4; 
		headpoint[1] = neckpoint[1] + up[1] * 4;
		headpoint[2] = neckpoint[2] + up[2] * 4;
	}
	muzzle[0] = gunpoint[0] + forward[0] * 8; // muzzle gets set to the computed gun tag position
	muzzle[1] = gunpoint[1] + forward[1] * 8;
	muzzle[2] = gunpoint[2] + forward[2] * 8;
	
	return waistpoint[2]; // return value is waist height, making it really easy to determine leg hits
}
