// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_weapons.c -- events and effects dealing with weapons
#include "cg_local.h"

/*
==========================
CG_MachineGunEjectBrass
==========================
*/

polyVert_t vert[4];
vec3_t point[8];

void CG_AddWeaponWithPowerups( refEntity_t *gun, int powerups ) {
	// add powerup effects
	if ( powerups & ( 1 << PW_INVIS ) )
	{
		
		if(gun->shaderRGBA[3] > 0)
			gun->customShader = cgs.media.invisShader2;
		else
			gun->customShader = cgs.media.invisShader;
		trap_R_AddRefEntityToScene( gun );
	} else {
		if(powerups & (1 << PW_HASTE) && !(gun->renderfx & RF_THIRD_PERSON) )
		{
			localEntity_t *le;
			refEntity_t *re;
		
			le = CG_AllocLocalEntity();
			re = &le->refEntity;
	
			le->leType = LE_FADE_RGB;
			le->startTime = cg.time;
			le->endTime = cg.time + 1000;
			le->lifeRate = 1.0 / ( le->endTime - le->startTime );
			memcpy(re, gun, sizeof(refEntity_t));
			re->renderfx = 0;
			re->customSkin = 0;
			re->customShader = cgs.media.invisShader2;
			le->color[0] = 1;
			le->color[1] = 0;
			le->color[2] = 0;
			le->color[3] = 1;

			
		}

		trap_R_AddRefEntityToScene( gun );

		if ( powerups & ( 1 << PW_BATTLESUIT ) ) {
			gun->customShader = cgs.media.battleWeaponShader;
			trap_R_AddRefEntityToScene( gun );
		}
		if ( powerups & ( 1 << PW_QUAD ) ) {
			gun->customShader = cgs.media.quadWeaponShader;
			trap_R_AddRefEntityToScene( gun );
		}
	}
}
extern vmCvar_t	cg_realammobelts;

void CG_DrawAmmoBelt(int bullets, int ammotype, int powerups, vec3_t start, vec3_t aimdir, vec3_t feeddir, vec3_t vel)
{ /* bullets is how many to render, 
	 start is where the belt comes from, 
	 aimdir is the direction the first bullet's head should be pointed, 
	 feeddir is the direction the belt should 'feed' from 
  */
	refEntity_t ent;
	vec3_t forward, right, oldright, up;
	vec3_t org, neworg, ground;
	vec3_t mins, maxs;
	int i, skip, min_skip, lastdraw;
	float err, a_err, size;
	trace_t tr;
	
	if(bullets <= 0)
		return;
	if(ammotype == AMMO_ROCKETS)
	{
		memset(&ent, 0, sizeof(ent));
		ent.hModel = trap_R_RegisterModel( "models/ammo/rocket2.md3" );
		
		size = 1.5;
		min_skip = 1;
	}
	else if(ammotype == AMMO_BULLETS)
	{
		size = 0.5;
		for(i = 0; i < 8; i++)
		{
			vert[i].modulate[0] = 255;
			vert[i].modulate[1] = 255;
			vert[i].modulate[2] = 255;
			vert[i].modulate[3] = 255;
		}
		min_skip = (bullets / 25 + 1);
		if(min_skip > 4)
			min_skip = 4;
	}
	if(bullets * size > 200)
	{ // are you INSANE, man!?
		bullets = 200 / size;
	}
	mins[0] = -size * 4;
	mins[1] = -size * 4;
	mins[2] = -size;
	maxs[0] = size * 4;
	maxs[1] = size * 4;
	maxs[2] = size;
	VectorSubtract(cg.refdef.vieworg, start, org);
	
	
	a_err = VectorLength(org) * sqrt(cg.refdef.fov_x * cg.refdef.fov_x + cg.refdef.fov_y * cg.refdef.fov_y) / 400000;
	if(a_err > 0.25)
	{
		a_err = 0.25;
	}
	VectorCopy(aimdir, forward);
	VectorNormalize(forward);
	ProjectPointOnPlane(right, feeddir, forward);
	VectorNormalize(right);
	CrossProduct(forward, right, up);
	VectorCopy(start, org);
	
	ground[0] = org[0];
	ground[1] = org[1];
	ground[2] = org[2] - 128;
	CG_Trace(&tr, org, mins, maxs, ground, 0, CONTENTS_SOLID);
	if(tr.startsolid)
		return; // give up, we're in a wall.
	VectorCopy(tr.endpos, ground);

	if(ammotype == AMMO_BULLETS)
	{
		VectorMA(org, -0.25, right, point[0]);
		VectorMA(point[0], -0.25, up, point[0]);
		VectorMA(point[0], -1, forward, point[0]);
		VectorMA(point[0], 0.5, up, point[1]);
		VectorMA(point[1], 2, forward, point[2]);
		VectorMA(point[2], -0.5, up, point[3]);
	}
	lastdraw = -1;
	oldright[0] = 0;
	oldright[1] = 0;
	oldright[2] = 0;
	for(i = 0; i < bullets; i++)
	{
		if(ammotype == AMMO_ROCKETS)
		{
			VectorCopy(forward, ent.axis[0]);
			VectorCopy(right, ent.axis[1]);
			VectorCopy(up, ent.axis[2]);
			VectorCopy(org, ent.lightingOrigin );
			VectorCopy(org, ent.origin );
			CG_AddWeaponWithPowerups(&ent, powerups);
			//trap_R_AddRefEntityToScene(&ent);
		}
		else if(ammotype == AMMO_BULLETS)
		{
			VectorMA(org, 0.25, right, point[4]);		
			VectorMA(point[4], -0.25, up, point[4]);
			VectorMA(point[4], -1, forward, point[4]);
			VectorMA(point[4], 0.5, up, point[5]);
			VectorMA(point[5], 2, forward, point[6]);
			VectorMA(point[6], -0.5, up, point[7]);
			err = 1 - DotProduct(oldright, right);
			skip = (i - lastdraw);
			if((err >= a_err && min_skip <= skip) || i == (bullets - 1))
			{
				
				vert[0].st[0] = 0;
				vert[0].st[1] = 0;
				vert[1].st[0] = skip;
				vert[1].st[1] = 0;
				vert[2].st[0] = skip;
				vert[2].st[1] = 1;
				vert[3].st[0] = 0;
				vert[3].st[1] = 1;
			
				vert[0].xyz[0] = point[3][0];
				vert[0].xyz[1] = point[3][1];
				vert[0].xyz[2] = point[3][2];
				vert[1].xyz[0] = point[7][0];
				vert[1].xyz[1] = point[7][1];
				vert[1].xyz[2] = point[7][2];
				vert[2].xyz[0] = point[4][0];
				vert[2].xyz[1] = point[4][1];
				vert[2].xyz[2] = point[4][2];
				vert[3].xyz[0] = point[0][0];
				vert[3].xyz[1] = point[0][1];
				vert[3].xyz[2] = point[0][2];
				trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[1], 4, vert );	

				vert[0].xyz[0] = point[6][0];
				vert[0].xyz[1] = point[6][1];
				vert[0].xyz[2] = point[6][2];
				vert[1].xyz[0] = point[2][0];
				vert[1].xyz[1] = point[2][1];
				vert[1].xyz[2] = point[2][2];
				vert[2].xyz[0] = point[1][0];
				vert[2].xyz[1] = point[1][1];
				vert[2].xyz[2] = point[1][2];
				vert[3].xyz[0] = point[5][0];
				vert[3].xyz[1] = point[5][1];
				vert[3].xyz[2] = point[5][2];
			
				trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[1], 4, vert );
				
				vert[0].st[0] = 0;
				vert[0].st[1] = 0;
				vert[1].st[0] = 0;
				vert[1].st[1] = 1;
				vert[2].st[0] = skip;
				vert[2].st[1] = 1;
				vert[3].st[0] = skip;
				vert[3].st[1] = 0;
				
				vert[0].xyz[0] = point[4][0];
				vert[0].xyz[1] = point[4][1];
				vert[0].xyz[2] = point[4][2];
				vert[1].xyz[0] = point[5][0];
				vert[1].xyz[1] = point[5][1];
				vert[1].xyz[2] = point[5][2];
				vert[2].xyz[0] = point[1][0];
				vert[2].xyz[1] = point[1][1];
				vert[2].xyz[2] = point[1][2];
				vert[3].xyz[0] = point[0][0];
				vert[3].xyz[1] = point[0][1];
				vert[3].xyz[2] = point[0][2];
				trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[2], 4, vert );
			
				vert[0].xyz[0] = point[3][0];
				vert[0].xyz[1] = point[3][1];
				vert[0].xyz[2] = point[3][2];
				vert[1].xyz[0] = point[2][0];
				vert[1].xyz[1] = point[2][1];	
				vert[1].xyz[2] = point[2][2];
				vert[2].xyz[0] = point[6][0];
				vert[2].xyz[1] = point[6][1];
				vert[2].xyz[2] = point[6][2];
				vert[3].xyz[0] = point[7][0];
				vert[3].xyz[1] = point[7][1];
				vert[3].xyz[2] = point[7][2];
				trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[0], 4, vert );

				VectorCopy(point[4], point[0]);
				VectorCopy(point[5], point[1]);
				VectorCopy(point[6], point[2]);
				VectorCopy(point[7], point[3]);
				lastdraw = i;
				VectorCopy(right, oldright);		
			}
		}
		if(ground[2] > (org[2] + 12 * size * min_skip * right[2]))
		{ // near the ground - make it start twisting in the direction of our movement
			right[0] -= vel[0] * 0.1;
			right[1] -= vel[1] * 0.1;
			forward[2] *= 0.1;
		}
		else
		{ // make it twist downward
			right[2] -= 0.1;	
		}
		VectorNormalize(right);
		VectorMA(org, size, right, neworg);
		if (cg_realammobelts.integer && (fabs(neworg[0] - ground[0]) > size || fabs(neworg[1] - ground[1]) > size))
		{ // hit the ground, make sure we don't go down below ground level
			int checks = 0;
			ground[0] = neworg[0];
			ground[1] = neworg[1];
			ground[2] = neworg[2] - 128;
			CG_Trace(&tr, org, mins, maxs, ground, 0, CONTENTS_SOLID);
			while(tr.startsolid)
			{
				org[2] += size;
				CG_Trace(&tr, org, mins, maxs, ground, 0, CONTENTS_SOLID);
				i++;
				if(tr.startsolid && ammotype == AMMO_BULLETS && i >= bullets)//checks > (bullets - i))
				{ // finish up the last draw
					skip = (i - lastdraw);
					vert[0].st[0] = 0;
					vert[0].st[1] = 0;
					vert[1].st[0] = skip;
					vert[1].st[1] = 0;
					vert[2].st[0] = skip;
					vert[2].st[1] = 1;
					vert[3].st[0] = 0;
					vert[3].st[1] = 1;
				
					vert[0].xyz[0] = point[3][0];
					vert[0].xyz[1] = point[3][1];
					vert[0].xyz[2] = point[3][2];
					vert[1].xyz[0] = point[7][0];
					vert[1].xyz[1] = point[7][1];
					vert[1].xyz[2] = point[7][2];
					vert[2].xyz[0] = point[4][0];
					vert[2].xyz[1] = point[4][1];
					vert[2].xyz[2] = point[4][2];
					vert[3].xyz[0] = point[0][0];
					vert[3].xyz[1] = point[0][1];
					vert[3].xyz[2] = point[0][2];
					trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[1], 4, vert );	

					vert[0].xyz[0] = point[6][0];
					vert[0].xyz[1] = point[6][1];
					vert[0].xyz[2] = point[6][2];
					vert[1].xyz[0] = point[2][0];
					vert[1].xyz[1] = point[2][1];
					vert[1].xyz[2] = point[2][2];
					vert[2].xyz[0] = point[1][0];
					vert[2].xyz[1] = point[1][1];
					vert[2].xyz[2] = point[1][2];
					vert[3].xyz[0] = point[5][0];
					vert[3].xyz[1] = point[5][1];
					vert[3].xyz[2] = point[5][2];
				
					trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[1], 4, vert );
				
					vert[0].st[0] = 0;
					vert[0].st[1] = 0;
					vert[1].st[0] = 0;
					vert[1].st[1] = 1;
					vert[2].st[0] = skip;
					vert[2].st[1] = 1;
					vert[3].st[0] = skip;
					vert[3].st[1] = 0;
					
					vert[0].xyz[0] = point[4][0];
					vert[0].xyz[1] = point[4][1];
					vert[0].xyz[2] = point[4][2];
					vert[1].xyz[0] = point[5][0];
					vert[1].xyz[1] = point[5][1];
					vert[1].xyz[2] = point[5][2];
					vert[2].xyz[0] = point[1][0];
					vert[2].xyz[1] = point[1][1];
					vert[2].xyz[2] = point[1][2];
					vert[3].xyz[0] = point[0][0];
					vert[3].xyz[1] = point[0][1];
					vert[3].xyz[2] = point[0][2];
					trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[2], 4, vert );
			
					vert[0].xyz[0] = point[3][0];
					vert[0].xyz[1] = point[3][1];
					vert[0].xyz[2] = point[3][2];
					vert[1].xyz[0] = point[2][0];
					vert[1].xyz[1] = point[2][1];	
					vert[1].xyz[2] = point[2][2];
					vert[2].xyz[0] = point[6][0];
					vert[2].xyz[1] = point[6][1];
					vert[2].xyz[2] = point[6][2];
					vert[3].xyz[0] = point[7][0];
					vert[3].xyz[1] = point[7][1];
					vert[3].xyz[2] = point[7][2];
					trap_R_AddPolyToScene( cgs.media.machinegunBrassShader[0], 4, vert );

					VectorCopy(point[4], point[0]);
					VectorCopy(point[5], point[1]);
					VectorCopy(point[6], point[2]);
					VectorCopy(point[7], point[3]);
					lastdraw = i;
					VectorCopy(right, oldright);		
					return; // give up, we're in a wall.
				}
				else if(i >= bullets)
					return; // we've been at this way too long
				neworg[2] = tr.endpos[2];
			}
			VectorCopy(tr.endpos, ground);			
		}
		ProjectPointOnPlane(forward, forward, right);
		VectorNormalize(forward);
		CrossProduct(forward, right, up);

		while(cg_realammobelts.integer && (neworg[2] + size * right[2] - fabs(forward[2]) - size * fabs(up[2])) < ground[2])
		{
			if(right[0] || right[1])
			{
				right[2] += 0.1;
			}
			else
			{				
				right[0] = (up[0] + vel[0]) * 0.01;
				right[1] = (up[1] + vel[1]) * 0.01;
			}
			forward[2] = 0;
			
			VectorNormalize(right);
			VectorMA(org, size, right, neworg);
			ProjectPointOnPlane(forward, forward, right);
			VectorNormalize(forward);
			CrossProduct(forward, right, up);
		}
		VectorCopy(neworg, org);
		
	}	
	return;
}

static void CG_MachineGunEjectBrass( centity_t *cent ) {
	localEntity_t	*le;
	refEntity_t		*re;
	vec3_t			velocity, xvelocity;
	vec3_t			offset, xoffset;
	float			waterScale = 1.0f;
	vec3_t			v[3];

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	velocity[0] = 25 * crandom() + cent->currentState.pos.trDelta[0];
	velocity[1] = -50 + 40 * crandom() + cent->currentState.pos.trDelta[1];
	velocity[2] = 100 + 50 * crandom() + cent->currentState.pos.trDelta[2];
	if(bg_itemlist[cent->currentState.weapon].giTag == WP_MACHINEGUN)
		velocity[1] = -velocity[1];
	le->leType = LE_FRAGMENT;
	le->startTime = cg.time;
	le->endTime = cg.time + 100000000;
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time - (rand()&15);

	AnglesToAxis( cent->lerpAngles, v );

	
	offset[0] = 8;
	offset[1] = -4;
	offset[2] = 24;

	xoffset[0] = offset[0] * v[0][0] + offset[1] * v[1][0] + offset[2] * v[2][0];
	xoffset[1] = offset[0] * v[0][1] + offset[1] * v[1][1] + offset[2] * v[2][1];
	xoffset[2] = offset[0] * v[0][2] + offset[1] * v[1][2] + offset[2] * v[2][2];
	if(cent->pe.railgunOrigin[0] || cent->pe.railgunOrigin[1] || cent->pe.railgunOrigin[2]  )
	{
		VectorMA( cent->pe.railgunOrigin, -16, v[0], re->origin );		
	}
	else
	{
		VectorAdd( cent->lerpOrigin, xoffset, re->origin );
	}

	VectorCopy( re->origin, le->pos.trBase );

	if ( CG_PointContents( re->origin, -1 ) & CONTENTS_WATER ) {
		waterScale = 0.10;
	}

	xvelocity[0] = velocity[0] * v[0][0] + velocity[1] * v[1][0] + velocity[2] * v[2][0];
	xvelocity[1] = velocity[0] * v[0][1] + velocity[1] * v[1][1] + velocity[2] * v[2][1];
	xvelocity[2] = velocity[0] * v[0][2] + velocity[1] * v[1][2] + velocity[2] * v[2][2];
	VectorScale( xvelocity, waterScale, le->pos.trDelta );

	AxisCopy( axisDefault, re->axis );
	re->hModel = cgs.media.machinegunBrassModel;

	le->bounceFactor = 0.4 * waterScale;

	le->angles.trType = TR_LINEAR;
	le->angles.trTime = cg.time;
	le->angles.trBase[0] = rand()&31;
	le->angles.trBase[1] = rand()&31;
	le->angles.trBase[2] = rand()&31;
	le->angles.trDelta[0] = 2;
	le->angles.trDelta[1] = 1;
	le->angles.trDelta[2] = 0;

	le->leFlags = LEF_TUMBLE;
	le->leBounceSoundType = LEBS_BRASS;
	le->leMarkType = LEMT_NONE;
}

/*
==========================
CG_ShotgunEjectBrass
==========================
*/
static void CG_ShotgunEjectBrass( centity_t *cent ) {
	localEntity_t	*le;
	refEntity_t		*re;
	vec3_t			velocity, xvelocity;
	vec3_t			offset, xoffset;
	vec3_t			v[3];
	int				i, shells;

	if(bg_itemlist[cent->currentState.weapon].giTag == WP_SHOTGUN)
		shells = 2;
	else
		shells = 1;

	for ( i = 0; i < shells; i++ ) {
		float	waterScale = 1.0f;

		le = CG_AllocLocalEntity();
		re = &le->refEntity;

		velocity[0] = 60 + 60 * crandom() + cent->currentState.pos.trDelta[0];
		if ( i == 0 ) {
			velocity[1] = 40 + 10 * crandom() + cent->currentState.pos.trDelta[1];
		} else {
			velocity[1] = -40 + 10 * crandom() + cent->currentState.pos.trDelta[1];
		}
		velocity[2] = 100 + 50 * crandom() + cent->currentState.pos.trDelta[2];

		le->leType = LE_FRAGMENT;
		le->startTime = cg.time;
		le->endTime = cg.time + 100000000;
	
		le->pos.trType = TR_GRAVITY;
		le->pos.trTime = cg.time;

		AnglesToAxis( cent->lerpAngles, v );

		
		offset[0] = 8;
		offset[1] = 0;
		offset[2] = 24;

		xoffset[0] = offset[0] * v[0][0] + offset[1] * v[1][0] + offset[2] * v[2][0];
		xoffset[1] = offset[0] * v[0][1] + offset[1] * v[1][1] + offset[2] * v[2][1];
		xoffset[2] = offset[0] * v[0][2] + offset[1] * v[1][2] + offset[2] * v[2][2];
		if(cent->pe.railgunOrigin[0] || cent->pe.railgunOrigin[1] || cent->pe.railgunOrigin[2]  )
		{
			VectorMA( cent->pe.railgunOrigin, -16, v[0], re->origin );		
		}
		else
		{
	
			VectorAdd( cent->lerpOrigin, xoffset, re->origin );
		}
		
		VectorCopy( re->origin, le->pos.trBase );
		if ( CG_PointContents( re->origin, -1 ) & CONTENTS_WATER ) {
			waterScale = 0.10;
		}

		xvelocity[0] = velocity[0] * v[0][0] + velocity[1] * v[1][0] + velocity[2] * v[2][0];
		xvelocity[1] = velocity[0] * v[0][1] + velocity[1] * v[1][1] + velocity[2] * v[2][1];
		xvelocity[2] = velocity[0] * v[0][2] + velocity[1] * v[1][2] + velocity[2] * v[2][2];
		VectorScale( xvelocity, waterScale, le->pos.trDelta );

		AxisCopy( axisDefault, re->axis );
		re->hModel = cgs.media.shotgunBrassModel;
		le->bounceFactor = 0.3;

		le->angles.trType = TR_LINEAR;
		le->angles.trTime = cg.time;
		le->angles.trBase[0] = rand()&31;
		le->angles.trBase[1] = rand()&31;
		le->angles.trBase[2] = rand()&31;
		le->angles.trDelta[0] = 1;
		le->angles.trDelta[1] = 0.5;
		le->angles.trDelta[2] = 0;

		le->leFlags = LEF_TUMBLE;
		le->leBounceSoundType = LEBS_BRASS;
		le->leMarkType = LEMT_NONE;
	}
}

static void CG_GrenadeEjectShrapnel( vec3_t org, vec3_t dir ) {
	localEntity_t	*le;
	refEntity_t		*re;

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	VectorCopy( org, le->pos.trBase );
	VectorScale(dir, 800, le->pos.trDelta);
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time;
	le->leType = LE_SHRAPNEL;
	le->startTime = cg.time;
	le->endTime = cg.time + random() * 5000;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	le->radius = 1;
	le->leFlags = cgs.media.tracerShader;
	le->bounceFactor = 1;
}

static void CG_RocketEjectShrapnel( vec3_t org, vec3_t dir ) {
	localEntity_t	*le;
	refEntity_t		*re;

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	VectorCopy( org, le->pos.trBase );
	VectorScale(dir, random() * 400 + 200, le->pos.trDelta);
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time;
	le->leType = LE_SHRAPNEL;
	le->startTime = cg.time;
	le->endTime = cg.time + random() * 5000;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	le->radius = 1;
	le->leFlags = cgs.media.tracerShader;
	le->bounceFactor = 1;
}

static void CG_GrenadeEjectJet( vec3_t org, vec3_t dir ) {
	localEntity_t	*le;
	refEntity_t		*re;

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	VectorCopy( org, le->pos.trBase );
	VectorScale(dir, (rand() & 255) + 400, le->pos.trDelta);
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time;
	le->leType = LE_SHRAPNEL;
	le->startTime = cg.time;
	le->endTime = cg.time + random() * 250 + 500;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	le->leFlags = cgs.media.flamethrowerShader; 
	le->radius = (rand() & 7) + 2;
	le->bounceFactor = 1;
}

/*
==========================
CG_RailTrail
==========================
*/
void CG_RailTrail( clientInfo_t *ci, vec3_t start, vec3_t end ) {
	localEntity_t	*le;
	refEntity_t		*re;

	vec3_t n0, n1, n2;
	float dis, rlength, rtime;
	int i, r;
	
	VectorSubtract(end, start, n0);
	dis = VectorNormalize(n0);
	rlength = 32;
	rtime = 1600;
	
	PerpendicularVector(n1, n0);
	CrossProduct(n0, n1, n2);
	
	for(r = 0; r < dis / rlength; r++)
	{
		le = CG_AllocLocalEntity();
		re = &le->refEntity;
	
		le->leType = LE_EXPAND_FADE;
		le->startTime = cg.time;
		le->endTime = cg.time + rtime;
		le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	
		re->shaderTime = cg.time / 1000.0f;
		VectorMA( start, r * rlength, n0, re->origin );
	
		re->reType = RT_MODEL;
		re->renderfx |= RF_NOSHADOW;
		re->hModel = cgs.media.teleportEffectModel;
		if(r == (int) dis / rlength)
			re->customShader = cgs.media.railEndShader;//trap_R_RegisterShader("railRingEnd");
		else if(r == 0)
			re->customShader = cgs.media.railStartShader;//trap_R_RegisterShader("railRingStart");
		else
			re->customShader = cgs.media.railRingsShader;
		
		for(i = 0; i < 3; i++)
		{
			re->axis[0][i] = n2[i];
			re->axis[1][i] = n1[i];
			re->axis[2][i] = n0[i] * rlength / 62.125;
			if(r == (int) dis / rlength)
				re->axis[2][i] *= (((int) dis) % ((int) rlength)) / rlength;
			
		}

		le->color[0] = 1;
		le->color[1] = 1;
		le->color[2] = 1;
		le->color[3] = 1;
		le->refEntity.shaderRGBA[3] = 1;
	}
	if(ci->powerups & (1 << PW_QUAD))
	{ // add quad ring
		le = CG_AllocLocalEntity();
		re = &le->refEntity;
	
		le->leType = LE_EXPAND_FADE;
		le->startTime = cg.time;
		le->endTime = cg.time + rtime;
		le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	
		re->shaderTime = cg.time / 1000.0f;
		VectorCopy( start, re->origin );
	
		re->reType = RT_MODEL;
		re->renderfx |= RF_NOSHADOW;
		re->hModel = cgs.media.teleportEffectModel;
		re->customShader = cgs.media.quadShader;
			
		for(i = 0; i < 3; i++)
		{
			re->axis[0][i] = n2[i];
			re->axis[1][i] = n1[i];
			re->axis[2][i] = n0[i] * dis / 62.125;
			
		}
	
		le->color[0] = 1;
		le->color[1] = 1;
		le->color[2] = 1;
		le->color[3] = 1;
		le->refEntity.shaderRGBA[3] = 0.15f;
	}
}

/*
==========================
CG_RocketTrail
==========================
*/
static void CG_RocketTrail( centity_t *ent, const weaponInfo_t *wi ) {
	int		step;
	vec3_t	origin, lastPos, lastPos2;
	int		t;
	int		startTime, contents;
	int		lastContents;
	entityState_t	*es;
	vec3_t	up;
	localEntity_t	*smoke;

	up[0] = 0;
	up[1] = 0;
	up[2] = 0;

	step = 50;

	es = &ent->currentState;
	startTime = ent->trailTime;
	t = step * ( (startTime + step) / step );

	BG_EvaluateTrajectory( &es->pos, cg.time, origin );
	contents = CG_PointContents( origin, -1 );

	// if object (e.g. grenade) is stationary, don't toss up smoke
	if ( es->pos.trType == TR_STATIONARY ) {
		ent->trailTime = cg.time;
		return;
	}

	BG_EvaluateTrajectory( &es->pos, ent->trailTime, lastPos );
	lastContents = CG_PointContents( lastPos, -1 );

	ent->trailTime = cg.time;

	if ( contents & ( CONTENTS_WATER | CONTENTS_SLIME | CONTENTS_LAVA ) ) {
		if ( contents & lastContents & CONTENTS_WATER ) {
			CG_BubbleTrail( lastPos, origin, 8 );
		}
		return;
	}
	for ( ; t <= ent->trailTime ; t += step ) {
		VectorCopy(lastPos, lastPos2);
		BG_EvaluateTrajectory( &es->pos, t, lastPos );
		
		if(ent->currentState.powerups & (1 << PW_QUAD) )
		{
			smoke = CG_SmokePuff( lastPos, up, 
					wi->trailRadius, 
					1, 1, 1, 0.33,
					wi->wiTrailTime, 
					t,
					0, 
					cgs.media.QuadSmokePuffShader );

			smoke->refEntity.renderfx |= RF_MINLIGHT;
					
		}
		else
		{
			smoke = CG_SmokePuff( lastPos, up, 
					wi->trailRadius, 
					1, 1, 1, 0.33,
					wi->wiTrailTime, 
					t,
					0, 
					cgs.media.smokePuffShader );
			//smoke->leFlags = LEF_AUTOSPRITE2;
			//VectorCopy(smoke->refEntity.oldorigin, lastPos2);
			//VectorSubtract(smoke->refEntity.oldorigin, smoke->refEntity.origin, smoke->refEntity.axis[0]);
			
		}
		// use the optimized local entity add
		smoke->leType = LE_SCALE_FADE;
	}

}


/*
==========================
CG_GrenadeTrail
==========================
*/
static void CG_GrenadeTrail( centity_t *ent, const weaponInfo_t *wi ) {
	CG_RocketTrail( ent, wi );
}


/*
=================
CG_RegisterWeapon

The server says this item is used on this level
=================
*/
void CG_RegisterWeapon( int weaponNum ) {
	weaponInfo_t	*weaponInfo;
	gitem_t			*item, *ammo;
	char			path[MAX_QPATH];
	vec3_t			mins, maxs;
	int				i;

	weaponInfo = &cg_weapons[weaponNum];

	if ( weaponNum <= 0 ) {
		return;
	}

	if ( weaponInfo->registered ) {
		return;
	}

	memset( weaponInfo, 0, sizeof( *weaponInfo ) );
	weaponInfo->registered = qtrue;

	for ( item = bg_itemlist + 1 ; item->classname ; item++ ) {
		if ( item->giType == IT_WEAPON && item->giTag == weaponNum ) {
			weaponInfo->item = item;
			break;
		}
	}
	if ( !item->classname ) {
		CG_Error( "Couldn't find weapon %i", weaponNum );
	}
	CG_RegisterItemVisuals( item - bg_itemlist );

	// load cmodel before model so filecache works

	weaponInfo->weaponIcon = trap_R_RegisterShader( item->icon );
	weaponInfo->ammoIcon = trap_R_RegisterShader( item->icon );

	for ( ammo = bg_itemlist + 1 ; ammo->classname ; ammo++ ) {
		if ( ammo->giType == IT_AMMO && ammo->giTag == weaponNum ) {
			break;
		}
	}
	if ( ammo->classname && ammo->world_model[0] ) {		
		weaponInfo->ammoModel = trap_R_RegisterModel( ammo->world_model[0] );
	}
	if(item->world_model[0])
	{
		weaponInfo->weaponModel = trap_R_RegisterModel( item->world_model[0] );
		weaponInfo->weaponSkin = 0;
		// calc midpoint for rotation
		trap_R_ModelBounds( weaponInfo->weaponModel, mins, maxs );
		for ( i = 0 ; i < 3 ; i++ ) {
			weaponInfo->weaponMidpoint[i] = mins[i] + 0.5 * ( maxs[i] - mins[i] );
		}
		strcpy( path, item->world_model[0] );
		COM_StripExtension( path, path );
		strcat( path, "_flash.md3" );
		weaponInfo->flashModel = trap_R_RegisterModel( path );
		if ( weaponNum == WP_CHAINGUN || weaponNum == WP_GAUNTLET ) {
			strcpy( path, item->world_model[0] );
			COM_StripExtension( path, path );
			strcat( path, "_barrel.md3" );
			weaponInfo->barrelModel = trap_R_RegisterModel( path );
		}
		else if(weaponNum == WP_BALL)
		{
			weaponInfo->barrelModel = trap_R_RegisterModel( "models/powerups/health/small_sphere.md3" );
		}

		strcpy( path, item->world_model[0] );
		COM_StripExtension( path, path );
	}

	switch ( weaponNum ) {
	case WP_SWORD:
	case WP_REDFLAG:
	case WP_BLUEFLAG:
		weaponInfo->firingSound = cgs.media.selectSound;
		weaponInfo->flashSound[0] = cgs.media.selectSound;
		break;

	
	case WP_GAUNTLET:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/melee/fstrun.wav" );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/melee/fstatck.wav" );
		break;

	case WP_FLAMETHROWER:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->barrelModel = trap_R_RegisterModel( "models/ammo/grenade1.md3" );
		weaponInfo->flashModel = trap_R_RegisterModel( "models/weapons2/grenadel/grenadel_flash.md3" );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/rocket/rockfly.wav" );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/items/s_health.wav" );		
		cgs.media.flamethrowerExplosionModel = trap_R_RegisterModel( "models/weaphits/crackle.md3" );

		break;

	
	case WP_LIGHTNING:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->readySound = trap_S_RegisterSound( "sound/weapons/melee/fsthum.wav" );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/lightning/lg_hum.wav" );
		
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/lightning/lg_fire.wav" );
		cgs.media.lightningShader = trap_R_RegisterShader( "lightningBolt" );
		cgs.media.lightningExplosionModel = trap_R_RegisterModel( "models/weaphits/crackle.md3" );
		cgs.media.sfx_lghit1 = trap_S_RegisterSound( "sound/weapons/lightning/lg_hit.wav" );
		cgs.media.sfx_lghit2 = trap_S_RegisterSound( "sound/weapons/lightning/lg_hit2.wav" );
		cgs.media.sfx_lghit3 = trap_S_RegisterSound( "sound/weapons/lightning/lg_hit3.wav" );

		break;

	case WP_MACHINEGUN:
	case WP_CHAINGUN:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf1b.wav" );
		weaponInfo->flashSound[1] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf2b.wav" );
		weaponInfo->flashSound[2] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf3b.wav" );
		weaponInfo->flashSound[3] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf4b.wav" );
		weaponInfo->ejectBrassFunc = CG_MachineGunEjectBrass;
		cgs.media.bulletExplosionShader = trap_R_RegisterShader( "bulletExplosion" );
		break;

	case WP_SHOTGUN:
	case WP_SINGLE_SHOTGUN:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/shotgun/sshotf1b.wav" );
		weaponInfo->ejectBrassFunc = CG_ShotgunEjectBrass;
		break;

	case WP_ROCKET_LAUNCHER:
		weaponInfo->missileModel = trap_R_RegisterModel( "models/ammo/rocket/rocket.md3" );
		weaponInfo->missileSound = trap_S_RegisterSound( "sound/weapons/rocket/rockfly.wav" );
		weaponInfo->missileTrailFunc = CG_RocketTrail;
		weaponInfo->missileDlight = 200;
		weaponInfo->wiTrailTime = 2000;
		weaponInfo->trailRadius = 64;
		MAKERGB( weaponInfo->missileDlightColor, 1, 0.75, 0 );
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.75, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/rocket/rocklf1a.wav" );
		cgs.media.rocketExplosionShader = trap_R_RegisterShader( "rocketExplosion3D" );
		cgs.media.smokeExplosionShader = trap_R_RegisterShader( "smokeExplosion3D" );
		break;

	case WP_GRENADE_LAUNCHER:
		weaponInfo->missileModel = trap_R_RegisterModel( "models/ammo/grenade2.md3" );
		weaponInfo->missileTrailFunc = CG_GrenadeTrail;
		weaponInfo->wiTrailTime = 700;
		weaponInfo->trailRadius = 32;
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.7, 0.5 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/grenade/grenlf1a.wav" );
		cgs.media.grenadeExplosionShader = trap_R_RegisterShader( "grenadeExplosion3D" );
		cgs.media.smokeExplosionShader = trap_R_RegisterShader( "smokeExplosion3D" );
		break;

	case WP_PLASMAGUN:
		weaponInfo->missileSound = trap_S_RegisterSound( "sound/weapons/plasma/lasfly.wav" );
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/plasma/hyprbf1a.wav" );
		cgs.media.plasmaExplosionShader = trap_R_RegisterShader( "plasmaExplosion" );
		weaponInfo->weaponSkin = trap_R_RegisterSkin("models/weapons2/plasma/plasma.skin");
		break;

	case WP_RAILGUN:
		weaponInfo->readySound = trap_S_RegisterSound( "sound/weapons/railgun/rg_hum.wav" );
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.5, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/railgun/railgf1a.wav" );
		cgs.media.railExplosionShader = trap_R_RegisterShader( "railExplosion" );
		cgs.media.railRingsShader = trap_R_RegisterShader( "railRing" );		
		cgs.media.railStartShader = trap_R_RegisterShader( "railRingStart" );
		cgs.media.railEndShader = trap_R_RegisterShader( "railRingEnd" );

		break;

	case WP_BALL:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/change.wav" );
		break;
		
	default:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/rocket/rocklf1a.wav" );
		break;
	}
}

/*
=================
CG_RegisterItemVisuals

The server says this item is used on this level
=================
*/
void CG_RegisterItemVisuals( int itemNum ) {
	itemInfo_t		*itemInfo;
	gitem_t			*item;

	itemInfo = &cg_items[ itemNum ];
	if ( itemInfo->registered ) {
		return;
	}

	item = &bg_itemlist[ itemNum ];

	memset( itemInfo, 0, sizeof( &itemInfo ) );
	itemInfo->registered = qtrue;

	itemInfo->models[0] = trap_R_RegisterModel( item->world_model[0] );

	itemInfo->icon = trap_R_RegisterShader( item->icon );

	if ( item->giType == IT_WEAPON ) {
		CG_RegisterWeapon( item->giTag );
	}

	//
	// powerups have an accompanying ring or sphere
	//
	if ( item->giType == IT_POWERUP || item->giType == IT_HEALTH || 
		item->giType == IT_ARMOR || item->giType == IT_HOLDABLE ) {
		if ( item->world_model[1] ) {
			itemInfo->models[1] = trap_R_RegisterModel( item->world_model[1] );
		}
	}
}


/*
========================================================================================

VIEW WEAPON

========================================================================================
*/

/*
=================
CG_MapTorsoToWeaponFrame

=================
*/
static int CG_MapTorsoToWeaponFrame( clientInfo_t *ci, int frame ) {

	// change weapon
	if ( frame >= ci->animations[TORSO_DROP].firstFrame 
		&& frame < ci->animations[TORSO_DROP].firstFrame + 9 ) {
		return frame - ci->animations[TORSO_DROP].firstFrame + 6;
	}

	// stand attack
	if ( frame >= ci->animations[TORSO_ATTACK].firstFrame 
		&& frame < ci->animations[TORSO_ATTACK].firstFrame + 6 ) {
		return 1 + frame - ci->animations[TORSO_ATTACK].firstFrame;
	}

	// stand attack 2
	if ( frame >= ci->animations[TORSO_ATTACK2].firstFrame 
		&& frame < ci->animations[TORSO_ATTACK2].firstFrame + 6 ) {
		return 1 + frame - ci->animations[TORSO_ATTACK].firstFrame;
	}
	
	return 0;
}


/*
==============
CG_CalculateWeaponPosition
==============
*/
static void CG_CalculateWeaponPosition( vec3_t origin, vec3_t angles ) {
	float	scale;
	int		delta;
	float	fracsin;

	VectorCopy( cg.refdef.vieworg, origin );
	VectorCopy( cg.refdefViewAngles, angles );

	// on odd legs, invert some angles
	if ( cg.bobcycle & 1 ) {
		scale = -cg.xyspeed;
	} else {
		scale = cg.xyspeed;
	}

	// gun angles from bobbing
	angles[ROLL] += scale * cg.bobfracsin * 0.005;
	angles[YAW] += scale * cg.bobfracsin * 0.01;
	angles[PITCH] += cg.xyspeed * cg.bobfracsin * 0.005;

	// drop the weapon when landing
	delta = cg.time - cg.landTime;
	if ( delta < LAND_DEFLECT_TIME ) {
		origin[2] += cg.landChange*0.25 * delta / LAND_DEFLECT_TIME;
	} else if ( delta < LAND_DEFLECT_TIME + LAND_RETURN_TIME ) {
		origin[2] += cg.landChange*0.25 * 
			(LAND_DEFLECT_TIME + LAND_RETURN_TIME - delta) / LAND_RETURN_TIME;
	}

#if 0
	// drop the weapon when stair climbing
	delta = cg.time - cg.stepTime;
	if ( delta < STEP_TIME/2 ) {
		origin[2] -= cg.stepChange*0.25 * delta / (STEP_TIME/2);
	} else if ( delta < STEP_TIME ) {
		origin[2] -= cg.stepChange*0.25 * (STEP_TIME - delta) / (STEP_TIME/2);
	}
#endif

	// idle drift
	scale = cg.xyspeed + 40;
	fracsin = sin( cg.time * 0.001 );
	angles[ROLL] += scale * fracsin * 0.01;
	angles[YAW] += scale * fracsin * 0.01;
	angles[PITCH] += scale * fracsin * 0.01;
}


/*
===============
CG_LightningBolt

Origin will be the exact tag point, which is slightly
different than the muzzle point used for determining hits.
The cent should be the non-predicted cent if it is from the player,
so the endpoint will reflect the simulated strike (lagging the predicted
angle)
===============
*/
static void CG_LightningBolt( centity_t *cent, vec3_t origin ) {
	trace_t		trace;
	refEntity_t		beam;
	vec3_t			forward;
	vec3_t			muzzlePoint, endPoint;

	if ( bg_itemlist[cent->currentState.weapon].giTag != WP_LIGHTNING ) {
		return;
	}

	memset( &beam, 0, sizeof( beam ) );

	// find muzzle point for this frame
	VectorCopy( cent->lerpOrigin, muzzlePoint );
	AngleVectors( cent->lerpAngles, forward, NULL, NULL );
	// FIXME: crouch
	muzzlePoint[2] += DEFAULT_VIEWHEIGHT;

	VectorMA( muzzlePoint, 14, forward, muzzlePoint );

	// project forward by the lightning range
	VectorMA( muzzlePoint, LIGHTNING_RANGE, forward, endPoint );

	// see if it hit a wall
	CG_Trace( &trace, muzzlePoint, vec3_origin, vec3_origin, endPoint, 
		cent->currentState.number, MASK_SHOT );

	// this is the endpoint
	VectorCopy( trace.endpos, beam.oldorigin );

	// use the provided origin, even though it may be slightly
	// different than the muzzle origin
	VectorCopy( origin, beam.origin );

	beam.reType = RT_LIGHTNING;
	beam.customShader = cgs.media.lightningShader;
	trap_R_AddRefEntityToScene( &beam );

	// add the impact flare if it hit something
	if ( trace.fraction < 1.0 ) {
		vec3_t	angles;
		vec3_t	dir;

		VectorSubtract( beam.oldorigin, beam.origin, dir );
		VectorNormalize( dir );

		memset( &beam, 0, sizeof( beam ) );
		beam.hModel = cgs.media.lightningExplosionModel;

		VectorMA( trace.endpos, -16, dir, beam.origin );

		// make a random orientation
		angles[0] = rand() % 360;
		angles[1] = rand() % 360;
		angles[2] = rand() % 360;
		AnglesToAxis( angles, beam.axis );
		VectorScale(beam.axis[0], 0.2, beam.axis[0]);
		VectorScale(beam.axis[1], 0.2, beam.axis[1]);
		VectorScale(beam.axis[2], 0.2, beam.axis[2]);
		trap_R_AddRefEntityToScene( &beam );
	}
}

/*
===============
CG_LightningBolt

Origin will be the exact tag point, which is slightly
different than the muzzle point used for determining hits.
The cent should be the non-predicted cent if it is from the player,
so the endpoint will reflect the simulated strike (lagging the predicted
angle)
===============
*/
static void CG_FlameBolt( centity_t *cent, vec3_t origin ) {
	trace_t		trace;
	refEntity_t		beam;
	vec3_t			forward, mins, maxs;
	vec3_t			muzzlePoint, endPoint;

	if ( bg_itemlist[cent->currentState.weapon].giTag != WP_FLAMETHROWER ) {
		return;
	}

	memset( &beam, 0, sizeof( beam ) );

	// find muzzle point for this frame
	VectorCopy( cent->lerpOrigin, muzzlePoint );
	AngleVectors( cent->lerpAngles, forward, NULL, NULL );
	// FIXME: crouch
	muzzlePoint[2] += DEFAULT_VIEWHEIGHT;

	VectorMA( muzzlePoint, 14, forward, muzzlePoint );

	// project forward by the lightning range
	VectorMA( muzzlePoint, FLAMETHROWER_RANGE, forward, endPoint );

	// see if it hit a wall
	mins[0] = -8;
	mins[1] = -8;
	mins[2] = -8;
	maxs[0] = 8;
	maxs[1] = 8;
	maxs[2] = 8;
	CG_Trace( &trace, muzzlePoint, mins, maxs, endPoint, 
		cent->currentState.number, MASK_SHOT | CONTENTS_WATER);

	// this is the endpoint
	VectorCopy( trace.endpos, beam.oldorigin );

	// use the provided origin, even though it may be slightly
	// different than the muzzle origin
	VectorCopy( origin, beam.origin );

	beam.reType = RT_LIGHTNING;
	beam.customShader = cgs.media.flamethrowerShader;
	
	trap_R_AddRefEntityToScene( &beam );

	{
		vec3_t	angles;
		vec3_t	org, dir;
		localEntity_t *le;
		VectorSubtract( beam.oldorigin, beam.origin, dir );
		VectorNormalize( dir );

		memset( &beam, 0, sizeof( beam ) );
		beam.hModel = cgs.media.flamethrowerExplosionModel;
		beam.customShader = cgs.media.flamethrowerShader;
		VectorMA( trace.endpos, -16, dir, beam.origin );

		// make a random orientation
		angles[0] = rand() % 360;
		angles[1] = rand() % 360;
		angles[2] = rand() % 360;
		AnglesToAxis( angles, beam.axis );
		trap_R_AddRefEntityToScene( &beam );
		org[0] = trace.endpos[0] + crandom() * 8;
		org[1] = trace.endpos[1] + crandom() * 8;
		org[2] = trace.endpos[2] + crandom() * 8;

		dir[0] = 0;//forward[0] * 128;
		dir[1] = 0;//forward[1] * 128;
		dir[2] = 0;//forward[2] * 128 + 40;
		if(!(trace.contents & CONTENTS_WATER))
		{
			le = CG_SmokePuff(org, dir, 32, 1, 1, 1, 1, 900, cg.time, 0, cgs.media.flamethrowerShader);
			le->leFlags |= LEF_PUFF_DONT_SCALE;
		}
		//CG_MakeExplosion( trace.endpos, forward, 
		//					   cgs.media.dishFlashModel, cgs.media.flamethrowerShader,
		//					   900, qtrue );		
		
	}
}


/*
===============
CG_SpawnRailTrail

Origin will be the exact tag point, which is slightly
different than the muzzle point used for determining hits.
===============
*/
static void CG_SpawnRailTrail( centity_t *cent, vec3_t origin ) {
	clientInfo_t	*ci;

	if ( bg_itemlist[cent->currentState.weapon].giTag != WP_RAILGUN ) {
		return;
	}
	if ( !cent->pe.railgunFlash ) {
		return;
	}
	cent->pe.railgunFlash = qtrue;
	ci = &cgs.clientinfo[ cent->currentState.clientNum ];
	ci->powerups = cent->currentState.powerups;
	CG_RailTrail( ci, origin, cent->pe.railgunImpact );
}


/*
======================
CG_MachinegunSpinAngle
======================
*/
#define		SPIN_SPEED	0.9
#define		COAST_TIME	1500
static float	CG_MachinegunSpinAngle( centity_t *cent ) {
	int		delta;
	float	angle;
	float	speed;
	if(bg_itemlist[cent->currentState.weapon].giType != IT_WEAPON)
		return 0;
	delta = cg.time - cent->pe.barrelTime;
	if ( cent->pe.barrelSpinning ) 
	{
		angle = cent->pe.barrelAngle + delta * SPIN_SPEED;
		if(bg_itemlist[cent->currentState.weapon].giTag == WP_CHAINGUN && !(cent->currentState.powerups & (1 << PW_QUAD)))
			trap_S_StartSound (cent->pe.railgunOrigin, cent->currentState.number, CHAN_ITEM, cgs.media.machinegunSpinSound );
	}
	else 
	{
		if ( delta > COAST_TIME ) {
			delta = COAST_TIME;
		}

		speed = 0.5 * ( SPIN_SPEED + (float)( COAST_TIME - delta ) / COAST_TIME );
		angle = cent->pe.barrelAngle + delta * speed;
	}

	if ( cent->pe.barrelSpinning == !((cent->currentState.eFlags & EF_FIRING) || (cent->currentState.torsoAnim & ~ANIM_TOGGLEBIT) == TORSO_ATTACK) )//!(cent->currentState.eFlags & EF_FIRING) 
	{
		cent->pe.barrelTime = cg.time;
		cent->pe.barrelAngle = AngleMod( angle );
		cent->pe.barrelSpinning = (cent->currentState.eFlags & EF_FIRING) || ((cent->currentState.torsoAnim & ~ANIM_TOGGLEBIT) == TORSO_ATTACK);//!!(cent->currentState.eFlags & EF_FIRING);		
	}

	return angle;
}


/*
========================
CG_AddWeaponWithPowerups
========================
*/

/*
=============
CG_AddPlayerWeapon

Used for both the view weapon (ps is valid) and the world modelother character models (ps is NULL)
The main player will have this called for BOTH cases, so effects like light and
sound should only be done on the world model case.
=============
*/
void CG_AddPlayerWeapon( refEntity_t *parent, playerState_t *ps, centity_t *cent ) {
	refEntity_t	gun;
	refEntity_t	barrel;
	refEntity_t	flash;
	vec3_t		angles;
	int			itemNum, weaponNum;
	weaponInfo_t	*weapon;
	centity_t	*nonPredictedCent;
	int anim;
	short ammo = 0;
	gitem_t *item;	
	
	itemNum = cent->currentState.weapon;
	ammo = cent->currentState.time2;
	if(ammo == 1023) // only sent as 10 bits
		ammo = -1;
	if(itemNum <= 0)
		return;
	item = &bg_itemlist[itemNum];
	CG_RegisterItemVisuals( itemNum );
	if(bg_itemlist[itemNum].giType == IT_WEAPON)
	{
		weaponNum = item->giTag;
		weapon = &cg_weapons[weaponNum];
	}
	else
	{
		weaponNum = 0;
		weapon = NULL;
	}

	// add the weapon
	memset( &gun, 0, sizeof( gun ) );
	VectorCopy( parent->lightingOrigin, gun.lightingOrigin );
	gun.shadowPlane = parent->shadowPlane;
	gun.renderfx = parent->renderfx;

	// set custom shading for railgun refire rate, or for plasmagun ammo capacity
	if(parent->hModel)
	{
		float	f, r, g, b;

		if(cent->currentState.powerups & (1 << PW_INVIS) )
			ammo = 0; // time2 is invisibility time left

		if(weaponNum == WP_RAILGUN)
		{
			f = (cg.time - cgs.clientinfo[cent->currentState.clientNum].lastfired) / 1500.0;
			if(cent->currentState.powerups & PW_HASTE)
				f *= g_HasteFactor.value;
			r = (ammo > 0);//cgs.clientinfo[cent->currentState.clientNum].color[0];
			g = (ammo > 0);//cgs.clientinfo[cent->currentState.clientNum].color[1];
			b = (ammo > 0);//cgs.clientinfo[cent->currentState.clientNum].color[2];
		
		}
		else if(weaponNum)
		{
			f = ammo * 1.0 / BG_FindItemForWeapon(weaponNum)->quantity;
			if(ammo < 0) 
				f = 0;
			r = 1;
			g = 1;
			b = 1;		
		}
		
		if(f > 1.0)
			f = 1.0;
		else if(f < 0)
			f = 0;
		gun.shaderRGBA[0] = r * 255 * f;
		gun.shaderRGBA[1] = g * 255 * f;
		gun.shaderRGBA[2] = b * 255 * f;
		gun.shaderRGBA[3] = 128 + 127 * f;
		
	}
	else
	{
		gun.shaderRGBA[0] = 0;
		gun.shaderRGBA[1] = 0;
		gun.shaderRGBA[2] = 0;
		gun.shaderRGBA[3] = 128;
	}

		

	if(weapon)
	{
		gun.hModel = weapon->weaponModel;
		if(weapon->weaponSkin)
			gun.customSkin = weapon->weaponSkin;
		if (!gun.hModel) {
			return;
		}
	}
	if ( weaponNum ) {
		// add weapon ready sound
		cent->pe.lightningFiring = qfalse;
		if ( ( cent->currentState.eFlags & EF_FIRING ) && weapon->firingSound ) {
			// lightning gun and guantlet make a different sound when fire is held down
			trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, weapon->firingSound );
			//CG_Printf("Skip?");
			cent->pe.lightningFiring = qtrue;
		} else if ( weapon->readySound ) {
			trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, weapon->readySound );
			
		}
	}
	
	CG_PositionEntityOnTag( &gun, parent, parent->hModel, "tag_weapon");
	if(weaponNum == WP_REDFLAG || weaponNum == WP_BLUEFLAG)
	{ // flags aren't centered right.

		VectorMA(gun.origin, -18, gun.axis[0], gun.origin);
		VectorMA(gun.origin, -2, gun.axis[1], gun.origin);
		VectorMA(gun.origin, -16, gun.axis[2], gun.origin);
	}
	if(!weaponNum)
	{
		VectorMA(gun.origin, -5, gun.axis[0], gun.origin);
		CG_RenderItem(item, ammo, cent->currentState.powerups & ~(PW_ARMOR), gun.origin, gun.axis, cent->vel);
		return;
	}
	anim = cent->pe.torso.animationNumber & 15;
	if(anim == TORSO_ATTACK || anim == TORSO_STAND)
	{
		
		if(DotProduct(parent->axis[0], gun.axis[0]) < 0.9)
		{ // gun's in a weird position; don't adjust it.
		}
		else
		{ // we need to adjust for paralax
			refEntity_t head;
			vec3_t endpos, dir;
			trace_t tr;
			
			memset( &head, 0, sizeof( head ) );
			AngleVectors(cent->lerpAngles, dir, NULL, NULL);
			VectorMA(gun.origin, -5, gun.axis[0], gun.origin);
			CG_PositionEntityOnTag( &head, parent, parent->hModel, "tag_head");
			if(ps)
			{ // we're rendering from org, which is different from lerpOrigin, which is DUMB.
				head.origin[0] += cent->lerpOrigin[0] - parent->lightingOrigin[0];
				head.origin[1] += cent->lerpOrigin[1] - parent->lightingOrigin[1];
				head.origin[2] += cent->lerpOrigin[2] - parent->lightingOrigin[2];
			}
			VectorMA(head.origin, 8192, dir, endpos);
			CG_Trace(&tr, head.origin, vec3_origin, vec3_origin, endpos, cent->currentState.number, MASK_SHOT);
			if(tr.fraction < 1)
			{ // do paralax
				VectorSubtract(tr.endpos, gun.origin, dir);
				if(ps)
				{ // we're rendering from org, which is different from lerpOrigin, which is DUMB.
					dir[0] -= cent->lerpOrigin[0] - parent->lightingOrigin[0];
					dir[1] -= cent->lerpOrigin[1] - parent->lightingOrigin[1];
					dir[2] -= cent->lerpOrigin[2] - parent->lightingOrigin[2];
				}
			
				VectorNormalize(dir);
				VectorCopy(dir, gun.axis[0]);
				ProjectPointOnPlane(gun.axis[2], gun.axis[2], gun.axis[0]);
				VectorNormalize(gun.axis[2]);
				CrossProduct(gun.axis[2], gun.axis[0], gun.axis[1]);
			}
			VectorMA(gun.origin, 5, gun.axis[0], gun.origin);
		}
	}
	if((cent->currentState.powerups & (1 << PW_INVIS) ))
	{
		if(cent->currentState.time2 < 5000)
		{
			gun.shaderRGBA[0] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
			gun.shaderRGBA[1] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
			gun.shaderRGBA[2] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
			gun.shaderRGBA[3] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
		}
		else
			gun.shaderRGBA[3] = 0;
	}
	
	CG_AddWeaponWithPowerups( &gun, cent->currentState.powerups );
	if(!weaponNum)
		return;

	// add the spinning barrel
	if ( weaponNum == WP_CHAINGUN || weaponNum == WP_GAUNTLET ) {
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( parent->lightingOrigin, barrel.lightingOrigin );
		barrel.shadowPlane = parent->shadowPlane;
		barrel.renderfx = parent->renderfx;

		barrel.hModel = weapon->barrelModel;
		angles[YAW] = 0;
		angles[PITCH] = 0;
		angles[ROLL] = CG_MachinegunSpinAngle( cent );
		AnglesToAxis( angles, barrel.axis );

		CG_PositionRotatedEntityOnTag( &barrel, &gun, weapon->weaponModel, "tag_barrel" );

		if(weapon->weaponSkin || (cent->currentState.powerups & (1 << PW_INVIS) ))
		{
			if(weapon->weaponSkin)
				barrel.customSkin = gun.customSkin;
			barrel.shaderRGBA[0] = gun.shaderRGBA[0];
			barrel.shaderRGBA[1] = gun.shaderRGBA[1];
			barrel.shaderRGBA[2] = gun.shaderRGBA[2];
			barrel.shaderRGBA[3] = gun.shaderRGBA[3];
			
		}
		
		CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups );
	}
	else if(weaponNum == WP_FLAMETHROWER)
	{ // add napalm canister
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( parent->lightingOrigin, barrel.lightingOrigin );
		barrel.shadowPlane = parent->shadowPlane;
		barrel.renderfx = parent->renderfx;

		barrel.hModel = weapon->barrelModel;
		barrel.customShader = cgs.media.armorShader;
		VectorScale(gun.axis[0], -0.6, barrel.axis[0]);
		VectorScale(gun.axis[1], 0.6, barrel.axis[1]);
		VectorScale(gun.axis[2], -0.6, barrel.axis[2]);
		VectorMA(gun.origin, 12, gun.axis[0], barrel.origin);
		if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
		{
			barrel.shaderRGBA[0] = 1;
			barrel.shaderRGBA[1] = 1;
			barrel.shaderRGBA[2] = 1;
			barrel.shaderRGBA[3] = (5000 - cent->currentState.time2	) / 5000.0;
		}
		else
			barrel.shaderRGBA[3] = 0;
		CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups );
		
	}
	
	else if(weaponNum == WP_BALL)
	{ // ball is built out of 2 models: health bubble and quad damage
		int i, j;
		vec3_t x, y;	
		float s, c;
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( parent->lightingOrigin, barrel.lightingOrigin );
		barrel.shadowPlane = parent->shadowPlane;
		barrel.renderfx = parent->renderfx;

		barrel.hModel = weapon->barrelModel;
		angles[YAW] = 0;
		angles[PITCH] = 0;
		angles[ROLL] = 0;
		AnglesToAxis( angles, barrel.axis );
		for(i=0;i<3;i++)
		{
			for(j=0;j<3;j++)
			{
				gun.axis[i][j] *= 0.18;
				barrel.axis[i][j] *= 0.4;
			}
		}
		
		CG_PositionRotatedEntityOnTag( &barrel, parent, parent->hModel, "tag_weapon" );
		
		if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
		{
			barrel.shaderRGBA[0] = 1;
			barrel.shaderRGBA[1] = 1;
			barrel.shaderRGBA[2] = 1;
			barrel.shaderRGBA[3] = (5000 - cent->currentState.time2	) / 5000.0;
		}
		CG_AddWeaponWithPowerups( &barrel, 0);
		c = cos(cg.time / 100.0);
		s = sin(cg.time / 100.0);
		x[0] = gun.axis[0][0] * c + gun.axis[1][0] * s * 1.22;
		x[1] = gun.axis[0][1] * c + gun.axis[1][1] * s * 1.22;
		x[2] = gun.axis[0][2] * c + gun.axis[1][2] * s * 1.22;
		y[0] = gun.axis[1][0] * c - gun.axis[0][0] * s * 1.22;
		y[1] = gun.axis[1][1] * c - gun.axis[0][1] * s * 1.22;
		y[2] = gun.axis[1][2] * c - gun.axis[0][2] * s * 1.22;
		VectorCopy(x, gun.axis[0]);
		VectorCopy(y, gun.axis[1]);
		gun.customShader=trap_R_RegisterShader("models/powerups/instant/regen");
	}
	

	// add ammo belt / clip / etc.
	if(weaponNum == WP_RAILGUN && ammo >= 0)
	{ 
		gun.hModel = trap_R_RegisterModel( "models/ammo/railgun.md3" );
		gun.frame = ammo;
		CG_AddWeaponWithPowerups( &gun, cent->currentState.powerups );
		gun.frame = 0;
	}
	if(weaponNum == WP_ROCKET_LAUNCHER && ammo > 0)
	{
		CG_PositionEntityOnTag( &barrel, &gun, cg_weapons[item->giTag].weaponModel, "tag_flash" );	
		VectorMA(barrel.origin, -6, barrel.axis[0], barrel.origin);
		VectorScale( barrel.axis[2], -1, barrel.axis[2] );
		CG_DrawAmmoBelt(ammo, AMMO_ROCKETS, cent->currentState.powerups, barrel.origin, barrel.axis[0], barrel.axis[2], cent->vel	);			

	}
	else if((weaponNum == WP_CHAINGUN || weaponNum == WP_MACHINEGUN) && ammo > 0)
	{
		vec3_t org, dir;
		memset( &barrel, 0, sizeof( barrel ) );
		if(weaponNum == WP_CHAINGUN)
		{
			VectorMA(gun.origin, 7.5, gun.axis[0], org);
			VectorCopy(gun.axis[1], dir);
		}
		else
		{
			VectorMA(gun.origin, -1.0, gun.axis[0], org);
			VectorMA(org, -1, gun.axis[1], org);
			VectorScale(gun.axis[1], -1, dir);
		}
		
		VectorMA(org, (ammo % 2) * 0.25, gun.axis[1], org);
		VectorMA(org, (3.6 - (ammo % 2) * 0.125), gun.axis[2], org);
		CG_DrawAmmoBelt(ammo, AMMO_BULLETS, cent->currentState.powerups, org, gun.axis[0], dir, cent->vel);
	}
	if(!parent->hModel) 
		return; // we don't want the gun on their back looking like it's firing.

	// make sure we aren't looking at cg.predictedPlayerEntity for LG
	nonPredictedCent = &cg_entities[cent->currentState.clientNum];

	// if the index of the nonPredictedCent is not the same as the clientNum
	// then this is a fake player (like on the single player podiums), so
	// go ahead and use the cent
	if( ( nonPredictedCent - cg_entities ) != cent->currentState.clientNum ) {
		nonPredictedCent = cent;
	}


	// add the flash
	
	memset( &flash, 0, sizeof( flash ) );
	VectorCopy( parent->lightingOrigin, flash.lightingOrigin );
	flash.shadowPlane = parent->shadowPlane;
	flash.renderfx = parent->renderfx;

	flash.hModel = weapon->flashModel;
	if (!flash.hModel) {
		return;
	}
	angles[YAW] = 0;
	angles[PITCH] = 0;
	angles[ROLL] = crandom() * 10;
	AnglesToAxis( angles, flash.axis );

	if ( weaponNum == WP_RAILGUN ) {
		flash.shaderRGBA[0] = 255;
		flash.shaderRGBA[1] = 255;
		flash.shaderRGBA[2] = 255;
	}

	CG_PositionRotatedEntityOnTag( &flash, &gun, weapon->weaponModel, "tag_flash");
	if(weaponNum == WP_MACHINEGUN || weaponNum == WP_SINGLE_SHOTGUN)
	{
		vec3_t mins, maxs;
		trap_R_ModelBounds(gun.hModel, mins, maxs);
		VectorMA(flash.origin, maxs[0], gun.axis[0], flash.origin);
		VectorMA(flash.origin, 2, gun.axis[2], flash.origin);
		VectorScale(flash.axis[1], 0.6, flash.axis[1]);
		VectorScale(flash.axis[2], 0.6, flash.axis[2]);
	}
	VectorCopy(flash.origin, cent->pe.railgunOrigin); 
	VectorCopy(flash.origin, cent->currentState.origin2); 

	if(weaponNum == WP_FLAMETHROWER)
	{
		VectorMA(flash.origin, 6.5, gun.axis[0], flash.origin);
		flash.customShader = trap_R_RegisterShader( "models/weapons2/rocketl/f_rocketl.md3" );
		VectorScale(flash.axis[0], 0.25, flash.axis[0]);
		VectorScale(flash.axis[1], 0.1, flash.axis[1]);
		VectorScale(flash.axis[2], 0.1, flash.axis[2]);
	}
	if ( ( weaponNum == WP_LIGHTNING || weaponNum == WP_GAUNTLET || weaponNum == WP_FLAMETHROWER)
		&& ( nonPredictedCent->currentState.eFlags & EF_FIRING ) ) 
	{
		// continuous flash
	} else {
		// impulse flash
		if ( cg.time - cent->muzzleFlashTime > MUZZLE_FLASH_TIME && !cent->pe.railgunFlash ) {
			return;
		}
	}
	
	trap_R_AddRefEntityToScene( &flash );
	
	if ( !(parent->renderfx & RF_THIRD_PERSON) ) {
		// add lightning bolt
		CG_LightningBolt( nonPredictedCent, flash.origin );
		CG_FlameBolt( nonPredictedCent, flash.origin );
		// add rail trail
		CG_SpawnRailTrail( nonPredictedCent, flash.origin );

		// make a dlight for the flash
		if ( weapon->flashDlightColor[0] || weapon->flashDlightColor[1] || weapon->flashDlightColor[2] ) {
			trap_R_AddLightToScene( flash.origin, 200 + (rand()&31), weapon->flashDlightColor[0],
				weapon->flashDlightColor[1], weapon->flashDlightColor[2] );
		}
	}
	
}

/*
==============================================================================

WEAPON SELECTION

==============================================================================
*/

/*
===================
CG_DrawWeaponSelect
===================
*/
void CG_DrawWeaponSelect( void ) {
	return; // Hentai - we'll totally redo this later.

}


/*
===============
CG_NextWeapon_f
===============
*/
void CG_NextWeapon_f( void ) {

	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}
	cg.weaponSelectTime = cg.time;	
	cg.weaponSelect++;
	if(cg.weaponSelect > 7)
		cg.weaponSelect = 0;
}

void CG_PrevWeapon_f( void ) {

	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}
	cg.weaponSelectTime = cg.time;	
	cg.weaponSelect--;
	if(cg.weaponSelect < 0)
		cg.weaponSelect = 7;
}

void CG_ReloadWeapon_f( void ) {

	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}
	cg.weaponSelectTime = cg.time;	
	cg.weaponSelect = WPM_RELOAD;
}

/*
===============
CG_PrevWeapon_f
===============
*/

/*
===============
CG_Weapon_f
===============
*/
qboolean	CG_ParseAnimationFile( const char *filename, clientInfo_t *ci );
void CG_MakeAnims_f(void)
{
	char buffer[1024];
	vec3_t mins, maxs;
	orientation_t tag, tag2;
	int frame, weight, anim, i;
	animation_t *frames;
	clientInfo_t ci;
	qhandle_t head, torso, legs;
	fileHandle_t	classfile;
	char model[32];
	// get model data and dump to *.animation
	
	strncpy(model, CG_Argv( 1 ), sizeof(model));
	head = trap_R_RegisterModel(va("models/players/%s/head.md3", model));
	torso = trap_R_RegisterModel(va("models/players/%s/upper.md3", model));	
	legs = trap_R_RegisterModel(va("models/players/%s/lower.md3", model));
			

	CG_ParseAnimationFile(va("models/players/%s/animation.cfg", model), &ci);
	frames = ci.animations;
				
	i = trap_FS_FOpenFile(va("models/players/%s/head.animation", model), &classfile, FS_WRITE );	
	if(classfile)
	{
		trap_R_ModelBounds(head, mins, maxs);	
	
		trap_FS_Write("[NONE]\r\nframes=	1	0", strlen("[NONE]\r\nframes=	1	0\r\n"), classfile);							
		Com_sprintf(buffer, sizeof(buffer), ":0	100	mins=	%f %f %f	maxs=	%f %f %f\r\n", mins[0], mins[1], mins[2], maxs[0], maxs[1], maxs[2]);
		trap_FS_Write(buffer, strlen(buffer), classfile);							
		trap_FS_FCloseFile(classfile);
	}
	i = trap_FS_FOpenFile(va("models/players/%s/upper.animation", model), &classfile, FS_WRITE );	
	if(classfile)
	{
		for(anim = 0; anim < TORSO_ANIMATIONS; anim++)
		{
			Com_sprintf(buffer, sizeof(buffer), "\r\n[%s]\r\nframes=	%d %d\r\n", animNames[anim], frames[anim].numFrames, frames[anim].loopFrames);
			trap_FS_Write(buffer, strlen(buffer), classfile);
			for(frame = frames[anim].firstFrame; frame < frames[anim].firstFrame + frames[anim].numFrames; frame++)
			{														
				if(frame == frames[anim].firstFrame)
					weight = frames[anim].initialLerp;
				else
					weight = frames[anim].frameLerp;
				trap_R_ModelBounds(torso, mins, maxs);	
				trap_R_LerpTag(&tag, torso, frame, frame, 0, "tag_head");
				trap_R_LerpTag(&tag2, torso, frame, frame, 0, "tag_weapon");
				Com_sprintf(buffer, sizeof(buffer), ":%d	%d	mins=	%f %f %f	maxs=	%f %f %f	tag_weapon=	%f %f %f %f %f %f %f %f %f %f %f %f 	tag_head=	%f %f %f %f %f %f %f %f %f %f %f %f \r\n", frame, weight, mins[0], mins[1], mins[2], maxs[0], maxs[1], maxs[2], tag2.origin[0], tag2.origin[1], tag2.origin[2], tag2.axis[0][0], tag2.axis[0][1], tag2.axis[0][2], tag2.axis[1][0], tag2.axis[1][1], tag2.axis[1][2], tag2.axis[2][0], tag2.axis[2][1], tag2.axis[2][2], tag.origin[0], tag.origin[1], tag.origin[2], tag.axis[0][0], tag.axis[0][1], tag.axis[0][2], tag.axis[1][0], tag.axis[1][1], tag.axis[1][2], tag.axis[2][0], tag.axis[2][1], tag.axis[2][2]);
				trap_FS_Write(buffer, strlen(buffer), classfile);							
			}
						
		}
	
		trap_FS_FCloseFile(classfile);
	}
	i = trap_FS_FOpenFile(va("models/players/%s/lower.animation", model), &classfile, FS_WRITE );	
	if(classfile)
	{
		for(anim = 0; anim < DEATH_ANIMATIONS; anim++)
		{
			Com_sprintf(buffer, sizeof(buffer), "\r\n[%s]\r\nframes=	%d %d\r\n", animNames[anim], frames[anim].numFrames, frames[anim].loopFrames);
			trap_FS_Write(buffer, strlen(buffer), classfile);
			for(frame = frames[anim].firstFrame; frame < frames[anim].firstFrame + frames[anim].numFrames; frame++)
			{
				if(frame == frames[anim].firstFrame)
					weight = frames[anim].initialLerp;
				else
					weight = frames[anim].frameLerp;
				trap_R_ModelBounds(legs, mins, maxs);	
				trap_R_LerpTag(&tag, legs, frame, frame, 0, "tag_torso");							
				Com_sprintf(buffer, sizeof(buffer), ":%d	%d	mins=	%f %f %f	maxs=	%f %f %f	tag_torso=	%f %f %f %f %f %f %f %f %f %f %f %f\r\n", frame, weight, mins[0], mins[1], mins[2], maxs[0], maxs[1], maxs[2], tag.origin[0], tag.origin[1], tag.origin[2], tag.axis[0][0], tag.axis[0][1], tag.axis[0][2], tag.axis[1][0], tag.axis[1][1], tag.axis[1][2], tag.axis[2][0], tag.axis[2][1], tag.axis[2][2]);
				trap_FS_Write(buffer, strlen(buffer), classfile);							
			}
		}
		for(anim = TORSO_ANIMATIONS; anim < MAX_ANIMATIONS; anim++)
		{
			Com_sprintf(buffer, sizeof(buffer), "\r\n[%s]\r\nframes=	%d %d\r\n", animNames[anim], frames[anim].numFrames, frames[anim].loopFrames);
			trap_FS_Write(buffer, strlen(buffer), classfile);
			for(frame = frames[anim].firstFrame; frame < frames[anim].firstFrame + frames[anim].numFrames; frame++)
			{
				int weight;
				
				if(frame == frames[anim].firstFrame)
					weight = frames[anim].initialLerp;
				else
					weight = frames[anim].frameLerp;
				trap_R_ModelBounds(legs, mins, maxs);	
				trap_R_LerpTag(&tag, legs, frame, frame, 0, "tag_torso");
				Com_sprintf(buffer, sizeof(buffer), ":%d	%d	mins=	%f %f %f	maxs=	%f %f %f	tag_torso=	%f %f %f %f %f %f %f %f %f %f %f %f\r\n", frame, weight, mins[0], mins[1], mins[2], maxs[0], maxs[1], maxs[2], tag.origin[0], tag.origin[1], tag.origin[2], tag.axis[0][0], tag.axis[0][1], tag.axis[0][2], tag.axis[1][0], tag.axis[1][1], tag.axis[1][2], tag.axis[2][0], tag.axis[2][1], tag.axis[2][2]);
				trap_FS_Write(buffer, strlen(buffer), classfile);							
			}
			
		}
				
		trap_FS_FCloseFile(classfile);
	}
}
void CG_Weapon_f( void ) {
	int		num;
	char item[32];
	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}

	strncpy(item, CG_Argv( 1 ), 32);
	num = atoi( item );

	if(num == 0)
	{
		if(!Q_stricmp(item, "none") || !Q_stricmp(item, "0"))
		{
			num = 0;
		}
		if(!Q_stricmp(item, "weapon"))
		{
			num = AMMO_ITEM_BIG / 2;
		}
		else if(!Q_stricmp(item, "empty"))
		{
			int i;
			if(bg_itemlist[cg.snap->ps.weapon].giType == IT_WEAPON)
			{
				if(cg.snap->ps.ammo[AMMO_ITEM_BIG])
					num = 0;
				else
					num = AMMO_ITEM_BIG / 2;
			}
			else
			{
				for(i = AMMO_ITEM1; i < AMMO_ITEMS; i+= 2)
				{
					if(!cg.snap->ps.ammo[i])
					{
						num = i;
						break;
					}
				}
			}
		}
		else if(!Q_stricmp(item, "ammo"))
		{
			int i;
			for(i = AMMO_ITEM1; i < AMMO_ITEMS; i+= 2)
			{
				if(bg_itemlist[cg.snap->ps.ammo[i]].giType == IT_AMMO && bg_itemlist[cg.snap->ps.ammo[i]].giTag == bg_itemlist[cg.snap->ps.weapon].giUses )
				{
					num = i;
					break;
				}
			}		
		}
		else
		{
			int i;
			for(i = AMMO_ITEM1; i < AMMO_ITEMS; i+= 2)
			{
				if(!Q_stricmp(item, bg_itemlist[cg.snap->ps.ammo[i]].pickup_name))
				{
					num = i;
					break;
				}
			}		
		}
	}
	else if ( num < 0 || num > 15 ) {
		return;
	}
	

	cg.weaponSelectTime = cg.time;
	cg.weaponSelect = num;
}

/*
===================
CG_OutOfAmmoChange

The current weapon has just run out of ammo
===================
*/
void CG_OutOfAmmoChange( void ) {
	return;
}



/*
===================================================================================================

WEAPON EVENTS

===================================================================================================
*/

/*
================
CG_FireWeapon

Caused by an EV_FIRE_WEAPON event
================
*/
void CG_FireWeapon( centity_t *cent ) {
	entityState_t *ent;
	int				c, weapon;
	weaponInfo_t	*weap;

	ent = &cent->currentState;
	if(ent->eType == ET_PLAYER)
	{

		if ( ent->weapon == 0 || bg_itemlist[ent->weapon].giType != IT_WEAPON ) {
			return;
		}
	
		weapon = bg_itemlist[ent->weapon].giTag;
	}
	else
	{
		weapon = ent->weapon;
	}
	if ( weapon >= WP_NUM_WEAPONS ) {
		CG_Error( "CG_FireWeapon: weapon >= WP_NUM_WEAPONS" );
		return;
	}
	weap = &cg_weapons[ weapon ];

	// mark the entity as muzzle flashing, so when it is added it will
	// append the flash to the weapon model
	cent->muzzleFlashTime = cg.time;

	// lightning gun only does this this on initial press
	if ( weapon == WP_LIGHTNING || weapon == WP_FLAMETHROWER) {
		if ( cent->pe.lightningFiring ) {
			//CG_Printf("Skipping");
			return;
		}
	}
	
	// play quad sound if needed
	if ( cent->currentState.powerups & ( 1 << PW_QUAD ) ) {
		trap_S_StartSound (NULL, cent->currentState.number, CHAN_ITEM, cgs.media.quadSound );
	}

	// play a sound
	for ( c = 0 ; c < 4 ; c++ ) {
		if ( !weap->flashSound[c] ) {
			break;
		}
	}
	if ( c > 0 ) {
		c = rand() % c;
		if ( weap->flashSound[c] )
		{
			trap_S_StartSound( NULL, ent->number, CHAN_WEAPON, weap->flashSound[c] );
		}
	}

	// do brass ejection
	if ( weap->ejectBrassFunc ) {
		weap->ejectBrassFunc( cent );
	}
}

void CG_FireWeaponFromItem( centity_t *cent ) {
	entityState_t *ent;
	int				c, weapon;
	weaponInfo_t	*weap;

	ent = &cent->currentState;
	weapon = ent->weapon;
	if ( weapon >= WP_NUM_WEAPONS ) {
		CG_Error( "CG_FireWeapon: weapon >= WP_NUM_WEAPONS" );
		return;
	}
	weap = &cg_weapons[ weapon ];

	// play a sound
	for ( c = 0 ; c < 4 ; c++ ) {
		if ( !weap->flashSound[c] ) {
			break;
		}
	}
	if ( c > 0 ) {
		c = rand() % c;
		if ( weap->flashSound[c] )
		{
			trap_S_StartSound( NULL, ent->number, CHAN_WEAPON, weap->flashSound[c] );
		}
	}

}

/*
=================
CG_MissileHitWall

Caused by an EV_MISSILE_MISS event, or directly by local bullet tracing
=================
*/
void CG_MissileHitWall( int weapon, int clientNum, vec3_t origin, vec3_t dir ) {
	qhandle_t		mod;
	qhandle_t		mark;
	qhandle_t		shader;
	sfxHandle_t		sfx;
	float			radius;
	float			mradius;
	float			light;
	vec3_t			lightColor;
	localEntity_t	*le;
	int				r;
	qboolean		alphaFade;
	qboolean		isSprite;
	int				duration;
	
	mark = 0;
	mradius = 0;
	radius = 32;
	sfx = 0;
	mod = 0;
	shader = 0;
	light = 0;
	lightColor[0] = 1;
	lightColor[1] = 1;
	lightColor[2] = 0;

	// set defaults
	isSprite = qfalse;
	duration = 600;

	switch ( weapon & 15) {
	default:
	case WP_FLAMETHROWER:
		mark = cgs.media.burnMarkShader; 
		radius = 48; 

		isSprite = qtrue;  
		duration = 900;    
		break;
	case WP_LIGHTNING:
		// no explosion at LG impact, it is added with the beam
		r = rand() & 3;
		if ( r < 2 ) {
			sfx = cgs.media.sfx_lghit2;
		} else if ( r == 2 ) {
			sfx = cgs.media.sfx_lghit1;
		} else {
			sfx = cgs.media.sfx_lghit3;
		}
		mark = cgs.media.holeMarkShader;
		radius = 12;
		duration = 0;
		break;
	case WP_GRENADE_LAUNCHER:
		mod = cgs.media.sphereFlashModel;
		shader = cgs.media.grenadeExplosionShader;
		sfx = cgs.media.sfx_rockexp;
		mark = cgs.media.burnMarkShader;
		radius = 32;
		light = 300;
		duration = DEFAULT_GRENADE_DURATION;
		mradius = DEFAULT_GRENADE_RADIUS / 2;
			
		if(VectorLength(dir))
		{ // it's actually against a wall
			int i, r;
			vec3_t dir2;
			for(i = 0; i < 64; i++)
			{
				r = rand() & 255;
				VectorCopy(bytedirs[r], dir2);
				if(DotProduct(dir, dir2) < 0)
					VectorMA(dir2, 2, dir, dir2);
				CG_GrenadeEjectShrapnel(origin, dir2);
			}
		}
		else
		{
			int i, r;
			for(i = 0; i < 64; i++)
			{
				r = rand() & 255;
				CG_GrenadeEjectShrapnel(origin, bytedirs[r]);
			}
		}
		
		break;
	case WP_ROCKET_LAUNCHER:
		mod = cgs.media.sphereFlashModel;
		shader = cgs.media.rocketExplosionShader;
		sfx = cgs.media.sfx_rockexp;
		mark = cgs.media.burnMarkShader;
		radius = 32;
		light = 300;
		duration = DEFAULT_ROCKET_DURATION;
		mradius = DEFAULT_ROCKET_RADIUS / 2;
			
		if(VectorLength(dir))
		{ // it's actually against a wall
			int i, r, s;
			vec3_t dir2;
			for(i = 0; i < 64; i++)
			{
				r = rand() & 255;
				VectorCopy(bytedirs[r], dir2);
				s = DotProduct(dir, dir2);
				if(s < 0)
					VectorMA(dir2, -2 * s, dir, dir2);
				VectorNormalize(dir2);
				CG_RocketEjectShrapnel(origin, dir2);
			}
		}
		else
		{
			int i, r;
			for(i = 0; i < 64; i++)
			{
				r = rand() & 255;
				CG_RocketEjectShrapnel(origin, bytedirs[r]);
			}
		}
			
		lightColor[0] = 1;
		lightColor[1] = 0.75;
		lightColor[2] = 0.0;
		break;
	case WP_RAILGUN:
		mod = cgs.media.ringFlashModel;
		shader = cgs.media.railExplosionShader;
		sfx = cgs.media.sfx_plasmaexp;
		mark = cgs.media.energyMarkShader;
		radius = 24;
		break;
	case WP_PLASMAGUN:
		mod = cgs.media.ringFlashModel;
		shader = cgs.media.plasmaExplosionShader;
		sfx = cgs.media.sfx_plasmaexp;
		mark = cgs.media.energyMarkShader;
		radius = 16;
		break;
	case WP_SHOTGUN:
		mod = cgs.media.bulletFlashModel;
		shader = cgs.media.bulletExplosionShader;
		mark = cgs.media.bulletMarkShader;
		sfx = 0;
		radius = 1;//4
		break;

	case WP_MACHINEGUN:
	case WP_CHAINGUN:
		mod = cgs.media.bulletFlashModel;
		shader = cgs.media.bulletExplosionShader;
		mark = cgs.media.bulletMarkShader;
		r = rand() & 3;
		if ( r < 2 ) {
			sfx = cgs.media.sfx_ric1;
		} else if ( r == 2 ) {
			sfx = cgs.media.sfx_ric2;
		} else {
			sfx = cgs.media.sfx_ric3;
		}

		radius = 8;
		break;
	}

	if ( sfx ) {
		trap_S_StartSound( origin, ENTITYNUM_WORLD, CHAN_AUTO, sfx );
	}

	//
	// create the explosion
	//
	if ( mod ) {
		if(weapon > 15)
		{
			weapon &= 15;
			le = CG_MakeExplosion( origin, dir, 
							   mod,	cgs.media.quadShader,
							   duration, isSprite );		
			le->light = light;
			le->lightColor[0] = 0;
			le->lightColor[1] = 0.5;
			le->lightColor[2] = 1;
			if(weapon == WP_ROCKET_LAUNCHER || weapon == WP_GRENADE_LAUNCHER || weapon == WP_NONE)
			{
				le->refEntity.renderfx |= RF_EXPANDING;			
				le->radius = 64;
			}
		
		}
		le = CG_MakeExplosion( origin, dir, 
							   mod,	shader,
							   duration, isSprite );		
		if(mradius)
			le->refEntity.radius = mradius;
		le->light = light;
		
		VectorCopy( lightColor, le->lightColor );
		if ( weapon == WP_RAILGUN ) {
			// colorize with client color
			VectorCopy( cgs.clientinfo[clientNum].color, le->color );
		}
		else if(weapon == WP_ROCKET_LAUNCHER || weapon == WP_GRENADE_LAUNCHER || weapon == WP_NONE)
		{
			//le->refEntity.customShader = ;//cgs.media.plasmaBallShader;
			le->refEntity.renderfx |= RF_EXPANDING;			
			le->startTime -= duration * 0.3; // make it start out with some size
			//le->radius = 24;
			//le = CG_MakeExplosion( origin, dir, 
			//				   mod,	cgs.media.smokeExplosionShader,
			//				   duration, isSprite );		
			//le->refEntity.renderfx |= RF_EXPANDING;			
			//le->radius = 64;
			
		}
		else if(weapon == WP_SHOTGUN || weapon == WP_SINGLE_SHOTGUN )
		{ // smaller damnit
			le->refEntity.axis[0][0] *= 0.2;
			le->refEntity.axis[1][0] *= 0.2;
			le->refEntity.axis[2][0] *= 0.2;
			le->refEntity.axis[0][1] *= 0.2;
			le->refEntity.axis[1][1] *= 0.2;
			le->refEntity.axis[2][1] *= 0.2;
			le->refEntity.axis[0][2] *= 0.2;
			le->refEntity.axis[1][2] *= 0.2;
			le->refEntity.axis[2][2] *= 0.2;
			
		}
	}
	weapon &= 15;
	//
	// impact mark
	//
	alphaFade = (mark == cgs.media.energyMarkShader);	// plasma fades alpha, all others fade color
	if ( weapon == WP_RAILGUN ) {
		float	*color;

		// colorize with client color
		color = cgs.clientinfo[clientNum].color;
		CG_ImpactMark( mark, origin, dir, random()*360, color[0],color[1], color[2],1, alphaFade, radius, qfalse );
	} else {
		CG_ImpactMark( mark, origin, dir, random()*360, 1,1,1,1, alphaFade, radius, qfalse );
	}
}


/*
=================
CG_MissileHitPlayer
=================
*/
void CG_MissileHitPlayer( int weapon, vec3_t origin, vec3_t dir, int entityNum ) {
	
	CG_Bleed( origin, entityNum );

	// some weapons will make an explosion with the blood, while
	// others will just make the blood
	switch ( weapon ) {
	case WP_GRENADE_LAUNCHER:
	case WP_ROCKET_LAUNCHER:
		CG_MissileHitWall( weapon, 0, origin, dir );
		break;
	default:
		break;
	}
}

/*
=================
CG_MissileHitPlayer
=================
*/
void CG_MissileHitRPlayer( int weapon, vec3_t origin, vec3_t dir, int entityNum ) {
	CG_RBleed( origin, entityNum );

	// some weapons will make an explosion with the blood, while
	// others will just make the blood
	switch ( weapon ) {
	case WP_GRENADE_LAUNCHER:
	case WP_ROCKET_LAUNCHER:
		CG_MissileHitWall( weapon, 0, origin, dir );
		break;
	default:
		break;
	}
}


/*
============================================================================

SHOTGUN TRACING

============================================================================
*/

/*
================
CG_ShotgunPellet
================
*/
static void CG_ShotgunPellet( vec3_t start, vec3_t end, int skipNum ) {
	trace_t		tr;
	int sourceContentType, destContentType;

	CG_Trace( &tr, start, NULL, NULL, end, skipNum, MASK_SHOT );
	
	sourceContentType = trap_CM_PointContents( start, 0 );
	destContentType = trap_CM_PointContents( tr.endpos, 0 );

	// FIXME: should probably move this cruft into CG_BubbleTrail
	if ( sourceContentType == destContentType ) {
		if ( sourceContentType & CONTENTS_WATER ) {
			CG_BubbleTrail( start, tr.endpos, 32 );
		}
	} else if ( sourceContentType & CONTENTS_WATER ) {
		trace_t trace;

		trap_CM_BoxTrace( &trace, end, start, NULL, NULL, 0, CONTENTS_WATER );
		CG_BubbleTrail( start, trace.endpos, 32 );
	} else if ( destContentType & CONTENTS_WATER ) {
		trace_t trace;

		trap_CM_BoxTrace( &trace, start, end, NULL, NULL, 0, CONTENTS_WATER );
		CG_BubbleTrail( tr.endpos, trace.endpos, 32 );
	}

	if (  tr.surfaceFlags & SURF_NOIMPACT ) {
		return;
	}
	
	if ( cg_entities[tr.entityNum].currentState.eType == ET_PLAYER ) {
	//	CG_MissileHitPlayer( WP_SHOTGUN, tr.endpos, tr.plane.normal, tr.entityNum );
	} else 
	{
		if ( tr.surfaceFlags & SURF_NOIMPACT ) {
			// SURF_NOIMPACT will not make a flame puff or a mark
			return;
		}
		CG_MissileHitWall( WP_SHOTGUN, 0, tr.endpos, tr.plane.normal );
	}
}

/*
================
CG_ShotgunPattern

Perform the same traces the server did to locate the
hit splashes (FIXME: ranom seed isn't synce anymore)
================
*/
static void CG_ShotgunPattern( vec3_t origin, vec3_t origin2, int seed, int otherEntNum ) {
	int			i;
	float		r, u;
	vec3_t		end;
	vec3_t		forward, right, up;

	// derive the right and up vectors from the forward vector, because
	// the client won't have any other information
	VectorNormalize2( origin2, forward );
	PerpendicularVector( right, forward );
	CrossProduct( forward, right, up );

	// generate the "random" spread pattern
	for ( i = 0 ; i < DEFAULT_SHOTGUN_COUNT ; i++ ) {		
		r = Q_crandom(&seed) * DEFAULT_SHOTGUN_SPREAD;
		u = Q_crandom(&seed) * DEFAULT_SHOTGUN_SPREAD;
		VectorMA( origin, 8192, forward, end);
		VectorMA (end, r, right, end);
		VectorMA (end, u, up, end);

		CG_ShotgunPellet( origin, end, otherEntNum );
	}
}

/*
==============
CG_ShotgunFire
==============
*/
void CG_ShotgunFire( entityState_t *es ) {
	vec3_t	v;
	int		contents;

	VectorSubtract( es->origin2, es->pos.trBase, v );
	VectorNormalize( v );
	VectorScale( v, 32, v );
	VectorAdd( es->pos.trBase, v, v );
	if ( cgs.glconfig.hardwareType != GLHW_RAGEPRO ) {
		// ragepro can't alpha fade, so don't even bother with smoke
		vec3_t			up;

		contents = trap_CM_PointContents( es->pos.trBase, 0 );
		if ( !( contents & CONTENTS_WATER ) ) {
			VectorSet( up, 0, 0, 8 );
			CG_SmokePuff( v, up, 32, 1, 1, 1, 0.33, 900, cg.time, LEF_PUFF_DONT_SCALE, cgs.media.shotgunSmokePuffShader );
		}
	}
	CG_ShotgunPattern( es->pos.trBase, es->origin2, es->eventParm, es->otherEntityNum );
}

/*
============================================================================

BULLETS

============================================================================
*/


/*
===============
CG_Tracer
===============
*/
void CG_Tracer( vec3_t source, vec3_t dest, float width, qhandle_t shader, vec4_t rgba ) {
	vec3_t		forward, right;
	polyVert_t	verts[4];
	vec3_t		line;
	float		len, begin, end;
	vec3_t		start, finish;
	vec3_t		midpoint;

	// tracer
	VectorSubtract( dest, source, forward );
	len = VectorNormalize( forward );

	// start at least a little ways from the muzzle
	begin = width * -0.5;
	end = len + width * 0.5;
	VectorMA( source, begin, forward, start );
	VectorMA( source, end, forward, finish );
	line[0] = DotProduct( forward, cg.refdef.viewaxis[1] );
	line[1] = DotProduct( forward, cg.refdef.viewaxis[2] );

	VectorScale( cg.refdef.viewaxis[1], line[1], right );
	VectorMA( right, -line[0], cg.refdef.viewaxis[2], right );
	VectorNormalize( right );

	VectorMA( finish, width, right, verts[0].xyz );
	verts[0].st[0] = 0;
	verts[0].st[1] = 1;
	verts[0].modulate[0] = 255 * rgba[0];
	verts[0].modulate[1] = 255 * rgba[1];
	verts[0].modulate[2] = 255 * rgba[2];
	verts[0].modulate[3] = 255 * rgba[3];
	
	VectorMA( finish, -width, right, verts[1].xyz );
	verts[1].st[0] = 1;
	verts[1].st[1] = 1;
	verts[1].modulate[0] = 255 * rgba[0];
	verts[1].modulate[1] = 255 * rgba[1];
	verts[1].modulate[2] = 255 * rgba[2];
	verts[1].modulate[3] = 255 * rgba[3];

	VectorMA( start, -width, right, verts[2].xyz );
	verts[2].st[0] = 1;
	verts[2].st[1] = 0;
	verts[2].modulate[0] = 255 * rgba[0];
	verts[2].modulate[1] = 255 * rgba[1];
	verts[2].modulate[2] = 255 * rgba[2];
	verts[2].modulate[3] = 255 * rgba[3];

	VectorMA( start, width, right, verts[3].xyz );
	verts[3].st[0] = 0;
	verts[3].st[1] = 0;
	verts[3].modulate[0] = 255 * rgba[0];
	verts[3].modulate[1] = 255 * rgba[1];
	verts[3].modulate[2] = 255 * rgba[2];
	verts[3].modulate[3] = 255 * rgba[3];

	trap_R_AddPolyToScene( shader, 4, verts );

	midpoint[0] = ( start[0] + finish[0] ) * 0.5;
	midpoint[1] = ( start[1] + finish[1] ) * 0.5;
	midpoint[2] = ( start[2] + finish[2] ) * 0.5;

	// add the tracer sound
	//trap_S_StartSound( midpoint, ENTITYNUM_WORLD, CHAN_AUTO, cgs.media.tracerSound );

}


