
// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_players.c -- handle the media and animation for player entities
#include "cg_local.h"
localEntity_t *CG_SpurtBlood( vec3_t org, vec3_t dir, int amount );

extern vec3_t	bytedirs[NUMVERTEXNORMALS];
char	*cg_customSoundNames[MAX_CUSTOM_SOUNDS] = {
	"*death1.wav",
	"*death2.wav",
	"*death3.wav",
	"*jump1.wav",
	"*pain25_1.wav",
	"*pain50_1.wav",
	"*pain75_1.wav",
	"*pain100_1.wav",
	"*falling1.wav",
	"*gasp.wav",
	"*drown.wav",
	"*fall1.wav",
	"*taunt.wav"
};

#define WINGS_SPREAD	-4
#define WINGS_IDLE		-3
#define WINGS_FLAP		-2
#define WINGS_FLAPB		-1

animation_t wingsAnim[] = 
{
	{0, 32, 0, 30, 30}, {0, 32, 0, 30, 30}, {0, 32, 32, 30, 30}, {0, 32, 32, 30, 30}

};
/*
================
CG_CustomSound

================
*/
sfxHandle_t	CG_CustomSound( int clientNum, const char *soundName ) {
	clientInfo_t *ci;
	int			i;

	if ( soundName[0] != '*' ) {
		return trap_S_RegisterSound( soundName );
	}

	if ( clientNum < 0 || clientNum >= MAX_CLIENTS ) {
		clientNum = 0;
	}
	ci = &cgs.clientinfo[ clientNum ];

	for ( i = 0 ; i < MAX_CUSTOM_SOUNDS && cg_customSoundNames[i] ; i++ ) {
		if ( !strcmp( soundName, cg_customSoundNames[i] ) ) {
			return ci->sounds[i];
		}
	}

	CG_Error( "Unknown custom sound: %s", soundName );
	return 0;
}



/*
=============================================================================

CLIENT INFO

=============================================================================
*/

/*
======================
CG_ParseAnimationFile

Read a configuration file containing animation coutns and rates
models/players/visor/animation.cfg, etc
======================
*/
qboolean	CG_ParseAnimationFile( const char *filename, clientInfo_t *ci ) {
	char		*text_p, *prev;
	int			len;
	int			i;
	char		*token;
	float		fps;
	int			skip;
	char		text[20000];
	fileHandle_t	f;
	animation_t *animations;

	animations = ci->animations;

	// load the file
	len = trap_FS_FOpenFile( filename, &f, FS_READ );
	if ( len <= 0 ) {
		return qfalse;
	}
	if ( len >= sizeof( text ) - 1 ) {
		CG_Printf( "File %s too long\n", filename );
		return qfalse;
	}
	trap_FS_Read( text, len, f );
	text[len] = 0;
	trap_FS_FCloseFile( f );

	// parse the text
	text_p = text;
	skip = 0;	// quite the compiler warning

	ci->footsteps = FOOTSTEP_NORMAL;
	VectorClear( ci->headOffset );
	ci->gender = GENDER_MALE;

	// read optional parameters
	while ( 1 ) {
		prev = text_p;	// so we can unget
		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		if ( !Q_stricmp( token, "footsteps" ) ) {
			token = COM_Parse( &text_p );
			if ( !token ) {
				break;
			}
			if ( !Q_stricmp( token, "default" ) || !Q_stricmp( token, "normal" ) ) {
				ci->footsteps = FOOTSTEP_NORMAL;
			} else if ( !Q_stricmp( token, "boot" ) ) {
				ci->footsteps = FOOTSTEP_BOOT;
			} else if ( !Q_stricmp( token, "flesh" ) ) {
				ci->footsteps = FOOTSTEP_FLESH;
			} else if ( !Q_stricmp( token, "mech" ) ) {
				ci->footsteps = FOOTSTEP_MECH;
			} else if ( !Q_stricmp( token, "energy" ) ) {
				ci->footsteps = FOOTSTEP_ENERGY;
			} else {
				CG_Printf( "Bad footsteps parm in %s: %s\n", filename, token );
			}
			continue;
		} else if ( !Q_stricmp( token, "headoffset" ) ) {
			for ( i = 0 ; i < 3 ; i++ ) {
				token = COM_Parse( &text_p );
				if ( !token ) {
					break;
				}
				ci->headOffset[i] = atof( token );
			}
			continue;
		} else if ( !Q_stricmp( token, "sex" ) ) {
			token = COM_Parse( &text_p );
			if ( !token ) {
				break;
			}
			if ( token[0] == 'f' || token[0] == 'F' ) {
				ci->gender = GENDER_FEMALE;
			} else if ( token[0] == 'n' || token[0] == 'N' ) {
				ci->gender = GENDER_NEUTER;
			} else {
				ci->gender = GENDER_MALE;
			}
			continue;
		}

		// if it is a number, start parsing animations
		if ( token[0] >= '0' && token[0] <= '9' ) {
			text_p = prev;	// unget the token
			break;
		}
		Com_Printf( "unknown token '%s' is %s\n", token, filename );
	}

	// read information for each frame
	for ( i = 0 ; i < MAX_ANIMATIONS ; i++ ) {

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		animations[i].firstFrame = atoi( token );
		// leg only frames are adjusted to not count the upper body only frames
		if ( i == LEGS_WALKCR ) {
			skip = animations[LEGS_WALKCR].firstFrame - animations[TORSO_GESTURE].firstFrame;
		}
		if ( i >= LEGS_WALKCR ) {
			animations[i].firstFrame -= skip;
		}

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		animations[i].numFrames = atoi( token );

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		animations[i].loopFrames = atoi( token );

		token = COM_Parse( &text_p );
		if ( !token ) {
			break;
		}
		fps = atof( token );
		if ( fps == 0 ) {
			fps = 1;
		}
		animations[i].frameLerp = 1000 / fps;
		animations[i].initialLerp = 1000 / fps;
	}
	animations[LEGS_JUMPB].numFrames = 1;
	animations[LEGS_JUMPB].loopFrames = 0;
	animations[LEGS_JUMPB].initialLerp = 0;
	animations[LEGS_JUMPB].frameLerp = 400;
	if ( i != MAX_ANIMATIONS ) {
		CG_Printf( "Error parsing animation file: %s", filename );
		return qfalse;
	}

	return qtrue;
}

/*
==========================
CG_RegisterClientSkin
==========================
*/
static qboolean	CG_RegisterClientSkin( clientInfo_t *ci, const char *modelName, const char *skinName, const char *headModelName, const char *headSkinName ) {
	char		filename[MAX_QPATH];

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/lower_%s.skin", modelName, skinName );
	ci->legsSkin = trap_R_RegisterSkin( filename );

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/upper_%s.skin", modelName, skinName );
	ci->torsoSkin = trap_R_RegisterSkin( filename );

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/head_%s.skin", headModelName, headSkinName );
	ci->headSkin = trap_R_RegisterSkin( filename );
	if(!ci->headSkin)
	{
		Com_sprintf( filename, sizeof( filename ), "models/players/%s/head_default.skin", modelName );
		ci->headSkin = trap_R_RegisterSkin( filename );
	
	}
	if ( !ci->legsSkin || !ci->torsoSkin || !ci->headSkin ) {
		return qfalse;
	}

	return qtrue;
}

/*
==========================
CG_RegisterClientModelname
==========================
*/
static qboolean CG_RegisterClientModelname( clientInfo_t *ci, const char *modelName, const char *skinName, const char *headModelName, const char *headSkinName ) {
	char		filename[MAX_QPATH];

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/lower.md3", modelName );
	ci->legsModel = trap_R_RegisterModel( filename );
	if ( !ci->legsModel ) {
		Com_Printf( "Failed to load model file %s\n", filename );
		return qfalse;
	}

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/upper.md3", modelName );
	ci->torsoModel = trap_R_RegisterModel( filename );
	if ( !ci->torsoModel ) {
		Com_Printf( "Failed to load model file %s\n", filename );
		return qfalse;
	}

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/head.md3", headModelName );
	ci->headModel = trap_R_RegisterModel( filename );
	if ( !ci->headModel ) {
		Com_Printf( "Failed to load model file %s\n", filename );
		return qfalse;
	}

	// if any skins failed to load, return failure
	if ( !CG_RegisterClientSkin( ci, modelName, skinName, headModelName, headSkinName ) ) {
		Com_Printf( "Failed to load skin file: %s/%s +%s/%s\n", modelName, skinName, headModelName, headSkinName );
		return qfalse;
	}

	// load the animations
	Com_sprintf( filename, sizeof( filename ), "models/players/%s/animation.cfg", modelName );
	if ( !CG_ParseAnimationFile( filename, ci ) ) {
		Com_Printf( "Failed to load animation file %s\n", filename );
		return qfalse;
	}

	Com_sprintf( filename, sizeof( filename ), "models/players/%s/icon_%s.tga", headModelName, headSkinName );
	ci->modelIcon = trap_R_RegisterShaderNoMip( filename );
	if ( !ci->modelIcon ) {
		Com_Printf( "Failed to load icon file: %s\n", filename );
		return qfalse;
	}

	return qtrue;
}

/*
====================
CG_ColorFromString
====================
*/
static void CG_ColorFromString( const char *v, vec3_t color ) {
	int val;

	VectorClear( color );

	val = atoi( v );

	if ( val < 1 || val > 7 ) {
		VectorSet( color, 1, 1, 1 );
		return;
	}

	if ( val & 1 ) {
		color[2] = 1.0f;
	}
	if ( val & 2 ) {
		color[1] = 1.0f;
	}
	if ( val & 4 ) {
		color[0] = 1.0f;
	}
}

/*
===================
CG_LoadClientInfo

Load it now, taking the disk hits.
This will usually be deferred to a safe time
===================
*/
void CG_LoadClientInfo( clientInfo_t *ci ) {
	const char	*dir, *fallback;
	int			i;
	const char	*s;
	int			clientNum;	
	CG_Printf("Registering model: [%s]/[%s] +[%s]/[%s]\n", ci->modelName, ci->skinName, ci->headModelName, ci->headSkinName);
	if ( !CG_RegisterClientModelname( ci, ci->modelName, ci->skinName, ci->headModelName, ci->headSkinName ) ) {
		if ( cg_buildScript.integer ) {
			CG_Error( "CG_RegisterClientModelname( %s/%s +%s/%s) failed", ci->modelName, ci->skinName, ci->headModelName, ci->headSkinName );
		}

		// fall back
		if ( cgs.gametype >= GT_TEAM ) {
			// keep skin name
			if ( !CG_RegisterClientModelname( ci, DEFAULT_MODEL, ci->skinName, DEFAULT_MODEL, ci->skinName ) ) 
			{
				CG_Error( "DEFAULT_MODEL / skin (%s/%s) failed to register", DEFAULT_MODEL, ci->skinName );
			}
		} 
		else 
		{			
			if ( !CG_RegisterClientModelname( ci, ci->modelName, "default", ci->modelName, "default" ) ) 
			{
				if ( !CG_RegisterClientModelname( ci, DEFAULT_MODEL, "default", DEFAULT_MODEL, "default" ) ) 
					CG_Error( "DEFAULT_MODEL (%s) failed to register", DEFAULT_MODEL );
			}
		}
	}

	// sounds
	dir = ci->modelName;
	fallback = DEFAULT_MODEL;

	for ( i = 0 ; i < MAX_CUSTOM_SOUNDS ; i++ ) {
		s = cg_customSoundNames[i];
		if ( !s ) {
			break;
		}
		ci->sounds[i] = trap_S_RegisterSound( va("sound/player/%s/%s", dir, s + 1) );
		if ( !ci->sounds[i] ) {
			ci->sounds[i] = trap_S_RegisterSound( va("sound/player/%s/%s", fallback, s + 1) );
		}
	}

	ci->deferred = qfalse;
	
	// reset any existing players and bodies, because they might be in bad
	// frames for this new model
	clientNum = ci - cgs.clientinfo;
	for ( i = 0 ; i < MAX_GENTITIES ; i++ ) {
		if ( cg_entities[i].currentState.clientNum == clientNum
			&& cg_entities[i].currentState.eType == ET_PLAYER ) {
			CG_ResetPlayerEntity( &cg_entities[i] );
		}
	}
}

/*
======================
CG_CopyClientInfoModel
======================
*/
static void CG_CopyClientInfoModel( clientInfo_t *from, clientInfo_t *to ) {
	VectorCopy( from->headOffset, to->headOffset );
	to->footsteps = from->footsteps;
	to->gender = from->gender;

	to->legsModel = from->legsModel;
	to->legsSkin = from->legsSkin;
	to->torsoModel = from->torsoModel;
	to->torsoSkin = from->torsoSkin;
	to->headModel = from->headModel;
	to->headSkin = from->headSkin;
	to->modelIcon = from->modelIcon;

	memcpy( to->animations, from->animations, sizeof( to->animations ) );
	memcpy( to->sounds, from->sounds, sizeof( to->sounds ) );
}

/*
======================
CG_ScanForExistingClientInfo
======================
*/
static qboolean CG_ScanForExistingClientInfo( clientInfo_t *ci ) {
	int		i;
	clientInfo_t	*match;

	for ( i = 0 ; i < cgs.maxclients ; i++ ) {
		match = &cgs.clientinfo[ i ];
		if ( !match->infoValid ) {
			continue;
		}
		if ( match->deferred ) {
			continue;
		}
		if ( !Q_stricmp( ci->modelName, match->modelName )
			&& !Q_stricmp( ci->skinName, match->skinName ) ) {
			// this clientinfo is identical, so use it's handles

			ci->deferred = qfalse;

			CG_CopyClientInfoModel( match, ci );

			return qtrue;
		}
	}

	// nothing matches, so defer the load
	return qfalse;
}

/*
======================
CG_SetDeferredClientInfo

We aren't going to load it now, so grab some other
client's info to use until we have some spare time.
======================
*/
static void CG_SetDeferredClientInfo( clientInfo_t *ci ) {
	// just load the damned info
	CG_LoadClientInfo( ci );
	
}


/*
======================
CG_NewClientInfo
======================
*/
void CG_NewClientInfo( int clientNum ) {
	clientInfo_t *ci;
	clientInfo_t newInfo;
	const char	*configstring;
	const char	*v;
	char		*slash;

	ci = &cgs.clientinfo[clientNum];

	configstring = CG_ConfigString( clientNum + CS_PLAYERS );
	if ( !configstring[0] ) {
		memset( ci, 0, sizeof( *ci ) );
		return;		// player just left
	}

	// build into a temp buffer so the defer checks can use
	// the old value
	memset( &newInfo, 0, sizeof( newInfo ) );

	// isolate the player's name
	v = Info_ValueForKey(configstring, "n");
	Q_strncpyz( newInfo.name, v, sizeof( newInfo.name ) );

	// colors
	v = Info_ValueForKey( configstring, "c1" );
	CG_ColorFromString( v, newInfo.color );

	// bot skill
	v = Info_ValueForKey( configstring, "skill" );
	newInfo.botSkill = atoi( v );

	// wins
	v = Info_ValueForKey( configstring, "w" );
	newInfo.wins = atoi( v );

	// losses
	v = Info_ValueForKey( configstring, "l" );
	newInfo.losses = atoi( v );

	// team
	v = Info_ValueForKey( configstring, "t" );
	newInfo.team = atoi( v );

	newInfo.head_explodey = 0; // this is important!
	// model
	v = Info_ValueForKey( configstring, "model" );
	/*if ( cg_forceModel.integer ) { // ## <HentaiAtWork> we don't do that joke anymore.
		// forcemodel makes everyone use a single model
		// to prevent load hitches
		char modelStr[MAX_QPATH];
		char *skin, *headskin, *headmodel;


		trap_Cvar_VariableStringBuffer( "model", modelStr, sizeof( modelStr ) );
		if ( ( skin = strchr( modelStr, '/' ) ) == NULL) {
			skin = "default";
		} else {
			*skin++ = 0;
		}
		if( ( headmodel = strchr(skin, ' ') ) == NULL)
		{
			headmodel = modelStr;
			headskin = skin;
		}
		else
		{
			*headmodel++ = 0;
			if ( ( headskin = strchr( headmodel, '/' ) ) == NULL) {
				headskin = "default";
			} else {
				*headskin++ = 0;
			}
			if(headmodel[0] == '*') // use default
				headmodel = modelStr;

		}

		Q_strncpyz( newInfo.skinName, skin, sizeof( newInfo.skinName ) );
		Q_strncpyz( newInfo.modelName, modelStr, sizeof( newInfo.modelName ) );

		Q_strncpyz( newInfo.headSkinName, headskin, sizeof( newInfo.headSkinName) );
		Q_strncpyz( newInfo.headModelName, headmodel, sizeof( newInfo.headModelName) );
		CG_Printf("model: %s skin: %s headmodel: %s headskin: %s\n", skin, modelStr, headskin, headmodel);
//		Q_strncpyz( newInfo.modelName, DEFAULT_MODEL, sizeof( newInfo.modelName ) );
//		Q_strncpyz( newInfo.skinName, "default", sizeof( newInfo.skinName ) );

		if ( cgs.gametype >= GT_TEAM ) {
			// keep skin name
			slash = strchr( v, '/' );
			if ( slash ) {
				Q_strncpyz( newInfo.skinName, slash + 1, sizeof( newInfo.skinName ) );
			}
		}
	} else {*/
		Q_strncpyz( newInfo.modelName, v, sizeof( newInfo.modelName ) );

		slash = strchr( newInfo.modelName, '/' );
		if ( !slash ) {
			// modelName didn not include a skin name
			Q_strncpyz( newInfo.skinName, "default", sizeof( newInfo.skinName ) );
		} else {
			Q_strncpyz( newInfo.skinName, slash + 1, sizeof( newInfo.skinName ) );
			// truncate modelName
			*slash = 0;
		}
		slash = strchr(newInfo.skinName, ' ');
		if( !slash ) {
			Q_strncpyz( newInfo.headModelName, newInfo.modelName, sizeof( newInfo.headModelName ) );
			Q_strncpyz( newInfo.headSkinName, newInfo.skinName, sizeof( newInfo.headSkinName ) );
			//CG_Printf("No special head");
		}
		else
		{
			*slash = 0;
			Q_strncpyz( newInfo.headModelName, slash + 1, sizeof( newInfo.headModelName ) );			
			slash = strchr( newInfo.headModelName, '/' );
			if ( !slash ) {
				// modelName did not include a skin name
				Q_strncpyz( newInfo.headSkinName, "default", sizeof( newInfo.headSkinName ) );
			} else {
				Q_strncpyz( newInfo.headSkinName, slash + 1, sizeof( newInfo.headSkinName ) );
				// truncate modelName
				*slash = 0;
			}
			if(newInfo.headModelName[0] == '*') // use default
				Q_strncpyz( newInfo.headModelName, newInfo.modelName, sizeof( newInfo.headModelName ) );	


		}

	//}

	// scan for an existing clientinfo that matches this modelname
	// so we can avoid loading checks if possible
	if ( !CG_ScanForExistingClientInfo( &newInfo ) ) {
		qboolean	forceDefer;

		forceDefer = trap_MemoryRemaining() < 4000000;

		// if we are defering loads, just have it pick the first valid
		if ( forceDefer || ( cg_deferPlayers.integer && !cg_buildScript.integer && !cg.loading ) ) {
			// keep whatever they had if it won't violate team skins
			if ( ci->infoValid && 
				( cgs.gametype < GT_TEAM || !Q_stricmp( newInfo.skinName, ci->skinName ) ) ) {
				CG_CopyClientInfoModel( ci, &newInfo );
				newInfo.deferred = qtrue;
			} else {
				// use whatever is available
				CG_SetDeferredClientInfo( &newInfo );
			}
			// if we are low on memory, leave them with this model
			if ( forceDefer ) {
				CG_Printf( "Memory is low.  Using deferred model.\n" );
				newInfo.deferred = qfalse;
			}
		} else {
			CG_LoadClientInfo( &newInfo );
		}
	}

	// replace whatever was there with the new one
	newInfo.infoValid = qtrue;
	*ci = newInfo;
}



/*
======================
CG_LoadDeferredPlayers

Called each frame when a player is dead
and the scoreboard is up
so deferred players can be loaded
======================
*/
void CG_LoadDeferredPlayers( void ) {
	int		i;
	clientInfo_t	*ci;

	// scan for a deferred player to load
	for ( i = 0, ci = cgs.clientinfo ; i < cgs.maxclients ; i++, ci++ ) {
		if ( ci->infoValid && ci->deferred ) {
			// if we are low on memory, leave it deferred
			if ( trap_MemoryRemaining() < 4000000 ) {
				CG_Printf( "Memory is low.  Using deferred model.\n" );
				ci->deferred = qfalse;
				continue;
			}
			CG_LoadClientInfo( ci );
//			break;
		}
	}
}

/*
=============================================================================

PLAYER ANIMATION

=============================================================================
*/


/*
===============
CG_SetLerpFrameAnimation

may include ANIM_TOGGLEBIT
===============
*/
static void CG_SetLerpFrameAnimation( clientInfo_t *ci, lerpFrame_t *lf, int newAnimation ) {
	animation_t	*anim;
	int savestate = newAnimation & ANIM_CLIMBWALLS;
	lf->animationNumber = newAnimation;
	newAnimation &= ~(ANIM_TOGGLEBIT | ANIM_CLIMBWALLS);

	if ( newAnimation < 0 || newAnimation >= MAX_ANIMATIONS ) {
		CG_Error( "Bad animation number: %i", newAnimation );
	}

	anim = &ci->animations[ newAnimation];

	lf->animation = anim;
	lf->animationTime = lf->frameTime + anim->initialLerp;
	lf->animationNumber |= savestate;
	
	if ( cg_debugAnim.integer ) {
		CG_Printf( "Anim: %i\n", newAnimation );
	}
}

/*
===============
CG_RunLerpFrame

Sets cg.snap, cg.oldFrame, and cg.backlerp
cg.time should be between oldFrameTime and frameTime after exit
===============
*/
static qboolean CG_RunLerpFrame( clientInfo_t *ci, lerpFrame_t *lf, int newAnimation, float speedScale ) {
	int			f, add = 0;
	animation_t	*anim;
	float oldlerp, advance;
	qboolean looped = qfalse;
	// debugging tool to get no animations
	if ( cg_animSpeed.value == 0) {
		lf->oldFrame = lf->frame = lf->backlerp = 0;
		return qfalse;
	}

	// see if the animation sequence is switching
	f = newAnimation & ~(ANIM_TOGGLEBIT | ANIM_CLIMBWALLS);

	{// ## Hentai ## - theoretically, this should do scaled looping better?
		
		if(newAnimation < 0)
		{ // wings
			if ( newAnimation != lf->animationNumber || !lf->animation ) {
				lf->animation = &wingsAnim[newAnimation + 4];
				lf->animationTime = lf->frameTime + lf->animation->initialLerp;
				lf->animationNumber = newAnimation;
				
				lf->oldFrameTime = cg.time + lf->animation->initialLerp;
				lf->frame = lf->animation->firstFrame;
			}
		}
		else if ( newAnimation != lf->animationNumber || !lf->animation ) {
			CG_SetLerpFrameAnimation( ci, lf, newAnimation );
			//lf->lerpFrame = 0;			
			lf->oldFrameTime = cg.time + lf->animation->initialLerp;
			lf->frame = lf->animation->firstFrame;
			if(f == LEGS_RUN || f == LEGS_BACK )
			{
				ci->lstep = lf->animation->firstFrame + lf->animation->numFrames - lf->animation->loopFrames;
				ci->rstep = lf->animation->firstFrame + lf->animation->numFrames - lf->animation->loopFrames / 2;
			}
			else if(f == LEGS_SWIM)
			{
				ci->lstep = ci->rstep = lf->animation->firstFrame + lf->animation->numFrames - lf->animation->loopFrames;
			}
		}
		anim = lf->animation;
		oldlerp = lf->lerpFrame;
		advance = speedScale * cg_animSpeed.value * (cg.time - lf->oldFrameTime) / anim->frameLerp;
		if(advance < 0)
		{ // we're between animations, so slow down a little to prevent a jerky 'snap'
			advance = 0.5 * speedScale * cg_animSpeed.value * (cg.time - lf->oldFrameTime + lf->animation->initialLerp) / anim->initialLerp;
		}
		lf->oldFrameTime = cg.time;// - lf->backlerp * anim->frameLerp;
		lf->lerpFrame += advance;
		
		add = floor(lf->lerpFrame) - floor(oldlerp);
		if(add)
		{
			
			// if we have passed the current frame, move it to
			// oldFrame and calculate a new frame	
			lf->oldFrame = lf->frame;
			
			f = lf->frame - anim->firstFrame;
			f += add;
			if(f >= anim->numFrames)
			{
				looped = qtrue;
				f -= anim->numFrames;
				if ( anim->loopFrames ) {
					f %= anim->loopFrames;
					f += anim->numFrames - anim->loopFrames;
					lf->lerpFrame -= floor(lf->lerpFrame); // keep fractional part to prevent lerping errors
					//lf->oldFrameTime = cg.time + lf->backlerp * anim->frameLerp;
				} else {
					f = anim->numFrames - 1;
					lf->frameTime = cg.time;
				}
			}
			lf->frame = anim->firstFrame + f;
					
		}
		lf->backlerp = ceil(lf->lerpFrame) - lf->lerpFrame;
		if(lf->frame == lf->oldFrame)
			lf->backlerp = 0;
		if ( cg_debugAnim.integer ) {
			CG_Printf( "speed: %f cycle: %f - advanced by %f - backlerp: %f - frame: %d -> %d\n", speedScale * 1000, lf->lerpFrame * 1000, advance * 1000, lf->backlerp * 1000, lf->oldFrame, lf->frame );
			//lf->oldFrame = lf->frame = lf->backlerp = 0;
		}
		
	}
	return add;
}


/*
===============
CG_ClearLerpFrame
===============
*/
static void CG_ClearLerpFrame( clientInfo_t *ci, lerpFrame_t *lf, int animationNumber ) {
	lf->frameTime = lf->oldFrameTime = cg.time;
	CG_SetLerpFrameAnimation( ci, lf, animationNumber );
	lf->oldFrame = lf->frame = lf->animation->firstFrame;
}

localEntity_t *CG_LaunchGib( vec3_t origin, vec3_t velocity, qhandle_t hModel );
/*
===============
CG_PlayerAnimation
===============
*/
static void CG_PlayerAnimation( centity_t *cent, int *legsOld, int *legs, float *legsBackLerp,
						int *torsoOld, int *torso, float *torsoBackLerp ) {
	clientInfo_t	*ci;
	int				clientNum, anim, wings_anim;
	float			xyspeed, speedScale;
	vec3_t dir;
	if(cent->oldorigin_time  < cg.time)// && (cg.time >= cent->pe.legs.frameTime))
	{ // only do this once per legframe
		float dx, dt;
		VectorSubtract(cent->lerpOrigin, cent->oldorigin, dir);
		dx = VectorLength(dir);
		if(dx)
		{
			VectorCopy(dir, cent->vel);
			VectorNormalize(cent->vel);
		}
		dt = (cg.time - cent->oldorigin_time) / 1000.0; // this is to scale down by 10
		cent->oldspeed = cent->speed;
		cent->speed = dx / dt;
		VectorCopy(cent->lerpOrigin, cent->oldorigin);
		cent->oldorigin_time = cg.time;		
	}
	
	//xyspeed = VectorLength(cent->currentState.pos.trDelta);//(cent->currentState.pos.trDelta[0] * cent->currentState.pos.trDelta[0] + cent->currentState.pos.trDelta[1] * cent->currentState.pos.trDelta[1]);
	clientNum = cent->currentState.clientNum;

	if ( cg_noPlayerAnims.integer ) {
		*legsOld = *legs = *torsoOld = *torso = 0;
		return;
	}
	
	anim = cent->currentState.legsAnim & ~(ANIM_TOGGLEBIT | ANIM_CLIMBWALLS);
	xyspeed = (cent->speed + cent->oldspeed) / 2;
	switch (anim)
	{
	case LEGS_WALKCR:
		speedScale = xyspeed / 40.0;
		wings_anim = WINGS_FLAPB;
		break;
	case LEGS_BACK:
	case LEGS_WALK:
	case LEGS_SWIM:
		speedScale = xyspeed / 80.0;
		wings_anim = WINGS_FLAP;
		break;
	case LEGS_RUN:
		speedScale = xyspeed / 160.0;
		wings_anim = WINGS_FLAP;
		break;
		
	case LEGS_IDLE:
	case LEGS_IDLECR:
		wings_anim = WINGS_SPREAD;
		if ( cent->currentState.powerups & ( 1 << PW_HASTE ) ) 
			speedScale = 1;
		else 
			speedScale = 0.5;
		break;

	case LEGS_TURN:	
	case LEGS_JUMP:
	default:
	//	cent->oldorigin_time = cg.time;
	//	cent->speed = 0;
	//	VectorCopy(cent->currentState.origin, cent->oldorigin);		
		wings_anim = WINGS_SPREAD;
		if ( cent->currentState.powerups & ( 1 << PW_HASTE ) ) 
			speedScale = 1.5;
		else 
			speedScale = 1;
		break;
	}
	//CG_Printf("%f [%f]\n", cent->speed, speedScale * 100);
	ci = &cgs.clientinfo[ clientNum ];

	if(cent->currentState.powerups & (1 << PW_FLIGHT))
	{		
		CG_RunLerpFrame( ci, &cent->pe.wings, wings_anim, xyspeed / 160.0 + 0.667);
		CG_RunLerpFrame( ci, &cent->pe.legs, cent->currentState.legsAnim, 0.05 + random() * 0.1 );
	}
	else if ( cent->pe.legs.yawing && ( cent->currentState.legsAnim & ~(ANIM_TOGGLEBIT | ANIM_CLIMBWALLS ) ) == LEGS_IDLE ) 
	{// do the shuffle turn frames locally
		CG_RunLerpFrame( ci, &cent->pe.legs, LEGS_TURN, speedScale );
	} 
	else 
	{		
		CG_RunLerpFrame( ci, &cent->pe.legs, cent->currentState.legsAnim, speedScale );
	}
	
	*legsOld = cent->pe.legs.oldFrame;
	*legs = cent->pe.legs.frame;
	*legsBackLerp = cent->pe.legs.backlerp;
	
	if ( cent->currentState.powerups & ( 1 << PW_HASTE ) ) 
		speedScale = 1.5;
	else 
		speedScale = 1;
	
	CG_RunLerpFrame( ci, &cent->pe.torso, cent->currentState.torsoAnim, speedScale );

	*torsoOld = cent->pe.torso.oldFrame;
	*torso = cent->pe.torso.frame;
	*torsoBackLerp = cent->pe.torso.backlerp;
}

/*
=============================================================================

PLAYER ANGLES

=============================================================================
*/

/*
==================
CG_SwingAngles
==================
*/
static void CG_SwingAngles( float destination, float swingTolerance, float clampTolerance,
					float speed, float *angle, qboolean *swinging ) {
	float	swing;
	float	move;
	float	scale;

	if ( !*swinging ) {
		// see if a swing should be started
		swing = AngleSubtract( *angle, destination );
		if ( swing > swingTolerance || swing < -swingTolerance ) {
			*swinging = qtrue;
		}
	}

	if ( !*swinging ) {
		return;
	}
	
	// modify the speed depending on the delta
	// so it doesn't seem so linear
	swing = AngleSubtract( destination, *angle );
	scale = fabs( swing );
	if ( scale < swingTolerance * 0.5 ) {
		scale = 0.5;
	} else if ( scale < swingTolerance ) {
		scale = 1.0;
	} else {
		scale = 2.0;
	}

	// swing towards the destination angle
	if ( swing >= 0 ) {
		move = cg.frametime * scale * speed;
		if ( move >= swing ) {
			move = swing;
			*swinging = qfalse;
		}
		*angle = AngleMod( *angle + move );
	} else if ( swing < 0 ) {
		move = cg.frametime * scale * -speed;
		if ( move <= swing ) {
			move = swing;
			*swinging = qfalse;
		}
		*angle = AngleMod( *angle + move );
	}

	// clamp to no more than tolerance
	swing = AngleSubtract( destination, *angle );
	if ( swing > clampTolerance ) {
		*angle = AngleMod( destination - (clampTolerance - 1) );
	} else if ( swing < -clampTolerance ) {
		*angle = AngleMod( destination + (clampTolerance - 1) );
	}
}

/*
=================
CG_AddPainTwitch
=================
*/
static void CG_AddPainTwitch( centity_t *cent, vec3_t torsoAngles ) {
	int		t;
	float	f;

	t = cg.time - cent->pe.painTime;
	if ( t >= PAIN_TWITCH_TIME ) {
		return;
	}

	f = 1.0 - (float)t / PAIN_TWITCH_TIME;

	if ( cent->pe.painDirection ) {
		torsoAngles[ROLL] += 20 * f;
	} else {
		torsoAngles[ROLL] -= 20 * f;
	}
}


/*
===============
CG_PlayerAngles

Handles seperate torso motion

  legs pivot based on direction of movement

  head always looks exactly at cent->lerpAngles

  if motion < 20 degrees, show in head only
  if < 45 degrees, also show in torso
===============
*/
static void CG_PlayerAngles( centity_t *cent, vec3_t legs[3], vec3_t torso[3], vec3_t head[3] ) {
	vec3_t		legsAngles, torsoAngles, headAngles;
	float		dest;
	static	int	movementOffsets[8] = { 0, 22, 45, -22, 0, 22, -45, -22 };
	vec3_t		velocity;
	float		speed;
	int			dir;
	VectorCopy( cent->lerpAngles, headAngles );
	headAngles[YAW] = AngleMod( headAngles[YAW] );
	VectorClear( legsAngles );
	VectorClear( torsoAngles );

	// --------- yaw -------------

	// allow yaw to drift a bit
	if ( ( cent->currentState.legsAnim & ~( ANIM_TOGGLEBIT | ANIM_CLIMBWALLS ) ) != LEGS_IDLE 
		|| ( (cent->currentState.torsoAnim & ~( ANIM_TOGGLEBIT | ANIM_CLIMBWALLS ) ) != TORSO_STAND && (cent->currentState.torsoAnim & ~( ANIM_TOGGLEBIT | ANIM_CLIMBWALLS ) ) != TORSO_STAND2  ) ) {
		// if not standing still, always point all in the same direction
		cent->pe.torso.yawing = qtrue;	// always center
		cent->pe.torso.pitching = qtrue;	// always center
		cent->pe.legs.yawing = qtrue;	// always center
	}

	// adjust legs for movement dir
	if ( cent->currentState.eFlags & EF_DEAD ) {
		// don't let dead bodies twitch
		dir = 0;
	} else {
		dir = cent->currentState.angles2[YAW];
		if ( dir < 0 || dir > 7 ) {
			CG_Error( "Bad player movement angle" );
		}
	}
	legsAngles[YAW] = headAngles[YAW] + movementOffsets[ dir ];

	torsoAngles[YAW] = headAngles[YAW] + 0.25 * movementOffsets[ dir ];

	// torso
	if(cent->currentState.clientNum == cg.snap->ps.clientNum && cg.zoomed)
	{
		CG_SwingAngles( torsoAngles[YAW], 45, 45, 0, &cent->pe.torso.yawAngle, &cent->pe.torso.yawing );
	}
	else
	{
		CG_SwingAngles( torsoAngles[YAW], 25, 90, cg_swingSpeed.value, &cent->pe.torso.yawAngle, &cent->pe.torso.yawing );
	}
	CG_SwingAngles( legsAngles[YAW], 40, 90, cg_swingSpeed.value, &cent->pe.legs.yawAngle, &cent->pe.legs.yawing );
	torsoAngles[YAW] = cent->pe.torso.yawAngle;
	legsAngles[YAW] = cent->pe.legs.yawAngle;
	if(cent->currentState.groundEntityNum == ENTITYNUM_NONE)
	{
		if ( headAngles[PITCH] > 180 ) {
			legsAngles[PITCH] = (-360 + headAngles[PITCH]) * 1.5;
			if(legsAngles[PITCH] < -180)
				legsAngles[PITCH] = -180;
		} else {
			legsAngles[PITCH] = headAngles[PITCH];
			//if(legsAngles[PITCH] > 180)
			//	legsAngles[PITCH] = 180;
		}
	
	}

	// --------- pitch -------------

	// only show a fraction of the pitch angle in the torso
	if(headAngles[PITCH] > 180)
		headAngles[PITCH] -= 360;
	if(fabs(headAngles[PITCH]) > 90)
	{
		dest = headAngles[PITCH];
	//}
	//else if ( headAngles[PITCH] > 180 ) {
	//	dest = (-360 + headAngles[PITCH]) * 0.75;
	} else {
		dest = headAngles[PITCH] * 0.75;
	}
	if(cent->currentState.clientNum == cg.snap->ps.clientNum && cg.zoomed)
	{
		if(headAngles[PITCH] < 45 || headAngles[PITCH] > 180 )
			dest = 0;
		else 
			dest = (headAngles[PITCH] - 45) * 0.75;
	}

	CG_SwingAngles( dest, 15, 30, 0.1, &cent->pe.torso.pitchAngle, &cent->pe.torso.pitching );
	torsoAngles[PITCH] = cent->pe.torso.pitchAngle;

	// --------- roll -------------


	// lean towards the direction of travel
	VectorCopy( cent->currentState.pos.trDelta, velocity );
	speed = VectorNormalize( velocity );
	if ( speed ) {
		vec3_t	axis[3];
		float	side;

		speed *= 0.05;

		AnglesToAxis( legsAngles, axis );
		side = speed * DotProduct( velocity, axis[1] );
		legsAngles[ROLL] -= side;

		side = speed * DotProduct( velocity, axis[0] );
		legsAngles[PITCH] += side;
	}

	// pain twitch
	CG_AddPainTwitch( cent, torsoAngles );	
	
	AnglesSubtract( headAngles, torsoAngles, headAngles );
	AnglesSubtract( torsoAngles, legsAngles, torsoAngles );

	AnglesToAxis( legsAngles, legs );
	AnglesToAxis( torsoAngles, torso );
	AnglesToAxis( headAngles, head );
}


//==========================================================================

/*
===============
CG_HasteTrail
===============
*/
static void CG_HasteTrail( centity_t *cent ) {
	localEntity_t	*smoke;
	vec3_t			origin;
	int				anim;

	if ( cent->trailTime > cg.time ) {
		return;
	}
	anim = cent->pe.legs.animationNumber & ~( ANIM_TOGGLEBIT | ANIM_CLIMBWALLS );
	if ( anim != LEGS_RUN && anim != LEGS_BACK ) {
		return;
	}

	cent->trailTime += 100;
	if ( cent->trailTime < cg.time ) {
		cent->trailTime = cg.time;
	}

	VectorCopy( cent->lerpOrigin, origin );
	origin[2] -= 16;

	smoke = CG_SmokePuff( origin, vec3_origin, 
				  8, 
				  1, 1, 1, 1,
				  500, 
				  cg.time,
				  0, 
				  cgs.media.hastePuffShader );

	// use the optimized local entity add
	smoke->leType = LE_SCALE_FADE;
}

/*
===============
CG_TrailItem
===============
*/
static void CG_TrailItem( centity_t *cent, qhandle_t hModel ) {
	refEntity_t		ent;
	vec3_t			angles;
	vec3_t			axis[3];

	VectorCopy( cent->lerpAngles, angles );
	angles[PITCH] = 0;
	angles[ROLL] = 0;
	AnglesToAxis( angles, axis );

	memset( &ent, 0, sizeof( ent ) );
	VectorMA( cent->lerpOrigin, -16, axis[0], ent.origin );
	ent.origin[2] += 16;
#if 0
	VectorScale( cg.autoAxis[0], 0.75, ent.axis[0] );
	VectorScale( cg.autoAxis[1], 0.75, ent.axis[1] );
	VectorScale( cg.autoAxis[2], 0.75, ent.axis[2] );
#else
	angles[YAW] += 90;
	AnglesToAxis( angles, ent.axis );
#endif

	ent.hModel = hModel;
	trap_R_AddRefEntityToScene( &ent );
}


/*
===============
CG_PlayerPowerups
===============
*/
static void CG_PlayerPowerups( centity_t *cent ) {
	int		powerups;

	powerups = cent->currentState.powerups;
	if ( !powerups ) {
		return;
	}

	// quad gives a dlight
	if ( powerups & ( 1 << PW_QUAD ) ) {
		trap_R_AddLightToScene( cent->lerpOrigin, 200 + (rand()&31), 0.2, 0.2, 1 );
	}

	// ## Hentai - these are now all handled in different ways
	/*
	 flight plays a looped sound
	if ( powerups & ( 1 << PW_FLIGHT ) ) {
		trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, cgs.media.flightSound );
	}

	// redflag
	if ( powerups & ( 1 << PW_REDFLAG ) ) {
		CG_TrailItem( cent, cgs.media.redFlagModel );
		trap_R_AddLightToScene( cent->lerpOrigin, 200 + (rand()&31), 1, 0.2, 0.2 );
	}

	// blueflag
	if ( powerups & ( 1 << PW_BLUEFLAG ) ) {
		CG_TrailItem( cent, cgs.media.blueFlagModel );
		trap_R_AddLightToScene( cent->lerpOrigin, 200 + (rand()&31), 0.2, 0.2, 1 );
	}

	// haste leaves smoke trails
	if ( powerups & ( 1 << PW_HASTE ) ) {
		CG_HasteTrail( cent );
	}
	*/
}


/*
===============
CG_PlayerFloatSprite

Float a sprite over the player's head
===============
*/
static void CG_PlayerFloatSprite( centity_t *cent, qhandle_t shader ) {
	int				rf;
	refEntity_t		ent;

	if ( cent->currentState.number == cg.snap->ps.clientNum && !cg.renderingThirdPerson ) {
		rf = RF_THIRD_PERSON;		// only show in mirrors
	} else {
		rf = 0;
	}

	memset( &ent, 0, sizeof( ent ) );
	VectorCopy( cent->lerpOrigin, ent.origin );
	ent.origin[2] += 48;
	ent.reType = RT_SPRITE;
	ent.customShader = shader;
	ent.radius = 10;
	ent.renderfx = rf;
	ent.shaderRGBA[0] = 255;
	ent.shaderRGBA[1] = 255;
	ent.shaderRGBA[2] = 255;
	ent.shaderRGBA[3] = 255;
	trap_R_AddRefEntityToScene( &ent );
}



/*
===============
CG_PlayerSprites

Float sprites over the player's head
===============
*/
static void CG_PlayerSprites( centity_t *cent ) {
	int		team;
	
	
	if ( cgs.clientinfo[ cent->currentState.clientNum ].deferred) {
		CG_PlayerFloatSprite( cent, cgs.media.deferShader );		
	}

	if ( cent->currentState.eFlags & EF_CONNECTION ) {
		CG_PlayerFloatSprite( cent, cgs.media.connectionShader );
		return;
	}

	if ( cent->currentState.eFlags & EF_TALK ) {
		CG_PlayerFloatSprite( cent, cgs.media.balloonShader );
		return;
	}

	if ( (cent->currentState.eFlags & EF_AWARD_IMPRESSIVE) && !(cent->currentState.eFlags & EF_DEAD)) { // impressive when dead means head blown off
		CG_PlayerFloatSprite( cent, cgs.media.medalImpressive );
		return;
	}

	if ( cent->currentState.eFlags & EF_AWARD_EXCELLENT ) {
		CG_PlayerFloatSprite( cent, cgs.media.medalExcellent );
		return;
	}

	if ( cent->currentState.eFlags & EF_AWARD_GAUNTLET ) {
		CG_PlayerFloatSprite( cent, cgs.media.medalGauntlet );
		return;
	}

	team = cgs.clientinfo[ cent->currentState.clientNum ].team;
	if ( !(cent->currentState.eFlags & EF_DEAD) && 
		cg.snap->ps.persistant[PERS_TEAM] == team &&
		cgs.gametype >= GT_TEAM) {
		CG_PlayerFloatSprite( cent, cgs.media.friendShader );
		return;
	}
}

/*
===============
CG_PlayerShadow

Returns the Z component of the surface being shadowed

  should it return a full plane instead of a Z?
===============
*/
#define	SHADOW_DISTANCE		128
static qboolean CG_PlayerShadow( centity_t *cent, float *shadowPlane ) {
	vec3_t		end, mins = {-15, -15, 0}, maxs = {15, 15, 2};
	trace_t		trace;
	float		alpha;

	*shadowPlane = 0;

	if ( cg_shadows.integer == 0 ) {
		return qfalse;
	}

	// no shadows when invisible
	if ( cent->currentState.powerups & ( 1 << PW_INVIS ) ) {
		return qfalse;
	}

	// send a trace down from the player to the ground
	VectorCopy( cent->lerpOrigin, end );
	end[2] -= SHADOW_DISTANCE;

	trap_CM_BoxTrace( &trace, cent->lerpOrigin, end, mins, maxs, 0, MASK_PLAYERSOLID );

	// no shadow if too high
	if ( trace.fraction == 1.0 ) {
		return qfalse;
	}

	*shadowPlane = trace.endpos[2] + 1;

	if ( cg_shadows.integer != 1 ) {	// no mark for stencil or projection shadows
		return qtrue;
	}

	// fade the shadow out with height
	alpha = 1.0 - trace.fraction;

	// add the mark as a temporary, so it goes directly to the renderer
	// without taking a spot in the cg_marks array
	CG_ImpactMark( cgs.media.shadowMarkShader, trace.endpos, trace.plane.normal, 
		cent->pe.legs.yawAngle, alpha,alpha,alpha,1, qfalse, 24, qtrue );

	return qtrue;
}


/*
===============
CG_PlayerSplash

Draw a mark at the water surface
===============
*/
static void CG_PlayerSplash( centity_t *cent ) {
	vec3_t		start, end;
	trace_t		trace;
	int			contents;
	polyVert_t	verts[4];

	if ( !cg_shadows.integer ) {
		return;
	}

	VectorCopy( cent->lerpOrigin, end );
	end[2] -= 24;

	// if the feet aren't in liquid, don't make a mark
	// this won't handle moving water brushes, but they wouldn't draw right anyway...
	contents = trap_CM_PointContents( end, 0 );
	if ( !( contents & ( CONTENTS_WATER | CONTENTS_SLIME | CONTENTS_LAVA ) ) ) {
		return;
	}

	VectorCopy( cent->lerpOrigin, start );
	start[2] += 32;

	// if the head isn't out of liquid, don't make a mark
	contents = trap_CM_PointContents( start, 0 );
	if ( contents & ( CONTENTS_SOLID | CONTENTS_WATER | CONTENTS_SLIME | CONTENTS_LAVA ) ) {
		return;
	}

	// trace down to find the surface
	trap_CM_BoxTrace( &trace, start, end, NULL, NULL, 0, ( CONTENTS_WATER | CONTENTS_SLIME | CONTENTS_LAVA ) );

	if ( trace.fraction == 1.0 ) {
		return;
	}

	// create a mark polygon
	VectorCopy( trace.endpos, verts[0].xyz );
	verts[0].xyz[0] -= 32;
	verts[0].xyz[1] -= 32;
	verts[0].st[0] = 0;
	verts[0].st[1] = 0;
	verts[0].modulate[0] = 255;
	verts[0].modulate[1] = 255;
	verts[0].modulate[2] = 255;
	verts[0].modulate[3] = 255;

	VectorCopy( trace.endpos, verts[1].xyz );
	verts[1].xyz[0] -= 32;
	verts[1].xyz[1] += 32;
	verts[1].st[0] = 0;
	verts[1].st[1] = 1;
	verts[1].modulate[0] = 255;
	verts[1].modulate[1] = 255;
	verts[1].modulate[2] = 255;
	verts[1].modulate[3] = 255;

	VectorCopy( trace.endpos, verts[2].xyz );
	verts[2].xyz[0] += 32;
	verts[2].xyz[1] += 32;
	verts[2].st[0] = 1;
	verts[2].st[1] = 1;
	verts[2].modulate[0] = 255;
	verts[2].modulate[1] = 255;
	verts[2].modulate[2] = 255;
	verts[2].modulate[3] = 255;

	VectorCopy( trace.endpos, verts[3].xyz );
	verts[3].xyz[0] += 32;
	verts[3].xyz[1] -= 32;
	verts[3].st[0] = 1;
	verts[3].st[1] = 0;
	verts[3].modulate[0] = 255;
	verts[3].modulate[1] = 255;
	verts[3].modulate[2] = 255;
	verts[3].modulate[3] = 255;

	trap_R_AddPolyToScene( cgs.media.wakeMarkShader, 4, verts );
}



/*
===============
CG_AddRefEntityWithPowerups

Adds a piece with modifications or duplications for powerups
Also called by CG_Missile for quad rockets, but nobody can tell...
===============
*/
void CG_AddRefEntityWithPowerups( refEntity_t *ent, int powerups, int team ) {

	if ( powerups & ( 1 << PW_INVIS ) ) {
		
		if(ent->shaderRGBA[3] > 0)
			ent->customShader = cgs.media.invisShader2;
		else
			ent->customShader = cgs.media.invisShader;
		trap_R_AddRefEntityToScene( ent );
	} else {
		
		if(powerups & (1 << PW_HASTE) )
		{
			localEntity_t *le;
			refEntity_t *re;
		
			le = CG_AllocLocalEntity();
			re = &le->refEntity;
	
			le->leType = LE_FADE_RGB;
			le->startTime = cg.time;
			le->endTime = cg.time + 1000 * random();
			le->lifeRate = 1.0 / ( le->endTime - le->startTime );
			memcpy(re, ent, sizeof(refEntity_t));
			re->renderfx = 0;
			re->customSkin = 0;
			re->customShader = cgs.media.invisShader2;
			le->color[0] = 1;
			le->color[1] = 0;
			le->color[2] = 0;
			le->color[3] = 1;

			
		}
		if ( powerups & ( 1 << PW_ARMOR ) ) 
			ent->customShader = cgs.media.armorShader;
		trap_R_AddRefEntityToScene( ent );

		if ( powerups & ( 1 << PW_QUAD ) ) 
		{
			if (team == TEAM_RED)
				ent->customShader = cgs.media.redQuadShader;
			else
				ent->customShader = cgs.media.quadShader;
			trap_R_AddRefEntityToScene( ent );
		}
		if ( powerups & ( 1 << PW_REGEN ) ) {
			if ( ( ( cg.time / 100 ) % 10 ) == 1 ) {
				ent->customShader = cgs.media.regenShader;
				trap_R_AddRefEntityToScene( ent );
			}
		}
		if ( powerups & ( 1 << PW_BATTLESUIT ) ) {
			ent->customShader = cgs.media.battleSuitShader;
			trap_R_AddRefEntityToScene( ent );
		}
	}
}

/*
===============
CG_Player
===============
*/

void CG_Corpse( centity_t *cent ) {
	clientInfo_t	*ci, realci;
	refEntity_t		legs;
	refEntity_t		torso;
	refEntity_t		head;
	int				clientNum;
	int				renderfx;
	qboolean		shadow;
	float			shadowPlane;
	
	if(cent->CorpseIndex <= 0)
		cent->CorpseIndex = CG_ClientInfoForCorpse(cent->currentState.modelindex);
	clientNum = cent->CorpseIndex;
	ci = &realci;
	// ## Hentai ##
	
	
	memset(ci, 0, sizeof(realci));
	memcpy(realci.animations, corpseinfo[clientNum].animations, sizeof(realci.animations));
	realci.legsModel = corpseinfo[clientNum].legsModel;
	realci.legsSkin = corpseinfo[clientNum].legsSkin;
	realci.torsoModel = corpseinfo[clientNum].torsoModel;
	realci.torsoSkin = corpseinfo[clientNum].torsoSkin;
	realci.headModel = corpseinfo[clientNum].headModel;
	realci.headSkin = corpseinfo[clientNum].headSkin;

	memset( &legs, 0, sizeof(legs) );
	memset( &torso, 0, sizeof(torso) );
	memset( &head, 0, sizeof(head) );

	if(!realci.legsModel)
	{
		cent->CorpseIndex = -1;
		return;
	}
	
	// get the rotation information
	AnglesToAxis(cent->currentState.angles, legs.axis);
	AxisClear(torso.axis);
	AxisClear(head.axis);
	
	CG_RunLerpFrame( ci, &cent->pe.legs, cent->currentState.legsAnim, 0);

	
	legs.oldframe = cent->pe.legs.oldFrame;
	legs.frame = cent->pe.legs.frame;
	legs.backlerp = cent->pe.legs.backlerp;

	CG_RunLerpFrame( ci, &cent->pe.torso, cent->currentState.torsoAnim, 0);

	torso.oldframe = cent->pe.torso.oldFrame;
	torso.frame = cent->pe.torso.frame;
	torso.backlerp = cent->pe.torso.backlerp;

	
	// add the shadow
	shadow = CG_PlayerShadow( cent, &shadowPlane );

	// add a water splash if partially in and out of water
	CG_PlayerSplash( cent );

	// get the player model information
	renderfx = 0;
	if ( cg_shadows.integer == 3 && shadow ) {
		renderfx |= RF_SHADOW_PLANE;
	}
	renderfx |= RF_LIGHTING_ORIGIN;			// use the same origin for all

	//
	// add the legs
	//
	legs.hModel = ci->legsModel;
	legs.customSkin = ci->legsSkin;
	VectorCopy( cent->lerpOrigin, legs.origin );

	VectorCopy( cent->lerpOrigin, legs.lightingOrigin );
	legs.shadowPlane = shadowPlane;
	legs.renderfx = renderfx;
	VectorCopy (legs.origin, legs.oldorigin);	// don't positionally lerp at all

	
	
	trap_R_AddRefEntityToScene( &legs );

	// if the model failed, allow the default nullmodel to be displayed
	if (!legs.hModel) {
		cent->CorpseIndex = -1;// use skeleton on next frame [this frame will show nullmodel]		
		return;
	}

	//
	// add the torso
	//
	torso.hModel = ci->torsoModel;
	if (!torso.hModel) {
		return;
	}

	if(cent->currentState.powerups & PW_ARMOR)
		torso.customShader = trap_R_RegisterShader("powerups/Armor");
	else
		torso.customSkin = ci->torsoSkin;

	VectorCopy( cent->lerpOrigin, torso.lightingOrigin );
	CG_PositionRotatedEntityOnTag( &torso, &legs, ci->legsModel, "tag_torso");

	
	torso.shadowPlane = shadowPlane;
	torso.renderfx = renderfx;

	trap_R_AddRefEntityToScene( &torso );

	//
	// add the head
	//
	
	head.hModel = ci->headModel;
	if (!head.hModel) {
		return;
	}
	head.customSkin = ci->headSkin;
	VectorCopy( cent->lerpOrigin, head.lightingOrigin );
	
	CG_PositionRotatedEntityOnTag( &head, &torso, ci->torsoModel, "tag_head");
	if(cent->currentState.eFlags & EF_AWARD_IMPRESSIVE)
	{
		localEntity_t *le = CG_SpurtBlood(head.origin, head.axis[2], (rand() & 10 )+ 1);
		le->leBounceSoundType = 0;
		return; // head blown clean off
	}

	head.shadowPlane = shadowPlane;
	head.renderfx = renderfx;

	trap_R_AddRefEntityToScene( &head );

}

extern vmCvar_t	cg_znear;
void CG_Player( centity_t *cent ) {
	clientInfo_t	*ci;
	refEntity_t		legs;
	refEntity_t		torso;
	refEntity_t		head;
	float			*origin;
	int				i, item[8];

	int				clientNum;
	int				renderfx;
	int				legsAnim;
	qboolean		shadow;
	float			shadowPlane;
	
	// the client number is stored in clientNum.  It can't be derived
	// from the entity number, because a single client may have
	// multiple corpses on the level using the same clientinfo
	clientNum = cent->currentState.clientNum;
	// ## Hentai ##
	if ( clientNum < 0 || clientNum >= MAX_CLIENTS ) {
		CG_Error( "Bad clientNum on player entity");
	}
	ci = &cgs.clientinfo[ clientNum ];
	// it is possible to see corpses from disconnected players that may
	// not have valid clientinfo
	if ( !ci->infoValid ) {
		return;
	}
	
	if(!cg.intermissionStarted && cent->currentState.clientNum == cg.snap->ps.clientNum)
		origin = cg.refdef.vieworg;
	else
		origin = cent->lerpOrigin;

	memset( &legs, 0, sizeof(legs) );
	memset( &torso, 0, sizeof(torso) );
	memset( &head, 0, sizeof(head) );

	// get the rotation information
	CG_PlayerAngles( cent, legs.axis, torso.axis, head.axis );
	//VectorScale(torso.axis[1], -1, torso.axis[1]);
	
	// get the animation state (after rotation, to allow feet shuffle)
	legsAnim = cent->currentState.legsAnim & ~(ANIM_TOGGLEBIT | ANIM_CLIMBWALLS);
	
	
	CG_PlayerAnimation( cent, &legs.oldframe, &legs.frame, &legs.backlerp,
		 &torso.oldframe, &torso.frame, &torso.backlerp );
	//CG_Printf("%d -> %d [%d] / [%d]\n", legs.oldframe, legs.frame, ci->lstep, ci->rstep);
	//CG_Printf("client thinks animation frames are L:%i T:%i\n", legs.frame, torso.frame);
	if(ci->footstepSound && ((legs.frame >= ci->lstep && legs.oldframe <= ci->lstep) || (legs.frame >= ci->rstep && legs.oldframe <= ci->rstep)))// && (legsAnim == LEGS_RUN || legsAnim == LEGS_SWIM))
	{ // just cycled through run - do a footstep		
		
		trap_S_StartSound (NULL, cent->currentState.number, CHAN_BODY, ci->footstepSound);
		ci->footstepSound = 0;
	}

	// add powerups floating behind the player
	CG_PlayerPowerups( cent );

	// add the talk baloon or disconnect icon
	CG_PlayerSprites( cent );

	// add the shadow
	shadow = CG_PlayerShadow( cent, &shadowPlane );

	// add a water splash if partially in and out of water
	CG_PlayerSplash( cent );

	// get the player model information
	renderfx = 0;
	if ( cg_shadows.integer == 3 && shadow ) {
		renderfx |= RF_SHADOW_PLANE;
	}
	renderfx |= RF_LIGHTING_ORIGIN;			// use the same origin for all

	//
	// add the legs
	//
	legs.hModel = ci->legsModel;
	legs.customSkin = ci->legsSkin;

	VectorCopy( origin, legs.origin );

	VectorCopy( origin, legs.lightingOrigin );
	legs.shadowPlane = shadowPlane;
	legs.renderfx = renderfx;
	VectorCopy (legs.origin, legs.oldorigin);	// don't positionally lerp at all

	
	
	if((cent->currentState.legsAnim & ANIM_CLIMBWALLS ) && !(cent->currentState.eFlags & EF_DEAD) && !(cg.intermissionStarted ))
	{
		vec3_t		point, mins, maxs;
		trace_t		tr, trace;
		int i;
		
		mins[0] = -15;
		mins[1] = -15;
		mins[2] = -24;
		maxs[0] = 15;
		maxs[1] = 15;
		maxs[2] = 24;
		trace.fraction = 1.1;
		for(i = 0; i < NUMVERTEXNORMALS; i++)
		{
			VectorMA(legs.origin, 40, bytedirs[i], point);
			CG_Trace(&tr, legs.origin, vec3_origin, vec3_origin, point, cent->currentState.number, MASK_SOLID);
			if(tr.fraction < trace.fraction && !(tr.surfaceFlags & (SURF_SKY | SURF_NOIMPACT)))
				trace = tr;
		}
		if(trace.startsolid)
		{ // this seems to happen on curves - why?
		
			for(i = 0; i < 6; i++)
			{
				VectorCopy(legs.origin, point);
				if(i < 3)
					point[i] -= 32;
				else
					point[i - 3] += 32;
				CG_Trace(&trace, legs.origin, vec3_origin, vec3_origin, point, cent->currentState.number, MASK_PLAYERSOLID);
				if(trace.fraction < 1.0)
					break;
			}
			if(trace.startsolid)
			{
				CG_Printf("WTF!?");
			}
		
		}
		if(trace.fraction < 1.0)
 		{
		
			vec3_t forward;
			trace_t tr;
			VectorMA(legs.origin, -40, trace.plane.normal, point);
			point[2] += 8;
			CG_Trace(&tr, legs.origin, vec3_origin, vec3_origin, point, cent->currentState.number, MASK_PLAYERSOLID);
			VectorMA(tr.endpos, 24, trace.plane.normal, legs.origin);
			
			AngleVectors(cent->lerpAngles, forward, NULL, NULL);
			VectorCopy(trace.plane.normal, legs.axis[2]);
			ProjectPointOnPlane(legs.axis[0], forward, legs.axis[2]);
			if(!VectorNormalize(legs.axis[0]))
			{
				AngleVectors(cent->lerpAngles, NULL, NULL, forward);
				ProjectPointOnPlane(legs.axis[0], forward, legs.axis[2]);
				VectorNormalize(legs.axis[0]);
			}
			CrossProduct(legs.axis[0], legs.axis[2], legs.axis[1]);			
			legs.axis[1][0] = -legs.axis[1][0];
			legs.axis[1][1] = -legs.axis[1][1];
			legs.axis[1][2] = -legs.axis[1][2];
			
			VectorCopy( legs.origin, legs.lightingOrigin );
			VectorCopy (legs.origin, legs.oldorigin);	// don't positionally lerp at all

		}
		else
			cent->currentState.legsAnim = LEGS_JUMP;
	}
	
	if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
	{
		legs.shaderRGBA[0] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		legs.shaderRGBA[1] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		legs.shaderRGBA[2] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		legs.shaderRGBA[3] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
	}

	//
	// add the torso
	//
	torso.hModel = ci->torsoModel;
	if (!torso.hModel) {
		return;
	}

	torso.customSkin = ci->torsoSkin;

	VectorCopy( origin, torso.lightingOrigin );
	CG_PositionRotatedEntityOnTag( &torso, &legs, ci->legsModel, "tag_torso");

	if((cent->currentState.legsAnim & ANIM_CLIMBWALLS) && !(cent->currentState.eFlags & EF_DEAD) && !(cg.intermissionStarted ))
	{
		AnglesToAxis(cent->lerpAngles, torso.axis);			
	}
	
	torso.shadowPlane = shadowPlane;
	torso.renderfx = renderfx;

	if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
	{
		torso.shaderRGBA[0] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		torso.shaderRGBA[1] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		torso.shaderRGBA[2] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		torso.shaderRGBA[3] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
	}
	//
	// add the head
	//
	
	head.hModel = ci->headModel;
	if (!head.hModel) {
		return;
	}
	head.customSkin = ci->headSkin;
	VectorCopy( origin, head.lightingOrigin );
	
	CG_PositionRotatedEntityOnTag( &head, &torso, ci->torsoModel, "tag_head");
	if((cent->currentState.legsAnim & ANIM_CLIMBWALLS) && !(cent->currentState.eFlags & EF_DEAD) && !(cg.intermissionStarted ))
	{
		AnglesToAxis(cent->lerpAngles, head.axis);			
	}
	//AngleVectors(cent->lerpAngles, head.axis[0], NULL, NULL);
	//ProjectPointOnPlane(head.axis[2], head.axis[2], head.axis[0]);
	//VectorNormalize(head.axis[2]);
	//CrossProduct(head.axis[2], head.axis[0], head.axis[1]);

	head.shadowPlane = shadowPlane;
	head.renderfx = renderfx;

	if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
	{
		head.shaderRGBA[0] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		head.shaderRGBA[1] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		head.shaderRGBA[2] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		head.shaderRGBA[3] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
	}
	if(clientNum == cg.snap->ps.clientNum && !cg.intermissionStarted )
	{ // hey, it's us! - better set our viewpoint.
		vec3_t mins, maxs, check_viewpoint;
		trace_t trace;
		VectorCopy(head.origin, ci->viewpoint);
	
		trap_R_ModelBounds( head.hModel, mins, maxs );
		VectorMA( ci->viewpoint, maxs[0] + cg_znear.value, head.axis[0], ci->viewpoint );	
		VectorMA( ci->viewpoint, (maxs[2] + mins[2]) / 2, head.axis[2], ci->viewpoint );
		
		check_viewpoint[0] = cg.refdef.vieworg[0];
		check_viewpoint[1] = cg.refdef.vieworg[1];
		check_viewpoint[2] = ci->viewpoint[2];

		mins[0] = -8;
		mins[1] = -8;
		mins[2] = -8;
		maxs[0] = 8;
		maxs[1] = 8;
		maxs[2] = 8;
		CG_Trace( &trace, check_viewpoint, mins, maxs, ci->viewpoint, cent->currentState.number, MASK_PLAYERSOLID);
		if(trace.fraction < 1)
		{ // we're in a wall!
			float xd, yd;		
			xd = ci->viewpoint[0] - trace.endpos[0];
			yd = ci->viewpoint[1] - trace.endpos[1];
			ci->viewpoint[0] = trace.endpos[0];
			ci->viewpoint[1] = trace.endpos[1];
			head.origin[0] -= xd;
			head.origin[1] -= yd;
			torso.origin[0] -= xd;
			torso.origin[1] -= yd;
			legs.origin[0] -= xd;
			legs.origin[1] -= yd;		
		}
	
		check_viewpoint[0] = ci->viewpoint[0];
		check_viewpoint[1] = ci->viewpoint[1];
		check_viewpoint[2] = cg.refdef.vieworg[2];
		CG_Trace( &trace, check_viewpoint, mins, maxs, ci->viewpoint, cent->currentState.number, MASK_PLAYERSOLID);
		if(trace.fraction < 1)
		{ // head's sticking up through the ceiling - this can happen while jumping for some models
			float zd;
			zd = ci->viewpoint[2] - trace.endpos[2];
			ci->viewpoint[2] = trace.endpos[2];
			head.origin[2] -= zd;
			torso.origin[2] -= zd;
			legs.origin[2] -= zd;		
		}
		VectorCopy(ci->viewpoint, cg.refdef.vieworg);
		//VectorCopy(cg.refdef.viewaxis[0], head.axis[0]);
		//VectorCopy(cg.refdef.viewaxis[1], head.axis[1]);
		//VectorCopy(cg.refdef.viewaxis[2], head.axis[2]);
		if(cent->currentState.eType == ET_INVISIBLE)
			return; // we got what we came for
	}

	CG_AddRefEntityWithPowerups( &legs, cent->currentState.powerups & ~(1 << PW_ARMOR), ci->team );

	// if the model failed, allow the default nullmodel to be displayed
	if (!legs.hModel) {
		return;
	}

	CG_AddRefEntityWithPowerups( &torso, cent->currentState.powerups, ci->team );
	VectorCopy(head.origin, cent->world_headpos);
	if((cent->currentState.eFlags & EF_DEAD) && (cent->currentState.eFlags & EF_AWARD_IMPRESSIVE) && !(cg.intermissionStarted ))
	{
		while(ci->head_explodey)
		{ // head was JUST blown off
			CG_SpurtBlood(head.origin, bytedirs[rand() & 0xff], (rand() & 10 )+ 1);
			ci->head_explodey--;
		}
		CG_SpurtBlood(head.origin, head.axis[2], (rand() & 10 )+ 1);
		return; // head blown clean off
	}


	CG_AddRefEntityWithPowerups( &head, cent->currentState.powerups & ~(1 << PW_ARMOR), ci->team );

	if(cent->currentState.powerups & (1 << PW_FLIGHT))
	{ // add the wings
		refEntity_t		wings;
		
		memset(&wings, 0, sizeof(wings));
		AxisCopy( torso.axis, wings.axis );
		VectorCopy( torso.origin, wings.origin );
		VectorCopy( wings.origin, wings.oldorigin );
		VectorCopy( wings.origin, wings.lightingOrigin );
		wings.hModel = cgs.media.flightWingsModel;
		wings.customShader = cgs.media.flightWingsShader;
		wings.renderfx = renderfx;
		wings.shadowPlane = shadowPlane;
		
		wings.oldframe = cent->pe.wings.oldFrame;
		wings.frame = cent->pe.wings.frame;
		wings.backlerp = cent->pe.wings.backlerp;
		if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
		{
			wings.shaderRGBA[0] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
			wings.shaderRGBA[1] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
			wings.shaderRGBA[2] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
			wings.shaderRGBA[3] = 255*(5000 -  cent->currentState.time2	) / 5000.0;
		}

		CG_AddRefEntityWithPowerups( &wings, cent->currentState.powerups & ~(1 << PW_ARMOR), ci->team );

		if(cent->pe.wings.oldFrame > cent->pe.wings.frame)
		{ // animation just looped
			localEntity_t *le;
			vec3_t org;
			org[0] = origin[0] + (rand() % 31) - 15;
			org[1] = origin[1] + (rand() % 31) - 15;
			org[2] = origin[2] + (rand() % 31);
			trap_S_StartSound( origin, cent->currentState.number, CHAN_BODY, cgs.media.flightSound );
			le = CG_LaunchGib( org, cent->currentState.pos.trDelta, cgs.media.flightFeatherModel);
			le->refEntity.customShader = cgs.media.flightFeatherShader;			
			le->leFlags |= LEF_TUMBLE;
			le->leBounceSoundType = LEBS_FEATHER;
			le->leMarkType = 0;
			le->angles.trBase[0] = rand() % 360;
			le->angles.trBase[1] = rand() % 360;
			le->angles.trBase[2] = rand() % 360;
			le->angles.trDelta[0] = rand() % 300;
			le->angles.trDelta[1] = rand() % 300;
			le->angles.trDelta[2] = rand() % 300;
			le->angles.trType = TR_LINEAR;
			le->angles.trTime = cg.time;
			le->endTime = cg.time + 100000000;
	
		}

	
	}
	else if(cent->pe.wings.animation)
	{
		int i;
		for(i = 0; i < 61; i++)
		{
			localEntity_t *le;
			vec3_t org;
			org[0] = origin[0] + (rand() % 31) - 15 + torso.axis[1][0] * (i - 30);
			org[1] = origin[1] + (rand() % 31) - 15 + torso.axis[1][1] * (i - 30);
			org[2] = origin[2] + (rand() % 31) + torso.axis[1][2] * (i - 30);
			trap_S_StartSound( origin, cent->currentState.number, CHAN_BODY, cgs.media.flightSound );
			le = CG_LaunchGib( org, cent->currentState.pos.trDelta, cgs.media.flightFeatherModel);
			le->refEntity.customShader = cgs.media.flightFeatherShader;			
			le->leFlags |= LEF_TUMBLE;
			le->leBounceSoundType = LEBS_FEATHER;
			le->leMarkType = 0;
			le->angles.trBase[0] = rand() % 360;
			le->angles.trBase[1] = rand() % 360;
			le->angles.trBase[2] = rand() % 360;
			le->angles.trDelta[0] = rand() % 300;
			le->angles.trDelta[1] = rand() % 300;
			le->angles.trDelta[2] = rand() % 300;
			le->angles.trType = TR_LINEAR;
			le->angles.trTime = cg.time;
			le->endTime = cg.time + 100000000;
		}
		cent->pe.wings.animation = NULL;
	}

	//
	// add the gun / barrel / flash
	//
	CG_AddPlayerWeapon( &torso, NULL, cent );
	item[0] = cent->currentState.otherEntityNum;
	item[1] = cent->currentState.otherEntityNum2;
	item[2] = cent->currentState.constantLight;
	item[3] = cent->currentState.loopSound;
	item[4] = cent->currentState.modelindex;
	item[5] = cent->currentState.modelindex2;
	for(i = 0; i < 6; i++)
	{
		int pclass;
		pclass = BG_PlayerClass(ci->modelName);
		if(item[i]) // on belt
		{
			int yaw = -20 * (i - 2), qty = (item[i] >> 8); // high 8 bits is quantity
			refEntity_t obj;
			memset(&obj, 0, sizeof(obj));
			VectorScale(torso.axis[0], cos(yaw), obj.axis[0]);
			VectorMA(obj.axis[0], sin(yaw), torso.axis[1], obj.axis[0]);
			
			VectorScale(torso.axis[0], -sin(yaw), obj.axis[1]);
			VectorMA(obj.axis[1], cos(yaw), torso.axis[1], obj.axis[1]);
			
			VectorCopy(torso.axis[2], obj.axis[2]);
			
			VectorMA(torso.origin, (18 + i * 0.25) * p_class[pclass].waistsize * 0.3, obj.axis[0], obj.origin);
			item[i] &= 0xff; // low 8 bits is item index
			
			if(bg_itemlist[item[i]].giType == IT_AMMO && bg_itemlist[item[i]].giTag == AMMO_SLUGS)
			{
				VectorMA(torso.origin, 20 * p_class[pclass].waistsize * 0.3 + 1.5, obj.axis[0], obj.origin);
			}
			else if(bg_itemlist[item[i]].giType == IT_HEALTH || (bg_itemlist[item[i]].giType == IT_AMMO && bg_itemlist[item[i]].giTag != AMMO_GRENADES))
			{ // special case - make them stick up, not out
				
				VectorMA(obj.axis[0], -2, torso.axis[2], obj.axis[0]);
				VectorNormalize(obj.axis[0]);
				CrossProduct(obj.axis[0], obj.axis[1], obj.axis[2]);
			}
			
			CG_RenderItem(&bg_itemlist[item[i]], qty, cent->currentState.powerups & ~(1 << PW_ARMOR), obj.origin, obj.axis, cent->vel);
		}
	}
	if(cent->currentState.frame && bg_itemlist[cent->currentState.frame].giType == IT_WEAPON)
	{
		refEntity_t		gun;
		short qty = cent->currentState.solid;
		if(qty == 1023) // only sent as 10 bits
			qty = -1;
		memset(&gun, 0, sizeof(gun));

		torso.hModel = 0;
		VectorMA(torso.origin, -5, torso.axis[0], gun.origin);
		VectorMA(gun.origin, -3, torso.axis[1], gun.origin);
		if(bg_itemlist[cent->currentState.modelindex].giType >= WP_REDFLAG)
		{
			VectorSubtract(torso.axis[1], torso.axis[2], gun.axis[0]);
		}
		else
		{
			VectorAdd(torso.axis[2], torso.axis[1], gun.axis[0]);
		}
		VectorNormalize(gun.axis[0]);
		VectorCopy(torso.lightingOrigin, gun.lightingOrigin);
		VectorCopy(torso.axis[0], gun.axis[1]);
		CrossProduct(gun.axis[0], gun.axis[1], gun.axis[2]);

		CG_RenderItem(&bg_itemlist[cent->currentState.frame], qty, cent->currentState.powerups & ~(1 << PW_ARMOR), gun.origin, gun.axis, cent->vel);
		//CG_AddPlayerWeapon( &gun, NULL, cent );
	}
	if (cg_lagometer.integer && cg_debugPosition.integer )
	{
		int pclass = BG_PlayerClass(ci->modelName);

		vec3_t dir, forward, right, up;
		dir[0] = (torso.origin[0] - legs.origin[0]);
		dir[1] = (torso.origin[1] - legs.origin[1]);
		dir[2] = (torso.origin[2] - legs.origin[2]);
		AngleVectors(cent->lerpAngles, forward, right, up);
		CG_Printf("torso->legs [%f | %d]: x:%f y:%f z:%f\n", cent->lerpAngles[YAW], cent->currentState.legsAnim & ~ANIM_TOGGLEBIT, DotProduct(forward, dir), DotProduct(right, dir), DotProduct(up, dir));
	
		dir[0] = (head.origin[0] - torso.origin[0]);
		dir[1] = (head.origin[1] - torso.origin[1]);
		dir[2] = (head.origin[2] - torso.origin[2]);
	
		CG_Printf("head->torso [%f | %d]: x:%f y:%f z:%f\n", cent->lerpAngles[YAW], cent->currentState.torsoAnim & ~ANIM_TOGGLEBIT, DotProduct(forward, dir), DotProduct(right, dir), DotProduct(up, dir));
	
		dir[0] = (cent->pe.railgunOrigin[0] - torso.origin[0]);
		dir[1] = (cent->pe.railgunOrigin[1] - torso.origin[1]);
		dir[2] = (cent->pe.railgunOrigin[2] - torso.origin[2]);
	
		CG_Printf("gun->torso [%f | %d]: x:%f y:%f z:%f\n", cent->lerpAngles[YAW], cent->currentState.torsoAnim & ~ANIM_TOGGLEBIT, DotProduct(forward, dir), DotProduct(right, dir), DotProduct(up, dir));

		if(pclass > -1)
		{
			BG_PlayerAngles(pclass, legs.frame, torso.frame, cent->lerpAngles, cent->currentState.pos.trDelta, forward, dir);
			memset( &head, 0, sizeof(head) );
			head.hModel = trap_R_RegisterModel( "models/powerups/health/medium_sphere.md3");//cgs.media.sphereFlashModel;
			VectorAdd(legs.origin, dir, head.origin);
			head.axis[0][0] = p_class[pclass].headsize / 16.0;
			head.axis[1][1] = p_class[pclass].headsize / 16.0;
			head.axis[2][2] = p_class[pclass].headsize / 16.0;
			VectorMA(head.origin, -4, head.axis[2], head.origin);
			trap_R_AddRefEntityToScene( &head ); // don't add powerups; they'll look funny

			head.hModel = trap_R_RegisterModel( "models/powerups/health/large_sphere.md3");//cgs.media.sphereFlashModel;
			VectorAdd(legs.origin, forward, head.origin);
			head.axis[0][0] = 0.1;
			head.axis[1][1] = 0.1;
			head.axis[2][2] = 0.1;
			VectorMA(head.origin, -4, head.axis[2], head.origin);
			trap_R_AddRefEntityToScene( &head ); // don't add powerups; they'll look funny
		}
	}
}

//=====================================================================

/*
===============
CG_ResetPlayerEntity

A player just came into view or teleported, so reset all animation info
===============
*/
void CG_ResetPlayerEntity( centity_t *cent ) {
	cent->errorTime = -99999;		// guarantee no error decay added
	cent->extrapolated = qfalse;	

	CG_ClearLerpFrame( &cgs.clientinfo[ cent->currentState.clientNum ], &cent->pe.legs, cent->currentState.legsAnim );
	CG_ClearLerpFrame( &cgs.clientinfo[ cent->currentState.clientNum ], &cent->pe.torso, cent->currentState.torsoAnim );

	BG_EvaluateTrajectory( &cent->currentState.pos, cg.time, cent->lerpOrigin );
	BG_EvaluateTrajectory( &cent->currentState.apos, cg.time, cent->lerpAngles );

	VectorCopy( cent->lerpOrigin, cent->rawOrigin );
	VectorCopy( cent->lerpAngles, cent->rawAngles );

	memset( &cent->pe.legs, 0, sizeof( cent->pe.legs ) );
	cent->pe.legs.yawAngle = cent->rawAngles[YAW];
	cent->pe.legs.yawing = qfalse;
	cent->pe.legs.pitchAngle = 0;
	cent->pe.legs.pitching = qfalse;

	memset( &cent->pe.torso, 0, sizeof( cent->pe.legs ) );
	cent->pe.torso.yawAngle = cent->rawAngles[YAW];
	cent->pe.torso.yawing = qfalse;
	cent->pe.torso.pitchAngle = cent->rawAngles[PITCH];
	cent->pe.torso.pitching = qfalse;

	if ( cg_debugPosition.integer ) {
		CG_Printf("%i ResetPlayerEntity yaw=%i\n", cent->currentState.number, cent->pe.torso.yawAngle );
	}
}

