// Copyright (C) 1999-2000 Id Software, Inc.
//

// cg_localents.c -- every frame, generate renderer commands for locally
// processed entities, like smoke puffs, gibs, shells, etc.

#include "cg_local.h"

#define	MAX_LOCAL_ENTITIES	512
#define	MAX_STATIC_LOCAL_ENTITIES	1024//8192

refEntity_t		cg_slocalEntities[MAX_STATIC_LOCAL_ENTITIES];
int				static_local_entities;
int				nextstatic_local_entity;

localEntity_t	cg_localEntities[MAX_LOCAL_ENTITIES];
localEntity_t	cg_activeLocalEntities;		// double linked list
localEntity_t	*cg_freeLocalEntities;		// single linked list

/*
===================
CG_InitLocalEntities

This is called at startup and for tournement restarts
===================
*/
void	CG_InitLocalEntities( void ) {
	int		i;

	memset( cg_localEntities, 0, sizeof( cg_localEntities ) );
	cg_activeLocalEntities.next = &cg_activeLocalEntities;
	cg_activeLocalEntities.prev = &cg_activeLocalEntities;
	cg_freeLocalEntities = cg_localEntities;
	for ( i = 0 ; i < MAX_LOCAL_ENTITIES - 1 ; i++ ) {
		cg_localEntities[i].next = &cg_localEntities[i+1];
	}

	memset( cg_slocalEntities, 0, sizeof( cg_slocalEntities ) );	
	static_local_entities = 0;
	nextstatic_local_entity = 0;

}


/*
==================
CG_FreeLocalEntity
==================
*/
void CG_FreeLocalEntity( localEntity_t *le ) {
	if ( !le->prev ) {
		CG_Error( "CG_FreeLocalEntity: not active" );
	}

	// remove from the doubly linked active list
	le->prev->next = le->next;
	le->next->prev = le->prev;

	// the free list is only singly linked
	le->next = cg_freeLocalEntities;
	cg_freeLocalEntities = le;
}

/*
===================
CG_AllocLocalEntity

Will allways succeed, even if it requires freeing an old active entity
===================
*/
localEntity_t	*CG_AllocLocalEntity( void ) {
	localEntity_t	*le;

	if ( !cg_freeLocalEntities ) {
		// no free entities, so free the one at the end of the chain
		// remove the oldest active entity
		CG_FreeLocalEntity( cg_activeLocalEntities.prev );
	}

	le = cg_freeLocalEntities;
	cg_freeLocalEntities = cg_freeLocalEntities->next;

	memset( le, 0, sizeof( *le ) );

	// link into the active list
	le->next = cg_activeLocalEntities.next;
	le->prev = &cg_activeLocalEntities;
	cg_activeLocalEntities.next->prev = le;
	cg_activeLocalEntities.next = le;
	return le;
}

refEntity_t	*CG_AllocStaticLocalEntity( void ) {

	/*while(cg_slocalEntities[nextstatic_local_entity].hModel && nextstatic_local_entity)
	{
		nextstatic_local_entity++;
		nextstatic_local_entity %= MAX_STATIC_LOCAL_ENTITIES;
	}
	if(nextstatic_local_entity >= static_local_entities)
		static_local_entities = nextstatic_local_entity + 1;*/
	nextstatic_local_entity = (static_local_entities++) % MAX_STATIC_LOCAL_ENTITIES;
	cg_slocalEntities[nextstatic_local_entity].hModel = 0;
	return &cg_slocalEntities[nextstatic_local_entity];	
}


/*
====================================================================================

FRAGMENT PROCESSING

A fragment localentity interacts with the environment in some way (hitting walls),
or generates more localentities along a trail.

====================================================================================
*/

/*
================
CG_BloodTrail

Leave expanding blood puffs behind gibs
================
*/
void CG_BloodTrail( localEntity_t *le ) {
	int		t;
	int		t2;
	int		step;
	float radius;
	vec3_t	newOrigin;
	localEntity_t	*blood;

	step = 150;
	t = step * ( (cg.time - cg.frametime + step ) / step );
	t2 = step * ( cg.time / step );

	for ( ; t <= t2; t += step ) {
		BG_EvaluateTrajectory( &le->pos, t, newOrigin );

		if(le->leBounceSoundType == LEBS_FEATHER)
			radius = random() * 6 + 2;
		else
			radius = random() * 12 + 12;
		blood = CG_SmokePuff( newOrigin, vec3_origin, 
					  radius,		// radius
					  1, 1, 1, 1,	// color
					  2000,		// trailTime
					  t,		// startTime
					  0,		// flags
					  cgs.media.bloodTrailShader );
		// use the optimized version
		blood->leType = LE_FALL_SCALE_FADE;
		VectorCopy(newOrigin, blood->refEntity.oldorigin);
		// drop a total of 40 units over its lifetime
		blood->pos.trDelta[2] = 40;
	}
}


/*
================
CG_FragmentBounceMark
================
*/
void CG_FragmentBounceMark( localEntity_t *le, trace_t *trace ) {
	int			radius;

	if ( le->leMarkType == LEMT_BLOOD ) {

		if(le->leBounceSoundType == LEBS_FEATHER)
			radius = 4 + (rand()&12);
		else
			radius = 16 + (rand()&31);
		CG_ImpactMark( cgs.media.bloodMarkShader, trace->endpos, trace->plane.normal, random()*360,
			1,1,1,1, qtrue, radius, qfalse );
	}

	// don't allow a fragment to make multiple marks, or they
	// pile up while settling
	le->leMarkType = LEMT_NONE;
}

/*
================
CG_FragmentBounceSound
================
*/
void CG_FragmentBounceSound( localEntity_t *le, trace_t *trace ) {
	if ( le->leBounceSoundType == LEBS_BLOOD ) {
		// half the gibs will make splat sounds
		if ( rand() & 1 ) {
			int r = rand()&3;
			sfxHandle_t	s;

			if ( r < 2 ) {
				s = cgs.media.gibBounce1Sound;
			} else if ( r == 2 ) {
				s = cgs.media.gibBounce2Sound;
			} else {
				s = cgs.media.gibBounce3Sound;
			}
			trap_S_StartSound( trace->endpos, ENTITYNUM_WORLD, CHAN_AUTO, s );
		}
		// don't allow a fragment to make multiple bounce sounds,
		// or it gets too noisy as they settle
		le->leBounceSoundType = LEBS_NONE;
	} else if ( le->leBounceSoundType == LEBS_BRASS ) {
		trap_S_StartSound( trace->endpos, ENTITYNUM_WORLD, CHAN_AUTO, cgs.media.brassBounceSound );
	}

	
}


/*
================
CG_ReflectVelocity
================
*/
void CG_ReflectVelocity( localEntity_t *le, trace_t *trace ) {
	vec3_t	velocity;
	float	dot;
	int		hitTime;

	// reflect the velocity on the trace plane
	hitTime = cg.time - cg.frametime + cg.frametime * trace->fraction;
	BG_EvaluateTrajectoryDelta( &le->pos, hitTime, velocity );
	dot = DotProduct( velocity, trace->plane.normal );
	VectorMA( velocity, -2*dot, trace->plane.normal, le->pos.trDelta );

	VectorScale( le->pos.trDelta, le->bounceFactor, le->pos.trDelta );

	VectorCopy( trace->endpos, le->pos.trBase );
	le->pos.trTime = cg.time;


	// check for stop, making sure that even on low FPS systems it doesn't bobble
	if ( trace->allsolid || 
		( trace->plane.normal[2] > 0 && 
		( le->pos.trDelta[2] < 40 || le->pos.trDelta[2] < -cg.frametime * le->pos.trDelta[2] ) ) ) {
		le->pos.trType = TR_STATIONARY;
	} else {

	}
}

/*
================
CG_AddFragment
================
*/
void CG_AddFragment( localEntity_t *le ) {
	vec3_t	newOrigin;
	trace_t	trace;

	if ( le->pos.trType == TR_STATIONARY ) {
		// sink into the ground if near the removal time
		int		t;
		float	oldZ;
		
		t = le->endTime - cg.time;
		if ( t < SINK_TIME ) {
			// we must use an explicit lighting origin, otherwise the
			// lighting would be lost as soon as the origin went
			// into the ground
			VectorCopy( le->refEntity.origin, le->refEntity.lightingOrigin );
			le->refEntity.renderfx |= RF_LIGHTING_ORIGIN;
			oldZ = le->refEntity.origin[2];
			le->refEntity.origin[2] -= 16 * ( 1.0 - (float)t / SINK_TIME );
			trap_R_AddRefEntityToScene( &le->refEntity );
			le->refEntity.origin[2] = oldZ;
		} else { 
			trap_R_AddRefEntityToScene( &le->refEntity );
		}

		return;
	}

	// calculate new position
	BG_EvaluateTrajectory( &le->pos, cg.time, newOrigin );

	if(le->leBounceSoundType == LEBS_FEATHER)
	{
		float decel = 1.0 - (cg.time - cg.oldTime) / 3000.0;
		if(decel < 0)
			decel = 0;
		VectorCopy(newOrigin, le->pos.trBase);
		le->pos.trTime = cg.time;
		le->pos.trDelta[0] *= decel;
		le->pos.trDelta[0] += ((rand() % 250) - 125.0) * (cg.time - cg.oldTime) / 1000.0;
		
		le->pos.trDelta[1] *= decel;
		le->pos.trDelta[1] += ((rand() % 250) - 125.0) * (cg.time - cg.oldTime) / 1000.0;
		//if(le->bounceFactor > 0 && le->pos.trDelta[2] < 0)
		//	le->leMarkType = 0; // stop bleeding after apogee

		if(le->leMarkType == LEMT_BLOOD)
		{
			{
				le->pos.trDelta[2] *= decel;
				le->pos.trDelta[2] -= g_gravity.value * (cg.time - cg.oldTime) / 1000.0;
			}
		}
		else
		{
			le->pos.trDelta[2] *= decel;
			le->pos.trDelta[2] += ((rand() % 250) - 125.0) * (cg.time - cg.oldTime) / 1000.0;
		}
	}
	
	// trace a line from previous position to new position
	CG_Trace( &trace, le->refEntity.origin, NULL, NULL, newOrigin, -1, CONTENTS_SOLID );
	if ( trace.fraction == 1.0 ) {
		// still in free fall
		VectorCopy( newOrigin, le->refEntity.origin );

		if ( le->leFlags & LEF_TUMBLE ) {
			vec3_t angles;

			BG_EvaluateTrajectory( &le->angles, cg.time, angles );
			
			AnglesToAxis( angles, le->refEntity.axis );
			VectorScale(le->refEntity.axis[0], 0.6, le->refEntity.axis[0]);
			VectorScale(le->refEntity.axis[1], 0.6, le->refEntity.axis[1]);
			VectorScale(le->refEntity.axis[2], 0.6, le->refEntity.axis[2]);
		
		}

		trap_R_AddRefEntityToScene( &le->refEntity );

		// add a blood trail
		if ( le->leBounceSoundType == LEBS_BLOOD || le->leMarkType == LEMT_BLOOD) {
			CG_BloodTrail( le );
		}

		return;
	}
	// if it is in a nodrop zone, remove it
	// this keeps gibs from waiting at the bottom of pits of death
	// and floating levels
	if ( trap_CM_PointContents( trace.endpos, 0 ) & CONTENTS_NODROP ) {
		CG_FreeLocalEntity( le );
		return;
	}

	
	// leave a mark
	CG_FragmentBounceMark( le, &trace );

	// do a bouncy sound
	CG_FragmentBounceSound( le, &trace );

	// reflect the velocity on the trace plane
	CG_ReflectVelocity( le, &trace );

	if((le->leFlags & LEF_TUMBLE) && trace.entityNum == ENTITYNUM_WORLD)
	{		
		if(le->leBounceSoundType == LEBS_FEATHER)
		{
			VectorCopy(trace.plane.normal, le->refEntity.axis[2]);
			le->refEntity.axis[1][0] = crandom() * 10;
			le->refEntity.axis[1][1] = crandom() * 10;
			le->refEntity.axis[1][2] = crandom() * 10;
			ProjectPointOnPlane(le->refEntity.axis[1], le->refEntity.axis[1], trace.plane.normal);
			VectorNormalize(le->refEntity.axis[1]);
			CrossProduct(le->refEntity.axis[1], le->refEntity.axis[2], le->refEntity.axis[0]);
			VectorScale(le->refEntity.axis[0], 0.6, le->refEntity.axis[0]);
			VectorScale(le->refEntity.axis[1], 0.6, le->refEntity.axis[1]);
			VectorScale(le->refEntity.axis[2], 0.6, le->refEntity.axis[2]);
			VectorMA(trace.endpos, 0.25, trace.plane.normal, le->refEntity.origin);
		}
		else
		{
			VectorCopy(trace.plane.normal, le->refEntity.axis[0]);
			le->refEntity.axis[1][0] = crandom() * 10;
			le->refEntity.axis[1][1] = crandom() * 10;
			le->refEntity.axis[1][2] = crandom() * 10;
			ProjectPointOnPlane(le->refEntity.axis[1], le->refEntity.axis[1], le->refEntity.axis[0]);
			VectorNormalize(le->refEntity.axis[1]);
			CrossProduct(le->refEntity.axis[0], le->refEntity.axis[1], le->refEntity.axis[2]);
			VectorScale(le->refEntity.axis[0], 0.6, le->refEntity.axis[0]);
			VectorScale(le->refEntity.axis[1], 0.6, le->refEntity.axis[1]);
			VectorScale(le->refEntity.axis[2], 0.6, le->refEntity.axis[2]);
		}
		if(le->pos.trType == TR_STATIONARY)
		{ // only leave it if we'd leave a mark here
			if(!(trace.surfaceFlags & SURF_NOMARKS))
			{
				refEntity_t *r = CG_AllocStaticLocalEntity();
				memcpy(r, &le->refEntity, sizeof(le->refEntity));
			}
			CG_FreeLocalEntity(le);
			return;
		}
		//ProjectPointOnPlane(le->refEntity.axis[1], le->refEntity.axis[1], le->refEntity.axis[2]);
	}
	else if(trace.entityNum != ENTITYNUM_NONE && (le->leBounceSoundType == LEBS_FEATHER || le->leBounceSoundType == LEBS_BRASS) && le->pos.trType == TR_STATIONARY)
	{
		CG_FreeLocalEntity(le);
		return;
	}
	trap_R_AddRefEntityToScene( &le->refEntity );
}

/*
=====================================================================

TRIVIAL LOCAL ENTITIES

These only do simple scaling or modulation before passing to the renderer
=====================================================================
*/


/*
====================
CG_AddFadeRGB
====================
*/
void CG_AddExpandingFade( localEntity_t *le ) {
	refEntity_t *re;
	float c, reScale;

	re = &le->refEntity;

	c = ( le->endTime - cg.time ) * le->lifeRate;
	
			
	reScale = 16 * 0.0075 / (c + 0.2);
	VectorNormalize(re->axis[0]);
	VectorNormalize(re->axis[1]);
	re->axis[0][0] *= reScale;
	re->axis[0][1] *= reScale;
	re->axis[0][2] *= reScale;

	re->axis[1][0] *= reScale;
	re->axis[1][1] *= reScale;
	re->axis[1][2] *= reScale;
	
	c *= 0xff;
	
	re->shaderRGBA[0] = le->color[0] * c;
	re->shaderRGBA[1] = le->color[1] * c;
	re->shaderRGBA[2] = le->color[2] * c;
	re->shaderRGBA[3] = le->color[3] * c;
	
	trap_R_AddRefEntityToScene( re );
}

/*
====================
CG_AddFadeRGB
====================
*/
void CG_AddFadeRGB( localEntity_t *le ) {
	refEntity_t *re;
	float c;

	re = &le->refEntity;

	c = ( le->endTime - cg.time ) * le->lifeRate;
	c *= 0xff;

	re->shaderRGBA[0] = le->color[0] * c;
	re->shaderRGBA[1] = le->color[1] * c;
	re->shaderRGBA[2] = le->color[2] * c;
	re->shaderRGBA[3] = le->color[3] * c;

	trap_R_AddRefEntityToScene( re );
}

/*
==================
CG_AddMoveScaleFade
==================
*/
static void CG_AddMoveScaleFade( localEntity_t *le ) {
	refEntity_t	*re;
	float		c;

	re = &le->refEntity;	
		

	// fade / grow time
	c = ( le->endTime - cg.time ) * le->lifeRate;

	re->shaderRGBA[3] = 0xff * c * le->color[3];

	if ( !( le->leFlags & LEF_PUFF_DONT_SCALE ) ) {
		re->radius = le->radius * ( 1.0 - c );
		if(le->leFlags & LEF_MINSIZE)
			re->radius += 8;
	}

	BG_EvaluateTrajectory( &le->pos, cg.time, re->origin );
	
	/*
	// if the view would be "inside" the sprite, kill the sprite
	// so it doesn't add too much overdraw
	VectorSubtract( re->origin, cg.refdef.vieworg, delta );
	len = VectorLength( delta );
	if ( len < le->radius ) {
		CG_FreeLocalEntity( le );
		return;
	}*/
	trap_R_AddRefEntityToScene( re );
}


static void CG_AddShrapnel( localEntity_t *le ) {
	vec3_t		org, org2;
	vec4_t		rgba;
	trace_t		tr;
	// fade / grow time
	rgba[0] = 1;
	rgba[1] = 1;
	rgba[2] = 1;
	rgba[3] = ( le->endTime - cg.time ) * le->lifeRate;

	BG_EvaluateTrajectory( &le->pos, cg.oldTime, org );
	BG_EvaluateTrajectory( &le->pos, cg.time, org2 );
	CG_Trace(&tr, org, vec3_origin, vec3_origin, org2, -1, MASK_SOLID);
	CG_Tracer(org, tr.endpos, le->radius, le->leFlags, rgba); // radius is shader
	if(tr.fraction < 1.0)
	{		
		if(le->leMarkType == LEMT_BLOOD)
			CG_ImpactMark( cgs.media.bloodMarkShader, tr.endpos, tr.plane.normal, random()*360, 1,1,1,1, qtrue, le->radius * 4, !(le->leBounceSoundType) );

		CG_FreeLocalEntity(le);
	}

}

/*
===================
CG_AddScaleFade

For rocket smokes that hang in place, fade out, and are
removed if the view passes through them.
There are often many of these, so it needs to be simple.
===================
*/
static void CG_AddScaleFade( localEntity_t *le ) {
	refEntity_t	*re;
	float		c;
	vec3_t		delta;
	float		len;

	re = &le->refEntity;

	// fade / grow time
	c = ( le->endTime - cg.time ) * le->lifeRate;

	re->shaderRGBA[3] = 0xff * c * le->color[3];
	re->radius = le->radius * ( 1.0 - c ) + 8;

	// if the view would be "inside" the sprite, kill the sprite
	// so it doesn't add too much overdraw
	VectorSubtract( re->origin, cg.refdef.vieworg, delta );
	len = VectorLength( delta );
	
	if ( len < le->radius ) {
		CG_FreeLocalEntity( le );
		return;
	}

	trap_R_AddRefEntityToScene( re );
}


/*
=================
CG_AddFallScaleFade

This is just an optimized CG_AddMoveScaleFade
For blood mists that drift down, fade out, and are
removed if the view passes through them.
There are often 100+ of these, so it needs to be simple.
=================
*/
static void CG_AddFallScaleFade( localEntity_t *le ) {
	refEntity_t	*re;
	float		c, deltaTime;
	vec3_t		delta;
	float		len;
	trace_t		tr;
	re = &le->refEntity;

	// fade time
	c = ( le->endTime - cg.time ) * le->lifeRate;
	deltaTime = (cg.time - le->startTime) / 1000.0;
	re->shaderRGBA[3] = 0xff * c * le->color[3];

	re->origin[2] = le->pos.trBase[2] - 0.5 * DEFAULT_GRAVITY * deltaTime * deltaTime;//( 1.0 - c ) * le->pos.trDelta[2];
	CG_Trace(&tr, re->oldorigin, vec3_origin, vec3_origin, re->origin, -1, MASK_SOLID);
	if(tr.fraction < 1)
	{
		CG_ImpactMark( cgs.media.bloodMarkShader, tr.endpos, tr.plane.normal, random()*360,
			1,1,1,1, qtrue, random()* re->radius, qfalse );
		CG_FreeLocalEntity( le );
		return;
	}
	VectorCopy(re->origin, re->oldorigin);
	re->radius = le->radius * ( 1.5 - c );

	// if the view would be "inside" the sprite, kill the sprite
	// so it doesn't add too much overdraw
	VectorSubtract( re->origin, cg.refdef.vieworg, delta );
	len = VectorLength( delta );
	if ( len < le->radius ) {
		CG_FreeLocalEntity( le );
		return;
	}

	trap_R_AddRefEntityToScene( re );
}



/*
================
CG_AddExplosion
================
*/
static void CG_AddExplosion( localEntity_t *ex ) {
	refEntity_t	*ent;
	int r;
	float radius;	
	ent = &ex->refEntity;
	if(ent->radius)
		radius = ent->radius / 8.0;/// 8.0;
	else
		radius = 8.0;
	r = ent->renderfx & RF_EXPANDING;
	ent->renderfx &= ~RF_EXPANDING;
	// add the entity
	if(r)
	{
		int i, j;
		float dt = ((float) (cg.time - ex->startTime)) / ((float) ( ex->endTime - ex->startTime ));
		vec3_t r;
		r[0] = ent->oldorigin[0] * dt;
		r[1] = ent->oldorigin[1] * dt;
		r[2] = ent->oldorigin[2] * dt;
		//AnglesToAxis(r, ent->axis);		
		//VectorScale(cg.refdef.viewaxis[0], -1, ent->axis[0]);
		VectorSubtract(cg.refdef.vieworg, ent->origin, ent->axis[0]);
		VectorNormalize(ent->axis[0]);
		VectorScale(cg.refdef.viewaxis[1], -1, ent->axis[1]);		
		VectorNormalize(ent->axis[1]);
		ProjectPointOnPlane(ent->axis[1], ent->axis[1], ent->axis[0]);
		CrossProduct(ent->axis[0], ent->axis[1], ent->axis[2]);//VectorScale(cg.refdef.viewaxis[2], -1, ent->axis[2]);
		for(i = 0; i < 3; i++)
			for(j = 0; j < 3; j++)
				ent->axis[i][j] *= radius * sqrt(dt);
		ent->shaderRGBA[0] = 255 - (255 * dt);		
		ent->shaderRGBA[1] = 255 - (255 * dt);
		ent->shaderRGBA[2] = 255 - (255 * dt);
		ent->shaderRGBA[3] = 255 - (255 * dt);
		//VectorMA(ex->pos.trBase, -8, ent->axis[2], ent->origin);
	}
	
	trap_R_AddRefEntityToScene(ent);
	ent->renderfx |= r;
	// add the dlight
	if ( ex->light ) {
		float		light;

		light = (float)( cg.time - ex->startTime ) / ( ex->endTime - ex->startTime );
		if ( light < 0.5 ) {
			light = 1.0;
		} else {
			light = 1.0 - ( light - 0.5 ) * 2;
		}
		light = ex->light * light;
		trap_R_AddLightToScene(ent->origin, light, ex->lightColor[0], ex->lightColor[1], ex->lightColor[2] );
	}
}

static void CG_AddFalling( localEntity_t *ex ) {
	refEntity_t	*ent;

	ex->refEntity.origin[2] -= cg.frametime / 20.0;
	ex->refEntity.shaderRGBA[3] *= ((rand() % 10 + 90) / 100.0);
	ent = &ex->refEntity;
	
	// add the entity
	trap_R_AddRefEntityToScene(ent);
}
/*
================
CG_AddSpriteExplosion
================
*/
static void CG_AddSpriteExplosion( localEntity_t *le ) {
	refEntity_t	re;
	float c;

	re = le->refEntity;

	c = ( le->endTime - cg.time ) / ( float ) ( le->endTime - le->startTime );
	if ( c > 1 ) {
		c = 1.0;	// can happen during connection problems
	}

	re.shaderRGBA[0] = 0xff;
	re.shaderRGBA[1] = 0xff;
	re.shaderRGBA[2] = 0xff;
	re.shaderRGBA[3] = 0xff * c * 0.33;

	re.reType = RT_SPRITE;
	re.radius = 42 * ( 1.0 - c ) + 30;

	trap_R_AddRefEntityToScene( &re );

	// add the dlight
	if ( le->light ) {
		float		light;

		light = (float)( cg.time - le->startTime ) / ( le->endTime - le->startTime );
		if ( light < 0.5 ) {
			light = 1.0;
		} else {
			light = 1.0 - ( light - 0.5 ) * 2;
		}
		light = le->light * light;
		trap_R_AddLightToScene(re.origin, light, le->lightColor[0], le->lightColor[1], le->lightColor[2] );
	}
}


//==============================================================================

/*
===================
CG_AddLocalEntities

===================
*/
void CG_AddLocalEntities( void ) {
	localEntity_t	*le, *next;
	int i, max;
	// walk the list backwards, so any new local entities generated
	// (trails, marks, etc) will be present this frame
	
	/*
	refEntity_t r;
	memset(&r, 0, sizeof(r));
	VectorMA(cg.refdef.vieworg, 50, cg.refdef.viewaxis[0], r.origin);

	r.axis[0][0] = 1;
	r.axis[1][1] = 1;
	r.axis[2][2] = 1;
	r.customShader = trap_R_RegisterShader("railRing");
	r.hModel = cgs.media.teleportEffectModel;
	trap_R_AddRefEntityToScene( &r );
	*/
	le = cg_activeLocalEntities.prev;
	for ( ; le != &cg_activeLocalEntities ; le = next ) {
		// grab next now, so if the local entity is freed we
		// still have it
		next = le->prev;

		if ( cg.time >= le->endTime ) {
			CG_FreeLocalEntity( le );
			continue;
		}
		switch ( le->leType ) {
		default:
			CG_Error( "Bad leType: %i", le->leType );
			break;

		case LE_MARK:
			break;

		case LE_SPRITE_EXPLOSION:
			CG_AddSpriteExplosion( le );
			break;

		case LE_EXPLOSION:
			CG_AddExplosion( le );
			break;

		case LE_SHRAPNEL:
			CG_AddShrapnel( le );
			break;

		case LE_FRAGMENT:			// gibs and brass
			CG_AddFragment( le );
			break;

		case LE_MOVE_SCALE_FADE:		// water bubbles
			CG_AddMoveScaleFade( le );
			break;

		case LE_FADE_RGB:				// teleporters, railtrails
			CG_AddFadeRGB( le );
			break;

		case LE_EXPAND_FADE:				// teleporters, railtrails
			CG_AddExpandingFade( le );
			break;

		case LE_FALL_SCALE_FADE: // gib blood trails
			CG_AddFallScaleFade( le );
			break;

		case LE_SCALE_FADE:		// rocket trails
			CG_AddScaleFade( le );
			break;

		case LE_FALLFAST:
			CG_AddFalling( le );
			break;

		}
	}
	// make it permanent
	{
		max = static_local_entities;
		if(max > MAX_STATIC_LOCAL_ENTITIES)
			max = MAX_STATIC_LOCAL_ENTITIES;
		for(i = 0; i < max; i++)
		{
			if(cg_slocalEntities[i].hModel)
			{
				float f;
				vec3_t test;
				test[0] = cg_slocalEntities[i].origin[0] - cg.refdef.vieworg[0] - cg.refdef.viewaxis[0][0] * 5;
				test[1] = cg_slocalEntities[i].origin[1] - cg.refdef.vieworg[1] - cg.refdef.viewaxis[0][1] * 5;
				test[2] = cg_slocalEntities[i].origin[2] - cg.refdef.vieworg[2] - cg.refdef.viewaxis[0][2] * 5;
				
				f = test[0] * cg.refdef.viewaxis[0][0] + test[1] * cg.refdef.viewaxis[0][1] + test[2] * cg.refdef.viewaxis[0][2];
				if(f > 0)
					trap_R_AddRefEntityToScene( &cg_slocalEntities[i] );
			}
		}
	}

}




