// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_event.c -- handle entity events at snapshot or playerstate transitions

#include "cg_local.h"

//==========================================================================

/*
===================
CG_PlaceString

Also called by scoreboard drawing
===================
*/
const char	*CG_PlaceString( int rank ) {
	static char	str[64];
	char	*s, *t;

	if ( rank & RANK_TIED_FLAG ) {
		rank &= ~RANK_TIED_FLAG;
		t = "Tied for ";
	} else {
		t = "";
	}

	if ( rank == 1 ) {
		s = S_COLOR_BLUE "1st" S_COLOR_WHITE;		// draw in blue
	} else if ( rank == 2 ) {
		s = S_COLOR_RED "2nd" S_COLOR_WHITE;		// draw in red
	} else if ( rank == 3 ) {
		s = S_COLOR_YELLOW "3rd" S_COLOR_WHITE;		// draw in yellow
	} else if ( rank == 11 ) {
		s = "11th";
	} else if ( rank == 12 ) {
		s = "12th";
	} else if ( rank == 13 ) {
		s = "13th";
	} else if ( rank % 10 == 1 ) {
		s = va("%ist", rank);
	} else if ( rank % 10 == 2 ) {
		s = va("%ind", rank);
	} else if ( rank % 10 == 3 ) {
		s = va("%ird", rank);
	} else {
		s = va("%ith", rank);
	}

	Com_sprintf( str, sizeof( str ), "%s%s", t, s );
	return str;
}

/*
=============
CG_Obituary
=============
*/
static void CG_Obituary( entityState_t *ent ) {
	int			mod;
	int			target, attacker;
	char		*message;
	char		*message2;
	const char	*targetInfo;
	const char	*attackerInfo;
	char		targetName[32];
	char		attackerName[32];
	gender_t	gender;
	clientInfo_t	*ci;

	target = ent->otherEntityNum;
	attacker = ent->otherEntityNum2;
	mod = ent->eventParm;

	if ( target < 0 || target >= MAX_CLIENTS ) {
		CG_Error( "CG_Obituary: target out of range" );
	}
	ci = &cgs.clientinfo[target];

	if ( attacker < 0 || attacker >= MAX_CLIENTS ) {
		attacker = ENTITYNUM_WORLD;
		attackerInfo = NULL;
	} else {
		attackerInfo = CG_ConfigString( CS_PLAYERS + attacker );
	}

	targetInfo = CG_ConfigString( CS_PLAYERS + target );
	if ( !targetInfo ) {
		return;
	}
	Q_strncpyz( targetName, Info_ValueForKey( targetInfo, "n" ), sizeof(targetName) - 2);
	strcat( targetName, S_COLOR_WHITE );

	message2 = "";

	// check for single client messages

	switch( mod ) {
	case MOD_SUICIDE:
		message = "suicides";
		break;
	case MOD_FALLING:
		message = "cratered";
		break;
	case MOD_CRUSH:
		message = "was squished";
		break;
	case MOD_WATER:
		message = "sank like a rock";
		break;
	case MOD_SLIME:
		message = "melted";
		break;
	case MOD_LAVA:
		message = "does a back flip into the lava";
		break;
	case MOD_TARGET_LASER:
		message = "saw the light";
		break;
	case MOD_TRIGGER_HURT:
		message = "was in the wrong place";
		break;
	case MOD_BLEED:
		message = "died of a sucking chest wound";
	default:
		message = NULL;
		break;
	}

	if (attacker == target) {
		gender = ci->gender;
		switch (mod) {
		case MOD_GRENADE_SPLASH:
			if ( gender == GENDER_FEMALE )
				message = "tripped on her own grenade";
			else if ( gender == GENDER_NEUTER )
				message = "tripped on its own grenade";
			else
				message = "tripped on his own grenade";
			break;
		case MOD_ROCKET_SPLASH:
			if ( gender == GENDER_FEMALE )
				message = "blew herself up";
			else if ( gender == GENDER_NEUTER )
				message = "blew itself up";
			else
				message = "blew himself up";
			break;
		case MOD_PLASMA_SPLASH:
			if ( gender == GENDER_FEMALE )
				message = "melted herself";
			else if ( gender == GENDER_NEUTER )
				message = "melted itself";
			else
				message = "melted himself";
			break;
		case MOD_CHAINGUN:
		case MOD_MACHINEGUN:
			if ( gender == GENDER_FEMALE )
				message = "caught herself on the rebound";
			else if ( gender == GENDER_NEUTER )
				message = "caught itself on the rebound";
			else
				message = "caught himself on the rebound";
			break;
		case MOD_BLEED:
			message = "died of a sucking chest wound.";
			
		default:
			if ( gender == GENDER_FEMALE )
				message = "killed herself";
			else if ( gender == GENDER_NEUTER )
				message = "killed itself";
			else
				message = "killed himself";
			break;
		}
	}

	if (message) {
		CG_Printf( "%s %s.\n", targetName, message);
		return;
	}

	// check for kill messages from the current clientNum
	if ( attacker == cg.snap->ps.clientNum ) {
		char	*s;

		if ( cgs.gametype < GT_TEAM ) {
			s = va("You fragged %s\n%s place with %i", targetName, 
				CG_PlaceString( cg.snap->ps.persistant[PERS_RANK] + 1 ),
				cg.snap->ps.persistant[PERS_SCORE] );
		} else {
			s = va("You fragged %s", targetName );
		}
		CG_CenterPrint( s, SCREEN_HEIGHT * 0.25, BIGCHAR_WIDTH );

		// print the text message as well
	}

	// check for double client messages
	if ( !attackerInfo ) {
		attacker = ENTITYNUM_WORLD;
		strcpy( attackerName, "noname" );
	} else {
		Q_strncpyz( attackerName, Info_ValueForKey( attackerInfo, "n" ), sizeof(attackerName) - 2);
		strcat( attackerName, S_COLOR_WHITE );
		// check for kill messages about the current clientNum
		if ( target == cg.snap->ps.clientNum ) {
			Q_strncpyz( cg.killerName, attackerName, sizeof( cg.killerName ) );
		}
	}

	if ( attacker != ENTITYNUM_WORLD ) {
		switch (mod) {
		case MOD_PUNCH:
			message = "was bitch-slapped by";
			break;
		case MOD_GAUNTLET:
			message = "was pummeled by";
			break;
		case MOD_MACHINEGUN:
			message = "was machinegunned by";
			break;
		case MOD_SHOTGUN:
			message = "was gunned down by";
			break;
		case MOD_GRENADE:
			message = "ate";
			message2 = "'s grenade";
			break;
		case MOD_GRENADE_SPLASH:
			message = "was shredded by";
			message2 = "'s shrapnel";
			break;
		case MOD_ROCKET:
			message = "ate";
			message2 = "'s rocket";
			break;
		case MOD_ROCKET_SPLASH:
			message = "almost dodged";
			message2 = "'s rocket";
			break;
		case MOD_PLASMA:
			message = "was melted by";
			message2 = "'s plasmagun";
			break;
		case MOD_PLASMA_SPLASH:
			message = "was melted by";
			message2 = "'s plasmagun";
			break;
		case MOD_RAILGUN:
			message = "was railed by";
			break;
		case MOD_LIGHTNING:
			message = "was electrocuted by";
			break;
		case MOD_CHAINGUN:
			message = "was cut in half by";
			message2 = "'s chaingun";
			break;
		case MOD_TELEFRAG:
			message = "tried to invade";
			message2 = "'s personal space";
			break;
		case MOD_BLEED:
			message = "finally bled to death from";
			message2 = "'s attack";
			break;
		case MOD_FLAMETHROWER:
			message = "was char-broiled by";
			break;

		default:
			message = "was killed by";
			break;
		}

		if (message) {
			CG_Printf( "%s %s %s%s\n", 
				targetName, message, attackerName, message2);
			return;
		}
	}

	// we don't know what it was
	CG_Printf( "%s died.\n", targetName );
}

//==========================================================================

/*
===============
CG_UseItem
===============
*/
/*
static void CG_UseItem( centity_t *cent ) {
	int			itemNum;
	gitem_t		*item;
	entityState_t *es;

	es = &cent->currentState;
	
	itemNum = es->event - EV_USE_ITEM0;
	if ( itemNum < 0 || itemNum > HI_NUM_HOLDABLE ) {
		itemNum = 0;
	}

	// print a message if the local player
	if ( es->number == cg.snap->ps.clientNum ) {
		if ( !itemNum ) {
			//CG_CenterPrint( "No item to use", SCREEN_HEIGHT * 0.25, BIGCHAR_WIDTH );
			
		} else {
			item = BG_FindItemForHoldable( itemNum );
			//CG_CenterPrint( va("Use %s", item->pickup_name), SCREEN_HEIGHT * 0.25, BIGCHAR_WIDTH );
		}
	}

	switch ( itemNum ) {
	default:
	case HI_NONE:
		trap_S_StartSound (NULL, es->number, CHAN_BODY, cgs.media.useNothingSound );
		break;

	case HI_TELEPORTER:
		break;

	case HI_MEDKIT:
		trap_S_StartSound (NULL, es->number, CHAN_BODY, cgs.media.medkitSound );
		break;
	}

}
*/
/*
================
CG_ItemPickup

A new item was picked up this frame
================
*/
static void CG_ItemPickup( int itemNum ) {
	cg.itemPickup = itemNum;
	cg.itemPickupTime = cg.time;
	cg.itemPickupBlendTime = cg.time;
}


/*
================
CG_PainEvent

Also called by playerstate transition
================
*/
void CG_PainEvent( centity_t *cent, int health ) {
	char	*snd;

	// don't do more than two pain sounds a second
	if ( cg.time - cent->pe.painTime < 500 ) {
		return;
	}
	if(CG_PointContents(cent->world_headpos, cent->currentState.number) & (CONTENTS_WATER | CONTENTS_SLIME | CONTENTS_LAVA))
	{ // under fluid: play a gurp sound instead of a normal pain sound
		if (health <= 0) {
			snd =  "*drown.wav";
		} else if (rand()&1) {
			snd = "sound/player/gurp1.wav";
		} else {
			snd = "sound/player/gurp2.wav";
		}
	}
	else if ( health < 25 ) {
		snd = "*pain25_1.wav";
	} else if ( health < 50 ) {
		snd = "*pain50_1.wav";
	} else if ( health < 75 ) {
		snd = "*pain75_1.wav";
	} else {
		snd = "*pain100_1.wav";
	}
	trap_S_StartSound( NULL, cent->currentState.number, CHAN_VOICE, 
		CG_CustomSound( cent->currentState.number, snd ) );

	// save pain time for programitic twitch animation
	cent->pe.painTime = cg.time;
	cent->pe.painDirection ^= 1;
}

/*
==============
CG_EntityEvent

An entity has an event value
also called by CG_CheckPlayerstateEvents
==============
*/
static void CG_FlameEjectSparks( vec3_t org, vec3_t dir ) {
	localEntity_t	*le;
	refEntity_t		*re;

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	VectorCopy( org, le->pos.trBase );
	VectorScale(dir, random() * 200 + 100, le->pos.trDelta);
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time;
	le->leType = LE_SHRAPNEL;
	le->startTime = cg.time;
	le->endTime = cg.time + random() * 5000;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	le->radius = 0.25;
	le->leFlags = cgs.media.tracerShader;
	le->bounceFactor = 1;
}

localEntity_t *CG_SpurtBlood( vec3_t org, vec3_t dir, int amount ) 
{
	localEntity_t	*le;
	refEntity_t		*re;

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	VectorCopy( org, le->pos.trBase );
	VectorScale(dir, random() * 100 + 50 * sqrt(amount), le->pos.trDelta);
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time;
	le->leType = LE_SHRAPNEL;
	le->leMarkType = LEMT_BLOOD;
	le->leBounceSoundType = LEBS_BLOOD;
	le->startTime = cg.time;
	le->endTime = cg.time + 5000 + random() * 2000;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	le->radius = amount * 0.1;
	le->leFlags = cgs.media.bloodMarkShader;
	le->bounceFactor = 0;
	
	return le;
}

localEntity_t *CG_LaunchGib( vec3_t origin, vec3_t velocity, qhandle_t hModel );
#define	DEBUGNAME(x) if(cg_debugEvents.integer){CG_Printf(x"\n");}
void CG_EntityEvent( centity_t *cent, vec3_t position ) {
	entityState_t	*es;
	int				event, weapon;
	vec3_t			dir;
	const char		*s;
	int				clientNum;
	clientInfo_t	*ci;

	es = &cent->currentState;
	event = es->event & ~EV_EVENT_BITS;
	if(bg_itemlist[es->weapon].giType == IT_WEAPON)
	{
		weapon = bg_itemlist[es->weapon].giTag;
	}
	else
	{
		weapon = WP_NONE;
	}
	if ( cg_debugEvents.integer ) {
		CG_Printf( "ent:%3i  event:%3i ", es->number, event );
	}

	if ( !event ) {
		DEBUGNAME("ZEROEVENT");
		return;
	}

	clientNum = es->clientNum;
	if ( clientNum < 0 || clientNum >= MAX_CLIENTS ) {
		clientNum = 0;
	}
	ci = &cgs.clientinfo[ clientNum ];

	switch ( event ) {
	//
	// movement generated events
	//
	case EV_FOOTSTEP:
		DEBUGNAME("EV_FOOTSTEP");
		trap_S_StartSound (NULL, es->number, CHAN_BODY, 
			cgs.media.footsteps[ ci->footsteps ][rand()&3] );
		break;
	case EV_FOOTSTEP_METAL:
		DEBUGNAME("EV_FOOTSTEP_METAL");
		trap_S_StartSound (NULL, es->number, CHAN_BODY, 
			cgs.media.footsteps[ FOOTSTEP_METAL ][rand()&3] );
		break;
	case EV_FOOTSPLASH:
		DEBUGNAME("EV_FOOTSPLASH");
		trap_S_StartSound (NULL, es->number, CHAN_BODY, 
			cgs.media.footsteps[ FOOTSTEP_SPLASH ][rand()&3] );
		break;
	case EV_FOOTWADE:
		DEBUGNAME("EV_FOOTWADE");
		trap_S_StartSound (NULL, es->number, CHAN_BODY, 
			cgs.media.footsteps[ FOOTSTEP_SPLASH ][rand()&3] );
		break;
		
	case EV_SWIM:
		DEBUGNAME("EV_SWIM");
		trap_S_StartSound (NULL, es->number, CHAN_BODY, 
			cgs.media.footsteps[ FOOTSTEP_SPLASH ][rand()&3] );
		break;


	case EV_FALL_SHORT:
		DEBUGNAME("EV_FALL_SHORT");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.landSound );
		if ( clientNum == cg.predictedPlayerState.clientNum ) {
			// smooth landing z changes
			cg.landChange = -8;
			cg.landTime = cg.time;
		}
		break;
	case EV_FALL_MEDIUM:
		DEBUGNAME("EV_FALL_MEDIUM");
		// use normal pain sound trap_S_StartSound( NULL, es->number, CHAN_VOICE, CG_CustomSound( es->number, "*pain100_1.wav" ) );
		if ( clientNum == cg.predictedPlayerState.clientNum ) {
			// smooth landing z changes
			cg.landChange = -16;
			cg.landTime = cg.time;
		}
		break;
	case EV_FALL_FAR:
		DEBUGNAME("EV_FALL_FAR");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, CG_CustomSound( es->number, "*fall1.wav" ) );
		cent->pe.painTime = cg.time;	// don't play a pain sound right after this
		if ( clientNum == cg.predictedPlayerState.clientNum ) {
			// smooth landing z changes
			cg.landChange = -24;
			cg.landTime = cg.time;
		}
		break;

	case EV_STEP_4:
	case EV_STEP_8:
	case EV_STEP_12:
	case EV_STEP_16:		// smooth out step up transitions
		DEBUGNAME("EV_STEP");
	{
		float	oldStep;
		int		delta;
		int		step;

		if ( clientNum != cg.predictedPlayerState.clientNum ) {
			break;
		}
		// if we are interpolating, we don't need to smooth steps
		if ( cg.demoPlayback || (cg.snap->ps.pm_flags & PMF_FOLLOW) ||
			cg_nopredict.integer || cg_syncronousClients.integer ) {
			break;
		}
		// check for stepping up before a previous step is completed
		delta = cg.time - cg.stepTime;
		if (delta < STEP_TIME) {
			oldStep = cg.stepChange * (STEP_TIME - delta) / STEP_TIME;
		} else {
			oldStep = 0;
		}

		// add this amount
		step = 4 * (event - EV_STEP_4 + 1 );
		cg.stepChange = oldStep + step;
		if ( cg.stepChange > MAX_STEP_CHANGE ) {
			cg.stepChange = MAX_STEP_CHANGE;
		}
		cg.stepTime = cg.time;
		break;
	}

	case EV_JUMP_PAD:
		DEBUGNAME("EV_JUMP_PAD");
		// boing sound at origin, jump sound on player
		trap_S_StartSound ( cent->lerpOrigin, -1, CHAN_VOICE, cgs.media.jumpPadSound );
		trap_S_StartSound (NULL, es->number, CHAN_VOICE, CG_CustomSound( es->number, "*jump1.wav" ) );
		break;

	case EV_JUMP:
		DEBUGNAME("EV_JUMP");
		trap_S_StartSound (NULL, es->number, CHAN_VOICE, CG_CustomSound( es->number, "*jump1.wav" ) );
		break;
	case EV_TAUNT:
		DEBUGNAME("EV_TAUNT");
		trap_S_StartSound (NULL, es->number, CHAN_VOICE, CG_CustomSound( es->number, "*taunt.wav" ) );
		break;
	case EV_WATER_TOUCH:
		DEBUGNAME("EV_WATER_TOUCH");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.watrInSound );
		break;
	case EV_WATER_LEAVE:
		DEBUGNAME("EV_WATER_LEAVE");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.watrOutSound );
		break;
	case EV_WATER_UNDER:
		DEBUGNAME("EV_WATER_UNDER");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.watrUnSound );
		break;
	case EV_WATER_CLEAR:
		DEBUGNAME("EV_WATER_CLEAR");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, CG_CustomSound( es->number, "*gasp.wav" ) );
		break;

	case EV_ITEM_PICKUP:
		DEBUGNAME("EV_ITEM_PICKUP");
		{
			gitem_t	*item;
			int		index;

			index = es->eventParm;		// player predicted

			if ( index < 1 || index >= bg_numItems ) {
				break;
			}
			item = &bg_itemlist[ index ];

			// powerups and team items will have a separate global sound, this one
			// will be played at prediction time
			if ( item->giType == IT_POWERUP || (item->giType == IT_WEAPON && (item->giTag == WP_REDFLAG || item->giTag == WP_BLUEFLAG || item->giTag == WP_BALL))) {
				trap_S_StartSound (NULL, es->number, CHAN_AUTO,	trap_S_RegisterSound( "sound/items/n_health.wav" ) );
				if(item->giType == IT_POWERUP && item->giTag == PW_FLIGHT)
				{
					int i;
					for(i = 0; i < 16; i++)
					{
						vec3_t pop;
						localEntity_t *le;

						pop[0] = rand() % 400 - 200;
						pop[1] = rand() % 400 - 200;
						pop[2] = rand() % 800 + 400;

						le = CG_LaunchGib( cent->lerpOrigin, pop, cgs.media.flightFeatherModel);
						le->refEntity.customShader = cgs.media.flightWingsShader;			
						le->leFlags |= LEF_TUMBLE;
						le->bounceFactor = 0.1;
						le->leBounceSoundType = LEBS_FEATHER;
						le->leMarkType = LEMT_BLOOD;
						le->angles.trBase[0] = rand() % 360;
						le->angles.trBase[1] = rand() % 360;
						le->angles.trBase[2] = rand() % 360;
						le->angles.trDelta[0] = rand() % 300;
						le->angles.trDelta[1] = rand() % 300;
						le->angles.trDelta[2] = rand() % 300;
						le->angles.trType = TR_LINEAR;
						le->angles.trTime = cg.time;
						le->endTime = cg.time + 100000000;

					}
				}
			} else {
				trap_S_StartSound (NULL, es->number, CHAN_AUTO,	trap_S_RegisterSound( item->pickup_sound ) );
			}
			// show icon and name on status bar
			if ( es->number == cg.snap->ps.clientNum ) {
				CG_ItemPickup( index );
			}
		}
		break;

	case EV_GLOBAL_ITEM_PICKUP:
		DEBUGNAME("EV_GLOBAL_ITEM_PICKUP");
		{
			gitem_t	*item;
			int		index;

			index = es->eventParm;		// player predicted

			if ( index < 1 || index >= bg_numItems ) {
				break;
			}
			item = &bg_itemlist[ index ];
			// powerup pickups are global
			trap_S_StartSound (NULL, cg.snap->ps.clientNum, CHAN_AUTO, trap_S_RegisterSound( item->pickup_sound ) );

			// show icon and name on status bar
			if ( es->number == cg.snap->ps.clientNum ) {
				CG_ItemPickup( index );
			}
		}
		break;

	//
	// weapon events
	//
	case EV_RELOAD:
		DEBUGNAME("EV_RELOAD");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.reloadSound );
	case EV_NOAMMO:
		DEBUGNAME("EV_NOAMMO");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.noAmmoSound );
		break;
	case EV_CHANGE_WEAPON:
		DEBUGNAME("EV_CHANGE_WEAPON");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.switchGunSound);
		break;
	case EV_CHANGE_STANCE:
		DEBUGNAME("EV_CHANGE_STANCE");
		break;
	case EV_DROP_WEAPON:
		DEBUGNAME("EV_DROP_WEAPON");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.selectSound );
		break;
	case EV_FIRE_WEAPON:
		DEBUGNAME("EV_FIRE_WEAPON");
		CG_FireWeapon( cent );
		break;
	case EV_FIRE_WEAPON_FROMITEM:
		DEBUGNAME("EV_FIRE_WEAPON_FROMITEM");
		CG_FireWeaponFromItem( cent );
		break;

//=================================================================

	//
	// other events
	//
	case EV_PLAYER_TELEPORT_IN:
		DEBUGNAME("EV_PLAYER_TELEPORT_IN");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.teleInSound );
		CG_SpawnEffect( position);
		break;

	case EV_PLAYER_TELEPORT_OUT:
		DEBUGNAME("EV_PLAYER_TELEPORT_OUT");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.teleOutSound );
		CG_SpawnEffect(  position);
		break;

	case EV_ITEM_POP:
		DEBUGNAME("EV_ITEM_POP");
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.respawnSound );
		break;
	case EV_ITEM_RESPAWN:
		DEBUGNAME("EV_ITEM_RESPAWN");
		cent->miscTime = cg.time;	// scale up from this
		trap_S_StartSound (NULL, es->number, CHAN_AUTO, cgs.media.respawnSound );
		break;

	case EV_GRENADE_BOUNCE:
		DEBUGNAME("EV_GRENADE_BOUNCE");
		if(es->eventParm)
		{
			if(es->eventParm > 512)
				trap_S_StartSound (NULL, es->number, CHAN_WEAPON, trap_S_RegisterSound("sound/weapons/melee/fstatck.wav") );
			if(es->eventParm > 256)
				trap_S_StartSound (NULL, es->number, CHAN_VOICE, trap_S_RegisterSound("sound/weapons/melee/fstatck.wav") );
			if(es->eventParm > 128)
				trap_S_StartSound (NULL, es->number, CHAN_ITEM, trap_S_RegisterSound("sound/weapons/melee/fstatck.wav") );
			if(es->eventParm > 64)
				trap_S_StartSound (NULL, es->number, CHAN_BODY, trap_S_RegisterSound("sound/weapons/melee/fstatck.wav") );
			trap_S_StartSound (NULL, es->number, CHAN_AUTO, trap_S_RegisterSound("sound/weapons/melee/fstatck.wav") );
		}
		else if ( rand() & 1 ) {
			trap_S_StartSound (NULL, es->number, CHAN_AUTO, trap_S_RegisterSound("sound/weapons/grenade/hgrenb1a.wav") );
		} else {
			trap_S_StartSound (NULL, es->number, CHAN_AUTO, trap_S_RegisterSound("sound/weapons/grenade/hgrenb2a.wav") );
		}
		break;

	//
	// missile impacts
	//
	case EV_MISSILE_HIT:
		DEBUGNAME("EV_MISSILE_HIT");
		ByteToDir( es->eventParm, dir );
		if(es->powerups & (1 << PW_QUAD))
			es->weapon |= 16;
		CG_MissileHitPlayer( es->weapon, position, dir, es->otherEntityNum );
		break;

	case EV_MISSILE_HIT_ROBOT:
		DEBUGNAME("EV_MISSILE_HIT_ROBOT");
		ByteToDir( es->eventParm, dir );
		if(es->powerups & (1 << PW_QUAD))
			es->weapon |= 16;
		CG_MissileHitRPlayer( es->weapon, position, dir, es->otherEntityNum );
		break;

	case EV_MISSILE_MISS:
		DEBUGNAME("EV_MISSILE_MISS");
		ByteToDir( es->eventParm, dir );
		if(weapon == WP_MACHINEGUN || weapon == WP_CHAINGUN)
		{
			vec4_t rgba;
			if(es->powerups & (1 << PW_QUAD))
			{
				rgba[0] = 1;
				rgba[1] = 1;
				rgba[2] = 1;
				rgba[3] = 0.5;
			}
			else
			{
				rgba[0] = 0;
				rgba[1] = 0.5;
				rgba[2] = 1;
				rgba[3] = 1;
			}
			CG_Tracer(es->origin, es->origin2, 1, cgs.media.tracerShader, rgba);
		}
		if(es->powerups & (1 << PW_QUAD))
		{
			es->weapon |= 16;
		}
		CG_MissileHitWall( es->weapon, 0, position, dir );
		break;

	case EV_RAILTRAIL:
		DEBUGNAME("EV_RAILTRAIL");
		//weapon = WP_RAILGUN;
		ci->lastfired = cg.time;
		// if the end was on a nomark surface, don't make an explosion
		if ( es->eventParm != 255 ) {
			ByteToDir( es->eventParm, dir );
			CG_MissileHitWall( WP_RAILGUN, es->clientNum, position, dir );
		}
		ci->powerups = cent->currentState.powerups;
		CG_RailTrail( ci, es->origin2, es->pos.trBase );
		break;

	case EV_SHOTGUN:
		DEBUGNAME("EV_SHOTGUN");
		CG_ShotgunFire( es );
		break;

	case EV_GENERAL_SOUND:
		if(es->eventParm == 8 && (es->legsAnim & ANIM_CLIMBWALLS)) // "*fallingi.wav" - wallclimbers aren't actually falling
		{
			break;
		}
		DEBUGNAME("EV_GENERAL_SOUND");
		if ( cgs.gameSounds[ es->eventParm ] ) {
			trap_S_StartSound (NULL, es->number, CHAN_VOICE, cgs.gameSounds[ es->eventParm ] );
		} else {
			s = CG_ConfigString( CS_SOUNDS + es->eventParm );
			trap_S_StartSound (NULL, es->number, CHAN_VOICE, CG_CustomSound( es->number, s ) );
		}
		break;

	case EV_GLOBAL_SOUND:	// play from the player's head so it never diminishes
		DEBUGNAME("EV_GLOBAL_SOUND");
		if ( cgs.gameSounds[ es->eventParm ] ) {
			trap_S_StartSound (NULL, cg.snap->ps.clientNum, CHAN_AUTO, cgs.gameSounds[ es->eventParm ] );
		} else {
			s = CG_ConfigString( CS_SOUNDS + es->eventParm );
			trap_S_StartSound (NULL, cg.snap->ps.clientNum, CHAN_AUTO, CG_CustomSound( es->number, s ) );
		}
		break;

	case EV_ONFIRE:
	{
		int t;
		for(t = 0; t < 5; t++)
		{
			vec3_t org, dir;
			localEntity_t *le;
			BG_EvaluateTrajectory( &cent->currentState.pos, cg.time - t * 20, org );
			org[0] += crandom() * 8;
			org[1] += crandom() * 8;
			org[2] += crandom() * 12 + 8;
			dir[0] = crandom() * 8;
			dir[1] = crandom() * 8;
			dir[2] = crandom() * 8 + 32;
			le = CG_SmokePuff(org, dir, 10 + crandom() * 3, 1, 1, 1, 1, 600, cg.time, LEF_MINSIZE, cgs.media.flamethrowerShader);
			if(cent->currentState.eType == ET_ITEM)
			{
				CG_FlameEjectSparks(cent->lerpOrigin, bytedirs[rand() & 0xff]);
			}
		}
		break;
	}
	case EV_BLOOD_SPURT:			
	{
		CG_SpurtBlood(es->origin, bytedirs[es->eventParm], es->powerups);
		break;
	}
	case EV_BLOOD:			
	{
		vec3_t pos;
		int radius = (sqrt(es->eventParm) + (rand()&4) + 2) * 3;
		pos[0] = position[0] + rand() % 25 - 12;
		pos[1] = position[1] + rand() % 25 - 12;
		pos[2] = position[2] + rand() % 25;
		CG_SpurtBlood(pos, vec3_origin, radius);
		break;
	}
	case EV_USE_BERZERK:
	case EV_PAIN:
		// local player sounds are triggered in CG_CheckLocalSounds,
		// so ignore events on the player
		DEBUGNAME("EV_PAIN");
		//CG_Printf("got EV_PAIN\n");
		if ( cent->currentState.number != cg.snap->ps.clientNum ) {
			CG_PainEvent( cent, es->eventParm );
		}
		if(cent->currentState.powerups & (1 << PW_FLIGHT))
		{
			int i;
			for(i = 0; i < 4; i++)
			{
				vec3_t pop;
				localEntity_t *le;

				pop[0] = rand() % 400 - 200;
				pop[1] = rand() % 400 - 200;
				pop[2] = rand() % 800 + 400;

				le = CG_LaunchGib( cent->lerpOrigin, pop, cgs.media.flightFeatherModel);
				le->refEntity.customShader = cgs.media.flightWingsShader;			
				le->leFlags |= LEF_TUMBLE;
				le->leBounceSoundType = LEBS_FEATHER;
				le->leMarkType = LEMT_BLOOD;
				le->angles.trBase[0] = rand() % 360;
				le->angles.trBase[1] = rand() % 360;
				le->angles.trBase[2] = rand() % 360;
				le->angles.trDelta[0] = rand() % 300;
				le->angles.trDelta[1] = rand() % 300;
				le->angles.trDelta[2] = rand() % 300;
				le->angles.trType = TR_LINEAR;
				le->angles.trTime = cg.time;
				le->endTime = cg.time + 100000000;
			}
		}
		break;

	
	case EV_HEAD_EXPLODEY1:
	case EV_HEAD_EXPLODEY2:
	case EV_HEAD_EXPLODEY3:
		DEBUGNAME("EV_HEAD_EXPLODEYx");
		ci->head_explodey = 16;
		trap_S_StartSound( NULL, es->number, CHAN_VOICE, cgs.media.gibSound );
		{
			vec3_t end;
			int radius;
			trace_t trace;
			end[0] = position[0];
			end[1] = position[1];
			end[2] = position[2] - 128; 
			radius = 16 + (rand()&31);
			CG_Trace(&trace, position, vec3_origin, vec3_origin, end, cent->currentState.number, MASK_SOLID);
			if(trace.fraction < 1)
				CG_ImpactMark( cgs.media.bloodMarkShader, trace.endpos, trace.plane.normal, random()*360,
					1,1,1,1, qtrue, radius, qfalse );
		}
		break;
	case EV_DEATH1: 
	case EV_DEATH2: 
	case EV_DEATH3: 
		DEBUGNAME("EV_DEATHx");		
		trap_S_StartSound( NULL, es->number, CHAN_VOICE, CG_CustomSound( es->number, va("*death%i.wav", event - EV_DEATH1 + 1) ) );
#if 0
		{
			vec3_t end;
			int radius;
			trace_t trace;
			end[0] = position[0];
			end[1] = position[1];
			end[2] = position[2] - 128; 
			radius = 16 + (rand()&31);
			CG_Trace(&trace, position, vec3_origin, vec3_origin, end, cent->currentState.number, MASK_SOLID);
			if(trace.fraction < 1)
				CG_ImpactMark( cgs.media.bloodMarkShader, trace.endpos, trace.plane.normal, random()*360,
					1,1,1,1, qtrue, radius, qfalse );
		}
#endif 
		break;
		
	case EV_OBITUARY:
		DEBUGNAME("EV_OBITUARY");
		CG_Obituary( es );
		break;

	//
	// powerup events
	//
	case EV_POWERUP_QUAD:
		DEBUGNAME("EV_POWERUP_QUAD");
		if ( es->number == cg.snap->ps.clientNum ) {
			cg.powerupActive = PW_QUAD;
			cg.powerupTime = cg.time;
		}
		trap_S_StartSound (NULL, es->number, CHAN_ITEM, cgs.media.quadSound );
		break;
	case EV_POWERUP_BATTLESUIT:
		DEBUGNAME("EV_POWERUP_BATTLESUIT");
		if ( es->number == cg.snap->ps.clientNum ) {
			cg.powerupActive = PW_BATTLESUIT;
			cg.powerupTime = cg.time;
		}
		trap_S_StartSound (NULL, es->number, CHAN_ITEM, trap_S_RegisterSound("sound/items/protect3.wav") );
		break;
	case EV_POWERUP_REGEN:
		DEBUGNAME("EV_POWERUP_REGEN");
		if ( es->number == cg.snap->ps.clientNum ) {
			cg.powerupActive = PW_REGEN;
			cg.powerupTime = cg.time;
		}
		trap_S_StartSound (NULL, es->number, CHAN_ITEM, trap_S_RegisterSound("sound/items/regen.wav") );
		break;

	case EV_SPAWN_CORPSE:
		DEBUGNAME("EV_SPAWN_CORPSE");
		cent->CorpseIndex = CG_ClientInfoForCorpse(es->eventParm);
		break;
	case EV_GIB_PLAYER:
		DEBUGNAME("EV_GIB_PLAYER");
		trap_S_StartSound( NULL, es->number, CHAN_BODY, cgs.media.gibSound );
		CG_GibPlayer( cent->lerpOrigin, cent->currentState.pos.trDelta, es->eventParm );
		
		break;
	case EV_GIB_NOBLOOD:
		DEBUGNAME("EV_GIB_NOBLOOD");
		//CG_Printf("Noblood?\n");
		trap_S_StartSound( NULL, es->number, CHAN_BODY, cgs.media.gibSound );
		CG_GibNoBlood( cent->lerpOrigin, cent->currentState.pos.trDelta, es->eventParm );
		break;

	case EV_DEBUG_LINE:
		DEBUGNAME("EV_DEBUG_LINE");
		CG_Beam( cent );
		break;
	default:
		DEBUGNAME("UNKNOWN");
		CG_Error( "Unknown event: %i", event );
		break;
	}

}


/*
==============
CG_CheckEvents

==============
*/
void CG_CheckEvents( centity_t *cent ) {
	// check for event-only entities
	if ( cent->currentState.eType > ET_EVENTS ) {
		if ( cent->previousEvent ) {
			return;	// already fired
		}
		cent->previousEvent = 1;

		cent->currentState.event = cent->currentState.eType - ET_EVENTS;
	} else {
		// check for events riding with another entity
		if ( cent->currentState.event == cent->previousEvent ) {
			return;
		}
		cent->previousEvent = cent->currentState.event;
		if ( ( cent->currentState.event & ~EV_EVENT_BITS ) == 0 ) {
			return;
		}
	}

	// calculate the position at exactly the frame time
	BG_EvaluateTrajectory( &cent->currentState.pos, cg.snap->serverTime, cent->lerpOrigin );
	CG_SetEntitySoundPosition( cent );

	CG_EntityEvent( cent, cent->lerpOrigin );
}

