// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_ents.c -- present snapshot entities, happens every single frame

#include "cg_local.h"


/*
======================
CG_PositionEntityOnTag

Modifies the entities position and axis by the given
tag location
======================
*/
void CG_PositionEntityOnTag( refEntity_t *entity, const refEntity_t *parent, 
							qhandle_t parentModel, char *tagName ) {
	int				i;
	orientation_t	lerped;
	
	// lerp the tag
	trap_R_LerpTag( &lerped, parentModel, parent->oldframe, parent->frame,
		1.0 - parent->backlerp, tagName );

	// FIXME: allow origin offsets along tag?
	VectorCopy( parent->origin, entity->origin );
	for ( i = 0 ; i < 3 ; i++ ) {
		VectorMA( entity->origin, lerped.origin[i], parent->axis[i], entity->origin );
	}

	// had to cast away the const to avoid compiler problems...
	MatrixMultiply( lerped.axis, ((refEntity_t *)parent)->axis, entity->axis );
	entity->backlerp = parent->backlerp;
}


/*
======================
CG_PositionRotatedEntityOnTag

Modifies the entities position and axis by the given
tag location
======================
*/
void CG_PositionRotatedEntityOnTag( refEntity_t *entity, const refEntity_t *parent, 
							qhandle_t parentModel, char *tagName ) {
	int				i;
	orientation_t	lerped;
	vec3_t			tempAxis[3];

//AxisClear( entity->axis );
	// lerp the tag
	trap_R_LerpTag( &lerped, parentModel, parent->oldframe, parent->frame,
		1.0 - parent->backlerp, tagName );

	// FIXME: allow origin offsets along tag?
	VectorCopy( parent->origin, entity->origin );
	for ( i = 0 ; i < 3 ; i++ ) {
		VectorMA( entity->origin, lerped.origin[i], parent->axis[i], entity->origin );
	}

	// had to cast away the const to avoid compiler problems...
	MatrixMultiply( entity->axis, lerped.axis, tempAxis );
	MatrixMultiply( tempAxis, ((refEntity_t *)parent)->axis, entity->axis );
}



/*
==========================================================================

FUNCTIONS CALLED EACH FRAME

==========================================================================
*/

/*
======================
CG_SetEntitySoundPosition

Also called by event processing code
======================
*/
void CG_SetEntitySoundPosition( centity_t *cent ) {
	if ( cent->currentState.solid == SOLID_BMODEL ) {
		vec3_t	origin;
		float	*v;

		v = cgs.inlineModelMidpoints[ cent->currentState.modelindex ];
		VectorAdd( cent->lerpOrigin, v, origin );
		trap_S_UpdateEntityPosition( cent->currentState.number, origin );
	} else {
		trap_S_UpdateEntityPosition( cent->currentState.number, cent->lerpOrigin );
	}
}

/*
==================
CG_EntityEffects

Add continuous entity effects, like local entity emission and lighting
==================
*/
static void CG_EntityEffects( centity_t *cent ) {

	// update sound origins
	CG_SetEntitySoundPosition( cent );
	if(cent->currentState.eType == ET_PLAYER || cent->currentState.eType == ET_INVISIBLE )
		return; // players currentState differently
	// add loop sound
	if ( cent->currentState.loopSound ) {
		trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, 
			cgs.gameSounds[ cent->currentState.loopSound ] );
	}


	// constant light glow
	if ( cent->currentState.constantLight ) {
		int		cl;
		int		i, r, g, b;

		cl = cent->currentState.constantLight;
		r = cl & 255;
		g = ( cl >> 8 ) & 255;
		b = ( cl >> 16 ) & 255;
		i = ( ( cl >> 24 ) & 255 ) * 4;
		trap_R_AddLightToScene( cent->lerpOrigin, i, r, g, b );
	}

}


/*
==================
CG_General
==================
*/
static void CG_General( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// if set to invisible, skip
	if (!s1->modelindex) {
		return;
	}

	memset (&ent, 0, sizeof(ent));

	// set frame

	ent.frame = s1->frame;
	ent.oldframe = ent.frame;
	ent.backlerp = 0;

	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);

	ent.hModel = cgs.gameModels[s1->modelindex];

	// player model
	if (s1->number == cg.snap->ps.clientNum) {
		ent.renderfx |= RF_THIRD_PERSON;	// only draw from mirrors
	}

	// convert angles to axis
	AnglesToAxis( cent->lerpAngles, ent.axis );

	// add to refresh list
	trap_R_AddRefEntityToScene (&ent);
}

/*
==================
CG_Speaker

Speaker entities can automatically play sounds
==================
*/
static void CG_Speaker( centity_t *cent ) {
	if ( ! cent->currentState.clientNum ) {	// FIXME: use something other than clientNum... 
		return;		// not auto triggering
	}

	if ( cg.time < cent->miscTime ) {
		return;
	}

	trap_S_StartSound (NULL, cent->currentState.number, CHAN_ITEM, cgs.gameSounds[cent->currentState.eventParm] );

	//	ent->s.frame = ent->wait * 10;
	//	ent->s.clientNum = ent->random * 10;
	cent->miscTime = cg.time + cent->currentState.frame * 100 + cent->currentState.clientNum * 100 * crandom();
}

/*
==================
CG_Item
==================
*/
void CG_AddWeaponWithPowerups( refEntity_t *gun, int powerups );
void CG_RenderItem(gitem_t *item, short quantity, int powerups, vec3_t org, vec3_t axis[3], vec3_t dir)
{
	refEntity_t			ent;
	memset (&ent, 0, sizeof(ent));		
	ent.hModel = cg_items[item - bg_itemlist].models[0];
	
	VectorCopy(org, ent.origin);
	VectorCopy(axis[0], ent.axis[0]);
	VectorCopy(axis[1], ent.axis[1]);
	VectorCopy(axis[2], ent.axis[2]);


	if(item->giType == IT_WEAPON && cg_weapons[item->giTag].weaponSkin)
	{ // if it has a custom skin, then ammo is shown on the skin.
		ent.customSkin = cg_weapons[item->giTag].weaponSkin;
		ent.shaderRGBA[0] = (quantity * 255.0 / item->quantity);
		ent.shaderRGBA[1] = (quantity * 255.0 / item->quantity);
		ent.shaderRGBA[2] = (quantity * 255.0 / item->quantity);
		ent.shaderRGBA[3] = (128 + quantity * 127.0 / item->quantity);					
	}

	if(item->giType == IT_WEAPON && cg_weapons[item->giTag].barrelModel)
	{ // ## Hentai - barrel is seperate model which must be drawn as well
		refEntity_t	barrel;					
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( ent.lightingOrigin, barrel.lightingOrigin );
		
		barrel.hModel = cg_weapons[item->giTag].barrelModel;
		if(item->giTag == WP_BALL)
		{
			barrel.axis[0][0] = 0.4;
			barrel.axis[1][1] = 0.4;
			barrel.axis[2][2] = 0.4;
		

			VectorCopy(ent.origin, barrel.origin);
			//trap_R_AddRefEntityToScene( &barrel );
			CG_AddWeaponWithPowerups(&barrel, powerups);
			ent.axis[0][0] = 0.22 * sin(cg.time / 100.0);
			ent.axis[0][1] = 0.22 * cos(cg.time / 100.0);
			ent.axis[0][2] = 0;
	
			ent.axis[1][0] = 0.22 * cos(cg.time / 100.0);
			ent.axis[1][1] = -0.22 * sin(cg.time / 100.0);
			ent.axis[1][2] = 0;
			ent.axis[2][0] = 0;
			ent.axis[2][1] = 0;
			ent.axis[2][2] = 0.18;

			ent.customShader = trap_R_RegisterShader("models/powerups/instant/regen");//.regenShader;

		}
		else if(item->giTag == WP_FLAMETHROWER)
		{
			barrel.customShader = cgs.media.armorShader;
			VectorScale(ent.axis[0], -0.6, barrel.axis[0]);
			VectorScale(ent.axis[1], 0.6, barrel.axis[1]);
			VectorScale(ent.axis[2], -0.6, barrel.axis[2]);
			VectorMA(ent.origin, 12, ent.axis[0], barrel.origin);
		}
		else
			CG_PositionEntityOnTag( &barrel, &ent, cg_weapons[item->giTag].weaponModel, "tag_barrel" );

		if(cg_weapons[item->giTag].weaponSkin)
		{ // if it has a custom skin, then ammo is shown on the skin.
			barrel.customSkin = cg_weapons[item->giTag].weaponSkin;
			barrel.shaderRGBA[0] = (quantity * 255.0 / item->quantity);
			barrel.shaderRGBA[1] = (quantity * 255.0 / item->quantity);
			barrel.shaderRGBA[2] = (quantity * 255.0 / item->quantity);
			barrel.shaderRGBA[3] = (128 + quantity * 127.0 / item->quantity);
					
		}
		CG_AddWeaponWithPowerups(&barrel, powerups);
		
	}

	else if(item->giType == IT_ARMOR)
	{
		VectorScale( ent.axis[0], -0.6, ent.axis[2] );
		VectorScale( ent.axis[1], 0.6, ent.axis[1] );
		CrossProduct(ent.axis[1], ent.axis[2], ent.axis[0]);
		//VectorScale( ent.axis[2], 0.6, ent.axis[2] );			
		ent.nonNormalizedAxes = qtrue;
	}

	if ( item->giType == IT_HEALTH && item->giTag == PW_BERZERK )
	{
		VectorScale(ent.axis[2], 0.2, ent.axis[0]);
		VectorScale(ent.axis[1], 0.2, ent.axis[1]);
		CrossProduct(ent.axis[0], ent.axis[1], ent.axis[2]);
	}
	if ( item->giType == IT_POWERUP )
	{
		ent.origin[2] += 5 + sin(cg.time / 1000.0) * 5;
	}
	if(item->giType != IT_AMMO) 
		CG_AddWeaponWithPowerups(&ent, powerups);
		//trap_R_AddRefEntityToScene(&ent);
	
	if(item->giType == IT_WEAPON )
	{ // draw ammo
		refEntity_t belt;
		vec3_t forward, right, dir;
		VectorCopy(ent.axis[0], forward);
		VectorCopy(ent.axis[1], right);
		right[2] = 0;
		VectorNormalize(right);
		switch(item->giTag)
		{
		default: // several don't show ammo
			ent.hModel = 0;
			break;
		case WP_RAILGUN:
			if(quantity >=0 && quantity <= 10)
			{
				ent.hModel = trap_R_RegisterModel( "models/ammo/railgun.md3" );
				ent.frame = quantity;
				CG_AddWeaponWithPowerups(&ent, powerups);
				ent.frame = 0;
			}
			break;
		case WP_MACHINEGUN:
			VectorMA(ent.origin, -1.0, ent.axis[0], ent.origin);
			VectorMA(ent.origin, 1, ent.axis[1], ent.origin);
			VectorScale(ent.axis[1], -1, ent.axis[1]);
			VectorMA(ent.origin, 3.6, ent.axis[2], ent.origin);
			CG_DrawAmmoBelt(quantity, item->giUses, powerups, ent.origin, ent.axis[0], ent.axis[1], vec3_origin);
			ent.hModel = 0;
			break;
		case WP_CHAINGUN:
			VectorMA(ent.origin, 7.5, ent.axis[0], ent.origin);
			VectorMA(ent.origin, 3.6, ent.axis[2], ent.origin);			
			CG_DrawAmmoBelt(quantity, item->giUses, powerups, ent.origin, ent.axis[0], ent.axis[1], vec3_origin);
			ent.hModel = 0;
			break;
		case WP_ROCKET_LAUNCHER:
			CG_PositionEntityOnTag( &belt, &ent, cg_weapons[item->giTag].weaponModel, "tag_flash" );	
			VectorMA(belt.origin, -6, belt.axis[0], belt.origin);
			VectorScale( belt.axis[2], -1, belt.axis[2] );
			CG_DrawAmmoBelt(quantity, item->giUses, powerups, belt.origin, belt.axis[0], belt.axis[2], dir);			
		}
	}
	else if((item->giType == IT_AMMO ) )
	{
		int i;
		switch(item->giTag)
		{
		default:
			CG_AddWeaponWithPowerups(&ent, powerups);
			break;		
		case AMMO_BULLETS:
			CG_DrawAmmoBelt(quantity, item->giTag, powerups, ent.origin, ent.axis[0], ent.axis[1], dir);
			break;
		case AMMO_SHELLS:
			ent.hModel = cgs.media.shotgunBrassModel;
			VectorScale( ent.axis[0], 0.6, ent.axis[0] );
			VectorScale( ent.axis[1], 0.6, ent.axis[1] );
			VectorScale( ent.axis[2], 0.6, ent.axis[2] );
			for(i = 0; i < quantity; i++)
			{	
				VectorCopy( org, ent.origin);
				ent.origin[0] += (i / 2) * 1.2 - quantity * 0.6;
				ent.origin[1] += (i % 2) * 1.2 - 0.6;
				CG_AddWeaponWithPowerups(&ent, powerups);
			}						
			break;			
		case AMMO_GRENADES:
			for(i = 0; i < quantity; i++)
			{
				ent.origin[1] += i * 2.5 - (quantity - 1) * 1.25;
				CG_AddWeaponWithPowerups(&ent, powerups);
			}
			break;
		case AMMO_ROCKETS:
			CG_DrawAmmoBelt(quantity, item->giTag, powerups, ent.origin, ent.axis[0], ent.axis[1], dir);
			break;
		case AMMO_SLUGS:
			ent.frame = quantity;
			CG_AddWeaponWithPowerups(&ent, powerups);
			break;
		case AMMO_CELLS:	
			VectorScale( ent.axis[0], 0.2, ent.axis[0] );
			VectorScale( ent.axis[1], 0.18, ent.axis[1] );
			VectorScale( ent.axis[2], 0.15, ent.axis[2] );
			CG_AddWeaponWithPowerups(&ent, powerups);
			break;
		case AMMO_NAPALM:
			ent.customShader = cgs.media.armorShader;						
			VectorScale( ent.axis[0], 1, ent.axis[0] );
			VectorScale( ent.axis[1], -1, ent.axis[1] );
			VectorScale( ent.axis[2], -1, ent.axis[2] );
			VectorMA(ent.origin, 0.5, ent.axis[2], ent.origin);
			CG_AddWeaponWithPowerups(&ent, powerups);
		
		}				
			
	}				
	else if ( item->giType == IT_POWERUP )
	{
		vec3_t spinAngles;
		if ( ( ent.hModel = cg_items[item - bg_itemlist ].models[1] ) != 0 )
		{
			ent.origin[2] += 12;
			spinAngles[1] = ( cg.time & 1023 ) * 360 / -1024.0f;
		}
		AnglesToAxis( spinAngles, ent.axis );
		trap_R_AddRefEntityToScene( &ent );
	}
	
}

static void CG_Item( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*es;
	gitem_t				*item;
	float				groundz;
	vec3_t end, mins, maxs;
	trace_t tr;
	es = &cent->currentState;
	if ( es->modelindex >= bg_numItems ) {
		CG_Error( "Bad item index %i on entity", es->modelindex );
	}

	// if set to invisible, skip
	if ( !es->modelindex || ( es->eFlags & EF_NODRAW ) ) {
		return;
	}


	item = &bg_itemlist[ es->modelindex ];
	if ( cg_simpleItems.integer) {
		memset( &ent, 0, sizeof( ent ) );
		ent.reType = RT_SPRITE;
		VectorCopy( cent->lerpOrigin, ent.origin );
		ent.radius = 14;
		ent.customShader = cg_items[es->modelindex].icon;
		ent.shaderRGBA[0] = 255;
		ent.shaderRGBA[1] = 255;
		ent.shaderRGBA[2] = 255;
		ent.shaderRGBA[3] = 255;
		trap_R_AddRefEntityToScene(&ent);
		return;
	}
	end[0] = cent->lerpOrigin[0];
	end[1] = cent->lerpOrigin[1];
	end[2] = cent->lerpOrigin[2] - 24;
	mins[0] = -10;
	mins[1] = -10;
	mins[2] = 0;
	maxs[0] = 10;
	maxs[1] = 10;
	maxs[2] = 0;
	CG_Trace( &tr, cent->lerpOrigin, mins, maxs, end, 0, MASK_SOLID );
	groundz = tr.endpos[2];
	
	if(item->giType == IT_WEAPON && item->giTag != WP_BALL)
	{
		if((item->giTag == WP_REDFLAG || item->giTag == WP_BLUEFLAG) && !(cent->currentState.eFlags & EF_DEAD))
		{
			cent->lerpOrigin[2] += 8;
		}
		else if(item->giTag == WP_REDFLAG || item->giTag == WP_BLUEFLAG)
		{
			if(tr.fraction == 1.0)
			{
				cent->rawAngles[YAW] = cent->lerpAngles[YAW] = cg.autoAnglesFast[YAW];
			
				cent->lerpOrigin[2] -= 15; // drop them to the ground
				cent->lerpAngles[PITCH] = 0;
				cent->lerpAngles[YAW] = cent->rawAngles[YAW];
				cent->lerpAngles[ROLL] = -45; // make it lean a little
			}
			else
			{
				cent->lerpOrigin[2] -= 12;
				cent->lerpAngles[PITCH] = 0;
				cent->lerpAngles[YAW] = cent->rawAngles[YAW];
				cent->lerpAngles[ROLL] = -85; // tip it on its side
			}
		}
		else
		{
			if(cent->currentState.groundEntityNum == ENTITYNUM_NONE)
				cent->rawAngles[YAW] = cent->lerpAngles[YAW] = cg.autoAnglesFast[YAW];
			
			cent->lerpAngles[PITCH] = 0;
			cent->lerpAngles[YAW] = cent->rawAngles[YAW];
			cent->lerpAngles[ROLL] = -45; // make it lean a little
			if(item->giTag == WP_FLAMETHROWER || item->giTag == WP_RAILGUN)
				cent->lerpAngles[ROLL] = -75; // make it lean a little more, so the ammo clip/canister doesn't clip through the ground
		}
	}
	else if ( item->giType == IT_AMMO ) 
	{
		float height = 2;
		switch(item->giTag)
		{
		case AMMO_SLUGS:
		case AMMO_NAPALM:
			height = 1;
			cent->lerpAngles[ROLL] = 90;
		case AMMO_BULLETS:
			height = 0.5;
		case AMMO_ROCKETS:
			cent->lerpOrigin[2] = groundz + height;
		default:
			break;
		}
	}
	AnglesToAxis(cent->lerpAngles, ent.axis);
	if ( item->giType == IT_WEAPON ) {		
		if(item->giTag == WP_REDFLAG || item->giTag == WP_BLUEFLAG)
		{
		}
		else
		{
			weaponInfo_t	*wi;
			wi = &cg_weapons[item->giTag];
			cent->lerpOrigin[0] -= 
				wi->weaponMidpoint[0] * ent.axis[0][0] +
				wi->weaponMidpoint[1] * ent.axis[1][0] +
				wi->weaponMidpoint[2] * ent.axis[2][0];
			cent->lerpOrigin[1] -= 
				wi->weaponMidpoint[0] * ent.axis[0][1] +
				wi->weaponMidpoint[1] * ent.axis[1][1] +
				wi->weaponMidpoint[2] * ent.axis[2][1];
			cent->lerpOrigin[2] -= 
				wi->weaponMidpoint[0] * ent.axis[0][2] +
				wi->weaponMidpoint[1] * ent.axis[1][2] +
				wi->weaponMidpoint[2] * ent.axis[2][2];
		}

	}
	if((item->giType == IT_POWERUP) || (cent->currentState.eFlags & EF_TELEPORT_BIT))
	{
		CG_RenderItem(item, cent->currentState.powerups, 0, cent->lerpOrigin, cg.autoAxisFast, vec3_origin);
	}
	else
	{
		CG_RenderItem(item, cent->currentState.powerups, 0, cent->lerpOrigin, ent.axis, vec3_origin);
	}
	return;
}

//============================================================================

/*
===============
CG_Missile
===============
*/
static void CG_Missile( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;
	const weaponInfo_t		*weapon;
	float frac = 0.7;
	
	s1 = &cent->currentState;
	if ( s1->weapon > WP_NUM_WEAPONS ) {
		s1->weapon = 0;
	}
	weapon = &cg_weapons[s1->weapon];

	// calculate the axis
	VectorCopy( s1->angles, cent->lerpAngles);

	// add trails
	if ( weapon->missileTrailFunc ) 
	{
		weapon->missileTrailFunc( cent, weapon );
	}

	// add dynamic light
	if(cent->currentState.powerups & (1 << PW_QUAD))
	{
		int radius = weapon->missileDlight;
		if(radius < 200)
			radius = 100 + radius / 2;
		trap_R_AddLightToScene(cent->lerpOrigin, radius, 
			0, 0.5, 1.0 );
	}

	else if ( weapon->missileDlight ) {
		trap_R_AddLightToScene(cent->lerpOrigin, weapon->missileDlight, 
			weapon->missileDlightColor[0], weapon->missileDlightColor[1], weapon->missileDlightColor[2] );
	}

	// add missile sound
	if ( weapon->missileSound ) {
		vec3_t	velocity;
		
		BG_EvaluateTrajectoryDelta( &cent->currentState.pos, cg.time, velocity );

		trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, velocity, weapon->missileSound );
	}

	if (cent->currentState.weapon == WP_MACHINEGUN || cent->currentState.weapon == WP_CHAINGUN )
	{
		vec4_t rgba;
		if(cent->oldorigin_time < cg.oldTime)
		{
			BG_EvaluateTrajectory( &cent->currentState.pos, cent->currentState.pos.trTime, cent->oldorigin);
			cent->oldorigin_time = cent->currentState.pos.trTime;
		}
		else if(cg.time - cent->oldorigin_time > 100)
		{
			BG_EvaluateTrajectory( &cent->currentState.pos, cg.oldTime, cent->oldorigin);
			cent->oldorigin_time = cg.oldTime;
		}
		if(cent->currentState.powerups & (1 << PW_QUAD))
		{
			rgba[0] = 1;
			rgba[1] = 1;
			rgba[2] = 1;
			rgba[3] = 0.5;
		}
		else
		{
			rgba[0] = 0;
			rgba[1] = 0.5;
			rgba[2] = 1;
			rgba[3] = 1;
		}
		CG_Tracer(cent->lerpOrigin, cent->oldorigin, 1, cgs.media.tracerShader, rgba);
		VectorCopy( cent->lerpOrigin, cent->oldorigin);
		cent->oldorigin_time = cg.time;
		return;
	}
	
	// create the render entity
	memset (&ent, 0, sizeof(ent));
	
	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);
	if ( cent->currentState.weapon == WP_PLASMAGUN ) {
		ent.reType = RT_SPRITE;
		ent.radius = 16;
		ent.rotation = 0;
		ent.customShader = cgs.media.plasmaBallShader;
		trap_R_AddRefEntityToScene( &ent );
		return;
	}
	else if(cent->currentState.weapon == WP_ROCKET_LAUNCHER)
	{
		AngleVectors(cent->currentState.angles, ent.axis[0], ent.axis[1], ent.axis[2]);	
	}
	else
	{
		if ( VectorNormalize2( s1->pos.trDelta, ent.axis[0] ) == 0 ) 
		{
			ent.axis[0][2] = 1;	
		}
	}


	// flicker between two skins
	ent.skinNum = cg.clientFrame & 1;
	ent.hModel = weapon->missileModel;
	ent.renderfx = weapon->missileRenderfx | RF_NOSHADOW;

	
	// spin as it moves
	if ( s1->pos.trType != TR_STATIONARY ) {
		RotateAroundDirection( ent.axis, cg.time / 4 );
	} else {
		RotateAroundDirection( ent.axis, s1->time );
	}

	
//	VectorScale( ent.axis[0], frac, ent.axis[0] );
//	VectorScale( ent.axis[1], frac, ent.axis[1] );
//	VectorScale( ent.axis[2], frac, ent.axis[2] );
		
	// add to refresh list, possibly with quad glow
	CG_AddRefEntityWithPowerups( &ent, s1->powerups, TEAM_FREE );
}

/*
===============
CG_Mover
===============
*/
static void CG_Mover( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);
	AnglesToAxis( cent->lerpAngles, ent.axis );

	ent.renderfx = RF_NOSHADOW;

	// flicker between two skins (FIXME?)
	ent.skinNum = ( cg.time >> 6 ) & 1;

	// get the model, either as a bmodel or a modelindex
	if ( s1->solid == SOLID_BMODEL ) {
		ent.hModel = cgs.inlineDrawModel[s1->modelindex];
	} else {
		ent.hModel = cgs.gameModels[s1->modelindex];
	}

	// add to refresh list
	trap_R_AddRefEntityToScene(&ent);

	// add the secondary model
	if ( s1->modelindex2 ) {
		ent.skinNum = 0;
		ent.hModel = cgs.gameModels[s1->modelindex2];
		trap_R_AddRefEntityToScene(&ent);
	}

}

/*
===============
CG_Beam

Also called as an event
===============
*/
void CG_Beam( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( s1->pos.trBase, ent.origin );
	VectorCopy( s1->origin2, ent.oldorigin );
	AxisClear( ent.axis );
	ent.reType = RT_BEAM;

	ent.renderfx = RF_NOSHADOW;

	// add to refresh list
	trap_R_AddRefEntityToScene(&ent);
}


/*
===============
CG_Portal
===============
*/
static void CG_Portal( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( cent->lerpOrigin, ent.origin );
	VectorCopy( s1->origin2, ent.oldorigin );
	ByteToDir( s1->eventParm, ent.axis[0] );
	PerpendicularVector( ent.axis[1], ent.axis[0] );

	// negating this tends to get the directions like they want
	// we really should have a camera roll value
	VectorSubtract( vec3_origin, ent.axis[1], ent.axis[1] );

	CrossProduct( ent.axis[0], ent.axis[1], ent.axis[2] );
	ent.reType = RT_PORTALSURFACE;
	ent.frame = s1->frame;		// rotation speed
	ent.skinNum = s1->clientNum/256.0 * 360;	// roll offset

	// add to refresh list
	trap_R_AddRefEntityToScene(&ent);
}


/*
=========================
CG_AdjustPositionForMover

Also called by client movement prediction code
=========================
*/
void CG_AdjustPositionForMover( const vec3_t in, int moverNum, int fromTime, int toTime, vec3_t out ) {
	centity_t	*cent;
	vec3_t	oldOrigin, origin, deltaOrigin;
	vec3_t	oldAngles, angles, deltaAngles;

	if ( moverNum <= 0 || moverNum >= ENTITYNUM_MAX_NORMAL ) {
		VectorCopy( in, out );
		return;
	}

	cent = &cg_entities[ moverNum ];
	if ( cent->currentState.eType != ET_MOVER ) {
		VectorCopy( in, out );
		return;
	}

	BG_EvaluateTrajectory( &cent->currentState.pos, fromTime, oldOrigin );
	BG_EvaluateTrajectory( &cent->currentState.apos, fromTime, oldAngles );

	BG_EvaluateTrajectory( &cent->currentState.pos, toTime, origin );
	BG_EvaluateTrajectory( &cent->currentState.apos, toTime, angles );

	VectorSubtract( origin, oldOrigin, deltaOrigin );
	VectorSubtract( angles, oldAngles, deltaAngles );

	VectorAdd( in, deltaOrigin, out );

	// FIXME: origin change when on a rotating object
}


/*
=============================
CG_InterpolateEntityPosition
=============================
*/
static void CG_InterpolateEntityPosition( centity_t *cent ) {
	vec3_t		current, next;
	float		f;

	// it would be an internal error to find an entity that interpolates without
	// a snapshot ahead of the current one
	if ( cg.nextSnap == NULL ) {
		CG_Error( "CG_InterpoateEntityPosition: cg.nextSnap == NULL" );
	}

	f = cg.frameInterpolation;

	// this will linearize a sine or parabolic curve, but it is important
	// to not extrapolate player positions if more recent data is available
	BG_EvaluateTrajectory( &cent->currentState.pos, cg.snap->serverTime, current );
	BG_EvaluateTrajectory( &cent->nextState.pos, cg.nextSnap->serverTime, next );

	cent->lerpOrigin[0] = current[0] + f * ( next[0] - current[0] );
	cent->lerpOrigin[1] = current[1] + f * ( next[1] - current[1] );
	cent->lerpOrigin[2] = current[2] + f * ( next[2] - current[2] );

	BG_EvaluateTrajectory( &cent->currentState.apos, cg.snap->serverTime, current );
	BG_EvaluateTrajectory( &cent->nextState.apos, cg.nextSnap->serverTime, next );

	cent->lerpAngles[0] = LerpAngle( current[0], next[0], f );
	cent->lerpAngles[1] = LerpAngle( current[1], next[1], f );
	cent->lerpAngles[2] = LerpAngle( current[2], next[2], f );

}

/*
===============
CG_CalcEntityLerpPositions

===============
*/
static void CG_CalcEntityLerpPositions( centity_t *cent ) {
	if ( cent->interpolate && cent->currentState.pos.trType == TR_INTERPOLATE ) {
		CG_InterpolateEntityPosition( cent );
		return;
	}
	
	// just use the current frame and evaluate as best we can
	BG_EvaluateTrajectory( &cent->currentState.pos, cg.time, cent->lerpOrigin );
	BG_EvaluateTrajectory( &cent->currentState.apos, cg.time, cent->lerpAngles );

	// adjust for riding a mover if it wasn't rolled into the predicted
	// player state
	if ( cent != &cg.predictedPlayerEntity ) {
		CG_AdjustPositionForMover( cent->lerpOrigin, cent->currentState.groundEntityNum, 
		cg.snap->serverTime, cg.time, cent->lerpOrigin );
	}
}
void CG_LoadClientInfo( clientInfo_t *ci );

int CG_ClientInfoForCorpse(int ModelIndex)
{
	int c;
	clientInfo_t newInfo;
	char *slash, *v;
	v = BG_GetCorpseModel(ModelIndex);
	Q_strncpyz( newInfo.modelName, v, sizeof( newInfo.modelName ) );
	slash = strchr( newInfo.modelName, '/' );
	if ( !slash ) {
		// modelName didn not include a skin name
		Q_strncpyz( newInfo.skinName, "default", sizeof( newInfo.skinName ) );
	} else {
		Q_strncpyz( newInfo.skinName, slash + 1, sizeof( newInfo.skinName ) );
		// truncate modelName
		*slash = 0;
	}
	slash = strchr(newInfo.skinName, ' ');
	if( !slash ) {
		Q_strncpyz( newInfo.headModelName, newInfo.modelName, sizeof( newInfo.headModelName ) );
		Q_strncpyz( newInfo.headSkinName, newInfo.skinName, sizeof( newInfo.headSkinName ) );
		//CG_Printf("No special head");
	}
	else
	{
		*slash = 0;
		Q_strncpyz( newInfo.headModelName, slash + 1, sizeof( newInfo.headModelName ) );			
		slash = strchr( newInfo.headModelName, '/' );
		if ( !slash ) {
			// modelName did not include a skin name
			Q_strncpyz( newInfo.headSkinName, "default", sizeof( newInfo.headSkinName ) );
		} else {
			Q_strncpyz( newInfo.headSkinName, slash + 1, sizeof( newInfo.headSkinName ) );
			// truncate modelName
			*slash = 0;
		}
		if(newInfo.headModelName[0] == '*') // use default
			Q_strncpyz( newInfo.headModelName, newInfo.modelName, sizeof( newInfo.headModelName ) );	
	}

	CG_LoadClientInfo( &newInfo );

	for(c = 1; c <= corpses; c++)
	{
		if(corpseinfo[c].legsModel == newInfo.legsModel && corpseinfo[c].headModel == newInfo.headModel && corpseinfo[c].legsSkin == newInfo.legsSkin)
		{			
			return c;
		}
	}
	corpses++;
	memcpy(corpseinfo[corpses].animations, newInfo.animations, sizeof(newInfo.animations));
	corpseinfo[corpses].torsoModel = newInfo.torsoModel;
	corpseinfo[corpses].legsModel = newInfo.legsModel;
	corpseinfo[corpses].headModel = newInfo.headModel;
	corpseinfo[corpses].torsoSkin = newInfo.torsoSkin;
	corpseinfo[corpses].legsSkin = newInfo.legsSkin;
	corpseinfo[corpses].headSkin = newInfo.headSkin;
	return corpses;
	
	
}


/*
===============
CG_AddCEntity

===============
*/
static void CG_AddCEntity( centity_t *cent ) {
	// event-only entities will have been dealt with already
	
	if ( cent->currentState.eType >= ET_EVENTS ) {
		return;
	}

	// calculate the current origin
	CG_CalcEntityLerpPositions( cent );

	// add automatic effects
	CG_EntityEffects( cent );

	switch ( cent->currentState.eType ) {
	default:
		CG_Error( "Bad entity type: %i\n", cent->currentState.eType );
		break;
	case ET_INVISIBLE:
	case ET_PUSH_TRIGGER:
	case ET_TELEPORT_TRIGGER:
		break;
	case ET_GENERAL:
		CG_General( cent );
		break;
	case ET_CORPSE:
		if(cent->CorpseIndex <= 0 || cent->CorpseIndex > corpses )
		{ // just died, so create a new faux client
			cent->CorpseIndex = CG_ClientInfoForCorpse(cent->currentState.modelindex);
		}
		CG_Corpse( cent );
		break;
	case ET_PLAYER:
		if(cent->currentState.number == cg.snap->ps.clientNum) 
			cgs.clientinfo[cg.snap->ps.clientNum].CurrentEntity = cent; // we'll draw ourselves first next time
		else
			CG_Player( cent );
		break;
	case ET_ITEM:
		CG_Item( cent );
		break;
	case ET_MISSILE:
		CG_Missile( cent );
		break;
	case ET_MOVER:
		CG_Mover( cent );
		break;
	case ET_BEAM:
		CG_Beam( cent );
		break;
	case ET_PORTAL:
		CG_Portal( cent );
		break;
	case ET_SPEAKER:
		CG_Speaker( cent );
		break;
	}
}

/*
===============
CG_AddPacketEntities

===============
*/
void CG_AddPacketEntities( void ) {
	int					num;
	centity_t			*cent;
	playerState_t		*ps;

	// set cg.frameInterpolation
	if ( cg.nextSnap ) {
		int		delta;

		delta = (cg.nextSnap->serverTime - cg.snap->serverTime);
		if ( delta == 0 ) {
			cg.frameInterpolation = 0;
		} else {
			cg.frameInterpolation = (float)( cg.time - cg.snap->serverTime ) / delta;
		}
	} else {
		cg.frameInterpolation = 0;	// actually, it should never be used, because 
									// no entities should be marked as interpolating
	}

	// the auto-rotating items will all have the same axis
	cg.autoAngles[0] = 0;
	cg.autoAngles[1] = ( cg.time & 2047 ) * 360 / 2048.0;
	cg.autoAngles[2] = 0;

	cg.autoAnglesFast[0] = 0;
	cg.autoAnglesFast[1] = ( cg.time & 1023 ) * 360 / 1024.0f;
	cg.autoAnglesFast[2] = 0;

	AnglesToAxis( cg.autoAngles, cg.autoAxis );
	AnglesToAxis( cg.autoAnglesFast, cg.autoAxisFast );

	// generate and add the entity from the playerstate
	ps = &cg.predictedPlayerState;
	BG_PlayerStateToEntityState( ps, &cg.predictedPlayerEntity.currentState, cg.time, qfalse );
	CG_AddCEntity( &cg.predictedPlayerEntity );

	// lerp the non-predicted value for lightning gun origins
	CG_CalcEntityLerpPositions( &cg_entities[ cg.snap->ps.clientNum ] );

	// add each entity sent over by the server
	for ( num = 0 ; num < cg.snap->numEntities ; num++ ) {
		cent = &cg_entities[ cg.snap->entities[ num ].number ];
		CG_AddCEntity( cent );
	}
}

