// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_weapons.c -- events and effects dealing with weapons
#include "cg_local.h"

/*
==========================
CG_MachineGunEjectBrass
==========================
*/
static void CG_MachineGunEjectBrass( centity_t *cent ) {
	localEntity_t	*le;
	refEntity_t		*re;
	vec3_t			velocity, xvelocity;
	vec3_t			offset, xoffset;
	float			waterScale = 1.0f;
	vec3_t			v[3];

	if ( !cg_brassTime.integer ) {
		return;
	}

	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	velocity[0] = 25 * crandom() + cent->currentState.pos.trDelta[0];
	velocity[1] = -50 + 40 * crandom() + cent->currentState.pos.trDelta[1];
	velocity[2] = 100 + 50 * crandom() + cent->currentState.pos.trDelta[2];

	le->leType = LE_FRAGMENT;
	le->startTime = cg.time;
	le->endTime = le->startTime + cg_brassTime.integer + ( cg_brassTime.integer / 4 ) * random();
	if(cg_brassTime.integer == -1 || g_ForceViz.integer == 2)
		le->endTime = cg.time + 100000000;
	le->pos.trType = TR_GRAVITY;
	le->pos.trTime = cg.time - (rand()&15);

	AnglesToAxis( cent->lerpAngles, v );

	
	offset[0] = 8;
	offset[1] = -4;
	offset[2] = 24;

	xoffset[0] = offset[0] * v[0][0] + offset[1] * v[1][0] + offset[2] * v[2][0];
	xoffset[1] = offset[0] * v[0][1] + offset[1] * v[1][1] + offset[2] * v[2][1];
	xoffset[2] = offset[0] * v[0][2] + offset[1] * v[1][2] + offset[2] * v[2][2];
	if(cent->pe.railgunOrigin[0] || cent->pe.railgunOrigin[1] || cent->pe.railgunOrigin[2]  )
	{
		VectorMA( cent->pe.railgunOrigin, -16, v[0], re->origin );		
	}
	else
	{
		VectorAdd( cent->lerpOrigin, xoffset, re->origin );
	}

	VectorCopy( re->origin, le->pos.trBase );

	if ( CG_PointContents( re->origin, -1 ) & CONTENTS_WATER ) {
		waterScale = 0.10;
	}

	xvelocity[0] = velocity[0] * v[0][0] + velocity[1] * v[1][0] + velocity[2] * v[2][0];
	xvelocity[1] = velocity[0] * v[0][1] + velocity[1] * v[1][1] + velocity[2] * v[2][1];
	xvelocity[2] = velocity[0] * v[0][2] + velocity[1] * v[1][2] + velocity[2] * v[2][2];
	VectorScale( xvelocity, waterScale, le->pos.trDelta );

	AxisCopy( axisDefault, re->axis );
	re->hModel = cgs.media.machinegunBrassModel;

	le->bounceFactor = 0.4 * waterScale;

	le->angles.trType = TR_LINEAR;
	le->angles.trTime = cg.time;
	le->angles.trBase[0] = rand()&31;
	le->angles.trBase[1] = rand()&31;
	le->angles.trBase[2] = rand()&31;
	le->angles.trDelta[0] = 2;
	le->angles.trDelta[1] = 1;
	le->angles.trDelta[2] = 0;

	le->leFlags = LEF_TUMBLE;
	le->leBounceSoundType = LEBS_BRASS;
	le->leMarkType = LEMT_NONE;
}

/*
==========================
CG_ShotgunEjectBrass
==========================
*/
static void CG_ShotgunEjectBrass( centity_t *cent ) {
	localEntity_t	*le;
	refEntity_t		*re;
	vec3_t			velocity, xvelocity;
	vec3_t			offset, xoffset;
	vec3_t			v[3];
	int				i;

	if ( !cg_brassTime.integer ) {
		return;
	}

	for ( i = 0; i < 2; i++ ) {
		float	waterScale = 1.0f;

		le = CG_AllocLocalEntity();
		re = &le->refEntity;

		velocity[0] = 60 + 60 * crandom() + cent->currentState.pos.trDelta[0];
		if ( i == 0 ) {
			velocity[1] = 40 + 10 * crandom() + cent->currentState.pos.trDelta[1];
		} else {
			velocity[1] = -40 + 10 * crandom() + cent->currentState.pos.trDelta[1];
		}
		velocity[2] = 100 + 50 * crandom() + cent->currentState.pos.trDelta[2];

		le->leType = LE_FRAGMENT;
		le->startTime = cg.time;
		le->endTime = le->startTime + cg_brassTime.integer*3 + cg_brassTime.integer * random();
		if(cg_brassTime.integer == -1 || g_ForceViz.integer == 2)
			le->endTime = cg.time + 100000000;
	
		le->pos.trType = TR_GRAVITY;
		le->pos.trTime = cg.time;

		AnglesToAxis( cent->lerpAngles, v );

		
		offset[0] = 8;
		offset[1] = 0;
		offset[2] = 24;

		xoffset[0] = offset[0] * v[0][0] + offset[1] * v[1][0] + offset[2] * v[2][0];
		xoffset[1] = offset[0] * v[0][1] + offset[1] * v[1][1] + offset[2] * v[2][1];
		xoffset[2] = offset[0] * v[0][2] + offset[1] * v[1][2] + offset[2] * v[2][2];
		if(cent->pe.railgunOrigin[0] || cent->pe.railgunOrigin[1] || cent->pe.railgunOrigin[2]  )
		{
			VectorMA( cent->pe.railgunOrigin, -16, v[0], re->origin );		
		}
		else
		{
	
			VectorAdd( cent->lerpOrigin, xoffset, re->origin );
		}
		
		VectorCopy( re->origin, le->pos.trBase );
		if ( CG_PointContents( re->origin, -1 ) & CONTENTS_WATER ) {
			waterScale = 0.10;
		}

		xvelocity[0] = velocity[0] * v[0][0] + velocity[1] * v[1][0] + velocity[2] * v[2][0];
		xvelocity[1] = velocity[0] * v[0][1] + velocity[1] * v[1][1] + velocity[2] * v[2][1];
		xvelocity[2] = velocity[0] * v[0][2] + velocity[1] * v[1][2] + velocity[2] * v[2][2];
		VectorScale( xvelocity, waterScale, le->pos.trDelta );

		AxisCopy( axisDefault, re->axis );
		re->hModel = cgs.media.shotgunBrassModel;
		le->bounceFactor = 0.3;

		le->angles.trType = TR_LINEAR;
		le->angles.trTime = cg.time;
		le->angles.trBase[0] = rand()&31;
		le->angles.trBase[1] = rand()&31;
		le->angles.trBase[2] = rand()&31;
		le->angles.trDelta[0] = 1;
		le->angles.trDelta[1] = 0.5;
		le->angles.trDelta[2] = 0;

		le->leFlags = LEF_TUMBLE;
		le->leBounceSoundType = LEBS_BRASS;
		le->leMarkType = LEMT_NONE;
	}
}

void CG_BFGBeam( int width, vec3_t start, vec3_t end ) {
	localEntity_t	*le;
	refEntity_t		*re;
	vec3_t n0, n1, n2;
	float dis;
	int i, r;
	
	VectorSubtract(end, start, n0);
	dis = VectorNormalize(n0);
		
	PerpendicularVector(n1, n0);
	CrossProduct(n0, n1, n2);

	le = CG_AllocLocalEntity();
	re = &le->refEntity;
	
	le->leType = LE_FADE_RGB;
	le->startTime = cg.time;
	le->endTime = cg.time + 300;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	
	re->shaderTime = cg.time / 1000.0f;
	VectorCopy( start, re->origin );
	
	re->reType = RT_MODEL;
	re->renderfx |= RF_NOSHADOW;
	re->hModel = cgs.media.teleportEffectModel;
	//re->hModel = cg_weapons[WP_RAILGUN].missileModel;
	re->customShader = cgs.media.bfgExplosionShader;
		
	for(i = 0; i < 3; i++)
	{
		re->axis[0][i] = n2[i] * width / 62.125;
		re->axis[1][i] = n1[i] * width / 62.125;
		re->axis[2][i] = n0[i] * dis / 62.125;
		
	}
	
	le->color[0] = 0;//ci->color[0] * 0.75;
	le->color[1] = 1;//ci->color[1] * 0.75;
	le->color[2] = 0;//ci->color[2] * 0.75;
	le->color[3] = 1;//0.15f;
	le->refEntity.shaderRGBA[3] = 0.15f;

	
}

/*
==========================
CG_RailTrail
==========================
*/
void CG_RailTrail( clientInfo_t *ci, vec3_t start, vec3_t end ) {
	localEntity_t	*le;
	refEntity_t		*re;

	if(cg_CoolRailGun.integer || g_ForceViz.integer)
	{
		vec3_t n0, n1, n2;
		float dis, rlength, rtime;
		int i, r;
	
		VectorSubtract(end, start, n0);
		dis = VectorNormalize(n0);
		rlength = cg_railLength.value;
		rtime = cg_railTrailTime.value;
		if(g_ForceViz.integer)
		{
			rlength = 32;
			cg_railTrailTime.value = rtime = 1600;
		}
		else if(rlength == 0)
		{
			rlength = 32;
			CG_Printf("Error: zero r_railSegmentLength\n");
		}
		
		PerpendicularVector(n1, n0);
		CrossProduct(n0, n1, n2);
		
		for(r = 0; r < dis / rlength; r++)
		{
			le = CG_AllocLocalEntity();
			re = &le->refEntity;
	
			le->leType = LE_EXPAND_FADE;
			le->startTime = cg.time;
			le->endTime = cg.time + rtime;
			le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	
			re->shaderTime = cg.time / 1000.0f;
			VectorMA( start, r * rlength, n0, re->origin );
	
			re->reType = RT_MODEL;
			re->renderfx |= RF_NOSHADOW;
			re->hModel = cgs.media.teleportEffectModel;
			//re->hModel = cg_weapons[WP_RAILGUN].missileModel;
			re->customShader = cgs.media.railEffectShader;
			
			//vectoangles(dir, a);
			//AnglesToAxis(a, re->axis);
			for(i = 0; i < 3; i++)
			{
				re->axis[0][i] = n2[i];
				re->axis[1][i] = n1[i];
				re->axis[2][i] = n0[i] * rlength / 62.125;
				if(r == (int) dis / rlength)
					re->axis[2][i] *= (((int) dis) % ((int) rlength)) / rlength;
				
			}
			//VectorCopy( end, re->oldorigin );
	
			le->color[0] = ci->color[0] * 0.75;
			le->color[1] = ci->color[1] * 0.75;
			le->color[2] = ci->color[2] * 0.75;
			le->color[3] = 1;//0.15f;
			le->refEntity.shaderRGBA[3] = 1;//0.15f;
		}
		if(ci->powerups & (1 << PW_QUAD) && (cg_coolPowerups.integer || g_ForceViz.integer))
		{ // add quad ring
			le = CG_AllocLocalEntity();
			re = &le->refEntity;
	
			le->leType = LE_EXPAND_FADE;
			le->startTime = cg.time;
			le->endTime = cg.time + rtime;
			le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	
			re->shaderTime = cg.time / 1000.0f;
			VectorCopy( start, re->origin );
	
			re->reType = RT_MODEL;
			re->renderfx |= RF_NOSHADOW;
			re->hModel = cgs.media.teleportEffectModel;
			//re->hModel = cg_weapons[WP_RAILGUN].missileModel;
			re->customShader = cgs.media.quadShader;
			
			//vectoangles(dir, a);
			//AnglesToAxis(a, re->axis);
			for(i = 0; i < 3; i++)
			{
				re->axis[0][i] = n2[i];
				re->axis[1][i] = n1[i];
				re->axis[2][i] = n0[i] * dis / 62.125;
				
			}
	
			le->color[0] = 1;//ci->color[0] * 0.75;
			le->color[1] = 1;//ci->color[1] * 0.75;
			le->color[2] = 1;//ci->color[2] * 0.75;
			le->color[3] = 1;//0.15f;
			le->refEntity.shaderRGBA[3] = 0.15f;
		}
	}
	else
	{
		//
		// rings
		//
		le = CG_AllocLocalEntity();
		re = &le->refEntity;
	
		le->leType = LE_FADE_RGB;
		le->startTime = cg.time;
		le->endTime = cg.time + cg_railTrailTime.value;
		le->lifeRate = 1.0 / ( le->endTime - le->startTime );
	
		re->shaderTime = cg.time / 1000.0f;
		re->reType = RT_RAIL_RINGS;
		re->customShader = cgs.media.railRingsShader;
		
		VectorCopy( start, re->origin );
		VectorCopy( end, re->oldorigin );
	
		
		// nudge down a bit so it isn't exactly in center
		//re->origin[2] -= 8;
		//re->oldorigin[2] -= 8;
		
		le->color[0] = ci->color[0] * 0.75;
		le->color[1] = ci->color[1] * 0.75;
		le->color[2] = ci->color[2] * 0.75;
		le->color[3] = 1.0f;
	
		AxisClear( re->axis );
	}
	
	// nudge down a bit so it isn't exactly in center
	//re->origin[2] -= 8;
	//re->oldorigin[2] -= 8;
	//
	// core
	//
	le = CG_AllocLocalEntity();
	re = &le->refEntity;

	le->leType = LE_FADE_RGB;
	le->startTime = cg.time;
	le->endTime = cg.time + cg_railTrailTime.value;
	le->lifeRate = 1.0 / ( le->endTime - le->startTime );

	re->shaderTime = cg.time / 1000.0f;
	VectorCopy( start, re->origin );
		

	re->reType = RT_RAIL_CORE;
	re->customShader = cgs.media.railCoreShader;	
	VectorCopy( end, re->oldorigin );
	AxisClear( re->axis );

	le->color[0] = ci->color[0] * 0.75;
	le->color[1] = ci->color[1] * 0.75;
	le->color[2] = ci->color[2] * 0.75;
	le->color[3] = 1.0f;

	
}

/*
==========================
CG_RocketTrail
==========================
*/
static void CG_RocketTrail( centity_t *ent, const weaponInfo_t *wi ) {
	int		step;
	vec3_t	origin, lastPos;
	int		t;
	int		startTime, contents;
	int		lastContents;
	entityState_t	*es;
	vec3_t	up;
	localEntity_t	*smoke;

	up[0] = 0;
	up[1] = 0;
	up[2] = 0;

	step = 50;

	es = &ent->currentState;
	startTime = ent->trailTime;
	t = step * ( (startTime + step) / step );

	BG_EvaluateTrajectory( &es->pos, cg.time, origin );
	contents = CG_PointContents( origin, -1 );

	// if object (e.g. grenade) is stationary, don't toss up smoke
	if ( es->pos.trType == TR_STATIONARY ) {
		ent->trailTime = cg.time;
		return;
	}

	BG_EvaluateTrajectory( &es->pos, ent->trailTime, lastPos );
	lastContents = CG_PointContents( lastPos, -1 );

	ent->trailTime = cg.time;

	if ( contents & ( CONTENTS_WATER | CONTENTS_SLIME | CONTENTS_LAVA ) ) {
		if ( contents & lastContents & CONTENTS_WATER ) {
			CG_BubbleTrail( lastPos, origin, 8 );
		}
		return;
	}

	for ( ; t <= ent->trailTime ; t += step ) {
		BG_EvaluateTrajectory( &es->pos, t, lastPos );
		
		if(ent->currentState.powerups & (1 << PW_QUAD) && (cg_coolPowerups.integer || g_ForceViz.integer))
		{
			smoke = CG_SmokePuff( lastPos, up, 
					wi->trailRadius, 
					1, 1, 1, 0.33,
					wi->wiTrailTime, 
					t,
					0, 
					cgs.media.QuadSmokePuffShader );

			smoke->refEntity.renderfx |= RF_MINLIGHT;
					
		}
		else
		{
			smoke = CG_SmokePuff( lastPos, up, 
					wi->trailRadius, 
					1, 1, 1, 0.33,
					wi->wiTrailTime, 
					t,
					0, 
					cgs.media.smokePuffShader );

			
		}
		// use the optimized local entity add
		smoke->leType = LE_SCALE_FADE;
	}

}

/*
==========================
CG_GrappleTrail
==========================
*/
void CG_GrappleTrail( centity_t *ent, const weaponInfo_t *wi ) {
	vec3_t	origin;
	entityState_t	*es;
	vec3_t			forward, up;
	refEntity_t		beam;

	es = &ent->currentState;

	BG_EvaluateTrajectory( &es->pos, cg.time, origin );
	ent->trailTime = cg.time;

	memset( &beam, 0, sizeof( beam ) );
	//FIXME adjust for muzzle position
	VectorCopy ( cg_entities[ ent->currentState.otherEntityNum ].lerpOrigin, beam.origin );
	beam.origin[2] += 26;
	AngleVectors( cg_entities[ ent->currentState.otherEntityNum ].lerpAngles, forward, NULL, up );
	VectorMA( beam.origin, -6, up, beam.origin );
	VectorCopy( origin, beam.oldorigin );

	if (Distance( beam.origin, beam.oldorigin ) < 64 )
		return; // Don't draw if close

	beam.reType = RT_LIGHTNING;
	beam.customShader = cgs.media.lightningShader;

	AxisClear( beam.axis );
	beam.shaderRGBA[0] = 0xff;
	beam.shaderRGBA[1] = 0xff;
	beam.shaderRGBA[2] = 0xff;
	beam.shaderRGBA[3] = 0xff;
	trap_R_AddRefEntityToScene( &beam );
}

/*
==========================
CG_GrenadeTrail
==========================
*/
static void CG_GrenadeTrail( centity_t *ent, const weaponInfo_t *wi ) {
	CG_RocketTrail( ent, wi );
}


/*
=================
CG_RegisterWeapon

The server says this item is used on this level
=================
*/
void CG_RegisterWeapon( int weaponNum ) {
	weaponInfo_t	*weaponInfo;
	gitem_t			*item, *ammo;
	char			path[MAX_QPATH];
	vec3_t			mins, maxs;
	int				i;

	weaponInfo = &cg_weapons[weaponNum];

	if ( weaponNum <= 0 ) {
		return;
	}

	if ( weaponInfo->registered ) {
		return;
	}

	memset( weaponInfo, 0, sizeof( *weaponInfo ) );
	weaponInfo->registered = qtrue;

	for ( item = bg_itemlist + 1 ; item->classname ; item++ ) {
		if ( item->giType == IT_WEAPON && item->giTag == weaponNum ) {
			weaponInfo->item = item;
			break;
		}
	}
	if ( !item->classname ) {
		CG_Error( "Couldn't find weapon %i", weaponNum );
	}
	CG_RegisterItemVisuals( item - bg_itemlist );

	// load cmodel before model so filecache works
	weaponInfo->weaponModel = trap_R_RegisterModel( item->world_model[0] );
	weaponInfo->weaponSkin = 0;
	// calc midpoint for rotation
	trap_R_ModelBounds( weaponInfo->weaponModel, mins, maxs );
	for ( i = 0 ; i < 3 ; i++ ) {
		weaponInfo->weaponMidpoint[i] = mins[i] + 0.5 * ( maxs[i] - mins[i] );
	}

	weaponInfo->weaponIcon = trap_R_RegisterShader( item->icon );
	weaponInfo->ammoIcon = trap_R_RegisterShader( item->icon );

	for ( ammo = bg_itemlist + 1 ; ammo->classname ; ammo++ ) {
		if ( ammo->giType == IT_AMMO && ammo->giTag == weaponNum ) {
			break;
		}
	}
	if ( ammo->classname && ammo->world_model[0] ) {
		weaponInfo->ammoModel = trap_R_RegisterModel( ammo->world_model[0] );
	}

	strcpy( path, item->world_model[0] );
	COM_StripExtension( path, path );
	strcat( path, "_flash.md3" );
	weaponInfo->flashModel = trap_R_RegisterModel( path );

	if ( weaponNum == WP_MACHINEGUN || weaponNum == WP_GAUNTLET || weaponNum == WP_BFG ) {
		strcpy( path, item->world_model[0] );
		COM_StripExtension( path, path );
		strcat( path, "_barrel.md3" );
		weaponInfo->barrelModel = trap_R_RegisterModel( path );
	}
	else if(weaponNum == WP_BALL)
	{
		weaponInfo->barrelModel = trap_R_RegisterModel( "models/powerups/health/small_sphere.md3" );
	}

	strcpy( path, item->world_model[0] );
	COM_StripExtension( path, path );
	strcat( path, "_hand.md3" );
	weaponInfo->handsModel = trap_R_RegisterModel( path );

	if ( !weaponInfo->handsModel ) {
		weaponInfo->handsModel = trap_R_RegisterModel( "models/weapons2/shotgun/shotgun_hand.md3" );
	}

	switch ( weaponNum ) {
	case WP_GAUNTLET:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/melee/fstrun.wav" );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/melee/fstatck.wav" );
		break;

	case WP_FLAMETHROWER:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->barrelModel = trap_R_RegisterModel( "models/ammo/grenade1.md3" );
		weaponInfo->flashModel = trap_R_RegisterModel( "models/weapons2/grenadel/grenadel_flash.md3" );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/rocket/rockfly.wav" );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/items/s_health.wav" );

		cgs.media.flamethrowerShader = trap_R_RegisterShader( "rocketExplosion" );
		
		cgs.media.flamethrowerExplosionModel = trap_R_RegisterModel( "models/weaphits/crackle.md3" );

		break;

	
	case WP_LIGHTNING:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->readySound = trap_S_RegisterSound( "sound/weapons/melee/fsthum.wav" );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/lightning/lg_hum.wav" );
		
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/lightning/lg_fire.wav" );
		cgs.media.lightningShader = trap_R_RegisterShader( "lightningBolt" );
		cgs.media.lightningExplosionModel = trap_R_RegisterModel( "models/weaphits/crackle.md3" );
		cgs.media.sfx_lghit1 = trap_S_RegisterSound( "sound/weapons/lightning/lg_hit.wav" );
		cgs.media.sfx_lghit2 = trap_S_RegisterSound( "sound/weapons/lightning/lg_hit2.wav" );
		cgs.media.sfx_lghit3 = trap_S_RegisterSound( "sound/weapons/lightning/lg_hit3.wav" );

		break;

	case WP_GRAPPLING_HOOK:
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->missileModel = trap_R_RegisterModel( "models/ammo/rocket/rocket.md3" );
		weaponInfo->missileTrailFunc = CG_GrappleTrail;
		weaponInfo->missileDlight = 200;
		weaponInfo->wiTrailTime = 2000;
		weaponInfo->trailRadius = 64;
		MAKERGB( weaponInfo->missileDlightColor, 1, 1, 0.5 );
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 0.5 );
		weaponInfo->readySound = trap_S_RegisterSound( "sound/weapons/melee/fsthum.wav" );
		weaponInfo->firingSound = trap_S_RegisterSound( "sound/weapons/melee/fstrun.wav" );
		break;

	case WP_MACHINEGUN:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf1b.wav" );
		weaponInfo->flashSound[1] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf2b.wav" );
		weaponInfo->flashSound[2] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf3b.wav" );
		weaponInfo->flashSound[3] = trap_S_RegisterSound( "sound/weapons/machinegun/machgf4b.wav" );
		weaponInfo->ejectBrassFunc = CG_MachineGunEjectBrass;
		cgs.media.bulletExplosionShader = trap_R_RegisterShader( "bulletExplosion" );
		break;

	case WP_SHOTGUN:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/shotgun/sshotf1b.wav" );
		weaponInfo->ejectBrassFunc = CG_ShotgunEjectBrass;
		break;

	case WP_ROCKET_LAUNCHER:
		weaponInfo->missileModel = trap_R_RegisterModel( "models/ammo/rocket/rocket.md3" );
		weaponInfo->missileSound = trap_S_RegisterSound( "sound/weapons/rocket/rockfly.wav" );
		weaponInfo->missileTrailFunc = CG_RocketTrail;
		weaponInfo->missileDlight = 200;
		weaponInfo->wiTrailTime = 2000;
		weaponInfo->trailRadius = 64;
		MAKERGB( weaponInfo->missileDlightColor, 1, 0.75, 0 );
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.75, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/rocket/rocklf1a.wav" );
		cgs.media.rocketExplosionShader = trap_R_RegisterShader( "rocketExplosion" );
		break;

	case WP_GRENADE_LAUNCHER:
		weaponInfo->missileModel = trap_R_RegisterModel( "models/ammo/grenade1.md3" );
		weaponInfo->missileTrailFunc = CG_GrenadeTrail;
		weaponInfo->wiTrailTime = 700;
		weaponInfo->trailRadius = 32;
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.7, 0.5 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/grenade/grenlf1a.wav" );
		cgs.media.grenadeExplosionShader = trap_R_RegisterShader( "grenadeExplosion" );
		break;

	case WP_PLASMAGUN:
		weaponInfo->missileSound = trap_S_RegisterSound( "sound/weapons/plasma/lasfly.wav" );
		MAKERGB( weaponInfo->flashDlightColor, 0.6, 0.6, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/plasma/hyprbf1a.wav" );
		cgs.media.plasmaExplosionShader = trap_R_RegisterShader( "plasmaExplosion" );
		weaponInfo->weaponSkin = trap_R_RegisterSkin("models/weapons2/plasma/plasma.skin");
		break;

	case WP_RAILGUN:
		weaponInfo->readySound = trap_S_RegisterSound( "sound/weapons/railgun/rg_hum.wav" );
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.5, 0 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/railgun/railgf1a.wav" );
		cgs.media.railExplosionShader = trap_R_RegisterShader( "railExplosion" );
		cgs.media.railRingsShader = trap_R_RegisterShader( "railDisc" );		
		cgs.media.railEffectShader = trap_R_RegisterShader( "railRing" );

		//weaponInfo->missileModel = trap_R_RegisterModel( "models/weaphits/bfg.md3" );
		break;

	case WP_BFG:
		weaponInfo->readySound = trap_S_RegisterSound( "sound/weapons/bfg/bfg_hum.wav" );
		MAKERGB( weaponInfo->flashDlightColor, 1, 0.7, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/bfg/bfg_fire.wav" );
		cgs.media.bfgExplosionShader = trap_R_RegisterShader( "bfgExplosion" );
		cgs.media.railCoreShader = trap_R_RegisterShader( "railCore" );
		weaponInfo->missileModel = trap_R_RegisterModel( "models/weaphits/bfg.md3" );
		weaponInfo->missileSound = trap_S_RegisterSound( "sound/weapons/rocket/rockfly.wav" );
		weaponInfo->weaponSkin = trap_R_RegisterSkin("models/weapons2/bfg/bfg.skin");
		break;
	
	case WP_BALL:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/change.wav" );
		break;
		
	default:
		MAKERGB( weaponInfo->flashDlightColor, 1, 1, 1 );
		weaponInfo->flashSound[0] = trap_S_RegisterSound( "sound/weapons/rocket/rocklf1a.wav" );
		break;
	}
}

/*
=================
CG_RegisterItemVisuals

The server says this item is used on this level
=================
*/
void CG_RegisterItemVisuals( int itemNum ) {
	itemInfo_t		*itemInfo;
	gitem_t			*item;

	itemInfo = &cg_items[ itemNum ];
	if ( itemInfo->registered ) {
		return;
	}

	item = &bg_itemlist[ itemNum ];

	memset( itemInfo, 0, sizeof( &itemInfo ) );
	itemInfo->registered = qtrue;

	itemInfo->models[0] = trap_R_RegisterModel( item->world_model[0] );

	itemInfo->icon = trap_R_RegisterShader( item->icon );

	if ( item->giType == IT_WEAPON ) {
		CG_RegisterWeapon( item->giTag );
	}

	//
	// powerups have an accompanying ring or sphere
	//
	if ( item->giType == IT_POWERUP || item->giType == IT_HEALTH || 
		item->giType == IT_ARMOR || item->giType == IT_HOLDABLE ) {
		if ( item->world_model[1] ) {
			itemInfo->models[1] = trap_R_RegisterModel( item->world_model[1] );
		}
	}
}


/*
========================================================================================

VIEW WEAPON

========================================================================================
*/

/*
=================
CG_MapTorsoToWeaponFrame

=================
*/
static int CG_MapTorsoToWeaponFrame( clientInfo_t *ci, int frame ) {

	// change weapon
	if ( frame >= ci->animations[TORSO_DROP].firstFrame 
		&& frame < ci->animations[TORSO_DROP].firstFrame + 9 ) {
		return frame - ci->animations[TORSO_DROP].firstFrame + 6;
	}

	// stand attack
	if ( frame >= ci->animations[TORSO_ATTACK].firstFrame 
		&& frame < ci->animations[TORSO_ATTACK].firstFrame + 6 ) {
		return 1 + frame - ci->animations[TORSO_ATTACK].firstFrame;
	}

	// stand attack 2
	if ( frame >= ci->animations[TORSO_ATTACK2].firstFrame 
		&& frame < ci->animations[TORSO_ATTACK2].firstFrame + 6 ) {
		return 1 + frame - ci->animations[TORSO_ATTACK].firstFrame;
	}
	
	return 0;
}


/*
==============
CG_CalculateWeaponPosition
==============
*/
static void CG_CalculateWeaponPosition( vec3_t origin, vec3_t angles ) {
	float	scale;
	int		delta;
	float	fracsin;

	VectorCopy( cg.refdef.vieworg, origin );
	VectorCopy( cg.refdefViewAngles, angles );

	// on odd legs, invert some angles
	if ( cg.bobcycle & 1 ) {
		scale = -cg.xyspeed;
	} else {
		scale = cg.xyspeed;
	}

	// gun angles from bobbing
	angles[ROLL] += scale * cg.bobfracsin * 0.005;
	angles[YAW] += scale * cg.bobfracsin * 0.01;
	angles[PITCH] += cg.xyspeed * cg.bobfracsin * 0.005;

	// drop the weapon when landing
	delta = cg.time - cg.landTime;
	if ( delta < LAND_DEFLECT_TIME ) {
		origin[2] += cg.landChange*0.25 * delta / LAND_DEFLECT_TIME;
	} else if ( delta < LAND_DEFLECT_TIME + LAND_RETURN_TIME ) {
		origin[2] += cg.landChange*0.25 * 
			(LAND_DEFLECT_TIME + LAND_RETURN_TIME - delta) / LAND_RETURN_TIME;
	}

#if 0
	// drop the weapon when stair climbing
	delta = cg.time - cg.stepTime;
	if ( delta < STEP_TIME/2 ) {
		origin[2] -= cg.stepChange*0.25 * delta / (STEP_TIME/2);
	} else if ( delta < STEP_TIME ) {
		origin[2] -= cg.stepChange*0.25 * (STEP_TIME - delta) / (STEP_TIME/2);
	}
#endif

	// idle drift
	scale = cg.xyspeed + 40;
	fracsin = sin( cg.time * 0.001 );
	angles[ROLL] += scale * fracsin * 0.01;
	angles[YAW] += scale * fracsin * 0.01;
	angles[PITCH] += scale * fracsin * 0.01;
}


/*
===============
CG_LightningBolt

Origin will be the exact tag point, which is slightly
different than the muzzle point used for determining hits.
The cent should be the non-predicted cent if it is from the player,
so the endpoint will reflect the simulated strike (lagging the predicted
angle)
===============
*/
static void CG_LightningBolt( centity_t *cent, vec3_t origin ) {
	trace_t		trace;
	refEntity_t		beam;
	vec3_t			forward;
	vec3_t			muzzlePoint, endPoint;

	if ( cent->currentState.weapon != WP_LIGHTNING ) {
		return;
	}

	memset( &beam, 0, sizeof( beam ) );

	// find muzzle point for this frame
	VectorCopy( cent->lerpOrigin, muzzlePoint );
	AngleVectors( cent->lerpAngles, forward, NULL, NULL );
	// FIXME: crouch
	muzzlePoint[2] += DEFAULT_VIEWHEIGHT;

	VectorMA( muzzlePoint, 14, forward, muzzlePoint );

	// project forward by the lightning range
	VectorMA( muzzlePoint, LIGHTNING_RANGE, forward, endPoint );

	// see if it hit a wall
	CG_Trace( &trace, muzzlePoint, vec3_origin, vec3_origin, endPoint, 
		cent->currentState.number, MASK_SHOT );

	// this is the endpoint
	VectorCopy( trace.endpos, beam.oldorigin );

	// use the provided origin, even though it may be slightly
	// different than the muzzle origin
	VectorCopy( origin, beam.origin );

	beam.reType = RT_LIGHTNING;
	beam.customShader = cgs.media.lightningShader;
	trap_R_AddRefEntityToScene( &beam );

	// add the impact flare if it hit something
	if ( trace.fraction < 1.0 ) {
		vec3_t	angles;
		vec3_t	dir;

		VectorSubtract( beam.oldorigin, beam.origin, dir );
		VectorNormalize( dir );

		memset( &beam, 0, sizeof( beam ) );
		beam.hModel = cgs.media.lightningExplosionModel;

		VectorMA( trace.endpos, -16, dir, beam.origin );

		// make a random orientation
		angles[0] = rand() % 360;
		angles[1] = rand() % 360;
		angles[2] = rand() % 360;
		AnglesToAxis( angles, beam.axis );
		trap_R_AddRefEntityToScene( &beam );
	}
}

/*
===============
CG_LightningBolt

Origin will be the exact tag point, which is slightly
different than the muzzle point used for determining hits.
The cent should be the non-predicted cent if it is from the player,
so the endpoint will reflect the simulated strike (lagging the predicted
angle)
===============
*/
static void CG_FlameBolt( centity_t *cent, vec3_t origin ) {
	trace_t		trace;
	refEntity_t		beam;
	vec3_t			forward, mins, maxs;
	vec3_t			muzzlePoint, endPoint;

	if ( cent->currentState.weapon != WP_FLAMETHROWER ) {
		return;
	}

	memset( &beam, 0, sizeof( beam ) );

	// find muzzle point for this frame
	VectorCopy( cent->lerpOrigin, muzzlePoint );
	AngleVectors( cent->lerpAngles, forward, NULL, NULL );
	// FIXME: crouch
	muzzlePoint[2] += DEFAULT_VIEWHEIGHT;

	VectorMA( muzzlePoint, 14, forward, muzzlePoint );

	// project forward by the lightning range
	VectorMA( muzzlePoint, FLAMETHROWER_RANGE, forward, endPoint );

	// see if it hit a wall
	mins[0] = -8;
	mins[1] = -8;
	mins[2] = -8;
	maxs[0] = 8;
	maxs[1] = 8;
	maxs[2] = 8;
	CG_Trace( &trace, muzzlePoint, mins, maxs, endPoint, 
		cent->currentState.number, MASK_SHOT | CONTENTS_WATER);

	// this is the endpoint
	VectorCopy( trace.endpos, beam.oldorigin );

	// use the provided origin, even though it may be slightly
	// different than the muzzle origin
	VectorCopy( origin, beam.origin );

	beam.reType = RT_LIGHTNING;
	beam.customShader = cgs.media.flamethrowerShader;
	
	trap_R_AddRefEntityToScene( &beam );

	{
		vec3_t	angles;
		vec3_t	org, dir;
		localEntity_t *le;
		VectorSubtract( beam.oldorigin, beam.origin, dir );
		VectorNormalize( dir );

		memset( &beam, 0, sizeof( beam ) );
		beam.hModel = cgs.media.flamethrowerExplosionModel;
		beam.customShader = cgs.media.flamethrowerShader;
		VectorMA( trace.endpos, -16, dir, beam.origin );

		// make a random orientation
		angles[0] = rand() % 360;
		angles[1] = rand() % 360;
		angles[2] = rand() % 360;
		AnglesToAxis( angles, beam.axis );
		trap_R_AddRefEntityToScene( &beam );
		org[0] = trace.endpos[0] + crandom() * 8;
		org[1] = trace.endpos[1] + crandom() * 8;
		org[2] = trace.endpos[2] + crandom() * 8;

		dir[0] = 0;//forward[0] * 128;
		dir[1] = 0;//forward[1] * 128;
		dir[2] = 0;//forward[2] * 128 + 40;
		if(!(trace.contents & CONTENTS_WATER))
		{
			le = CG_SmokePuff(org, dir, 32, 1, 1, 1, 1, 900, cg.time, 0, cgs.media.flamethrowerShader);
			le->leFlags |= LEF_PUFF_DONT_SCALE;
		}
		//CG_MakeExplosion( trace.endpos, forward, 
		//					   cgs.media.dishFlashModel, cgs.media.flamethrowerShader,
		//					   900, qtrue );		
		
	}
}

/*
===============
CG_SpawnRailTrail

Origin will be the exact tag point, which is slightly
different than the muzzle point used for determining hits.
===============
*/
static void CG_SpawnRailTrail( centity_t *cent, vec3_t origin ) {
	clientInfo_t	*ci;

	if ( cent->currentState.weapon != WP_RAILGUN ) {
		return;
	}
	if ( !cent->pe.railgunFlash ) {
		return;
	}
	cent->pe.railgunFlash = qtrue;
	ci = &cgs.clientinfo[ cent->currentState.clientNum ];
	ci->powerups = cent->currentState.powerups;
	CG_RailTrail( ci, origin, cent->pe.railgunImpact );
}


/*
======================
CG_MachinegunSpinAngle
======================
*/
#define		SPIN_SPEED	0.9
#define		COAST_TIME	1500
static float	CG_MachinegunSpinAngle( centity_t *cent ) {
	int		delta;
	float	angle;
	float	speed;

	delta = cg.time - cent->pe.barrelTime;
	if ( cent->pe.barrelSpinning ) 
	{
		angle = cent->pe.barrelAngle + delta * SPIN_SPEED;
		if(cent->currentState.weapon == WP_MACHINEGUN && !(cent->currentState.powerups & (1 << PW_QUAD)))
			trap_S_StartSound (cent->pe.railgunOrigin, cent->currentState.number, CHAN_ITEM, cgs.media.machinegunSpinSound );
	}
	else 
	{
		if ( delta > COAST_TIME ) {
			delta = COAST_TIME;
		}

		speed = 0.5 * ( SPIN_SPEED + (float)( COAST_TIME - delta ) / COAST_TIME );
		angle = cent->pe.barrelAngle + delta * speed;
	}

	if ( cent->pe.barrelSpinning == !((cent->currentState.eFlags & EF_FIRING) || (cent->currentState.torsoAnim & ~ANIM_TOGGLEBIT) == TORSO_ATTACK) )//!(cent->currentState.eFlags & EF_FIRING) 
	{
		cent->pe.barrelTime = cg.time;
		cent->pe.barrelAngle = AngleMod( angle );
		cent->pe.barrelSpinning = (cent->currentState.eFlags & EF_FIRING) || ((cent->currentState.torsoAnim & ~ANIM_TOGGLEBIT) == TORSO_ATTACK);//!!(cent->currentState.eFlags & EF_FIRING);		
	}

	return angle;
}


/*
========================
CG_AddWeaponWithPowerups
========================
*/
static void CG_AddWeaponWithPowerups( refEntity_t *gun, int powerups ) {
	// add powerup effects
	if ( powerups & ( 1 << PW_INVIS ) )
	{
		
		if(gun->shaderRGBA[3] > 0)
			gun->customShader = cgs.media.invisShader2;
		else
			gun->customShader = cgs.media.invisShader;
		trap_R_AddRefEntityToScene( gun );
	} else {
		if(powerups & (1 << PW_HASTE) && !(gun->renderfx & RF_THIRD_PERSON) && (cg_coolPowerups.integer || g_ForceViz.integer == 2))
		{
			localEntity_t *le;
			refEntity_t *re;
		
			le = CG_AllocLocalEntity();
			re = &le->refEntity;
	
			le->leType = LE_FADE_RGB;
			le->startTime = cg.time;
			le->endTime = cg.time + 1000;
			le->lifeRate = 1.0 / ( le->endTime - le->startTime );
			memcpy(re, gun, sizeof(refEntity_t));
			re->renderfx = 0;
			re->customSkin = 0;
			re->customShader = cgs.media.invisShader2;
			le->color[0] = 1;
			le->color[1] = 0;
			le->color[2] = 0;
			le->color[3] = 1;

			
		}

		trap_R_AddRefEntityToScene( gun );

		if ( powerups & ( 1 << PW_BATTLESUIT ) ) {
			gun->customShader = cgs.media.battleWeaponShader;
			trap_R_AddRefEntityToScene( gun );
		}
		if ( powerups & ( 1 << PW_QUAD ) ) {
			gun->customShader = cgs.media.quadWeaponShader;
			trap_R_AddRefEntityToScene( gun );
		}
	}
}


/*
=============
CG_AddPlayerWeapon

Used for both the view weapon (ps is valid) and the world modelother character models (ps is NULL)
The main player will have this called for BOTH cases, so effects like light and
sound should only be done on the world model case.
=============
*/
void CG_AddPlayerWeapon( refEntity_t *parent, playerState_t *ps, centity_t *cent ) {
	refEntity_t	gun;
	refEntity_t	barrel;
	refEntity_t	flash;
	vec3_t		angles;
	weapon_t	weaponNum;
	weaponInfo_t	*weapon;
	centity_t	*nonPredictedCent;
	int anim, ammo = 0;
	
	weaponNum = cent->currentState.weapon;
	if(weaponNum <= WP_NONE)
		return;
	CG_RegisterWeapon( weaponNum );
	weapon = &cg_weapons[weaponNum];

	// add the weapon
	memset( &gun, 0, sizeof( gun ) );
	VectorCopy( parent->lightingOrigin, gun.lightingOrigin );
	gun.shadowPlane = parent->shadowPlane;
	gun.renderfx = parent->renderfx;

	// set custom shading for railgun refire rate
	if ( ps ) {
		float r, g, b, f;
		if(cent->currentState.powerups & (1 << PW_INVIS) )
			ammo = 0;
		else
			ammo = ps->ammo[ps->weapon];
		if(weaponNum == WP_RAILGUN)
		{
			f = 1.0 - cg.predictedPlayerState.weaponTime / 1500.0;
			if(cent->currentState.powerups & PW_HASTE)
				f *= g_HasteFactor.value;
			r = cgs.clientinfo[cent->currentState.clientNum].color[0];
			g = cgs.clientinfo[cent->currentState.clientNum].color[1];
			b = cgs.clientinfo[cent->currentState.clientNum].color[2];
		}
		else if(weaponNum)
		{
			f = ammo * 1.0 / BG_FindItemForWeapon(weaponNum)->quantity;
			r = 1;
			g = 1;
			b = 1;
		}
		
		if(f > 1.0)
			f = 1.0;
		else if(f < 0)
			f = 0;

		gun.shaderRGBA[0] = r * 255 * f;
		gun.shaderRGBA[1] = g * 255 * f;
		gun.shaderRGBA[2] = b * 255 * f;
		gun.shaderRGBA[3] = 128 + 127 * f;
			
		
	}
	else
	{
		float	f, r, g, b;

		if(cent->currentState.powerups & (1 << PW_INVIS) )
			ammo = 0; // time2 is invisibility time left
		else
			ammo = cent->currentState.time2;//time2 is ammo

		if(weaponNum == WP_RAILGUN)
		{
			f = (cg.time - cgs.clientinfo[cent->currentState.clientNum].lastfired) / 1500.0;
			if(cent->currentState.powerups & PW_HASTE)
				f *= g_HasteFactor.value;
			r = cgs.clientinfo[cent->currentState.clientNum].color[0];
			g = cgs.clientinfo[cent->currentState.clientNum].color[1];
			b = cgs.clientinfo[cent->currentState.clientNum].color[2];
		
		}
		else if(weaponNum)
		{
			f = ammo * 1.0 / BG_FindItemForWeapon(weaponNum)->quantity;
			r = 1;
			g = 1;
			b = 1;		
		}
		
		if(f > 1.0)
			f = 1.0;
		else if(f < 0)
			f = 0;
		gun.shaderRGBA[0] = r * 255 * f;
		gun.shaderRGBA[1] = g * 255 * f;
		gun.shaderRGBA[2] = b * 255 * f;
		gun.shaderRGBA[3] = 128 + 127 * f;
		
	}
	
		

	if(!(cg_coolAmmo.integer || g_ForceViz.integer == 2) || ammo < 0)
		ammo = 0;

	gun.hModel = weapon->weaponModel;
	if(weapon->weaponSkin)
		gun.customSkin = weapon->weaponSkin;
	if (!gun.hModel) {
		return;
	}

	//if ( !ps ) {
		// add weapon ready sound
		cent->pe.lightningFiring = qfalse;
		if ( ( cent->currentState.eFlags & EF_FIRING ) && weapon->firingSound ) {
			// lightning gun and guantlet make a different sound when fire is held down
			trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, weapon->firingSound );
			//CG_Printf("Skip?");
			cent->pe.lightningFiring = qtrue;
		} else if ( weapon->readySound ) {
			trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, weapon->readySound );
			
		}
	//}
	CG_PositionEntityOnTag( &gun, parent, parent->hModel, "tag_weapon");
	
	anim = cent->pe.torso.animationNumber & 15;
	if(!(parent->renderfx & RF_DEPTHHACK) && (anim == TORSO_ATTACK || anim == TORSO_STAND))// && (weaponNum == WP_LIGHTNING || weaponNum == WP_BFG || weaponNum == WP_PLASMAGUN || weaponNum == WP_MACHINEGUN || (cent->currentState.eFlags & EF_FIRING)) )
	{
		
		float test;

		if(DotProduct(parent->axis[0], gun.axis[0]) < 0.9)
		{ // gun's in a weird position; don't adjust it.
		}
		else
		{ // this is the view gun, so we need to adjust for paralax
			refEntity_t head;
			vec3_t endpos, dir;
			trace_t tr;
			
			memset( &head, 0, sizeof( head ) );
			AngleVectors(cent->lerpAngles, dir, NULL, NULL);
			CG_PositionEntityOnTag( &head, parent, parent->hModel, "tag_head");
			if(ps && (cg_thirdPerson.integer == -1 || g_ForceViz.integer) )
			{ // we're rendering from org, which is different from lerpOrigin, which is DUMB.
				head.origin[0] += cent->lerpOrigin[0] - parent->lightingOrigin[0];
				head.origin[1] += cent->lerpOrigin[1] - parent->lightingOrigin[1];
				head.origin[2] += cent->lerpOrigin[2] - parent->lightingOrigin[2];
			}
			VectorMA(head.origin, 8192, dir, endpos);
			CG_Trace(&tr, head.origin, vec3_origin, vec3_origin, endpos, cent->currentState.number, MASK_SHOT);
			if(tr.fraction < 1)
			{ // do paralax
				VectorSubtract(tr.endpos, gun.origin, dir);
				if(ps && (cg_thirdPerson.integer == -1 || g_ForceViz.integer) )
				{ // we're rendering from org, which is different from lerpOrigin, which is DUMB.
					dir[0] -= cent->lerpOrigin[0] - parent->lightingOrigin[0];
					dir[1] -= cent->lerpOrigin[1] - parent->lightingOrigin[1];
					dir[2] -= cent->lerpOrigin[2] - parent->lightingOrigin[2];
				}
			
				VectorNormalize(dir);
				VectorCopy(dir, gun.axis[0]);
				ProjectPointOnPlane(gun.axis[1], gun.axis[1], gun.axis[0]);
				VectorNormalize(gun.axis[1]);
				CrossProduct(gun.axis[0], gun.axis[1], gun.axis[2]);
			}
		}
	}
	if(weaponNum == WP_FLAMETHROWER)
	{ // add napalm canister
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( parent->lightingOrigin, barrel.lightingOrigin );
		barrel.shadowPlane = parent->shadowPlane;
		barrel.renderfx = parent->renderfx;

		barrel.hModel = weapon->barrelModel;
		barrel.customShader = cgs.media.armorShader;
		VectorScale(gun.axis[0], -0.6, barrel.axis[0]);
		VectorScale(gun.axis[1], 0.6, barrel.axis[1]);
		VectorScale(gun.axis[2], -0.6, barrel.axis[2]);
		VectorMA(gun.origin, 12, gun.axis[0], barrel.origin);
		if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
		{
			barrel.shaderRGBA[0] = 1;
			barrel.shaderRGBA[1] = 1;
			barrel.shaderRGBA[2] = 1;
			barrel.shaderRGBA[3] = (5000 - cent->currentState.time2	) / 5000.0;
		}
		else
			barrel.shaderRGBA[3] = 0;
		CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups );
		
	}
	
	else if(weaponNum == WP_BALL)
	{
		int i, j;
		vec3_t x, y;	
		float s, c;
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( parent->lightingOrigin, barrel.lightingOrigin );
		barrel.shadowPlane = parent->shadowPlane;
		barrel.renderfx = parent->renderfx;

		barrel.hModel = weapon->barrelModel;
		angles[YAW] = 0;
		angles[PITCH] = 0;
		angles[ROLL] = 0;
		AnglesToAxis( angles, barrel.axis );
		for(i=0;i<3;i++)
		{
			for(j=0;j<3;j++)
			{
				gun.axis[i][j] *= 0.18;
				barrel.axis[i][j] *= 0.4;
			}
		}
		
		CG_PositionRotatedEntityOnTag( &barrel, parent, parent->hModel, "tag_weapon" );
		
		if((cent->currentState.powerups & (1 << PW_INVIS) ) && cent->currentState.time2 < 5000)
		{
			barrel.shaderRGBA[0] = 1;
			barrel.shaderRGBA[1] = 1;
			barrel.shaderRGBA[2] = 1;
			barrel.shaderRGBA[3] = (5000 - cent->currentState.time2	) / 5000.0;
		}
		CG_AddWeaponWithPowerups( &barrel, 0);
		c = cos(cg.time / 100.0);
		s = sin(cg.time / 100.0);
		x[0] = gun.axis[0][0] * c + gun.axis[1][0] * s * 1.22;
		x[1] = gun.axis[0][1] * c + gun.axis[1][1] * s * 1.22;
		x[2] = gun.axis[0][2] * c + gun.axis[1][2] * s * 1.22;
		y[0] = gun.axis[1][0] * c - gun.axis[0][0] * s * 1.22;
		y[1] = gun.axis[1][1] * c - gun.axis[0][1] * s * 1.22;
		y[2] = gun.axis[1][2] * c - gun.axis[0][2] * s * 1.22;
		VectorCopy(x, gun.axis[0]);
		VectorCopy(y, gun.axis[1]);
		gun.customShader=trap_R_RegisterShader("models/powerups/instant/regen");
	}
	
	if((cent->currentState.powerups & (1 << PW_INVIS) ))
	{
		if(cent->currentState.time2 < 5000)
		{
			gun.shaderRGBA[0] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
			gun.shaderRGBA[1] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
			gun.shaderRGBA[2] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
			gun.shaderRGBA[3] = 255 * (5000 - cent->currentState.time2 ) / 5000.0;
		}
		else
			gun.shaderRGBA[3] = 0;
	}
	

	CG_AddWeaponWithPowerups( &gun, cent->currentState.powerups );

	// add the spinning barrel
	if ( weaponNum == WP_MACHINEGUN || weaponNum == WP_GAUNTLET || weaponNum == WP_BFG ) {
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( parent->lightingOrigin, barrel.lightingOrigin );
		barrel.shadowPlane = parent->shadowPlane;
		barrel.renderfx = parent->renderfx;

		barrel.hModel = weapon->barrelModel;
		angles[YAW] = 0;
		angles[PITCH] = 0;
		angles[ROLL] = CG_MachinegunSpinAngle( cent );
		AnglesToAxis( angles, barrel.axis );

		CG_PositionRotatedEntityOnTag( &barrel, &gun, weapon->weaponModel, "tag_barrel" );

		if(weapon->weaponSkin || (cent->currentState.powerups & (1 << PW_INVIS) ))
		{
			if(weapon->weaponSkin)
				barrel.customSkin = gun.customSkin;
			barrel.shaderRGBA[0] = gun.shaderRGBA[0];
			barrel.shaderRGBA[1] = gun.shaderRGBA[1];
			barrel.shaderRGBA[2] = gun.shaderRGBA[2];
			barrel.shaderRGBA[3] = gun.shaderRGBA[3];
			
		}
		
		CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups );
	}

	// add ammo belt
	if((weaponNum == WP_ROCKET_LAUNCHER || weaponNum == WP_RAILGUN) && ammo > 0)
	{
		int i;
		float twist;
		vec3_t org, sax;
		memset( &barrel, 0, sizeof( barrel ) );
		CG_PositionEntityOnTag( &barrel, &gun, weapon->weaponModel, "tag_flash" );
		VectorMA(barrel.origin, -8, barrel.axis[0], org);
		barrel.renderfx = parent->renderfx & (RF_THIRD_PERSON | RF_FIRST_PERSON);
		

		if(weaponNum == WP_ROCKET_LAUNCHER)
		{
			barrel.hModel = trap_R_RegisterModel( "models/powerups/ammo/rocket.md3" );			
		}
		else
		{
			barrel.hModel = cgs.media.sphereFlashModel;//trap_R_RegisterModel( "models/powerups/health/medium_sphere.md3" );
			barrel.customShader = cgs.media.armorShader;// trap_R_RegisterShader("models/weapons2/grapple/grapple06.tga");
			VectorMA(org, -8, barrel.axis[0], org);
		}
		VectorScale(barrel.axis[1], -1, barrel.axis[1]);
		VectorScale(barrel.axis[2], -1, barrel.axis[2]);
		for(i = 0; i < ammo; i++)
		{
			if (i > 20)
			{
				break;
				
			}
			else
			{
				twist = 0.003 * i;
				VectorNormalize(barrel.axis[2]);
				barrel.axis[2][2] -= 0.03;
				VectorNormalize(barrel.axis[2]);
				ProjectPointOnPlane(barrel.axis[1], barrel.axis[1], barrel.axis[2]);
				VectorNormalize(barrel.axis[1]);
				CrossProduct(barrel.axis[1], barrel.axis[2], barrel.axis[0]);
				
				VectorMA(org,  i * 2, barrel.axis[2], barrel.origin);			
				VectorMA(barrel.origin, sqrt(i) * sin(i / 4.0 + cg.time / 400.0) * 0.1, barrel.axis[1], barrel.origin);
				
				if(weaponNum == WP_ROCKET_LAUNCHER)
				{
				}
				else
				{
					VectorScale(barrel.axis[0], 0.225, barrel.axis[0]);
					VectorScale(barrel.axis[1], 0.1, barrel.axis[1]);
					VectorScale(barrel.axis[2], 0.1, barrel.axis[2]);
				}
				
				
			}
			CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups & ~(1 << PW_HASTE));
			

		}
		
	}
	else if(weaponNum == WP_MACHINEGUN && ammo > 0)
	{
		int i;
		float twist;
		vec3_t org, sax;
		memset( &barrel, 0, sizeof( barrel ) );
		VectorMA(gun.origin, 7.5, gun.axis[0], org);
		VectorMA(org, (ammo % 2) * 0.25, gun.axis[1], org);
		VectorMA(org, 3.6 - (ammo % 2) * 0.125, gun.axis[2], org);
		VectorScale(gun.axis[2], -0.6, barrel.axis[0]);
		VectorScale(gun.axis[1], 0.6, barrel.axis[1]);
		VectorScale(gun.axis[0], 0.6, barrel.axis[2]);
		barrel.hModel = cgs.media.machinegunBrassModel;
		barrel.renderfx = parent->renderfx & (RF_THIRD_PERSON | RF_FIRST_PERSON);
		for(i = 0; i < ammo; i++)
		{
			if(i == 9)
			{
				VectorMA(org, i / 15.0, gun.axis[1], org);
				VectorMA(org, -i * i / 400.0, gun.axis[2], org);
				VectorMA(org, 0.6, gun.axis[1], barrel.origin);
			}
			else if(i >= 10)
			{
				break;
			}
			else
			{
				VectorMA(org, i / 15.0, gun.axis[1], org);
				VectorMA(org, -i * i / 400.0, gun.axis[2], org);
				VectorMA(org, 0.75, gun.axis[1], barrel.origin);
			}
			//VectorMA(org, 1, gun.axis[0], barrel.origin);

			CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups & ~(1 << PW_HASTE) );	
			
		}
		VectorMA(org, -0.4 + sin(cg.time / 400.0) * 0.05, gun.axis[2], org);
		for(i = 0; i < ammo - 10; i++)
		{
			if (i > 90)// (i > 50)
			{
				break;
				//int x = i % 2, y = i / 2;
				
			}
			else
			{
				if(i > 10)
					twist = 0.03;
				else
					twist = 0.003 * i;
				barrel.axis[0][2] -= 0.03;
				VectorNormalize(barrel.axis[0]);
				ProjectPointOnPlane(barrel.axis[1], barrel.axis[1], barrel.axis[0]);
				VectorNormalize(barrel.axis[1]);
				CrossProduct(barrel.axis[0], barrel.axis[1], barrel.axis[2]);
				//VectorNormalize(barrel.axis[2]);
				VectorScale(barrel.axis[0], 0.6, barrel.axis[0]);
				VectorScale(barrel.axis[1], 0.6, barrel.axis[1]);
				VectorScale(barrel.axis[2], 0.6, barrel.axis[2]);
			
				VectorMA(org,  0.7 * (i - sqrt(i) * 0.1), barrel.axis[0], barrel.origin);			
				VectorMA(barrel.origin, sqrt(i + 1) * 1.5 + sqrt(i) * sin(i / 4.0 + cg.time / 400.0) * 0.1, barrel.axis[1], barrel.origin);			
			}
			CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups & ~(1 << PW_HASTE) );
			/*
			if(ammo > 60 && (100 - i) < ammo)
			{
				VectorMA(barrel.origin, -0.8, barrel.axis[1], barrel.origin);
				CG_AddWeaponWithPowerups( &barrel, cent->currentState.powerups );
			}
			*/


		}
	}

	// make sure we aren't looking at cg.predictedPlayerEntity for LG
	nonPredictedCent = &cg_entities[cent->currentState.clientNum];

	// if the index of the nonPredictedCent is not the same as the clientNum
	// then this is a fake player (like on teh single player podiums), so
	// go ahead and use the cent
	if( ( nonPredictedCent - cg_entities ) != cent->currentState.clientNum ) {
		nonPredictedCent = cent;
	}

	// add the flash
	
	memset( &flash, 0, sizeof( flash ) );
	VectorCopy( parent->lightingOrigin, flash.lightingOrigin );
	flash.shadowPlane = parent->shadowPlane;
	flash.renderfx = parent->renderfx;

	flash.hModel = weapon->flashModel;
	if (!flash.hModel) {
		return;
	}
	angles[YAW] = 0;
	angles[PITCH] = 0;
	angles[ROLL] = crandom() * 10;
	AnglesToAxis( angles, flash.axis );

	// colorize the railgun blast
	if ( weaponNum == WP_RAILGUN ) {
		clientInfo_t	*ci;

		ci = &cgs.clientinfo[ cent->currentState.clientNum ];
		flash.shaderRGBA[0] = 255 * ci->color[0];
		flash.shaderRGBA[1] = 255 * ci->color[1];
		flash.shaderRGBA[2] = 255 * ci->color[2];
	}

	CG_PositionRotatedEntityOnTag( &flash, &gun, weapon->weaponModel, "tag_flash");
	VectorCopy(flash.origin, cent->pe.railgunOrigin); 
	VectorCopy(flash.origin, cent->currentState.origin2); 
#if 0 // laser sight
	{
		vec3_t			forward, start, end;
		localEntity_t	*le;
		refEntity_t		*re;
		trace_t			tr;
		le = CG_AllocLocalEntity();
		re = &le->refEntity;

		le->leType = LE_FADE_RGB;
		le->startTime = cg.time;
		le->endTime = cg.time + 1;
		le->lifeRate = 1;

		re->shaderTime = cg.time / 1000.0f;
		re->reType = RT_RAIL_CORE;
		re->customShader = cgs.media.railCoreShader;
		
		VectorCopy( flash.origin, re->origin );
		AngleVectors(cent->lerpAngles, forward, NULL, NULL);
		VectorMA( re->origin, 8192, forward, end);
		CG_Trace(&tr, re->origin, vec3_origin, vec3_origin, end, cent->currentState.number, CONTENTS_SOLID | CONTENTS_PLAYERCLIP);
		VectorCopy(tr.endpos, re->oldorigin);
		le->color[0] = 1.0;
		le->color[1] = 0.0;
		le->color[2] = 0.0;
		le->color[3] = 0.025;

		AxisClear( re->axis );
	}
#endif
	if(weaponNum == WP_FLAMETHROWER)
	{
		VectorMA(flash.origin, 6.5, gun.axis[0], flash.origin);
		flash.customShader = trap_R_RegisterShader( "models/weapons2/rocketl/f_rocketl.md3" );
		VectorScale(flash.axis[0], 0.25, flash.axis[0]);
		VectorScale(flash.axis[1], 0.1, flash.axis[1]);
		VectorScale(flash.axis[2], 0.1, flash.axis[2]);
	}
	if ( ( weaponNum == WP_LIGHTNING || weaponNum == WP_GAUNTLET || weaponNum == WP_GRAPPLING_HOOK || weaponNum == WP_FLAMETHROWER)
		&& ( nonPredictedCent->currentState.eFlags & EF_FIRING ) ) 
	{
//		nonPredictedCent->pe.lightningFiring = qtrue;
		// continuous flash
	} else {
		// impulse flash
		if ( cg.time - cent->muzzleFlashTime > MUZZLE_FLASH_TIME && !cent->pe.railgunFlash ) {
			return;
		}
	}
	
	trap_R_AddRefEntityToScene( &flash );
	
	if ( !(parent->renderfx & RF_THIRD_PERSON) ) {
		// add lightning bolt
		CG_LightningBolt( nonPredictedCent, flash.origin );
		CG_FlameBolt( nonPredictedCent, flash.origin );
		// add rail trail
		CG_SpawnRailTrail( nonPredictedCent, flash.origin );

		// make a dlight for the flash
		if ( weapon->flashDlightColor[0] || weapon->flashDlightColor[1] || weapon->flashDlightColor[2] ) {
			trap_R_AddLightToScene( flash.origin, 200 + (rand()&31), weapon->flashDlightColor[0],
				weapon->flashDlightColor[1], weapon->flashDlightColor[2] );
		}
	}
	
}

/*
==============
CG_AddViewWeapon

Add the weapon, and flash for the player's view
==============
*/
void CG_AddViewWeapon( playerState_t *ps ) {
	refEntity_t	hand;
	centity_t	*cent;
	clientInfo_t	*ci;
	float		fovOffset;
	vec3_t		angles;
	weaponInfo_t	*weapon;

	if ( ps->persistant[PERS_TEAM] == TEAM_SPECTATOR ) {
		return;
	}

	if ( ps->pm_type == PM_INTERMISSION ) {
		return;
	}

	// no gun if in third person view
	if ( cg.renderingThirdPerson || cg_thirdPerson.integer || g_ForceViz.integer) {
		return;
	}

	// allow the gun to be completely removed
	if ( !cg_drawGun.integer ) {
		vec3_t		origin;

		if ( cg.predictedPlayerState.eFlags & EF_FIRING ) {
			// special hack for lightning gun...
			VectorCopy( cg.refdef.vieworg, origin );
			VectorMA( origin, -8, cg.refdef.viewaxis[2], origin );
			CG_FlameBolt( &cg_entities[ps->clientNum], origin );
			CG_LightningBolt( &cg_entities[ps->clientNum], origin );
		}
		return;
	}

	// don't draw if testing a gun model
	if ( cg.testGun ) {
		return;
	}

	// drop gun lower at higher fov
	if ( cg_fov.integer > 90 ) {
		fovOffset = -0.2 * ( cg_fov.integer - 90 );
	} else {
		fovOffset = 0;
	}

	cent = &cg.predictedPlayerEntity;	// &cg_entities[cg.snap->ps.clientNum];
	CG_RegisterWeapon( ps->weapon );
	weapon = &cg_weapons[ ps->weapon ];

	memset (&hand, 0, sizeof(hand));

	// set up gun position
	CG_CalculateWeaponPosition( hand.origin, angles );

	VectorMA( hand.origin, cg_gun_x.value, cg.refdef.viewaxis[0], hand.origin );
	VectorMA( hand.origin, cg_gun_y.value, cg.refdef.viewaxis[1], hand.origin );
	VectorMA( hand.origin, (cg_gun_z.value+fovOffset), cg.refdef.viewaxis[2], hand.origin );

	AnglesToAxis( angles, hand.axis );

	// map torso animations to weapon animations
	if ( cg_gun_frame.integer ) {
		// development tool
		hand.frame = hand.oldframe = cg_gun_frame.integer;
		hand.backlerp = 0;
	} else {
		// get clientinfo for animation map
		ci = &cgs.clientinfo[ cent->currentState.clientNum ];
		hand.frame = CG_MapTorsoToWeaponFrame( ci, cent->pe.torso.frame );
		hand.oldframe = CG_MapTorsoToWeaponFrame( ci, cent->pe.torso.oldFrame );
		hand.backlerp = cent->pe.torso.backlerp;
	}

	hand.hModel = weapon->handsModel;
	hand.renderfx = RF_DEPTHHACK | RF_FIRST_PERSON | RF_MINLIGHT;

	// add everything onto the hand
	CG_AddPlayerWeapon( &hand, ps, &cg.predictedPlayerEntity );
}

/*
==============================================================================

WEAPON SELECTION

==============================================================================
*/

/*
===================
CG_DrawWeaponSelect
===================
*/
void CG_DrawWeaponSelect( void ) {
	int		i;
	int		bits;
	int		count;
	int		x, y, w;
	float	*color;

	// don't display if dead
	if ( cg.predictedPlayerState.stats[STAT_HEALTH] <= 0 ) {
		return;
	}

	color = CG_FadeColor( cg.weaponSelectTime, WEAPON_SELECT_TIME );
	if ( !color ) {
		return;
	}
	trap_R_SetColor( color );

	// showing weapon select clears pickup item display, but not the blend blob
	cg.itemPickupTime = 0;

	// count the number of weapons owned
	bits = cg.snap->ps.stats[ STAT_WEAPONS ];
	count = 0; // ALWAYS at least have a fist/gauntlet
	for ( i = 1 ; i < 16 ; i++ ) {
		if ( bits & ( 2 << i ) ) {
			count++;
		}
	}
	x = 320 - count * 20;
	y = 380;

	for ( i = 1 ; i < 16 ; i++ ) {
		if (!( bits & ( 1 << i ) ) ) {
			continue;
		}

		CG_RegisterWeapon( i );

		// draw weapon icon
		CG_DrawPic( x, y, 32, 32, cg_weapons[i].weaponIcon );

		// draw selection marker
		if ( i == cg.weaponSelect ) {
			CG_DrawPic( x-4, y-4, 40, 40, cgs.media.selectShader );
		}

		// no ammo cross on top
		if ( !cg.snap->ps.ammo[ i ] ) {
			CG_DrawPic( x, y, 32, 32, cgs.media.noammoShader );
		}

		x += 40;
	}

	// draw the selected name
	if ( cg_weapons[ cg.weaponSelect ].item ) {
		char name[128];
		name[0] = '\0';
		if(!cg.predictedPlayerState.stats[STAT_WEAPONS] & (1 << cg.weaponSelect))
		{
			strcpy(name, "Dropped: ");
		}
		strcat(name, cg_weapons[ cg.weaponSelect ].item->pickup_name);
		if ( name ) {
			w = CG_DrawStrlen( name ) * BIGCHAR_WIDTH;
			x = ( SCREEN_WIDTH - w ) / 2;
			CG_DrawBigStringColor(x, y - 22, name, color);
		}
	}

	trap_R_SetColor( NULL );
}


/*
===============
CG_WeaponSelectable
===============
*/
static qboolean CG_WeaponSelectable( int i ) {
	if(cg.snap->ps.stats[STAT_EXTENDED_INFO] & EXT_LIMIT_WEAPONS)
	{
		
	}
	else if ( !cg.snap->ps.ammo[i] ) {
		return qfalse;
	}
	if ( ! (cg.snap->ps.stats[ STAT_WEAPONS ] & ( 1 << i ) ) ) {
		return qfalse;
	}

	return qtrue;
}

/*
===============
CG_NextWeapon_f
===============
*/
void CG_NextWeapon_f( void ) {
	int		i;
	int		original;

	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}

	cg.weaponSelectTime = cg.time;
	original = cg.weaponSelect;

	for ( i = 0 ; i < 16 ; i++ ) {
		cg.weaponSelect++;
		if ( cg.weaponSelect == 16 ) {
			cg.weaponSelect = 0;
		}
		//if ( cg.weaponSelect == WP_GAUNTLET ) {
		//	continue;		// never cycle to gauntlet
		//}
		if ( CG_WeaponSelectable( cg.weaponSelect ) ) {
			break;
		}
	}
	if ( i == 16 ) {
		cg.weaponSelect = original;
	}
}

/*
===============
CG_PrevWeapon_f
===============
*/
void CG_PrevWeapon_f( void ) {
	int		i;
	int		original;

	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}

	cg.weaponSelectTime = cg.time;
	original = cg.weaponSelect;

	for ( i = 0 ; i < 16 ; i++ ) {
		cg.weaponSelect--;
		if ( cg.weaponSelect == -1 ) {
			cg.weaponSelect = 15;
		}
		//if ( cg.weaponSelect == WP_GAUNTLET ) {
		//	continue;		// never cycle to gauntlet
		//}
		if ( CG_WeaponSelectable( cg.weaponSelect ) ) {
			break;
		}
	}
	if ( i == 16 ) {
		cg.weaponSelect = original;
	}
}

/*
===============
CG_Weapon_f
===============
*/
void CG_Weapon_f( void ) {
	int		num;

	if ( !cg.snap ) {
		return;
	}
	if ( cg.snap->ps.pm_flags & PMF_FOLLOW ) {
		return;
	}

	num = atoi( CG_Argv( 1 ) );

	if ( num < 0 || num > 15 ) {
		return;
	}

	cg.weaponSelectTime = cg.time;

	if ( ! ( cg.snap->ps.stats[STAT_WEAPONS] & ( 1 << num ) ) ) {
		return;		// don't have the weapon
	}
	//if(num == 0 && cg.snap->ps.stats[STAT_WEAPONS] > 1 && cg.snap->ps.weaponstate == WEAPON_RAISING)
	//{
	//	int i;
	//	for(i = 1; i < 16; i++)
	//	{
	//		if(cg.snap->ps.stats[STAT_WEAPONS] & (1 << i))
	//		{
	//			num = i;
	//			break;
	//		}
	//	}
	//}
	cg.weaponSelect = num;
}

/*
===================
CG_OutOfAmmoChange

The current weapon has just run out of ammo
===================
*/
void CG_OutOfAmmoChange( void ) {
	int		i;

	if(cg.snap->ps.stats[STAT_EXTENDED_INFO] & EXT_LIMIT_WEAPONS)
		return;
	cg.weaponSelectTime = cg.time;

	for ( i = 15 ; i >= 0 ; i-- ) {
		if ( CG_WeaponSelectable( i ) ) {
			cg.weaponSelect = i;
			break;
		}
	}
}



/*
===================================================================================================

WEAPON EVENTS

===================================================================================================
*/

/*
================
CG_FireWeapon

Caused by an EV_FIRE_WEAPON event
================
*/
void CG_FireWeapon( centity_t *cent ) {
	entityState_t *ent;
	int				c;
	weaponInfo_t	*weap;

	ent = &cent->currentState;
	if ( ent->weapon == WP_NONE ) {
		return;
	}
	if ( ent->weapon >= WP_NUM_WEAPONS ) {
		CG_Error( "CG_FireWeapon: ent->weapon >= WP_NUM_WEAPONS" );
		return;
	}
	weap = &cg_weapons[ ent->weapon ];

	// mark the entity as muzzle flashing, so when it is added it will
	// append the flash to the weapon model
	cent->muzzleFlashTime = cg.time;

	// lightning gun only does this this on initial press
	if ( ent->weapon == WP_LIGHTNING || ent->weapon == WP_FLAMETHROWER) {
		if ( cent->pe.lightningFiring ) {
			//CG_Printf("Skipping");
			return;
		}
	}
	
	// play quad sound if needed
	if ( cent->currentState.powerups & ( 1 << PW_QUAD ) ) {
		trap_S_StartSound (NULL, cent->currentState.number, CHAN_ITEM, cgs.media.quadSound );
	}

	// play a sound
	for ( c = 0 ; c < 4 ; c++ ) {
		if ( !weap->flashSound[c] ) {
			break;
		}
	}
	if ( c > 0 ) {
		c = rand() % c;
		if ( weap->flashSound[c] )
		{
			trap_S_StartSound( NULL, ent->number, CHAN_WEAPON, weap->flashSound[c] );
		}
	}

	// do brass ejection
	if ( weap->ejectBrassFunc && (cg_brassTime.integer || g_ForceViz.integer) ) {
		weap->ejectBrassFunc( cent );
	}
}


/*
=================
CG_MissileHitWall

Caused by an EV_MISSILE_MISS event, or directly by local bullet tracing
=================
*/
void CG_MissileHitWall( int weapon, int clientNum, vec3_t origin, vec3_t dir ) {
	qhandle_t		mod;
	qhandle_t		mark;
	qhandle_t		shader;
	sfxHandle_t		sfx;
	float			radius;
	float			mradius;
	float			light;
	vec3_t			lightColor;
	localEntity_t	*le;
	int				r;
	qboolean		alphaFade;
	qboolean		isSprite;
	int				duration;
	
	mark = 0;
	mradius = 0;
	radius = 32;
	sfx = 0;
	mod = 0;
	shader = 0;
	light = 0;
	lightColor[0] = 1;
	lightColor[1] = 1;
	lightColor[2] = 0;

	// set defaults
	isSprite = qfalse;
	duration = 600;

	switch ( weapon & 15) {
	default:
	case WP_FLAMETHROWER:
		mark = cgs.media.burnMarkShader; 
		radius = 48; 

		isSprite = qtrue;  
		duration = 900;    
		break;
	case WP_LIGHTNING:
		// no explosion at LG impact, it is added with the beam
		r = rand() & 3;
		if ( r < 2 ) {
			sfx = cgs.media.sfx_lghit2;
		} else if ( r == 2 ) {
			sfx = cgs.media.sfx_lghit1;
		} else {
			sfx = cgs.media.sfx_lghit3;
		}
		mark = cgs.media.holeMarkShader;
		radius = 12;
		break;
	case WP_GRENADE_LAUNCHER:
		mod = cgs.media.sphereFlashModel;
		shader = cgs.media.grenadeExplosionShader;
		sfx = cgs.media.sfx_rockexp;
		mark = cgs.media.burnMarkShader;
		radius = 64;
		light = 300;
		isSprite = !(cg_SphericalExplosions.integer || g_ForceViz.integer);
		if(!isSprite)
		{
			mradius = 64;
			le = CG_MakeExplosion( origin, dir, 
				   mod,	shader,
				   duration, qtrue);		
			
			le = CG_MakeExplosion( origin, dir, 
				   cgs.media.dishFlashModel, cgs.media.railExplosionShader,
				   300, qfalse);		
			
			le = CG_MakeExplosion( origin, dir, 
				   cgs.media.dishFlashModel,	shader,
				   duration, qfalse);		
			le->refEntity.axis[0][0] *= 0.5;
			le->refEntity.axis[1][0] *= 0.5;
			le->refEntity.axis[2][0] *= 0.5;
			le->refEntity.axis[0][1] *= 0.5;
			le->refEntity.axis[1][1] *= 0.5;
			le->refEntity.axis[2][1] *= 0.5;
			le->refEntity.axis[0][2] *= 0.5;
			le->refEntity.axis[1][2] *= 0.5;
			le->refEntity.axis[2][2] *= 0.5;
			r = rand() % 360;
			le->refEntity.rotation = r + 180;		
			

		}
		break;
	case WP_ROCKET_LAUNCHER:
		mod = cgs.media.sphereFlashModel;
		shader = cgs.media.rocketExplosionShader;
		sfx = cgs.media.sfx_rockexp;
		mark = cgs.media.burnMarkShader;
		radius = 64;
		light = 300;
		isSprite = !(cg_SphericalExplosions.integer || g_ForceViz.integer);
		if(!isSprite)
		{
			mradius = 64;
			le = CG_MakeExplosion( origin, dir, 
				   mod,	shader,
				   duration, qtrue);		
			
			le = CG_MakeExplosion( origin, dir, 
				   cgs.media.dishFlashModel, cgs.media.railExplosionShader,
				   300, qfalse);		
			
			le = CG_MakeExplosion( origin, dir, 
				   cgs.media.dishFlashModel,	shader,
				   duration, qfalse);		
			le->refEntity.axis[0][0] *= 0.5;
			le->refEntity.axis[1][0] *= 0.5;
			le->refEntity.axis[2][0] *= 0.5;
			le->refEntity.axis[0][1] *= 0.5;
			le->refEntity.axis[1][1] *= 0.5;
			le->refEntity.axis[2][1] *= 0.5;
			le->refEntity.axis[0][2] *= 0.5;
			le->refEntity.axis[1][2] *= 0.5;
			le->refEntity.axis[2][2] *= 0.5;
			r = rand() % 360;
			le->refEntity.rotation = r + 180;		
			
		}
		duration = 1000;
		lightColor[0] = 1;
		lightColor[1] = 0.75;
		lightColor[2] = 0.0;
		break;
	case WP_RAILGUN:
		mod = cgs.media.ringFlashModel;
		shader = cgs.media.railExplosionShader;
		sfx = cgs.media.sfx_plasmaexp;
		mark = cgs.media.energyMarkShader;
		radius = 24;
		break;
	case WP_PLASMAGUN:
		mod = cgs.media.ringFlashModel;
		shader = cgs.media.plasmaExplosionShader;
		sfx = cgs.media.sfx_plasmaexp;
		mark = cgs.media.energyMarkShader;
		radius = 16;
		break;
	case WP_BFG:
		mod = cgs.media.sphereFlashModel;
		shader = cgs.media.bfgExplosionShader;
		sfx = cgs.media.sfx_rockexp;
		mark = cgs.media.burnMarkShader;
		duration = 300;
		radius = 32;
		mradius = 64;
		isSprite = !(cg_SphericalExplosions.integer || g_ForceViz.integer);
		break;
	case WP_SHOTGUN:
		mod = cgs.media.bulletFlashModel;
		shader = cgs.media.bulletExplosionShader;
		mark = cgs.media.bulletMarkShader;
		sfx = 0;
		radius = 1;//4
		break;

	case WP_MACHINEGUN:
		mod = cgs.media.bulletFlashModel;
		shader = cgs.media.bulletExplosionShader;
		mark = cgs.media.bulletMarkShader;

		r = rand() & 3;
		if ( r < 2 ) {
			sfx = cgs.media.sfx_ric1;
		} else if ( r == 2 ) {
			sfx = cgs.media.sfx_ric2;
		} else {
			sfx = cgs.media.sfx_ric3;
		}

		radius = 8;
		break;
	}

	if ( sfx ) {
		trap_S_StartSound( origin, ENTITYNUM_WORLD, CHAN_AUTO, sfx );
	}

	//
	// create the explosion
	//
	if ( mod ) {
		if(weapon > 15)
		{
			weapon &= 15;
			le = CG_MakeExplosion( origin, dir, 
							   mod,	cgs.media.quadShader,
							   duration, isSprite );		
			le->light = light;
			le->lightColor[0] = 0;
			le->lightColor[1] = 0.5;
			le->lightColor[2] = 1;
			if(weapon == WP_ROCKET_LAUNCHER || weapon == WP_GRENADE_LAUNCHER || weapon == WP_BFG)
			{
			//le->refEntity.customShader = ;//cgs.media.plasmaBallShader;
				le->refEntity.renderfx |= RF_EXPANDING;			
				le->refEntity.oldorigin[0] = crandom() * 180;
				le->refEntity.oldorigin[1] = crandom() * 180;
				le->refEntity.oldorigin[2] = crandom() * 180;
			/*le = CG_MakeExplosion( origin, dir, 
							   mod,	shader,
							   duration, qtrue);		*/

				le->refEntity.rotation = r;		
			}
		
		}
		le = CG_MakeExplosion( origin, dir, 
							   mod,	shader,
							   duration, isSprite );		
		if(mradius)
			le->refEntity.radius = mradius;
		le->light = light;
		
		VectorCopy( lightColor, le->lightColor );
		if ( weapon == WP_RAILGUN ) {
			// colorize with client color
			VectorCopy( cgs.clientinfo[clientNum].color, le->color );
		}
		else if(weapon == WP_ROCKET_LAUNCHER || weapon == WP_GRENADE_LAUNCHER || weapon == WP_BFG)
		{
			//le->refEntity.customShader = ;//cgs.media.plasmaBallShader;
			le->refEntity.renderfx |= RF_EXPANDING;			
			le->refEntity.oldorigin[0] = crandom() * 180;
			le->refEntity.oldorigin[1] = crandom() * 180;
			le->refEntity.oldorigin[2] = crandom() * 180;
			/*le = CG_MakeExplosion( origin, dir, 
							   mod,	shader,
							   duration, qtrue);		*/

			le->refEntity.rotation = r;		
		}
		else if(weapon == WP_SHOTGUN)
		{ // smaller damnit
			le->refEntity.axis[0][0] *= 0.2;
			le->refEntity.axis[1][0] *= 0.2;
			le->refEntity.axis[2][0] *= 0.2;
			le->refEntity.axis[0][1] *= 0.2;
			le->refEntity.axis[1][1] *= 0.2;
			le->refEntity.axis[2][1] *= 0.2;
			le->refEntity.axis[0][2] *= 0.2;
			le->refEntity.axis[1][2] *= 0.2;
			le->refEntity.axis[2][2] *= 0.2;
			
		}
	}
	weapon &= 15;
	//
	// impact mark
	//
	alphaFade = (mark == cgs.media.energyMarkShader);	// plasma fades alpha, all others fade color
	if ( weapon == WP_RAILGUN ) {
		float	*color;

		// colorize with client color
		color = cgs.clientinfo[clientNum].color;
		CG_ImpactMark( mark, origin, dir, random()*360, color[0],color[1], color[2],1, alphaFade, radius, qfalse );
	} else {
		CG_ImpactMark( mark, origin, dir, random()*360, 1,1,1,1, alphaFade, radius, qfalse );
	}
}


/*
=================
CG_MissileHitPlayer
=================
*/
void CG_MissileHitPlayer( int weapon, vec3_t origin, vec3_t dir, int entityNum ) {
	
	CG_Bleed( origin, entityNum );

	// some weapons will make an explosion with the blood, while
	// others will just make the blood
	switch ( weapon ) {
	case WP_GRENADE_LAUNCHER:
	case WP_ROCKET_LAUNCHER:
		// sam mdy ft: include flame animation if hit a player (for burning players)
	case WP_FLAMETHROWER:

		CG_MissileHitWall( weapon, 0, origin, dir );
		break;
	default:
		break;
	}
}

/*
=================
CG_MissileHitPlayer
=================
*/
void CG_MissileHitRPlayer( int weapon, vec3_t origin, vec3_t dir, int entityNum ) {
	CG_RBleed( origin, entityNum );

	// some weapons will make an explosion with the blood, while
	// others will just make the blood
	switch ( weapon ) {
	case WP_GRENADE_LAUNCHER:
	case WP_ROCKET_LAUNCHER:
		CG_MissileHitWall( weapon, 0, origin, dir );
		break;
	default:
		break;
	}
}


/*
============================================================================

SHOTGUN TRACING

============================================================================
*/

/*
================
CG_ShotgunPellet
================
*/
static void CG_ShotgunPellet( vec3_t start, vec3_t end, int skipNum ) {
	trace_t		tr;
	int sourceContentType, destContentType;

	CG_Trace( &tr, start, NULL, NULL, end, skipNum, MASK_SHOT );
	
	sourceContentType = trap_CM_PointContents( start, 0 );
	destContentType = trap_CM_PointContents( tr.endpos, 0 );

	// FIXME: should probably move this cruft into CG_BubbleTrail
	if ( sourceContentType == destContentType ) {
		if ( sourceContentType & CONTENTS_WATER ) {
			CG_BubbleTrail( start, tr.endpos, 32 );
		}
	} else if ( sourceContentType & CONTENTS_WATER ) {
		trace_t trace;

		trap_CM_BoxTrace( &trace, end, start, NULL, NULL, 0, CONTENTS_WATER );
		CG_BubbleTrail( start, trace.endpos, 32 );
	} else if ( destContentType & CONTENTS_WATER ) {
		trace_t trace;

		trap_CM_BoxTrace( &trace, start, end, NULL, NULL, 0, CONTENTS_WATER );
		CG_BubbleTrail( tr.endpos, trace.endpos, 32 );
	}

	if (  tr.surfaceFlags & SURF_NOIMPACT ) {
		return;
	}
	
	if ( cg_entities[tr.entityNum].currentState.eType == ET_PLAYER ) {
	//	CG_MissileHitPlayer( WP_SHOTGUN, tr.endpos, tr.plane.normal, tr.entityNum );
	} else 
	{
		if ( tr.surfaceFlags & SURF_NOIMPACT ) {
			// SURF_NOIMPACT will not make a flame puff or a mark
			return;
		}
		CG_MissileHitWall( WP_SHOTGUN, 0, tr.endpos, tr.plane.normal );
	}
}

/*
================
CG_ShotgunPattern

Perform the same traces the server did to locate the
hit splashes (FIXME: ranom seed isn't synce anymore)
================
*/
static void CG_ShotgunPattern( vec3_t origin, vec3_t origin2, int otherEntNum ) {
	int			i;
	float		r, u;
	vec3_t		end;
	vec3_t		forward, right, up;

	// derive the right and up vectors from the forward vector, because
	// the client won't have any other information
	VectorNormalize2( origin2, forward );
	PerpendicularVector( right, forward );
	CrossProduct( forward, right, up );

	// generate the "random" spread pattern
	for ( i = 0 ; i < DEFAULT_SHOTGUN_COUNT ; i++ ) {		
		r = crandom() * DEFAULT_SHOTGUN_SPREAD;
		u = crandom() * DEFAULT_SHOTGUN_SPREAD;
		VectorMA( origin, 8192, forward, end);
		VectorMA (end, r, right, end);
		VectorMA (end, u, up, end);

		CG_ShotgunPellet( origin, end, otherEntNum );
	}
}

/*
==============
CG_ShotgunFire
==============
*/
void CG_ShotgunFire( entityState_t *es ) {
	vec3_t	v;
	int		contents;

	VectorSubtract( es->origin2, es->pos.trBase, v );
	VectorNormalize( v );
	VectorScale( v, 32, v );
	VectorAdd( es->pos.trBase, v, v );
	if ( cgs.glconfig.hardwareType != GLHW_RAGEPRO ) {
		// ragepro can't alpha fade, so don't even bother with smoke
		vec3_t			up;

		contents = trap_CM_PointContents( es->pos.trBase, 0 );
		if ( !( contents & CONTENTS_WATER ) ) {
			VectorSet( up, 0, 0, 8 );
			CG_SmokePuff( v, up, 32, 1, 1, 1, 0.33, 900, cg.time, LEF_PUFF_DONT_SCALE, cgs.media.shotgunSmokePuffShader );
		}
	}
	CG_ShotgunPattern( es->pos.trBase, es->origin2, es->otherEntityNum );
}

/*
============================================================================

BULLETS

============================================================================
*/


/*
===============
CG_Tracer
===============
*/
void CG_Tracer( vec3_t source, vec3_t dest ) {
	vec3_t		forward, right;
	polyVert_t	verts[4];
	vec3_t		line;
	float		len, begin, end;
	vec3_t		start, finish;
	vec3_t		midpoint;

	// tracer
	VectorSubtract( dest, source, forward );
	len = VectorNormalize( forward );

	// start at least a little ways from the muzzle
	if ( len < 100 ) {
		return;
	}
	begin = 50 + random() * (len - 100) * 0.5;
	end = begin + 500;//cg_tracerLength.value;
	if ( end > len ) {
		end = len;
	}
	VectorMA( source, begin, forward, start );
	VectorMA( source, end, forward, finish );

	line[0] = DotProduct( forward, cg.refdef.viewaxis[1] );
	line[1] = DotProduct( forward, cg.refdef.viewaxis[2] );

	VectorScale( cg.refdef.viewaxis[1], line[1], right );
	VectorMA( right, -line[0], cg.refdef.viewaxis[2], right );
	VectorNormalize( right );

	VectorMA( finish, cg_tracerWidth.value, right, verts[0].xyz );//100 was cg_tracerWidth.value
	verts[0].st[0] = 0;
	verts[0].st[1] = 1;
	verts[0].modulate[0] = 255;
	verts[0].modulate[1] = 255;
	verts[0].modulate[2] = 255;
	verts[0].modulate[3] = 255;
	
	VectorMA( finish, -cg_tracerWidth.value, right, verts[1].xyz );//100 was cg_tracerWidth.value
	verts[1].st[0] = 1;
	verts[1].st[1] = 1;
	verts[1].modulate[0] = 255;
	verts[1].modulate[1] = 255;
	verts[1].modulate[2] = 255;
	verts[1].modulate[3] = 255;

	VectorMA( start, -cg_tracerWidth.value, right, verts[2].xyz );//100 was cg_tracerWidth.value
	verts[2].st[0] = 1;
	verts[2].st[1] = 0;
	verts[2].modulate[0] = 255;
	verts[2].modulate[1] = 255;
	verts[2].modulate[2] = 255;
	verts[2].modulate[3] = 255;

	VectorMA( start, cg_tracerWidth.value, right, verts[3].xyz );//100 was cg_tracerWidth.value
	verts[3].st[0] = 0;
	verts[3].st[1] = 0;
	verts[3].modulate[0] = 255;
	verts[3].modulate[1] = 255;
	verts[3].modulate[2] = 255;
	verts[3].modulate[3] = 255;

	trap_R_AddPolyToScene( cgs.media.tracerShader, 4, verts );

	midpoint[0] = ( start[0] + finish[0] ) * 0.5;
	midpoint[1] = ( start[1] + finish[1] ) * 0.5;
	midpoint[2] = ( start[2] + finish[2] ) * 0.5;

	// add the tracer sound
	trap_S_StartSound( midpoint, ENTITYNUM_WORLD, CHAN_AUTO, cgs.media.tracerSound );

}


/*
======================
CG_CalcMuzzlePoint
======================
*/
static qboolean	CG_CalcMuzzlePoint( int entityNum, vec3_t muzzle ) {
	vec3_t		forward;
	centity_t	*cent;
	int			anim;
	// ## Hentai ##
	cent = &cg_entities[entityNum];
	//if ( !cent->currentValid ) {
//		CG_Printf("ent %d not valid\n", entityNum);
	//	return qfalse;
	//}
	VectorCopy(cent->lerpOrigin, muzzle);
	//VectorCopy(cent->pe.railgunOrigin, muzzle);
	/*
	if ( entityNum == cg.snap->ps.clientNum ) {
		VectorCopy( cg.snap->ps.origin, muzzle );
		muzzle[2] += cg.snap->ps.viewheight;
		AngleVectors( cg.snap->ps.viewangles, forward, NULL, NULL );
		VectorMA( muzzle, 14, forward, muzzle );
		return qtrue;
	}

	cent = &cg_entities[entityNum];	
	if ( !cent->currentValid ) {
		return qfalse;
	}

	VectorCopy( cent->currentState.pos.trBase, muzzle );

	AngleVectors( cent->currentState.apos.trBase, forward, NULL, NULL );
	anim = cent->currentState.legsAnim & ~ANIM_TOGGLEBIT;
	if ( anim == LEGS_WALKCR || anim == LEGS_IDLECR ) {
		muzzle[2] += CROUCH_VIEWHEIGHT;
	} else {
		muzzle[2] += DEFAULT_VIEWHEIGHT;
	}

	VectorMA( muzzle, 14, forward, muzzle );
	*/
	return qtrue;

}

/*
======================
CG_Bullet

Renders bullet effects.
======================
*/

void CG_QuadTracer(vec3_t start, vec3_t end)
{
	localEntity_t	*le;
	refEntity_t		*re;
	le = CG_AllocLocalEntity();
	re = &le->refEntity;
		le->leType = LE_FADE_RGB;
	le->startTime = cg.time;
	le->endTime = cg.time + 1;
	le->lifeRate = 1;
		re->shaderTime = cg.time / 1000.0f;
	re->reType = RT_RAIL_CORE;
	re->customShader = cgs.media.railCoreShader;
	
	VectorCopy( start, re->origin );
	VectorCopy(end, re->oldorigin);
	le->color[0] = 0.0;
	le->color[1] = 0.6;
	le->color[2] = 1.0;
	le->color[3] = 1.0;
	AxisClear( re->axis );

}
void CG_Bullet( vec3_t start, vec3_t end, int sourceEntityNum, vec3_t normal, qboolean flesh, int fleshEntityNum ) {
	trace_t trace;
	int sourceContentType, destContentType;
	//vec3_t		start;
	// if the shooter is currently valid, calc a source point and possibly
	// do trail effects
	
	
	if ( sourceEntityNum >= 0 && cg_tracerChance.value > 0 ) {
		//if ( CG_CalcMuzzlePoint( sourceEntityNum, start ) ) {
			sourceContentType = trap_CM_PointContents( start, 0 );
			destContentType = trap_CM_PointContents( end, 0 );

			// do a complete bubble trail if necessary
			if ( ( sourceContentType == destContentType ) && ( sourceContentType & CONTENTS_WATER ) ) {
				CG_BubbleTrail( start, end, 32 );
			}
			// bubble trail from water into air
			else if ( ( sourceContentType & CONTENTS_WATER ) ) {
				trap_CM_BoxTrace( &trace, end, start, NULL, NULL, 0, CONTENTS_WATER );
				CG_BubbleTrail( start, trace.endpos, 32 );
			}
			// bubble trail from air into water
			else if ( ( destContentType & CONTENTS_WATER ) ) {
				trap_CM_BoxTrace( &trace, start, end, NULL, NULL, 0, CONTENTS_WATER );
				CG_BubbleTrail( end, trace.endpos, 32 );
			}

			// draw a tracer
			if ( 1 ) {//random() < cg_tracerChance.value ) {
				CG_Tracer( start, end );
			}
		//}
	}

	// impact splash and mark
	
	if(fleshEntityNum == ENTITYNUM_NONE)
		return;
	if(flesh == -1)
	{
		CG_RBleed( end, fleshEntityNum );
	}
	else if ( flesh ) {
		CG_Bleed( end, fleshEntityNum );
	} else {
		CG_MissileHitWall( WP_MACHINEGUN, 0, end, normal );
	}

}
