// Copyright (C) 1999-2000 Id Software, Inc.
//
// cg_ents.c -- present snapshot entities, happens every single frame

#include "cg_local.h"


/*
======================
CG_PositionEntityOnTag

Modifies the entities position and axis by the given
tag location
======================
*/
void CG_PositionEntityOnTag( refEntity_t *entity, const refEntity_t *parent, 
							qhandle_t parentModel, char *tagName ) {
	int				i;
	orientation_t	lerped;
	
	// lerp the tag
	trap_R_LerpTag( &lerped, parentModel, parent->oldframe, parent->frame,
		1.0 - parent->backlerp, tagName );

	// FIXME: allow origin offsets along tag?
	VectorCopy( parent->origin, entity->origin );
	for ( i = 0 ; i < 3 ; i++ ) {
		VectorMA( entity->origin, lerped.origin[i], parent->axis[i], entity->origin );
	}

	// had to cast away the const to avoid compiler problems...
	MatrixMultiply( lerped.axis, ((refEntity_t *)parent)->axis, entity->axis );
	entity->backlerp = parent->backlerp;
}


/*
======================
CG_PositionRotatedEntityOnTag

Modifies the entities position and axis by the given
tag location
======================
*/
void CG_PositionRotatedEntityOnTag( refEntity_t *entity, const refEntity_t *parent, 
							qhandle_t parentModel, char *tagName ) {
	int				i;
	orientation_t	lerped;
	vec3_t			tempAxis[3];

//AxisClear( entity->axis );
	// lerp the tag
	trap_R_LerpTag( &lerped, parentModel, parent->oldframe, parent->frame,
		1.0 - parent->backlerp, tagName );

	// FIXME: allow origin offsets along tag?
	VectorCopy( parent->origin, entity->origin );
	for ( i = 0 ; i < 3 ; i++ ) {
		VectorMA( entity->origin, lerped.origin[i], parent->axis[i], entity->origin );
	}

	// had to cast away the const to avoid compiler problems...
	MatrixMultiply( entity->axis, lerped.axis, tempAxis );
	MatrixMultiply( tempAxis, ((refEntity_t *)parent)->axis, entity->axis );
}



/*
==========================================================================

FUNCTIONS CALLED EACH FRAME

==========================================================================
*/

/*
======================
CG_SetEntitySoundPosition

Also called by event processing code
======================
*/
void CG_SetEntitySoundPosition( centity_t *cent ) {
	if ( cent->currentState.solid == SOLID_BMODEL ) {
		vec3_t	origin;
		float	*v;

		v = cgs.inlineModelMidpoints[ cent->currentState.modelindex ];
		VectorAdd( cent->lerpOrigin, v, origin );
		trap_S_UpdateEntityPosition( cent->currentState.number, origin );
	} else {
		trap_S_UpdateEntityPosition( cent->currentState.number, cent->lerpOrigin );
	}
}

/*
==================
CG_EntityEffects

Add continuous entity effects, like local entity emission and lighting
==================
*/
static void CG_EntityEffects( centity_t *cent ) {

	// update sound origins
	CG_SetEntitySoundPosition( cent );

	// add loop sound
	if ( cent->currentState.loopSound ) {
		trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, 
			cgs.gameSounds[ cent->currentState.loopSound ] );
	}


	// constant light glow
	if ( cent->currentState.constantLight && cent->currentState.eType != ET_PLAYER) {
		int		cl;
		int		i, r, g, b;

		cl = cent->currentState.constantLight;
		r = cl & 255;
		g = ( cl >> 8 ) & 255;
		b = ( cl >> 16 ) & 255;
		i = ( ( cl >> 24 ) & 255 ) * 4;
		trap_R_AddLightToScene( cent->lerpOrigin, i, r, g, b );
	}

}


/*
==================
CG_General
==================
*/
static void CG_General( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// if set to invisible, skip
	if (!s1->modelindex) {
		return;
	}

	memset (&ent, 0, sizeof(ent));

	// set frame

	ent.frame = s1->frame;
	ent.oldframe = ent.frame;
	ent.backlerp = 0;

	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);

	ent.hModel = cgs.gameModels[s1->modelindex];

	// player model
	if (s1->number == cg.snap->ps.clientNum) {
		ent.renderfx |= RF_THIRD_PERSON;	// only draw from mirrors
	}

	// convert angles to axis
	AnglesToAxis( cent->lerpAngles, ent.axis );

	// add to refresh list
	trap_R_AddRefEntityToScene (&ent);
}

/*
==================
CG_Speaker

Speaker entities can automatically play sounds
==================
*/
static void CG_Speaker( centity_t *cent ) {
	if ( ! cent->currentState.clientNum ) {	// FIXME: use something other than clientNum...
		return;		// not auto triggering
	}

	if ( cg.time < cent->miscTime ) {
		return;
	}

	trap_S_StartSound (NULL, cent->currentState.number, CHAN_ITEM, cgs.gameSounds[cent->currentState.eventParm] );

	//	ent->s.frame = ent->wait * 10;
	//	ent->s.clientNum = ent->random * 10;
	cent->miscTime = cg.time + cent->currentState.frame * 100 + cent->currentState.clientNum * 100 * crandom();
}

/*
==================
CG_Item
==================
*/
static void CG_Item( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*es;
	gitem_t				*item;
	int					msec;
	float				frac;
	float				scale;
	float				groundz;
	vec3_t end, mins, maxs;
	trace_t tr;
	es = &cent->currentState;
	if ( es->modelindex >= bg_numItems ) {
		CG_Error( "Bad item index %i on entity", es->modelindex );
	}

	// if set to invisible, skip
	if ( !es->modelindex || ( es->eFlags & EF_NODRAW ) ) {
		return;
	}


	item = &bg_itemlist[ es->modelindex ];
	if ( cg_simpleItems.integer && item->giType != IT_TEAM ) {
		memset( &ent, 0, sizeof( ent ) );
		ent.reType = RT_SPRITE;
		VectorCopy( cent->lerpOrigin, ent.origin );
		ent.radius = 14;
		ent.customShader = cg_items[es->modelindex].icon;
		ent.shaderRGBA[0] = 255;
		ent.shaderRGBA[1] = 255;
		ent.shaderRGBA[2] = 255;
		ent.shaderRGBA[3] = 255;
		trap_R_AddRefEntityToScene(&ent);
		return;
	}
	end[0] = cent->lerpOrigin[0];
	end[1] = cent->lerpOrigin[1];
	end[2] = cent->lerpOrigin[2] - 24;
	mins[0] = -10;
	mins[1] = -10;
	mins[2] = 0;
	maxs[0] = 10;
	maxs[1] = 10;
	maxs[2] = 0;
	CG_Trace( &tr, cent->lerpOrigin, mins, maxs, end, 0, MASK_SOLID );
	groundz = tr.endpos[2];
	
	// ### HENTAI ###
#if 1
	{ // test backclip
		float f;
		vec3_t test;
		test[0] = cent->lerpOrigin[0] - cg.refdef.vieworg[0];
		test[1] = cent->lerpOrigin[1] - cg.refdef.vieworg[1];
		test[2] = cent->lerpOrigin[2] - cg.refdef.vieworg[2];
		
		f = test[0] * cg.refdef.viewaxis[0][0] + test[1] * cg.refdef.viewaxis[0][1] + test[2] * cg.refdef.viewaxis[0][2];
		if(f < ITEM_RADIUS * -2)
			return;
		
	}
#endif
	
	
	if((item->giType != IT_TEAM) && ((cg_BobItems.value > 0 && g_ForceViz.integer == 0) || item->giType == IT_POWERUP) || (cent->currentState.eFlags & EF_TELEPORT_BIT)) // powerups always bob; flags never bob
	{
		// items bob up and down continuously
		scale = (0.005 + cent->currentState.number * 0.00001) * fabs(cg_BobItems.value);		
		cent->lerpOrigin[2] += 4 + cos( ( cg.time + 1000 ) *  scale ) * 4;
	
		memset (&ent, 0, sizeof(ent));
	
		// autorotate at one of two speeds
		if ( item->giType == IT_HEALTH ) { 
			VectorCopy( cg.autoAnglesFast, cent->lerpAngles );
			AxisCopy( cg.autoAxisFast, ent.axis );
		} else {
			VectorCopy( cg.autoAngles, cent->lerpAngles );
			AxisCopy( cg.autoAxis, ent.axis );
		}
	}
	else
	{
		memset (&ent, 0, sizeof(ent));		
		
		if(item->giType == IT_WEAPON && item->giTag != WP_BALL)
		{
			
			if(tr.fraction == 1.0)
				cent->rawAngles[YAW] = cent->lerpAngles[YAW] = cg.autoAnglesFast[YAW];
			
			cent->lerpOrigin[2] -= 18; // drop them to the ground
			cent->lerpAngles[PITCH] = 0;
			cent->lerpAngles[YAW] = cent->rawAngles[YAW];
			cent->lerpAngles[ROLL] = -45; // make it lean a little
			if(item->giTag == WP_FLAMETHROWER)
				cent->lerpAngles[ROLL] = -75; // make it lean a little more, so the canister doesn't clip through the ground
		}
		AnglesToAxis(cent->lerpAngles, ent.axis);
	}
	// ### HENTAI ###

	// the weapons have their origin where they attatch to player
	// models, so we need to offset them or they will rotate
	// eccentricly
	
	if ( item->giType == IT_WEAPON ) {
		weaponInfo_t	*wi;

		wi = &cg_weapons[item->giTag];
		cent->lerpOrigin[0] -= 
			wi->weaponMidpoint[0] * ent.axis[0][0] +
			wi->weaponMidpoint[1] * ent.axis[1][0] +
			wi->weaponMidpoint[2] * ent.axis[2][0];
		cent->lerpOrigin[1] -= 
			wi->weaponMidpoint[0] * ent.axis[0][1] +
			wi->weaponMidpoint[1] * ent.axis[1][1] +
			wi->weaponMidpoint[2] * ent.axis[2][1];
		cent->lerpOrigin[2] -= 
			wi->weaponMidpoint[0] * ent.axis[0][2] +
			wi->weaponMidpoint[1] * ent.axis[1][2] +
			wi->weaponMidpoint[2] * ent.axis[2][2];

		cent->lerpOrigin[2] += 8;	// an extra height boost
	}
	
	ent.hModel = cg_items[es->modelindex].models[0];
	if(item->giType == IT_AMMO && cent->currentState.powerups > 0 && (g_ForceViz.integer == 2 || cg_coolAmmo.integer))
	{
		switch(item->giTag)
		{
		default:
			break;
		case WP_MACHINEGUN:
			ent.hModel = cgs.media.machinegunBrassModel;
			cent->lerpOrigin[2] -= 12;
			VectorScale( ent.axis[0], 0.6, ent.axis[0] );
			VectorScale( ent.axis[1], 0.6, ent.axis[1] );
			VectorScale( ent.axis[2], 0.6, ent.axis[2] );
			break;
		case WP_SHOTGUN:
			ent.hModel = cgs.media.shotgunBrassModel;
			VectorScale( ent.axis[0], 0.6, ent.axis[0] );
			VectorScale( ent.axis[1], 0.6, ent.axis[1] );
			VectorScale( ent.axis[2], 0.6, ent.axis[2] );
			cent->lerpOrigin[2] -= 11;
			break;
		case WP_FLAMETHROWER:
		{
			vec3_t naxis2;
			ent.hModel = trap_R_RegisterModel( "models/ammo/grenade1.md3" );
			ent.customShader = cgs.media.armorShader;
			VectorScale( ent.axis[0], 0.6, naxis2 );
			VectorScale( ent.axis[1], 0.6, ent.axis[1] );
			VectorScale( ent.axis[2], -0.6, ent.axis[0] );
			VectorCopy(naxis2, ent.axis[2]);
			cent->lerpOrigin[2] -= 10;
			break;
		}
		case WP_GRENADE_LAUNCHER:
			ent.hModel = trap_R_RegisterModel( "models/ammo/grenade1.md3" );
			VectorScale( ent.axis[0], 0.3, ent.axis[0] );
			VectorScale( ent.axis[1], 0.3, ent.axis[1] );
			VectorScale( ent.axis[2], 0.3, ent.axis[2] );
			cent->lerpOrigin[2] -= 14;
			break;
		
		case WP_ROCKET_LAUNCHER:
			ent.hModel = trap_R_RegisterModel( "models/powerups/ammo/rocket.md3" );
			cent->lerpOrigin[2] -= 14;
			break;
		case WP_RAILGUN:
			ent.hModel = cgs.media.sphereFlashModel; //trap_R_RegisterModel( "models/powerups/health/medium_sphere.md3" );
			ent.customShader = cgs.media.armorShader;//trap_R_RegisterShader("models/weapons2/grapple/grapple06.tga");
			VectorScale( ent.axis[0], 0.225, ent.axis[0] );
			VectorScale( ent.axis[1], 0.1, ent.axis[1] );
			VectorScale( ent.axis[2], 0.1, ent.axis[2] );
			cent->lerpOrigin[2] -= 14;
			break;
		case WP_PLASMAGUN:
		case WP_LIGHTNING:
		case WP_BFG:
		case WP_ENERGY:
			VectorScale( ent.axis[0], 0.25, ent.axis[0] );
			VectorScale( ent.axis[1], 0.25, ent.axis[1] );
			VectorScale( ent.axis[2], 0.25, ent.axis[2] );
			cent->lerpOrigin[2] -= 12;
			break;
		}
		
	}
	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);
	
	ent.nonNormalizedAxes = qfalse;

	// if just respawned, slowly scale up
	msec = cg.time - cent->miscTime;
	if ( msec >= 0 && msec < ITEM_SCALEUP_TIME && cg_BobItems.value && !g_ForceViz.integer) {
		frac = (float)msec / ITEM_SCALEUP_TIME;
		VectorScale( ent.axis[0], frac, ent.axis[0] );
		VectorScale( ent.axis[1], frac, ent.axis[1] );
		VectorScale( ent.axis[2], frac, ent.axis[2] );
		ent.nonNormalizedAxes = qtrue;
	} else {
		frac = 1.0;
	}

	// items without glow textures need to keep a minimum light value
	// so they are always visible
	if ( ( item->giType == IT_WEAPON ) ||
		 ( item->giType == IT_ARMOR ) ) {
		ent.renderfx |= RF_MINLIGHT;
	}

	// increase the size of the weapons when they are presented as items
	// ### HENTAI ###
	if(item->giType == IT_ARMOR && ((cg_ScaleWeapons.integer != 150) || g_ForceViz.integer))
	{
		float scale = ((float) cg_ScaleWeapons.integer) / 150.0;
		vec3_t v;
		if(g_ForceViz.integer)
		{
			scale = 0.6;
			
		}
		if(cent->currentState.eFlags & EF_TELEPORT_BIT)
		{
			VectorScale( ent.axis[0], scale, ent.axis[0] );
			VectorScale( ent.axis[1], scale, ent.axis[1] );
			VectorScale( ent.axis[2], scale, ent.axis[2] );			
		}
		else
		{
			VectorScale( ent.axis[0], scale, v );
			VectorScale( ent.axis[1], -scale, ent.axis[1] );
			VectorScale( ent.axis[2], scale, ent.axis[0] );
			VectorCopy(v, ent.axis[2]);
			ent.origin[2] -= 10;
		}
			
		ent.nonNormalizedAxes = qtrue;
	}
	else if ( item->giType == IT_WEAPON && (cg_ScaleWeapons.integer != 100) && !g_ForceViz.integer) {		
		float wscale = ((float) cg_ScaleWeapons.integer) / 100.0;
		VectorScale( ent.axis[0], wscale, ent.axis[0] );
		VectorScale( ent.axis[1], wscale, ent.axis[1] );
		VectorScale( ent.axis[2], wscale, ent.axis[2] );
		ent.nonNormalizedAxes = qtrue;
	}
	// ### HENTAI ###

	// add to refresh list
	if(item->giType == IT_WEAPON)
	{
		if(cg_weapons[item->giTag].weaponSkin)
		{ // if it has a custom skin, then ammo is shown on the skin.
			ent.customSkin = cg_weapons[item->giTag].weaponSkin;
			ent.shaderRGBA[0] = (cent->currentState.powerups * 255.0 / item->quantity);
			ent.shaderRGBA[1] = (cent->currentState.powerups * 255.0 / item->quantity);
			ent.shaderRGBA[2] = (cent->currentState.powerups * 255.0 / item->quantity);
			ent.shaderRGBA[3] = (128 + cent->currentState.powerups * 127.0 / item->quantity);
					
		}
	}
	if(item->giType == IT_WEAPON && item->giTag == WP_BALL)
	{		
		refEntity_t	barrel;					
		float wscale = ((float) cg_ScaleWeapons.integer) / 100.0;
		if(g_ForceViz.integer)
			wscale = 1;
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( ent.lightingOrigin, barrel.lightingOrigin );
		
		barrel.hModel = cg_weapons[WP_BALL].barrelModel;
		barrel.axis[0][0] = 0.4 * wscale;
		barrel.axis[1][1] = 0.4 * wscale;
		barrel.axis[2][2] = 0.4 * wscale;
		

		//CG_PositionEntityOnTag( &barrel, &ent, cg_weapons[WP_MACHINEGUN].weaponModel, "tag_barrel" );
		VectorCopy(ent.origin, barrel.origin);
		trap_R_AddRefEntityToScene( &barrel );
		ent.axis[0][0] = 0.22 * sin(cg.time / 100.0) * wscale;
		ent.axis[0][1] = 0.22 * cos(cg.time / 100.0) * wscale;
		ent.axis[0][2] = 0;

		ent.axis[1][0] = 0.22 * cos(cg.time / 100.0) * wscale;
		ent.axis[1][1] = -0.22 * sin(cg.time / 100.0) * wscale;
		ent.axis[1][2] = 0;
		ent.axis[2][0] = 0;
		ent.axis[2][1] = 0;
		ent.axis[2][2] = 0.18 * wscale;

		ent.customShader = trap_R_RegisterShader("models/powerups/instant/regen");//.regenShader;
	}
	
	if(item->giType != IT_AMMO || cent->currentState.powerups == 0 || !(g_ForceViz.integer == 2 || cg_coolAmmo.integer) )
		trap_R_AddRefEntityToScene(&ent);
	
	if(item->giType == IT_WEAPON && cg_weapons[item->giTag].barrelModel)// && (item->giTag != WP_FLAMETHROWER || cent->currentState.powerups > 0))
	{ // ## Hentai - barrel is seperate model which must be drawn as well
		refEntity_t	barrel;					
		memset( &barrel, 0, sizeof( barrel ) );
		VectorCopy( ent.lightingOrigin, barrel.lightingOrigin );
		
		barrel.hModel = cg_weapons[item->giTag].barrelModel;
		if(item->giTag == WP_FLAMETHROWER)
		{
			barrel.customShader = cgs.media.armorShader;
			VectorScale(ent.axis[0], -0.6, barrel.axis[0]);
			VectorScale(ent.axis[1], 0.6, barrel.axis[1]);
			VectorScale(ent.axis[2], -0.6, barrel.axis[2]);
			VectorMA(ent.origin, 12, ent.axis[0], barrel.origin);
		}
		else
			CG_PositionEntityOnTag( &barrel, &ent, cg_weapons[item->giTag].weaponModel, "tag_barrel" );

		if(cg_weapons[item->giTag].weaponSkin)
		{ // if it has a custom skin, then ammo is shown on the skin.
			barrel.customSkin = cg_weapons[item->giTag].weaponSkin;
			barrel.shaderRGBA[0] = (cent->currentState.powerups * 255.0 / item->quantity);
			barrel.shaderRGBA[1] = (cent->currentState.powerups * 255.0 / item->quantity);
			barrel.shaderRGBA[2] = (cent->currentState.powerups * 255.0 / item->quantity);
			barrel.shaderRGBA[3] = (128 + cent->currentState.powerups * 127.0 / item->quantity);
					
		}
		trap_R_AddRefEntityToScene( &barrel );
	}
		
	// accompanying rings / spheres for powerups
	if ( !cg_simpleItems.integer ) 
	{
		vec3_t spinAngles;

		VectorClear( spinAngles );

		if((item->giType == IT_WEAPON || item->giType == IT_AMMO) && cent->currentState.powerups > 0 && (g_ForceViz.integer == 2 || cg_coolAmmo.integer) )
		{ // cent->currentState.powerups is ammo quantity
			int i;
			vec3_t forward, right;
			VectorCopy(ent.axis[0], forward);
			VectorCopy(ent.axis[1], right);
			right[2] = 0;
			VectorNormalize(right);
			if(item->giType == IT_WEAPON)
			{
				cent->lerpAngles[2] = 0;
				AnglesToAxis(cent->lerpAngles, ent.axis);
				switch(item->giTag)
				{
				default: // several don't show ammo
					ent.hModel = 0;
					break;
				case WP_MACHINEGUN:
					{
						vec3_t s;
						ent.hModel = cgs.media.machinegunBrassModel;
						VectorMA(cent->lerpOrigin, 8.3, forward, cent->lerpOrigin);
						cent->lerpOrigin[2] += 3;
						VectorScale( ent.axis[0], 0.6, s );
						VectorScale( ent.axis[1], 0.6, ent.axis[1] );
						VectorScale( ent.axis[2], -0.6, ent.axis[0] );
						VectorCopy(s, ent.axis[2]);
					}
					break;
				case WP_ROCKET_LAUNCHER:
					ent.hModel = trap_R_RegisterModel( "models/powerups/ammo/rocket.md3" );
					VectorMA(cent->lerpOrigin, 12, forward, cent->lerpOrigin);
					VectorMA(cent->lerpOrigin, -cent->currentState.powerups, right, cent->lerpOrigin);
					cent->lerpOrigin[2] -= 2;
					
					break;
				case WP_RAILGUN:
					ent.hModel = cgs.media.sphereFlashModel;//trap_R_RegisterModel( "models/powerups/health/medium_sphere.md3" );
					ent.customShader = cgs.media.armorShader;//trap_R_RegisterShader("models/weapons2/grapple/grapple06.tga");
					VectorMA(cent->lerpOrigin, 8, forward, cent->lerpOrigin);
					VectorMA(cent->lerpOrigin, -cent->currentState.powerups, right, cent->lerpOrigin);
					
					VectorScale( ent.axis[0], 0.225, ent.axis[0] );
					VectorScale( ent.axis[1], 0.1, ent.axis[1] );
					VectorScale( ent.axis[2], 0.1, ent.axis[2] );
					break;
				}

			}
			if(ent.hModel)
			{
				switch(item->giTag)
				{
				default:
					trap_R_AddRefEntityToScene( &ent );
					break;		
				case WP_MACHINEGUN:
					for(i = 0; i < cent->currentState.powerups; i++)
					{	
						if(i > 200) break;
						VectorCopy( cent->lerpOrigin, ent.origin);
						if(item->giType == IT_AMMO)
						{
							ent.origin[0] += cos(sqrt(i * 2.5)) * sqrt(i) * 0.6;
							ent.origin[1] += sin(sqrt(i * 2.5)) * sqrt(i) * 0.6;
						}
						else
						{
							if(i < 8)
								i = 8;
							VectorMA(ent.origin, i * 0.4, right, ent.origin);
							ent.origin[2] += 6 - sqrt(i * 9.0);//(cos(sqrt(i*5 + ent.origin[0] / 10)) - 1) * sqrt(i / 10.0 + 1) + 1.2 - i * 0.125;
							if(ent.origin[2] < groundz + 0.35 && tr.fraction < 1.0)
								ent.origin[2] = groundz + 0.35;
						}

						trap_R_AddRefEntityToScene( &ent );
					}						
					break;
				case WP_SHOTGUN:
					for(i = 0; i < cent->currentState.powerups; i++)
					{	
						VectorCopy( cent->lerpOrigin, ent.origin);
						ent.origin[0] += (i / 2) * 1.6 - cent->currentState.powerups * 0.8;
						ent.origin[1] += (i % 2) * 1.6 - 0.8;
						trap_R_AddRefEntityToScene( &ent );
					}						
					break;			
				case WP_RAILGUN:				
					for(i = 0; i < cent->currentState.powerups; i++)
					{
						if(i > 10) break;
						VectorCopy( cent->lerpOrigin, ent.origin);
						VectorMA(ent.origin, i * 2 - cent->currentState.powerups, right, ent.origin);
						trap_R_AddRefEntityToScene( &ent );
					}	
					break;
				case WP_GRENADE_LAUNCHER:
				
					for(i = 0; i < cent->currentState.powerups; i++)
					{
						VectorCopy( cent->lerpOrigin, ent.origin);
						ent.origin[1] += i * 2 - cent->currentState.powerups;
						trap_R_AddRefEntityToScene( &ent );
					}
					break;
				case WP_ROCKET_LAUNCHER:
					for(i = 0; i < cent->currentState.powerups; i++)
					{
						if(i > 10) break;
						VectorCopy( cent->lerpOrigin, ent.origin);
						VectorMA(ent.origin, i * 2 - cent->currentState.powerups, right, ent.origin);
						if((i < cent->currentState.powerups - 1) || (item->giType == IT_AMMO))
							trap_R_AddRefEntityToScene( &ent );
					}						
					break;
				}
			
			}				
			
		}				
		else if ( item->giType == IT_HEALTH || item->giType == IT_POWERUP )
		{
			if ( ( ent.hModel = cg_items[es->modelindex].models[1] ) != 0 )
			{
				if ( item->giType == IT_POWERUP )
				{
					ent.origin[2] += 12;
					spinAngles[1] = ( cg.time & 1023 ) * 360 / -1024.0f;
				}
				AnglesToAxis( spinAngles, ent.axis );
				
				// scale up if respawning
				if ( frac != 1.0 ) {
					VectorScale( ent.axis[0], frac, ent.axis[0] );
					VectorScale( ent.axis[1], frac, ent.axis[1] );
					VectorScale( ent.axis[2], frac, ent.axis[2] );
					ent.nonNormalizedAxes = qtrue;
				}
				trap_R_AddRefEntityToScene( &ent );
			}
		}
	}
}

//============================================================================

/*
===============
CG_Missile
===============
*/
static void CG_Missile( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;
	const weaponInfo_t		*weapon;
	float frac = ((float) cg_ScaleWeapons.integer) / 150.0;
	
	s1 = &cent->currentState;
	if ( s1->weapon > WP_NUM_WEAPONS ) {
		s1->weapon = 0;
	}
	weapon = &cg_weapons[s1->weapon];

	// calculate the axis
	VectorCopy( s1->angles, cent->lerpAngles);

	// add trails
	if ( weapon->missileTrailFunc ) 
	{
		weapon->missileTrailFunc( cent, weapon );
	}

	// add dynamic light
	if(cent->currentState.powerups & (1 << PW_QUAD))
	{
		int radius = weapon->missileDlight;
		if(radius < 200)
			radius = 100 + radius / 2;
		trap_R_AddLightToScene(cent->lerpOrigin, radius, 
			0, 0.5, 1.0 );
	}

	else if ( weapon->missileDlight ) {
		trap_R_AddLightToScene(cent->lerpOrigin, weapon->missileDlight, 
			weapon->missileDlightColor[0], weapon->missileDlightColor[1], weapon->missileDlightColor[2] );
	}

	// add missile sound
	if ( weapon->missileSound ) {
		vec3_t	velocity;

		BG_EvaluateTrajectoryDelta( &cent->currentState.pos, cg.time, velocity );

		trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, velocity, weapon->missileSound );
	}

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);
	if ( cent->currentState.weapon == WP_PLASMAGUN ) {
		ent.reType = RT_SPRITE;
		ent.radius = 16;
		ent.rotation = 0;
		ent.customShader = cgs.media.plasmaBallShader;
		trap_R_AddRefEntityToScene( &ent );
		return;
	}
	else if(cent->currentState.weapon == WP_ROCKET_LAUNCHER)
	{
		AngleVectors(cent->currentState.angles, ent.axis[0], ent.axis[1], ent.axis[2]);	
		// ### Hentai ### - FIXME: do client-side prediction of rocket acceleration?
		frac = 1.0;
	}
	else
	{
		if ( VectorNormalize2( s1->pos.trDelta, ent.axis[0] ) == 0 ) 
		{
			ent.axis[0][2] = 1;	
		}
		if(cent->currentState.weapon == WP_GRENADE_LAUNCHER)
			frac = 0.3;
		else
			frac = 1.0;
		
	}


	// flicker between two skins
	ent.skinNum = cg.clientFrame & 1;
	ent.hModel = weapon->missileModel;
	ent.renderfx = weapon->missileRenderfx | RF_NOSHADOW;

	
	// spin as it moves
	if ( s1->pos.trType != TR_STATIONARY ) {
		RotateAroundDirection( ent.axis, cg.time / 4 );
	} else {
		RotateAroundDirection( ent.axis, s1->time );
	}

	
	VectorScale( ent.axis[0], frac, ent.axis[0] );
	VectorScale( ent.axis[1], frac, ent.axis[1] );
	VectorScale( ent.axis[2], frac, ent.axis[2] );
		
	// add to refresh list, possibly with quad glow
	CG_AddRefEntityWithPowerups( &ent, s1->powerups, TEAM_FREE );
}

/*
===============
CG_Grapple

This is called when the grapple is sitting up against the wall
===============
*/
static void CG_Grapple( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;
	const weaponInfo_t		*weapon;

	s1 = &cent->currentState;
	if ( s1->weapon > WP_NUM_WEAPONS ) {
		s1->weapon = 0;
	}
	weapon = &cg_weapons[s1->weapon];

	// calculate the axis
	VectorCopy( s1->angles, cent->lerpAngles);

#if 0 // FIXME add grapple pull sound here..?
	// add missile sound
	if ( weapon->missileSound ) {
		trap_S_AddLoopingSound( cent->currentState.number, cent->lerpOrigin, vec3_origin, weapon->missileSound );
	}
#endif

	// Will draw cable if needed
	CG_GrappleTrail ( cent, weapon );

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);

	// flicker between two skins
	ent.skinNum = cg.clientFrame & 1;
	ent.hModel = weapon->missileModel;
	ent.renderfx = weapon->missileRenderfx | RF_NOSHADOW;

	// convert direction of travel into axis
	if ( VectorNormalize2( s1->pos.trDelta, ent.axis[0] ) == 0 ) {
		ent.axis[0][2] = 1;
	}

	trap_R_AddRefEntityToScene( &ent );
}

/*
===============
CG_Mover
===============
*/
static void CG_Mover( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( cent->lerpOrigin, ent.origin);
	VectorCopy( cent->lerpOrigin, ent.oldorigin);
	AnglesToAxis( cent->lerpAngles, ent.axis );

	ent.renderfx = RF_NOSHADOW;

	// flicker between two skins (FIXME?)
	ent.skinNum = ( cg.time >> 6 ) & 1;

	// get the model, either as a bmodel or a modelindex
	if ( s1->solid == SOLID_BMODEL ) {
		ent.hModel = cgs.inlineDrawModel[s1->modelindex];
	} else {
		ent.hModel = cgs.gameModels[s1->modelindex];
	}

	// add to refresh list
	trap_R_AddRefEntityToScene(&ent);

	// add the secondary model
	if ( s1->modelindex2 ) {
		ent.skinNum = 0;
		ent.hModel = cgs.gameModels[s1->modelindex2];
		trap_R_AddRefEntityToScene(&ent);
	}

}

/*
===============
CG_Beam

Also called as an event
===============
*/
void CG_Beam( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( s1->pos.trBase, ent.origin );
	VectorCopy( s1->origin2, ent.oldorigin );
	AxisClear( ent.axis );
	ent.reType = RT_BEAM;

	ent.renderfx = RF_NOSHADOW;

	// add to refresh list
	trap_R_AddRefEntityToScene(&ent);
}


/*
===============
CG_Portal
===============
*/
static void CG_Portal( centity_t *cent ) {
	refEntity_t			ent;
	entityState_t		*s1;

	s1 = &cent->currentState;

	// create the render entity
	memset (&ent, 0, sizeof(ent));
	VectorCopy( cent->lerpOrigin, ent.origin );
	VectorCopy( s1->origin2, ent.oldorigin );
	ByteToDir( s1->eventParm, ent.axis[0] );
	PerpendicularVector( ent.axis[1], ent.axis[0] );

	// negating this tends to get the directions like they want
	// we really should have a camera roll value
	VectorSubtract( vec3_origin, ent.axis[1], ent.axis[1] );

	CrossProduct( ent.axis[0], ent.axis[1], ent.axis[2] );
	ent.reType = RT_PORTALSURFACE;
	ent.frame = s1->frame;		// rotation speed
	ent.skinNum = s1->clientNum/256.0 * 360;	// roll offset

	// add to refresh list
	trap_R_AddRefEntityToScene(&ent);
}


/*
=========================
CG_AdjustPositionForMover

Also called by client movement prediction code
=========================
*/
void CG_AdjustPositionForMover( const vec3_t in, int moverNum, int fromTime, int toTime, vec3_t out ) {
	centity_t	*cent;
	vec3_t	oldOrigin, origin, deltaOrigin;
	vec3_t	oldAngles, angles, deltaAngles;

	if ( moverNum <= 0 || moverNum >= ENTITYNUM_MAX_NORMAL ) {
		VectorCopy( in, out );
		return;
	}

	cent = &cg_entities[ moverNum ];
	if ( cent->currentState.eType != ET_MOVER ) {
		VectorCopy( in, out );
		return;
	}

	BG_EvaluateTrajectory( &cent->currentState.pos, fromTime, oldOrigin );
	BG_EvaluateTrajectory( &cent->currentState.apos, fromTime, oldAngles );

	BG_EvaluateTrajectory( &cent->currentState.pos, toTime, origin );
	BG_EvaluateTrajectory( &cent->currentState.apos, toTime, angles );

	VectorSubtract( origin, oldOrigin, deltaOrigin );
	VectorSubtract( angles, oldAngles, deltaAngles );

	VectorAdd( in, deltaOrigin, out );

	// FIXME: origin change when on a rotating object
}


/*
=============================
CG_InterpolateEntityPosition
=============================
*/
static void CG_InterpolateEntityPosition( centity_t *cent ) {
	vec3_t		current, next;
	float		f;

	// it would be an internal error to find an entity that interpolates without
	// a snapshot ahead of the current one
	if ( cg.nextSnap == NULL ) {
		CG_Error( "CG_InterpoateEntityPosition: cg.nextSnap == NULL" );
	}

	f = cg.frameInterpolation;

	// this will linearize a sine or parabolic curve, but it is important
	// to not extrapolate player positions if more recent data is available
	BG_EvaluateTrajectory( &cent->currentState.pos, cg.snap->serverTime, current );
	BG_EvaluateTrajectory( &cent->nextState.pos, cg.nextSnap->serverTime, next );

	cent->lerpOrigin[0] = current[0] + f * ( next[0] - current[0] );
	cent->lerpOrigin[1] = current[1] + f * ( next[1] - current[1] );
	cent->lerpOrigin[2] = current[2] + f * ( next[2] - current[2] );

	BG_EvaluateTrajectory( &cent->currentState.apos, cg.snap->serverTime, current );
	BG_EvaluateTrajectory( &cent->nextState.apos, cg.nextSnap->serverTime, next );

	cent->lerpAngles[0] = LerpAngle( current[0], next[0], f );
	cent->lerpAngles[1] = LerpAngle( current[1], next[1], f );
	cent->lerpAngles[2] = LerpAngle( current[2], next[2], f );

}

/*
===============
CG_CalcEntityLerpPositions

===============
*/
static void CG_CalcEntityLerpPositions( centity_t *cent ) {
	if ( cent->interpolate && cent->currentState.pos.trType == TR_INTERPOLATE ) {
		CG_InterpolateEntityPosition( cent );
		return;
	}
	
	// just use the current frame and evaluate as best we can
	BG_EvaluateTrajectory( &cent->currentState.pos, cg.time, cent->lerpOrigin );
	BG_EvaluateTrajectory( &cent->currentState.apos, cg.time, cent->lerpAngles );

	// adjust for riding a mover if it wasn't rolled into the predicted
	// player state
	if ( cent != &cg.predictedPlayerEntity ) {
		CG_AdjustPositionForMover( cent->lerpOrigin, cent->currentState.groundEntityNum, 
		cg.snap->serverTime, cg.time, cent->lerpOrigin );
	}
}

int CG_ClientInfoForCorpse(int ClientNum)
{
	int c;
	clientInfo_t *ci = &cgs.clientinfo[ClientNum];
	for(c = 1; c <= corpses; c++)
	{
		if(corpseinfo[c].legsModel == ci->legsModel && corpseinfo[c].headModel == ci->headModel && corpseinfo[c].legsSkin == ci->legsSkin)
		{			
			//CG_Printf("matching corpse %d [%d(%d)/%d(%d)/%d(%d)]\n", c, ci->headModel, corpseinfo[c].headModel, ci->torsoModel, corpseinfo[c].torsoModel, ci->legsModel, corpseinfo[c].legsModel);
			return c;
		}
	}
	corpses++;
	memcpy(corpseinfo[corpses].animations, ci->animations, sizeof(ci->animations));
	corpseinfo[corpses].torsoModel = ci->torsoModel;
	corpseinfo[corpses].legsModel = ci->legsModel;
	corpseinfo[corpses].headModel = ci->headModel;
	corpseinfo[corpses].torsoSkin = ci->torsoSkin;
	corpseinfo[corpses].legsSkin = ci->legsSkin;
	corpseinfo[corpses].headSkin = ci->headSkin;
	//CG_Printf("creating corpse %d [%d(%d)/%d(%d)/%d(%d)]\n", corpses, ci->headModel, corpseinfo[corpses].headModel, ci->torsoModel, corpseinfo[corpses].torsoModel, ci->legsModel, corpseinfo[corpses].legsModel);
	return corpses;
	
	
}

/*
===============
CG_AddCEntity

===============
*/
static void CG_AddCEntity( centity_t *cent ) {
	// event-only entities will have been dealt with already
	if ( cent->currentState.eType >= ET_EVENTS ) {
		return;
	}

	// calculate the current origin
	CG_CalcEntityLerpPositions( cent );

	// add automatic effects
	CG_EntityEffects( cent );

	switch ( cent->currentState.eType ) {
	default:
		CG_Error( "Bad entity type: %i\n", cent->currentState.eType );
		break;
	case ET_INVISIBLE:
	case ET_PUSH_TRIGGER:
	case ET_TELEPORT_TRIGGER:
		break;
	case ET_GENERAL:
		CG_General( cent );
		break;
	case ET_CORPSE:
		if(cent->CorpseIndex <= 0 || cent->CorpseIndex > corpses )
		{ // just died, so create a new faux client
			cent->CorpseIndex = CG_ClientInfoForCorpse(cent->currentState.clientNum);
		}
		CG_Corpse( cent );
		break;
	case ET_PLAYER:
		CG_Player( cent );
		break;
	case ET_ITEM:
		CG_Item( cent );
		break;
	case ET_MISSILE:
		CG_Missile( cent );
		break;
	case ET_MOVER:
		CG_Mover( cent );
		break;
	case ET_BEAM:
		CG_Beam( cent );
		break;
	case ET_PORTAL:
		CG_Portal( cent );
		break;
	case ET_SPEAKER:
		CG_Speaker( cent );
		break;
	case ET_GRAPPLE:
		CG_Grapple( cent );
		break;
	}
}

/*
===============
CG_AddPacketEntities

===============
*/
void CG_AddPacketEntities( void ) {
	int					num;
	centity_t			*cent;
	playerState_t		*ps;

	// set cg.frameInterpolation
	if ( cg.nextSnap ) {
		int		delta;

		delta = (cg.nextSnap->serverTime - cg.snap->serverTime);
		if ( delta == 0 ) {
			cg.frameInterpolation = 0;
		} else {
			cg.frameInterpolation = (float)( cg.time - cg.snap->serverTime ) / delta;
		}
	} else {
		cg.frameInterpolation = 0;	// actually, it should never be used, because 
									// no entities should be marked as interpolating
	}

	// the auto-rotating items will all have the same axis
	cg.autoAngles[0] = 0;
	cg.autoAngles[1] = ( cg.time & 2047 ) * 360 / 2048.0;
	cg.autoAngles[2] = 0;

	cg.autoAnglesFast[0] = 0;
	cg.autoAnglesFast[1] = ( cg.time & 1023 ) * 360 / 1024.0f;
	cg.autoAnglesFast[2] = 0;

	AnglesToAxis( cg.autoAngles, cg.autoAxis );
	AnglesToAxis( cg.autoAnglesFast, cg.autoAxisFast );

	// generate and add the entity from the playerstate
	ps = &cg.predictedPlayerState;
	BG_PlayerStateToEntityState( ps, &cg.predictedPlayerEntity.currentState, cg.time, qfalse );
	CG_AddCEntity( &cg.predictedPlayerEntity );

	// lerp the non-predicted value for lightning gun origins
	CG_CalcEntityLerpPositions( &cg_entities[ cg.snap->ps.clientNum ] );

	// add each entity sent over by the server
	for ( num = 0 ; num < cg.snap->numEntities ; num++ ) {
		cent = &cg_entities[ cg.snap->entities[ num ].number ];
		CG_AddCEntity( cent );
	}
}

