// g_combat.c

#include "g_local.h"

/*
============
CanDamage

Returns true if the inflictor can directly damage the target.  Used for
explosions and melee attacks.
============
*/
qboolean CanDamage (edict_t *targ, edict_t *inflictor)
{
	vec3_t	dest;
	trace_t	trace;

// bmodels need special checking because their origin is 0,0,0
	if (targ->movetype == MOVETYPE_PUSH)
	{
		VectorAdd (targ->absmin, targ->absmax, dest);
		VectorScale (dest, 0.5, dest);
		trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
		if (trace.fraction == 1.0)
			return true;
		if (trace.ent == targ)
			return true;
		return false;
	}
	
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, targ->s.origin, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] += 15.0;
	dest[1] += 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] += 15.0;
	dest[1] -= 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] -= 15.0;
	dest[1] += 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] -= 15.0;
	dest[1] -= 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;


	return false;
}


/*
============
Killed
============
*/
void Killed (edict_t *targ, edict_t *inflictor, edict_t *attacker, int damage, vec3_t point)
{
	if (targ->health < -999)
		targ->health = -999;

	targ->enemy = attacker;

	if ((targ->svflags & SVF_MONSTER) && (targ->deadflag != DEAD_DEAD))
	{
//		targ->svflags |= SVF_DEADMONSTER;	// now treat as a different content type
		if (!(targ->monsterinfo.aiflags & AI_GOOD_GUY))
		{
			level.killed_monsters++;
			if (coop->value && attacker->client)
				attacker->client->resp.score++;
			// medics won't heal monsters that they kill themselves
			if (strcmp(attacker->classname, "monster_medic") == 0)
				targ->owner = attacker;
		}
	}

	if (targ->movetype == MOVETYPE_PUSH || targ->movetype == MOVETYPE_STOP || targ->movetype == MOVETYPE_NONE)
	{	// doors, triggers, etc
		targ->die (targ, inflictor, attacker, damage, point);
		return;
	}

	if ((targ->svflags & SVF_MONSTER) && (targ->deadflag != DEAD_DEAD))
	{
		targ->touch = NULL;
		monster_death_use (targ);
	}

	targ->die (targ, inflictor, attacker, damage, point);
}


/*
================
SpawnDamage
================
*/
void SpawnDamage (int type, vec3_t origin, vec3_t normal, int damage)
{
	if (damage > 255)
		damage = 255;
	gi.WriteByte (svc_temp_entity);
	gi.WriteByte (type);
//	gi.WriteByte (damage);
	gi.WritePosition (origin);
	gi.WriteDir (normal);
	gi.multicast (origin, MULTICAST_PVS);
}


/*
============
T_Damage

targ		entity that is being damaged
inflictor	entity that is causing the damage
attacker	entity that caused the inflictor to damage targ
	example: targ=monster, inflictor=rocket, attacker=player

dir			direction of the attack
point		point at which the damage is being inflicted
normal		normal vector from that point
damage		amount of damage being inflicted
knockback	force to be applied against targ as a result of the damage

dflags		these flags are used to control how T_Damage works
	DAMAGE_RADIUS			damage was indirect (from a nearby explosion)
	DAMAGE_NO_ARMOR			armor does not protect from this damage
	DAMAGE_ENERGY			damage is from an energy based weapon
	DAMAGE_NO_KNOCKBACK		do not affect velocity, just view angles
	DAMAGE_BULLET			damage is from a bullet (used for ricochets)
	DAMAGE_NO_PROTECTION	kills godmode, armor, everything
	DAMAGE_ARMOR_PIERCING	damage is armor-piercing (1/2 armor protection)
============
*/
static int CheckPowerArmor (edict_t *ent, vec3_t point, vec3_t normal, int damage, int dflags)
{
	gclient_t	*client;
	int			save;
	int			power_armor_type;
	int			index;
	int			damagePerCell;
	int			pa_te_type;
	int			power;
	int			power_used;

	if (!damage)
		return 0;

	client = ent->client;
	if(!client)
		return 0;
	if (dflags & DAMAGE_NO_ARMOR)
		return 0;

	power_armor_type = PowerArmorType (ent);
	if (power_armor_type != POWER_ARMOR_NONE)
	{
		index = ITEM_INDEX(FindItem("Cells"));			
		power = client->pers.inventory[index];
	}
	else
		return 0;

	if (power_armor_type == POWER_ARMOR_NONE)
		return 0;
	if (!power)
		return 0;

	if (power_armor_type == POWER_ARMOR_SCREEN)
	{
		vec3_t		vec;
		float		dot;
		vec3_t		forward;

		// only works if damage point is in front
		AngleVectors (ent->s.angles, forward, NULL, NULL);
		VectorSubtract (point, ent->s.origin, vec);
		VectorNormalize (vec);
		dot = DotProduct (vec, forward);
		if (dot <= 0.3)
			return 0;

		damagePerCell = 1;
		pa_te_type = TE_SCREEN_SPARKS;
		damage = damage / 3;
	}
	else
	{
		damagePerCell = 1; // power armor is weaker in CTF
		pa_te_type = TE_SHIELD_SPARKS;
		damage = (2 * damage) / 3;
	}

	save = power * damagePerCell;
	if (!save)
		return 0;
	if (save > damage)
		save = damage;

	SpawnDamage (pa_te_type, point, normal, save);
	ent->powerarmor_time = level.time + 0.2;

	power_used = save / damagePerCell;

	client->pers.inventory[index] -= power_used;
	return save;
}

static int CheckArmor (edict_t *ent, vec3_t point, vec3_t normal, int damage, int te_sparks, int dflags, int hitfactor)
{
	gclient_t	*client;
	int			save;
	int			index;
	gitem_t		*armor;

	if (!damage)
		return 0;

	client = ent->client;

	if (!client)
		return 0;

	if (dflags & DAMAGE_NO_ARMOR)
		return 0;

	index = ArmorIndex (ent);
	if (!index)
		return 0;

	armor = GetItemByIndex (index);
	if(hitfactor == -1)
	{
		if (dflags & DAMAGE_ENERGY)
			save = ceil(((gitem_armor_t *)armor->info)->energy_protection * damage);
		else if(((gitem_armor_t *)armor->info)->armor == ARMOR_JACKET)
			save = ceil(((gitem_armor_t *)armor->info)->normal_protection * damage * 0.5);
		else if(((gitem_armor_t *)armor->info)->armor == ARMOR_COMBAT)
			save = ceil(((gitem_armor_t *)armor->info)->normal_protection * damage * 0.6);
		else
			save = ceil(((gitem_armor_t *)armor->info)->normal_protection * damage);
	}
	else if(((gitem_armor_t *)armor->info)->protects_head && hitfactor < 0.25)
		return 0;
	else if(((gitem_armor_t *)armor->info)->protects_legs && hitfactor > 0.70)
		return 0;
	else if (dflags & DAMAGE_ENERGY)
		save = ceil(((gitem_armor_t *)armor->info)->energy_protection * damage);
	else
		save = ceil(((gitem_armor_t *)armor->info)->normal_protection * damage);
	if (save >= client->pers.inventory[index])
		save = client->pers.inventory[index];
	if (dflags & DAMAGE_ARMOR_PIERCING)
		save /= 2;

	if (!save)
		return 0;

	client->pers.inventory[index] -= save;
	SpawnDamage (te_sparks, point, normal, save);

	return save;
}

void M_ReactToDamage (edict_t *targ, edict_t *attacker)
{
	if (!(attacker->client) && !(attacker->svflags & SVF_MONSTER))
		return;

	if (attacker == targ || attacker == targ->enemy)
		return;

	// if we are a good guy monster and our attacker is a player
	// or another good guy, do not get mad at them
	if (targ->monsterinfo.aiflags & AI_GOOD_GUY)
	{
		if (attacker->client || (attacker->monsterinfo.aiflags & AI_GOOD_GUY))
			return;
	}

	// we now know that we are not both good guys

	// if attacker is a client, get mad at them because he's good and we're not
	if (attacker->client)
	{
		// this can only happen in coop (both new and old enemies are clients)
		// only switch if can't see the current enemy
		if (targ->enemy && targ->enemy->client)
		{
			if (visible(targ, targ->enemy))
			{
				targ->oldenemy = attacker;
				return;
			}
			targ->oldenemy = targ->enemy;
		}
		targ->enemy = attacker;
		if (!(targ->monsterinfo.aiflags & AI_DUCKED))
			FoundTarget (targ);
		return;
	}

	// it's the same base (walk/swim/fly) type and a different classname and it's not a tank
	// (they spray too much), get mad at them
	if (((targ->flags & (FL_FLY|FL_SWIM)) == (attacker->flags & (FL_FLY|FL_SWIM))) &&
		 (strcmp (targ->classname, attacker->classname) != 0) &&
		 (strcmp(attacker->classname, "monster_tank") != 0) &&
		 (strcmp(attacker->classname, "monster_supertank") != 0) &&
		 (strcmp(attacker->classname, "monster_makron") != 0) &&
		 (strcmp(attacker->classname, "monster_jorg") != 0) )
	{
		if (targ->enemy)
			if (targ->enemy->client)
				targ->oldenemy = targ->enemy;
		targ->enemy = attacker;
		if (!(targ->monsterinfo.aiflags & AI_DUCKED))
			FoundTarget (targ);
	}
	else
	// otherwise get mad at whoever they are mad at (help our buddy)
	{
		if (targ->enemy)
			if (targ->enemy->client)
				targ->oldenemy = targ->enemy;
		targ->enemy = attacker->enemy;
		FoundTarget (targ);
	}
}

qboolean CheckTeamDamage (edict_t *targ, edict_t *attacker)
{
//ZOID	
	if (ctf->value && targ->client && attacker->client)
		if (targ->client->resp.ctf_team == attacker->client->resp.ctf_team &&
			targ != attacker)
			return true;
//ZOID

		//FIXME make the next line real and uncomment this block
		// if ((ability to damage a teammate == OFF) && (targ's team == attacker's team))
	return false;
}

void T_Damage (edict_t *targ, edict_t *inflictor, edict_t *attacker, vec3_t dir, vec3_t point, vec3_t normal, int damage, int knockback, int dflags, int mod)
{
	gclient_t	*client;
	int			take;
	int			save;
	int			asave;
	int			psave;
	int			te_sparks;
	float		hitfactor, hitz;

	if (!targ->takedamage)
		return;
	if(inflictor->owner)
		inflictor = inflictor->owner;
	// friendly fire avoidance
	// if enabled you can't hurt teammates (but you can hurt yourself)
	// knockback still occurs
	if ((targ != attacker) && ((deathmatch->value && ((int)(dmflags->value) & (DF_MODELTEAMS | DF_SKINTEAMS))) || coop->value))
	{
		if (OnSameTeam (targ, attacker))
		{
			//if ((int)(dmflags->value) & DF_NO_FRIENDLY_FIRE)
			//	damage = 0;
			//else
				mod |= MOD_FRIENDLY_FIRE;
		}
	}
	meansOfDeath = mod;

	// easy mode takes half damage
	if (skill->value == 0 && deathmatch->value == 0 && targ->client)
	{
		damage *= 0.5;
		if (!damage)
			damage = 1;
	}

	client = targ->client;

	if (dflags & DAMAGE_BULLET)
		te_sparks = TE_BULLET_SPARKS;
	else
		te_sparks = TE_SPARKS;

	VectorNormalize(dir);

// bonus damage for suprising a monster
	if (!(dflags & DAMAGE_RADIUS) && (targ->svflags & SVF_MONSTER) && (attacker->client) && (!targ->enemy) && (targ->health > 0))
		damage *= 2;

	if (targ->flags & FL_NO_KNOCKBACK)
		knockback = 0;

// figure momentum add
	if (!(dflags & DAMAGE_NO_KNOCKBACK))
	{
		if ((knockback) && (targ->movetype != MOVETYPE_NONE) && (targ->movetype != MOVETYPE_PUSH) && (targ->movetype != MOVETYPE_STOP))
		{
			vec3_t	kvel;
			float	mass;

			if (targ->mass < 1)
				mass = 1;
			else
				mass = targ->mass;

			if (targ->client  && attacker == targ)
				VectorScale (dir, 1600.0 * (float)knockback / mass, kvel);	// the rocket jump hack...
			else
				VectorScale (dir, 500.0 * (float)knockback / mass, kvel);

			VectorAdd (targ->velocity, kvel, targ->velocity);
			if(targ->groundentity)
			{
				targ->groundentity = NULL;
				targ->s.origin[2]++;
				targ->velocity[2] = fabs(targ->velocity[2]);
			}
		}
	}

	if(mod == MOD_FALLING)
	{
		hitz = 1;
		hitfactor = 1;
	}
	else if(!(dflags & DAMAGE_RADIUS) && (dir) && (point) && (targ->client != NULL))
	{
		vec3_t offset;
		float hitoffset, dx, dy, dlen;
	
		offset[0] = targ->s.origin[0] - point[0];
		offset[1] = targ->s.origin[1] - point[1];
		offset[2] = targ->s.origin[2] + targ->viewheight - point[2];

		dlen = sqrt(dir[0] * dir[0] + dir[1] * dir[1]);
		dx = dir[0] / dlen;
		dy = dir[1] / dlen;
		dlen = offset[0] * dir[0] + offset[1] * dir[1];
		offset[0] -= dx * dlen;
		offset[1] -= dy * dlen;
				
		offset[0] = offset[0] / targ->client->width;
		offset[1] = offset[1] / targ->client->width;
		offset[2] = 2 * offset[2] / targ->client->height;
		hitz = offset[2];
		hitoffset = VectorLength(offset);
		if(hitoffset > 2 || hitoffset < 0)
		{
			hitfactor = 0;
			hitz = -1;
		}
		else
		//gi.dprintf ("hit offset: %f %f %f\n", offset[0], offset[1], offset[2]);				
			hitfactor = 1 / (hitoffset + 0.1);
		//gi.dprintf ("Damage multiplier: %f\n", hitfactor);
	}
	else
	{
		hitfactor = 0;
		hitz = -1;
	}
		
	take = damage;
	save = 0;

	// check for godmode
	if ( (targ->flags & FL_GODMODE) && !(dflags & DAMAGE_NO_PROTECTION) )
	{
		take = 0;
		save = damage;
		SpawnDamage (te_sparks, point, normal, save);
	}

	psave = CheckPowerArmor (targ, point, normal, take, dflags);
	take -= psave;

	asave = CheckArmor (targ, point, normal, take, te_sparks, dflags, hitz);
	take -= asave;

	//treat cheat/powerup savings the same as armor
	asave += save;

//ZOID
//resistance tech
//	take = CTFApplyResistance(targ, take);
//ZOID

	// ### Hentai ### BEGIN
	
	if(hitfactor > 0)
	{
		take *= hitfactor;
		if((targ->client != NULL) && targ->client->canbleed)
		{
			if(dflags & DAMAGE_NO_BLEED)
			{
				if(dflags & DAMAGE_ENERGY)
					targ->client->bleeding *= 0.9;
			}
			else
				targ->client->bleeding += (take * hitfactor) / targ->max_health;			
			targ->enemy = attacker;
		}
		//gi.dprintf("%s hit in area [%f]\n", targ->client->pers.netname, hitfactor);
		if((targ->client != NULL) && hitz > 0.70 && take > 0)
		{
			targ->client->leg_damage += take * 10.0 / targ->max_health;
			//gi.dprintf("%s gimped [%f]\n", targ->client->pers.netname, targ->client->leg_damage);
		}
			
	}
	else
	{
	}
	//	gi.dprintf("%s hit\n", targ->client->pers.netname);	

	// ### Hentai ### END
	// team damage avoidance
	//if (!(dflags & DAMAGE_NO_PROTECTION) && CheckTeamDamage (targ, attacker))
	//	return;

//ZOID
	CTFCheckHurtCarrier(targ, attacker);
//ZOID

// do the damage
	if (take)
	{
		if(mod == MOD_WATER)
		{
					gi.WriteByte (svc_temp_entity);
					gi.WriteByte (TE_SPLASH);
					gi.WriteByte (8);
					gi.WritePosition (point);
					gi.WriteDir (normal);
					gi.WriteByte (SPLASH_BLUE_WATER);
					gi.multicast (point, MULTICAST_PVS);

		//	SpawnDamage (SPLASH_WATER, point, normal, take);
		}
		else if(targ->client && !targ->client->canbleed)
			SpawnDamage (te_sparks, point, normal, take);
		else if ((targ->svflags & SVF_MONSTER) || (client))
			SpawnDamage (TE_BLOOD, point, normal, take);
		else
			SpawnDamage (te_sparks, point, normal, take);


		targ->health -= take;
		if(hitz > 0.75)
			targ->max_health -= floor(hitfactor * 2);
		if(targ->max_health < targ->health)
			targ->health = targ->max_health;
		if (targ->health <= 0)
		{
			if ((targ->svflags & SVF_MONSTER) || (client))
				targ->flags |= FL_NO_KNOCKBACK;
			Killed (targ, inflictor, attacker, take, point);
			return;
		}
	}

	if (targ->svflags & SVF_MONSTER)
	{
		M_ReactToDamage (targ, attacker);
		if (!(targ->monsterinfo.aiflags & AI_DUCKED) && (take))
		{
			targ->pain (targ, attacker, knockback, take);
			// nightmare mode monsters don't go into pain frames often
			if (skill->value == 3)
				targ->pain_debounce_time = level.time + 5;
		}
	}
	else if (client)
	{
		if (!(targ->flags & FL_GODMODE) && (take))
			targ->pain (targ, attacker, knockback, take);
	}
	else if (take)
	{
		if (targ->pain)
			targ->pain (targ, attacker, knockback, take);
	}

	// add to the damage inflicted on a player this frame
	// the total will be turned into screen blends and view angle kicks
	// at the end of the frame
	if (client)
	{
		client->damage_parmor += psave;
		client->damage_armor += asave;
		client->damage_blood += take;
		client->damage_knockback += knockback;
		VectorCopy (point, client->damage_from);
	}
}


/*
============
T_RadiusDamage
============
*/
void T_RadiusDamage (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);		
		points = damage * (1 - VectorLength(v) / radius);
		//if (ent == attacker)
		//	points = points * 0.5;
		if (points > 0)
		{
			if (CanDamage (ent, inflictor))
			{
				vec3_t forward;
				float dot;
				VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
				T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);

				if(ent->client)
				{
					AngleVectors (ent->client->v_angle, forward, NULL, NULL);
				

					VectorNormalize (dir);
					
					if ((dot = DotProduct (dir, forward)) < -0.1) // was facing blast
					{
						if(ent->client->pers.inventory[ITEM_INDEX(FindItem("Imaging Goggles"))] == 0)
							ent->client->blinded -= dot * points / 5;
						ent->client->bonus_alpha = 1;
					}
					if(ent->client->blinded > 0.95)
						ent->client->blinded = 0.95;
				}
			}
		}
	}
}

void ConcussionThink(edict_t *ent)
{
	vec3_t dest, mins, maxs;
	dpnt_t *pnt = ent->concussion, *last = NULL;
	int bounces, i;	
		
	mins[0] = -ent->dmg_radius * ent->dmg_radius;
	mins[1] = -ent->dmg_radius * ent->dmg_radius;
	mins[2] = -ent->dmg_radius * ent->dmg_radius;

	maxs[0] = ent->dmg_radius * ent->dmg_radius;
	maxs[1] = ent->dmg_radius * ent->dmg_radius;
	maxs[2] = ent->dmg_radius * ent->dmg_radius;
	while(pnt != NULL)
	{	
		trace_t trace;
		qboolean remove = false;
		
		VectorAdd(pnt->org, pnt->dir, dest);
		trace = gi.trace (pnt->org, mins, maxs, dest, NULL, MASK_SHOT);
		bounces = 0;
		while(trace.fraction < 1 && bounces < 10)
		{
			/*
			gi.WriteByte (svc_temp_entity);
			gi.WriteByte (TE_BFG_LASER);
			gi.WritePosition (pnt->org);
			gi.WritePosition (trace.endpos);
			gi.multicast (ent->s.origin, MULTICAST_PHS);
			*/
			bounces++;
			if(trace.ent->takedamage) 
			{
				T_Damage(trace.ent, ent, ent->owner, pnt->dir, trace.endpos, trace.plane.normal, ent->dmg, ent->dmg - floor(trace.fraction * 5), DAMAGE_RADIUS, ent->flags);
				remove = true;
				break;
			}
			else if((trace.surface != NULL) && (trace.surface->flags & SURF_SKY))
			{
				remove = true;
				break;
			}
			else if(trace.surface != NULL)
			{
				float backoff;					
				backoff =  DotProduct (pnt->dir, trace.plane.normal) * 2;
				VectorMA(pnt->org, trace.fraction, pnt->dir, pnt->org);

				for (i=0 ; i<3 ; i++)
					pnt->dir[i] = pnt->dir[i] - trace.plane.normal[i]*backoff;
				
				VectorMA(pnt->org, trace.fraction, pnt->dir, dest);		
					trace = gi.trace (pnt->org, mins, maxs, dest, NULL, MASK_SHOT);
	
			}
			else
			{
				remove = true;
				break;
			}
		}
		if(remove)
		{
			ent->count--;
			if(last)
			{
				last->next = pnt->next;
				free(pnt);
				pnt = last->next;
			}
			else
			{
				ent->concussion = pnt->next;
				free(pnt);
				pnt = ent->concussion;
			}
		}
		else
		{
			VectorCopy(pnt->org, trace.endpos);
			last = pnt;
			pnt = pnt->next;
		}
	}
	ent->dmg -= 5;
	if(ent->dmg <= 0 || ent->concussion == NULL || ent->count < 1)
	{		
		G_FreeEdict(ent);
		return;
	}
	ent->dmg_radius++;		
	ent->nextthink = level.time + FRAMETIME;

}

void T_BecomeExplosion(edict_t *ent, int mod)
{
	int i;
	dpnt_t *pnt;	
	
	if(ent->dmg > 0 && ent->dmg < 150)
		ent->count = ent->dmg;
	else
		ent->count = 150;

	ent->dmg = 15;
	ent->dmg_radius = 0;
	ent->flags = mod;
	ent->s.modelindex = 0;
	ent->solid = 0;
	ent->concussion = pnt = malloc(sizeof(dpnt_t));
	for(i = 1; i < ent->count; i++)
	{
		VectorCopy(ent->s.origin, pnt->org);		
		pnt->dir[0] = crandom();
		pnt->dir[1] = crandom();
		pnt->dir[2] = crandom();
		VectorNormalize(pnt->dir);
		VectorMA(ent->velocity, 250, pnt->dir, pnt->dir);
		pnt->next = malloc(sizeof(dpnt_t));
		pnt = pnt->next;
	}
	VectorCopy(ent->s.origin, pnt->org);		
	pnt->dir[0] = crandom();
	pnt->dir[1] = crandom();
	pnt->dir[2] = crandom();
	VectorNormalize(pnt->dir);
	VectorMA(ent->velocity, 2500, pnt->dir, pnt->dir);
	VectorScale(pnt->dir, 0.1, pnt->dir);
	pnt->next = NULL;
	
	ent->think = ConcussionThink;
	ent->nextthink = level.time + FRAMETIME;
}