#include "main.h"

int g_multilayeredClutChoice = 0;

static BYTE *g_fileBuffer = 0;
static int g_bufferLen = 0;
static int g_fileHandle = -1;

static const int g_minTimSize = 64;
static const int g_maxTimSize = 4000000;
static const int g_typeFilter = -1;

//get the clut entry given its index and obtain our pixel color from it
static BYTE *GetRGBFromCLUT(BYTE clutIndex, tim_t *tim)
{
	static BYTE rgba[4];
	BYTE r, g, b, t;
	timCLUT_t *c = tim->clut;
	WORD *clutEntry;

	//step over the header data
	c++;

	//assign the pointer to the first clut entry plus offset by the clut index
	if (tim->clut->height > 1)
	{ //special crazy type with a > 1 height clut
		clutEntry = (WORD *)c + g_multilayeredClutChoice + clutIndex;//(15-clutIndex);
	}
	else
	{
		clutEntry = (WORD *)c + clutIndex;
	}

	r = (*clutEntry >> 0) & 31; //extract 5 bits from 0
	g = (*clutEntry >> 5) & 31; //extract 5 bits from 5
	b = (*clutEntry >> 10) & 31; //extract 5 bits from 10
	t = (*clutEntry >> 15) & 1; //extract 1 bit from 15

	//1 bit from 15 is the transparency bit.
	rgba[0] = r*GAMMA_CORRECTION;
	rgba[1] = g*GAMMA_CORRECTION;
	rgba[2] = b*GAMMA_CORRECTION;

	if (!t && (!r && !g && !b))
	{ //completely see-through
		rgba[3] = 0;
	}
	else if (t)
	{ //supposed to be partially transparent?
		rgba[3] = 127;
	}
	else
	{ //opaque
		rgba[3] = 255;
	}

	return rgba;
}

//create and fill out a pixelImage based on the contents of a tim
static pixelImage_t *GeneratePixelImageFromTIM(tim_t *tim)
{
	static pixelImage_t *pImg;
	rgbaPixel_t *pix;
	BYTE *rgba;
	int i = 0;
	int x = tim->width, y = tim->height;
	int pixels = (x*y)*sizeof(rgbaPixel_t);
	int bSize = sizeof(pixelImage_t)-sizeof(rgbaPixel_t);
	timPixels_t *p;

	pImg = (pixelImage_t *)malloc(bSize+pixels);

	memset(pImg, 0, bSize+pixels);

	//info about our image
	pImg->i.width = x;
	pImg->i.height = y;

	if (tim->pixelMode == TIM_PM_4BIT)
	{ //4-bit stores 4 pixels per "chunk", each "pixel" being an index to a CLUT entry
		tim4BitPixelEntry_t *pixelData;

		p = tim->pixels;

		//step over the header data
		p++;

		//the base of the pixel data
		pixelData = (tim4BitPixelEntry_t *)p;

		pix = &pImg->p;
		while (i < (x*y))
		{ //iterate through the pixels and place them into the pixel "array"
			rgba = GetRGBFromCLUT(pixelData->p1, tim);
			pix->r = rgba[0];
			pix->g = rgba[1];
			pix->b = rgba[2];
			pix->a = rgba[3];
			pix++;

			rgba = GetRGBFromCLUT(pixelData->p2, tim);
			pix->r = rgba[0];
			pix->g = rgba[1];
			pix->b = rgba[2];
			pix->a = rgba[3];
			pix++;

			rgba = GetRGBFromCLUT(pixelData->p3, tim);
			pix->r = rgba[0];
			pix->g = rgba[1];
			pix->b = rgba[2];
			pix->a = rgba[3];
			pix++;

			rgba = GetRGBFromCLUT(pixelData->p4, tim);
			pix->r = rgba[0];
			pix->g = rgba[1];
			pix->b = rgba[2];
			pix->a = rgba[3];
			pix++;
			
			i += 4;
			pixelData++;
		}
	}
	else if (tim->pixelMode == TIM_PM_8BIT)
	{ //similar to 4-bit, except there are 2 pixels per chunk and 256 possible clut entries instead of 16
		tim8BitPixelEntry_t *pixelData;

		p = tim->pixels;

		//step over the header data
		p++;

		//the base of the pixel data
		pixelData = (tim8BitPixelEntry_t *)p;

		pix = &pImg->p;
		while (i < (x*y))
		{ //iterate through the pixels and place them into the pixel "array"
			rgba = GetRGBFromCLUT(pixelData->p1, tim);
			pix->r = rgba[0];
			pix->g = rgba[1];
			pix->b = rgba[2];
			pix->a = rgba[3];
			pix++;

			rgba = GetRGBFromCLUT(pixelData->p2, tim);
			pix->r = rgba[0];
			pix->g = rgba[1];
			pix->b = rgba[2];
			pix->a = rgba[3];
			pix++;
			
			i += 2;
			pixelData++;
		}
	}
	else if (tim->pixelMode == TIM_PM_15BIT)
	{ //in direct 15-bit, each pixel is a 16-bit value with rgb in the first 5 bits and, I guess, an additional 1-bit transparency bit
		WORD *pixelData;
		BYTE r, g, b, t;

		p = tim->pixels;

		//step over the header data
		p++;

		//the base of the pixel data
		pixelData = (WORD *)p;

		pix = &pImg->p;
		while (i < (x*y))
		{ //iterate through the pixels and place them into the pixel "array"
			r = (*pixelData >> 0) & 31; //extract 5 bits from 0
			g = (*pixelData >> 5) & 31; //extract 5 bits from 5
			b = (*pixelData >> 10) & 31; //extract 5 bits from 10
			t = (*pixelData >> 15) & 1; //extract 1 bit from 15

			//Assuming I need to "gamma correct" still since we only have 5 bits per pixel
			pix->r = r*GAMMA_CORRECTION;
			pix->g = g*GAMMA_CORRECTION;
			pix->b = b*GAMMA_CORRECTION;
			if (!t && (!r && !g && !b))
			{ //completely see-through
				pix->a = 0;
			}
			else if (t)
			{ //opaque
				pix->a = 255;
			}
			else
			{ //supposed to be partially transparent?
				pix->a = 127;
			}
			pix++;
			
			i++;
			pixelData++;
		}
	}
	else if (tim->pixelMode == TIM_PM_24BIT)
	{ //in 24-bit direct, each pixel chunk is 6 bytes worth of data and contains 2 pixels each with its own set of 8 bit rgb components
		tim24BitPixelEntry_t *pixelData;

		p = tim->pixels;

		//step over the header data
		p++;

		//the base of the pixel data
		pixelData = (tim24BitPixelEntry_t *)p;

		pix = &pImg->p;
		while (i < (x*y))
		{ //iterate through the pixels and place them into the pixel "array"
			//I have not tested 24-bit mode, but I make the assumption that "gamma correction" is not needed,
			//since each rgb value is 8 bits and thus have the ability to correspond with the pixel standard
			//of 0-255
			pix->r = pixelData->r0;
			pix->g = pixelData->g0;
			pix->b = pixelData->b0;
			pix->a = 255;
			pix++;
			pix->r = pixelData->r1;
			pix->g = pixelData->g1;
			pix->b = pixelData->b1;
			pix->a = 255;
			pix++;
			
			i += 2;
			pixelData++;
		}
	}
	else //mixed mode? I can't find a tim in this format to mess with.
	{
		Game_Print("Can't create image - tim in unsupported format.\n");
		free(pImg);
		return NULL;
	}

	return pImg;
}

//opens whatever fileName is and goes through it looking for tims
pixelImage_t *GetTIMFromBuffer(int byteOffset, int readLen)
{
	bool localCRC = false;
tryAgain:
	DWORD dwTotal;
	int i = 0;
	unsigned int crc;
	int realReadLen;
	int crcReadPoint = 0;

	if (g_useCRC || localCRC)
	{
		crc = GetCRCForOffset(byteOffset);

		realReadLen = readLen;
		readLen = g_maxTimSize*2;
	}

	if (g_fileHandle == -1)
	{
		Game_Print("GetTIMFromBuffer call with no file buffer\n");
		return NULL;
	}

	if (g_fileBuffer)
	{ //free if we have one in memory from before
		free(g_fileBuffer);
	}

	g_fileBuffer = (BYTE *)malloc(readLen);

	if ((g_useCRC || localCRC) && crc)
	{ //if checking by crc set the initial read point to 0
		_lseek(g_fileHandle, crcReadPoint, SEEK_SET);
	}
	else
	{ //set the "file pointer" x bytes into the file from the beginning
		_lseek(g_fileHandle, byteOffset, SEEK_SET);
	}
	//read x bytes from the offset
	_read(g_fileHandle, g_fileBuffer, readLen);

	while (crcReadPoint < g_bufferLen)
	{ //this while is only for crc reading, otherwise it will just break at the end
		if ((i+g_minTimSize) < readLen &&
		//	g_fileBuffer[i] == 0x10 && //id always starts with 0x10
			g_fileBuffer[i+1] == 0x00 && //version is always 0x00
			g_fileBuffer[i+2] == 0x00 && //reserved byte 1
			g_fileBuffer[i+3] == 0x00 && //reserved byte 2
			g_fileBuffer[i+5] == 0x00 && //reserved byte 1 after flag offset
			g_fileBuffer[i+6] == 0x00 && //reserved byte 2 after flag offset
			g_fileBuffer[i+7] == 0x00) //reserved byte 3 after flag offset
		{
			BYTE flag = g_fileBuffer[i+4];
			int valid = 0;

			if (flag)
			{ //should always at least be non-0, you can never have a 4-bit CLUT with no CLUT section
				timFLAG_t *timFlag = (timFLAG_t *)&g_fileBuffer[i+4];
				DWORD bytesForCLUT = 0;
				WORD bits;

				if (timFlag->pixelMode <= TIM_PM_MIXED &&
					timFlag->pixelMode >= 0 &&
					(g_typeFilter == -1 || timFlag->pixelMode == g_typeFilter))
				{
					if ((timFlag->pixelMode < TIM_PM_15BIT ||
						timFlag->pixelMode == TIM_PM_MIXED) &&
						timFlag->hasCLUT)
					{ //4-bit clut, 8-bit clut, or mixed, and has clut section
						//a CLUT entry is 2 bytes long (16 bits).
						//For 4-bit mode there are 16 entries, for 8-bit mode there are 256 entries.
						if (timFlag->pixelMode == TIM_PM_4BIT)
						{ //4-bit
							bytesForCLUT = 16*2; //a 4-bit tim contains 16 clut entries at 2 bytes (16 bits) per entry
							bits = 4;
						}
						else
						{ //8-bit
							bytesForCLUT = 256*2; //an 8-bit tim contains 256 clut entries at 2 bytes (16 bits) per entry
							bits = 8;
						}
						bytesForCLUT += sizeof(timCLUT_t); //clut size includes the clut data itself
						valid = 1;
					}
					else if ((timFlag->pixelMode == TIM_PM_15BIT ||
						timFlag->pixelMode == TIM_PM_24BIT) &&
						!timFlag->hasCLUT)
					{ //direct, no CLUT
						if (timFlag->pixelMode == TIM_PM_15BIT)
						{
							bits = 15;
						}
						else
						{
							bits = 24;
						}
						valid = 1;
					}
				}

				if (valid)
				{
					DWORD *clutLen = NULL;
					DWORD *dw = NULL;
					DWORD pixelStart;

					if (timFlag->hasCLUT)
					{ //get the CLUT block size, the total size is it + the pixel size
						clutLen = (DWORD *)&g_fileBuffer[i+8];

						if (*clutLen == bytesForCLUT)
						{ //the numbers match up
							pixelStart = i + 8 + *clutLen;
							dw = (DWORD *)&g_fileBuffer[pixelStart];
						}
						else
						{ //check if it's a multi-clut (> 1 height)
							timCLUT_t *clut = (timCLUT_t *)&g_fileBuffer[i+8];

							//2 bytes per clut entry
							bytesForCLUT = (clut->width*2)*clut->height;
		
							if (timFlag->pixelMode == TIM_PM_4BIT)
							{ //4-bit
								bits = 4;
							}
							else
							{ //8-bit
								bits = 8;
							}
							bytesForCLUT += sizeof(timCLUT_t); //clut size includes the clut data itself

							if (clut->length == bytesForCLUT)
							{ //ok then
								pixelStart = i + 8 + *clutLen;
								dw = (DWORD *)&g_fileBuffer[pixelStart];
							}
						}
					}
					else
					{ //just get the pixel size
						pixelStart = i + 8;
						dw = (DWORD *)&g_fileBuffer[pixelStart];
					}

					if (dw)
					{
						WORD *width, *height;
						WORD realWidth;

						//Get the total file size based on the length of the CLUT (if applicable)
						//and the pixel data.
						if (timFlag->hasCLUT && clutLen)
						{
							dwTotal = 8 + *dw + *clutLen;
						}
						else
						{
							dwTotal = 8 + *dw;
						}

						//start of pixel data, plus 4 bytes for data size, plus 4 for x/y coords
						width = (WORD *)&g_fileBuffer[pixelStart+4+4];

						//the actual image width is relative to the pixel mode
						realWidth = (*width)*(16/bits);


						//start of pixel data, plus 4 bytes for data size, plus 4 for x/y coords, plus 2 for over width
						height = (WORD *)&g_fileBuffer[pixelStart+4+4+2];

						if (realWidth && *height)
						{
							int fSize = i + dwTotal;

							if ((g_useCRC || localCRC) && crc)
							{ //check the crc
								if (dwTotal >= (DWORD)g_minTimSize &&
									dwTotal <= (DWORD)g_maxTimSize)
								{
									unsigned long potentialCRC = CRC_CalcChecksum(&g_fileBuffer[i], dwTotal);
									if (potentialCRC != crc)
									{
										i += dwTotal;
										goto onWithLoop;
									}
								}
								else
								{
									i++;
									goto onWithLoop;
								}
							}

							if (fSize <= /*g_bufferLen*/readLen)
							{ //alright, since it all checks out, we'll go ahead and make a tim object and cast the pointers over the appropriate buffer spots.
								tim_t tim;

								tim.bpp = bits;
								tim.pixelMode = timFlag->pixelMode;
								tim.height = *height;
								tim.width = realWidth;
								if (timFlag->hasCLUT)
								{
									tim.clut = (timCLUT_t *)&g_fileBuffer[i+8];
								}
								else
								{
									tim.clut = NULL;
								}
								tim.pixels = (timPixels_t *)&g_fileBuffer[pixelStart];

								if (dwTotal >= (DWORD)g_minTimSize &&
									dwTotal <= (DWORD)g_maxTimSize)
								{
									if ((g_useCRC || localCRC) && crc)
									{ //this means we found the right guy
										SetOffsetForCRC(crc, crcReadPoint+i);
									}
									return GeneratePixelImageFromTIM(&tim);
								}
							}
						}
					}
				}
			}
		}

		if ((!g_useCRC && !localCRC) || !crc)
		{ //wel den
			break;
		}

		i++;
onWithLoop:
		if (i >= realReadLen)
		{
			i = 0;
			crcReadPoint += realReadLen;
			_lseek(g_fileHandle, crcReadPoint, SEEK_SET);
			//read x bytes from the offset
			_read(g_fileHandle, g_fileBuffer, readLen);
		}
	}

	if (!localCRC)
	{ //try again then.
		localCRC = true;
		goto tryAgain;
	}
	return NULL;
}

//opens a file and puts it into the buffer for access
int CreateTIMBuffer(const char *fileName)
{
	g_fileHandle = _open(fileName, _O_RDONLY|_O_BINARY, _S_IREAD);

	if (g_fileHandle == -1)
	{
		Game_Print("Open for %s failed\n", fileName);
		return 0;
	}

	g_bufferLen = _filelength(g_fileHandle);

	return 1;
}

//clear buffer memory and close file
void FreeTIMBuffer(void)
{
	if (g_fileBuffer)
	{
		free(g_fileBuffer);
		g_fileBuffer = 0;
	}

	if (g_fileHandle != -1)
	{
		_close(g_fileHandle);
		g_fileHandle = -1;
	}
}
