#include "main.h"

#define _NO_CARD_CARVING

drawCard_t *g_drawCards = NULL;
drawObject_t *g_drawObjects = NULL;
int g_glTexNum = 1;
int g_cardBackTex = -1;
int g_linearTex = 1;
void GL_Ortho(void);
void GL_Perspective(void);

float g_cScale = 1.0f;

//================================================
//Texture generation
//================================================
//finds the next highest power of 2 number
int NextPow2(int val)
{
	int i = 0;
	int v = 0;
	while (v < val)
	{
		v = (1<<i);
		i++;
	}

	return v;
}

//try to make the backing of the card transparent by finding
//pixel neighbors of similar pixel color within a set tolerance.
//Starts where specified, and recursively calls on all neighbors.
//This is pretty rough in general.
#ifndef _NO_CARD_CARVING
static inline void CarveCardBackground(rgbaPixel_t *base, int x, int y, int w, int h)
{
	rgbaPixel_t *p = base;
	p += (y*w);
	p += x;

	if (p->r+16 <= p->g &&
		p->r+32 >= p->g &&
		p->g+16 <= p->b &&
		p->g+32 >= p->b &&
		(p->r+p->g+p->b) <= 512 &&
		(p->r+p->g+p->b) > 256 &&
		p->a)
	{ //this one passes.. set alpha to 0 and check for bottom/right neighbors
		p->a = 0;
		if (x < w)
		{ //call for right neighbor
			CarveCardBackground(base, x+1, y, w, h);
		}
		if (y < h)
		{ //call for bottom neighbor
			CarveCardBackground(base, x, y+1, w, h);
		}
		if (x > 0)
		{ //call for left neighbor
			CarveCardBackground(base, x-1, y, w, h);
		}
		if (y > 0)
		{ //call for top neighbor
			CarveCardBackground(base, x, y-1, w, h);
		}
	}
}
#endif

//go through and tint card bg reddish
void TintCardBackground(rgbaPixel_t *p, int w, int h)
{
	int i = 0;

	while (i < (w*h))
	{
		if (p->r+8 <= p->g &&
			p->r+48 >= p->g &&
			p->g+8 <= p->b &&
			p->g+48 >= p->b &&
			(p->r+p->g+p->b) <= 550 &&
			(p->r+p->g+p->b) > 256 &&
			p->a)
		{ //passed
			float r = (float)p->r*1.5f;
			if (r > 255)
			{
				r = 255;
			}
			p->r = (BYTE)r;
		}
		p++;
		i++;
	}
}

//creates a gl texture from card image data
int GL_CreateGLTextureFromCard(playingCard_t *card, int tint)
{
	int i = 0;
	int row = 0;
	int h = 0;
	int t = g_glTexNum;
	int cols = card->col-1;
	int rows = card->row-1;
	rgbaPixel_t cardImage[(CARD_PIXELS+2)*(CARD_PIXELS+2)];
	rgbaPixel_t *p;

	assert(card->image && card->image->pImage);
	
	memset(cardImage, 0, sizeof(cardImage));

	p = &card->image->pImage->p;

	//should never be out of this range
	assert(card->row >= 1 && card->row <= 3);
	assert(card->col >= 1 && card->col <= 4);

	//hop to the upper-left pixel on the card we want
	p += ((rows*((CARD_INIMAGE_PIXELS*CARD_INIMAGE_PIXELS)*4)));
	p += (cols*CARD_INIMAGE_PIXELS);
	p += CARD_BORDERING_PIXELS;
	p += (4*CARD_INIMAGE_PIXELS);
	while (row < CARD_PIXELS+2) //+2 is since it has to be power of 2, we have 2 extra rows, 1 top 1 bottom
	{
		if (h == 0)
		{
			//we need a power of 2 image, but the initial card image is 62x62,
			//so we just have 2 extra pixels per row.
			//pixel at start of row, same as one that will follow it
			cardImage[i].r = p->r;
			cardImage[i].g = p->g;
			cardImage[i].b = p->b;
			cardImage[i].a = p->a;
			i++;
		}
		cardImage[i].r = p->r;
		cardImage[i].g = p->g;
		cardImage[i].b = p->b;
		cardImage[i].a = p->a;
		i++;
		p++;
		h++;

		if (h == CARD_PIXELS)
		{ //horizontal reach, hop the pixel point around the image to the beginning point for this card on the
			//next row
			h = 0;
			row++;
			if (row == 1 || row == (CARD_PIXELS+1))
			{
				p -= CARD_PIXELS;
			}
			else
			{
				p += (3*CARD_INIMAGE_PIXELS) + (CARD_BORDERING_PIXELS*2);
			}

			//we need a power of 2 image, but the initial card image is 62x62,
			//so we just have 2 extra pixels per row.
			//pixel at end of row, same as one before it
			cardImage[i].r = cardImage[i-1].r;
			cardImage[i].g = cardImage[i-1].g;
			cardImage[i].b = cardImage[i-1].b;
			cardImage[i].a = cardImage[i-1].a;
			i++;
		}
	}

	//kind of works, but not really. Still have the problem of background in area
	//surrounded by other non-background pixels.
#ifndef _NO_CARD_CARVING
	CarveCardBackground(&cardImage[0], 3, 3, CARD_PIXELS+2, CARD_PIXELS+2);
	CarveCardBackground(&cardImage[0], CARD_PIXELS-2, CARD_PIXELS-2, CARD_PIXELS+2, CARD_PIXELS+2);
	CarveCardBackground(&cardImage[0], 3, CARD_PIXELS-2, CARD_PIXELS+2, CARD_PIXELS+2);
	CarveCardBackground(&cardImage[0], CARD_PIXELS-2, 3, CARD_PIXELS+2, CARD_PIXELS+2);
#endif

	if (tint)
	{ //just tint them red/pink. This sucks too and gets a bunch of pixels I don't want it to get, but oh well.
		TintCardBackground(&cardImage[0], CARD_PIXELS+2, CARD_PIXELS+2);
	}

	//now let's make our gl texture
	glBindTexture(GL_TEXTURE_2D, t);
	glTexImage2D(GL_TEXTURE_2D, 0, 4, CARD_PIXELS+2, CARD_PIXELS+2, 0, GL_RGBA, GL_UNSIGNED_BYTE, cardImage);

	//always clamp the cards
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	if (g_linearTex)
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
	else
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	}

	g_glTexNum++;
	return t;
}

//creates a gl texture from image data.
//If not power of 2, will make a call to ImageRegion function
//instead.
int GL_CreateGLTextureFromImage(image_t *image)
{
	int t = g_glTexNum;
	int pW, pH;

	assert(image && image->pImage);

	pW = NextPow2(image->pImage->i.width);
	pH = NextPow2(image->pImage->i.height);

	if (pW != image->pImage->i.width ||
		pH != image->pImage->i.height)
	{ //not a power of 2
		return GL_CreateGLTextureFromImageRegion(image, 0, 0, image->pImage->i.width, image->pImage->i.height);
	}

	glBindTexture(GL_TEXTURE_2D, t);
	glTexImage2D(GL_TEXTURE_2D, 0, 4, image->pImage->i.width, image->pImage->i.height, 0, GL_RGBA, GL_UNSIGNED_BYTE, &image->pImage->p);

	if (g_linearTex >= 2)
	{ //special background case
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

		if (g_linearTex == 3)
		{ //don't want linear tex filter
			g_linearTex = 0;
		}
	}

	if (g_linearTex)
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
	else
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	}

	g_glTexNum++;
	return t;
}

//creates a gl texture from a specified area of image data
int GL_CreateGLTextureFromImageRegion(image_t *image, int x, int y, int w, int h)
{
	int i = 0;
	int j = 0;
	int k = 0;
	int t = g_glTexNum;
	int imgSize;
	int pW, pH;
	rgbaPixel_t *p;
	rgbaPixel_t *finalImage;

	pW = NextPow2(w);
	pH = NextPow2(h);

	assert(image && image->pImage);

	imgSize = pW*pH;
	finalImage = (rgbaPixel_t *)malloc(imgSize*sizeof(rgbaPixel_t));

	memset(finalImage, 0, sizeof(finalImage));

	p = &image->pImage->p;

	//jump to the vertical start
	p += image->pImage->i.width*y;

	//jump to the horizontal start
	p += x;

	while (i < imgSize)
	{
		j = 0;
		k++;
		while (j < pW)
		{
			if (j < w && k < h)
			{
				finalImage[i].r = p->r;
				finalImage[i].g = p->g;
				finalImage[i].b = p->b;
				finalImage[i].a = p->a;
				p++;
			}
			else
			{ //fill the excess region with black no-alpha pixels
				finalImage[i].r = 0;
				finalImage[i].g = 0;
				finalImage[i].b = 0;
				finalImage[i].a = 0;
			}
			i++;
			j++;
		}

		if (k < h)
		{ //jump around to start of row with same offset horizontally
			p += image->pImage->i.width-w;
		}
	}

	//now let's make our gl texture
	glBindTexture(GL_TEXTURE_2D, t);
	glTexImage2D(GL_TEXTURE_2D, 0, 4, pW, pH, 0, GL_RGBA, GL_UNSIGNED_BYTE, finalImage);

	if (g_linearTex >= 2)
	{ //special background case
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

		if (g_linearTex == 3)
		{ //don't want linear tex filter
			g_linearTex = 0;
		}
	}

	if (g_linearTex)
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
	else
	{
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	}

	free(finalImage);
	g_glTexNum++;
	return t;
}

//create the texture for the back of a card
void GL_CardBackface(image_t *image)
{
	playingCard_t back;
	back.col = 4;
	back.row = 3;
	back.image = image;
	g_cardBackTex = GL_CreateGLTextureFromCard(&back, false);
}


//================================================
//Card rendering
//================================================
//add card draw object
void GL_AddDrawCard(drawCard_t *card)
{
	drawCard_t **nextFree = &g_drawCards;

	while (*nextFree)
	{
		nextFree = &((*nextFree)->next);
	}

	(*nextFree) = (drawCard_t *)malloc(sizeof(drawCard_t));
	memcpy(*nextFree, card, sizeof(drawCard_t));
	(*nextFree)->next = NULL;
}

extern int g_depthHack;
//draw a card object
void GL_DrawCard(drawCard_t *card)
{
	float wRadius = card->wRadius;
	float hRadius = card->hRadius;
	int elemHandicap = Game_ElementalHandicap(card);

	//reset viewmatrix, translate point so we are drawing relative to the
	//object pos, and rotate relative to the object angles
	glLoadIdentity();
	glTranslatef(card->pos[0], card->pos[1], card->pos[2]);
	glRotatef(card->ang[0], 1, 0, 0);
	glRotatef(card->ang[1], 0, 1, 0);
	glRotatef(card->ang[2], 0, 0, 1);

	glEnable(GL_CULL_FACE);
	
	//bind texture for the card
	glAlphaFunc(GL_GREATER, 0.0f);
	glEnable(GL_ALPHA_TEST);

	if (card->altTex)
	{
		glBindTexture(GL_TEXTURE_2D, card->pCard->image->glTexNum2);
	}
	else
	{
		glBindTexture(GL_TEXTURE_2D, card->pCard->image->glTexNum);
	}
	glCullFace(GL_BACK);
#if 1
	glBegin(GL_QUADS);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
	glEnd();
#else
	glBegin(GL_QUADS);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
	glEnd();
#endif

	if (g_depthHack)
	{
		glDisable(GL_DEPTH_TEST);
	}

	//while depth buffer precision works just fine on the geforce,
	//apparently lequal for radeons means "well not really less than
	//or equal but it'll be correct as long as you give me an aligned
	//transform matrix!"
	const float atiSucks = -0.01f;

	//draw other graphics, like card values and elemental properties
	//change the depth func so these show up on top of the cards
	glDepthFunc(GL_LEQUAL);

	//draw the top num
	glBindTexture(GL_TEXTURE_2D, g_texStorage.cardNumTextures[card->pCard->up]);
	glBegin(GL_QUADS);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(0.45f*wRadius, 0.65f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(0.45f*wRadius, 0.9f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(0.8f*wRadius, 0.9f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(0.8f*wRadius, 0.65f*hRadius, 0.0f+atiSucks);
	glEnd();

	//draw the bottom num
	glBindTexture(GL_TEXTURE_2D, g_texStorage.cardNumTextures[card->pCard->down]);
	glBegin(GL_QUADS);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(0.45f*wRadius, 0.25f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(0.45f*wRadius, 0.5f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(0.8f*wRadius, 0.5f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(0.8f*wRadius, 0.25f*hRadius, 0.0f+atiSucks);
	glEnd();

	//draw the left num
	glBindTexture(GL_TEXTURE_2D, g_texStorage.cardNumTextures[card->pCard->left]);
	glBegin(GL_QUADS);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(0.55f*wRadius, 0.45f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(0.55f*wRadius, 0.7f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(0.9f*wRadius, 0.7f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(0.9f*wRadius, 0.45f*hRadius, 0.0f+atiSucks);
	glEnd();

	//draw the right num
	glBindTexture(GL_TEXTURE_2D, g_texStorage.cardNumTextures[card->pCard->right]);
	glBegin(GL_QUADS);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(0.35f*wRadius, 0.45f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(0.35f*wRadius, 0.7f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(0.7f*wRadius, 0.7f*hRadius, 0.0f+atiSucks);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(0.7f*wRadius, 0.45f*hRadius, 0.0f+atiSucks);
	glEnd();

	//draw the elemental property if there is one
	if (card->pCard->elemental != CARD_ELEM_NONE)
	{
		glBindTexture(GL_TEXTURE_2D, g_texStorage.cardElemTextures[card->pCard->elemental]);
		glBegin(GL_QUADS);
			glTexCoord2f(1.0f, 1.0f);
			glVertex3f(-0.9f*wRadius, 0.45f*hRadius, 0.0f+atiSucks);
			glTexCoord2f(1.0f, 0.0f);
			glVertex3f(-0.9f*wRadius, 0.9f*hRadius, 0.0f+atiSucks);
			glTexCoord2f(0.0f, 0.0f);
			glVertex3f(-0.5f*wRadius, 0.9f*hRadius, 0.0f+atiSucks);
			glTexCoord2f(0.0f, 1.0f);
			glVertex3f(-0.5f*wRadius, 0.45f*hRadius, 0.0f+atiSucks);
		glEnd();
	}

	//draw + or - if elemental effect
	if (elemHandicap)
	{
		if (elemHandicap > 0)
		{
			glBindTexture(GL_TEXTURE_2D, g_texStorage.letterTextures['+']);
		}
		else
		{
			glBindTexture(GL_TEXTURE_2D, g_texStorage.letterTextures['-']);
		}
		glBegin(GL_QUADS);
			glTexCoord2f(1.0f, 1.0f);
			glVertex3f(-1.3f*wRadius, -1.9f*hRadius, 0.0f+atiSucks);
			glTexCoord2f(1.0f, 0.0f);
			glVertex3f(-1.3f*wRadius, -0.2f*hRadius, 0.0f+atiSucks);
			glTexCoord2f(0.0f, 0.0f);
			glVertex3f(-0.3f*wRadius, -0.2f*hRadius, 0.0f+atiSucks);
			glTexCoord2f(0.0f, 1.0f);
			glVertex3f(-0.3f*wRadius, -1.9f*hRadius, 0.0f+atiSucks);
		glEnd();
	}

	//standard depth test again
	glDepthFunc(GL_LESS);

	if (g_depthHack)
	{
		glEnable(GL_DEPTH_TEST);
	}

	//now draw the back of the card
	glDisable(GL_ALPHA_TEST);
	glBindTexture(GL_TEXTURE_2D, g_cardBackTex);
	glCullFace(GL_FRONT);

	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 1.0f);
		glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
		glTexCoord2f(0.0f, 0.0f);
		glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
		glTexCoord2f(1.0f, 0.0f);
		glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
		glTexCoord2f(1.0f, 1.0f);
		glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
	glEnd();

	glDisable(GL_CULL_FACE);
}

//draw and remove card render objects recursively
void GL_RecursiveCardDraw(drawCard_t *card)
{
	if (!card)
	{
		return;
	}
	GL_RecursiveCardDraw(card->next);
	assert(card->altTex >= 0);
	if (card->altTex >= 0)
	{
		GL_DrawCard(card);
	}
	free(card);
}


//================================================
//Object rendering
//================================================
//draw the yellow "turn arrow"
void GL_DrawTurnArrow(void)
{
	//these things are made up of untextured colored polygons
	glDisable(GL_TEXTURE_2D);
	glColor4f(0.5f*g_cScale, 0.5f*g_cScale, 0.0f, 1.0f);

	glBegin(GL_TRIANGLES);
		glVertex3f(-0.2f, 0.0f, 0.2f);
		glColor4f(0.0f, 0.0f, 0.0f, 1.0f); //darken toward tip
		glVertex3f(0.0f, -0.6f, 0.0f);
		glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 0.0f, 1.0f); //normal color again
		glVertex3f(0.2f, 0.0f, 0.2f);
	glEnd();
	glBegin(GL_TRIANGLES);
		glVertex3f(0.2f, 0.0f, 0.2f);
		glColor4f(0.0f, 0.0f, 0.0f, 1.0f); //darken toward tip
		glVertex3f(0.0f, -0.6f, 0.0f);
		glColor4f(0.4f*g_cScale, 0.4f*g_cScale, 0.0f, 1.0f); //normal color again
		glVertex3f(0.0f, 0.0f, -0.2f);
	glEnd();
	glBegin(GL_TRIANGLES);
		glVertex3f(0.0f, 0.0f, -0.2f);
		glColor4f(0.0f, 0.0f, 0.0f, 1.0f); //darken toward tip
		glVertex3f(0.0f, -0.6f, 0.0f);
		glColor4f(0.8f*g_cScale, 0.8f*g_cScale, 0.0f, 1.0f); //normal color again
		glVertex3f(-0.2f, 0.0f, 0.2f);
	glEnd();

	//drew the 3 faces, now draw the top connecting face
	glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 0.0f, 1.0f); //normal color again
	glBegin(GL_TRIANGLES);
		glVertex3f(-0.2f, 0.0f, 0.2f);
		glVertex3f(0.2f, 0.0f, 0.2f);
		glVertex3f(0.0f, 0.0f, -0.2f);
	glEnd();


	//reenable texturing and default color
	glEnable(GL_TEXTURE_2D);
	glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 1.0f*g_cScale, 1.0f);
}

//add object draw object
void GL_AddDrawObject(drawObject_t *obj)
{
	drawObject_t **nextFree = &g_drawObjects;

	while (*nextFree)
	{
		nextFree = &((*nextFree)->next);
	}

	(*nextFree) = (drawObject_t *)malloc(sizeof(drawObject_t));
	memcpy(*nextFree, obj, sizeof(drawObject_t));
	(*nextFree)->next = NULL;
}

//draw a draw object
void GL_DrawObject(drawObject_t *obj)
{
	float wRadius = obj->wRadius;
	float hRadius = obj->hRadius;
	int passes = 0;

	//reset viewmatrix, translate point so we are drawing relative to the
	//object pos, and rotate relative to the object angles
	glLoadIdentity();
	glTranslatef(obj->pos[0], obj->pos[1], obj->pos[2]);
	glRotatef(obj->ang[0], 1, 0, 0);
	glRotatef(obj->ang[1], 0, 1, 0);
	glRotatef(obj->ang[2], 0, 0, 1);

	if (obj->img == -1)
	{ //special case, draw turn arrow
		GL_DrawTurnArrow();
		return;
	}

	if (obj->noDepth)
	{ //don't write to depth buffer
		glDepthMask(0);
	}

	if (obj->srcBlend != -1)
	{
		if (obj->dstBlend != -1)
		{
			glBlendFunc(obj->srcBlend, obj->dstBlend);
		}
		else
		{
			glBlendFunc(obj->srcBlend, obj->srcBlend);
		}
		glEnable(GL_BLEND);
	}

	if (obj->alphaTest != -1)
	{
		glAlphaFunc(obj->alphaTest, 0.0f);
		glEnable(GL_ALPHA_TEST);
	}

	glEnable(GL_CULL_FACE);
	while (passes < 2)
	{
		//bind texture for the object
		if (passes == 0)
		{
			glBindTexture(GL_TEXTURE_2D, obj->img);
			glCullFace(GL_BACK);
		}
		else
		{
			if (obj->imgBack != -1)
			{ //otherwise just draw with the first tex
				glBindTexture(GL_TEXTURE_2D, obj->imgBack);
			}
			glCullFace(GL_FRONT);
		}

		glBegin(GL_QUADS);
			glTexCoord2f(1.0f, 1.0f);
			glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
			glTexCoord2f(1.0f, 0.0f);
			glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
			glTexCoord2f(0.0f, 0.0f);
			glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
			glTexCoord2f(0.0f, 1.0f);
			glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
		glEnd();

		passes++;
	}
	glDisable(GL_CULL_FACE);

	if (obj->srcBlend != -1)
	{
		glDisable(GL_BLEND);
	}

	if (obj->alphaTest != -1)
	{
		glDisable(GL_ALPHA_TEST);
	}

	if (obj->noDepth)
	{
		glDepthMask(1);
	}
}

//draw and remove object render objects recursively
void GL_RecursiveObjectDraw(drawObject_t *obj)
{
	if (!obj)
	{
		return;
	}
	GL_RecursiveObjectDraw(obj->next);
	GL_DrawObject(obj);
	free(obj);
}


//================================================
//Background rendering
//================================================
//draw the background "underlay"
void GL_DrawBackground(void)
{
	const float wRadius = 9.0f;
	const float hRadius = 6.7f;
	glLoadIdentity();
	glTranslatef(0.0f, 0.0f, -16.0f);

	//don't want to ever cover anything else up with the bg
	glDepthMask(0);

	if (g_bgOnePart)
	{
		glBindTexture(GL_TEXTURE_2D, g_texStorage.backgroundImage);

		glBegin(GL_QUADS);
			glTexCoord2f(0.0f, 1.0f);
			glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
			glTexCoord2f(0.0f, 0.0f);
			glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
			glTexCoord2f(1.0f, 0.0f);
			glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
			glTexCoord2f(1.0f, 1.0f);
			glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
		glEnd();
	}
	else
	{
		glBindTexture(GL_TEXTURE_2D, g_texStorage.backgroundImage);
		glBegin(GL_QUADS);
			if (g_bgMirrorFirst)
			{
				glTexCoord2f(0.0f, 1.0f);
				glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
				glTexCoord2f(0.0f, 0.0f);
				glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(1.0f, 0.0f);
				glVertex3f(0.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(1.0f, 1.0f);
				glVertex3f(0.0f*wRadius, -1.0f*hRadius, 0.0f);
			}
			else
			{
				glTexCoord2f(1.0f, 1.0f);
				glVertex3f(-1.0f*wRadius, -1.0f*hRadius, 0.0f);
				glTexCoord2f(1.0f, 0.0f);
				glVertex3f(-1.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(0.0f, 0.0f);
				glVertex3f(0.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(0.0f, 1.0f);
				glVertex3f(0.0f*wRadius, -1.0f*hRadius, 0.0f);
			}
		glEnd();


		glBindTexture(GL_TEXTURE_2D, g_texStorage.backgroundImage2);
		glBegin(GL_QUADS);
			if (g_bgMirrorSecond)
			{
				glTexCoord2f(0.0f, 1.0f);
				glVertex3f(0.0f*wRadius, -1.0f*hRadius, 0.0f);
				glTexCoord2f(0.0f, 0.0f);
				glVertex3f(0.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(1.0f, 0.0f);
				glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(1.0f, 1.0f);
				glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
			}
			else
			{
				glTexCoord2f(1.0f, 1.0f);
				glVertex3f(0.0f*wRadius, -1.0f*hRadius, 0.0f);
				glTexCoord2f(1.0f, 0.0f);
				glVertex3f(0.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(0.0f, 0.0f);
				glVertex3f(1.0f*wRadius, 1.0f*hRadius, 0.0f);
				glTexCoord2f(0.0f, 1.0f);
				glVertex3f(1.0f*wRadius, -1.0f*hRadius, 0.0f);
			}
		glEnd();
	}

	glDepthMask(1);
}


//================================================
//Menu/string drawing
//================================================
//scale value assuming 640x480 is the "ideal"
static inline float ScaleBasedOnResW(float f)
{
	return f*(g_displayWidth/640.0f);
}

//for height
static inline float ScaleBasedOnResH(float f)
{
	return f*(g_displayHeight/480.0f);
}

//draw a string message
void GL_DrawStatusString(char *str, float x, float y, int fade, float charHeight, float charWidth)
{
	int t;
	float cX;
	float cY;
	int blend = 0;
	float blendAmt = 0.0f;

	x = ScaleBasedOnResW(x);
	y = ScaleBasedOnResH(y);
	charHeight = ScaleBasedOnResH(charHeight);
	charWidth = ScaleBasedOnResW(charWidth);

	cX = x;
	cY = y;

	glAlphaFunc(GL_GREATER, 0.0f);
	glEnable(GL_ALPHA_TEST);

	if (fade == 2)
	{
		if ((g_CGS.pCStrTime - g_Time) < 1000)
		{
			blend = 1;
			blendAmt = (g_CGS.pCStrTime - g_Time)*0.001f;
		}
	}
	else if (fade && (g_CGS.pStrTime - g_Time) < 1000)
	{
		blend = 1;
		blendAmt = (g_CGS.pStrTime - g_Time)*0.001f;
	}

	if (blend)
	{
		glBlendFunc(GL_SRC_ALPHA, GL_DST_ALPHA);
		glEnable(GL_BLEND);
		glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 1.0f*g_cScale, blendAmt);
	}

	while (*str)
	{
		if (*str == '\n')
		{
			cX = x;
			cY += charHeight;
			str++;
			continue;
		}

		t = g_texStorage.letterTextures[*str];
		if (!t)
		{ //if don't have a tex for this char, use empty space
			t = g_texStorage.letterTextures[' '];
		}

		glBindTexture(GL_TEXTURE_2D, t);
		
		glBegin(GL_QUADS);
			glTexCoord2f(0.0f, 0.0f);
			glVertex2f(cX, cY);

			glTexCoord2f(0.0f, 1.0f);
			glVertex2f(cX, cY+charHeight);

			glTexCoord2f(1.0f, 1.0f);
			glVertex2f(cX+charWidth, cY+charHeight);

			glTexCoord2f(1.0f, 0.0f);
			glVertex2f(cX+charWidth, cY);
		glEnd();
		if (*str == 'I' || *str == 'i' || *str == 'l')
		{ //kind of hacky, these have a lot of "right-space"
			cX += charWidth*0.6f;
		}
		else
		{
			cX += charWidth*0.8f;
		}
		if (cX >= g_displayWidth)
		{
			cX = x;
			cY += charHeight/1.5f;
		}
		str++;
	}
	glDisable(GL_ALPHA_TEST);

	if (blend)
	{
		glDisable(GL_BLEND);
	}
}

//draws a draw object in ortho screen coords
void GL_DrawObjectOrtho(drawObject_t *obj, float x, float y, float w, float h)
{
	x = ScaleBasedOnResW(x);
	y = ScaleBasedOnResH(y);
	w = ScaleBasedOnResW(w);
	h = ScaleBasedOnResH(h);

	if (obj->alphaTest != -1)
	{
		glAlphaFunc(obj->alphaTest, 0.0f);
		glEnable(GL_ALPHA_TEST);
	}

	if (obj->srcBlend != -1)
	{
		if (obj->dstBlend != -1)
		{
			glBlendFunc(obj->srcBlend, obj->dstBlend);
		}
		else
		{
			glBlendFunc(obj->srcBlend, obj->srcBlend);
		}
		glEnable(GL_BLEND);
	}

	glBindTexture(GL_TEXTURE_2D, obj->img);
		
	glBegin(GL_QUADS);
		glTexCoord2f(0.0f, 0.0f);
		glVertex2f(x, y);

		glTexCoord2f(0.0f, 1.0f);
		glVertex2f(x, y+h);

		glTexCoord2f(1.0f, 1.0f);
		glVertex2f(x+w, y+h);

		glTexCoord2f(1.0f, 0.0f);
		glVertex2f(x+w, y);
	glEnd();

	if (obj->alphaTest != -1)
	{
		glDisable(GL_ALPHA_TEST);
	}

	if (obj->srcBlend != -1)
	{
		glDisable(GL_BLEND);
	}
}

//draw the menu
void GL_DrawMenu(gameMenu_t *menu)
{
	int i = 0;
	float x = 64.0f;
	float y = 128.0f;
	int l = 0;
	drawObject_t pointy;

	memset(&pointy, 0, sizeof(drawObject_t));
	pointy.img = g_texStorage.misterPointy;
	pointy.alphaTest = GL_GREATER;
	pointy.srcBlend = -1;

	GL_Ortho();

	while (i < menu->numMenus)
	{
		if (menu->menu == i)
		{
			if (menu->menus[i].title[0] == '\n')
			{
				GL_DrawObjectOrtho(&pointy, x-38.0f, y+20.0f, 32.0f, 32.0f);

				if (g_CGS.doink > g_Time)
				{
					drawObject_t doink;

					memset(&doink, 0, sizeof(drawObject_t));
					doink.img = g_texStorage.doinkBubble;
					doink.alphaTest = GL_GREATER;
					doink.srcBlend = -1;
					GL_DrawObjectOrtho(&doink, x-20.0f, y, 64.0f, 32.0f);
					doink.img = g_texStorage.doinkTxt;
					GL_DrawObjectOrtho(&doink, x-12.0f, y+6.0f, 60.0f, 16.0f);
				}
			}
			else
			{
				GL_DrawObjectOrtho(&pointy, x-38.0f, y-12.0f, 32.0f, 32.0f);
			}

			if (g_menuInput.active)
			{
				glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 0.0f, 1.0f);
			}
		}
		GL_DrawStatusString(menu->menus[i].title, x, y, 0, 32.0f, 20.0f);
		GL_DrawStatusString(menu->menus[i].status, 352.0f, y, 0, 32.0f, 20.0f);
		l = strlen(menu->menus[i].status);
		y += 20.0f;
		while (l > 18)
		{
			y += 20.0f;
			l -= 18;
		}

		if (g_menuInput.active)
		{
			glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 1.0f*g_cScale, 1.0f);
		}

		i++;
	}

	GL_Perspective();
}

//draw the card selection screen
void GL_DrawCardSelection(void)
{
	int i = 0;
	float x = 50.0f;
	float y = 50.0f;
	char lvStr[MAX_PRINT_CHARS];
	drawObject_t pointy;
	playingCard_t *baseList = &g_playingCards[0];

	memset(&pointy, 0, sizeof(drawObject_t));
	pointy.img = g_texStorage.misterPointy;
	pointy.alphaTest = GL_GREATER;
	pointy.srcBlend = -1;

	baseList += (g_gameState.cardPage*NUM_CARDS_PER_LEVEL);

	GL_Ortho();

	sprintf(lvStr, "Lv.%i", g_gameState.cardPage+1);
	GL_DrawStatusString(lvStr, x, y, 0, 50.0f, 30.0f);
	y += 40.0f;

	while (i < NUM_CARDS_PER_LEVEL)
	{
		if (g_gameState.cardSelected == i)
		{
			GL_DrawObjectOrtho(&pointy, x-56.0f, y-18.0f, 50.0f, 50.0f);
		}
		GL_DrawStatusString(baseList[i].cardName, x, y, 0, 50.0f, 30.0f);
		y += 30.0f;
		i++;
	}

	x = 200.0f;
	y = 20.0f;

	if (g_gameState.cardsChosen == NUM_CARDS_PER_PLAYER)
	{ //confirmation
		GL_DrawStatusString("A to begin.\nB to cancel.", x, y, 0, 32.0f, 20.0f);
	}

	GL_Perspective();
}


//================================================
//Initialization, frame functions
//================================================
//same functionality as gluPerspective
void GL_CalcPerspective(float fovy, float aspect, float zNear, float zFar)
{
	/*
	float xmin, xmax, ymin, ymax;

	ymax = zNear * (float)tan(fovy * PI / 360.0f);
	ymin = -ymax;

	xmin = ymin * aspect;
	xmax = ymax * aspect;

	glFrustum(xmin, xmax, ymin, ymax, zNear, zFar);
	*/

	//xbox
	D3DXMATRIX matProj;

	//xbox hax
	aspect = 640.0f/475.0f;
	fovy -= 0.158f;

	D3DXMatrixPerspectiveFovRH(&matProj, fovy, aspect, zNear, zFar);
	g_pd3dDevice->SetTransform(D3DTS_PROJECTION, &matProj);
}

//ortho view matrix
void GL_Ortho(void)
{
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho(0, g_displayWidth, g_displayHeight, 0, -1, 1);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
}

//perspective view matrix
void GL_Perspective(void)
{
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

//init gl state
int GL_Init(void)
{
	glEnable(GL_TEXTURE_2D);
	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClearDepth(1.0f);
	glDepthFunc(GL_LESS);
	glEnable(GL_DEPTH_TEST);
	glShadeModel(GL_SMOOTH);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

	glDisable(GL_CULL_FACE);

	//xbox stuff
	g_pd3dDevice->SetRenderState(D3DRS_LIGHTING, FALSE);

	return 1;
}

//draw gl frame
void GL_DrawFrame(void)
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	//for xbox
	g_pd3dDevice->BeginScene();

	if (g_gameMenu.active || g_ruleMenu.active)
	{
		g_cScale = 0.2f;
	}
	else
	{
		g_cScale = 1.0f;
	}

	glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 1.0f*g_cScale, 1.0f);

	GL_DrawBackground();

	GL_RecursiveCardDraw(g_drawCards);
	g_drawCards = NULL;

	GL_RecursiveObjectDraw(g_drawObjects);
	g_drawObjects = NULL;

	//always draw menus/strings over everything else
	glDisable(GL_DEPTH_TEST);

	if (g_CGS.pCStrTime > g_Time)
	{
		GL_Ortho();
		GL_DrawStatusString(g_CGS.pCStr, g_CGS.pCX, g_CGS.pCY, 2, 64.0f, 40.0f);
		GL_Perspective();
	}

	if (g_gameMenu.active)
	{
		g_cScale = 1.0f;
		glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 1.0f*g_cScale, 1.0f);
		GL_DrawMenu(&g_gameMenu);
	}
	else if (g_ruleMenu.active)
	{ //selecting rules for beginning of game
		g_cScale = 1.0f;
		glColor4f(1.0f*g_cScale, 1.0f*g_cScale, 1.0f*g_cScale, 1.0f);
		GL_DrawMenu(&g_ruleMenu);
	}
	else if (!g_gameState.chosenCards)
	{ //display card selection then
		if (!g_listenSocketCreated)
		{
			GL_DrawCardSelection();
		}
	}

	if (g_CGS.pStrTime > g_Time)
	{
		GL_Ortho();
		GL_DrawStatusString(g_CGS.pStr, 10.0f, 10.0f, 1, 32.0f, 20.0f);
		GL_Perspective();
	}

	glEnable(GL_DEPTH_TEST);

	//for xbox
	g_pd3dDevice->EndScene();
}
