/*
=============================================================================
Module Information
------------------
Name:			util.cpp
Author:			Rich Whitehouse
Description:	random server logic utility/convenience functions
=============================================================================
*/

#include "main.h"

typedef struct objSpawnMatch_s
{
	void				(*a)(gameObject_s *obj, BYTE *b, const objArgs_t *args, int numArgs);
	char				*b;
} objSpawnMatch_t;

static const objSpawnMatch_t g_objSpawnTable[] =
{
	{ObjProjectile_Spawn, "ObjProjectile_Spawn"},
	{ObjCamMarker_Spawn, "ObjCamMarker_Spawn"},
	{ObjLift_Spawn, "ObjLift_Spawn"},
	{ObjLight_Spawn, "ObjLight_Spawn"},
	{ObjPulser_Spawn, "ObjPulser_Spawn"},
	{ObjVisBlock_Spawn, "ObjVisBlock_Spawn"},
	{ObjSoundEmit_Spawn, "ObjSoundEmit_Spawn"},
	{ObjSnakeHead_Spawn, "ObjSnakeHead_Spawn"},
	{ObjFighter_Spawn, "ObjFighter_Spawn"},
	{ObjSupernova_Spawn, "ObjSupernova_Spawn"},
	{ObjBigJesus_Spawn, "ObjBigJesus_Spawn"},
	{0, 0}
};

//parse a spawn function from the table
bool Util_ParseObjSpawn(const char *str, gameObject_t *obj)
{
	obj->customSpawn = NULL;

	if (!str || !str[0])
	{
		return false;
	}

	int i = 0;
	while (g_objSpawnTable[i].b)
	{
		if (!strcmp(str, g_objSpawnTable[i].b))
		{ //got it
			obj->customSpawn = g_objSpawnTable[i].a;
			return true;
		}
		i++;
	}

	return false;
}

typedef struct objValMatch_s
{
	objTypes_e	a;
	char		*b;
} objValMatch_t;

static const objValMatch_t g_objValTable[] =
{
	{OBJ_TYPE_SPRITE, "OBJ_TYPE_SPRITE"},
	{OBJ_TYPE_MAP, "OBJ_TYPE_MAP"},
	{OBJ_TYPE_MODEL, "OBJ_TYPE_MODEL"},
	{OBJ_TYPE_NETEVENT, "OBJ_TYPE_NETEVENT"},
	{OBJ_TYPE_CAM, "OBJ_TYPE_CAM"},
	{OBJ_TYPE_PROJECTILE, "OBJ_TYPE_PROJECTILE"},
	{OBJ_TYPE_GENERAL, "OBJ_TYPE_GENERAL"},
	{OBJ_TYPE_LIGHT, "OBJ_TYPE_LIGHT"},
	{OBJ_TYPE_VISBLOCK, "OBJ_TYPE_VISBLOCK"},
	{OBJ_TYPE_USER, "OBJ_TYPE_USER"},
	{NUM_OBJ_TYPES, NULL}
};

//parse an object type from the table
bool Util_ParseObjType(const char *str, int *out)
{
	*out = OBJ_TYPE_SPRITE;

	if (!str || !str[0])
	{
		return false;
	}

	int i = 0;
	while (g_objValTable[i].b)
	{
		if (!strcmp(str, g_objValTable[i].b))
		{ //got it
			*out = g_objValTable[i].a;
			return true;
		}
		i++;
	}

	return false;
}

//parse string vector to float
bool Util_ParseVector(const char *str, float *out)
{
	if (!str || !str[0])
	{
		out[0] = 0.0f;
		out[1] = 0.0f;
		out[2] = 0.0f;
		return false;
	}
	sscanf(str, "(%f %f %f)", &out[0], &out[1], &out[2]);
	return true;
}

//parse string to int
bool Util_ParseInt(const char *str, int *out)
{
	if (!str || !str[0])
	{
		*out = 0;
		return false;
	}
	*out = atoi(str);
	return true;
}

//nudge value
float Util_NudgeValue(float cur, float dst, float fact)
{
	if (fact > 1.0f)
	{
		fact = 1.0f;
	}
	else if (fact < 0.000001f)
	{
		fact = 0.000001f;
	}

	float dif = (dst-cur);
	if (fabsf(dif) < 0.00001f)
	{
		return dst;
	}

	float f = cur+(dif*fact);

	return f;
}

//cheesy function to get a radius from a bounds for very quick frustum checking
float Util_RadiusFromBounds(float *mins, float *maxs, float padMul)
{
	float b = 0.0f;
	for (int i = 0; i < 3; i++)
	{
		if (fabsf(mins[i]) > b)
		{
			b = fabsf(mins[i]);
		}
		if (maxs[i] > b)
		{
			b = maxs[i];
		}
	}

	return b*padMul;
}

//checks if two game objects overlap
bool Util_GameObjectsOverlap(gameObject_t *obja, gameObject_t *objb)
{
	for (int i = 0; i < 3; i++)
	{
		float mina = obja->net.pos[i]+obja->spawnMins[i];
		float maxa = obja->net.pos[i]+obja->spawnMaxs[i];
		float minb = objb->net.pos[i]+objb->spawnMins[i];
		float maxb = objb->net.pos[i]+objb->spawnMaxs[i];
		if (mina < minb &&
			maxa < minb)
		{
			return false;
		}
		if (mina > maxb &&
			maxa > maxb)
		{
			return false;
		}
	}
	return true;
}

//touch objects to each other
void Util_TouchObjects(gameObject_t *obj1, gameObject_t *obj2, const collObj_t *col)
{
	if (!obj1 || !obj2)
	{
		return;
	}
	if (obj1->touch)
	{
		obj1->touch(obj1, obj2, col);
	}
	if (obj2->touch)
	{
		obj2->touch(obj2, obj1, col);
	}
}

//project a decal
void Util_ProjectDecal(gameObject_t *obj, const float *start, const float *end, const char *decal,
					   float decalSize, int decalLife, int decalFade)
{
	if (!obj || !obj->inuse)
	{
		return;
	}

	gameObject_t *netEvent = LServ_CreateObject();
	if (!netEvent)
	{
		return;
	}

	netEvent->net.type = OBJ_TYPE_NETEVENT;
	netEvent->net.frame = NETEVENT_DECAL;
	netEvent->net.pos[0] = start[0];
	netEvent->net.pos[1] = start[1];
	netEvent->net.pos[2] = start[2];

	netEvent->net.ang[0] = end[0];
	netEvent->net.ang[1] = end[1];
	netEvent->net.ang[2] = end[2];

	netEvent->net.modelScale[0] = decalSize; //projected decal size
	netEvent->net.modelScale[1] = (float)decalLife; //lifetime (in ms)
	netEvent->net.modelScale[2] = (float)decalFade; //fadetime (in ms)

	netEvent->net.solid = obj->net.index;

	netEvent->think = ObjGeneral_RemoveThink;
	netEvent->thinkTime = g_curTime+50;

	char str[64];
    sprintf(str, "^%s", decal);
	netEvent->net.strIndex = g_sharedFn->Common_ServerString(str);

}

//do damage
bool Util_DamageObject(gameObject_t *damager, gameObject_t *victim, int damage)
{
	if (!victim->hurtable)
	{
		return false;
	}

    if (victim->health <= 0)
	{ //already dead
		return true;
	}

	if (victim->localFlags & LFL_ENEMY)
	{ //don't take damage from other enemies
		if (damager && damager != victim &&
			damager->inuse)
		{
			if (damager->localFlags & LFL_ENEMY)
			{ //attacker is enemy
				return false;
			}
			if (damager->net.owner >= 0 && damager->net.owner < MAX_GAME_OBJECTS)
			{
				if (g_gameObjects[damager->net.owner].inuse &&
					g_gameObjects[damager->net.owner].localFlags & LFL_ENEMY)
				{ //owner is enemy
					return false;
				}
			}
		}
	}

	if (victim->net.index < MAX_NET_CLIENTS)
	{
		if (victim->debounce2 > g_curTime)
		{ //player invincible
			return false;
		}
		if (damager->net.index < MAX_NET_CLIENTS ||
			damager->net.owner < MAX_NET_CLIENTS)
		{ //no friendly fire
			return false;
		}
	}

	victim->health -= damage;
	if (victim->pain)
	{
		victim->pain(victim, damager, damage);
	}
	if (victim->health <= 0 && victim->death)
	{
		victim->death(victim, damager);
	}
	return true;
}

//cheap check for general cam proximity on xy
bool Util_CamProx2D(float *pos, const float d)
{
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	if (cam->inuse)
	{
		float xyD[2] = {cam->net.pos[0]-pos[0], cam->net.pos[1]-pos[1]};
		float xyDist = (xyD[0]*xyD[0] + xyD[1]*xyD[1]);
		if (xyDist < d)
		{
			return true;
		}
	}

	return false;
}

//hax
void Util_DebugBox(float *mins, float *maxs, float *color)
{
	gameObject_t *netEvent = LServ_CreateObject();
	if (!netEvent)
	{
		return;
	}

	netEvent->net.type = OBJ_TYPE_NETEVENT;
	netEvent->net.frame = NETEVENT_DEBUGBOX;
	
	netEvent->net.mins[0] = mins[0];
	netEvent->net.mins[1] = mins[1];
	netEvent->net.mins[2] = mins[2];
	netEvent->net.maxs[0] = maxs[0];
	netEvent->net.maxs[1] = maxs[1];
	netEvent->net.maxs[2] = maxs[2];
	netEvent->net.ang[0] = color[0];
	netEvent->net.ang[1] = color[1];
	netEvent->net.ang[2] = color[2];

	netEvent->think = ObjGeneral_RemoveThink;
	netEvent->thinkTime = g_curTime+50;
}

//more hax
void Util_DebugFX(float *pos, float *dir)
{
	float impactFxPos[3], impactFxAng[3];
	Math_VecToAngles(dir, impactFxAng);
	Math_VecCopy(pos, impactFxPos);
	ObjParticles_Create("weapons/laserimpact", impactFxPos, impactFxAng, -1);
}
