/*
=============================================================================
Module Information
------------------
Name:			obj_player.cpp
Author:			Rich Whitehouse
Description:	server logic object: player
=============================================================================
*/

#include "main.h"

typedef enum
{
	WEAPON_EYEBEAM,
	WEAPON_BOMB,
	NUM_FETAL_WEAPONS
} fetalWeaponIndex_e;

typedef struct fetalWeapon_s
{
	const char					*projectile;
	fetalWeaponIndex_e			weaponIndex;
	float						muzzleOffset[3];
	bool						charge;
	int							nonDepressCount;
	int							depressPause;
	int							shotDelay;
} fetalWeapon_t;

static fetalWeapon_t g_fetalWeapons[NUM_FETAL_WEAPONS] =
{
	{
		"obj_proj_smallbeam",
		WEAPON_EYEBEAM,
		{64, 0, 0},
		false,
		6,
		200,
		50
	},
	{
		"obj_supernova",		//projectile
		WEAPON_BOMB,			//type
		{64, 0, 0},				//muzzle offset
		false,					//charge
		-1,//1,						//non-depress count
		0,//1000,					//depress pause
		3000					//shot delay
	}
};

float g_initialFPSPos[3] = {0.0f, 0.0f, 0.0f};
float g_initialFPSAng[3] = {0.0f, 210.0f, 0.0f}; //hax

//player touch
void ObjPlayer_Touch(gameObject_s *obj, gameObject_s *other, const collObj_t *col)
{
	if (other && other->inuse && other->projDamage <= 0)
	{ //touching anything without a specific damage is DEATH
		int touchDmg = 10000;
		if (g_musical && other->hurtable)
		{
			Util_DamageObject(obj, other, touchDmg);
			touchDmg = 50;
		}
		Util_DamageObject(other, obj, touchDmg);
	}
}

//player die
void ObjPlayer_Death(gameObject_s *obj, gameObject_s *killer)
{
	if (killer && killer->inuse && obj != killer)
	{ //hurt whatever killed me
		Util_DamageObject(obj, killer, 100);
	}

	ObjSound_Create(obj->net.pos, "assets/sound/weapons/explode2.wav", 2.0f, -1);
	float up[3] = {90.0f, 0.0f, 0.0f};
	ObjParticles_Create("player/death", obj->net.pos, up, -1);

	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	obj->net.pos[0] = cam->camTrackPos[0]-4096.0f;
	obj->net.pos[1] = cam->camTrackPos[1];
	obj->net.pos[2] = cam->camTrackPos[2]-4096.0f;
	obj->net.ang[0] = 0.0f;
	obj->net.ang[1] = 0.0f;
	obj->net.ang[2] = 0.0f;
	
	obj->health = DEFAULT_PLAYER_HEALTH;
	if (g_musical)
	{
		obj->health = 200;
	}
//	obj->altFls.clWeaponAmmo = 3;
//	obj->net.plAmmo = obj->altFls.clWeaponAmmo;

	obj->debounce = g_curTime+500;//1000;
	obj->debounce2 = g_curTime+3000; //invincibility period

	//if (!g_runningMultiplayer)
	if (obj->net.lives > 0 || g_musical)
	{ //lives
		if (!g_musical)
		{ //infinite lives in musical mode
			obj->net.lives--;
		}
		else
		{ //restore ammo in musical mode
			obj->altFls.clWeaponAmmo = 0;
			obj->net.plAmmo = obj->altFls.clWeaponAmmo;
			obj->net.lives -= 10001;
			obj->musicalKills = 0;
			if (obj->net.lives < -9999999)
			{
				obj->net.lives = -9999999;
			}
			LServ_MusicalScoreLog(obj);
		}
		if (obj->net.lives <= 0 && !g_musical)
		{
			obj->think = ObjGeneral_RemoveThink;
			bool anyoneAlive = false;
			for (int i = 0; i < MAX_NET_CLIENTS; i++)
			{
				gameObject_t *pl = &g_gameObjects[i];
				if (pl->inuse && pl->net.lives > 0)
				{
					anyoneAlive = true;
					break;
				}
			}
			if (!anyoneAlive)
			{ //they're all dead, restart the map
				g_sharedFn->Common_MapChange(NULL);
			}
		}
	}
}

//confine player object within the camera scope
void ObjPlayer_CameraConfine(gameObject_t *obj)
{
	if (g_fpsMode)
	{ //do not confine in fps mode
		return;
	}
	float dMin[3] = {-750.0f, -1000.0f, -700.0f};
	float dMax[3] = {750.0f, 1000.0f, 700.0f};

	if (!g_camFrustumFresh)
	{
		return;
	}

	if (g_noConfineTime >= g_curTime)
	{
		return;
	}

	if (obj->fls.playerSpawnSeq)
	{ //don't confine during spawn sequence
		return;
	}

	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	if (!cam->inuse)
	{
		return;
	}

	float camUsePos[3];
	camUsePos[0] = cam->net.pos[0];
	camUsePos[1] = cam->net.pos[1];
	camUsePos[2] = cam->net.pos[2];

	//derive the confinement bounds from the current camera frustum
	//these values should probably be based on something less magic.
	float xyRatio = (640.0f/480.0f)*1.03f; //compensate for aspect ratio
	//float baseScaling = 0.23f;
	float baseScaling = 0.23f*(cam->net.modelScale[0]/45.0f);
	float a[3], b[3];
	float t[3];
	float dLen, camLen;

	//calculate an appropriate distance on the exact near frustum plane normal
	a[0] = camUsePos[0]*g_camFrustum.planes[4][0];
	a[1] = camUsePos[1]*g_camFrustum.planes[4][1];
	a[2] = camUsePos[2]*g_camFrustum.planes[4][2];
	b[0] = obj->net.pos[0]*g_camFrustum.planes[4][0];
	b[1] = obj->net.pos[1]*g_camFrustum.planes[4][1];
	b[2] = obj->net.pos[2]*g_camFrustum.planes[4][2];
	Math_VecSub(a, b, t);
	dLen = Math_VecLen(t);

	//calculate x confine based on the inward distance of the object
	a[0] = dLen*g_camFrustum.planes[2][0];
	a[1] = dLen*g_camFrustum.planes[2][1];
	a[2] = dLen*g_camFrustum.planes[2][2];
	b[0] = dLen*g_camFrustum.planes[3][0];
	b[1] = dLen*g_camFrustum.planes[3][1];
	b[2] = dLen*g_camFrustum.planes[3][2];
	Math_VecSub(a, b, t);
	camLen = Math_VecLen(t);
	dMin[0] = -(camLen*baseScaling);
	dMax[0] = (camLen*baseScaling);

	//calculate y confine based on the inward distance of the object
	a[0] = dLen*g_camFrustum.planes[0][0];
	a[1] = dLen*g_camFrustum.planes[0][1];
	a[2] = dLen*g_camFrustum.planes[0][2];
	b[0] = dLen*g_camFrustum.planes[1][0];
	b[1] = dLen*g_camFrustum.planes[1][1];
	b[2] = dLen*g_camFrustum.planes[1][2];
	Math_VecSub(a, b, t);
	camLen = Math_VecLen(t);
	dMin[1] = -(camLen*baseScaling*xyRatio);
	dMax[1] = (camLen*baseScaling*xyRatio);

	if (g_fetusSideMode)
	{
		dMin[2] = dMin[0];
		dMax[2] = dMax[0];
		dMin[0] = dMin[1];
		dMax[0] = dMax[1];
		dMin[2] -= obj->net.mins[2]*4.0f;
		dMax[2] -= obj->net.maxs[2]*4.0f;
		dMin[0] -= obj->net.mins[0]*4.0f;
		dMax[0] -= obj->net.maxs[0]*4.0f;
	}
	else
	{
		dMin[0] -= obj->net.mins[0]*4.0f;
		dMax[0] -= obj->net.maxs[0]*4.0f;
		dMin[1] -= obj->net.mins[1]*4.0f;
		dMax[1] -= obj->net.maxs[1]*4.0f;
	}
	int i = 0;
	while (i < 2)
	{
		if (g_fetusSideMode && i == 1)
		{
			i = 2;
		}
		float min = dMin[i];
		float max = dMax[i];
		if (obj->net.pos[i] < camUsePos[i])
		{
			if ((obj->net.pos[i]-camUsePos[i]) < min)
			{
				obj->net.pos[i] = camUsePos[i]+min;
			}
		}
		else if (obj->net.pos[i] > camUsePos[i])
		{
			if ((obj->net.pos[i]-camUsePos[i]) > max)
			{
				obj->net.pos[i] = camUsePos[i]+max;
			}
		}
		i++;
	}
}

//handle firing a weapon
void ObjPlayer_HandleWeapon(gameObject_t *obj, float timeMod, fetalWeapon_t *weap,
							fetusLiveState_t *fls)
{
	if (fls->clWeaponAmmo == 0)
	{
		return;
	}

	if (fls->clWeaponDelay < g_curTime)
	{
		if (weap->nonDepressCount == -1 ||
			fls->clWeaponPressShots < weap->nonDepressCount)
		{
			float projPos[3];
			float projDir[3];
			float fwd[3], right[3], up[3];

			Math_VecCopy(obj->net.pos, projPos);

			Math_AngleVectors(obj->net.ang, fwd, right, up);
			Math_VecCopy(fwd, projDir);

            projPos[0] += fwd[0]*weap->muzzleOffset[0];
			projPos[1] += fwd[1]*weap->muzzleOffset[0];
			projPos[2] += fwd[2]*weap->muzzleOffset[0];

            projPos[0] += right[0]*weap->muzzleOffset[1];
			projPos[1] += right[1]*weap->muzzleOffset[1];
			projPos[2] += right[2]*weap->muzzleOffset[1];

            projPos[0] += up[0]*weap->muzzleOffset[2];
			projPos[1] += up[1]*weap->muzzleOffset[2];
			projPos[2] += up[2]*weap->muzzleOffset[2];

			ObjProjectile_Create(obj, weap->projectile, projPos, projDir);

			fls->clWeaponPressShots++;
			fls->clWeaponDelay = g_curTime + weap->shotDelay;
			fls->clWeaponDepressDur = g_curTime + weap->depressPause;
			if (g_musical)
			{ //musical hax
				fls->clWeaponDepressDur = 0;
				fls->clWeaponDelay = g_curTime + weap->shotDelay/4;
			}
			if (fls->clWeaponAmmo > 0)
			{
				fls->clWeaponAmmo--;
				obj->net.plAmmo = fls->clWeaponAmmo;
			}
		}
		else if (fls->clWeaponDepressDur < g_curTime)
		{
			fls->clWeaponPressShots = 0;
		}
	}
}

//weapon logic
void ObjPlayer_Weapons(gameObject_t *obj, float timeMod)
{
	if (obj->clButtons[BUTTON_ACTION])
	{
		if (obj->fls.clWeapon >= 0 && obj->fls.clWeapon < NUM_FETAL_WEAPONS)
		{
			fetalWeapon_t *weap = &g_fetalWeapons[obj->fls.clWeapon];
			ObjPlayer_HandleWeapon(obj, timeMod, weap, &obj->fls);
		}
	}
	else
	{
		obj->fls.clWeaponPressShots = 0;
	}

	if (obj->clButtons[BUTTON_JUMP])
	{
		if (obj->altFls.clWeapon >= 0 && obj->altFls.clWeapon < NUM_FETAL_WEAPONS)
		{
			fetalWeapon_t *weap = &g_fetalWeapons[obj->altFls.clWeapon];
			ObjPlayer_HandleWeapon(obj, timeMod, weap, &obj->altFls);
		}
	}
	else
	{
		obj->altFls.clWeaponPressShots = 0;
	}
}

float g_worldMins[3], g_worldMaxs[3];

//check analog input movement
void ObjPlayer_AnalogInputFPS(gameObject_t *obj, float &move, float &moveRight,
							  const float moveSpeed, const float timeMod)
{
	if (!obj->hasClAnalog)
	{
		return;
	}

	float moveDir[4];
	for (int i = 0; i < 4; i++)
	{
		const float minMove = 0.2f;
		moveDir[i] = (float)((int)obj->clAnalog[i] - 32767)/32767.0f;
		if (fabsf(moveDir[i]) < minMove)
		{
			moveDir[i] = 0.0f;
		}
	}

	if (moveDir[0] != 0.0f)
	{
		moveRight = moveDir[0]*moveSpeed;
	}
	if (moveDir[1] != 0.0f)
	{
		move = -moveDir[1]*moveSpeed;
	}

	obj->net.ang[0] += moveDir[3]*(5.0f*timeMod);
	if (obj->net.ang[0] > 90.0f)
	{
		obj->net.ang[0] = 90.0f;
	}
	else if (obj->net.ang[0] < -90.0f)
	{
		obj->net.ang[0] = -90.0f;
	}
	obj->net.ang[1] -= moveDir[2]*(5.0f*timeMod);
}

//check analog input movement (shooter mode)
void ObjPlayer_AnalogInputShooter(gameObject_t *obj, float &move, float &moveRight, float &moveAng,
								  const float moveSpeed, const float timeMod)
{
	if (!obj->hasClAnalog)
	{
		return;
	}

	float moveDir[4];
	for (int i = 0; i < 4; i++)
	{
		float minMove = (i >= 2) ? 0.1f : 0.2f;
		moveDir[i] = (float)((int)obj->clAnalog[i] - 32767)/32767.0f;
		if (fabsf(moveDir[i]) < minMove)
		{
			moveDir[i] = 0.0f;
		}
	}

	const float analogMoveSpeed = moveSpeed*1.5f;
	if (moveDir[0] || moveDir[1])
	{
		moveAng = 0.0f;
		moveRight = moveDir[0]*analogMoveSpeed;
		move = -moveDir[1]*analogMoveSpeed;
	}

	if (moveDir[2] || moveDir[3])
	{
		float moveXY[2] = {moveDir[2], moveDir[3]};
		float dist = Math_VecNorm2(moveXY);
		if (dist > 0.3f)
		{
			const float up[2] = {0.0f, -1.0f};
			float dot2D = moveXY[0]*up[0] + moveXY[1]*up[1];
			float ang = acosf(dot2D);
			if (moveDir[2] > 0.0f)
			{
				ang = -ang;
			}
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], ang*57.2957795785f, timeMod*0.5f);
		}
	}
}

//frame function
void ObjPlayer_Think(gameObject_t *obj, float timeMod)
{
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	const float moveBlendScale = 0.4f;//0.1f;
	const float angleBlendScale = 0.1f;

	if (g_fpsMode && obj->net.index != 0)
	{ //other players just spectate in fps mode
		Math_VecCopy(g_gameObjects[0].net.pos, obj->net.pos);
		Math_VecCopy(g_gameObjects[0].net.ang, obj->net.ang);
		obj->net.solid = 0;
		LServ_UpdateRClip(obj);
		return;
	}

	if (obj->net.lives <= 0 && !g_musical)
	{ //freeze
		return;
	}

	if (obj->debounce2 > g_curTime)
	{ //invincible
		obj->net.renderEffects |= FXFL_SHELLRED;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_SHELLRED;
	}

	if (g_runningMultiplayer)
	{
		if (obj->net.index > 0)
		{ //subsequent players
			obj->net.renderEffects &= ~FXFL_SHELL1;
			obj->net.renderEffects |= FXFL_SHELL2;
		}
		else
		{ //player 1
			obj->net.renderEffects |= FXFL_SHELL1;
			obj->net.renderEffects &= ~FXFL_SHELL2;
		}
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_SHELL1;
		obj->net.renderEffects &= ~FXFL_SHELL2;
	}

	if (!g_fpsMode)
	{
		if (cam->inuse)
		{
			g_fetusPilotZ = cam->camTrackPos[2]-2048.0f;
		}
		if (obj->fls.playerSpawnSeq)
		{
			if (obj->net.index == 0)
			{
				float plStartX = cam->camTrackPos[0]-700.0f;
				if (obj->net.pos[0] < plStartX)
				{
					obj->net.pos[0] += 64.0f*timeMod;
					if (obj->net.pos[0] > plStartX)
					{
						obj->net.pos[0] = plStartX;
					}
				}
				if (obj->net.pos[2] < g_fetusPilotZ)
				{
					obj->net.pos[2] += 64.0f*timeMod;
					if (obj->net.pos[2] > g_fetusPilotZ)
					{
						obj->net.pos[2] = g_fetusPilotZ;
					}
				}
				if (fabsf(obj->net.pos[0]-plStartX) <= 0.1f && fabsf(obj->net.pos[2]-g_fetusPilotZ) <= 0.1f &&
					obj->net.ang[ROLL] == 0.0f)
				{
					obj->fls.playerSpawnSeq = false;
					g_levelScroll[0] = 16.0f;
					g_levelScroll[1] = 16.0f;
					g_levelScroll[2] = 16.0f;
				}
				else
				{
					return;
				}
			}
			else
			{
				obj->fls.playerSpawnSeq = false;
			}
		}
		obj->net.solid = 1;
	}
	else
	{ //fps mode
		gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
		if (cam->inuse)
		{
			if (obj->fls.playerSpawnSeq)
			{
				obj->net.pos[0] = g_initialFPSPos[0];
				obj->net.pos[1] = g_initialFPSPos[1];
				obj->net.pos[2] = g_initialFPSPos[2];
				obj->net.ang[0] = g_initialFPSAng[0];
				obj->net.ang[1] = g_initialFPSAng[1];
				obj->net.ang[2] = g_initialFPSAng[2];
				Math_VecCopy(obj->net.pos, obj->safePos);
				g_sharedFn->Coll_GetWorldBounds(g_worldMins, g_worldMaxs);
				obj->fls.playerSpawnSeq = false;
			}

			//terrible hackery to prevent player from falling out of world
			float plAbsMins[3], plAbsMaxs[3];
			for (int k = 0; k < 3; k++)
			{
				plAbsMins[k] = obj->net.pos[k]+obj->net.mins[k];
				plAbsMaxs[k] = obj->net.pos[k]+obj->net.maxs[k];
			}
			if (!Math_BoxesOverlap(plAbsMins, plAbsMaxs, g_worldMins, g_worldMaxs))
			{
				obj->net.pos[0] = g_initialFPSPos[0];
				obj->net.pos[1] = g_initialFPSPos[1];
				obj->net.pos[2] = g_initialFPSPos[2];
				obj->net.ang[0] = g_initialFPSAng[0];
				obj->net.ang[1] = g_initialFPSAng[1];
				obj->net.ang[2] = g_initialFPSAng[2];
				Math_VecCopy(obj->net.pos, obj->safePos);
			}
		}
		//obj->net.solid = 0;
	}

	//g_fetusPilotZ -= 64.0f;

	if (g_fpsMode)
	{
		float moveSpeed = 230.0f;//48.0f;
		float turnRate = 5.0f; //10.0f
		float pitchTurnRate = 2.5f; //5.0f
		float move = 0.0f, rightMove = 0.0f;
		float fwd[3], right[3];

		if (!obj->onGround)
		{
			moveSpeed *= 0.0625f;
		}
		else
		{
			if (obj->clButtons[BUTTON_JUMP])
			{
				obj->net.vel[2] = 500.0f;
			}
		}

		if (obj->clButtons[BUTTON_UP])
		{ //up
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_DOWN])
		{ //down
			move = -moveSpeed;
		}
		if (obj->clButtons[BUTTON_RIGHT])
		{ //right
			rightMove = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_LEFT])
		{ //left
			rightMove = -moveSpeed;
		}

		int turnAxis = YAW;
		if (obj->clButtons[BUTTON_EVENT2])
		{ //turn fetus
			obj->net.ang[turnAxis] += turnRate*timeMod;		
		}
		if (obj->clButtons[BUTTON_EVENT3])
		{ //turn fetus
			obj->net.ang[turnAxis] -= turnRate*timeMod;		
		}

		if (obj->clButtons[BUTTON_EVENT5])
		{ //fps mode - look down
			obj->net.ang[0] += pitchTurnRate*timeMod;
			if (obj->net.ang[0] > 90.0f)
			{
				obj->net.ang[0] = 90.0f;
			}
		}
		if (obj->clButtons[BUTTON_EVENT4])
		{ //fps mode - look up
			obj->net.ang[0] -= pitchTurnRate*timeMod;
			if (obj->net.ang[0] < -90.0f)
			{
				obj->net.ang[0] = -90.0f;
			}
		}

		if (obj->hasLastClAngles)
		{
			for (int n = 0; n < 3; n++)
			{
				obj->net.ang[n] += (obj->clAngles[n]-obj->lastClAngles[n]);
				obj->lastClAngles[n] = obj->clAngles[n];
			}
			if (obj->net.ang[0] > 90.0f)
			{
				obj->net.ang[0] = 90.0f;
			}
			else if (obj->net.ang[0] < -90.0f)
			{
				obj->net.ang[0] = -90.0f;
			}
		}

		if (move && rightMove)
		{
			move *= 0.5f;
			rightMove *= 0.5f;
		}

		ObjPlayer_AnalogInputFPS(obj, move, rightMove, moveSpeed, timeMod);

		//compensate for time
		move *= timeMod;
		rightMove *= timeMod;
		float moveAng[3];
		moveAng[0] = 0.0f;
		moveAng[1] = obj->net.ang[1];
		moveAng[2] = 0.0f;

		Math_AngleVectors(moveAng, fwd, right, 0);
		/*
		obj->net.pos[0] += (fwd[0]*move);
		obj->net.pos[1] += (fwd[1]*move);
		obj->net.pos[2] += (fwd[2]*move);
		obj->net.pos[0] += (right[0]*rightMove);
		obj->net.pos[1] += (right[1]*rightMove);
		obj->net.pos[2] += (right[2]*rightMove);
		*/
		obj->net.vel[0] += (fwd[0]*move);
		obj->net.vel[1] += (fwd[1]*move);
		obj->net.vel[2] += (fwd[2]*move);
		obj->net.vel[0] += (right[0]*rightMove);
		obj->net.vel[1] += (right[1]*rightMove);
		obj->net.vel[2] += (right[2]*rightMove);
	}
	else if (obj->debounce < g_curTime)
	{ //shooter mode
		float mus = 0.0f;
		float musMove = 0.0f;
		if (g_musical)
		{
			mus = (g_musicalRunningAvg/(float)g_musicalRunningCount)*10.0f;
			musMove = (g_musicalRunningAvg/(float)g_musicalRunningCount)*52.0f;
		}

		float oldPos[3];
		oldPos[0] = obj->net.pos[0];
		oldPos[1] = obj->net.pos[1];
		oldPos[2] = obj->net.pos[2];
		float moveAng = 0.0f;
		float move = 0.0f;
		float moveSpeed = 52.0f+musMove;//48.0f;//28.0f;//50.0f;
		float turnRate = 10.0f+mus;//5.0f;

		if (obj->clButtons[BUTTON_UP] && obj->clButtons[BUTTON_LEFT])
		{ //up-left
			moveAng = 45.0f;
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_UP] && obj->clButtons[BUTTON_RIGHT])
		{ //up-right
			moveAng = -45.0f;
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_DOWN] && obj->clButtons[BUTTON_LEFT])
		{ //down-left
			moveAng = 180.0f-45.0f;
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_DOWN] && obj->clButtons[BUTTON_RIGHT])
		{ //down-right
			moveAng = -(180.0f-45.0f);
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_UP])
		{ //up
			moveAng = 0.0f;
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_DOWN])
		{ //down
			moveAng = 180.0f;
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_RIGHT])
		{ //right
			moveAng = -90.0f;
			move = moveSpeed;
		}
		else if (obj->clButtons[BUTTON_LEFT])
		{ //left
			moveAng = 90.0f;
			move = moveSpeed;
		}

		int turnAxis = YAW;

		if (g_fetusSideMode)
		{
			turnRate = -turnRate;
			turnAxis = PITCH;
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], 0.0f, timeMod*angleBlendScale);
		}
		else
		{
			obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], 0.0f, timeMod*angleBlendScale);
		}

		if (g_noConfineTime < g_curTime)
		{
			if (obj->clButtons[BUTTON_EVENT2])
			{ //turn fetus
				obj->net.ang[turnAxis] += turnRate*timeMod;		
			}
			if (obj->clButtons[BUTTON_EVENT3])
			{ //turn fetus
				obj->net.ang[turnAxis] -= turnRate*timeMod;		
			}

			//todo - use y factor somehow
			if (obj->hasLastClAngles)
			{
				if (obj->net.ang[turnAxis] >= 0.0f &&
					obj->net.ang[turnAxis] < 180.0f)
				{
					//obj->net.ang[turnAxis] += (obj->clAngles[0]-obj->lastClAngles[0]);
					obj->net.ang[turnAxis] += (obj->clAngles[1]-obj->lastClAngles[1]);
				}
				else
				{
					//obj->net.ang[turnAxis] -= (obj->clAngles[0]-obj->lastClAngles[0]);
					obj->net.ang[turnAxis] += (obj->clAngles[1]-obj->lastClAngles[1]);
				}
				for (int n = 0; n < 3; n++)
				{
					obj->lastClAngles[n] = obj->clAngles[n];
				}
			}
		}

		int i = 0;
		while (i < 3)
		{
			if (obj->net.ang[i] < 0.0f)
			{
				obj->net.ang[i] += 360.0f;
			}
			else if (obj->net.ang[i] > 360.0f)
			{
				obj->net.ang[i] -= 360.0f;
			}
			i++;
		}

		float rightMove = 0.0f;
		ObjPlayer_AnalogInputShooter(obj, move, rightMove, moveAng, moveSpeed, timeMod);

		//compensate for time
		move *= timeMod;
		rightMove *= timeMod;

		if (g_noConfineTime < g_curTime)
		{
			float fwd[3];
			float ang[3];

			if (g_fetusSideMode)
			{
				ang[ROLL] = 0.0f;
				ang[YAW] = 0.0f;
				ang[PITCH] = moveAng+90;
			}
			else
			{
				ang[ROLL] = 0.0f;
				ang[YAW] = moveAng;
				ang[PITCH] = 0.0f;
			}

			float right[3];
			if (g_fetusSideMode)
			{
				Math_AngleVectors(ang, fwd, right, 0);
				fwd[2] = -fwd[2];
			}
			else
			{
				Math_AngleVectors(ang, fwd, right, 0);
			}
			obj->net.pos[0] += (fwd[0]*move);
			obj->net.pos[1] += (fwd[1]*move);
			obj->net.pos[2] += (fwd[2]*move);
			obj->net.pos[0] += (right[0]*rightMove);
			obj->net.pos[1] += (right[1]*rightMove);
			obj->net.pos[2] += (right[2]*rightMove);

			collObj_t col;
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, oldPos, obj->net.pos, obj->radius, NULL, false);
			if ((col.hit || col.containsSolid) && col.hitObjectIndex >= 0 &&
				col.hitObjectIndex < MAX_GAME_OBJECTS)
			{
				gameObject_t *other = &g_gameObjects[col.hitObjectIndex];
				if (other->inuse)
				{
					Util_TouchObjects(obj, other, &col);
				}
			}
		}
	}

	if (!g_fpsMode)
	{
		if (g_noConfineTime >= g_curTime && cam->inuse)
		{
			const float catchupMoveBlendScale = 0.07f;
			if (!g_fetusSideMode)
			{
				obj->net.pos[0] = Util_NudgeValue(obj->net.pos[0], cam->net.pos[0], timeMod*catchupMoveBlendScale);
				obj->net.pos[1] = Util_NudgeValue(obj->net.pos[1], cam->net.pos[1], timeMod*catchupMoveBlendScale);
			}
			else
			{
				obj->net.pos[0] = Util_NudgeValue(obj->net.pos[0], cam->net.pos[0], timeMod*catchupMoveBlendScale);
				obj->net.pos[2] = Util_NudgeValue(obj->net.pos[2], cam->net.pos[2], timeMod*catchupMoveBlendScale);
			}
		}

		if (!g_fetusSideMode)
		{
			obj->net.pos[2] = Util_NudgeValue(obj->net.pos[2], g_fetusPilotZ, timeMod*moveBlendScale);
		}
		else
		{
			obj->net.pos[1] = Util_NudgeValue(obj->net.pos[1], g_fetusSideModeX, timeMod*moveBlendScale);
		}

		ObjPlayer_CameraConfine(obj);
	}

	const float plRad = obj->radius;
	if (!obj->fls.playerSpawnSeq)
	{ //translation safety checking
		if (g_fpsMode)
		{
			Phys_ApplyObjectPhysics(obj, timeMod, plRad, 80.0f, 0.0f);

			collObj_t col;
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, obj->net.pos, obj->net.pos, plRad, NULL, true);
			if (!col.containsSolid)
			{
				Math_VecCopy(obj->net.pos, obj->safePos);
			}
			else
			{
				g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, obj->net.pos, obj->net.pos, plRad*0.5f, NULL, true);
				if (col.containsSolid)
				{ //ended up in solid
					Math_VecCopy(obj->safePos, obj->net.pos);
				}
			}
		}
	}

	if (g_fpsMode)
	{
		if (cam->inuse && obj->net.index == 0)
		{
			ObjVisBlock_Update(obj->net.pos);
			Math_VecCopy(obj->net.pos, cam->net.pos);
			cam->net.pos[2] += plRad*0.5f;
			Math_VecCopy(obj->net.ang, cam->net.ang);
			cam->net.ang[0] += 90.0f;
		}
	}

	if (g_noConfineTime < g_curTime && obj->debounce < g_curTime && !g_fpsMode)
	{
		//perform weapon logic after all movement has been completed
		ObjPlayer_Weapons(obj, timeMod);
	}

	LServ_UpdateRClip(obj);
}
