/*
=============================================================================
Module Information
------------------
Name:			main.cpp
Author:			Rich Whitehouse
Description:	core server logic module implementation.
=============================================================================
*/

#include "main.h"

float g_timeFactor = 0.0f;
float g_timeScale = 1.0f;
float g_invTimeScale = 1.0f;
sharedSVFunctions_t *g_sharedFn;
gameValues_t g_gameValues;

bool g_musical = false;
int g_musicStrIndex = -1;
float g_musicalAmp = 0.0f;
float g_musicalRunningAvg = 0.0f;
float g_musicalLastFrame = 0.0f;
unsigned long g_musicalSpawnTime = 0;
int g_musicalRunningCount = 1;
unsigned long g_musicReset = 0;
int g_musicHighScore = 0;
int g_musicHighScorePrev = 0;
char g_musicResetName[4096];
char g_musicResetNameNext[4096];

unsigned long g_curTime = 0;
bool g_runningMultiplayer = false;

//fetus blaster global things
float g_fetusPilotZ = 512.0f;
float g_levelScroll[3] = {0.0f, 0.0f, 0.0f};

bool g_fetusSideMode = false;
float g_fetusSideModeX = 0.0f;

gameObject_t g_gameObjects[MAX_GAME_OBJECTS];
clientInfo_t g_clientInfo[MAX_NET_CLIENTS];
int g_gameObjectSlots = 0;

//return api version
int LServ_GetAPIVersion(void)
{
	return LOGIC_API_VERSION;
}

//relink all objects
void LServ_Relink(void)
{
#ifdef _RC_NOVODEX
	int i = 0;
	while (i < g_gameObjectSlots)
	{
		if (g_gameObjects[i].inuse &&
			g_gameObjects[i].nxActor &&
			g_gameObjects[i].net.solid &&
			!g_gameObjects[i].staticphys)
		{
			//first, link it to the world
			g_sharedFn->Common_NxLinkContact(g_gameObjects[i].nxActor, NULL);

			int j = 0;
			while (j < g_gameObjectSlots)
			{
				if (g_gameObjects[j].inuse &&
					g_gameObjects[j].nxActor &&
					g_gameObjects[j].net.solid)
				{
					g_sharedFn->Common_NxLinkContact(g_gameObjects[i].nxActor, g_gameObjects[j].nxActor);
				}
				j++;
			}
		}
		i++;
	}
#endif
}

//find one that isn't in use
gameObject_t *LServ_CreateObject(void)
{
	int i = MAP_OBJ_NUM+1;
	while (i < MAX_GAME_OBJECTS)
	{
		if (!g_gameObjects[i].inuse && g_gameObjects[i].noReuseTime <= g_curTime)
		{
			g_gameObjects[i].inuse = INUSE_ACTIVE;
			g_gameObjects[i].net.index = i;
			g_gameObjects[i].net.owner = MAP_OBJ_NUM;
			g_gameObjects[i].net.staticIndex = -1;

			if (g_gameObjectSlots <= i)
			{
				g_gameObjectSlots = i+1;
			}

			return &g_gameObjects[i];
		}
		i++;
	}

	return NULL;
}

//kill it
void LServ_FreeObject(gameObject_t *obj)
{
#ifdef _RC_NOVODEX
	if (obj->nxActor)
	{
		g_sharedFn->Common_FreeNxActor(obj->nxActor);
	}
#endif
	if (obj->rcColModel)
	{
		g_sharedFn->Coll_DestroyModelInstance(obj->rcColModel);
	}
	if (obj->visBlockData)
	{
		g_sharedFn->Common_RCFree(obj->visBlockData);
	}
	memset(obj, 0, sizeof(gameObject_t));
	obj->noReuseTime = g_curTime + 1000; //don't reuse it so that the client can get an update on its removal first
}

//cheesy function to get a radius from a bounds for very quick frustum checking
float LServ_RadiusFromBounds(float *mins, float *maxs, float padMul)
{
	float b = 0.0f;
	for (int i = 0; i < 3; i++)
	{
		if (fabsf(mins[i]) > b)
		{
			b = fabsf(mins[i]);
		}
		if (maxs[i] > b)
		{
			b = maxs[i];
		}
	}

	return b*padMul;
}

//flag INUSE_NONET based on visibility within camera frustum
void LServ_NetCull(gameObject_t *obj)
{
	if (!g_camFrustumFresh)
	{
		return;
	}

	if (obj->net.type == OBJ_TYPE_VISBLOCK)
	{ //don't attempt to cull these
		return;
	}

	if (obj->localFlags & LFL_NONET)
	{ //never send
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (obj->spawnMins[0] == 0.0f &&
		obj->spawnMins[1] == 0.0f &&
		obj->spawnMins[2] == 0.0f &&
		obj->spawnMaxs[0] == 0.0f &&
		obj->spawnMaxs[1] == 0.0f &&
		obj->spawnMaxs[2] == 0.0f)
	{ //spawn bounds not set
		return;
	}

	if (!Math_PointInFrustum(&g_wideCamFrustum, obj->net.pos, Util_RadiusFromBounds(obj->spawnMins, obj->spawnMaxs, 2.0f)))
	{ //not in frustum
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (!ObjVisBlock_CheckVis(obj))
	{ //visibility system culled it
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (!obj->rcColModel || !g_sharedFn->Coll_IsTreeModel(obj->rcColModel))
	{
		gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
		float rad = Math_Max3(obj->spawnMaxs[0]-obj->spawnMins[0],
								obj->spawnMaxs[1]-obj->spawnMins[1],
								obj->spawnMaxs[2]-obj->spawnMins[2])*0.5f;
		float midPos[3];
		midPos[0] = obj->net.pos[0] + obj->spawnMins[0] + (obj->spawnMaxs[0]-obj->spawnMins[0])*0.5f;
		midPos[1] = obj->net.pos[1] + obj->spawnMins[1] + (obj->spawnMaxs[1]-obj->spawnMins[1])*0.5f;
		midPos[2] = obj->net.pos[2] + obj->spawnMins[2] + (obj->spawnMaxs[2]-obj->spawnMins[2])*0.5f;
		if (!g_sharedFn->Coll_GeneralVisibility(midPos, cam->net.pos, rad))
		{ //sphere visibility failed
			obj->inuse |= INUSE_NONET;
			return;
		}
	}

	//visible
	obj->inuse &= ~INUSE_NONET;
}

//update an object's rclip
void LServ_UpdateRClip(gameObject_t *obj)
{
	if (!obj || !obj->rcColModel)
	{
		return;
	}
	Math_VecCopy(obj->net.mins, obj->rcColModel->mins);
	Math_VecCopy(obj->net.maxs, obj->rcColModel->maxs);
	Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
	Math_VecCopy(obj->net.ang, obj->rcColModel->ang);
	Math_VecCopy(obj->net.modelScale, obj->rcColModel->modelScale);
	obj->rcColModel->gameOwner = obj->net.index;
	obj->rcColModel->solid = obj->net.solid;
	obj->rcColModel->frame = obj->net.frame;
	g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
}

//spawn an enemy object
void LServ_SpawnEnemy(const char *name, float *pos)
{
	float ang[3];
	ang[0] = 0.0f;
	ang[1] = 0.0f;
	ang[2] = 0.0f;
	LServ_ObjectFromName(name, pos, ang, NULL, 0);
	//optionally set extra object parameters
}

//spawn relative to camera in a spread
void LServ_SpawnEnemySpread(const char *name)
{
	if (!name)
	{
		int r = rand()%13;//9;
		if (r >= 6)
		{
			name = "obj_gen_fighter01";
		}
		else if (r <= 3)
		{
			name = "obj_gen_fighterspin";
		}
		else
		{
			name = "obj_gen_fightershooter";
		}
	}
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	if (!cam->inuse || !cam->target || !cam->target->inuse)
	{
		return;
	}

	float d[3];
	Math_VecSub(cam->target->net.pos, cam->camTrackPos, d);
	
	Math_VecNorm(d);
	float a[3], right[3];
	Math_VecToAngles(d, a);
	Math_AngleVectors(a, 0, right, 0);

	float r = -1500.0f + (float)(rand()%3000);
	Math_VecScale(d, 1500.0f);
	d[0] += cam->camTrackPos[0] + right[0]*r;
	d[1] += cam->camTrackPos[1] + right[1]*r;
	d[2] += g_fetusPilotZ + right[2]*r;

	LServ_SpawnEnemy(name, d);
}

//in musical mode, spawn enemies constantly
void LServ_MusicalSpawns(void)
{
	static bool nextInBatch = false;
	if ((g_musicalAmp-g_musicalLastFrame) > 0.4f)
	{ //the music went up rapidly between frames, spawn a batch
		nextInBatch = true;
	}

	if (g_musicalSpawnTime > g_curTime)
	{ //not long enough since the last spawn
		return;
	}

	if (g_musicalAmp < 0.05f)
	{ //don't spawn during silent periods
		return;
	}

	int activeEnemies = 0;
	for (int i = MAX_NET_CLIENTS; i < g_gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->hurtable && obj->health > 0 && obj->net.solid)
		{
			activeEnemies++;
			if (activeEnemies > 15)
			{ //start killing the bastards
				Util_DamageObject(obj, obj, 9999999);
			}
		}
	}

	LServ_SpawnEnemySpread(NULL);
	if (nextInBatch)
	{
		LServ_SpawnEnemySpread(NULL);
		LServ_SpawnEnemySpread(NULL);
		LServ_SpawnEnemySpread(NULL);
		nextInBatch = false;
	}
	else if ((g_musicalRunningAvg/(float)g_musicalRunningCount) > 0.37f)
	{
		LServ_SpawnEnemySpread(NULL);
		LServ_SpawnEnemySpread(NULL);
	}

	float modAmp = g_musicalAmp*1.4f;
	if (modAmp > 1.0f)
	{
		modAmp = 1.0f;
	}
	unsigned long spawnTime = (unsigned long)((1.0f-modAmp)*3000.0f);
	if (spawnTime < 50)
	{
		spawnTime = 50;
	}
	g_musicalSpawnTime = g_curTime+spawnTime;
}

//get existing data
char *LServ_GetHighScoreData(void)
{
	int fileHandle = g_sharedFn->FileSys_OpenFile("assets/persdata/musicalscores.txt", _O_RDONLY, _S_IREAD);
	if (fileHandle == -1)
	{
		return NULL;
	}

	int l = g_sharedFn->FileSys_GetLen(fileHandle);
	if (l <= 0)
	{
		g_sharedFn->FileSys_CloseFile(fileHandle);
		return NULL;
	}
	char *scoreData = (char *)g_sharedFn->Common_RCMalloc(l+1);
	g_sharedFn->FileSys_ReadFile(fileHandle, scoreData, l);
	g_sharedFn->FileSys_CloseFile(fileHandle);
	scoreData[l] = 0;

	return scoreData;
}

//write new highscore data
void LServ_SetHighScoreData(char *scoreData)
{
	int fileHandle = g_sharedFn->FileSys_OpenFile("assets/persdata/musicalscores.txt", _O_WRONLY|_O_CREAT|_O_TRUNC, _S_IWRITE);
	if (fileHandle == -1)
	{
		return;
	}

	g_sharedFn->FileSys_WriteFile(fileHandle, scoreData, (int)strlen(scoreData));
	g_sharedFn->FileSys_CloseFile(fileHandle);
}

//get the highscore for this media
int LServ_GetMusicHighScore(char *musicName)
{
	char *scoreData = LServ_GetHighScoreData();
	if (!scoreData)
	{
		return 0;
	}

	char scoreStr[4096];
	sprintf(scoreStr, "\n%s\n", musicName);
	char *p = strstr(scoreData, scoreStr);
	int score = 0;
	if (p)
	{
		strcat(scoreStr, "%i\n");
		sscanf(p, scoreStr, &score);
	}
	g_sharedFn->Common_RCFree(scoreData);
	return score;
}

//record highscore
void LServ_SaveNewHighscore(char *musicName)
{
	char scoreStr[4096];
	sprintf(scoreStr, "\n%s\n", musicName);

	char *scoreData = LServ_GetHighScoreData();
	if (!scoreData)
	{
		sprintf(scoreStr, "\n%s\n%i\n", musicName, g_musicHighScore);
		LServ_SetHighScoreData(scoreStr);
		return;
	}

	char *p = strstr(scoreData, scoreStr);
	if (!p)
	{
		sprintf(scoreStr, "\n%s\n%i\n", musicName, g_musicHighScore);
		char *out = (char *)g_sharedFn->Common_RCMalloc((int)strlen(scoreData)+(int)strlen(scoreStr)+1);
		strcpy(out, scoreData);
		strcat(out, scoreStr);
		LServ_SetHighScoreData(out);
		g_sharedFn->Common_RCFree(scoreData);
		g_sharedFn->Common_RCFree(out);
		return;
	}

	size_t l = p-scoreData;
	int numEnd = 0;
	while (numEnd < 3)
	{
		if (!*p)
		{
			break;
		}
		if (*p == '\n')
		{
			numEnd++;
		}
		p++;
	}
	sprintf(scoreStr, "\n%s\n%i\n", musicName, g_musicHighScore);
	size_t preSize = l;
	l = l + strlen(p) + strlen(scoreStr);
	char *out = (char *)g_sharedFn->Common_RCMalloc((int)l+4);
	memcpy(out, scoreData, preSize);
	*(out+preSize) = 0;
	strcat(out, scoreStr);
	strcat(out, p);

	LServ_SetHighScoreData(out);
	g_sharedFn->Common_RCFree(out);
	g_sharedFn->Common_RCFree(scoreData);
}

//check the new score against records and such
void LServ_MusicalScoreLog(gameObject_t *client)
{
	/*
	if (client->net.lives > g_musicHighScore)
	{
		g_musicHighScore = client->net.lives;
		g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), -1);
	}
	*/
	//wait until after the song is done to update i guess, in case they started screwing up and dying a lot
}

//reset music stuff
void LServ_MusicReset(void)
{
	g_musicReset = 0;
	g_musicalRunningAvg = 0.0f;
	g_musicalLastFrame = 0.0f;
	g_musicalSpawnTime = 0;
	g_musicalRunningCount = 1;

	char str[1024];
	int plNum = 1;
	bool wasPlayingMusical = (g_musicResetName[0] != 0);
	if (wasPlayingMusical)
	{
		sprintf(str, "Results for %s:", g_musicResetName);
		g_sharedFn->Net_SendMessage(str);
	}
	int highestScore = -99999999;
	int winningClient = 0;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (pl->inuse)
		{
			if (wasPlayingMusical)
			{
				if (pl->net.lives > g_musicHighScorePrev)
				{
					sprintf(str, "Player %i (%s) destroyed the best score with %i.", plNum, g_clientInfo[pl->net.index].infoStr, pl->net.lives);
					winningClient = i;
				}
				else if (pl->net.lives >= 0)
				{
					sprintf(str, "Player %i (%s) finished with a score of %i.", plNum, g_clientInfo[pl->net.index].infoStr, pl->net.lives);
				}
				else
				{
					sprintf(str, "Player %i (%s) failed miserably with a score of %i.", plNum, g_clientInfo[pl->net.index].infoStr, pl->net.lives);
				}
				g_sharedFn->Net_SendMessage(str);
			}
			if (pl->net.lives > highestScore)
			{
				highestScore = pl->net.lives;
			}
			pl->net.lives = 0;
			pl->musicalKills = 0;
			pl->health = 200;
			pl->altFls.clWeaponAmmo = 0;
			pl->net.plAmmo = pl->altFls.clWeaponAmmo;
			plNum++;
		}
	}

	if (wasPlayingMusical && highestScore > g_musicHighScore)
	{
		g_musicHighScore = highestScore;
		if (g_musicHighScore > 0)
		{
			LServ_SaveNewHighscore(g_musicResetName);
		}
	}

	g_musicHighScore = LServ_GetMusicHighScore(g_musicResetNameNext);
	g_musicHighScorePrev = g_musicHighScore;
	g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), -1);

	sprintf(str, "Now playing: %s", g_musicResetNameNext);
	g_sharedFn->Net_SendMessage(str);
	//todo high score tracking
	strcpy(g_musicResetName, g_musicResetNameNext);

	//destroy all enemies and such
	for (int i = MAX_NET_CLIENTS; i < g_gameObjectSlots; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (!other->inuse)
		{
			continue;
		}
		if (other->projDamage)
		{ //remove projectile
			other->think = ObjGeneral_RemoveThink;
			other->thinkTime = g_curTime-1;
		}
		else if (other->net.solid && other->hurtable && other->health > 0)
		{ //kill it
			Util_DamageObject(other, other, 9999999);
		}
	}
}

//logic frame
void LServ_RunFrame(unsigned long prvTime, unsigned long curTime, bool paused)
{
	float timeDif = (float)(curTime-prvTime);

	g_invTimeScale = 1.0f/g_timeScale;
	timeDif *= g_timeScale;

	g_timeFactor = timeDif/50.0f; //compensate for dropped frames
	if (g_timeFactor > 2.0f)
	{
		g_timeFactor = 2.0f;
	}
	else if (g_timeFactor < 0.1f)
	{
		g_timeFactor = 0.1f;
	}

	if (!g_curTime || prvTime == curTime)
	{
		g_curTime = curTime;
	}
	else
	{
		g_curTime += (unsigned long)timeDif;
	}

	if (paused)
	{
		return;
	}

	if (g_musicReset && g_musicReset < g_curTime)
	{
		LServ_MusicReset();
	}

	if (g_musical)
	{
		float amp, pos;
		g_sharedFn->Common_QueryMusic(&amp, &pos);
		gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
		if (cam->inuse)
		{
			cam->net.modelScale[1] = pos;
		}
		g_musicalLastFrame = g_musicalAmp;
		g_musicalAmp = amp;

		g_musicalRunningAvg += amp;
		g_musicalRunningCount++;
		int avgChunk = 256;
		if (g_musicalRunningCount > avgChunk)
		{
			g_musicalRunningAvg /= (float)g_musicalRunningCount;
			g_musicalRunningAvg *= ((float)avgChunk/2.0f);
			g_musicalRunningCount = avgChunk/2;
		}

		LServ_MusicalSpawns();
	}

#ifdef _RC_NOVODEX
	//run extra compensation frames (novodex refuses higher values for some reason)
#if 1
	#define PHYSICS_RATE	20
	while (timeDif > PHYSICS_RATE)
	{
		g_sharedFn->Common_RunPhysics(1.0f);
		timeDif -= PHYSICS_RATE;
	}
	g_sharedFn->Common_RunPhysics(timeDif/(float)PHYSICS_RATE);
#else
	g_sharedFn->Common_RunPhysics(timeDif/50.0f);
#endif
#endif //_RC_NOVODEX

	gameObject_t *activeLights[MAX_GAME_OBJECTS];
	int numActiveLights = 0;
	gameObject_t *obj;
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		obj = &g_gameObjects[i];
		if (obj->inuse)
		{
			LServ_NetCull(obj);
			if (obj->net.type == OBJ_TYPE_LIGHT &&
				!(obj->inuse & INUSE_NONET) &&
				obj->net.lerpDuration)
			{ //add to the unculled lights list (only if it's a shadowing light)
				activeLights[numActiveLights++] = obj;
			}
			if (obj->think && obj->thinkTime < g_curTime && !obj->parent)
			{
				obj->think(obj, g_timeFactor);
				if (obj->child)
				{ //think for the child
					obj->child->think(obj->child, g_timeFactor);
				}
			}
#ifdef _RC_NOVODEX
			if (obj->nxActor)
			{
				if (!obj->dynphys)
				{
					g_sharedFn->Common_NxActorOverride(obj->nxActor, &obj->net);
				}
				if (obj->getPhysFromNovodex)
				{
					float oldAng[3];
					if (obj->localAngles)
					{
						Math_VecCopy(obj->net.ang, oldAng);
					}
					g_sharedFn->Common_UpdateFromNxActor(obj->nxActor, &obj->net);
					if (obj->localAngles)
					{
						Math_VecCopy(oldAng, obj->net.ang);
					}
				}
			}
#endif
			/*
			if (obj->rcColModel)
			{ //update rclip model
				LServ_UpdateRClip(obj);
			}
			*/
		}
	}

	if (numActiveLights > 0)
	{ //check culled objects against lights
		for (int i = 0; i < g_gameObjectSlots; i++)
		{
			obj = &g_gameObjects[i];

			if (!obj->inuse || !(obj->inuse & INUSE_NONET))
			{ //only interested if it is inuse and culled
				continue;
			}

			if (obj->net.type == OBJ_TYPE_LIGHT)
			{ //lights don't affect each other
				continue;
			}

			if (obj->localFlags & LFL_NONET)
			{ //never wants to be seen by the client
				continue;
			}

			if (obj->spawnMins[0] == 0.0f &&
				obj->spawnMins[1] == 0.0f &&
				obj->spawnMins[2] == 0.0f &&
				obj->spawnMaxs[0] == 0.0f &&
				obj->spawnMaxs[1] == 0.0f &&
				obj->spawnMaxs[2] == 0.0f)
			{ //spawn bounds not set
				continue;
			}

			for (int j = 0; j < numActiveLights; j++)
			{
				if (activeLights[j]->net.frame & LIGHTFL_FULLAMB)
				{ //do not consider pure ambient lights
					continue;
				}
				if (Util_GameObjectsOverlap(obj, activeLights[j]))
				{ //it interacts with a light in view, so make sure the client gets it.
					obj->inuse &= ~INUSE_NONET;
					break;
				}
			}
		}
	}

	//ObjVisBlock_DebugDraw();
}

//make sure global resources are cached by the client
void LServ_GlobalResources(void)
{
	g_sharedFn->Common_ServerString("^assets/textures/fetusredshell");
	g_sharedFn->Common_ServerString("$assets/sound/menu/introsnd.wav");
	ObjProjectile_Init();
}

//set up the main map object
void LServ_InitGlobalMapObj(void)
{
	gameObject_t *map = &g_gameObjects[MAP_OBJ_NUM];

	map->inuse = INUSE_ACTIVE;
	map->net.index = MAP_OBJ_NUM;
	map->net.type = OBJ_TYPE_MAP;
	map->net.pos[0] = 0.0f;
	map->net.pos[1] = 0.0f;
	map->net.pos[2] = 0.0f;

	map->net.ang[0] = 0.0f;
	map->net.ang[1] = 0.0f;
	map->net.ang[2] = 0.0f;

	map->net.entNameIndex = g_sharedFn->Common_ServerString("obj_map");

	map->inuse |= INUSE_NONET; //for fetus blaster

	map->net.solid = 1;

	//Fetus Blaster - init the camera tracker
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];

	g_noConfineTime = 0;
	g_camFrustumFresh = false;
	cam->inuse = INUSE_ACTIVE;
	cam->net.staticIndex = -1;
	cam->net.index = CAM_TRACK_NUM;
	cam->net.type = OBJ_TYPE_CAM;
	cam->net.pos[0] = 0.0f;
	cam->net.pos[1] = 0.0f;
	cam->net.pos[2] = DEFAULT_CAM_Z;//g_fetusPilotZ+2048.0f;

	cam->camTrackPos[0] = 0.0f;
	cam->camTrackPos[1] = 0.0f;
	cam->camTrackPos[2] = DEFAULT_CAM_Z;

	cam->net.ang[0] = 180.0f;
	cam->net.ang[1] = 0.0f;
	cam->net.ang[2] = 0.0f;

	g_levelScroll[0] = 0.0f;
	g_levelScroll[1] = 0.0f;
	g_levelScroll[2] = 0.0f;

	cam->net.maxs[0] = g_levelScroll[0];
	cam->net.maxs[1] = g_levelScroll[1];

	cam->net.modelScale[0] = 70.0f; //default fov

	//cam->net.renderEffects |= FXFL_FPSMODE;

	cam->net.entNameIndex = g_sharedFn->Common_ServerString("obj_camtrack");

	cam->think = ObjCam_Think;
	cam->thinkTime = 0;
}

//client joined, create its map object.
void LServ_InitPlayerObj(int clientNum, const char *clientInfo)
{
	gameObject_t *pl = &g_gameObjects[clientNum];

	strcpy(g_clientInfo[clientNum].infoStr, clientInfo);
	g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), clientNum);

	g_sharedFn->Common_ServerString("$assets/sound/weapons/explode2.wav");
	g_sharedFn->Common_ServerString("&&player/death");

	pl->net.lives = DEFAULT_PLAYER_LIVES;
	pl->musicalKills = 0;

	pl->net.owner = MAP_OBJ_NUM;
	pl->net.staticIndex = -1;
	pl->inuse = INUSE_ACTIVE;
	pl->net.index = clientNum;
	pl->net.type = OBJ_TYPE_MODEL;
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	if (cam->inuse)
	{
		pl->net.pos[0] = cam->net.pos[0]-2048.0f;
		pl->net.pos[1] = cam->net.pos[1];
		pl->net.pos[2] = cam->net.pos[2]-4096.0f;
	}
	else
	{
		pl->net.pos[0] = -2048.0f;
		pl->net.pos[1] = 0.0f;
		pl->net.pos[2] = DEFAULT_PLR_Z-2048.0f;//g_fetusPilotZ-2048.0f;
	}

	pl->health = DEFAULT_PLAYER_HEALTH;
	if (g_musical)
	{
		pl->health = 200;
	}
	if (!g_fpsMode)
	{
		pl->hurtable = true;
	}

	//pl->dynphys = 2;

	pl->fls.playerSpawnSeq = true;

	pl->net.ang[0] = 0.0f;
	pl->net.ang[1] = 0.0f;
	pl->net.ang[2] = 0.0f;

	if (g_fpsMode)
	{
		pl->net.mins[0] = -256.0f;
		pl->net.mins[1] = -256.0f;
		pl->net.mins[2] = -256.0f;
		pl->net.maxs[0] = 256.0f;
		pl->net.maxs[1] = 256.0f;
		pl->net.maxs[2] = 256.0f;
	}
	else
	{
		float shooterRad = 64.0f;
		pl->net.mins[0] = -shooterRad;
		pl->net.mins[1] = -shooterRad;
		pl->net.mins[2] = -shooterRad;
		pl->net.maxs[0] = shooterRad;
		pl->net.maxs[1] = shooterRad;
		pl->net.maxs[2] = shooterRad;
	}

	pl->fls.clWeapon = 0;
	pl->fls.clWeaponAmmo = -1;
	pl->altFls.clWeapon = 1;
	pl->altFls.clWeaponAmmo = 3;
	pl->net.plAmmo = 3;

	if (g_musical)
	{
		pl->altFls.clWeaponAmmo = 0;
		pl->net.plAmmo = 0;
	}

	pl->radius = LServ_RadiusFromBounds(pl->net.mins, pl->net.maxs, 1.0f);

	pl->net.solid = 1;

	pl->touch = ObjPlayer_Touch;
	pl->death = ObjPlayer_Death;
	pl->think = ObjPlayer_Think;
	pl->thinkTime = 0;

	pl->localAngles = true;

	//pl->net.renderEffects = FXFL_SHELLRED;

	pl->net.entNameIndex = g_sharedFn->Common_ServerString("obj_player");

	//pl->net.strIndex = g_sharedFn->Common_ServerString("#assets/sprites/marche.rsp");
	//pl->net.strIndex = g_sharedFn->Common_ServerString("&assets/models/player.rdm");
	if (g_fpsMode)
	{
		pl->net.strIndex = 0;
	}
	else
	{
		pl->net.strIndex = g_sharedFn->Common_ServerString("&assets/models/ship.rdm");
	}

#ifdef _RC_NOVODEX
	//update nxactor
	//g_sharedFn->Common_CreateNxActor(&pl->nxActor, &pl->net, "assets/models/player.rdm");
	g_sharedFn->Common_CreateNxActor(&pl->nxActor, &pl->net, "assets/models/ship.rdm");

	/*
	pl->net.modelScale[0] = 8.0f;
	pl->net.modelScale[1] = 8.0f;
	pl->net.modelScale[2] = 8.0f;
	char assetName[2048];
	assetName[0] = '@';
	assetName[1] = 0;
	strcat(assetName, "assets/models/pinky/pinky_idle.rda");
	pl->net.strIndexB = g_sharedFn->Common_ServerString(assetName);
	*/
	//g_sharedFn->Common_CreateNxActor(&pl->nxActor, &pl->net, "*sphere");
	if (pl->nxActor)
	{
		g_sharedFn->Common_NxActorOverride(pl->nxActor, &pl->net);
		LServ_Relink();
	}
#endif

    //pl->rcColModel = g_sharedFn->Coll_RegisterModelInstance("*sphere");
	pl->rcColModel = g_sharedFn->Coll_RegisterModelInstance("assets/models/ship.rdm");
	if (pl->rcColModel)
	{
		pl->rcColModel->radius = pl->radius;
		if (g_fpsMode)
		{
			pl->rcColModel->clipFlags = CLIPFL_BOXMOVE;
		}
		Math_VecCopy(pl->net.mins, pl->rcColModel->mins);
		Math_VecCopy(pl->net.maxs, pl->rcColModel->maxs);
		Math_VecCopy(pl->net.pos, pl->rcColModel->pos);
		Math_VecCopy(pl->net.ang, pl->rcColModel->ang);
		Math_VecCopy(pl->net.modelScale, pl->rcColModel->modelScale);
		pl->rcColModel->gameOwner = pl->net.index;
		pl->rcColModel->solid = pl->net.solid;
		pl->rcColModel->frame = pl->net.frame;
		g_sharedFn->Coll_UpdateModel(pl->rcColModel, NULL);
	}
}

//client disconnected
void LServ_FreePlayerObj(int clientNum)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (pl->inuse)
	{
		LServ_FreeObject(pl);
	}
}

//cache all stuff applicable
void LServ_CacheObj(const char *objName)
{
	const char *s;
	char assetName[2048];
	int i;
	int type;
	BYTE *b = Common_GetEntryForObject(objName);
	if (!b)
	{ //no entry for this object then
		return;
	}

	Util_ParseObjType(Common_GetValForKey(b, "type"), &type);

	char soundCacheStr[256];
	i = 0;
	sprintf(soundCacheStr, "soundcache%i", i);
	s = Common_GetValForKey(b, soundCacheStr);
	while (s && s[0] && i < 256)
	{
		sprintf(assetName, "$%s", s);
		g_sharedFn->Common_ServerString(assetName);
		i++;
		sprintf(soundCacheStr, "soundcache%i", i);
		s = Common_GetValForKey(b, soundCacheStr);
	}

	s = Common_GetValForKey(b, "customTex");
	if (s && s[0])
	{ //make sure the client caches this texture
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//precache particle effects
	s = Common_GetValForKey(b, "rpl_muzzle");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}
	s = Common_GetValForKey(b, "rpl_muzzlechg");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}
	s = Common_GetValForKey(b, "rpl_impact");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//precache decals
	s = Common_GetValForKey(b, "dcl_impact");
	if (s && s[0])
	{
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//base on the type we need a different prepended char so the client
	//knows what to cache based on string name alone
	i = 0;
	if (type == OBJ_TYPE_SPRITE || type == OBJ_TYPE_PROJECTILE)
	{
		assetName[i++] = '#';
	}
	else if (type == OBJ_TYPE_MODEL)
	{
		assetName[i++] = '&';
	}
	assetName[i] = 0;
	s = Common_GetValForKey(b, "assetName");
	if ((!s || !s[0]) && assetName[0])
	{ //if it's a type that needs an asset specified, stick error string in
		s = "NO_ASSET_SPECIFIED";
	}

	if (s && s[0])
	{
		strcat(assetName, s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//check modelAnim
	s = Common_GetValForKey(b, "modelAnim");
	if (s && s[0])
	{
		assetName[0] = '@';
		assetName[1] = 0;
		strcat(assetName, s);
		g_sharedFn->Common_ServerString(assetName);
	}

#ifdef _RC_NOVODEX
	//update nxactor
	s = Common_GetValForKey(b, "clipModel");
	if (!s)
	{
		s = "*sphere";
	}
	if (s[0] != '*')
	{ //if it's a custom clip model, it needs to be cached
		g_sharedFn->Common_CacheModel(s);
	}
#endif

	//rclip functionality
	s = Common_GetValForKey(b, "clipModel");
	if (!s)
	{
		s = "*sphere";
	}
	if (s[0] != '*')
	{ //if it's a custom clip model, it needs to be cached
		const char *clipAnim = Common_GetValForKey(b, "clipAnim");
		g_sharedFn->Coll_CacheModel(s, clipAnim);
	}
}

gameObject_t *LServ_ObjectFromName(const char *objName, float *pos, float *ang, const objArgs_t *args, int numArgs)
{
	const char *s;
	char assetName[2048];
	BYTE *b = Common_GetEntryForObject(objName);
	int i;
	if (!b)
	{ //no entry for this object then
		return NULL;
	}

	gameObject_t *obj = LServ_CreateObject();
	if (!obj)
	{ //oh no
		return NULL;
	}

	obj->objData = b;
	obj->spawnArgs = args;
	obj->numSpawnArgs = numArgs;

	Util_ParseObjType(Common_GetValForKey(b, "type"), &obj->net.type);

	Util_ParseObjSpawn(Common_GetValForKey(b, "customSpawn"), obj);

	obj->think = ObjGeneral_Think;
	obj->thinkTime = 0;

	obj->net.pos[0] = pos[0];
	obj->net.pos[1] = pos[1];
	obj->net.pos[2] = pos[2];

	obj->net.ang[0] = ang[0];
	obj->net.ang[1] = ang[1];
	obj->net.ang[2] = ang[2];

	Util_ParseVector(Common_GetValForKey(b, "spawnMins"), obj->spawnMins);
	Util_ParseVector(Common_GetValForKey(b, "spawnMaxs"), obj->spawnMaxs);

	Util_ParseVector(Common_GetValForKey(b, "mins"), obj->net.mins);
	Util_ParseVector(Common_GetValForKey(b, "maxs"), obj->net.maxs);
	obj->radius = LServ_RadiusFromBounds(obj->net.mins, obj->net.maxs, 1.0f);
	Util_ParseVector(Common_GetValForKey(b, "modelScale"), obj->net.modelScale);

	Util_ParseInt(Common_GetValForKey(b, "hurtable"), &obj->hurtable);
	Util_ParseInt(Common_GetValForKey(b, "health"), &obj->health);

	Util_ParseInt(Common_GetValForKey(b, "solid"), &obj->net.solid);

	Util_ParseInt(Common_GetValForKey(b, "renderfx"), &obj->net.renderEffects);

	s = Common_GetValForKey(b, "customTex");
	if (s && s[0])
	{ //make sure the client caches this texture
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	obj->net.entNameIndex = g_sharedFn->Common_ServerString(objName);

	//base on the type we need a different prepended char so the client
	//knows what to cache based on string name alone
	i = 0;
	if (obj->net.type == OBJ_TYPE_SPRITE || obj->net.type == OBJ_TYPE_PROJECTILE)
	{
		assetName[i++] = '#';
	}
	else if (obj->net.type == OBJ_TYPE_MODEL)
	{
		assetName[i++] = '&';
	}
	assetName[i] = 0;
	s = Common_GetValForKey(b, "assetName");
	if ((!s || !s[0]) && assetName[0])
	{ //if it's a type that needs an asset specified, stick error string in
		s = "NO_ASSET_SPECIFIED";
	}

	if (s && s[0])
	{
		strcat(assetName, s);
		obj->net.strIndex = g_sharedFn->Common_ServerString(assetName);
	}

	//check modelAnim
	s = Common_GetValForKey(b, "modelAnim");
	if (s && s[0])
	{
		assetName[0] = '@';
		assetName[1] = 0;
		strcat(assetName, s);
		obj->net.strIndexB = g_sharedFn->Common_ServerString(assetName);
	}
	else
	{
		obj->net.strIndexB = 0;
	}

	s = Common_GetValForKey(b, "lightBlocking");
	if (s && s[0] && atoi(s) == 1)
	{ //blocks static light, take this to mean it is static
		obj->net.staticIndex = obj->net.index;
	}

#ifdef _RC_NOVODEX
	Util_ParseInt(Common_GetValForKey(b, "dynphys"), &obj->dynphys);
	if (obj->dynphys)
	{
		obj->getPhysFromNovodex = true;
	}
	Util_ParseInt(Common_GetValForKey(b, "staticphys"), &obj->staticphys);
	if (obj->net.solid)
	{
		//update nxactor
		s = Common_GetValForKey(b, "clipModel");
		if (!s)
		{
			s = "*sphere";
		}
		g_sharedFn->Common_CreateNxActor(&obj->nxActor, &obj->net, s);
		if (obj->nxActor)
		{
			g_sharedFn->Common_NxActorOverride(obj->nxActor, &obj->net);
			LServ_Relink();

			if (obj->dynphys || obj->staticphys == 2)
			{
				int flags = 0;

				if (obj->staticphys == 2)
				{ //does not move or do anything, but actively reacts to physics objects colliding with it
					flags |= (OBJPHYS_NOROTATION|OBJPHYS_NOTRANSLATION|OBJPHYS_NOFRICTION|OBJPHYS_NOGRAVITY);
				}
				else if (obj->dynphys == 2)
				{
					flags |= (OBJPHYS_NOROTATION|OBJPHYS_NOPHYSREACTION|OBJPHYS_NOFRICTION|OBJPHYS_NOGRAVITY);
				}
				g_sharedFn->Common_NxFullCollision(obj->nxActor, flags);
			}
		}
	}
#endif

	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		for (int i = 0; i < obj->numSpawnArgs; i++)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "objMins"))
			{ //mins override
				Util_ParseVector(arg->val, obj->spawnMins);
				Math_VecCopy(obj->spawnMins, obj->net.mins);
			}
			else if (!_stricmp(arg->key, "objMaxs"))
			{ //maxs override
				Util_ParseVector(arg->val, obj->spawnMaxs);
				Math_VecCopy(obj->spawnMaxs, obj->net.maxs);
			}
		}
	}

	//rclip functionality
	if (obj->net.solid)
	{
		//update nxactor
		s = Common_GetValForKey(b, "clipModel");
		if (!s)
		{
			s = "*sphere";
		}
		if (s)
		{
            obj->rcColModel = g_sharedFn->Coll_RegisterModelInstance(s);
			if (obj->rcColModel)
			{
				obj->rcColModel->radius = obj->radius;
				s = Common_GetValForKey(b, "clipBox");
				if (s && atoi(s))
				{
					obj->rcColModel->clipFlags = CLIPFL_BOXMOVE;
				}
				Math_VecCopy(obj->net.mins, obj->rcColModel->mins);
				Math_VecCopy(obj->net.maxs, obj->rcColModel->maxs);
				Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
				Math_VecCopy(obj->net.ang, obj->rcColModel->ang);
				Math_VecCopy(obj->net.modelScale, obj->rcColModel->modelScale);
				obj->rcColModel->gameOwner = obj->net.index;
				obj->rcColModel->solid = obj->net.solid;
				obj->rcColModel->frame = obj->net.frame;
				const char *clipAnim = Common_GetValForKey(b, "clipAnim");
				g_sharedFn->Coll_UpdateModel(obj->rcColModel, clipAnim);
			}
		}
	}

	if (obj->customSpawn)
	{
		obj->customSpawn(obj, b, args, numArgs);
	}

	return obj;
}

//time update outside of frame when necessary
void LServ_UpdateTime(unsigned long time)
{
	g_curTime = time;
}

//the engine will call this function for each map object.
//it's our responsibility to create an object in the game structure.
void LServ_InitMapObj(const char *objName, float *pos, float *ang, const objArgs_t *args, int numArgs)
{
	LServ_ObjectFromName(objName, pos, ang, args, numArgs);
}

//find a targeted object based on target name
gameObject_t *LServ_FindTargetObject(const char *targetName)
{
    int i = 0;
	while (i < g_gameObjectSlots)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->spawnArgs && obj->numSpawnArgs > 0)
		{
			int j = 0;
			while (j < obj->numSpawnArgs)
			{
				const objArgs_t *arg = obj->spawnArgs+j;
				if (!_stricmp(arg->key, "targname") && !_stricmp(arg->val, targetName))
				{
					return obj;
				}
				j++;
			}
		}
		i++;
	}

	return NULL;
}

//perform any tasks that should be performed immediately after map spawn
void LServ_PostMapSpawn(void)
{
    int i = 0;
	while (i < g_gameObjectSlots)
	{
		//target entities to each other
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->spawnArgs && obj->numSpawnArgs > 0)
		{
			int j = 0;
			while (j < obj->numSpawnArgs)
			{
				const objArgs_t *arg = obj->spawnArgs+j;
				if (!_stricmp(arg->key, "camstart"))
				{ //camera's target
					gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
					if (cam->inuse)
					{
                        cam->target = obj;
						cam->net.pos[0] = obj->net.pos[0];
						cam->net.pos[1] = obj->net.pos[1];
						cam->net.pos[2] = obj->net.pos[2];
						cam->camTrackPos[0] = obj->net.pos[0];
						cam->camTrackPos[1] = obj->net.pos[1];
						cam->camTrackPos[2] = obj->net.pos[2];
					}
				}
				else if (!_stricmp(arg->key, "targobj"))
				{ //targeting something else
					obj->target = LServ_FindTargetObject(arg->val);
				}
				j++;
			}
		}
		i++;
	}

	ObjVisBlock_EstablishVis();

	//call post-spawns
	i = 0;
	while (i < g_gameObjectSlots)
	{
		//target entities to each other
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->postSpawn)
		{
			obj->postSpawn(obj);
		}
		i++;
	}
}

//physics impact callback (for novodex)
void LServ_PhysicsImpact(int objNum, int otherNum, const float *planeDir, const float *point)
{
#ifdef _RC_NOVODEX
	gameObject_t *obj = &g_gameObjects[objNum];
	gameObject_t *other = &g_gameObjects[otherNum];
	if (!obj->inuse || !other->inuse)
	{ //stale objects
		return;
	}
	if (!obj->net.solid || !other->net.solid)
	{ //non-solid things won't collide
		return;
	}
	if (obj->net.owner != MAP_OBJ_NUM && obj->net.owner == otherNum)
	{ //my owner is not the world, and the thing i'm hitting is it
		return;
	}
	if (other->net.owner == objNum)
	{ //i am the thing's owner
		return;
	}
	if (obj->net.owner != MAP_OBJ_NUM &&
		obj->net.owner == other->net.owner)
	{ //same owner
		return;
	}
	if (obj->net.index == other->net.index)
	{ //hit myself?!
		return;
	}

	float up[3];
	up[0] = 0.0f;
	up[1] = 0.0f;
	up[2] = 1.0f;
	float dp = Math_DotProduct(planeDir, up);
	if (dp < -0.4f)
	{
		obj->onGround = true;
	}

	if (obj->touch)
	{
		obj->touch(obj, other, planeDir, point);
	}
	//check still in use in case a touch function removed one of the objects
	if (other->touch)
	{
		other->touch(other, obj, planeDir, point);
	}
#endif
}

//get information about the game object
void LServ_ObjectInfo(void **baseAddr, int *size, int *num, int *netSize, int **activeObjPtr)
{
	*baseAddr = &g_gameObjects[0];
	*size = sizeof(gameObject_t);
	*num = MAX_GAME_OBJECTS;
	*netSize = sizeof(gameObjNet_t);
	*activeObjPtr = &g_gameObjectSlots;
}

//new input from a client
void LServ_ClientInput(int clientNum, BYTE *buttons)
{ //it would be better to cache these and run them all in a single server frame for smoother input.
	memcpy(g_gameObjects[clientNum].clButtons, buttons, sizeof(g_gameObjects[clientNum].clButtons));
}

//the client's view angles changed (in respect to the player object)
void LServ_ClientInputAngle(int clientNum, float *angles)
{
	memcpy(g_gameObjects[clientNum].clAngles, angles, sizeof(g_gameObjects[clientNum].clAngles));
	if (!g_gameObjects[clientNum].hasLastClAngles)
	{
		Math_VecCopy(g_gameObjects[clientNum].clAngles, g_gameObjects[clientNum].lastClAngles);
		g_gameObjects[clientNum].hasLastClAngles = true;
	}
}

//the client's analog controller movement
void LServ_ClientInputAnalog(int clientNum, WORD *analogs)
{
	memcpy(g_gameObjects[clientNum].clAnalog, analogs, sizeof(g_gameObjects[clientNum].clAnalog));
	g_gameObjects[clientNum].hasClAnalog = true;
}

//in musical mode, modify the music string
void LServ_ChangeMusic(char *music)
{
	if (!g_musical || g_musicStrIndex == -1)
	{
		return;
	}
	g_sharedFn->Common_ModifyString(g_musicStrIndex, ""); //force cleansing
	char str[1024];
	sprintf(str, "$@assets/music/%s", music);
	g_sharedFn->Common_ModifyString(g_musicStrIndex, str);
}

//music was restarted/changed
void LServ_MusicalReset(const char *musicName)
{
	if (!g_musical)
	{
		return;
	}
	g_musicReset = g_curTime + 500;
	strcpy(g_musicResetNameNext, musicName);
}

//module init
int LServ_Initialize(sharedSVFunctions_t *sharedFunc, void *val, bool queryOnly, bool multiplayer)
{
	g_musical = false;
	g_musicStrIndex = -1;
	g_musicalRunningAvg = 0.0f;
	g_musicalLastFrame = 0.0f;
	g_musicalSpawnTime = 0;
	g_musicalRunningCount = 1;
	g_musicReset = 0;
	g_musicHighScore = 0;
	g_musicHighScorePrev = 0;
	g_musicResetName[0] = 0;
	g_musicResetNameNext[0] = 0;

	g_sharedFn = sharedFunc;
	g_sharedFn->LServ_RunFrame = LServ_RunFrame;
	g_sharedFn->LServ_ObjectInfo = LServ_ObjectInfo;
	g_sharedFn->LServ_ClientInput = LServ_ClientInput;
	g_sharedFn->LServ_ClientInputAngle = LServ_ClientInputAngle;
	g_sharedFn->LServ_ClientInputAnalog = LServ_ClientInputAnalog;
	g_sharedFn->LServ_InitPlayerObj = LServ_InitPlayerObj;
	g_sharedFn->LServ_FreePlayerObj = LServ_FreePlayerObj;
	g_sharedFn->LServ_UpdateTime = LServ_UpdateTime;
	g_sharedFn->LServ_InitMapObj = LServ_InitMapObj;
	g_sharedFn->LServ_PostMapSpawn = LServ_PostMapSpawn;
	g_sharedFn->LServ_ChangeMusic = LServ_ChangeMusic;
	g_sharedFn->LServ_MusicalReset = LServ_MusicalReset;
	g_sharedFn->LServ_PhysicsImpact = LServ_PhysicsImpact;
	if (val)
	{
		memcpy(&g_gameValues, val, sizeof(g_gameValues));
	}

	g_runningMultiplayer = multiplayer;

	if (!queryOnly)
	{ //don't bother loading stuff if this is only a query
		LServ_InitGlobalMapObj();
		LServ_GlobalResources();
	}

	return 1;
}

//module shutdown
void LServ_Shutdown(void)
{
	g_sharedFn->Common_SaveGameValues(&g_gameValues, sizeof(g_gameValues));

	int i = 0;
	while (i < MAX_GAME_OBJECTS)
	{
		if (g_gameObjects[i].inuse)
		{
			LServ_FreeObject(&g_gameObjects[i]);
		}
		i++;
	}
	g_gameObjectSlots = 0;
}
