/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// models.c -- model loading and caching

#include "gl_local.h"

model_t	*loadmodel;
int		modfilelen;

void Mod_LoadSpriteModel (model_t *mod, void *buffer);
void Mod_LoadBrushModel (model_t *mod, void *buffer);
void Mod_LoadAliasModel (model_t *mod, void *buffer);
model_t *Mod_LoadModel (model_t *mod, qboolean crash);

byte	mod_novis[MAX_MAP_LEAFS/8];

#define	MAX_MOD_KNOWN	512
model_t	mod_known[MAX_MOD_KNOWN];
int		mod_numknown;

// the inline * models from the current map are kept seperate
model_t	mod_inline[MAX_MOD_KNOWN];

int		registration_sequence;

/*
===============
Mod_PointInLeaf
===============
*/
mleaf_t *Mod_PointInLeaf (vec3_t p, model_t *model)
{
	mnode_t		*node;
	float		d;
	cplane_t	*plane;
	
	if (!model || !model->nodes)
		ri.Sys_Error (ERR_DROP, "Mod_PointInLeaf: bad model");

	node = model->nodes;
	while (1)
	{
		if (node->contents != -1)
			return (mleaf_t *)node;
		plane = node->plane;
		d = DotProduct (p,plane->normal) - plane->dist;
		if (d > 0)
			node = node->children[0];
		else
			node = node->children[1];
	}
	
	return NULL;	// never reached
}


/*
===================
Mod_DecompressVis
===================
*/
byte *Mod_DecompressVis (byte *in, model_t *model)
{
	static byte	decompressed[MAX_MAP_LEAFS/8];
	int		c;
	byte	*out;
	int		row;

	row = (model->vis->numclusters+7)>>3;	
	out = decompressed;

	if (!in)
	{	// no vis info, so make all visible
		while (row)
		{
			*out++ = 0xff;
			row--;
		}
		return decompressed;		
	}

	do
	{
		if (*in)
		{
			*out++ = *in++;
			continue;
		}
	
		c = in[1];
		in += 2;
		while (c)
		{
			*out++ = 0;
			c--;
		}
	} while (out - decompressed < row);
	
	return decompressed;
}

/*
==============
Mod_ClusterPVS
==============
*/
byte *Mod_ClusterPVS (int cluster, model_t *model)
{
	if (cluster == -1 || !model->vis)
		return mod_novis;
	return Mod_DecompressVis ( (byte *)model->vis + model->vis->bitofs[cluster][DVIS_PVS],
		model);
}


//===============================================================================

/*
================
Mod_Modellist_f
================
*/
void Mod_Modellist_f (void)
{
	int		i;
	model_t	*mod;
	int		total;

	total = 0;
	ri.Con_Printf (PRINT_ALL,"Loaded models:\n");
	for (i=0, mod=mod_known ; i < mod_numknown ; i++, mod++)
	{
		if (!mod->name[0])
			continue;
		ri.Con_Printf (PRINT_ALL, "%8i : %s\n",mod->extradatasize, mod->name);
		total += mod->extradatasize;
	}
	ri.Con_Printf (PRINT_ALL, "Total resident: %i\n", total);
}

/*
===============
Mod_Init
===============
*/
void Mod_Init (void)
{
	memset (mod_novis, 0xff, sizeof(mod_novis));
}


//rww begin
//i'm sure there are some duplications of existing quake functionality in here.
//some of this code is ported directly from my model conversion utility (that
//i use to convert to my own engine's format) and i didn't feel like making sure
//all the quake math functions operate the same and changing them and blah.
modelMatrix_t g_identityMatrix =
{
	{1.0f, 0.0f, 0.0f},
	{0.0f, 1.0f, 0.0f},
	{0.0f, 0.0f, 1.0f},
	{0.0f, 0.0f, 0.0f}
};

//standard plane equation formula
void Math_CalcPlaneEq(float *x, float *y, float *z, float *planeEq)
{
	planeEq[0] = x[1]*(y[2]-z[2]) + y[1]*(z[2]-x[2]) + z[1]*(x[2]-y[2]);
	planeEq[1] = x[2]*(y[0]-z[0]) + y[2]*(z[0]-x[0]) + z[2]*(x[0]-y[0]);
	planeEq[2] = x[0]*(y[1]-z[1]) + y[0]*(z[1]-x[1]) + z[0]*(x[1]-y[1]);
	planeEq[3] = -( x[0]*( y[1]*z[2] - z[1]*y[2] ) + y[0]*(z[1]*x[2] - x[1]*z[2]) + z[0]*(x[1]*y[2] - y[1]*x[2]) );
}

//vector normalize func
float Math_VecNorm(float *v)
{
	float	length, ilength;

	length = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
	length = (float)sqrt (length);

	if ( length )
	{
		ilength = 1/length;
		v[0] *= ilength;
		v[1] *= ilength;
		v[2] *= ilength;
	}
		
	return length;
}

//vector length func
float Math_VecLen(float *a)
{
	return (float)sqrt (a[0]*a[0] + a[1]*a[1] + a[2]*a[2]);
}

//vector copy func
void Math_VecCopy(float *a, float *out)
{
	out[0] = a[0];
	out[1] = a[1];
	out[2] = a[2];
}

//vector subtract func
void Math_VecSub(float *a, float *b, float *out)
{
	out[0] = a[0]-b[0];
	out[1] = a[1]-b[1];
	out[2] = a[2]-b[2];
}

//vector add func
void Math_VecAdd(float *a, float *b, float *out)
{
	out[0] = a[0]+b[0];
	out[1] = a[1]+b[1];
	out[2] = a[2]+b[2];
}

//vector scale func
void Math_VecScale(float *a, float scale)
{
	a[0] *= scale;
	a[1] *= scale;
	a[2] *= scale;
}

//good matrix refrence: http://www.ibrtses.com/opengl/matrices.html

//transform a point by a given matrix
void Math_TransformPointByMatrix(modelMatrix_t *matrix, float *in, float *out)
{
	out[0] = matrix->x1[0]*in[0] + matrix->x1[1]*in[1] + matrix->x1[2]*in[2] + matrix->o[0];
	out[1] = matrix->x2[0]*in[0] + matrix->x2[1]*in[1] + matrix->x2[2]*in[2] + matrix->o[1];
	out[2] = matrix->x3[0]*in[0] + matrix->x3[1]*in[1] + matrix->x3[2]*in[2] + matrix->o[2];
}

//transform a point by a given matrix
void Math_TransformPointByMatrix3x3(modelMatrix_t *matrix, float *in, float *out)
{
	out[0] = matrix->x1[0]*in[0] + matrix->x1[1]*in[1] + matrix->x1[2]*in[2];
	out[1] = matrix->x2[0]*in[0] + matrix->x2[1]*in[1] + matrix->x2[2]*in[2];
	out[2] = matrix->x3[0]*in[0] + matrix->x3[1]*in[1] + matrix->x3[2]*in[2];
}

//matrix inverse
void Math_MatrixInverse(modelMatrix_t *in, modelMatrix_t *out)
{
	int i, j;
	float *mat;
	float *srcMat;

    for (i = 0; i < 3; i++)
	{
		if (i == 0)
		{
			mat = out->x1;
		}
		else if (i == 1)
		{
			mat = out->x2;
		}
		else
		{
			mat = out->x3;
		}

        for (j = 0; j < 3; j++)
		{
			if (j == 0)
			{
				srcMat = in->x1;
			}
			else if (j == 1)
			{
				srcMat = in->x2;
			}
			else
			{
				srcMat = in->x3;
			}
            mat[j] = srcMat[i];
		}
	}
    for (i = 0; i < 3; i++)
	{
		if (i == 0)
		{
			mat = out->x1;
		}
		else if (i == 1)
		{
			mat = out->x2;
		}
		else
		{
			mat = out->x3;
		}

		out->o[i] = 0.0f;
        for (j = 0; j < 3; j++)
		{
			out->o[i] -= (mat[j]*in->o[j]);
		}
	}
}

//multiply a matrix by another matrix into out
void Math_MatrixMultiply(modelMatrix_t *in, modelMatrix_t *in2, modelMatrix_t *out)
{
	out->x1[0] = (in->x1[0]*in2->x1[0]) + (in->x1[1]*in2->x2[0]) + (in->x1[2]*in2->x3[0]);
	out->x1[1] = (in->x1[0]*in2->x1[1]) + (in->x1[1]*in2->x2[1]) + (in->x1[2]*in2->x3[1]);
	out->x1[2] = (in->x1[0]*in2->x1[2]) + (in->x1[1]*in2->x2[2]) + (in->x1[2]*in2->x3[2]);
	out->o[0] = (in->x1[0]*in2->o[0]) + (in->x1[1]*in2->o[1]) + (in->x1[2]*in2->o[2]) + in->o[0];

	out->x2[0] = (in->x2[0]*in2->x1[0]) + (in->x2[1]*in2->x2[0]) + (in->x2[2]*in2->x3[0]);
	out->x2[1] = (in->x2[0]*in2->x1[1]) + (in->x2[1]*in2->x2[1]) + (in->x2[2]*in2->x3[1]);
	out->x2[2] = (in->x2[0]*in2->x1[2]) + (in->x2[1]*in2->x2[2]) + (in->x2[2]*in2->x3[2]);
	out->o[1] = (in->x2[0]*in2->o[0]) + (in->x2[1]*in2->o[1]) + (in->x2[2]*in2->o[2]) + in->o[1];

	out->x3[0] = (in->x3[0]*in2->x1[0]) + (in->x3[1]*in2->x2[0]) + (in->x3[2]*in2->x3[0]);
	out->x3[1] = (in->x3[0]*in2->x1[1]) + (in->x3[1]*in2->x2[1]) + (in->x3[2]*in2->x3[1]);
	out->x3[2] = (in->x3[0]*in2->x1[2]) + (in->x3[1]*in2->x2[2]) + (in->x3[2]*in2->x3[2]);
	out->o[2] = (in->x3[0]*in2->o[0]) + (in->x3[1]*in2->o[1]) + (in->x3[2]*in2->o[2]) + in->o[2];
}

//multiply a matrix by another matrix into out
void Math_MatrixMultiply3x3(modelMatrix_t *in, modelMatrix_t *in2, modelMatrix_t *out)
{
	out->x1[0] = (in->x1[0]*in2->x1[0]) + (in->x1[1]*in2->x2[0]) + (in->x1[2]*in2->x3[0]);
	out->x1[1] = (in->x1[0]*in2->x1[1]) + (in->x1[1]*in2->x2[1]) + (in->x1[2]*in2->x3[1]);
	out->x1[2] = (in->x1[0]*in2->x1[2]) + (in->x1[1]*in2->x2[2]) + (in->x1[2]*in2->x3[2]);

	out->x2[0] = (in->x2[0]*in2->x1[0]) + (in->x2[1]*in2->x2[0]) + (in->x2[2]*in2->x3[0]);
	out->x2[1] = (in->x2[0]*in2->x1[1]) + (in->x2[1]*in2->x2[1]) + (in->x2[2]*in2->x3[1]);
	out->x2[2] = (in->x2[0]*in2->x1[2]) + (in->x2[1]*in2->x2[2]) + (in->x2[2]*in2->x3[2]);

	out->x3[0] = (in->x3[0]*in2->x1[0]) + (in->x3[1]*in2->x2[0]) + (in->x3[2]*in2->x3[0]);
	out->x3[1] = (in->x3[0]*in2->x1[1]) + (in->x3[1]*in2->x2[1]) + (in->x3[2]*in2->x3[1]);
	out->x3[2] = (in->x3[0]*in2->x1[2]) + (in->x3[1]*in2->x2[2]) + (in->x3[2]*in2->x3[2]);
}

//calculate crossproduct between 2 vectors
void Math_CrossProduct(const float *v1, const float *v2, float *cross)
{
	cross[0] = v1[1]*v2[2] - v1[2]*v2[1];
	cross[1] = v1[2]*v2[0] - v1[0]*v2[2];
	cross[2] = v1[0]*v2[1] - v1[1]*v2[0];
}

//calculate dot product of 2 vectors
float Math_DotProduct(const float *v1, const float *v2)
{
	return v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2];
}

//creates a matrix given pos and angles
void Math_CreateMatrixFromOrientation(float *origin, float *angles, modelMatrix_t *mat)
{
	vec3_t axis[3];

	AngleVectors(angles, axis[0], axis[1], axis[2]);
	VectorInverse(axis[1]);

	mat->x1[0] = axis[0][0];
	mat->x2[0] = axis[0][1];
	mat->x3[0] = axis[0][2];

	mat->x1[1] = axis[1][0];
	mat->x2[1] = axis[1][1];
	mat->x3[1] = axis[1][2];

	mat->x1[2] = axis[2][0];
	mat->x2[2] = axis[2][1];
	mat->x3[2] = axis[2][2];

	mat->o[0] = origin[0];
	mat->o[1] = origin[1];
	mat->o[2] = origin[2];
}

//build a rotation matrix
void Math_RotationMatrix(float phi, int axis, modelMatrix_t *mat)
{
	switch (axis)
	{
	case 0: //x
		Math_VecCopy(g_identityMatrix.x1, mat->x1);
		mat->x2[0] = 0.0f;
		mat->x2[1] = cos(phi);
		mat->x2[2] = sin(phi);
		mat->x3[0] = 0.0f;
		mat->x3[1] = -sin(phi);
		mat->x3[2] = cos(phi);
		break;
	case 1: //y
		mat->x1[0] = cos(phi);
		mat->x1[1] = 0.0f;
		mat->x1[2] = sin(phi);
		Math_VecCopy(g_identityMatrix.x2, mat->x2);
		mat->x3[0] = -sin(phi);
		mat->x3[1] = 0.0f;
		mat->x3[2] = cos(phi);
		break;
	case 2: //z
		mat->x1[0] = cos(phi);
		mat->x1[1] = sin(phi);
		mat->x1[2] = 0.0f;
		mat->x2[0] = -sin(phi);
		mat->x2[1] = cos(phi);
		mat->x2[2] = 0.0f;
		Math_VecCopy(g_identityMatrix.x3, mat->x3);
		break;

	default:
		break;
	}

	mat->o[0] = 0.0f;
	mat->o[1] = 0.0f;
	mat->o[2] = 0.0f;
}

//used to orthogonalize a vector
static void Mod_ClosestPointOnLine(float *a, float *b, float *p, float *res)
{
	float dif[3];
	float difx[3];
	float d, t;

	Math_VecSub(p, a, dif);
	Math_VecSub(b, a, difx);
	d = Math_VecLen(difx);
	Math_VecNorm(difx);
	t = Math_DotProduct(difx, dif);
	if (t < 0.0f)
	{
		Math_VecCopy(a, res);
	}
	if (t > d)
	{
		Math_VecCopy(b, res);
	}
	Math_VecScale(difx, t);
	Math_VecAdd(a, difx, res);
}

//orthogonalize vector (tangent/binormal) along normal
static void Mod_Orthogonalize(float *v1, float *v2, float *res)
{
	float v[3];
	float vx[3];
	Math_VecCopy(v1, vx);
	Math_VecScale(vx, -1.0f);
	Mod_ClosestPointOnLine(v1, vx, v2, v);
	Math_VecSub(v2, v, res);
	Math_VecNorm(res);
}

//do the tangent calculations
static void Mod_CreateTangentMatrices(modelMeshObject_t *obj, int numVerts, modelVert_t *verts, int numFaces, modelTriFace_t *faces,
											  modelTangent_t *argTangentData, modelTexCoord_t *coords, float *vertNormals)
{
	int i = 0;
	modelTriFace_t *face = faces;
	modelTexCoord_t *coordBase = coords;

	modelTangent_t *tan = argTangentData;

	memset(tan, 0, sizeof(modelTangent_t)*numVerts);

	while (i < numFaces)
	{
		modelVert_t *v1, *v2, *v3;
		modelTexCoord_t *t1, *t2, *t3;
		modelTangent_t *tan1, *tan2, *tan3;
		float *v1use, *v2use, *v3use;
		float planes[3][3];
		int x;

		v1 = (verts+face->a);
		v2 = (verts+face->b);
		v3 = (verts+face->c);

		tan1 = (tan+face->a);
		tan2 = (tan+face->b);
		tan3 = (tan+face->c);

		v1use = &v1->x;
		v2use = &v2->x;
		v3use = &v3->x;

		t1 = (coordBase+face->a);
		t2 = (coordBase+face->b);
		t3 = (coordBase+face->c);

		x = 0;
		while (x < 3)
		{
			float sc1[3], sc2[3];
			sc1[0] = v2use[x]-v1use[x];
			sc1[1] = t2->u-t1->u;
			sc1[2] = t2->v-t1->v;

			sc2[0] = v3use[x]-v1use[x];
			sc2[1] = t3->u-t1->u;
			sc2[2] = t3->v-t1->v;

			Math_CrossProduct(sc1, sc2, planes[x]);
			x++;
		}

		if (planes[0][0] != 0.0f &&
			planes[1][0] != 0.0f &&
			planes[2][0] != 0.0f)
		{ //otherwise would divide by 0, so, looks like we have us here an invalid tangent.
			float dsdt[3];
			float dtdt[3];

			dsdt[0] = -planes[0][1]/planes[0][0];
			dsdt[1] = -planes[1][1]/planes[1][0];
			dsdt[2] = -planes[2][1]/planes[2][0];
			dtdt[0] = -planes[0][2]/planes[0][0];
			dtdt[1] = -planes[1][2]/planes[1][0];
			dtdt[2] = -planes[2][2]/planes[2][0];

			Math_VecAdd(tan1->tangentMatrix.x1, dsdt, tan1->tangentMatrix.x1);
			Math_VecAdd(tan1->tangentMatrix.x2, dtdt, tan1->tangentMatrix.x2);

			Math_VecAdd(tan2->tangentMatrix.x1, dsdt, tan2->tangentMatrix.x1);
			Math_VecAdd(tan2->tangentMatrix.x2, dtdt, tan2->tangentMatrix.x2);

			Math_VecAdd(tan3->tangentMatrix.x1, dsdt, tan3->tangentMatrix.x1);
			Math_VecAdd(tan3->tangentMatrix.x2, dtdt, tan3->tangentMatrix.x2);
		}

		tan1->numNormals++;
		tan2->numNormals++;
		tan3->numNormals++;

		face++;
		i++;
	}

	i = 0;
	while (i < numVerts)
	{
		float *vNorm;

		tan->tangentMatrix.x1[0] /= tan->numNormals;
		tan->tangentMatrix.x1[1] /= tan->numNormals;
		tan->tangentMatrix.x1[2] /= tan->numNormals;

		tan->tangentMatrix.x2[0] /= tan->numNormals;
		tan->tangentMatrix.x2[1] /= tan->numNormals;
		tan->tangentMatrix.x2[2] /= tan->numNormals;


		Math_VecNorm(tan->tangentMatrix.x1);
		Math_VecNorm(tan->tangentMatrix.x2);

		vNorm = vertNormals+(i*3);

		Math_VecCopy(vNorm, tan->tangentMatrix.x3);

		Mod_Orthogonalize(tan->tangentMatrix.x3, tan->tangentMatrix.x1, tan->tangentMatrix.x1);
		Mod_Orthogonalize(tan->tangentMatrix.x3, tan->tangentMatrix.x2, tan->tangentMatrix.x2);

		tan++;
		i++;
	}
}

//recalculate light vectors etc
void Mod_TangentUpdates(entity_t *e, modelMeshObject_t *obj, float *lightPos)
{
	modelTangent_t *t;
	modelVert_t *verts;
	float *lightVectors;
	int i;

	Mod_CreateTangentMatrices(obj, obj->meshData.numVerts, obj->meshData.vertDataTransformed, obj->meshData.numFaces,
		(modelTriFace_t *)obj->meshData.faceData, obj->meshData.tangents, (modelTexCoord_t *)obj->meshData.uvCoord, obj->meshData.vertNormalsTransformed);

	//now that tangents are updated, calculate light vectors
	t = obj->meshData.tangents;
	verts = obj->meshData.vertDataTransformed;
	lightVectors = obj->meshData.lightVectors;
	i = 0;
	while (i < obj->meshData.numVerts)
	{
		float lightVertDir[3];

		Math_VecSub(lightPos, (float *)verts, lightVertDir);

		lightVectors[0] = -Math_DotProduct(lightVertDir, t->tangentMatrix.x1);
		lightVectors[1] = -Math_DotProduct(lightVertDir, t->tangentMatrix.x2);
		lightVectors[2] = -Math_DotProduct(lightVertDir, t->tangentMatrix.x3);
		Math_VecNorm(lightVectors);

		t++;
		verts++;
		lightVectors += 3;
		i++;
	}
}

#define CHAR_TAB 9 //^^;

//parse through a group for a given key value.
char *ParseMD5GroupForVal(char *group, const char *keyName, int parseSpaces)
{
	static char value[4096];
	int i = 0;

	if (group[i] != '{')
	{ //don't want to be given something that doesn't really start on a group
		return NULL;
	}
	
	value[0] = 0;
	while (group[i])
	{
		while (group[i] &&
			(group[i] == '{' || group[i] == ' ' || group[i] == '\r' || group[i] == '\n' || group[i] == CHAR_TAB))
		{
			i++;
		}

		if (group[i] && group[i] != '}')
		{
			static char checkKey[4096];
			int j = 0;
			int spaces = 0;

			while (group[i] && group[i] != '\r' && group[i] != '\n' && group[i] != CHAR_TAB)
			{
				if (group[i] == '}')
				{
					return NULL;
				}
				else if (group[i] == ' ')
				{
					spaces++;
					if (spaces > parseSpaces)
					{
						break;
					}
				}

				checkKey[j] = group[i];
				j++;
				i++;
			}
			checkKey[j] = 0;
			if (!stricmp(checkKey, keyName))
			{ //found it!
				if (parseSpaces == -1)
				{ //this means, the caller wants me to return the address of the key
					return &group[i];
				}

				while (group[i] &&
					(group[i] == ' ' || group[i] == '\r' || group[i] == '\n' || group[i] == CHAR_TAB))
				{
					i++;
				}
				if (group[i])
				{ //now we should be on the value, so parse it into the static char.
					int stopOnQuote = 0;
					j = 0;

					if (group[i] == '\"')
					{
						stopOnQuote = 1;
						i++;
					}

					while (group[i] && group[i] != '\r' && group[i] != '\n')
					{
						if (stopOnQuote && group[i] == '\"')
						{
							break;
						}
						value[j] = group[i];
						j++;
						i++;
					}
					value[j] = 0;

					//now we should have it, so return the address of our static.
					return value;
				}
				else
				{
					break;
				}
			}
			else
			{
				while (group[i] && group[i] != '\n' && group[i] != '\r')
				{
					i++;
				}
			}
		}
		else
		{
			break;
		}
		i++;
	}

	return NULL;
}

float *StringToVec(const char *val)
{
	static float v[3];
	sscanf(val, "%f %f %f", &v[0], &v[1], &v[2]);

	return v;
}

//create formatted face data
void Mod_CreateFormattedFaces(modelMeshObject_t *obj)
{
	int x = 0;
	modelTriFaceGLFormat_t *faceForm = obj->meshData.faceDataFormatted;
	modelTriFace_t *face = (modelTriFace_t *)obj->meshData.faceData;

	while (x < obj->meshData.numFaces)
	{
		faceForm->a = face->a;
		faceForm->b = face->b;
		faceForm->c = face->c;

		assert(faceForm->a >= 0 && faceForm->a < obj->meshData.numVerts);
		assert(faceForm->b >= 0 && faceForm->b < obj->meshData.numVerts);
		assert(faceForm->c >= 0 && faceForm->c < obj->meshData.numVerts);

		faceForm++;
		face++;
		x++;
	}
}

#define NUMVERTEXNORMALS	162
extern float	r_avertexnormals[NUMVERTEXNORMALS][3];
//calc vert normals
void Mod_CalculateVertNormals(modelMeshObject_t *obj)
{
	int i = 0;
	int j;
	modelVert_t *verts = (modelVert_t *)obj->meshData.vertData;
	modelVert_t *vert = verts;
	modelTriFace_t *face;
	float *normal = obj->meshData.vertNormals;

	modelVert_t *v;
	BYTE *normIndex;

	while (i < obj->meshData.numVerts)
	{
		normal[0] = 0.0f;
		normal[1] = 0.0f;
		normal[2] = 0.0f;
		j = 0;
		face = (modelTriFace_t *)obj->meshData.faceData;
		while (j < obj->meshData.numFaces)
		{
			modelVert_t *v1, *v2, *v3;
			v1 = (verts+face->a);
			v2 = (verts+face->b);
			v3 = (verts+face->c);

			if (vert == v1 ||
				vert == v2 ||
				vert == v3)
			{ //this vert is attached to this face, so add its plane into the will-be average
				float planeEq[4];

				Math_CalcPlaneEq(&v1->x, &v2->x, &v3->x, planeEq);
				Math_VecNorm(planeEq);
				Math_VecAdd(normal, planeEq, normal);
			}

			face++;
			j++;
		}
		Math_VecNorm(normal);

		normal += 3;
		vert++;
		i++;
	}

	//copy all the vert normals over the transformed vert normal array, for until we calculate the transformed normals.
	memcpy(obj->meshData.vertNormalsTransformed, obj->meshData.vertNormals, sizeof(modelVert_t)*obj->meshData.numVerts);

	//to be compliant with the quake vertex lighting system
	//this code is adapted from the modelgen code
	normIndex = obj->meshData.vertNormalIndexes;
	v = (modelVert_t *)obj->meshData.vertNormals;
	for (i=0 ; i<obj->meshData.numVerts ; i++)
	{
		int		j;
		float	maxdot;
		int		maxdotindex;

		maxdot = -999999.0;
		maxdotindex = -1;

		for (j=0 ; j<NUMVERTEXNORMALS ; j++)
		{
			float	dot;

			dot = DotProduct ((&v->x), r_avertexnormals[j]);
			if (dot > maxdot)
			{
				maxdot = dot;
				maxdotindex = j;
			}
		}

		*normIndex = maxdotindex;
		normIndex++;
		v++;
	}
}


//get the anim joint for a bone
modelAnim_t *Mod_GetAnimJointForBone(modelAnim_t *anim, modelBone_t *bone)
{
	modelAnim_t *a = anim;

	while (a)
	{
		if (a->completed &&
			!stricmp(a->name, bone->name))
		{
			return a;
		}

		a = a->next;
	}

	return NULL;
}

//get the matrix for the given frame on a bone
void Mod_FrameMatrixForBoneAnim(modelBone_t *bone, modelAnim_t *boneAnim, int frame, modelMatrix_t *out)
{
	float *xyz, *ang;
	modelMatrix_t baseMat;
	modelMatrix_t m1, m2, m3, tmp, tmp2;
	float transXYZ[3];
	const int ignoreOriginPos = 1;

	//it would actually be good to precalculate all this into a matrix.
	//however, with my current seperation of animation and model objects,
	//this isn't doable, since the animation object contains no actual
	//hierarchal data.
	//if i were doing this from scratch outside of quake architecture,
	//i would probably bind animations and models together to allow for
	//this. it would also prevent the necessary "animation joint" lookup
	//for every bone transform.

	if (ignoreOriginPos && !bone->eData.parent)
	{ //this is a bit of a hack, because i don't want to deal with models moving out of their bbox gameside
		static float zero[3] = {0.0f, 0.0f, 0.0f};

		xyz = zero;
	}
	else
	{
		xyz = (boneAnim->xyz+(frame*3));
	}
	ang = (boneAnim->angles+(frame*3));

	//we need to use the parent's matrix as our base, since offsets
	//are stored additively. if no parent, identity will do.
	if (bone->eData.parent)
	{
		baseMat = bone->eData.parent->eData.finalMat;
	}
	else
	{
		baseMat = g_identityMatrix;
	}

	//quite frankly, i'm not sure why i have to invert the yaw and roll.
	//but i do. so, bling.
	Math_RotationMatrix((-ang[1])*0.0174532925f, 2, &m1);
	Math_RotationMatrix((ang[0])*0.0174532925f, 1, &m2);
	Math_RotationMatrix((-ang[2])*0.0174532925f, 0, &m3);

	//the origin for the bone needs to be transformed by the parent's
	//matrix (the 3x3 portion only, because bone positions are
	//additive on their own)
	Math_TransformPointByMatrix3x3(&baseMat, xyz, transXYZ);
	Math_VecAdd(baseMat.o, transXYZ, baseMat.o);

	//now multiply the matrix by the new rotation matrix into the
	//output matrix.
	Math_MatrixMultiply(&baseMat, &m1, &tmp);
	Math_MatrixMultiply(&tmp, &m2, &tmp2);
	Math_MatrixMultiply(&tmp2, &m3, out);
}

//transform a bone, making sure its parent is transformed as well.
void Mod_TransformBone(modelMeshObject_t *obj, modelBone_t *bone, modelAnim_t *animRoot, int frame, int oldframe, float lerpAmt)
{
	modelMatrix_t matCur, matLast;
	modelAnim_t *boneAnim;

	if (bone->eData.parent)
	{ //make sure parent is transformed first.
		Mod_TransformBone(obj, bone->eData.parent, animRoot, frame, oldframe, lerpAmt);
	}

	if (bone->eData.renderCount == obj->meshData.renderCountCurrent)
	{ //already transformed
		return;
	}
	bone->eData.renderCount = obj->meshData.renderCountCurrent;

	boneAnim = Mod_GetAnimJointForBone(animRoot, bone);
	if (!boneAnim)
	{ //well then.
		if (bone->eData.parent)
		{
			bone->eData.finalMat = bone->eData.parent->eData.finalMat;
		}
		else
		{
			bone->eData.finalMat = g_identityMatrix;
		}
		return;
	}

	//make sure frame and oldframe are within range
	if (frame >= boneAnim->numKeys)
	{
		frame = boneAnim->numKeys-1;
	}
	if (frame < 0)
	{
		frame = 0;
	}
	if (oldframe >= boneAnim->numKeys)
	{
		oldframe = boneAnim->numKeys-1;
	}
	if (oldframe < 0)
	{
		oldframe = 0;
	}

	//lets get the matrices for the current frame and last frame
	Mod_FrameMatrixForBoneAnim(bone, boneAnim, frame, &matCur);
	Mod_FrameMatrixForBoneAnim(bone, boneAnim, oldframe, &matLast);

	//now, lets lerp between them.
	if (!lerpAmt)
	{ //looks like it's totally up to date, so don't lerp at all
		bone->eData.finalMat = matCur;
	}
	else
	{
		modelMatrix_t m;
		float lerpFrac = 1.0f-lerpAmt;

		Math_VecSub(matCur.x1, matLast.x1, m.x1);
		Math_VecSub(matCur.x2, matLast.x2, m.x2);
		Math_VecSub(matCur.x3, matLast.x3, m.x3);
		Math_VecSub(matCur.o, matLast.o, m.o);
		Math_VecScale(m.x1, lerpFrac);
		Math_VecScale(m.x2, lerpFrac);
		Math_VecScale(m.x3, lerpFrac);
		Math_VecScale(m.o, lerpFrac);

		bone->eData.finalMat = matLast;
		Math_VecAdd(bone->eData.finalMat.x1, m.x1, bone->eData.finalMat.x1);
		Math_VecAdd(bone->eData.finalMat.x2, m.x2, bone->eData.finalMat.x2);
		Math_VecAdd(bone->eData.finalMat.x3, m.x3, bone->eData.finalMat.x3);
		Math_VecAdd(bone->eData.finalMat.o, m.o, bone->eData.finalMat.o);
	}
}

//make sure all bones are transformed.
void Mod_TransformBones(entity_t *e, modelMeshObject_t *obj, modelAnim_t *anim)
{
	modelBone_t *b = obj->meshData.bones;
	int numB = 0;
	while (numB < obj->meshData.numBones)
	{
		Mod_TransformBone(obj, b, anim, e->frame, e->oldframe, e->backlerp);

		b++;
		numB++;
	}
}

//create transformed verts for animations etc
void Mod_CreateTransformedVerts(entity_t *e, modelMeshObject_t *obj, modelMeshObject_t *root, model_t *anm)
{
	modelAnim_t *anim;
	modelVert_t *transNormals;
	modelVert_t *firstNormals;
	modelVert_t *v;
	modelVertWInfo_t *vWInfo;
	int i;

	if (!e || !anm || !anm->md5anim)
	{ //just copy the verts across and be done with it.
		modelVert_t *t = obj->meshData.vertDataTransformed;
		modelVert_t *o = (modelVert_t *)obj->meshData.vertData;

		memcpy(t, o, sizeof(modelVert_t)*obj->meshData.numVerts);
		return;
	}

	anim = anm->md5anim;

	//ok, transform the bones.
	Mod_TransformBones(e, obj, anim);

	//now set up the verts
	transNormals = (modelVert_t *)obj->meshData.vertNormalsTransformed;
	firstNormals = (modelVert_t *)obj->meshData.vertNormals;
	v = obj->meshData.vertDataTransformed;
	vWInfo = obj->meshData.vertWInfo;
	i = 0;
	while (i < obj->meshData.numVerts)
	{
		int w = 0;

		v->x = 0.0f;
		v->y = 0.0f;
		v->z = 0.0f;
		if (transNormals)
		{
			transNormals->x = 0.0f;
			transNormals->y = 0.0f;
			transNormals->z = 0.0f;
		}

		while (w < vWInfo->numWeights)
		{
			modelVertWeight_t *weight;
			modelBone_t *weightedBone;
			modelMatrix_t *transMat;
			float p[3];

			weight = obj->meshData.vertWeights+(vWInfo->weightIndex+w);
			weightedBone = root->meshData.bones+weight->boneIndex;

			assert(weightedBone && weightedBone->index == weight->boneIndex && weight->index == (vWInfo->weightIndex+w));

			transMat = &weightedBone->eData.finalMat;

			//you know, it would really make more sense if animations were additive off the
			//initially formed static mesh. what the hell is the point of even having original
			//matrix/position in the bones within the mesh then? i guess it's convenience so
			//the model can be rendered without animations. oh well.
			Math_TransformPointByMatrix(transMat, weight->pos, p);
			Math_VecScale(p, weight->weightFactor);

			Math_VecAdd((float *)v, p, (float *)v);

			if (transNormals)
			{ //this is tricky. extrude a point off the base vert and transform that,
				//then we know the direction between the extruded transformed point
				//and normal transformed point is our transformed vertex normal.
				float x[3];

				float extr[3];
				Math_VecAdd((float *)weight->pos, (float *)firstNormals, extr);

				Math_TransformPointByMatrix(transMat, extr, x);
				Math_VecScale(x, weight->weightFactor);

				Math_VecSub(x, p, x);
				Math_VecAdd((float *)transNormals, x, (float *)transNormals);
			}

			w++;
		}

		if (transNormals)
		{
			Math_VecNorm((float *)transNormals);

			transNormals++;
			firstNormals++;
		}

		vWInfo++;
		v++;

		i++;
	}
}

//get bone by name
modelBone_t *Mod_GetBoneByName(int numBones, modelBone_t *boneRoot, const char *name)
{
	modelBone_t *b = boneRoot;
	int i = 0;

	while (i < numBones)
	{
		if (!stricmp(b->name, name))
		{
			return b;
		}

		i++;
		b++;
	}

	return NULL;
}

//fill in parent pointers for quick access while checking for matrix overrides
void Mod_FillBoneParentPtrs(modelMeshObject_t *root)
{
	modelBone_t *b = root->meshData.bones;
	int i = 0;

	while (i < root->meshData.numBones)
	{
		if (b->parentName[0])
		{
			b->eData.parent = Mod_GetBoneByName(root->meshData.numBones, root->meshData.bones, b->parentName);
		}
		else
		{
			b->eData.parent = NULL;
		}

		i++;
		b++;
	}
}

//most of the parsing code here is disgusting, as i ripped it from my command line tool.
//please excuse the mess. ;_;

//parse the md5 and shove it into a modelMeshObject_t
modelMeshObject_t *Mod_LoadMD5Parse(BYTE *fileBuffer, int bufferLen)
{
	modelMeshObject_t *firstObj = (modelMeshObject_t *)malloc(sizeof(modelMeshObject_t));
	modelMeshObject_t **nObj = &firstObj;
	modelBone_t **bone = &firstObj->meshData.bones;
	char *p = (char *)fileBuffer;
	char *val;
	int bones = 0;
	int i = 0;
	int objectNum = 0;
	int somethingWentWrong = 0;

	memset(firstObj, 0, sizeof(modelMeshObject_t));

	while (i < bufferLen)
	{
		while (p[i] && p[i] != '{')
		{
			i++;
		}

		if (p[i] == '{')
		{
			int openBraces;
			int x = i-1;

			while (x > 0 && p[x] != '\n' && p[x] != '\r')
			{
				x--;
			}

			if (x >= 0)
			{
				char groupName[1024];
				int j = 0;

				while (p[x] && p[x] != '{')
				{
					groupName[j] = p[x];
					j++;
					x++;
				}
				groupName[j] = 0;

				//now, lets see what kind of object this sucker is.
				if (p[x] == '{')
				{
					if (strstr(groupName, "bone"))
					{ //it's a bone
						if (!firstObj->meshData.numBones)
						{
							modelBone_t *b;

							//allocate room for a new bone to shove data into.
							while (*bone)
							{
								bone = &((*bone)->eData.next);
							}
							*bone = (modelBone_t *)malloc(sizeof(modelBone_t));
							memset(*bone, 0, sizeof(modelBone_t));

							b = *bone;

							val = ParseMD5GroupForVal(&p[x], "name", 0);
							if (val)
							{
								strcpy(b->name, val);
							}
							else
							{
								strcpy(b->name, "unknownName");
							}

							val = ParseMD5GroupForVal(&p[x], "parent", 0);
							if (val)
							{
								strcpy(b->parentName, val);
							}
							else
							{
								b->parentName[0] = 0;
							}

							val = ParseMD5GroupForVal(&p[x], "bindpos", 0);
							if (val)
							{
								float *v = StringToVec(val);
								b->mat.o[0] = v[0];
								b->mat.o[1] = v[1];
								b->mat.o[2] = v[2];
							}
							else
							{
								b->mat.o[0] = g_identityMatrix.o[0];
								b->mat.o[1] = g_identityMatrix.o[1];
								b->mat.o[2] = g_identityMatrix.o[2];
								somethingWentWrong = 1;
							}

							val = ParseMD5GroupForVal(&p[x], "bindmat", 0);
							if (val)
							{
								sscanf(val, "%f %f %f %f %f %f %f %f %f", &b->mat.x1[0], &b->mat.x2[0], &b->mat.x3[0],
									&b->mat.x1[1], &b->mat.x2[1], &b->mat.x3[1], &b->mat.x1[2], &b->mat.x2[2], &b->mat.x3[2]);
							}
							else
							{
								b->mat.x1[0] = g_identityMatrix.x1[0];
								b->mat.x1[1] = g_identityMatrix.x1[1];
								b->mat.x1[2] = g_identityMatrix.x1[2];
								b->mat.x2[0] = g_identityMatrix.x2[0];
								b->mat.x2[1] = g_identityMatrix.x2[1];
								b->mat.x2[2] = g_identityMatrix.x2[2];
								b->mat.x3[0] = g_identityMatrix.x3[0];
								b->mat.x3[1] = g_identityMatrix.x3[1];
								b->mat.x3[2] = g_identityMatrix.x3[2];
								somethingWentWrong = 1;
							}

							b->index = bones;
							bones++;
						}
						else
						{
							ri.Con_Printf(PRINT_ALL, "Found a bone out of order!\n");
						}
					}
					else if (strstr(groupName, "mesh"))
					{ //vert, tri, weight data
						modelMeshObject_t *obj;

						//ok, hack, presume this means we allocated all the bones.
						if (bones > 0 && !firstObj->meshData.numBones && firstObj->meshData.bones)
						{
							//note that i am only doing it this way because my tool didn't have to care about efficiently accessing
							//bone data, but since we are parsing into the same format we will be using in game, having the bones
							//aligned is a must. and of course, i am too lazy to rewrite things to do this the right way by just
							//getting the number of bones first off. ^^;
							modelBone_t *boneAllocation, *b, *bTo, *bN;

							firstObj->meshData.numBones = bones;

							boneAllocation = (modelBone_t *)malloc(sizeof(modelBone_t)*firstObj->meshData.numBones);
							
							//now lets copy them over
							b = firstObj->meshData.bones;
							bTo = boneAllocation;
							while (b)
							{
								memcpy(bTo, b, sizeof(modelBone_t));

								bTo++;
								b = b->eData.next;
							}

							//now free all the old ones
							b = firstObj->meshData.bones;
							while (b)
							{
								bN = b->eData.next;
								free(b);
								b = bN;
							}
							
							//now, set the pointer on the object to the correct one
							firstObj->meshData.bones = boneAllocation;
						}

						if (!(*nObj))
						{ //no current object ptr is null, make a new one
							*nObj = (modelMeshObject_t *)malloc(sizeof(modelMeshObject_t));
							memset(*nObj, 0, sizeof(modelMeshObject_t));
						}						
						obj = *nObj;

						val = ParseMD5GroupForVal(&p[x], "numverts", 0);
						if (val)
						{ //we totally need to know how many verts there are beforehand
							obj->meshData.numVerts = atoi(val);

							val = ParseMD5GroupForVal(&p[x], "numtris", 0);
							if (val)
							{ //need to know tris too
								obj->meshData.numFaces = atoi(val);

								val = ParseMD5GroupForVal(&p[x], "numweights", 0);
								if (val)
								{ //and finally, weights
									modelTexCoord_t *uvCoord;
									modelVertWInfo_t *vWInfo;
									modelTriFace_t *face;
									modelVertWeight_t *weights;
									modelVert_t *verts;
									int v;
									char findStr[256];

									obj->meshData.numWeights = atoi(val);

									//now, let us allocate room for everything.
									obj->meshData.vertData = (BYTE *)malloc(sizeof(modelVert_t)*obj->meshData.numVerts);
									obj->meshData.vertWInfo = (modelVertWInfo_t *)malloc(sizeof(modelVertWInfo_t)*obj->meshData.numVerts);
									obj->meshData.uvCoord = (BYTE *)malloc(sizeof(modelTexCoord_t)*obj->meshData.numVerts);
									obj->meshData.faceData = (BYTE *)malloc(sizeof(modelTriFace_t)*obj->meshData.numFaces);
									obj->meshData.vertWeights = (modelVertWeight_t *)malloc(sizeof(modelVertWeight_t)*obj->meshData.numWeights);

									//extra
									obj->meshData.vertDataTransformed = (modelVert_t *)malloc(sizeof(modelVert_t)*obj->meshData.numVerts);
									obj->meshData.faceDataFormatted = (modelTriFaceGLFormat_t *)malloc(sizeof(modelTriFaceGLFormat_t)*obj->meshData.numFaces);
									obj->meshData.colorPointer = (modelRGBA_t *)malloc(sizeof(modelRGBA_t)*obj->meshData.numVerts);
									obj->meshData.colorPointerVec = (modelRGB_t *)malloc(sizeof(modelRGB_t)*obj->meshData.numVerts);
									obj->meshData.vertNormals = (float *)malloc(sizeof(modelVert_t)*obj->meshData.numVerts);
									obj->meshData.vertNormalsTransformed = (float *)malloc(sizeof(modelVert_t)*obj->meshData.numVerts);
									obj->meshData.vertNormalIndexes = (BYTE *)malloc(sizeof(BYTE)*obj->meshData.numVerts);
									obj->meshData.tangents = (modelTangent_t *)malloc(sizeof(modelTangent_t)*obj->meshData.numVerts);
									obj->meshData.lightVectors = (float *)malloc(sizeof(modelVert_t)*obj->meshData.numVerts);


									memset(obj->meshData.vertData, 0, sizeof(modelVert_t)*obj->meshData.numVerts);

									val = ParseMD5GroupForVal(&p[x], "shader", 0);
									if (val)
									{
										strcpy(obj->meshData.skinName, val);
									}
									else
									{
										strcpy(obj->meshData.skinName, "default");
									}
									val = ParseMD5GroupForVal(&p[x], "shader_normal", 0);
									if (val)
									{
										strcpy(obj->meshData.skinNameNormal, val);
									}
									else
									{
										strcpy(obj->meshData.skinNameNormal, "");
									}


									uvCoord = (modelTexCoord_t *)obj->meshData.uvCoord;
									vWInfo = (modelVertWInfo_t *)obj->meshData.vertWInfo;
									v = 0;

									while (v < obj->meshData.numVerts)
									{
										sprintf(findStr, "vert %i", v);
										val = ParseMD5GroupForVal(&p[x], findStr, 1); //fixme: this is a crappy slow way to look up subsequent values
										if (val)
										{
											sscanf(val, "%f %f %i %i", &uvCoord->u, &uvCoord->v, &vWInfo->weightIndex, &vWInfo->numWeights);
										}

										uvCoord++;
										vWInfo++;

										v++;
									}
									ri.Con_Printf(PRINT_ALL, "Finished vertex processing for object %i.\n", objectNum);

									face = (modelTriFace_t *)obj->meshData.faceData;
									v = 0;

									while (v < obj->meshData.numFaces)
									{
										sprintf(findStr, "tri %i", v);
										val = ParseMD5GroupForVal(&p[x], findStr, 1); //fixme: this is a crappy slow way to look up subsequent values
										if (val)
										{
											int faceA, faceB, faceC;
											sscanf(val, "%i %i %i", &faceA, &faceB, &faceC);
											face->a = (WORD)faceA;
											face->b = (WORD)faceB;
											face->c = (WORD)faceC;
											face->flag = 0;
										}

										face++;

										v++;
									}
									ri.Con_Printf(PRINT_ALL, "Finished tri processing for object %i.\n", objectNum);

									weights = obj->meshData.vertWeights;
									v = 0;

									while (v < obj->meshData.numWeights)
									{
										sprintf(findStr, "weight %i", v);
										val = ParseMD5GroupForVal(&p[x], findStr, 1); //fixme: this is a crappy slow way to look up subsequent values
										if (val)
										{
											sscanf(val, "%i %f %f %f %f", &weights->boneIndex, &weights->weightFactor, &weights->pos[0], &weights->pos[1], &weights->pos[2]);
											weights->index = v;
										}

										weights++;

										v++;
									}
									ri.Con_Printf(PRINT_ALL, "Finished weight processing for object %i.\n", objectNum);

									ri.Con_Printf(PRINT_ALL, "Creating standard static mesh from vertex weights for object %i...", objectNum);
									verts = (modelVert_t *)obj->meshData.vertData;
									vWInfo = (modelVertWInfo_t *)obj->meshData.vertWInfo;
									v = 0;
									while (v < obj->meshData.numVerts)
									{
										int w;

										verts->x = 0;
										verts->y = 0;
										verts->z = 0;

										w = 0;
										while (w < vWInfo->numWeights)
										{
											modelVertWeight_t *weight;
											modelBone_t *weightedBone;

											weight = obj->meshData.vertWeights+(vWInfo->weightIndex+w);
											weightedBone = firstObj->meshData.bones+weight->boneIndex;

											if (weightedBone &&
												weightedBone->index == weight->boneIndex && weight->index == (vWInfo->weightIndex+w))
											{
												float p[3];
												modelMatrix_t *transMat = &weightedBone->mat;

												Math_TransformPointByMatrix(transMat, weight->pos, p);
												Math_VecScale(p, weight->weightFactor);

												Math_VecAdd((float *)verts, p, (float *)verts);
											}
											else
											{
												ri.Con_Printf(PRINT_ALL, "CRITICAL ERROR: Bone/weight index values do not match up!\n");
												somethingWentWrong = 1;
											}

											w++;
										}

										vWInfo++;
										verts++;

										v++;
									}
									ri.Con_Printf(PRINT_ALL, "Finished!\n");
								}
								else
								{
									somethingWentWrong = 1;
								}
							}
							else
							{
								somethingWentWrong = 1;
							}
						}
						else
						{
							somethingWentWrong = 1;
						}

						nObj = &(*nObj)->next;
						objectNum++;
					}
				}
			}

			openBraces = 0;
			while (p[i] != '}' || openBraces > 0)
			{
				i++;
				if (p[i] == '{')
				{
					openBraces++;
				}
				else if (p[i] == '}')
				{
					openBraces--;
				}
			}
		}
		else
		{
			break;
		}

		i++;
	}

	if (somethingWentWrong)
	{
		ri.Con_Printf(PRINT_ALL, "Errors occured while parsing the md5mesh.\nThe mesh may contain errors.\n");
	}

	//fill in parent pointers for bones
	Mod_FillBoneParentPtrs(firstObj);

	return firstObj;
}

//kill all the mallocs made for the md5
void Mod_FreeUpMD5(modelMeshObject_t *obj)
{
	modelMeshObject_t *n;

	while (obj)
	{
		if (obj->meshData.bones)
		{
			free(obj->meshData.bones);
		}
		if (obj->meshData.vertData)
		{
			free(obj->meshData.vertData);
		}
		if (obj->meshData.vertWInfo)
		{
			free(obj->meshData.vertWInfo);
		}
		if (obj->meshData.uvCoord)
		{
			free(obj->meshData.uvCoord);
		}
		if (obj->meshData.faceData)
		{
			free(obj->meshData.faceData);
		}
		if (obj->meshData.vertWeights)
		{
			free(obj->meshData.vertWeights);
		}
		if (obj->meshData.vertDataTransformed)
		{
			free(obj->meshData.vertDataTransformed);
		}
		if (obj->meshData.faceDataFormatted)
		{
			free(obj->meshData.faceDataFormatted);
		}
		if (obj->meshData.colorPointer)
		{
			free(obj->meshData.colorPointer);
		}
		if (obj->meshData.colorPointerVec)
		{
			free(obj->meshData.colorPointerVec);
		}
		if (obj->meshData.vertNormals)
		{
			free(obj->meshData.vertNormals);
		}
		if (obj->meshData.vertNormalsTransformed)
		{
			free(obj->meshData.vertNormalsTransformed);
		}
		if (obj->meshData.vertNormalIndexes)
		{
			free(obj->meshData.vertNormalIndexes);
		}
		if (obj->meshData.tangents)
		{
			free(obj->meshData.tangents);
		}
		if (obj->meshData.lightVectors)
		{
			free(obj->meshData.lightVectors);
		}

		n = obj->next;
		free(obj);
		obj = n;
	}
}

//free up md5anim allocations
void Mod_FreeUpMD5Anim(modelAnim_t *anm)
{
	modelAnim_t *n;
	while (anm)
	{
		if (anm->xyz)
		{
			free(anm->xyz);
		}
		if (anm->angles)
		{
			free(anm->angles);
		}

		n = anm->next;
		free(anm);
		anm = n;
	}
	return;
}

//interface with q2's model poop, to call my md5 model poop
int Mod_AllocateMD5Mesh(model_t *mod, BYTE *buf, int len)
{
	modelMeshObject_t *root = Mod_LoadMD5Parse(buf, len);
	modelMeshObject_t *obj;
	if (!root)
	{
		return 0;
	}

	//precalculate some stuff
	obj = root;
	while (obj)
	{
		Mod_CreateFormattedFaces(obj);
		Mod_CreateTransformedVerts(NULL, obj, root, NULL);
		Mod_CalculateVertNormals(obj);
		obj = obj->next;
	}

	mod->type = mod_md5;
	mod->md5 = root;
	return 1;
}

//see if one exists by said name
modelAnim_t *Mod_GetAnimObject(modelAnim_t *root, const char *name)
{
	modelAnim_t *a = root;
	while (a)
	{
		if (!stricmp(a->name, name))
		{
			return a;
		}
		a = a->next;
	}

	return NULL;
}

//allocate or find unused object from root
modelAnim_t *Mod_AllocNewAnimObj(modelAnim_t *root)
{
	modelAnim_t **a = &root;
	while (*a)
	{
		if (!(*a)->inuse)
		{ //not in use, so just use this.
			return *a;
		}

		a = &(*a)->next;
	}

	//if we got here, then we've reached the end of the chain, so malloc a new one.
	*a = (modelAnim_t *)malloc(sizeof(modelAnim_t));
	memset(*a, 0, sizeof(modelAnim_t));

	return *a;
}

//this.. sucks, kind of. but it would be a pain to precalculate the number of keys for
//a bone based on all the channels, so i'm just allocating a set amount, for now.
#define MAX_ANIM_KEYS	4096

//parse all the keys in a group to a float buffer.
int ParseMD5AnimKeys(char *parseGroup, float *outBuf, modelAnim_t *joint, int attribKeys, int rangeStart, int keySize)
{
	int i = 0;
	char keyVal[256];
	char *keyAdd = ParseMD5GroupForVal(parseGroup, "keys", -1);
	float lastKey;

	if (!keyAdd)
	{
		ri.Con_Printf(PRINT_ALL, "Group parse error on joint %s.\n", joint->name);
		return 0;
	}

	keyAdd++;

	while (keyAdd[i] && keyAdd[i] != ' ' &&
		keyAdd[i] != '\r' && keyAdd[i] != '\n' &&
		keyAdd[i] != CHAR_TAB)
	{
		i++;
	}

	if (keyAdd[i])
	{ //kk, this should be the beginning of the keys themselves
		int j = i;
		i = 0;
		while (i < attribKeys)
		{
			int n = 0;
			while (keyAdd[j] && 
				(keyAdd[j] == ' ' ||
				 keyAdd[j] == '\r' ||
				 keyAdd[j] == '\n' ||
				 keyAdd[j] == CHAR_TAB) )
			{
				j++;
			}
			//this should be the start of the key
			while (keyAdd[j] && 
				(keyAdd[j] != ' ' &&
				 keyAdd[j] != '\r' &&
				 keyAdd[j] != '\n' &&
				 keyAdd[j] != CHAR_TAB) )
			{
				keyVal[n] = keyAdd[j];
				n++;
				j++;
			}
			keyVal[n] = 0;

			*outBuf = atof(keyVal);
			outBuf += keySize;
			i++;
		}
	}

	lastKey = *(outBuf-keySize);

	//fill in the rest with the last value... this is a little bad, and is some of the logic
	//that would have to be reworked if i did the attribute allocation the right way.
	while (i < (MAX_ANIM_KEYS-rangeStart))
	{
		*outBuf = lastKey;
		outBuf += keySize;

		i++;
	}

	return 1;
}

//parse the contents of the md5anim
modelAnim_t *Mod_LoadMD5AnimParse(BYTE *buf, int len)
{
	modelAnim_t *root = (modelAnim_t *)malloc(sizeof(modelAnim_t));
	char *p = (char *)buf;
	char *val;
	int i = 0;

	memset(root, 0, sizeof(modelAnim_t));

	while (i < len)
	{
		while (p[i] && p[i] != '{')
		{
			i++;
		}

		if (p[i] == '{')
		{
			int openBraces;
			int x = i-1;

			while (x > 0 && p[x] != '\n' && p[x] != '\r')
			{
				x--;
			}

			if (x >= 0)
			{
				char groupName[1024];
				int j = 0;

				while (p[x] && p[x] != '{')
				{
					groupName[j] = p[x];
					j++;
					x++;
				}
				groupName[j] = 0;

				//now, lets see what kind of object this sucker is.
				if (p[x] == '{')
				{
					if (strstr(groupName, "channel"))
					{ //alright then...
						val = ParseMD5GroupForVal(&p[x], "joint", 0);
						if (val)
						{
							modelAnim_t *joint;
							int rangeStart;
							const int keySize = sizeof(float)*3;

							joint = Mod_GetAnimObject(root, val);
							if (!joint)
							{ //not found, so make a new one
								joint = Mod_AllocNewAnimObj(root);
								assert(joint); //should never fail

								strcpy(joint->name, val);
								joint->inuse = 1;

								joint->xyz = (float *)malloc(keySize*MAX_ANIM_KEYS);
								memset(joint->xyz, 0, keySize*MAX_ANIM_KEYS);
								joint->angles = (float *)malloc(keySize*MAX_ANIM_KEYS);
								memset(joint->angles, 0, keySize*MAX_ANIM_KEYS);

								joint->startTime = 99999; //hack (obviously)

								//just get the framerate once. if it changes between tracks for one bone, then.. blargh.
								val = ParseMD5GroupForVal(&p[x], "framerate", 0);
								if (val)
								{
									joint->frameRate = atof(val);
								}
								//ri.Con_Printf(PRINT_ALL, "Created bone %s.\n", joint->name);
							}

							val = ParseMD5GroupForVal(&p[x], "starttime", 0);
							if (val)
							{
								float fVal = atof(val);
								if (fVal < joint->startTime)
								{ //before the current startTime, so use it instead.
									joint->startTime = fVal;
								}
							}
							val = ParseMD5GroupForVal(&p[x], "endtime", 0);
							if (val)
							{
								float fVal = atof(val);
								if (fVal > joint->endTime)
								{ //after the current endTime, so use it instead.
									joint->endTime = fVal;
								}
							}
							val = ParseMD5GroupForVal(&p[x], "range", 0);
							if (val)
							{
								char findNextPart[512];
								int l = 0;

								while (val[l] && val[l] != '\n' && val[l] != '\r' && val[l] != ' ')
								{
									l++;
								}

								if (val[l] == ' ')
								{ //should always land on a space, since this should be a two-part arg, but just in case, we check.
									val[l] = 0;
									rangeStart = atoi(val);
									sprintf(findNextPart, "range %s", val);
									val = ParseMD5GroupForVal(&p[x], findNextPart, 1);
									if (val)
									{ 
										joint->rangeEnd = atoi(val);

										val = ParseMD5GroupForVal(&p[x], "keys", 0);
										if (val)
										{
											int attribKeys = atoi(val);

											if (attribKeys > joint->numKeys)
											{ //joint's numKeys keeps the highest number of keys in any channel for the joint
												joint->numKeys = attribKeys;
											}

											if (joint->numKeys < MAX_ANIM_KEYS)
											{
												val = ParseMD5GroupForVal(&p[x], "attribute", 0);
												if (val)
												{ //ok, got the attribute, now we know where we should be parsing to.
													float *parseFloatTo = NULL;

													if (!stricmp(val, "x"))
													{
														parseFloatTo = (joint->xyz);
													}
													else if (!stricmp(val, "y"))
													{
														parseFloatTo = (joint->xyz+1);
													}
													else if (!stricmp(val, "z"))
													{
														parseFloatTo = (joint->xyz+2);
													}
													else if (!stricmp(val, "pitch"))
													{
														parseFloatTo = (joint->angles); //PITCH == 0
													}
													else if (!stricmp(val, "yaw"))
													{
														parseFloatTo = (joint->angles+1); //YAW == 1
													}
													else if (!stricmp(val, "roll"))
													{
														parseFloatTo = (joint->angles+2); //ROLL == 2
													}

													if (parseFloatTo)
													{ //ok, we got something to parse to, so lets finally parse the keys.
														//skip to the start of the range for this guy
														parseFloatTo += (rangeStart*3);

														if (ParseMD5AnimKeys(&p[x], parseFloatTo, joint, attribKeys, rangeStart, 3))
														{
															joint->completed = 1; //so we know this bone contains data we can really use
														}
													}
												}
											}
											else
											{ //fixme, maybe if i'm gonna keep using my lazy method, at least try re-allocating the keys?
												//the reason it's difficult is because numKeys can vary from attribute to attribute on a
												//single joint/bone, and assumptions are made about ranges, so reallocating even keeping
												//the same data causes some extra logic issues.
												ri.Con_Printf(PRINT_ALL, "WARNING: keys for bone %s exceed MAX_ANIM_KEYS!\n", joint->name);
											}
										}
									}
								}
							}
						}
					}
				}
			}

			openBraces = 0;
			while (p[i] != '}' || openBraces > 0)
			{
				i++;
				if (p[i] == '{')
				{
					openBraces++;
				}
				else if (p[i] == '}')
				{
					openBraces--;
				}
			}
		}
		else
		{
			break;
		}

		i++;
	}

	return root;
}

//animation instances are stored as models, purely for convenience.
int Mod_AllocateMD5Anim(model_t *mod, BYTE *buf, int len)
{
	modelAnim_t *root = Mod_LoadMD5AnimParse(buf, len);
	if (!root)
	{
		return 0;
	}

	mod->type = mod_md5anim;
	mod->md5anim = root;
	return 1;
}
//rww end

/*
==================
Mod_ForName

Loads in a model for the given name
==================
*/
model_t *Mod_ForName (char *name, qboolean crash)
{
	model_t	*mod;
	unsigned *buf;
	int		i;
	//rww begin
	int		wasMD5 = 0;
	//rww end
	
	if (!name[0])
		ri.Sys_Error (ERR_DROP, "Mod_ForName: NULL name");
		
	//
	// inline models are grabbed only from worldmodel
	//
	if (name[0] == '*')
	{
		i = atoi(name+1);
		if (i < 1 || !r_worldmodel || i >= r_worldmodel->numsubmodels)
			ri.Sys_Error (ERR_DROP, "bad inline model number");
		return &mod_inline[i];
	}

	//
	// search the currently loaded models
	//
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++)
	{
		if (!mod->name[0])
			continue;
		if (!strcmp (mod->name, name) )
			return mod;
	}
	
	//
	// find a free model slot spot
	//
	for (i=0 , mod=mod_known ; i<mod_numknown ; i++, mod++)
	{
		if (!mod->name[0])
			break;	// free spot
	}
	if (i == mod_numknown)
	{
		if (mod_numknown == MAX_MOD_KNOWN)
			ri.Sys_Error (ERR_DROP, "mod_numknown == MAX_MOD_KNOWN");
		mod_numknown++;
	}
	strcpy (mod->name, name);
	
	//
	// load the file
	//
	modfilelen = ri.FS_LoadFile (mod->name, &buf);
	if (!buf)
	{
		if (crash)
			ri.Sys_Error (ERR_DROP, "Mod_NumForName: %s not found", mod->name);
		memset (mod->name, 0, sizeof(mod->name));
		return NULL;
	}
	
	loadmodel = mod;

	//
	// fill it in
	//


	//rww begin
	// call the apropriate loader
	if (strstr(mod->name, "md5mesh"))
	{ //if it's an md5 mesh
		if (Mod_AllocateMD5Mesh(mod, (BYTE *)buf, modfilelen))
		{
			wasMD5 = 1;
		}
	}

	if (!wasMD5)
	{ //ok, see if it's an anim
		if (strstr(mod->name, "md5anim"))
		{
			if (Mod_AllocateMD5Anim(mod, (BYTE *)buf, modfilelen))
			{
				wasMD5 = 1;
			}
		}
	}
	
	if (!wasMD5)
	{ //if my md5 stuff didn't handle it, do this
		//rww end
		switch (LittleLong(*(unsigned *)buf))
		{
		case IDALIASHEADER:
			loadmodel->extradata = Hunk_Begin (0x200000);
			Mod_LoadAliasModel (mod, buf);
			break;
			
		case IDSPRITEHEADER:
			loadmodel->extradata = Hunk_Begin (0x10000);
			Mod_LoadSpriteModel (mod, buf);
			break;
		
		case IDBSPHEADER:
			loadmodel->extradata = Hunk_Begin (0x1000000);
			Mod_LoadBrushModel (mod, buf);
			break;

		default:
			ri.Sys_Error (ERR_DROP,"Mod_NumForName: unknown fileid for %s", mod->name);
			break;
		}
	}

	loadmodel->extradatasize = Hunk_End ();

	ri.FS_FreeFile (buf);

	return mod;
}

/*
===============================================================================

					BRUSHMODEL LOADING

===============================================================================
*/

byte	*mod_base;


/*
=================
Mod_LoadLighting
=================
*/
void Mod_LoadLighting (lump_t *l)
{
	if (!l->filelen)
	{
		loadmodel->lightdata = NULL;
		return;
	}
	loadmodel->lightdata = Hunk_Alloc ( l->filelen);	
	memcpy (loadmodel->lightdata, mod_base + l->fileofs, l->filelen);
}


/*
=================
Mod_LoadVisibility
=================
*/
void Mod_LoadVisibility (lump_t *l)
{
	int		i;

	if (!l->filelen)
	{
		loadmodel->vis = NULL;
		return;
	}
	loadmodel->vis = Hunk_Alloc ( l->filelen);	
	memcpy (loadmodel->vis, mod_base + l->fileofs, l->filelen);

	loadmodel->vis->numclusters = LittleLong (loadmodel->vis->numclusters);
	for (i=0 ; i<loadmodel->vis->numclusters ; i++)
	{
		loadmodel->vis->bitofs[i][0] = LittleLong (loadmodel->vis->bitofs[i][0]);
		loadmodel->vis->bitofs[i][1] = LittleLong (loadmodel->vis->bitofs[i][1]);
	}
}


/*
=================
Mod_LoadVertexes
=================
*/
void Mod_LoadVertexes (lump_t *l)
{
	dvertex_t	*in;
	mvertex_t	*out;
	int			i, count;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->vertexes = out;
	loadmodel->numvertexes = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		out->position[0] = LittleFloat (in->point[0]);
		out->position[1] = LittleFloat (in->point[1]);
		out->position[2] = LittleFloat (in->point[2]);
	}
}

/*
=================
RadiusFromBounds
=================
*/
float RadiusFromBounds (vec3_t mins, vec3_t maxs)
{
	int		i;
	vec3_t	corner;

	for (i=0 ; i<3 ; i++)
	{
		corner[i] = fabs(mins[i]) > fabs(maxs[i]) ? fabs(mins[i]) : fabs(maxs[i]);
	}

	return VectorLength (corner);
}


/*
=================
Mod_LoadSubmodels
=================
*/
void Mod_LoadSubmodels (lump_t *l)
{
	dmodel_t	*in;
	mmodel_t	*out;
	int			i, j, count;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->submodels = out;
	loadmodel->numsubmodels = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<3 ; j++)
		{	// spread the mins / maxs by a pixel
			out->mins[j] = LittleFloat (in->mins[j]) - 1;
			out->maxs[j] = LittleFloat (in->maxs[j]) + 1;
			out->origin[j] = LittleFloat (in->origin[j]);
		}
		out->radius = RadiusFromBounds (out->mins, out->maxs);
		out->headnode = LittleLong (in->headnode);
		out->firstface = LittleLong (in->firstface);
		out->numfaces = LittleLong (in->numfaces);
	}
}

/*
=================
Mod_LoadEdges
=================
*/
void Mod_LoadEdges (lump_t *l)
{
	dedge_t *in;
	medge_t *out;
	int 	i, count;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( (count + 1) * sizeof(*out));	

	loadmodel->edges = out;
	loadmodel->numedges = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		out->v[0] = (unsigned short)LittleShort(in->v[0]);
		out->v[1] = (unsigned short)LittleShort(in->v[1]);
	}
}

/*
=================
Mod_LoadTexinfo
=================
*/
void Mod_LoadTexinfo (lump_t *l)
{
	texinfo_t *in;
	mtexinfo_t *out, *step;
	int 	i, j, count;
	char	name[MAX_QPATH];
	int		next;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->texinfo = out;
	loadmodel->numtexinfo = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<8 ; j++)
			out->vecs[0][j] = LittleFloat (in->vecs[0][j]);

		out->flags = LittleLong (in->flags);
		next = LittleLong (in->nexttexinfo);
		if (next > 0)
			out->next = loadmodel->texinfo + next;
		else
		    out->next = NULL;
		Com_sprintf (name, sizeof(name), "textures/%s.wal", in->texture);

		out->image = GL_FindImage (name, it_wall);
		if (!out->image)
		{
			ri.Con_Printf (PRINT_ALL, "Couldn't load %s\n", name);
			out->image = r_notexture;
		}
	}

	// count animation frames
	for (i=0 ; i<count ; i++)
	{
		out = &loadmodel->texinfo[i];
		out->numframes = 1;
		for (step = out->next ; step && step != out ; step=step->next)
			out->numframes++;
	}
}

/*
================
CalcSurfaceExtents

Fills in s->texturemins[] and s->extents[]
================
*/
void CalcSurfaceExtents (msurface_t *s)
{
	float	mins[2], maxs[2], val;
	int		i,j, e;
	mvertex_t	*v;
	mtexinfo_t	*tex;
	int		bmins[2], bmaxs[2];

	mins[0] = mins[1] = 999999;
	maxs[0] = maxs[1] = -99999;

	tex = s->texinfo;
	
	for (i=0 ; i<s->numedges ; i++)
	{
		e = loadmodel->surfedges[s->firstedge+i];
		if (e >= 0)
			v = &loadmodel->vertexes[loadmodel->edges[e].v[0]];
		else
			v = &loadmodel->vertexes[loadmodel->edges[-e].v[1]];
		
		for (j=0 ; j<2 ; j++)
		{
			val = v->position[0] * tex->vecs[j][0] + 
				v->position[1] * tex->vecs[j][1] +
				v->position[2] * tex->vecs[j][2] +
				tex->vecs[j][3];
			if (val < mins[j])
				mins[j] = val;
			if (val > maxs[j])
				maxs[j] = val;
		}
	}

	for (i=0 ; i<2 ; i++)
	{	
		bmins[i] = floor(mins[i]/16);
		bmaxs[i] = ceil(maxs[i]/16);

		s->texturemins[i] = bmins[i] * 16;
		s->extents[i] = (bmaxs[i] - bmins[i]) * 16;

//		if ( !(tex->flags & TEX_SPECIAL) && s->extents[i] > 512 /* 256 */ )
//			ri.Sys_Error (ERR_DROP, "Bad surface extents");
	}
}


void GL_BuildPolygonFromSurface(msurface_t *fa);
void GL_CreateSurfaceLightmap (msurface_t *surf);
void GL_EndBuildingLightmaps (void);
void GL_BeginBuildingLightmaps (model_t *m);

/*
=================
Mod_LoadFaces
=================
*/
void Mod_LoadFaces (lump_t *l)
{
	dface_t		*in;
	msurface_t 	*out;
	int			i, count, surfnum;
	int			planenum, side;
	int			ti;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->surfaces = out;
	loadmodel->numsurfaces = count;

	currentmodel = loadmodel;

	GL_BeginBuildingLightmaps (loadmodel);

	for ( surfnum=0 ; surfnum<count ; surfnum++, in++, out++)
	{
		out->firstedge = LittleLong(in->firstedge);
		out->numedges = LittleShort(in->numedges);		
		out->flags = 0;
		out->polys = NULL;

		planenum = LittleShort(in->planenum);
		side = LittleShort(in->side);
		if (side)
			out->flags |= SURF_PLANEBACK;			

		out->plane = loadmodel->planes + planenum;

		ti = LittleShort (in->texinfo);
		if (ti < 0 || ti >= loadmodel->numtexinfo)
			ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: bad texinfo number");
		out->texinfo = loadmodel->texinfo + ti;

		CalcSurfaceExtents (out);
				
	// lighting info

		for (i=0 ; i<MAXLIGHTMAPS ; i++)
			out->styles[i] = in->styles[i];
		i = LittleLong(in->lightofs);
		if (i == -1)
			out->samples = NULL;
		else
			out->samples = loadmodel->lightdata + i;
		
	// set the drawing flags
		
		if (out->texinfo->flags & SURF_WARP)
		{
			out->flags |= SURF_DRAWTURB;
			for (i=0 ; i<2 ; i++)
			{
				out->extents[i] = 16384;
				out->texturemins[i] = -8192;
			}
			GL_SubdivideSurface (out);	// cut up polygon for warps
		}

		// create lightmaps and polygons
		if ( !(out->texinfo->flags & (SURF_SKY|SURF_TRANS33|SURF_TRANS66|SURF_WARP) ) )
			GL_CreateSurfaceLightmap (out);

		if (! (out->texinfo->flags & SURF_WARP) ) 
			GL_BuildPolygonFromSurface(out);

	}

	GL_EndBuildingLightmaps ();
}


/*
=================
Mod_SetParent
=================
*/
void Mod_SetParent (mnode_t *node, mnode_t *parent)
{
	node->parent = parent;
	if (node->contents != -1)
		return;
	Mod_SetParent (node->children[0], node);
	Mod_SetParent (node->children[1], node);
}

/*
=================
Mod_LoadNodes
=================
*/
void Mod_LoadNodes (lump_t *l)
{
	int			i, j, count, p;
	dnode_t		*in;
	mnode_t 	*out;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->nodes = out;
	loadmodel->numnodes = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<3 ; j++)
		{
			out->minmaxs[j] = LittleShort (in->mins[j]);
			out->minmaxs[3+j] = LittleShort (in->maxs[j]);
		}
	
		p = LittleLong(in->planenum);
		out->plane = loadmodel->planes + p;

		out->firstsurface = LittleShort (in->firstface);
		out->numsurfaces = LittleShort (in->numfaces);
		out->contents = -1;	// differentiate from leafs

		for (j=0 ; j<2 ; j++)
		{
			p = LittleLong (in->children[j]);
			if (p >= 0)
				out->children[j] = loadmodel->nodes + p;
			else
				out->children[j] = (mnode_t *)(loadmodel->leafs + (-1 - p));
		}
	}
	
	Mod_SetParent (loadmodel->nodes, NULL);	// sets nodes and leafs
}

/*
=================
Mod_LoadLeafs
=================
*/
void Mod_LoadLeafs (lump_t *l)
{
	dleaf_t 	*in;
	mleaf_t 	*out;
	int			i, j, count, p;
//	glpoly_t	*poly;

	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->leafs = out;
	loadmodel->numleafs = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		for (j=0 ; j<3 ; j++)
		{
			out->minmaxs[j] = LittleShort (in->mins[j]);
			out->minmaxs[3+j] = LittleShort (in->maxs[j]);
		}

		p = LittleLong(in->contents);
		out->contents = p;

		out->cluster = LittleShort(in->cluster);
		out->area = LittleShort(in->area);

		out->firstmarksurface = loadmodel->marksurfaces +
			LittleShort(in->firstleafface);
		out->nummarksurfaces = LittleShort(in->numleaffaces);
		
		// gl underwater warp
#if 0
		if (out->contents & (CONTENTS_WATER|CONTENTS_SLIME|CONTENTS_LAVA|CONTENTS_THINWATER) )
		{
			for (j=0 ; j<out->nummarksurfaces ; j++)
			{
				out->firstmarksurface[j]->flags |= SURF_UNDERWATER;
				for (poly = out->firstmarksurface[j]->polys ; poly ; poly=poly->next)
					poly->flags |= SURF_UNDERWATER;
			}
		}
#endif
	}	
}

/*
=================
Mod_LoadMarksurfaces
=================
*/
void Mod_LoadMarksurfaces (lump_t *l)
{	
	int		i, j, count;
	short		*in;
	msurface_t **out;
	
	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->marksurfaces = out;
	loadmodel->nummarksurfaces = count;

	for ( i=0 ; i<count ; i++)
	{
		j = LittleShort(in[i]);
		if (j < 0 ||  j >= loadmodel->numsurfaces)
			ri.Sys_Error (ERR_DROP, "Mod_ParseMarksurfaces: bad surface number");
		out[i] = loadmodel->surfaces + j;
	}
}

/*
=================
Mod_LoadSurfedges
=================
*/
void Mod_LoadSurfedges (lump_t *l)
{	
	int		i, count;
	int		*in, *out;
	
	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	if (count < 1 || count >= MAX_MAP_SURFEDGES)
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: bad surfedges count in %s: %i",
		loadmodel->name, count);

	out = Hunk_Alloc ( count*sizeof(*out));	

	loadmodel->surfedges = out;
	loadmodel->numsurfedges = count;

	for ( i=0 ; i<count ; i++)
		out[i] = LittleLong (in[i]);
}


/*
=================
Mod_LoadPlanes
=================
*/
void Mod_LoadPlanes (lump_t *l)
{
	int			i, j;
	cplane_t	*out;
	dplane_t 	*in;
	int			count;
	int			bits;
	
	in = (void *)(mod_base + l->fileofs);
	if (l->filelen % sizeof(*in))
		ri.Sys_Error (ERR_DROP, "MOD_LoadBmodel: funny lump size in %s",loadmodel->name);
	count = l->filelen / sizeof(*in);
	out = Hunk_Alloc ( count*2*sizeof(*out));	
	
	loadmodel->planes = out;
	loadmodel->numplanes = count;

	for ( i=0 ; i<count ; i++, in++, out++)
	{
		bits = 0;
		for (j=0 ; j<3 ; j++)
		{
			out->normal[j] = LittleFloat (in->normal[j]);
			if (out->normal[j] < 0)
				bits |= 1<<j;
		}

		out->dist = LittleFloat (in->dist);
		out->type = LittleLong (in->type);
		out->signbits = bits;
	}
}

/*
=================
Mod_LoadBrushModel
=================
*/
void Mod_LoadBrushModel (model_t *mod, void *buffer)
{
	int			i;
	dheader_t	*header;
	mmodel_t 	*bm;
	
	loadmodel->type = mod_brush;
	if (loadmodel != mod_known)
		ri.Sys_Error (ERR_DROP, "Loaded a brush model after the world");

	header = (dheader_t *)buffer;

	i = LittleLong (header->version);
	if (i != BSPVERSION)
		ri.Sys_Error (ERR_DROP, "Mod_LoadBrushModel: %s has wrong version number (%i should be %i)", mod->name, i, BSPVERSION);

// swap all the lumps
	mod_base = (byte *)header;

	for (i=0 ; i<sizeof(dheader_t)/4 ; i++)
		((int *)header)[i] = LittleLong ( ((int *)header)[i]);

// load into heap
	
	Mod_LoadVertexes (&header->lumps[LUMP_VERTEXES]);
	Mod_LoadEdges (&header->lumps[LUMP_EDGES]);
	Mod_LoadSurfedges (&header->lumps[LUMP_SURFEDGES]);
	Mod_LoadLighting (&header->lumps[LUMP_LIGHTING]);
	Mod_LoadPlanes (&header->lumps[LUMP_PLANES]);
	Mod_LoadTexinfo (&header->lumps[LUMP_TEXINFO]);
	Mod_LoadFaces (&header->lumps[LUMP_FACES]);
	Mod_LoadMarksurfaces (&header->lumps[LUMP_LEAFFACES]);
	Mod_LoadVisibility (&header->lumps[LUMP_VISIBILITY]);
	Mod_LoadLeafs (&header->lumps[LUMP_LEAFS]);
	Mod_LoadNodes (&header->lumps[LUMP_NODES]);
	Mod_LoadSubmodels (&header->lumps[LUMP_MODELS]);
	mod->numframes = 2;		// regular and alternate animation
	
//
// set up the submodels
//
	for (i=0 ; i<mod->numsubmodels ; i++)
	{
		model_t	*starmod;

		bm = &mod->submodels[i];
		starmod = &mod_inline[i];

		*starmod = *loadmodel;
		
		starmod->firstmodelsurface = bm->firstface;
		starmod->nummodelsurfaces = bm->numfaces;
		starmod->firstnode = bm->headnode;
		if (starmod->firstnode >= loadmodel->numnodes)
			ri.Sys_Error (ERR_DROP, "Inline model %i has bad firstnode", i);

		VectorCopy (bm->maxs, starmod->maxs);
		VectorCopy (bm->mins, starmod->mins);
		starmod->radius = bm->radius;
	
		if (i == 0)
			*loadmodel = *starmod;

		starmod->numleafs = bm->visleafs;
	}
}

/*
==============================================================================

ALIAS MODELS

==============================================================================
*/

/*
=================
Mod_LoadAliasModel
=================
*/
void Mod_LoadAliasModel (model_t *mod, void *buffer)
{
	int					i, j;
	dmdl_t				*pinmodel, *pheader;
	dstvert_t			*pinst, *poutst;
	dtriangle_t			*pintri, *pouttri;
	daliasframe_t		*pinframe, *poutframe;
	int					*pincmd, *poutcmd;
	int					version;

	pinmodel = (dmdl_t *)buffer;

	version = LittleLong (pinmodel->version);
	if (version != ALIAS_VERSION)
		ri.Sys_Error (ERR_DROP, "%s has wrong version number (%i should be %i)",
				 mod->name, version, ALIAS_VERSION);

	pheader = Hunk_Alloc (LittleLong(pinmodel->ofs_end));
	
	// byte swap the header fields and sanity check
	for (i=0 ; i<sizeof(dmdl_t)/4 ; i++)
		((int *)pheader)[i] = LittleLong (((int *)buffer)[i]);

	if (pheader->skinheight > MAX_LBM_HEIGHT)
		ri.Sys_Error (ERR_DROP, "model %s has a skin taller than %d", mod->name,
				   MAX_LBM_HEIGHT);

	if (pheader->num_xyz <= 0)
		ri.Sys_Error (ERR_DROP, "model %s has no vertices", mod->name);

	if (pheader->num_xyz > MAX_VERTS)
		ri.Sys_Error (ERR_DROP, "model %s has too many vertices", mod->name);

	if (pheader->num_st <= 0)
		ri.Sys_Error (ERR_DROP, "model %s has no st vertices", mod->name);

	if (pheader->num_tris <= 0)
		ri.Sys_Error (ERR_DROP, "model %s has no triangles", mod->name);

	if (pheader->num_frames <= 0)
		ri.Sys_Error (ERR_DROP, "model %s has no frames", mod->name);

//
// load base s and t vertices (not used in gl version)
//
	pinst = (dstvert_t *) ((byte *)pinmodel + pheader->ofs_st);
	poutst = (dstvert_t *) ((byte *)pheader + pheader->ofs_st);

	for (i=0 ; i<pheader->num_st ; i++)
	{
		poutst[i].s = LittleShort (pinst[i].s);
		poutst[i].t = LittleShort (pinst[i].t);
	}

//
// load triangle lists
//
	pintri = (dtriangle_t *) ((byte *)pinmodel + pheader->ofs_tris);
	pouttri = (dtriangle_t *) ((byte *)pheader + pheader->ofs_tris);

	for (i=0 ; i<pheader->num_tris ; i++)
	{
		for (j=0 ; j<3 ; j++)
		{
			pouttri[i].index_xyz[j] = LittleShort (pintri[i].index_xyz[j]);
			pouttri[i].index_st[j] = LittleShort (pintri[i].index_st[j]);
		}
	}

//
// load the frames
//
	for (i=0 ; i<pheader->num_frames ; i++)
	{
		pinframe = (daliasframe_t *) ((byte *)pinmodel 
			+ pheader->ofs_frames + i * pheader->framesize);
		poutframe = (daliasframe_t *) ((byte *)pheader 
			+ pheader->ofs_frames + i * pheader->framesize);

		memcpy (poutframe->name, pinframe->name, sizeof(poutframe->name));
		for (j=0 ; j<3 ; j++)
		{
			poutframe->scale[j] = LittleFloat (pinframe->scale[j]);
			poutframe->translate[j] = LittleFloat (pinframe->translate[j]);
		}
		// verts are all 8 bit, so no swapping needed
		memcpy (poutframe->verts, pinframe->verts, 
			pheader->num_xyz*sizeof(dtrivertx_t));

	}

	mod->type = mod_alias;

	//
	// load the glcmds
	//
	pincmd = (int *) ((byte *)pinmodel + pheader->ofs_glcmds);
	poutcmd = (int *) ((byte *)pheader + pheader->ofs_glcmds);
	for (i=0 ; i<pheader->num_glcmds ; i++)
		poutcmd[i] = LittleLong (pincmd[i]);


	// register all skins
	memcpy ((char *)pheader + pheader->ofs_skins, (char *)pinmodel + pheader->ofs_skins,
		pheader->num_skins*MAX_SKINNAME);
	for (i=0 ; i<pheader->num_skins ; i++)
	{
		mod->skins[i] = GL_FindImage ((char *)pheader + pheader->ofs_skins + i*MAX_SKINNAME
			, it_skin);
	}

	mod->mins[0] = -32;
	mod->mins[1] = -32;
	mod->mins[2] = -32;
	mod->maxs[0] = 32;
	mod->maxs[1] = 32;
	mod->maxs[2] = 32;
}

/*
==============================================================================

SPRITE MODELS

==============================================================================
*/

/*
=================
Mod_LoadSpriteModel
=================
*/
void Mod_LoadSpriteModel (model_t *mod, void *buffer)
{
	dsprite_t	*sprin, *sprout;
	int			i;

	sprin = (dsprite_t *)buffer;
	sprout = Hunk_Alloc (modfilelen);

	sprout->ident = LittleLong (sprin->ident);
	sprout->version = LittleLong (sprin->version);
	sprout->numframes = LittleLong (sprin->numframes);

	if (sprout->version != SPRITE_VERSION)
		ri.Sys_Error (ERR_DROP, "%s has wrong version number (%i should be %i)",
				 mod->name, sprout->version, SPRITE_VERSION);

	if (sprout->numframes > MAX_MD2SKINS)
		ri.Sys_Error (ERR_DROP, "%s has too many frames (%i > %i)",
				 mod->name, sprout->numframes, MAX_MD2SKINS);

	// byte swap everything
	for (i=0 ; i<sprout->numframes ; i++)
	{
		sprout->frames[i].width = LittleLong (sprin->frames[i].width);
		sprout->frames[i].height = LittleLong (sprin->frames[i].height);
		sprout->frames[i].origin_x = LittleLong (sprin->frames[i].origin_x);
		sprout->frames[i].origin_y = LittleLong (sprin->frames[i].origin_y);
		memcpy (sprout->frames[i].name, sprin->frames[i].name, MAX_SKINNAME);
		mod->skins[i] = GL_FindImage (sprout->frames[i].name,
			it_sprite);
	}

	mod->type = mod_sprite;
}

//=============================================================================

/*
@@@@@@@@@@@@@@@@@@@@@
R_BeginRegistration

Specifies the model that will be used as the world
@@@@@@@@@@@@@@@@@@@@@
*/
void R_BeginRegistration (char *model)
{
	char	fullname[MAX_QPATH];
	cvar_t	*flushmap;

	registration_sequence++;
	r_oldviewcluster = -1;		// force markleafs

	Com_sprintf (fullname, sizeof(fullname), "maps/%s.bsp", model);

	// explicitly free the old map if different
	// this guarantees that mod_known[0] is the world map
	flushmap = ri.Cvar_Get ("flushmap", "0", 0);
	if ( strcmp(mod_known[0].name, fullname) || flushmap->value)
		Mod_Free (&mod_known[0]);
	r_worldmodel = Mod_ForName(fullname, true);

	r_viewcluster = -1;
}


/*
@@@@@@@@@@@@@@@@@@@@@
R_RegisterModel

@@@@@@@@@@@@@@@@@@@@@
*/
struct model_s *R_RegisterModel (char *name)
{
	model_t	*mod;
	int		i;
	dsprite_t	*sprout;
	dmdl_t		*pheader;

	mod = Mod_ForName (name, false);
	if (mod)
	{
		mod->registration_sequence = registration_sequence;

		// register any images used by the models
		if (mod->type == mod_sprite)
		{
			sprout = (dsprite_t *)mod->extradata;
			for (i=0 ; i<sprout->numframes ; i++)
				mod->skins[i] = GL_FindImage (sprout->frames[i].name, it_sprite);
		}
		else if (mod->type == mod_alias)
		{
			pheader = (dmdl_t *)mod->extradata;
			for (i=0 ; i<pheader->num_skins ; i++)
				mod->skins[i] = GL_FindImage ((char *)pheader + pheader->ofs_skins + i*MAX_SKINNAME, it_skin);
//PGM
			mod->numframes = pheader->num_frames;
//PGM
		}
		else if (mod->type == mod_brush)
		{
			for (i=0 ; i<mod->numtexinfo ; i++)
				mod->texinfo[i].image->registration_sequence = registration_sequence;
		}
		//rww begin
		else if (mod->type == mod_md5)
		{
			modelMeshObject_t *obj = mod->md5;
			while (obj)
			{
				obj->meshData.skin = GL_FindImage(obj->meshData.skinName, it_skin);
				if (!obj->meshData.skin)
				{ //then just try for a placeholder
					obj->meshData.skin = GL_FindImage("players/male/cipher.pcx", it_skin);
				}
				//check for a normal map
				if (obj->meshData.skinNameNormal[0])
				{
					obj->meshData.skin_normal = GL_FindImage(obj->meshData.skinNameNormal, it_skin);
				}
				obj = obj->next;
			}

			if (gl_ext_palettedtexture->value)
			{ //just a little bitching
				ri.Con_Printf(PRINT_ALL, "YOU HAVE gl_ext_palettedtexture ENABLED!\nThis will make things look terrible.\n");
			}
		}
		//rww end
	}
	return mod;
}

//rww begin
struct model_s *R_RegisterAnim (char *name)
{
	model_t	*mod;

	assert(strstr(name, "md5anim"));

	mod = Mod_ForName (name, false);
	if (mod)
	{
		mod->registration_sequence = registration_sequence;

		//do more stuff
	}
	return mod;
}
//rww end


/*
@@@@@@@@@@@@@@@@@@@@@
R_EndRegistration

@@@@@@@@@@@@@@@@@@@@@
*/
void R_EndRegistration (void)
{
	int		i;
	model_t	*mod;

	for (i=0, mod=mod_known ; i<mod_numknown ; i++, mod++)
	{
		if (!mod->name[0])
			continue;
		if (mod->registration_sequence != registration_sequence)
		{	// don't need this model
			Mod_Free (mod);
		}
	}

	GL_FreeUnusedImages ();
}


//=============================================================================


/*
================
Mod_Free
================
*/
void Mod_Free (model_t *mod)
{
	//rww begin
	if (mod->type == mod_md5)
	{ //free up the extra md5 allocations
		if (mod->md5)
		{
			Mod_FreeUpMD5(mod->md5);
		}
	}
	else if (mod->type == mod_md5anim)
	{
		if (mod->md5anim)
		{
			Mod_FreeUpMD5Anim(mod->md5anim);
		}
	}
	//rww end

	Hunk_Free (mod->extradata);
	memset (mod, 0, sizeof(*mod));
}

/*
================
Mod_FreeAll
================
*/
void Mod_FreeAll (void)
{
	int		i;

	for (i=0 ; i<mod_numknown ; i++)
	{
		if (mod_known[i].extradatasize)
			Mod_Free (&mod_known[i]);
	}
}
