/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// gl_mesh.c: triangle model functions

#include "gl_local.h"

/*
=============================================================

  ALIAS MODELS

=============================================================
*/

#define NUMVERTEXNORMALS	162

float	r_avertexnormals[NUMVERTEXNORMALS][3] = {
#include "anorms.h"
};

typedef float vec4_t[4];

static	vec4_t	s_lerped[MAX_VERTS];
//static	vec3_t	lerped[MAX_VERTS];

vec3_t	shadevector;
float	shadelight[3];

// precalculated dot products for quantized angles
#define SHADEDOT_QUANT 16
float	r_avertexnormal_dots[SHADEDOT_QUANT][256] =
#include "anormtab.h"
;

float	*shadedots = r_avertexnormal_dots[0];

void GL_LerpVerts( int nverts, dtrivertx_t *v, dtrivertx_t *ov, dtrivertx_t *verts, float *lerp, float move[3], float frontv[3], float backv[3] )
{
	int i;

	//PMM -- added RF_SHELL_DOUBLE, RF_SHELL_HALF_DAM
	if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE | RF_SHELL_DOUBLE | RF_SHELL_HALF_DAM) )
	{
		for (i=0 ; i < nverts; i++, v++, ov++, lerp+=4 )
		{
			float *normal = r_avertexnormals[verts[i].lightnormalindex];

			lerp[0] = move[0] + ov->v[0]*backv[0] + v->v[0]*frontv[0] + normal[0] * POWERSUIT_SCALE;
			lerp[1] = move[1] + ov->v[1]*backv[1] + v->v[1]*frontv[1] + normal[1] * POWERSUIT_SCALE;
			lerp[2] = move[2] + ov->v[2]*backv[2] + v->v[2]*frontv[2] + normal[2] * POWERSUIT_SCALE; 
		}
	}
	else
	{
		for (i=0 ; i < nverts; i++, v++, ov++, lerp+=4)
		{
			lerp[0] = move[0] + ov->v[0]*backv[0] + v->v[0]*frontv[0];
			lerp[1] = move[1] + ov->v[1]*backv[1] + v->v[1]*frontv[1];
			lerp[2] = move[2] + ov->v[2]*backv[2] + v->v[2]*frontv[2];
		}
	}

}

/*
=============
GL_DrawAliasFrameLerp

interpolates between two frames and origins
FIXME: batch lerp all vertexes
=============
*/
void GL_DrawAliasFrameLerp (dmdl_t *paliashdr, float backlerp)
{
	float 	l;
	daliasframe_t	*frame, *oldframe;
	dtrivertx_t	*v, *ov, *verts;
	int		*order;
	int		count;
	float	frontlerp;
	float	alpha;
	vec3_t	move, delta, vectors[3];
	vec3_t	frontv, backv;
	int		i;
	int		index_xyz;
	float	*lerp;

	frame = (daliasframe_t *)((byte *)paliashdr + paliashdr->ofs_frames 
		+ currententity->frame * paliashdr->framesize);
	verts = v = frame->verts;

	oldframe = (daliasframe_t *)((byte *)paliashdr + paliashdr->ofs_frames 
		+ currententity->oldframe * paliashdr->framesize);
	ov = oldframe->verts;

	order = (int *)((byte *)paliashdr + paliashdr->ofs_glcmds);

//	glTranslatef (frame->translate[0], frame->translate[1], frame->translate[2]);
//	glScalef (frame->scale[0], frame->scale[1], frame->scale[2]);

	if (currententity->flags & RF_TRANSLUCENT)
		alpha = currententity->alpha;
	else
		alpha = 1.0;

	// PMM - added double shell
	if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE | RF_SHELL_DOUBLE | RF_SHELL_HALF_DAM) )
		qglDisable( GL_TEXTURE_2D );

	frontlerp = 1.0 - backlerp;

	// move should be the delta back to the previous frame * backlerp
	VectorSubtract (currententity->oldorigin, currententity->origin, delta);
	AngleVectors (currententity->angles, vectors[0], vectors[1], vectors[2]);

	move[0] = DotProduct (delta, vectors[0]);	// forward
	move[1] = -DotProduct (delta, vectors[1]);	// left
	move[2] = DotProduct (delta, vectors[2]);	// up

	VectorAdd (move, oldframe->translate, move);

	for (i=0 ; i<3 ; i++)
	{
		move[i] = backlerp*move[i] + frontlerp*frame->translate[i];
	}

	for (i=0 ; i<3 ; i++)
	{
		frontv[i] = frontlerp*frame->scale[i];
		backv[i] = backlerp*oldframe->scale[i];
	}

	lerp = s_lerped[0];

	GL_LerpVerts( paliashdr->num_xyz, v, ov, verts, lerp, move, frontv, backv );

	if ( gl_vertex_arrays->value )
	{
		float colorArray[MAX_VERTS*4];

		qglEnableClientState( GL_VERTEX_ARRAY );
		qglVertexPointer( 3, GL_FLOAT, 16, s_lerped );	// padded for SIMD

//		if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE ) )
		// PMM - added double damage shell
		if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE | RF_SHELL_DOUBLE | RF_SHELL_HALF_DAM) )
		{
			qglColor4f( shadelight[0], shadelight[1], shadelight[2], alpha );
		}
		else
		{
			qglEnableClientState( GL_COLOR_ARRAY );
			qglColorPointer( 3, GL_FLOAT, 0, colorArray );

			//
			// pre light everything
			//
			for ( i = 0; i < paliashdr->num_xyz; i++ )
			{
				float l = shadedots[verts[i].lightnormalindex];

				colorArray[i*3+0] = l * shadelight[0];
				colorArray[i*3+1] = l * shadelight[1];
				colorArray[i*3+2] = l * shadelight[2];
			}
		}

		if ( qglLockArraysEXT != 0 )
			qglLockArraysEXT( 0, paliashdr->num_xyz );

		while (1)
		{
			// get the vertex count and primitive type
			count = *order++;
			if (!count)
				break;		// done
			if (count < 0)
			{
				count = -count;
				qglBegin (GL_TRIANGLE_FAN);
			}
			else
			{
				qglBegin (GL_TRIANGLE_STRIP);
			}

			// PMM - added double damage shell
			if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE | RF_SHELL_DOUBLE | RF_SHELL_HALF_DAM) )
			{
				do
				{
					index_xyz = order[2];
					order += 3;

					qglVertex3fv( s_lerped[index_xyz] );

				} while (--count);
			}
			else
			{
				do
				{
					// texture coordinates come from the draw list
					qglTexCoord2f (((float *)order)[0], ((float *)order)[1]);
					index_xyz = order[2];

					order += 3;

					// normals and vertexes come from the frame list
//					l = shadedots[verts[index_xyz].lightnormalindex];
					
//					qglColor4f (l* shadelight[0], l*shadelight[1], l*shadelight[2], alpha);
					qglArrayElement( index_xyz );

				} while (--count);
			}
			qglEnd ();
		}

		if ( qglUnlockArraysEXT != 0 )
			qglUnlockArraysEXT();
	}
	else
	{
		while (1)
		{
			// get the vertex count and primitive type
			count = *order++;
			if (!count)
				break;		// done
			if (count < 0)
			{
				count = -count;
				qglBegin (GL_TRIANGLE_FAN);
			}
			else
			{
				qglBegin (GL_TRIANGLE_STRIP);
			}

			if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE ) )
			{
				do
				{
					index_xyz = order[2];
					order += 3;

					qglColor4f( shadelight[0], shadelight[1], shadelight[2], alpha);
					qglVertex3fv (s_lerped[index_xyz]);

				} while (--count);
			}
			else
			{
				do
				{
					// texture coordinates come from the draw list
					qglTexCoord2f (((float *)order)[0], ((float *)order)[1]);
					index_xyz = order[2];
					order += 3;

					// normals and vertexes come from the frame list
					l = shadedots[verts[index_xyz].lightnormalindex];
					
					qglColor4f (l* shadelight[0], l*shadelight[1], l*shadelight[2], alpha);
					qglVertex3fv (s_lerped[index_xyz]);
				} while (--count);
			}

			qglEnd ();
		}
	}

//	if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE ) )
	// PMM - added double damage shell
	if ( currententity->flags & ( RF_SHELL_RED | RF_SHELL_GREEN | RF_SHELL_BLUE | RF_SHELL_DOUBLE | RF_SHELL_HALF_DAM) )
		qglEnable( GL_TEXTURE_2D );
}


#if 1
/*
=============
GL_DrawAliasShadow
=============
*/
extern	vec3_t			lightspot;

void GL_DrawAliasShadow (dmdl_t *paliashdr, int posenum)
{
	dtrivertx_t	*verts;
	int		*order;
	vec3_t	point;
	float	height, lheight;
	int		count;
	daliasframe_t	*frame;

	lheight = currententity->origin[2] - lightspot[2];

	frame = (daliasframe_t *)((byte *)paliashdr + paliashdr->ofs_frames 
		+ currententity->frame * paliashdr->framesize);
	verts = frame->verts;

	height = 0;

	order = (int *)((byte *)paliashdr + paliashdr->ofs_glcmds);

	height = -lheight + 1.0;

	while (1)
	{
		// get the vertex count and primitive type
		count = *order++;
		if (!count)
			break;		// done
		if (count < 0)
		{
			count = -count;
			qglBegin (GL_TRIANGLE_FAN);
		}
		else
			qglBegin (GL_TRIANGLE_STRIP);

		do
		{
			// normals and vertexes come from the frame list
/*
			point[0] = verts[order[2]].v[0] * frame->scale[0] + frame->translate[0];
			point[1] = verts[order[2]].v[1] * frame->scale[1] + frame->translate[1];
			point[2] = verts[order[2]].v[2] * frame->scale[2] + frame->translate[2];
*/

			memcpy( point, s_lerped[order[2]], sizeof( point )  );

			point[0] -= shadevector[0]*(point[2]+lheight);
			point[1] -= shadevector[1]*(point[2]+lheight);
			point[2] = height;
//			height -= 0.001;
			qglVertex3fv (point);

			order += 3;

//			verts++;

		} while (--count);

		qglEnd ();
	}	
}

#endif

/*
** R_CullAliasModel
*/
static qboolean R_CullAliasModel( vec3_t bbox[8], entity_t *e )
{
	int i;
	vec3_t		mins, maxs;
	dmdl_t		*paliashdr;
	vec3_t		vectors[3];
	vec3_t		thismins, oldmins, thismaxs, oldmaxs;
	daliasframe_t *pframe, *poldframe;
	vec3_t angles;

	paliashdr = (dmdl_t *)currentmodel->extradata;

	if ( ( e->frame >= paliashdr->num_frames ) || ( e->frame < 0 ) )
	{
		ri.Con_Printf (PRINT_ALL, "R_CullAliasModel %s: no such frame %d\n", 
			currentmodel->name, e->frame);
		e->frame = 0;
	}
	if ( ( e->oldframe >= paliashdr->num_frames ) || ( e->oldframe < 0 ) )
	{
		ri.Con_Printf (PRINT_ALL, "R_CullAliasModel %s: no such oldframe %d\n", 
			currentmodel->name, e->oldframe);
		e->oldframe = 0;
	}

	pframe = ( daliasframe_t * ) ( ( byte * ) paliashdr + 
		                              paliashdr->ofs_frames +
									  e->frame * paliashdr->framesize);

	poldframe = ( daliasframe_t * ) ( ( byte * ) paliashdr + 
		                              paliashdr->ofs_frames +
									  e->oldframe * paliashdr->framesize);

	/*
	** compute axially aligned mins and maxs
	*/
	if ( pframe == poldframe )
	{
		for ( i = 0; i < 3; i++ )
		{
			mins[i] = pframe->translate[i];
			maxs[i] = mins[i] + pframe->scale[i]*255;
		}
	}
	else
	{
		for ( i = 0; i < 3; i++ )
		{
			thismins[i] = pframe->translate[i];
			thismaxs[i] = thismins[i] + pframe->scale[i]*255;

			oldmins[i]  = poldframe->translate[i];
			oldmaxs[i]  = oldmins[i] + poldframe->scale[i]*255;

			if ( thismins[i] < oldmins[i] )
				mins[i] = thismins[i];
			else
				mins[i] = oldmins[i];

			if ( thismaxs[i] > oldmaxs[i] )
				maxs[i] = thismaxs[i];
			else
				maxs[i] = oldmaxs[i];
		}
	}

	/*
	** compute a full bounding box
	*/
	for ( i = 0; i < 8; i++ )
	{
		vec3_t   tmp;

		if ( i & 1 )
			tmp[0] = mins[0];
		else
			tmp[0] = maxs[0];

		if ( i & 2 )
			tmp[1] = mins[1];
		else
			tmp[1] = maxs[1];

		if ( i & 4 )
			tmp[2] = mins[2];
		else
			tmp[2] = maxs[2];

		VectorCopy( tmp, bbox[i] );
	}

	/*
	** rotate the bounding box
	*/
	VectorCopy( e->angles, angles );
	angles[YAW] = -angles[YAW];
	AngleVectors( angles, vectors[0], vectors[1], vectors[2] );

	for ( i = 0; i < 8; i++ )
	{
		vec3_t tmp;

		VectorCopy( bbox[i], tmp );

		bbox[i][0] = DotProduct( vectors[0], tmp );
		bbox[i][1] = -DotProduct( vectors[1], tmp );
		bbox[i][2] = DotProduct( vectors[2], tmp );

		VectorAdd( e->origin, bbox[i], bbox[i] );
	}

	{
		int p, f, aggregatemask = ~0;

		for ( p = 0; p < 8; p++ )
		{
			int mask = 0;

			for ( f = 0; f < 4; f++ )
			{
				float dp = DotProduct( frustum[f].normal, bbox[p] );

				if ( ( dp - frustum[f].dist ) < 0 )
				{
					mask |= ( 1 << f );
				}
			}

			aggregatemask &= mask;
		}

		if ( aggregatemask )
		{
			return true;
		}

		return false;
	}
}

//rww begin
//wow, this is some ugly code
void R_CalcShadeLight(entity_t *e)
{
	int			i;
	float		an;

	//
	// get lighting information
	//
	// PMM - rewrote, reordered to handle new shells & mixing
	// PMM - 3.20 code .. replaced with original way of doing it to keep mod authors happy
	//
	if ( e->flags & ( RF_SHELL_HALF_DAM | RF_SHELL_GREEN | RF_SHELL_RED | RF_SHELL_BLUE | RF_SHELL_DOUBLE ) )
	{
		VectorClear (shadelight);
		if (e->flags & RF_SHELL_HALF_DAM)
		{
				shadelight[0] = 0.56;
				shadelight[1] = 0.59;
				shadelight[2] = 0.45;
		}
		if ( e->flags & RF_SHELL_DOUBLE )
		{
			shadelight[0] = 0.9;
			shadelight[1] = 0.7;
		}
		if ( e->flags & RF_SHELL_RED )
			shadelight[0] = 1.0;
		if ( e->flags & RF_SHELL_GREEN )
			shadelight[1] = 1.0;
		if ( e->flags & RF_SHELL_BLUE )
			shadelight[2] = 1.0;
	}
/*
		// PMM -special case for godmode
		if ( (e->flags & RF_SHELL_RED) &&
			(e->flags & RF_SHELL_BLUE) &&
			(e->flags & RF_SHELL_GREEN) )
		{
			for (i=0 ; i<3 ; i++)
				shadelight[i] = 1.0;
		}
		else if ( e->flags & ( RF_SHELL_RED | RF_SHELL_BLUE | RF_SHELL_DOUBLE ) )
		{
			VectorClear (shadelight);

			if ( e->flags & RF_SHELL_RED )
			{
				shadelight[0] = 1.0;
				if (e->flags & (RF_SHELL_BLUE|RF_SHELL_DOUBLE) )
					shadelight[2] = 1.0;
			}
			else if ( e->flags & RF_SHELL_BLUE )
			{
				if ( e->flags & RF_SHELL_DOUBLE )
				{
					shadelight[1] = 1.0;
					shadelight[2] = 1.0;
				}
				else
				{
					shadelight[2] = 1.0;
				}
			}
			else if ( e->flags & RF_SHELL_DOUBLE )
			{
				shadelight[0] = 0.9;
				shadelight[1] = 0.7;
			}
		}
		else if ( e->flags & ( RF_SHELL_HALF_DAM | RF_SHELL_GREEN ) )
		{
			VectorClear (shadelight);
			// PMM - new colors
			if ( e->flags & RF_SHELL_HALF_DAM )
			{
				shadelight[0] = 0.56;
				shadelight[1] = 0.59;
				shadelight[2] = 0.45;
			}
			if ( e->flags & RF_SHELL_GREEN )
			{
				shadelight[1] = 1.0;
			}
		}
	}
			//PMM - ok, now flatten these down to range from 0 to 1.0.
	//		max_shell_val = max(shadelight[0], max(shadelight[1], shadelight[2]));
	//		if (max_shell_val > 0)
	//		{
	//			for (i=0; i<3; i++)
	//			{
	//				shadelight[i] = shadelight[i] / max_shell_val;
	//			}
	//		}
	// pmm
*/
	else if ( e->flags & RF_FULLBRIGHT )
	{
		for (i=0 ; i<3 ; i++)
			shadelight[i] = 1.0;
	}
	else
	{
		R_LightPoint (e->origin, shadelight);

		// player lighting hack for communication back to server
		// big hack!
		if ( e->flags & RF_WEAPONMODEL )
		{
			// pick the greatest component, which should be the same
			// as the mono value returned by software
			if (shadelight[0] > shadelight[1])
			{
				if (shadelight[0] > shadelight[2])
					r_lightlevel->value = 150*shadelight[0];
				else
					r_lightlevel->value = 150*shadelight[2];
			}
			else
			{
				if (shadelight[1] > shadelight[2])
					r_lightlevel->value = 150*shadelight[1];
				else
					r_lightlevel->value = 150*shadelight[2];
			}

		}
		
		if ( gl_monolightmap->string[0] != '0' )
		{
			float s = shadelight[0];

			if ( s < shadelight[1] )
				s = shadelight[1];
			if ( s < shadelight[2] )
				s = shadelight[2];

			shadelight[0] = s;
			shadelight[1] = s;
			shadelight[2] = s;
		}
	}

	if ( e->flags & RF_MINLIGHT )
	{
		for (i=0 ; i<3 ; i++)
			if (shadelight[i] > 0.1)
				break;
		if (i == 3)
		{
			shadelight[0] = 0.1;
			shadelight[1] = 0.1;
			shadelight[2] = 0.1;
		}
	}

	if ( e->flags & RF_GLOW )
	{	// bonus items will pulse with time
		float	scale;
		float	min;

		scale = 0.1 * sin(r_newrefdef.time*7);
		for (i=0 ; i<3 ; i++)
		{
			min = shadelight[i] * 0.8;
			shadelight[i] += scale;
			if (shadelight[i] < min)
				shadelight[i] = min;
		}
	}

// =================
// PGM	ir goggles color override
	if ( r_newrefdef.rdflags & RDF_IRGOGGLES && e->flags & RF_IR_VISIBLE)
	{
		shadelight[0] = 1.0;
		shadelight[1] = 0.0;
		shadelight[2] = 0.0;
	}
// PGM	
// =================

	shadedots = r_avertexnormal_dots[((int)(e->angles[1] * (SHADEDOT_QUANT / 360.0))) & (SHADEDOT_QUANT - 1)];
	
	an = e->angles[1]/180*M_PI;
	shadevector[0] = cos(-an);
	shadevector[1] = sin(-an);
	shadevector[2] = 1;
	VectorNormalize (shadevector);
}

/*
=================
R_DrawAliasModel

=================
*/
void R_DrawAliasModel (entity_t *e)
{
	dmdl_t		*paliashdr;
	vec3_t		bbox[8];
	image_t		*skin;

	if ( !( e->flags & RF_WEAPONMODEL ) )
	{
		if ( R_CullAliasModel( bbox, e ) )
			return;
	}

	if ( e->flags & RF_WEAPONMODEL )
	{
		if ( r_lefthand->value == 2 )
			return;
	}

	paliashdr = (dmdl_t *)currentmodel->extradata;

	//rww begin
	//(moved that mess into its own function)
	R_CalcShadeLight(e);
	//rww end

	//
	// locate the proper data
	//

	c_alias_polys += paliashdr->num_tris;

	//
	// draw all the triangles
	//
	if (currententity->flags & RF_DEPTHHACK) // hack the depth range to prevent view model from poking into walls
		qglDepthRange (gldepthmin, gldepthmin + 0.3*(gldepthmax-gldepthmin));

	if ( ( currententity->flags & RF_WEAPONMODEL ) && ( r_lefthand->value == 1.0F ) )
	{
		extern void MYgluPerspective( GLdouble fovy, GLdouble aspect, GLdouble zNear, GLdouble zFar );

		qglMatrixMode( GL_PROJECTION );
		qglPushMatrix();
		qglLoadIdentity();
		qglScalef( -1, 1, 1 );
	    MYgluPerspective( r_newrefdef.fov_y, ( float ) r_newrefdef.width / r_newrefdef.height,  4,  4096);
		qglMatrixMode( GL_MODELVIEW );

		qglCullFace( GL_BACK );
	}

    qglPushMatrix ();
	e->angles[PITCH] = -e->angles[PITCH];	// sigh.
	R_RotateForEntity (e);
	e->angles[PITCH] = -e->angles[PITCH];	// sigh.

	// select skin
	if (currententity->skin)
		skin = currententity->skin;	// custom player skin
	else
	{
		if (currententity->skinnum >= MAX_MD2SKINS)
			skin = currentmodel->skins[0];
		else
		{
			skin = currentmodel->skins[currententity->skinnum];
			if (!skin)
				skin = currentmodel->skins[0];
		}
	}
	if (!skin)
		skin = r_notexture;	// fallback...
	GL_Bind(skin->texnum);

	// draw it

	qglShadeModel (GL_SMOOTH);

	GL_TexEnv( GL_MODULATE );
	if ( currententity->flags & RF_TRANSLUCENT )
	{
		qglEnable (GL_BLEND);
	}


	if ( (currententity->frame >= paliashdr->num_frames) 
		|| (currententity->frame < 0) )
	{
		ri.Con_Printf (PRINT_ALL, "R_DrawAliasModel %s: no such frame %d\n",
			currentmodel->name, currententity->frame);
		currententity->frame = 0;
		currententity->oldframe = 0;
	}

	if ( (currententity->oldframe >= paliashdr->num_frames)
		|| (currententity->oldframe < 0))
	{
		ri.Con_Printf (PRINT_ALL, "R_DrawAliasModel %s: no such oldframe %d\n",
			currentmodel->name, currententity->oldframe);
		currententity->frame = 0;
		currententity->oldframe = 0;
	}

	if ( !r_lerpmodels->value )
		currententity->backlerp = 0;
	GL_DrawAliasFrameLerp (paliashdr, currententity->backlerp);

	GL_TexEnv( GL_REPLACE );
	qglShadeModel (GL_FLAT);

	qglPopMatrix ();

#if 0
	qglDisable( GL_CULL_FACE );
	qglPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
	qglDisable( GL_TEXTURE_2D );
	qglBegin( GL_TRIANGLE_STRIP );
	for ( i = 0; i < 8; i++ )
	{
		qglVertex3fv( bbox[i] );
	}
	qglEnd();
	qglEnable( GL_TEXTURE_2D );
	qglPolygonMode( GL_FRONT_AND_BACK, GL_FILL );
	qglEnable( GL_CULL_FACE );
#endif

	if ( ( currententity->flags & RF_WEAPONMODEL ) && ( r_lefthand->value == 1.0F ) )
	{
		qglMatrixMode( GL_PROJECTION );
		qglPopMatrix();
		qglMatrixMode( GL_MODELVIEW );
		qglCullFace( GL_FRONT );
	}

	if ( currententity->flags & RF_TRANSLUCENT )
	{
		qglDisable (GL_BLEND);
	}

	if (currententity->flags & RF_DEPTHHACK)
		qglDepthRange (gldepthmin, gldepthmax);

#if 1
	if (gl_shadows->value && !(currententity->flags & (RF_TRANSLUCENT | RF_WEAPONMODEL)))
	{
		qglPushMatrix ();
		R_RotateForEntity (e);
		qglDisable (GL_TEXTURE_2D);
		qglEnable (GL_BLEND);
		qglColor4f (0,0,0,0.5);
		GL_DrawAliasShadow (paliashdr, currententity->frame );
		qglEnable (GL_TEXTURE_2D);
		qglDisable (GL_BLEND);
		qglPopMatrix ();
	}
#endif
	qglColor4f (1,1,1,1);
}

//rww begin
#define GL_COMBINE_EXT						0x8570
#define GL_COMBINE_RGB_EXT					0x8571
#define GL_COMBINE_ALPHA_EXT				0x8572
#define GL_SOURCE0_RGB_EXT					0x8580
#define GL_SOURCE1_RGB_EXT					0x8581
#define GL_SOURCE2_RGB_EXT					0x8582
#define GL_SOURCE0_ALPHA_EXT				0x8588
#define GL_SOURCE1_ALPHA_EXT				0x8589
#define GL_SOURCE2_ALPHA_EXT				0x858A
#define GL_OPERAND0_RGB_EXT					0x8590
#define GL_OPERAND1_RGB_EXT					0x8591
#define GL_OPERAND2_RGB_EXT					0x8592
#define GL_OPERAND0_ALPHA_EXT				0x8598
#define GL_OPERAND1_ALPHA_EXT				0x8599
#define GL_OPERAND2_ALPHA_EXT				0x859A
#define GL_RGB_SCALE_EXT					0x8573
#define GL_ADD_SIGNED_EXT					0x8574
#define GL_INTERPOLATE_EXT					0x8575
#define GL_CONSTANT_EXT						0x8576
#define GL_PRIMARY_COLOR_EXT				0x8577
#define GL_PREVIOUS_EXT						0x8578

#define GL_DOT3_RGB_EXT						0x8740
#define GL_DOT3_RGBA_EXT					0x8741

//calc the light vectors into the color pointer
void GL_CalculateMD5LightVectors(modelMeshObject_t *obj)
{
	int i = 0;
	float *lightDir = obj->meshData.lightVectors;
	modelVert_t *clr = (modelVert_t *)obj->meshData.colorPointerVec;
	while (i < obj->meshData.numVerts)
	{
		//scale and bias into a color vector
		clr->x = lightDir[0] * 0.5f + 0.5f;
		clr->y = lightDir[1] * 0.5f + 0.5f;
		clr->z = lightDir[2] * 0.5f + 0.5f;

		lightDir += 3;
		clr++;
		i++;
	}
}

//calculate vert lighting
void GL_CalculateMD5ColorPointer(modelMeshObject_t *obj, float arbitraryMultiplier)
{
	int v = 0;
	modelRGBA_t *clr = obj->meshData.colorPointer;
	BYTE *vNorm = obj->meshData.vertNormalIndexes;

	while (v < obj->meshData.numVerts)
	{
		float l = shadedots[*vNorm];

		clr->rgba[0] = (l*shadelight[0])*arbitraryMultiplier;
		clr->rgba[1] = (l*shadelight[1])*arbitraryMultiplier;
		clr->rgba[2] = (l*shadelight[2])*arbitraryMultiplier;
		clr->rgba[3] = 1.0f;

		v++;
		clr++;
		vNorm++;
	}
}

extern void Math_TransformPointByMatrix(modelMatrix_t *matrix, float *in, float *out);
extern void Math_MatrixInverse(modelMatrix_t *in, modelMatrix_t *out);
void GL_GetWorldToLocalMatrix(entity_t *e, modelMatrix_t *invModelMat)
{
	modelMatrix_t modelMat;
	float modelMatrix[16];

	//push current modelview matrix, then translate/rotate for the object, and grab the matrix to the float array
	qglPushMatrix();
	qglLoadIdentity();

	e->angles[PITCH] = -e->angles[PITCH];	// sigh.
	R_RotateForEntity (e);
	e->angles[PITCH] = -e->angles[PITCH];	// sigh.
	
	qglGetFloatv (GL_MODELVIEW_MATRIX, modelMatrix);

	//pop back the modelview matrix since i'm done
	qglPopMatrix();

	//now i translate the gl 4x4 matrix to one of my matrices
	modelMat.x1[0] = modelMatrix[0];
	modelMat.x1[1] = modelMatrix[4];
	modelMat.x1[2] = modelMatrix[8];

	modelMat.x2[0] = modelMatrix[1];
	modelMat.x2[1] = modelMatrix[5];
	modelMat.x2[2] = modelMatrix[9];

	modelMat.x3[0] = modelMatrix[2];
	modelMat.x3[1] = modelMatrix[6];
	modelMat.x3[2] = modelMatrix[10];

	modelMat.o[0] = modelMatrix[12];
	modelMat.o[1] = modelMatrix[13];
	modelMat.o[2] = modelMatrix[14];

	//invert the matrix to transform the world coordinates with it
	Math_MatrixInverse(&modelMat, invModelMat);
}


extern void Mod_CreateTransformedVerts(entity_t *e, modelMeshObject_t *obj, modelMeshObject_t *root, model_t *anm);
extern void Mod_TangentUpdates(entity_t *e, modelMeshObject_t *obj, float *lightPos);
extern int g_canSupportDOT3;
void R_DrawMD5Model(entity_t *e)
{
	float wLPos[3], lPos[3];
	float upDir[3];
	modelMatrix_t invModelMat;
	modelMeshObject_t *root;
	modelMeshObject_t *obj;
	model_t *m = e->model;
	float lightDir[3];
	float an;
	if (!m ||
		m->type != mod_md5 ||
		!m->md5)
	{
		ri.Con_Printf(PRINT_ALL, "R_DrawMD5Model call with invalid model type.\n");
		return;
	}

	R_CalcShadeLight(e);

	//orient as per other quake objects.
    qglPushMatrix();
	e->angles[PITCH] = -e->angles[PITCH];	// sigh.
	R_RotateForEntity (e);
	e->angles[PITCH] = -e->angles[PITCH];	// sigh.

	root = m->md5;
	
	qglShadeModel (GL_SMOOTH);
	GL_TexEnv( GL_MODULATE );
	if ( e->flags & RF_TRANSLUCENT )
	{
		qglEnable (GL_BLEND);
	}

	//enable alpha test for textures that have masks on them
	qglEnable(GL_ALPHA_TEST);

#if 1
	AngleVectors(e->angles, 0, 0, upDir);
	an = e->angles[1]/180*M_PI;
	lightDir[0] = cos(-an);
	lightDir[1] = sin(-an);
	lightDir[2] = 1;
	VectorNormalize (lightDir);
	lightDir[0] += upDir[0]*0.2f;
	lightDir[1] += upDir[1]*0.2f;
	lightDir[2] += upDir[2]*0.2f;
	VectorNormalize(lightDir);

	VectorScale(lightDir, 0.5f, lightDir);
	//get dir from dlights
	{
		dlight_t	*dl;
		int			lnum;
		float		add;
		vec3_t		lightVec;
		for (lnum=0 ; lnum<r_newrefdef.num_dlights ; lnum++)
		{
			dl = &r_newrefdef.dlights[lnum];
			VectorSubtract (dl->origin,
							e->origin,
							lightVec);
			add = VectorLength(lightVec)/dl->intensity;
			if (add <= 1.0f)
			{
				VectorNormalize(lightVec);
				add = (1.0f-add)*8.0f;
				VectorScale(lightVec, add, lightVec);
				VectorAdd(lightDir, lightVec, lightDir);
			}
		}
		VectorNormalize(lightDir);
	}

	VectorMA(e->origin, 1000.0f, lightDir, wLPos);

	GL_GetWorldToLocalMatrix(e, &invModelMat);
	Math_TransformPointByMatrix(&invModelMat, wLPos, lPos);
#else //debug test, arbitrary constant light pos
	lPos[0] = 1500.0f;//sin(r_newrefdef.time)*2000.0f;
	lPos[1] = 0.0f;//cos(r_newrefdef.time)*2000.0f;
	lPos[2] = 500.0f;
#endif

	//debug to draw a line to the light position
	/*
	qglDisable(GL_TEXTURE_2D);
	qglColor3f(1.0f, 0.0f, 0.0f);
	qglLineWidth(8);
	qglBegin(GL_LINES);
		qglVertex3f(0.0f, 0.0f, 0.0f);
		qglVertex3f(lPos[0], lPos[1], lPos[2]);
	qglEnd();
	qglEnable(GL_TEXTURE_2D);
	*/

	//run through and render all objects
	obj = root;
	while (obj)
	{
		//this lets us know we need to transform again.
		obj->meshData.renderCountCurrent++;

		//make sure the verts are transformed for the current state.
		Mod_CreateTransformedVerts(e, obj, root, e->anim);
		//now update tangent vectors
		Mod_TangentUpdates(e, obj, lPos);

		//set the tex coord array
		qglEnableClientState(GL_TEXTURE_COORD_ARRAY);
		qglTexCoordPointer(2, GL_FLOAT, 2 * sizeof(GL_FLOAT), obj->meshData.uvCoord);

		//set the vertex array for our transformed verts.
		qglEnableClientState(GL_VERTEX_ARRAY);
		qglVertexPointer(3, GL_FLOAT, 3 * sizeof(GL_FLOAT), obj->meshData.vertDataTransformed);

		if (g_canSupportDOT3 && obj->meshData.skin && obj->meshData.skin_normal)
		{ //dot3 env bumpmapping
			//i would like to do this in less than 3 passes. but given the resources here,
			//i'm not sure how else to get a proper ambience with colored light on the dot3 pass.

			//first we draw a shell with normal vert lighting to give the dot3 pass something to blend with
			GL_CalculateMD5ColorPointer(obj, 1.0f);

			//set the color pointer
			qglEnableClientState(GL_COLOR_ARRAY);
			qglColorPointer(4, GL_FLOAT, 4 * sizeof(GL_FLOAT), obj->meshData.colorPointer);

			if (!e->skin)
			{ //no override, so use the obj skin
				if (obj->meshData.skin)
				{
					GL_Bind(obj->meshData.skin->texnum);
				}
			}
			else
			{
				GL_Bind(e->skin->texnum);
			}

			//now, draw.
			qglDrawElements(GL_TRIANGLES, obj->meshData.numFaces*3, GL_UNSIGNED_INT, obj->meshData.faceDataFormatted);

			
			//now blend and draw env bump map (dot3) pass
			qglEnable(GL_BLEND);
			qglBlendFunc (GL_DST_COLOR, GL_SRC_COLOR);

			GL_EnableMultitexture(true);

			GL_SelectTexture(GL_TEXTURE0);
			GL_Bind(obj->meshData.skin_normal->texnum);
			qglTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_EXT);
			qglTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_EXT, GL_DOT3_RGB_EXT);

			qglTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_EXT, GL_TEXTURE);
			qglTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_EXT, GL_SRC_COLOR);
			qglTexEnvf(GL_TEXTURE_ENV, GL_SOURCE1_RGB_EXT, GL_PRIMARY_COLOR_EXT);
			qglTexEnvf(GL_TEXTURE_ENV, GL_OPERAND1_RGB_EXT, GL_SRC_COLOR);


			GL_SelectTexture(GL_TEXTURE1);
			qglEnableClientState(GL_TEXTURE_COORD_ARRAY);
			qglTexCoordPointer(2, GL_FLOAT, 2 * sizeof(GL_FLOAT), obj->meshData.uvCoord);
			GL_Bind(obj->meshData.skin->texnum);
			qglTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_EXT);
			qglTexEnvf(GL_TEXTURE_ENV, GL_COMBINE_RGB_EXT, GL_MODULATE);

			qglTexEnvf(GL_TEXTURE_ENV, GL_SOURCE0_RGB_EXT, GL_PREVIOUS_EXT);
			qglTexEnvf(GL_TEXTURE_ENV, GL_OPERAND0_RGB_EXT, GL_SRC_COLOR);
			qglTexEnvf(GL_TEXTURE_ENV, GL_SOURCE1_RGB_EXT, GL_TEXTURE);
			qglTexEnvf(GL_TEXTURE_ENV, GL_OPERAND1_RGB_EXT, GL_SRC_COLOR);

			GL_SelectTexture(GL_TEXTURE0);

			GL_CalculateMD5LightVectors(obj);
			//set the color pointer
			qglEnableClientState(GL_COLOR_ARRAY);
			qglColorPointer(3, GL_FLOAT, 3 * sizeof(GL_FLOAT), obj->meshData.colorPointerVec);

			//now, draw.
			qglDrawElements(GL_TRIANGLES, obj->meshData.numFaces*3, GL_UNSIGNED_INT, obj->meshData.faceDataFormatted);

			{ //draw again, setting an ambient tone, to avoid "pitch black" areas
				const float ambientAdjustment = 0.3f;
				int l = 0;
				modelRGBA_t *clr = obj->meshData.colorPointer;

				//disable mt and set to modulate
				GL_EnableMultitexture(false);
				GL_TexEnv( GL_MODULATE );

				qglBlendFunc(GL_ONE, GL_ONE);

				//set the color pointer
				//cheesy hack, go through and multiply down based on ambience
				while (l < obj->meshData.numVerts)
				{
					clr->rgba[0] *= ambientAdjustment;
					clr->rgba[1] *= ambientAdjustment;
					clr->rgba[2] *= ambientAdjustment;

					clr++;
					l++;
				}
				qglColorPointer(4, GL_FLOAT, 4 * sizeof(GL_FLOAT), obj->meshData.colorPointer);

				if (!e->skin)
				{ //no override, so use the obj skin
					if (obj->meshData.skin)
					{
						GL_Bind(obj->meshData.skin->texnum);
					}
				}
				else
				{
					GL_Bind(e->skin->texnum);
				}

				//now, draw.
				qglDrawElements(GL_TRIANGLES, obj->meshData.numFaces*3, GL_UNSIGNED_INT, obj->meshData.faceDataFormatted);
			}

			//disable blend
			qglDisable (GL_BLEND);
			qglBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		}
		else
		{ //this is for normal vert lighting if no normal map is available or there is no dot3 env mapping.
			GL_CalculateMD5ColorPointer(obj, 1.0f);

			//set the color pointer
			qglEnableClientState(GL_COLOR_ARRAY);
			qglColorPointer(4, GL_FLOAT, 4 * sizeof(GL_FLOAT), obj->meshData.colorPointer);

			if (!e->skin)
			{ //no override, so use the obj skin
				if (obj->meshData.skin)
				{
					GL_Bind(obj->meshData.skin->texnum);
				}
			}
			else
			{
				GL_Bind(e->skin->texnum);
			}

			//now, draw.
			qglDrawElements(GL_TRIANGLES, obj->meshData.numFaces*3, GL_UNSIGNED_INT, obj->meshData.faceDataFormatted);
		}

		//disable arrays
		qglDisableClientState(GL_VERTEX_ARRAY);
		qglDisableClientState(GL_TEXTURE_COORD_ARRAY);
		qglDisableClientState(GL_COLOR_ARRAY);

		if (g_canSupportDOT3 && obj->meshData.skin && obj->meshData.skin_normal)
		{ //reset env mapping and texture units
			GL_SelectTexture(GL_TEXTURE1);
			qglDisableClientState(GL_TEXTURE_COORD_ARRAY);
			GL_TexEnv( GL_MODULATE );
			GL_TexEnv( GL_REPLACE );
			GL_SelectTexture(GL_TEXTURE0);
			GL_TexEnv( GL_REPLACE );
			GL_TexEnv( GL_MODULATE );

			GL_EnableMultitexture(false);

			GL_TexEnv( GL_MODULATE );
		}

		obj = obj->next;
	}

	qglShadeModel (GL_FLAT);
	GL_TexEnv( GL_REPLACE );
	if ( e->flags & RF_TRANSLUCENT )
	{
		qglDisable (GL_BLEND);
	}

	//back to the view matrix
	qglPopMatrix();
}
//rww end


