/*
=============================================================================
Module Information
------------------
Name:			main.cpp
Author:			Rich Whitehouse
Description:	core server logic module implementation.
=============================================================================
*/

#include "main.h"
#include "ai.h"
#include "rscript.h"

float g_timeFactor = 0.0f;
float g_timeScale = 1.0f;
float g_invTimeScale = 1.0f;
int g_timeType = TIMETYPE_NORMAL;
int g_lastTimeType = TIMETYPE_NORMAL;
serverTime_t g_actionTime = 0;
float g_actionTimeScale = 0.125f;
serverTime_t g_tonberryTime = 0;
int g_enSpawnBlock = 0;
bool g_noDeath = false;
int g_cinema = 0;
sharedSVFunctions_t *g_sharedFn;
globalNetData_t g_globalNet;

char g_mapName[512];
bool g_musical = false;
int g_musicStrIndex = -1;
float g_musicalAmp = 0.0f;
float g_musicalRunningAvg = 0.0f;
float g_musicalLastFrame = 0.0f;
serverTime_t g_musicalSpawnTime = 0;
int g_musicalRunningCount = 1;
serverTime_t g_musicReset = 0;
int g_musicHighScore = 0;
int g_musicHighScorePrev = 0;
char g_musicResetName[4096];
char g_musicResetNameNext[4096];

char g_levelMusic[256] = {'$', '$', '\0'};
serverTime_t g_battleTime = 0;
static bool g_battleMusic = false;
static bool g_chocoboMusic = false;
bool g_battleMusicEnabled = true;

int g_magicMode = 0;
endlBattleInfo_t g_endlessBattle = {0};
int g_mpVersusMode = 0;
int g_mpCountDown = 0;
serverTime_t g_mpCountTime = 0;

serverTime_t g_curTime = 0;
bool g_runningMultiplayer = false;

int g_soundFootsteps[NUM_SOUNDS_FOOTSTEPS];
int g_soundHitLight[NUM_SOUNDS_HITLIGHT];
int g_soundHitHeavy[NUM_SOUNDS_HITHEAVY];
int g_soundKick[NUM_SOUNDS_KICK];
int g_soundPunch[NUM_SOUNDS_PUNCH];
int g_soundDeepBlow[NUM_SOUNDS_DEEPBLOW];

//fetus blaster global things
float g_fetusPilotZ = 512.0f;
float g_levelScroll[3] = {0.0f, 0.0f, 0.0f};

bool g_fetusSideMode = false;
float g_fetusSideModeX = 0.0f;

gameObject_t g_gameObjects[MAX_GAME_OBJECTS];
clientInfo_t g_clientInfo[MAX_NET_CLIENTS];
int g_gameObjectSlots = 0;
int g_creationCount = 0;

//return api version
int LServ_GetAPIVersion(void)
{
	return LOGIC_API_VERSION;
}

//find one that isn't in use
gameObject_t *LServ_CreateObject(void)
{
	int i = MAP_OBJ_NUM+1;
	while (i < MAX_GAME_OBJECTS)
	{
		if (!g_gameObjects[i].inuse && g_gameObjects[i].noReuseTime <= g_curTime)
		{
			g_gameObjects[i].inuse = INUSE_ACTIVE;
			g_gameObjects[i].net.index = i;
			g_gameObjects[i].net.owner = MAP_OBJ_NUM;
			g_gameObjects[i].net.staticIndex = -1;
			g_gameObjects[i].creationCount = g_creationCount;
			g_creationCount++;
			g_gameObjects[i].localTimeScale = 1.0f;

			if (g_gameObjectSlots <= i)
			{
				g_gameObjectSlots = i+1;
			}

			return &g_gameObjects[i];
		}
		i++;
	}

	return NULL;
}

//kill it
void LServ_FreeObject(gameObject_t *obj)
{
	if (obj->onremove)
	{
		obj->onremove(obj);
	}
	if (obj->rcColModel)
	{
		g_sharedFn->Coll_DestroyModelInstance(obj->rcColModel);
	}
	if (obj->visBlockData)
	{
		g_sharedFn->Common_RCFree(obj->visBlockData);
	}
	if (obj->plObj)
	{
		if (obj->plObj->targetObj)
		{
			obj->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(obj);
		}
		g_sharedFn->Common_RCFree(obj->plObj);
	}
	if (obj->aiObj)
	{
		AI_FreeAIObject(obj->aiObj);
	}
	if (obj->cinObj)
	{
		g_sharedFn->Common_RCFree(obj->cinObj);
	}
	if (obj->rscript)
	{
		RScr_DestroyScript(obj->rscript);
	}
	if (obj->swordObj)
	{
		g_sharedFn->Common_RCFree(obj->swordObj);
	}
	if (obj->customDynamic)
	{
		g_sharedFn->Common_RCFree(obj->customDynamic);
	}
	memset(obj, 0, sizeof(gameObject_t));
	obj->noReuseTime = g_curTime + 1000; //don't reuse it so that the client can get an update on its removal first
}

//flag INUSE_NONET based on visibility within camera frustum
void LServ_NetCull(gameObject_t *obj)
{
	if (!g_camFrustumFresh)
	{
		return;
	}

	if (obj->net.type == OBJ_TYPE_VISBLOCK)
	{ //don't attempt to cull these
		return;
	}

	if (obj->noCullTime >= g_curTime || obj->alwaysSend)
	{
		obj->inuse &= ~INUSE_NONET;
		return;
	}

	if (obj->localFlags & LFL_NONET)
	{ //never send
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (obj->net.type == OBJ_TYPE_LIGHT &&
		(obj->net.renderEffects & FXFL_HIDDEN))
	{ //cull out invisible lights
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (obj->cinObj)
	{
		if (obj->cinObj->hideable && (obj->net.renderEffects & FXFL_HIDDEN))
		{
			obj->inuse |= INUSE_NONET;
			return;
		}
		else if (!obj->net.solid)
		{ //hack for cinematic objects so they can animate way outside of their bounds
			obj->inuse &= ~INUSE_NONET;
			return;
		}
	}

	if (obj->spawnMins[0] == 0.0f &&
		obj->spawnMins[1] == 0.0f &&
		obj->spawnMins[2] == 0.0f &&
		obj->spawnMaxs[0] == 0.0f &&
		obj->spawnMaxs[1] == 0.0f &&
		obj->spawnMaxs[2] == 0.0f)
	{ //spawn bounds not set
		return;
	}

	if (!Math_PointInFrustum(&g_wideCamFrustum, obj->net.pos, Util_RadiusFromBounds(obj->spawnMins, obj->spawnMaxs, 2.0f)))
	{ //not in frustum
		if (!(obj->net.renderEffects & FXFL_DEATH))
		{ //don't frustum-cull things with death fx
			obj->inuse |= INUSE_NONET;
			return;
		}
	}

	if (!ObjVisBlock_CheckVis(obj))
	{ //visibility system culled it
		obj->inuse |= INUSE_NONET;
		return;
	}

	if (!obj->rcColModel || !g_sharedFn->Coll_IsTreeModel(obj->rcColModel))
	{
		gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
		float rad;
		float midPos[3];
		if (obj->net.type == OBJ_TYPE_LIGHT)
		{
			rad = obj->net.modelScale[0];
			Math_VecCopy(obj->net.pos, midPos);
		}
		else
		{
			rad = Math_Max3(obj->spawnMaxs[0]-obj->spawnMins[0],
								obj->spawnMaxs[1]-obj->spawnMins[1],
								obj->spawnMaxs[2]-obj->spawnMins[2])*0.5f;
			midPos[0] = obj->net.pos[0] + obj->spawnMins[0] + (obj->spawnMaxs[0]-obj->spawnMins[0])*0.5f;
			midPos[1] = obj->net.pos[1] + obj->spawnMins[1] + (obj->spawnMaxs[1]-obj->spawnMins[1])*0.5f;
			midPos[2] = obj->net.pos[2] + obj->spawnMins[2] + (obj->spawnMaxs[2]-obj->spawnMins[2])*0.5f;
		}
		if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(midPos, cam->net.pos, rad, obj->spawnMins, obj->spawnMaxs))
		{ //sphere visibility failed
			obj->inuse |= INUSE_NONET;
			return;
		}
	}

	//visible
	obj->inuse &= ~INUSE_NONET;
}

//update an object's rclip
void LServ_UpdateRClip(gameObject_t *obj)
{
	if (!obj || !obj->rcColModel)
	{
		return;
	}
	Math_VecCopy(obj->net.mins, obj->rcColModel->mins);
	Math_VecCopy(obj->net.maxs, obj->rcColModel->maxs);
	Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
	Math_VecCopy(obj->net.ang, obj->rcColModel->ang);
	Math_VecCopy(obj->net.modelScale, obj->rcColModel->modelScale);
	obj->rcColModel->gameOwner = obj->net.index;
	obj->rcColModel->solid = obj->net.solid;
	if (obj->rcColModel->frame != obj->net.frame)
	{
		obj->rcColModel->frame = obj->net.frame;
		obj->rcColModel->blendFrame = obj->net.frame;
	}
	g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
}

//spawn an enemy object
void LServ_SpawnEnemy(const char *name, float *pos)
{
	float ang[3];
	ang[0] = 0.0f;
	ang[1] = 0.0f;
	ang[2] = 0.0f;
	LServ_ObjectFromName(name, pos, ang, NULL, 0);
	//optionally set extra object parameters
}

//spawn relative to camera in a spread
void LServ_SpawnEnemySpread(const char *name)
{
	if (!name)
	{
		int r = rand()%13;//9;
		if (r >= 6)
		{
			name = "obj_gen_fighter01";
		}
		else if (r <= 3)
		{
			name = "obj_gen_fighterspin";
		}
		else
		{
			name = "obj_gen_fightershooter";
		}
	}
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	if (!cam->inuse || !cam->target || !cam->target->inuse)
	{
		return;
	}

	float d[3];
	Math_VecSub(cam->target->net.pos, cam->camTrackPos, d);
	
	Math_VecNorm(d);
	float a[3], right[3];
	Math_VecToAngles(d, a);
	Math_AngleVectors(a, 0, right, 0);

	float r = -1500.0f + (float)(rand()%3000);
	Math_VecScale(d, 1500.0f);
	d[0] += cam->camTrackPos[0] + right[0]*r;
	d[1] += cam->camTrackPos[1] + right[1]*r;
	d[2] += g_fetusPilotZ + right[2]*r;

	LServ_SpawnEnemy(name, d);
}

//in musical mode, spawn enemies constantly
void LServ_MusicalSpawns(void)
{
	static bool nextInBatch = false;
	if ((g_musicalAmp-g_musicalLastFrame) > 0.4f)
	{ //the music went up rapidly between frames, spawn a batch
		nextInBatch = true;
	}

	if (g_musicalSpawnTime > g_curTime)
	{ //not long enough since the last spawn
		return;
	}

	if (g_musicalAmp < 0.05f)
	{ //don't spawn during silent periods
		return;
	}

	int activeEnemies = 0;
	for (int i = MAX_NET_CLIENTS; i < g_gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->hurtable && obj->health > 0 && obj->net.solid)
		{
			activeEnemies++;
			if (activeEnemies > 15)
			{ //start killing the bastards
				Util_DamageObject(obj, obj, 9999999);
			}
		}
	}

	LServ_SpawnEnemySpread(NULL);
	if (nextInBatch)
	{
		LServ_SpawnEnemySpread(NULL);
		LServ_SpawnEnemySpread(NULL);
		LServ_SpawnEnemySpread(NULL);
		nextInBatch = false;
	}
	else if ((g_musicalRunningAvg/(float)g_musicalRunningCount) > 0.37f)
	{
		LServ_SpawnEnemySpread(NULL);
		LServ_SpawnEnemySpread(NULL);
	}

	float modAmp = g_musicalAmp*1.4f;
	if (modAmp > 1.0f)
	{
		modAmp = 1.0f;
	}
	serverTime_t spawnTime = (serverTime_t)((1.0f-modAmp)*3000.0f);
	if (spawnTime < 50)
	{
		spawnTime = 50;
	}
	g_musicalSpawnTime = g_curTime+spawnTime;
}

//get existing data
char *LServ_GetHighScoreData(void)
{
	int fileHandle = g_sharedFn->FileSys_OpenFile("assets/persdata/musicalscores.txt", _O_RDONLY, _S_IREAD);
	if (fileHandle == -1)
	{
		return NULL;
	}

	int l = g_sharedFn->FileSys_GetLen(fileHandle);
	if (l <= 0)
	{
		g_sharedFn->FileSys_CloseFile(fileHandle);
		return NULL;
	}
	char *scoreData = (char *)g_sharedFn->Common_RCMalloc(l+1);
	g_sharedFn->FileSys_ReadFile(fileHandle, scoreData, l);
	g_sharedFn->FileSys_CloseFile(fileHandle);
	scoreData[l] = 0;

	return scoreData;
}

//write new highscore data
void LServ_SetHighScoreData(char *scoreData)
{
	int fileHandle = g_sharedFn->FileSys_OpenFile("assets/persdata/musicalscores.txt", _O_WRONLY|_O_CREAT|_O_TRUNC, _S_IWRITE);
	if (fileHandle == -1)
	{
		return;
	}

	g_sharedFn->FileSys_WriteFile(fileHandle, scoreData, (int)strlen(scoreData));
	g_sharedFn->FileSys_CloseFile(fileHandle);
}

//get the highscore for this media
int LServ_GetMusicHighScore(char *musicName)
{
	char *scoreData = LServ_GetHighScoreData();
	if (!scoreData)
	{
		return 0;
	}

	char scoreStr[4096];
	sprintf(scoreStr, "\n%s\n", musicName);
	char *p = strstr(scoreData, scoreStr);
	int score = 0;
	if (p)
	{
		strcat(scoreStr, "%i\n");
		sscanf(p, scoreStr, &score);
	}
	g_sharedFn->Common_RCFree(scoreData);
	return score;
}

//record highscore
void LServ_SaveNewHighscore(char *musicName)
{
	char scoreStr[4096];
	sprintf(scoreStr, "\n%s\n", musicName);

	char *scoreData = LServ_GetHighScoreData();
	if (!scoreData)
	{
		sprintf(scoreStr, "\n%s\n%i\n", musicName, g_musicHighScore);
		LServ_SetHighScoreData(scoreStr);
		return;
	}

	char *p = strstr(scoreData, scoreStr);
	if (!p)
	{
		sprintf(scoreStr, "\n%s\n%i\n", musicName, g_musicHighScore);
		char *out = (char *)g_sharedFn->Common_RCMalloc((int)strlen(scoreData)+(int)strlen(scoreStr)+1);
		strcpy(out, scoreData);
		strcat(out, scoreStr);
		LServ_SetHighScoreData(out);
		g_sharedFn->Common_RCFree(scoreData);
		g_sharedFn->Common_RCFree(out);
		return;
	}

	size_t l = p-scoreData;
	int numEnd = 0;
	while (numEnd < 3)
	{
		if (!*p)
		{
			break;
		}
		if (*p == '\n')
		{
			numEnd++;
		}
		p++;
	}
	sprintf(scoreStr, "\n%s\n%i\n", musicName, g_musicHighScore);
	size_t preSize = l;
	l = l + strlen(p) + strlen(scoreStr);
	char *out = (char *)g_sharedFn->Common_RCMalloc((int)l+4);
	memcpy(out, scoreData, preSize);
	*(out+preSize) = 0;
	strcat(out, scoreStr);
	strcat(out, p);

	LServ_SetHighScoreData(out);
	g_sharedFn->Common_RCFree(out);
	g_sharedFn->Common_RCFree(scoreData);
}

//check the new score against records and such
void LServ_MusicalScoreLog(gameObject_t *client)
{
	/*
	if (client->net.lives > g_musicHighScore)
	{
		g_musicHighScore = client->net.lives;
		g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), -1);
	}
	*/
	//wait until after the song is done to update i guess, in case they started screwing up and dying a lot
}

//reset music stuff
void LServ_MusicReset(void)
{
	g_musicReset = 0;
	g_musicalRunningAvg = 0.0f;
	g_musicalLastFrame = 0.0f;
	g_musicalSpawnTime = 0;
	g_musicalRunningCount = 1;

	char str[1024];
	int plNum = 1;
	bool wasPlayingMusical = (g_musicResetName[0] != 0);
	if (wasPlayingMusical)
	{
		sprintf(str, "Results for %s:", g_musicResetName);
		g_sharedFn->Net_SendMessage(str);
	}
	int highestScore = -99999999;
	int winningClient = 0;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (pl->inuse)
		{
			if (wasPlayingMusical)
			{
				if (pl->net.lives > g_musicHighScorePrev)
				{
					sprintf(str, "Player %i (%s) destroyed the best score with %i.", plNum, g_clientInfo[pl->net.index].infoStr, pl->net.lives);
					winningClient = i;
				}
				else if (pl->net.lives >= 0)
				{
					sprintf(str, "Player %i (%s) finished with a score of %i.", plNum, g_clientInfo[pl->net.index].infoStr, pl->net.lives);
				}
				else
				{
					sprintf(str, "Player %i (%s) failed miserably with a score of %i.", plNum, g_clientInfo[pl->net.index].infoStr, pl->net.lives);
				}
				g_sharedFn->Net_SendMessage(str);
			}
			if (pl->net.lives > highestScore)
			{
				highestScore = pl->net.lives;
			}
			pl->health = 200;
			pl->net.plAmmo = 0;
			plNum++;
		}
	}

	if (wasPlayingMusical && highestScore > g_musicHighScore)
	{
		g_musicHighScore = highestScore;
		if (g_musicHighScore > 0)
		{
			LServ_SaveNewHighscore(g_musicResetName);
		}
	}

	g_musicHighScore = LServ_GetMusicHighScore(g_musicResetNameNext);
	g_musicHighScorePrev = g_musicHighScore;
	g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), -1);

	sprintf(str, "Now playing: %s", g_musicResetNameNext);
	g_sharedFn->Net_SendMessage(str);
	//todo high score tracking
	strcpy(g_musicResetName, g_musicResetNameNext);

	//destroy all enemies and such
	for (int i = MAX_NET_CLIENTS; i < g_gameObjectSlots; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (!other->inuse)
		{
			continue;
		}
		if (other->projDamage)
		{ //remove projectile
			other->think = ObjGeneral_RemoveThink;
			other->thinkTime = g_curTime-1;
		}
		else if (other->net.solid && other->hurtable && other->health > 0)
		{ //kill it
			Util_DamageObject(other, other, 9999999);
		}
	}
}

//in musical mode, modify the music string
void LServ_ChangeMusic(char *music)
{
	if (g_musicStrIndex == -1)
	{
		g_musicStrIndex = g_sharedFn->Common_ServerString(music);
	}
	else
	{
		g_sharedFn->Common_ModifyString(g_musicStrIndex, ""); //force cleansing
		g_sharedFn->Common_ModifyString(g_musicStrIndex, music);
	}
}

//event
void LServ_ServerEvent(int clientIndex, serverEventTypes_e ev, BYTE *data, int dataSize)
{
	switch (ev)
	{
	case SVEV_FORGEABILITY:
		if (dataSize == sizeof(int))
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				Shared_ForgeAbility(&obj->plObj->plData, *((int *)data), obj->plObj->makoStash);
				obj->plObj->makoCharges = obj->plObj->plData.maxMakoCharges;
			}
		}
		break;
	case SVEV_FORGEITEM:
		if (dataSize == sizeof(int))
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				Shared_ForgeItem(&obj->plObj->plData, *((int *)data), obj->plObj->makoStash);
			}
		}
		break;
	case SVEV_FORGESTAT:
		if (dataSize == sizeof(int))
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				Shared_ForgeStat(&obj->plObj->plData, *((int *)data), obj->plObj->makoStash);
				obj->health = obj->plObj->plData.maxHealth;
			}
		}
		break;
	case SVEV_INVSORT:
		if (dataSize == sizeof(int)*2)
		{
			gameObject_t *obj = &g_gameObjects[clientIndex];
			if (obj->inuse && obj->plObj)
			{
				int src = *((int *)data);
				int dst = *(((int *)data)+1);
				playerInvItem_t itemDst = obj->plObj->plData.inventory[dst];
				obj->plObj->plData.inventory[dst] = obj->plObj->plData.inventory[src];
				obj->plObj->plData.inventory[src] = itemDst;
			}
		}
		break;
	default:
		break;
	}
}

//a development-only command
void LServ_DevCommand(char *cmd)
{
	GVar_SetInt("useddevcmd", 1);

	textParser_t *parser = g_sharedFn->Parse_InitParser(cmd);
	if (!parser)
	{
		return;
	}
	parseToken_t token;
	while (g_sharedFn->Parse_GetNextToken(parser, &token))
	{
		if (!stricmp(token.text, "spawn"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
				float pos[3], ang[3], dir[3];
				Math_AngleVectors(cam->net.ang, dir, 0, 0);
				ang[0] = 0.0f;
				ang[1] = cam->net.ang[1];
				ang[2] = 0.0f;
				Math_VecMA(cam->net.pos, 2500.0f, dir, pos);
				gameObject_t *obj = LServ_ObjectFromName(token.text, pos, ang, NULL, 0);
				if (!obj)
				{
					Util_StatusMessage("Object '%s' is not valid.", token.text);
				}
			}
			else
			{
				Util_StatusMessage("spawn command requires an object name.");
			}
		}
		else if (!stricmp(token.text, "<3"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->plData.maxHealth = 1000;
				pl->health = 1000;
				pl->plObj->plData.maxMakoCharges = 4;
				pl->plObj->makoCharges = 4;
				pl->plObj->plData.abilities = (1<<PLABIL_WALLJUMP)|(1<<PLABIL_GETUP)|(1<<PLABIL_FLIPKICK)|(1<<PLABIL_FLURRY)|(1<<PLABIL_THRUST)|(1<<PLABIL_TACKLE)|
						(1<<PLABIL_GROUNDCRUSH)|(1<<PLABIL_METEORKICK)|(1<<PLABIL_MAKOFLAME)|(1<<PLABIL_MAKOTIME)|(1<<PLABIL_MAKOHEAL)|(1<<PLABIL_MAKOAURA)|
						(1<<PLABIL_CHARGEFORCE);
			}
		}
		else if (!stricmp(token.text, "mako"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				if (g_sharedFn->Parse_GetNextToken(parser, &token))
				{
					ObjPlayer_GiveMako(pl, atoi(token.text));
					if (pl->plObj->makoStash < 0)
					{
						pl->plObj->makoStash = 0;
					}
				}
				else
				{
					Util_StatusMessage("mako command requires a value.");
				}
			}
		}
		else if (!stricmp(token.text, "str") || !stricmp(token.text, "def") || !stricmp(token.text, "dex") ||
			!stricmp(token.text, "luck"))
		{
			char cmdName[64];
			strcpy(cmdName, token.text);
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				if (pl->plObj)
				{
					if (!stricmp(cmdName, "str"))
					{
						pl->plObj->plData.statStr = (BYTE)atoi(token.text);
					}
					else if (!stricmp(cmdName, "def"))
					{
						pl->plObj->plData.statDef = (BYTE)atoi(token.text);
					}
					else if (!stricmp(cmdName, "dex"))
					{
						pl->plObj->plData.statDex = (BYTE)atoi(token.text);
					}
					else if (!stricmp(cmdName, "luck"))
					{
						pl->plObj->plData.statLuck = (BYTE)atoi(token.text);
					}
				}
			}
			else
			{
				Util_StatusMessage("%s command requires value.", cmdName);
			}
		}
		else if (!stricmp(token.text, "noclip"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->noClip = !pl->plObj->noClip;
			}
		}
		else if (!stricmp(token.text, "jesus"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->invincible = !pl->plObj->invincible;
			}
		}
		else if (!stricmp(token.text, "notarget"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				pl->plObj->noTarget = !pl->plObj->noTarget;
			}
		}
		else if (!stricmp(token.text, "npsngs"))
		{
			g_noDeath = !g_noDeath;
		}
		else if (!stricmp(token.text, "aipath"))
		{
			g_ai.showPaths = !g_ai.showPaths;
		}
		else if (!stricmp(token.text, "getpos"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			Util_StatusMessage("pos: (%f %f %f)", pl->net.pos[0], pl->net.pos[1], pl->net.pos[2]);
		}
		else if (!stricmp(token.text, "setpos"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				sscanf(token.text, "(%f %f %f)", &pl->net.pos[0], &pl->net.pos[1], &pl->net.pos[2]);
			}
			else
			{
				Util_StatusMessage("setpos command requires argument.");
			}
		}
		else if (!stricmp(token.text, "killemall"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			for (int i = 0; i < g_gameObjectSlots; i++)
			{
				gameObject_t *obj = &g_gameObjects[i];
				if (obj->inuse && obj->hurtable && (obj->localFlags & LFL_ENEMY))
				{
					Util_DamageObject(pl, obj, 99999999, 0);
				}
			}
		}
		else if (!stricmp(token.text, "giveitup"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->inuse && pl->plObj)
			{
				for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
				{
					const invItemDef_t *item = &g_invItems[i];
					pl->plObj->plData.inventory[i].itemIndex = i;
					pl->plObj->plData.inventory[i].itemQuantity = item->maxAmount;
				}
			}
		}
		else if (!stricmp(token.text, "timescale"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				g_timeScale = (float)atof(token.text);
			}
			else
			{
				Util_StatusMessage("timescale command requires argument.");
			}
		}
		else if (!stricmp(token.text, "testfx"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				float c[3];
				float ang[3] = {0.0f, 0.0f, 0.0f};
				Util_GameObjectCenter(pl, c);
				ObjParticles_Create(token.text, c, ang, -1);
			}
			else
			{
				Util_StatusMessage("testfx command requires argument.");
			}
		}
		else if (!stricmp(token.text, "rfx"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				pl->net.renderEffects ^= (1<<atoi(token.text));
			}
			else
			{
				Util_StatusMessage("rfx command requires argument.");
			}
		}
		else if (!stricmp(token.text, "rfx2"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				gameObject_t *pl = &g_gameObjects[0];
				pl->net.renderEffects2 ^= (1<<atoi(token.text));
			}
			else
			{
				Util_StatusMessage("rfx2 command requires argument.");
			}
		}
		else if (!stricmp(token.text, "^#$^%"))
		{
			gameObject_t *pl = &g_gameObjects[0];
			if (pl->plObj)
			{
				ObjPlBar_Init(pl);
			}
		}
		else if (!stricmp(token.text, "runscript"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				Util_RunScript("obj_testrun", token.text);
			}
			else
			{
				Util_StatusMessage("runscript command requires a script name.");
			}
		}
		else if (!stricmp(token.text, "map"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				g_sharedFn->Common_MapChange(token.text, 1000);
			}
			else
			{
				Util_StatusMessage("map command requires a map name.");
			}
		}
		else if (!stricmp(token.text, "gvar"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				char gvarName[2048];
				strcpy(gvarName, token.text);
				if (g_sharedFn->Parse_GetNextToken(parser, &token))
				{
					if (!stricmp(gvarName, "useddevcmd") || !stricmp(gvarName, "cdt_usingmod"))
					{
						Util_StatusMessage("Why would you want to do that?");
					}
					else
					{
						GVar_SetValue(gvarName, token.text);
					}
				}
				else
				{
					Util_StatusMessage("gvar '%s' is '%s'.", gvarName, GVar_GetValue(gvarName));
				}
			}
			else
			{
				Util_StatusMessage("gvar command requires a gvar name.");
			}
		}
		else if (!stricmp(token.text, "gvarlist"))
		{
			GVar_ListGVars();
		}
		else if (!stricmp(token.text, "clearerror"))
		{
			g_sharedFn->Net_SendEventType(CLEV_ERRORMESSAGE, "\0", 1, -1);
		}
		else if (!stricmp(token.text, "mscript"))
		{
			if (g_sharedFn->Parse_GetNextToken(parser, &token))
			{
				g_sharedFn->Net_SendEventType(CLEV_RUNMENUSCRIPT, token.text, (int)strlen(token.text)+1, -1);
			}
			else
			{
				Util_StatusMessage("mscript command requires a script name.");
			}
		}
		else
		{
			Util_StatusMessage("Unknown dev command: '%s'", token.text);
		}
	}

	g_sharedFn->Parse_FreeParser(parser);
}

//music logic
void LServ_MusicLogic(void)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (g_battleTime > g_curTime && g_battleMusicEnabled)
	{
		if (!g_battleMusic)
		{
			LServ_ChangeMusic("$#assets/music/FullFrontalAssault.ogg");
			g_battleMusic = true;
		}
	}
	else if (!g_battleMusic && pl->inuse && pl->plObj && pl->plObj->ride)
	{ //chocobo!
		if (!g_chocoboMusic)
		{
			LServ_ChangeMusic("$#assets/music/Kweh.ogg");
			g_chocoboMusic = true;
		}
	}
	else
	{
		if (g_battleMusic || g_chocoboMusic)
		{
			LServ_ChangeMusic(g_levelMusic);
			g_battleMusic = false;
			g_chocoboMusic = false;
		}
	}
}

//show stats
void LServ_ShowMPStats(void)
{
	bool tied = false;
	int bestCl = -1;
	int bestScore = -1;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *pl = &g_gameObjects[i];
		if (!pl->inuse || !pl->plObj || !g_clientInfo[i].infoStr)
		{
			continue;
		}

		char str[64];
		sprintf(str, "pls_%s", g_clientInfo[i].infoStr);
		int score = GVar_GetInt(str);
		bool cheating = (i == 0) ? (GVar_GetInt("useddevcmd") || GVar_GetInt("cdt_usingmod")) : pl->plObj->clCheater;
		char *winStr = (score == 1) ? "win" : "wins";
		if (cheating)
		{
			Util_StatusMessage("%s has %i %s, but may be cheating.", g_clientInfo[i].infoStr, score, winStr);
		}
		else
		{
			Util_StatusMessage("%s has %i %s.", g_clientInfo[i].infoStr, score, winStr);
		}
		if (score > bestScore)
		{
			bestScore = score;
			bestCl = i;
		}
		else if (score == bestScore)
		{
			tied = true;
		}
	}

	if (tied)
	{
		Util_StatusMessage("The winning position tied.");
	}
	else if (bestCl != -1)
	{
		Util_StatusMessage("%s is winning with %i.", g_clientInfo[bestCl].infoStr, bestScore);
	}
}

//mp versus mode checks
static void LServ_VersusLogic(void)
{
	if (g_mpCountDown > 0)
	{
		if (g_mpCountTime < g_curTime)
		{
			gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
			g_mpCountDown--;
			if (g_mpCountDown == 0)
			{
				int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
				*t = 600;
				float *c = (float *)(t+1);
				c[0] = 1.0f;
				c[1] = 1.0f;
				c[2] = 1.0f;
				g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
				ObjSound_Create(cam->net.pos, "assets/sound/cb/fistpower.wav", 1.0f, -1);
				Util_StatusMessage("*(1 0 0 1)FIGHT!");
				for (int i = 0; i < MAX_NET_CLIENTS; i++)
				{
					gameObject_t *obj = &g_gameObjects[i];
					if (!obj->inuse || !obj->plObj)
					{
						continue;
					}
					AI_StartAnim(obj, HUMANIM_IDLE, true);
				}
			}
			else
			{
				ObjSound_Create(cam->net.pos, "assets/sound/cb/bigfoot1.wav", 1.0f, -1);
				Util_StatusMessage("*(1 1 1 1)%i...", g_mpCountDown);
				g_mpCountTime = g_curTime+1000;
			}
		}
	}

	gameObject_t *living[MAX_NET_CLIENTS];
	int numLiving = 0;
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (!obj->inuse || !obj->plObj)
		{
			continue;
		}

		if (i < 2)
		{
			g_globalNet.vsHealth[i] = obj->health;
			if (g_globalNet.vsHealth[i] < 0)
			{
				g_globalNet.vsHealth[i] = 0;
			}
		}

		if (obj->health <= 0)
		{
			continue;
		}

		living[numLiving] = obj;
		numLiving++;
	}

	if (numLiving >= 2)
	{
		return;
	}

	if (numLiving <= 0)
	{
		Util_StatusMessage("Double K.O.!");
	}
	else
	{
		int clIndex = living[0]->net.index;
		Util_StatusMessage("%s is the victor!", g_clientInfo[clIndex].infoStr);
		char str[64];
		sprintf(str, "pls_%s", g_clientInfo[clIndex].infoStr);
		int score = GVar_GetInt(str);
		score++;
		GVar_SetInt(str, score);
	}

	LServ_ShowMPStats();

	g_mpVersusMode = false;
	Util_RunScript("obj_mpscript", "lshub_mpfinished");
}

//logic frame
void LServ_RunFrame(unsigned long prvTime, unsigned long curTime, bool paused)
{
	static float secondCounter = 0.0f;
	float timeDif = (float)(curTime-prvTime);

	secondCounter += timeDif;
	while (secondCounter >= 1000.0f)
	{ //increment persistent play time counter
		secondCounter -= 1000.0f;
		int sec = GVar_GetInt("playsec");
		sec++;
		GVar_SetInt("playsec", sec);
	}

	if (timeDif > 100.0f)
	{ //cap to something sane
		timeDif = 100.0f;
	}

	if (g_actionTime)
	{
		if (g_actionTime > g_curTime)
		{
			if (g_timeType == TIMETYPE_NORMAL)
			{
				g_timeType = TIMETYPE_ACTION;
			}
		}
		else
		{
			g_actionTime = 0;
			if (g_timeType == TIMETYPE_ACTION)
			{
				g_timeType = TIMETYPE_NORMAL;
			}
		}
	}

	if (g_tonberryTime)
	{
		if (g_tonberryTime > g_curTime)
		{
			if (g_timeType == TIMETYPE_NORMAL)
			{
				g_timeType = TIMETYPE_TONBERRY;
			}
		}
		else
		{
			g_tonberryTime = 0;
			if (g_timeType == TIMETYPE_TONBERRY)
			{
				g_timeType = TIMETYPE_NORMAL;
			}
		}
	}

	if (g_cinema == 1 && g_gameObjects[0].inuse && g_gameObjects[0].plObj &&
		g_gameObjects[0].plObj->clButtons[BUTTON_EVENT1])
	{ //fast-forward
		g_timeScale = 8.0f;
		g_lastTimeType = -1;
	}
	else if (g_timeType != g_lastTimeType)
	{
		float plTime;
		switch (g_timeType)
		{
		case TIMETYPE_NORMAL:
			g_timeScale = 1.0f;
			plTime = 1.0f;
			break;
		case TIMETYPE_TIMEATTACK:
			g_timeScale = 0.25f;
			plTime = 4.0f;
			break;
		case TIMETYPE_TONBERRY:
			g_timeScale = 0.125f;
			plTime = 1.0f;
			break;
		case TIMETYPE_DEATH:
			g_timeScale = 0.25f;
			plTime = 1.0f;
			break;
		case TIMETYPE_ACTION:
			g_timeScale = g_actionTimeScale;
			plTime = 1.0f;
			break;
		default:
			g_timeScale = 1.0f;
			plTime = 1.0f;
			break;
		}
		for (int i = 0; i < MAX_NET_CLIENTS; i++)
		{
			gameObject_t *obj = &g_gameObjects[i];
			if (g_timeType == TIMETYPE_TIMEATTACK)
			{
				obj->net.renderEffects |= FXFL_CONTRAST;
			}
			else
			{
				obj->net.renderEffects &= ~FXFL_CONTRAST;
			}
			obj->localTimeScale = plTime;
		}
		g_gameObjects[CAM_TRACK_NUM].localTimeScale = plTime;
		int ltt = g_lastTimeType;
		g_lastTimeType = g_timeType;
		if (g_timeType != TIMETYPE_DEATH && ltt != -1)
		{
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 300;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 1.0f;
			c[2] = 1.0f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
	}
	else if (g_timeType == TIMETYPE_ACTION)
	{
		g_timeScale = g_actionTimeScale;
	}

	g_invTimeScale = 1.0f/g_timeScale;
	timeDif *= g_timeScale;

	g_timeFactor = timeDif/50.0f; //compensate for dropped frames
	if (g_timeFactor > 2.0f)
	{
		g_timeFactor = 2.0f;
	}
	else if (g_timeFactor < 0.1f)
	{
		g_timeFactor = 0.1f;
	}

	if (paused)
	{
		return;
	}

	g_sharedFn->Coll_SetupVisibility(g_gameObjects[CAM_TRACK_NUM].net.pos);

	if (!g_curTime || prvTime == curTime)
	{
		g_curTime = (serverTime_t)curTime;
	}
	else
	{
		g_curTime += (serverTime_t)timeDif;
	}

	LServ_MusicLogic();

	if (g_musicReset && g_musicReset < g_curTime)
	{
		LServ_MusicReset();
	}

	if (g_musical)
	{
		float amp, pos;
		g_sharedFn->Common_QueryMusic(&amp, &pos);
		gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
		if (cam->inuse)
		{
			cam->net.modelScale[1] = pos;
		}
		g_musicalLastFrame = g_musicalAmp;
		g_musicalAmp = amp;

		g_musicalRunningAvg += amp;
		g_musicalRunningCount++;
		int avgChunk = 256;
		if (g_musicalRunningCount > avgChunk)
		{
			g_musicalRunningAvg /= (float)g_musicalRunningCount;
			g_musicalRunningAvg *= ((float)avgChunk/2.0f);
			g_musicalRunningCount = avgChunk/2;
		}

		LServ_MusicalSpawns();
	}

	AI_GlobalFrame();

	gameObject_t *activeLights[MAX_GAME_OBJECTS];
	int numActiveLights = 0;
	gameObject_t *obj;
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		obj = &g_gameObjects[i];
		if (obj->inuse)
		{
			LServ_NetCull(obj);
			if (obj->net.type == OBJ_TYPE_LIGHT &&
				!(obj->inuse & INUSE_NONET) &&
				obj->net.lerpDuration)
			{ //add to the unculled lights list (only if it's a shadowing light)
				activeLights[numActiveLights++] = obj;
			}
			if (obj->think && obj->thinkTime < g_curTime && !obj->parent)
			{
				obj->think(obj, g_timeFactor*obj->localTimeScale);
				if (obj->child)
				{ //think for the child
					obj->child->think(obj->child, g_timeFactor*obj->localTimeScale);
				}
			}
		}
	}

	if (g_mpVersusMode)
	{
		LServ_VersusLogic();
	}

	if (numActiveLights > 0)
	{ //check culled objects against lights
		for (int i = 0; i < g_gameObjectSlots; i++)
		{
			obj = &g_gameObjects[i];

			if (!obj->inuse || !(obj->inuse & INUSE_NONET))
			{ //only interested if it is inuse and culled
				continue;
			}

			if (obj->net.type == OBJ_TYPE_LIGHT)
			{ //lights don't affect each other
				continue;
			}

			if (obj->localFlags & LFL_NONET)
			{ //never wants to be seen by the client
				continue;
			}

			if (obj->spawnMins[0] == 0.0f &&
				obj->spawnMins[1] == 0.0f &&
				obj->spawnMins[2] == 0.0f &&
				obj->spawnMaxs[0] == 0.0f &&
				obj->spawnMaxs[1] == 0.0f &&
				obj->spawnMaxs[2] == 0.0f)
			{ //spawn bounds not set
				continue;
			}

			for (int j = 0; j < numActiveLights; j++)
			{
				if (activeLights[j]->net.frame & LIGHTFL_FULLAMB)
				{ //do not consider pure ambient lights
					continue;
				}
				if (Util_GameObjectsOverlap(obj, activeLights[j]))
				{ //it interacts with a light in view, so make sure the client gets it.
					obj->inuse &= ~INUSE_NONET;
					break;
				}
			}
		}
	}

	//update client state data
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		obj = &g_gameObjects[i];
		if (!obj->inuse || !obj->plObj)
		{
			continue;
		}
		ObjPlayer_UpdatePlayerStates(obj);
		assert(sizeof(g_globalNet) < 32768); //make sure the delta can't get too big
		if (memcmp(&g_globalNet, &obj->plObj->lastGlobal, sizeof(g_globalNet)) != 0)
		{ //update global net data
			int msgSize = sizeof(g_globalNet)+1024;
			WORD *msgOut = (WORD *)_alloca(msgSize);
			BYTE *deltaOut = (BYTE *)(msgOut+1);
			int deltaSize = msgSize-4;

			int realDeltaSize = g_sharedFn->Net_CreateDelta(&obj->plObj->lastGlobal, &g_globalNet, sizeof(g_globalNet),
				deltaOut, deltaSize);
			memcpy(&obj->plObj->lastGlobal, &g_globalNet, sizeof(g_globalNet));
			assert(realDeltaSize > 0);
		
			*msgOut = realDeltaSize; //first word for the event type is the delta size

			g_sharedFn->Net_SendEventType(CLEV_GLOBALDATA, msgOut, realDeltaSize+2, obj->net.index);
		}
	}

	//ObjVisBlock_DebugDraw();
}

//make sure global resources are cached by the client
void LServ_GlobalResources(void)
{
	g_sharedFn->Common_ServerString("^assets/textures/2dattn");
	g_sharedFn->Common_ServerString("^assets/textures/flashattn");
	g_sharedFn->Common_ServerString("^assets/textures/fetusredshell");
	g_sharedFn->Common_ServerString("$assets/sound/menu/introsnd.wav");

	//battle music
	g_sharedFn->Common_ServerString("$assets/music/FullFrontalAssault.ogg");

	g_sharedFn->Common_ServerString("$assets/sound/cb/jump01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/meteorsmash.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/objland.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/hitbounce.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swingaura.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/walljump01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/groundsmash.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/hitaura.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstream.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstreamsh.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstreamint.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lstreamint2.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/lunge.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/makopickup.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/makopop.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/roll01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swingflame.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swingflame02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeaura.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeflame.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeheal.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargetime.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeoff.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chargeup.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/fire01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/healstep.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/pldeath.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/jesus.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/itempickup.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/itempop.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/ironhit.wav");
	g_soundHitHeavy[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitheavy01.wav");
	g_soundHitHeavy[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitheavy02.wav");
	g_soundHitHeavy[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitheavy03.wav");
	g_soundHitLight[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight01.wav");
	g_soundHitLight[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight02.wav");
	g_soundHitLight[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight03.wav");
	g_soundHitLight[3] = g_sharedFn->Common_ServerString("$assets/sound/cb/hitlight04.wav");
	g_soundPunch[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/punch01.wav");
	g_soundPunch[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/punch02.wav");
	g_soundPunch[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/punch03.wav");
	g_soundKick[0] = g_soundPunch[0];//g_sharedFn->Common_ServerString("$assets/sound/cb/kick01.wav");
	g_soundKick[1] = g_soundPunch[1];//g_sharedFn->Common_ServerString("$assets/sound/cb/kick02.wav");
	g_soundKick[2] = g_soundPunch[2];//g_sharedFn->Common_ServerString("$assets/sound/cb/kick03.wav");
	g_soundDeepBlow[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/kick01.wav");
	g_soundDeepBlow[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/kick02.wav");
	g_soundDeepBlow[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/kick03.wav");
	g_soundFootsteps[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/step01.wav");
	g_soundFootsteps[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/step02.wav");
	g_soundFootsteps[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/step03.wav");
	g_soundFootsteps[3] = g_sharedFn->Common_ServerString("$assets/sound/cb/step04.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swdun.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swdre.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/swdhit.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/slash01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/slash02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/slash03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bite01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bite02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/buscut01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/buscut02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/buscut03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/hitmetal.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/espawn.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/suck.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/quake.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/confuse.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cin/fast.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/crack01.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/crack02.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/crack03.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/fistpower.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bigfoot1.wav");

	g_sharedFn->Common_ServerString("$assets/sound/cb/glock.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bullet.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/bulflesh.wav");
	g_sharedFn->Common_ServerString("&&bulletspark");
	g_sharedFn->Common_ServerString("&&melee/bigslash");

	g_sharedFn->Common_ServerString("$assets/sound/items/rifle.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/usepotion.wav");
	g_sharedFn->Common_ServerString("$assets/sound/other/mmts.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/tr_silver.wav");
	g_sharedFn->Common_ServerString("$assets/sound/items/tr_gold.wav");
	for (int i = 0; i < NUM_INVENTORY_ITEM_DEFS; i++)
	{
		char s[256];
		const invItemDef_t *item = &g_invItems[i];
		if (item->useFX && item->useFX[0])
		{
			if (item->useFX[0] != '*')
			{
				sprintf(s, "&&%s", item->useFX);
				g_sharedFn->Common_ServerString(s);
			}
			else
			{
				LServ_CacheObj(&item->useFX[1]);
			}
		}
	}

	g_sharedFn->Common_ServerString("&&melee/bleed");
	g_sharedFn->Common_ServerString("&&melee/bleedless");
	g_sharedFn->Common_ServerString("&&melee/impact");
	g_sharedFn->Common_ServerString("&&melee/impacten");
	g_sharedFn->Common_ServerString("&&melee/impactexplosive");
	g_sharedFn->Common_ServerString("&&melee/impactslash");
	g_sharedFn->Common_ServerString("&&impacts/flyback");
	g_sharedFn->Common_ServerString("&&impacts/meteorimpact");
	g_sharedFn->Common_ServerString("&&impacts/aircrush");
	g_sharedFn->Common_ServerString("&&impacts/aircrushsuper");
	g_sharedFn->Common_ServerString("&&impacts/airsword");
	g_sharedFn->Common_ServerString("&&general/makopickup");
	g_sharedFn->Common_ServerString("&&general/makofade");
	g_sharedFn->Common_ServerString("&&general/makodeath");
	g_sharedFn->Common_ServerString("&&general/makosit");
	g_sharedFn->Common_ServerString("&&general/itemfade");
	g_sharedFn->Common_ServerString("&&general/itempickup");
	g_sharedFn->Common_ServerString("&&general/droppeditem");
	g_sharedFn->Common_ServerString("&&general/makochargeup");
	g_sharedFn->Common_ServerString("&&impacts/wallhop");
	g_sharedFn->Common_ServerString("&&playerfire");
	g_sharedFn->Common_ServerString("&&items/vortex");

	LServ_CacheObj("obj_mako_drop");
	LServ_CacheObj("obj_item_drop");

	LServ_CacheObj("obj_aerithshell");

	ObjProjectile_Init();

	g_globalNet.skyboxModel = g_sharedFn->Common_ServerString("&assets/models/levels/skyshell4.rdm");
	//g_globalNet.skyboxModel = g_sharedFn->Common_ServerString("&assets/models/levels/skyshell5.rdm");
	//g_globalNet.skyboxTerrain = g_sharedFn->Common_ServerString("&assets/models/terrain2_01.rdm");
}

//set up the main map object
void LServ_InitGlobalMapObj(void)
{
	gameObject_t *map = &g_gameObjects[MAP_OBJ_NUM];

	map->inuse = INUSE_ACTIVE;
	map->net.index = MAP_OBJ_NUM;
	map->creationCount = g_creationCount;
	g_creationCount++;
	map->localTimeScale = 1.0f;
	map->net.type = OBJ_TYPE_MAP;
	map->net.pos[0] = 0.0f;
	map->net.pos[1] = 0.0f;
	map->net.pos[2] = 0.0f;

	map->net.ang[0] = 0.0f;
	map->net.ang[1] = 0.0f;
	map->net.ang[2] = 0.0f;

	map->net.entNameIndex = g_sharedFn->Common_ServerString("obj_map");

	map->inuse |= INUSE_NONET; //for fetus blaster

	map->net.solid = 1;

	//Fetus Blaster - init the camera tracker
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];

	g_noConfineTime = 0;
	g_camFrustumFresh = false;
	g_camBoneBolt[0] = 0;
	g_camBoltOffset[0] = 0.0f;
	g_camBoltOffset[1] = 0.0f;
	g_camBoltOffset[2] = 0.0f;
	g_camAbsOffset[0] = 0.0f;
	g_camAbsOffset[1] = 0.0f;
	g_camAbsOffset[2] = 0.0f;
	g_camAbsAngOffset[0] = 0.0f;
	g_camAbsAngOffset[1] = 0.0f;
	g_camAbsAngOffset[2] = 0.0f;
	g_camSlack = 1.0f;
	cam->inuse = INUSE_ACTIVE;
	cam->targetName = "__the_camera";
	cam->net.staticIndex = -1;
	cam->net.index = CAM_TRACK_NUM;
	cam->creationCount = g_creationCount;
	g_creationCount++;
	cam->localTimeScale = 1.0f;
	cam->net.type = OBJ_TYPE_CAM;
	cam->net.pos[0] = 0.0f;
	cam->net.pos[1] = 0.0f;
	cam->net.pos[2] = DEFAULT_CAM_Z;//g_fetusPilotZ+2048.0f;

	cam->camTrackPos[0] = 0.0f;
	cam->camTrackPos[1] = 0.0f;
	cam->camTrackPos[2] = DEFAULT_CAM_Z;

	cam->net.ang[0] = 180.0f;
	cam->net.ang[1] = 0.0f;
	cam->net.ang[2] = 0.0f;

	g_levelScroll[0] = 0.0f;
	g_levelScroll[1] = 0.0f;
	g_levelScroll[2] = 0.0f;

	cam->net.maxs[0] = g_levelScroll[0];
	cam->net.maxs[1] = g_levelScroll[1];

	cam->net.modelScale[0] = 70.0f; //default fov

	//cam->net.renderEffects |= FXFL_FPSMODE;

	cam->net.entNameIndex = g_sharedFn->Common_ServerString("obj_camtrack");

	cam->think = ObjCam_Think;
	cam->thinkTime = 0;
}

//client disconnected
void LServ_FreePlayerObj(int clientNum)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (pl->inuse)
	{
		if (pl->plObj)
		{
			if (pl->plObj->targetObj)
			{
				pl->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(pl);
			}
			if (pl->plObj->ride)
			{
				ObjPlayer_UnmountRide(pl, true);
				pl->plObj->ride = NULL;
			}
			if (pl->plObj->flashlight)
			{
				pl->plObj->flashlight->think = ObjGeneral_RemoveThink;
				pl->plObj->flashlight = NULL;
			}
			if (pl->plObj->sword)
			{
				pl->plObj->sword->think = ObjGeneral_RemoveThink;
				pl->plObj->sword = NULL;
			}
			if (pl->plObj->gun)
			{
				pl->plObj->gun->think = ObjGeneral_RemoveThink;
				pl->plObj->gun = NULL;
			}
			if (pl->plObj->aerith)
			{
				pl->plObj->aerith->think = ObjGeneral_RemoveThink;
				pl->plObj->aerith = NULL;
			}
		}
		LServ_FreeObject(pl);
	}
}

//cache all stuff applicable
void LServ_CacheObj(const char *objName)
{
	const char *s;
	char assetName[2048];
	int i;
	int type;
	BYTE *b = Common_GetEntryForObject(objName);
	if (!b)
	{ //no entry for this object then
		return;
	}

	Util_ParseObjType(Common_GetValForKey(b, "type"), &type);

	s = Common_GetValForKey(b, "swordcache");
	if (s && atoi(s))
	{
		ObjSword_RegMedia(b, "swdHitSnd", NULL, false);
		ObjSword_RegMedia(b, "swdSwing", NULL, false);
		ObjSword_RegMedia(b, "swdImpact", NULL, false);
		ObjSword_RegMedia(b, "swdUnholst", NULL, false);
		ObjSword_RegMedia(b, "swdHolst", NULL, false);
		ObjSword_RegMedia(b, "swdHitFx", NULL, true);
		ObjSword_RegMedia(b, "swdImpactFx", NULL, true);
	}

	char resCacheStr[256];
	i = 0;
	sprintf(resCacheStr, "soundcache%i", i);
	s = Common_GetValForKey(b, resCacheStr);
	while (s && s[0] && i < 256)
	{
		sprintf(assetName, "$%s", s);
		g_sharedFn->Common_ServerString(assetName);
		i++;
		sprintf(resCacheStr, "soundcache%i", i);
		s = Common_GetValForKey(b, resCacheStr);
	}

	i = 0;
	sprintf(resCacheStr, "rplcache%i", i);
	s = Common_GetValForKey(b, resCacheStr);
	while (s && s[0] && i < 256)
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
		i++;
		sprintf(resCacheStr, "rplcache%i", i);
		s = Common_GetValForKey(b, resCacheStr);
	}

	s = Common_GetValForKey(b, "customTex");
	if (s && s[0])
	{ //make sure the client caches this texture
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//precache particle effects
	s = Common_GetValForKey(b, "rpl_muzzle");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}
	s = Common_GetValForKey(b, "rpl_muzzlechg");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}
	s = Common_GetValForKey(b, "rpl_impact");
	if (s && s[0])
	{
		sprintf(assetName, "&&%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//precache decals
	s = Common_GetValForKey(b, "dcl_impact");
	if (s && s[0])
	{
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//base on the type we need a different prepended char so the client
	//knows what to cache based on string name alone
	i = 0;
	if (type == OBJ_TYPE_SPRITE || type == OBJ_TYPE_PROJECTILE)
	{
		assetName[i++] = '#';
	}
	else if (type == OBJ_TYPE_MODEL)
	{
		assetName[i++] = '&';
	}
	assetName[i] = 0;
	s = Common_GetValForKey(b, "assetName");
	if ((!s || !s[0]) && assetName[0])
	{ //if it's a type that needs an asset specified, stick error string in
		s = "NO_ASSET_SPECIFIED";
	}

	if (s && s[0])
	{
		strcat(assetName, s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//check modelAnim
	s = Common_GetValForKey(b, "modelAnim");
	if (s && s[0])
	{
		assetName[0] = '@';
		assetName[1] = 0;
		strcat(assetName, s);
		g_sharedFn->Common_ServerString(assetName);
	}

	//rclip functionality
	s = Common_GetValForKey(b, "clipModel");
	if (!s)
	{
		s = "*sphere";
	}
	if (s[0] != '*')
	{ //if it's a custom clip model, it needs to be cached
		const char *clipAnim = Common_GetValForKey(b, "clipAnim");
		g_sharedFn->Coll_CacheModel(s, clipAnim);
	}
}

gameObject_t *LServ_ObjectFromName(const char *objName, float *pos, float *ang, const objArgs_t *args, int numArgs)
{
	const char *s;
	char assetName[2048];
	BYTE *b = Common_GetEntryForObject(objName);
	int i;
	if (!b)
	{ //no entry for this object then
		return NULL;
	}

	gameObject_t *obj = LServ_CreateObject();
	if (!obj)
	{ //oh no
		return NULL;
	}

	obj->objData = b;
	obj->spawnArgs = args;
	obj->numSpawnArgs = numArgs;

	Util_ParseObjType(Common_GetValForKey(b, "type"), &obj->net.type);

	Util_ParseObjSpawn(Common_GetValForKey(b, "customSpawn"), obj);

	obj->think = ObjGeneral_Think;
	obj->thinkTime = 0;

	obj->net.pos[0] = pos[0];
	obj->net.pos[1] = pos[1];
	obj->net.pos[2] = pos[2];

	obj->net.ang[0] = ang[0];
	obj->net.ang[1] = ang[1];
	obj->net.ang[2] = ang[2];

	Util_ParseVector(Common_GetValForKey(b, "spawnMins"), obj->spawnMins);
	Util_ParseVector(Common_GetValForKey(b, "spawnMaxs"), obj->spawnMaxs);

	Util_ParseVector(Common_GetValForKey(b, "mins"), obj->net.mins);
	Util_ParseVector(Common_GetValForKey(b, "maxs"), obj->net.maxs);
	obj->radius = Util_RadiusFromBounds(obj->net.mins, obj->net.maxs, 1.0f);
	Util_ParseVector(Common_GetValForKey(b, "modelScale"), obj->net.modelScale);

	Util_ParseInt(Common_GetValForKey(b, "hurtable"), &obj->hurtable);
	Util_ParseInt(Common_GetValForKey(b, "health"), &obj->health);

	Util_ParseInt(Common_GetValForKey(b, "solid"), &obj->net.solid);

	Util_ParseInt(Common_GetValForKey(b, "renderfx"), &obj->net.renderEffects);

	s = Common_GetValForKey(b, "customTex");
	if (s && s[0])
	{ //make sure the client caches this texture
		sprintf(assetName, "^%s", s);
		g_sharedFn->Common_ServerString(assetName);
	}

	obj->net.entNameIndex = g_sharedFn->Common_ServerString(objName);

	//base on the type we need a different prepended char so the client
	//knows what to cache based on string name alone
	i = 0;
	if (obj->net.type == OBJ_TYPE_SPRITE || obj->net.type == OBJ_TYPE_PROJECTILE)
	{
		assetName[i++] = '#';
	}
	else if (obj->net.type == OBJ_TYPE_MODEL)
	{
		assetName[i++] = '&';
	}
	else if (obj->net.type == OBJ_TYPE_BEAM)
	{
		assetName[i++] = '^';
	}
	assetName[i] = 0;
	s = Common_GetValForKey(b, "assetName");
	if ((!s || !s[0]) && assetName[0])
	{ //if it's a type that needs an asset specified, stick error string in
		s = "NO_ASSET_SPECIFIED";
	}

	if (s && s[0])
	{
		strcat(assetName, s);
		obj->net.strIndex = g_sharedFn->Common_ServerString(assetName);
	}

	//check modelAnim
	s = Common_GetValForKey(b, "modelAnim");
	if (s && s[0])
	{
		assetName[0] = '@';
		assetName[1] = 0;
		strcat(assetName, s);
		obj->net.strIndexB = g_sharedFn->Common_ServerString(assetName);
	}
	else
	{
		obj->net.strIndexB = 0;
	}

	s = Common_GetValForKey(b, "lightBlocking");
	if (s && s[0] && atoi(s) == 1)
	{ //blocks static light, take this to mean it is static
		obj->net.staticIndex = obj->net.index;
	}

	if (obj->spawnArgs && obj->numSpawnArgs > 0)
	{
		for (int i = 0; i < obj->numSpawnArgs; i++)
		{
			const objArgs_t *arg = obj->spawnArgs+i;
			if (!_stricmp(arg->key, "objMins"))
			{ //mins override
				Util_ParseVector(arg->val, obj->spawnMins);
				Math_VecCopy(obj->spawnMins, obj->net.mins);
			}
			else if (!_stricmp(arg->key, "objMaxs"))
			{ //maxs override
				Util_ParseVector(arg->val, obj->spawnMaxs);
				Math_VecCopy(obj->spawnMaxs, obj->net.maxs);
			}
		}
	}

	//rclip functionality
	if (obj->net.solid)
	{
		//update nxactor
		s = Common_GetValForKey(b, "clipModel");
		if (s && s[0])
		{
            obj->rcColModel = g_sharedFn->Coll_RegisterModelInstance(s);
			if (obj->rcColModel)
			{
				const char *clipRadStr = Common_GetValForKey(b, "clipRadius");
				obj->rcColModel->radius = (clipRadStr && clipRadStr[0]) ? (float)atof(clipRadStr) : Math_Max2(obj->net.maxs[0], obj->net.maxs[1]);
				s = Common_GetValForKey(b, "clipBox");
				if (s && atoi(s))
				{
					obj->rcColModel->clipFlags = CLIPFL_BOXMOVE;
				}
				Math_VecCopy(obj->net.mins, obj->rcColModel->mins);
				Math_VecCopy(obj->net.maxs, obj->rcColModel->maxs);
				Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
				Math_VecCopy(obj->net.ang, obj->rcColModel->ang);
				Math_VecCopy(obj->net.modelScale, obj->rcColModel->modelScale);
				obj->rcColModel->gameOwner = obj->net.index;
				obj->rcColModel->solid = obj->net.solid;
				obj->rcColModel->frame = obj->net.frame;
				obj->rcColModel->blendFrame = obj->net.frame;
				const char *clipAnim = Common_GetValForKey(b, "clipAnim");
				g_sharedFn->Coll_UpdateModel(obj->rcColModel, clipAnim);
			}
		}
	}

	if (obj->customSpawn)
	{
		obj->customSpawn(obj, b, args, numArgs);
	}

	return obj;
}

//time update outside of frame when necessary
void LServ_UpdateTime(unsigned long time)
{
	g_curTime = (serverTime_t)time;
}

//the engine will call this function for each map object.
//it's our responsibility to create an object in the game structure.
void LServ_InitMapObj(const char *objName, float *pos, float *ang, const objArgs_t *args, int numArgs)
{
	LServ_ObjectFromName(objName, pos, ang, args, numArgs);
}

//perform any tasks that should be performed immediately after map spawn
void LServ_PostMapSpawn(void)
{
	//set up target name pointers
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->spawnArgs && obj->numSpawnArgs > 0)
		{
			for (int j = 0; j < obj->numSpawnArgs; j++)
			{
				const objArgs_t *arg = obj->spawnArgs+j;
				if (!_stricmp(arg->key, "targname"))
				{
					obj->targetName = arg->val;
					break;
				}
			}
		}
	}

	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		//target objects to each other
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->inuse && obj->spawnArgs && obj->numSpawnArgs > 0)
		{
			for (int j = 0; j < obj->numSpawnArgs; j++)
			{
				const objArgs_t *arg = obj->spawnArgs+j;
				if (!_stricmp(arg->key, "camstart"))
				{ //camera's target
					gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
					if (cam->inuse)
					{
                        //cam->target = obj;
						cam->net.pos[0] = obj->net.pos[0];
						cam->net.pos[1] = obj->net.pos[1];
						cam->net.pos[2] = obj->net.pos[2];
						cam->camTrackPos[0] = obj->net.pos[0];
						cam->camTrackPos[1] = obj->net.pos[1];
						cam->camTrackPos[2] = obj->net.pos[2];
					}
				}
				else if (!_stricmp(arg->key, "targobj"))
				{ //targeting something else
					obj->target = Util_FindTargetObject(arg->val);
				}
			}
		}
	}

	ObjVisBlock_EstablishVis();

	//call post-spawns
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (obj->postSpawn)
		{
			obj->postSpawn(obj);
		}
	}
}

//physics impact callback (for novodex)
void LServ_PhysicsImpact(int objNum, int otherNum, const float *planeDir, const float *point)
{
}

//get information about the game object
void LServ_ObjectInfo(void **baseAddr, int *size, int *num, int *netSize, int **activeObjPtr)
{
	*baseAddr = &g_gameObjects[0];
	*size = sizeof(gameObject_t);
	*num = MAX_GAME_OBJECTS;
	*netSize = sizeof(gameObjNet_t);
	*activeObjPtr = &g_gameObjectSlots;
}

//savegame from a client
void LServ_ClientSaveGame(int clientIndex, BYTE *saveData, int saveSize)
{
	if (saveSize < sizeof(saveHeader_t))
	{
		Util_StatusMessage("Client sent a malformed save header.");
		return;
	}
	saveHeader_t *hdr = (saveHeader_t *)saveData;
	if (hdr->version != SAVE_GAME_VERSION)
	{
		Util_StatusMessage("Client sent a bad savegame version (%i).", hdr->version);
		return;
	}
	saveSize -= sizeof(saveHeader_t);
	if (saveSize != hdr->dataSize)
	{
		Util_StatusMessage("Client sent corrupted save data (%i != %i).", saveSize, hdr->dataSize);
		return;
	}

	cntStream_t *st = Stream_Alloc(hdr+1, hdr->dataSize);

	ObjPlayer_ReadGameData(&g_gameObjects[clientIndex], st);
	for (int i = 1; i < MAX_NET_CLIENTS; i++)
	{
		ObjPlayer_ReadGameData(NULL, st);
	}
	gameObject_t *clObj = &g_gameObjects[clientIndex];
	int c = Stream_ReadInt(st);
	for (int i = 0; i < c; i++)
	{
		char gvName[MAX_GVAR_LEN];
		char gvVal[MAX_GVAR_LEN];
		Stream_ReadString(st, gvName, MAX_GVAR_LEN);
		Stream_ReadString(st, gvVal, MAX_GVAR_LEN);
		if (!stricmp(gvName, "useddevcmd") && atoi(gvVal))
		{
			Util_StatusMessage("Connecting client is using a developer save.");
			if (clObj->plObj)
			{
				clObj->plObj->clCheater = true;
			}
		}
		else if (!stricmp(gvName, "cdt_usingmod") && atoi(gvVal))
		{
			Util_StatusMessage("Connecting client is using a mod save.");
			if (clObj->plObj)
			{
				clObj->plObj->clCheater = true;
			}
		}
	}

	Stream_Free(st);
}

//new input from a client
void LServ_ClientInput(int clientNum, BYTE *buttons, BYTE *userData, int userDataSize)
{ //it would be better to cache these and run them all in a single server frame for smoother input.
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (!pl->plObj)
	{
		return;
	}
	if (userDataSize == sizeof(int))
	{
		int eq = *((int *)userData);
		if (eq >= 0 && eq < MAX_PLAYER_INVENTORY_ITEMS)
		{
			pl->plObj->desiredItem = eq;
		}
	}
	memcpy(pl->plObj->clButtonsBackBuffer[pl->plObj->numButtonsBuffered], pl->plObj->clButtons, sizeof(pl->plObj->clButtons));
	if (pl->plObj->numButtonsBuffered < MAX_BUTTON_BUFFER-1)
	{
		pl->plObj->numButtonsBuffered++;
	}
	memcpy(pl->plObj->clButtons, buttons, sizeof(pl->plObj->clButtons));
}

//the client's view angles changed (in respect to the player object)
void LServ_ClientInputAngle(int clientNum, float *angles)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (!pl->plObj)
	{
		return;
	}
	memcpy(pl->plObj->clAngles, angles, sizeof(pl->plObj->clAngles));
	if (!pl->plObj->hasLastClAngles)
	{
		Math_VecCopy(pl->plObj->clAngles, pl->plObj->lastClAngles);
		pl->plObj->hasLastClAngles = true;
	}
}

//the client's analog controller movement
void LServ_ClientInputAnalog(int clientNum, WORD *analogs)
{
	gameObject_t *pl = &g_gameObjects[clientNum];
	if (!pl->plObj)
	{
		return;
	}
	memcpy(pl->plObj->clAnalog, analogs, sizeof(pl->plObj->clAnalog));
	pl->plObj->hasClAnalog = true;
}

//music was restarted/changed
void LServ_MusicalReset(const char *musicName)
{
	if (!g_musical)
	{
		return;
	}
	g_musicReset = g_curTime + 500;
	strcpy(g_musicResetNameNext, musicName);
}

//fill up a stream with persistent game data
void LServ_WriteGameStream(cntStream_t *st)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		ObjPlayer_WriteGameData(&g_gameObjects[i], st);
	}
	GVar_WriteGVars(st);
}

//restore game data from stream
void LServ_ReadGameStream(cntStream_t *st)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		ObjPlayer_ReadGameData(&g_gameObjects[i], st);
	}
	GVar_ReadGVars(st);

	//make ai weak (only in terms of logic) until after chocobo wonderland becomes accessable
	g_ai.weakAI = !GVar_GetInt("lshub_gotchoco");
}

//save game
void LServ_SaveGame(const char *saveName, const char *modName)
{
	int fh = g_sharedFn->FileSys_OpenFile(saveName, _O_WRONLY|_O_BINARY|_O_CREAT, _S_IWRITE);
	if (fh == -1)
	{
		return;
	}

	if (modName && modName[0])
	{
		GVar_SetInt("cdt_usingmod", 1);
	}

	cntStream_t *st = Stream_Alloc(NULL, 8192);
	LServ_WriteGameStream(st);

	//write out values now in case we somehow die before changing maps
	g_sharedFn->Common_SaveGameValues(Stream_Buffer(st), Stream_Size(st));

	saveHeader_t hdr;
	memset(&hdr, 0, sizeof(hdr));
	hdr.version = SAVE_GAME_VERSION;
	hdr.dataSize = Stream_Size(st);
	hdr.timeSec = GVar_GetInt("playsec");
	strcpy(hdr.map, g_mapName);
	gameObject_t *pl = &g_gameObjects[0];
	if (!pl->inuse || !pl->plObj)
	{
		strcpy(hdr.desc, "Unknown");
	}
	else
	{
		sprintf(hdr.desc, "\nHP*(1 1 1 1):*(-1 -1 -1 1) %i*(1 1 1 1)/*(-1 -1 -1 1)%i\n"
			"STR*(1 1 1 1)/*(-1 -1 -1 1)DEF*(1 1 1 1)/*(-1 -1 -1 1)DEX*(1 1 1 1)/*(-1 -1 -1 1)LUCK*(1 1 1 1): "
			"*(-1 -1 -1 1)%i*(1 1 1 1)/*(-1 -1 -1 1)%i*(1 1 1 1)/*(-1 -1 -1 1)%i*(1 1 1 1)/*(-1 -1 -1 1)%i\n"
			"Mako*(1 1 1 1):*(-1 -1 -1 1) %i\n",
			pl->health, pl->plObj->plData.maxHealth,
			pl->plObj->plData.statStr, pl->plObj->plData.statDef, pl->plObj->plData.statDex,
			pl->plObj->plData.statLuck, pl->plObj->makoStash);
	}
	g_sharedFn->FileSys_Write(fh, &hdr, sizeof(hdr));
	g_sharedFn->FileSys_Write(fh, Stream_Buffer(st), hdr.dataSize);
	Stream_Free(st);
	g_sharedFn->FileSys_CloseFile(fh);
}

//load game
void LServ_LoadGame(const char *saveName)
{
	int fh = g_sharedFn->FileSys_OpenFile(saveName, _O_RDONLY|_O_BINARY, _S_IREAD);
	if (fh == -1)
	{
		return;
	}

	saveHeader_t hdr;
	g_sharedFn->FileSys_Read(fh, &hdr, sizeof(hdr));
	if (hdr.version != SAVE_GAME_VERSION)
	{
		Util_StatusMessage("Bad savegame version %i.", hdr.version);
		return;
	}

	unsigned char *data = (unsigned char *)g_sharedFn->Common_RCMalloc(hdr.dataSize);
	g_sharedFn->FileSys_Read(fh, data, hdr.dataSize);
	cntStream_t *st = Stream_Alloc(data, hdr.dataSize);
	LServ_ReadGameStream(st);

	g_sharedFn->Common_SaveGameValues(Stream_Buffer(st), Stream_Size(st));

	Stream_Free(st);
	g_sharedFn->Common_RCFree(data);
}

//module init
int LServ_Initialize(sharedSVFunctions_t *sharedFunc, const char *mapName, void *val, int valSize, bool queryOnly, bool multiplayer)
{
	strcpy(g_mapName, mapName);
	g_musical = false;
	g_musicStrIndex = -1;
	g_musicalRunningAvg = 0.0f;
	g_musicalLastFrame = 0.0f;
	g_musicalSpawnTime = 0;
	g_musicalRunningCount = 1;
	g_musicReset = 0;
	g_musicHighScore = 0;
	g_musicHighScorePrev = 0;
	g_musicResetName[0] = 0;
	g_musicResetNameNext[0] = 0;

	g_sharedFn = sharedFunc;
	g_sharedFn->LServ_RunFrame = LServ_RunFrame;
	g_sharedFn->LServ_ObjectInfo = LServ_ObjectInfo;
	g_sharedFn->LServ_ClientSaveGame = LServ_ClientSaveGame;
	g_sharedFn->LServ_ClientInput = LServ_ClientInput;
	g_sharedFn->LServ_ClientInputAngle = LServ_ClientInputAngle;
	g_sharedFn->LServ_ClientInputAnalog = LServ_ClientInputAnalog;
	g_sharedFn->LServ_InitPlayerObj = LServ_InitPlayerObj;
	g_sharedFn->LServ_FreePlayerObj = LServ_FreePlayerObj;
	g_sharedFn->LServ_UpdateTime = LServ_UpdateTime;
	g_sharedFn->LServ_InitMapObj = LServ_InitMapObj;
	g_sharedFn->LServ_PostMapSpawn = LServ_PostMapSpawn;
	g_sharedFn->LServ_ChangeMusic = LServ_ChangeMusic;
	g_sharedFn->LServ_DevCommand = LServ_DevCommand;
	g_sharedFn->LServ_ServerEvent = LServ_ServerEvent;
	g_sharedFn->LServ_MusicalReset = LServ_MusicalReset;
	g_sharedFn->LServ_PhysicsImpact = LServ_PhysicsImpact;
	g_sharedFn->LServ_SaveGame = LServ_SaveGame;
	g_sharedFn->LServ_LoadGame = LServ_LoadGame;

	g_runningMultiplayer = multiplayer;

	if (!queryOnly)
	{ //don't bother loading stuff if this is only a query
		sharedInterface_t si;
		si.Parse_InitParser = g_sharedFn->Parse_InitParser;
		si.Parse_InitParserFromFile = g_sharedFn->Parse_InitParserFromFile;
		si.Parse_FreeParser = g_sharedFn->Parse_FreeParser;
		si.Parse_EnableInclude = g_sharedFn->Parse_EnableInclude;
		si.Parse_GetNextToken = g_sharedFn->Parse_GetNextToken;
		Shared_LoadUserItems(&si);

		g_creationCount = 0;
		LServ_InitGlobalMapObj();
		LServ_GlobalResources();
		RScr_Init();
		GVar_Init();
		g_ai.weakAI = true;
		if (val)
		{
			cntStream_t *st = Stream_Alloc(val, valSize);
			LServ_ReadGameStream(st);
			Stream_Free(st);
		}

		//don't preserve this gvar across map changes
		GVar_SetInt("intriggersequence", 0);

		//parse the general script file now
		RScr_ParseScriptFile("assets/rscript/general.rsc");
	}

	return 1;
}

//module shutdown
void LServ_Shutdown(void)
{
	gameObject_t *pl = &g_gameObjects[0];
	if (!pl->inuse || pl->health > 0)
	{ //don't write out values if the player is dead (let the values from last time they were alive be restored instead)
		cntStream_t *st = Stream_Alloc(NULL, 8192);
		LServ_WriteGameStream(st);
		g_sharedFn->Common_SaveGameValues(Stream_Buffer(st), Stream_Size(st));
		Stream_Free(st);
	}

	RScr_Shutdown();
	GVar_Shutdown();

	int i = 0;
	while (i < MAX_GAME_OBJECTS)
	{
		if (g_gameObjects[i].inuse)
		{
			LServ_FreeObject(&g_gameObjects[i]);
		}
		i++;
	}
	g_gameObjectSlots = 0;
}
