/*
=============================================================================
Module Information
------------------
Name:			gvar.cpp
Author:			Rich Whitehouse
Description:	game variable tracking
=============================================================================
*/

#include "main.h"

typedef struct gvar_s
{
	char		val[MAX_GVAR_LEN];
} gvarVal_t;

static strHash_t *g_gvHash = NULL;
static cntArray_t *g_gvVals = NULL;

//init
void GVar_Init(void)
{
	g_gvHash = SHash_Alloc(32768);
	g_gvVals = Array_Alloc(sizeof(gvarVal_t), 512);
	Array_SetGrowth(g_gvVals, true);
}

//shutdown
void GVar_Shutdown(void)
{
	if (g_gvHash)
	{
		SHash_Free(g_gvHash);
		g_gvHash = NULL;
	}
	if (g_gvVals)
	{
		Array_Free(g_gvVals);
		g_gvVals = NULL;
	}
}

//list
void GVar_ListGVars(void)
{
	int c = SHash_GetCount(g_gvHash);
	Util_StatusMessage("Gvar list:");
	for (int i = 0; i < c; i++)
	{
		char *varName = SHash_String(g_gvHash, i);
		Util_StatusMessage("'%s' is '%s'.", varName, GVar_GetValue(varName));
	}
	Util_StatusMessage("End list.");
}

//internal get
static char *GVar_GetValueInternal(char *varName)
{
	int index = SHash_Index(g_gvHash, varName);
	gvarVal_t *v = (gvarVal_t *)Array_GetElementGrow(g_gvVals, index);
	return v->val;
}

//write vars
void GVar_WriteGVars(cntStream_t *st)
{
	int c = SHash_GetCount(g_gvHash);
	Stream_WriteInt(st, c);
	for (int i = 0; i < c; i++)
	{
		char *varName = SHash_String(g_gvHash, i);
		Stream_WriteString(st, varName);
		Stream_WriteString(st, GVar_GetValueInternal(varName));
	}
}

//read vars
void GVar_ReadGVars(cntStream_t *st)
{
	int c = Stream_ReadInt(st);
	for (int i = 0; i < c; i++)
	{
		char gvName[MAX_GVAR_LEN];
		char gvVal[MAX_GVAR_LEN];
		Stream_ReadString(st, gvName, MAX_GVAR_LEN);
		Stream_ReadString(st, gvVal, MAX_GVAR_LEN);
		GVar_SetValue(gvName, gvVal);
	}
}

//get
char *GVar_GetValue(char *varName)
{
	static char str[256][MAX_GVAR_LEN];
	static int c = 0;

	char *s = GVar_GetValueInternal(varName);
	char *d = str[c];
	strcpy(d, s);
	c = (c+1)&255;
	return d;
}

//set
void GVar_SetValue(char *varName, char *value)
{
	char *s = GVar_GetValueInternal(varName);
	assert(strlen(value) < MAX_GVAR_LEN);
	strcpy(s, value);
}

//convenience
int GVar_GetInt(char *varName)
{
	return atoi(GVar_GetValueInternal(varName));
}
void GVar_SetInt(char *varName, int value)
{
	char str[MAX_GVAR_LEN];
	sprintf(str, "%i", value);
	GVar_SetValue(varName, str);
}

//convenience
float GVar_GetFloat(char *varName)
{
	return (float)atof(GVar_GetValueInternal(varName));
}
void GVar_SetFloat(char *varName, float value)
{
	char str[MAX_GVAR_LEN];
	sprintf(str, "%f", value);
	GVar_SetValue(varName, str);
}
