/*
=============================================================================
Module Information
------------------
Name:			obj_grandhorn.cpp
Author:			Rich Whitehouse
Description:	server logic object: grand horn (aborted)
=============================================================================
*/

#include "main.h"
#include "ai.h"

//just to make things a bit more readable
#define debounceLungeTime		debounce2
#define debounceLungeDuration	debounce3
#define debounceFastLunge		debounce4
#define debounceFastLook		debounce5

typedef enum
{
	HORNANIM_WINDUP1 = NUM_HUMAN_ANIMS,
	HORNANIM_WINDUP2,
	HORNANIM_SWING1,
	HORNANIM_ROUNDSWING,
	HORNANIM_LUNGE,
	HORNANIM_LUNGE2,
	HORNANIM_LUNGE_END,
	HORNANIM_DODGE_R,
	HORNANIM_DODGE_L,
	HORNANIM_BLOCK,
	HORNANIM_GETUPFAST_BACK,
	HORNANIM_GETUPFAST_FRONT,
	NUM_HORN_ANIMS
} hornAnims_e;

static gameAnim_t g_hornAnims[NUM_HORN_ANIMS] =
{
	{0, 23, 68.0f, true},			//HUMANIM_IDLE
	{142, 145, 168.0f, true},		//HUMANIM_WALK
	{142, 145, 118.0f, true},		//HUMANIM_RUNSLOW
	{142, 145, 68.0f, true},		//HUMANIM_RUN
	{141, 143, 68.0f, false},		//HUMANIM_JUMP
	{143, 143, 68.0f, false},		//HUMANIM_FALL
	{144, 145, 68.0f, false},		//HUMANIM_LAND
	{24, 31, 68.0f, false},			//HUMANIM_PAIN_HIGH1
	{146, 153, 68.0f, false},		//HUMANIM_PAIN_HIGH2
	{154, 161, 68.0f, false},		//HUMANIM_PAIN_LOW1
	{154, 161, 68.0f, false},		//HUMANIM_PAIN_LOW2
	{163, 167, 68.0f, false},		//HUMANIM_PAIN_AIR
	{162, 167, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{181, 183, 118.0f, false},		//HUMANIM_GETUP_BACK
	{184, 187, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{172, 172, 68.0f, false},		//HUMANIM_FLYBACK
	{172, 172, 68.0f, false},		//HUMANIM_FLYBACK2
	{173, 176, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{177, 180, 68.0f, false},		//HUMANIM_FALL_LAND
	{168, 171, 68.0f, false},		//HUMANIM_POPUP_LAND
	{163, 163, 1568.0f, false},		//HUMANIM_DEATH
	{188, 194, 68.0f, false},		//HORNANIM_WINDUP1
	{192, 194, 68.0f, false},		//HORNANIM_WINDUP2
	{195, 199, 68.0f, false},		//HORNANIM_SWING1
	{201, 210, 118.0f, false},		//HORNANIM_ROUNDSWING
	{211, 215, 168.0f, false},		//HORNANIM_LUNGE
	{211, 215, 68.0f, false},		//HORNANIM_LUNGE2
	{216, 220, 118.0f, false},		//HORNANIM_LUNGE_END
	{221, 222, 68.0f, false},		//HORNANIM_DODGE_R
	{223, 224, 68.0f, false},		//HORNANIM_DODGE_L
	{225, 228, 118.0f, false},		//HORNANIM_BLOCK
	{181, 182, 68.0f, false},		//HORNANIM_GETUPFAST_BACK
	{185, 186, 68.0f, false}		//HORNANIM_GETUPFAST_FRONT
};

#define NUM_HORN_DAMAGE_BONES		3
#define HORN_DMGBONE_FIST_LR		2
static damageBone_t g_hornDamageBones[NUM_HORN_DAMAGE_BONES] =
{
	{"b21", "b20", {0.0f, 0.0f, 0.0f}, 128.0f},			//DMGBONE_FIST_R
	{"b7", "b6", {0.0f, 0.0f, 0.0f}, 128.0f},			//DMGBONE_FIST_L
	{"b21", "b7", {0.0f, 0.0f, 0.0f}, 128.0f}			//HORN_DMGBONE_FIST_LR
};

//returns damage bone or -1 if not in attack
int ObjHorn_InAttack(gameObject_t *obj)
{
	if (obj->curAnim == HORNANIM_SWING1)
	{
		if (obj->net.frame >= 195 && obj->net.frame <= 198)
		{
			return DMGBONE_FIST_R;
		}
	}
	else if (obj->curAnim == HORNANIM_ROUNDSWING)
	{
		if (obj->net.frame >= 203 && obj->net.frame <= 209)
		{
			return DMGBONE_FIST_R;
		}
	}
	else if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2)
	{
		if (obj->net.frame >= 213 && obj->net.frame <= 215)
		{
			return HORN_DMGBONE_FIST_LR;
		}
	}
	else if (obj->curAnim == HORNANIM_LUNGE_END)
	{
		if (obj->net.frame >= 216 && obj->net.frame <= 217)
		{
			return HORN_DMGBONE_FIST_LR;
		}
	}
	else if (obj->curAnim == HORNANIM_BLOCK)
	{
		if (obj->net.frame == 227)
		{
			return HORN_DMGBONE_FIST_LR;
		}
	}


	return -1;
}

//occupied?
bool ObjHorn_CanAct(gameObject_t *obj)
{
	if (!obj->onGround)
	{
		return false;
	}
	if (obj->curAnim >= HORNANIM_WINDUP1)
	{
		return false;
	}
	if (AI_NonMovingAnim(obj))
	{
		return false;
	}
	if (obj->aiObj->noMoveTime >= g_curTime)
	{
		return false;
	}
	return true;
}

//can i dodge?
bool ObjHorn_CanDodge(gameObject_t *obj)
{
	if (obj->generalFlag >= 0)
	{
		return false;
	}
	if (!ObjHorn_CanAct(obj))
	{
		return false;
	}

	return true;
}

//dodge
bool ObjHorn_DodgeEnemy(gameObject_t *obj, gameObject_t *enemy)
{
	float d[3];
	Math_VecSub(obj->net.pos, enemy->net.pos, d);
	if (Math_VecLen(d) > 1200.0f)
	{
		return false;
	}

	float fwd[3];
	Math_AngleVectors(obj->net.ang, 0, fwd, 0);
	float nv[3];
	Math_VecCopy(enemy->net.vel, nv);
	Math_VecNorm(nv);
	float dp = Math_DotProduct(fwd, nv) + 0.5f;
	if (dp <= 0.0f || dp >= 1.0f)
	{
		return false;
	}

	if (dp < 0.5f)
	{
		AI_StartAnim(obj, HORNANIM_DODGE_R, true);
	}
	else
	{
		AI_StartAnim(obj, HORNANIM_DODGE_L, true);
	}
	ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
	return true;
}

//pick which lunge to do
int ObjHorn_PickLungeAnim(gameObject_t *obj)
{
	if (obj->generalFlag <= -1)
	{
		return HORNANIM_LUNGE2;
	}
	return HORNANIM_LUNGE;
}

//think
void ObjHorn_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	AI_StandardGoals(obj, timeMod);
	AI_GenericThink(obj, timeMod);
	AI_GetToGoal(obj, timeMod);

	if (obj->generalFlag <= -1 && (obj->curAnim == HUMANIM_GETUP_BACK || obj->curAnim == HUMANIM_GETUP_FRONT))
	{
		int anim = (obj->curAnim == HUMANIM_GETUP_BACK) ? HORNANIM_GETUPFAST_BACK : HORNANIM_GETUPFAST_FRONT;
		AI_StartAnim(obj, HORNANIM_GETUPFAST_BACK, true);
		if (obj->onGround)
		{
			if (obj->aiObj->enemy)
			{
				float d[3];
				Math_VecSub(obj->net.pos, obj->aiObj->enemy->net.pos, d);
				d[2] = 0.0f;
				Math_VecNorm(d);
				obj->net.vel[0] = d[0]*800.0f;
				obj->net.vel[1] = d[1]*800.0f;
			}
			obj->net.vel[2] = 1000.0f;
		}
		ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
	}

	if (obj->debounceFastLook >= g_curTime)
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 45.0f);
	}

	float f;
	float fwd[3];
	if (obj->net.frame >= 204 && obj->net.frame <= 207)
	{ //round swing
		f = 200.0f;
		Math_AngleVectors(obj->net.ang, fwd, 0, 0);
		obj->net.vel[0] += fwd[0]*f;
		obj->net.vel[1] += fwd[1]*f;
		obj->net.vel[2] += fwd[2]*f;
	}
	else if (obj->net.frame >= 214 && obj->net.frame <= 215)
	{ //lunge
		f = (obj->debounceFastLunge >= g_curTime) ? 325.0f : 300.0f;
		if (obj->generalFlag <= -1)
		{
			f *= 1.25f;
		}
		Math_AngleVectors(obj->net.ang, fwd, 0, 0);
		obj->net.vel[0] += fwd[0]*f;
		obj->net.vel[1] += fwd[1]*f;
		obj->net.vel[2] += fwd[2]*f;
	}
	else if (obj->curAnim == HORNANIM_DODGE_R)
	{
		obj->aiObj->noMoveTime = g_curTime+100;
		f = 800.0f;
		Math_AngleVectors(obj->net.ang, 0, fwd, 0);
		obj->net.vel[0] += fwd[0]*f;
		obj->net.vel[1] += fwd[1]*f;
		obj->net.vel[2] += fwd[2]*f;
	}
	else if (obj->curAnim == HORNANIM_DODGE_L)
	{
		obj->aiObj->noMoveTime = g_curTime+100;
		f = 800.0f;
		Math_AngleVectors(obj->net.ang, 0, fwd, 0);
		obj->net.vel[0] -= fwd[0]*f;
		obj->net.vel[1] -= fwd[1]*f;
		obj->net.vel[2] -= fwd[2]*f;
	}

	dmgBoneLine_t dmgPos[NUM_HORN_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_hornDamageBones, dmgPos, NUM_HORN_DAMAGE_BONES);
	int dmgBone = ObjHorn_InAttack(obj);
	if (obj->aiObj->hasLastDmg && dmgBone >= 0)
	{
		int dmg = (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2 || obj->curAnim == HORNANIM_LUNGE_END) ? 70 : 40;
		dmg = AI_LevelDamage(obj, dmg);
		AI_RunDamageBone(obj, &g_hornDamageBones[dmgBone], &obj->aiObj->lastDmg[dmgBone],
			&dmgPos[dmgBone], dmg);
	}

	if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2 || obj->curAnim == HORNANIM_LUNGE_END)
	{
		if (obj->atkLastTime == g_curTime)
		{
			char *s[3] =
			{
				"assets/sound/cb/buscut01.wav",
				"assets/sound/cb/buscut02.wav",
				"assets/sound/cb/buscut03.wav"
			};
			ObjSound_Create(obj->net.pos, s[rand()%3], 1.0f, -1);
			if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2)
			{
				AI_StartAnim(obj, HORNANIM_LUNGE_END, true);
				obj->atkLastSeq--; //hack to prevent double lunge damage
			}
		}
		obj->atkType = ATKTYPE_KNOCKBACK2;
	}
	else
	{
		obj->atkType = ATKTYPE_NORMAL;
	}

	if (obj->curAnim == HORNANIM_DODGE_R || obj->curAnim == HORNANIM_DODGE_L)
	{
		obj->net.renderEffects &= ~FXFL_BLADETRAIL;
		obj->net.renderEffects |= FXFL_MODELTRAIL;
	}
	else if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2 || obj->curAnim == HORNANIM_LUNGE_END)
	{
		obj->atkType = ATKTYPE_KNOCKBACK;
		obj->net.renderEffects &= ~FXFL_BLADETRAIL;
		if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL;
		}
	}
	else if (obj->aiObj->hasLastDmg && dmgBone >= 0 && obj->curAnim != HORNANIM_BLOCK)
	{
		obj->net.renderEffects |= FXFL_BLADETRAIL;
		obj->net.effectLen = 196.0f;
		obj->net.strIndexC = g_sharedFn->Common_ServerString(g_hornDamageBones[dmgBone].boneName);
		obj->net.strIndexD = g_sharedFn->Common_ServerString(g_hornDamageBones[dmgBone].parentName);
	}
	else if (obj->curAnim == HORNANIM_ROUNDSWING)
	{
		obj->net.renderEffects |= FXFL_BLADETRAIL;
		obj->net.effectLen = 196.0f;
		obj->net.strIndexC = g_sharedFn->Common_ServerString(g_hornDamageBones[0].boneName);
		obj->net.strIndexD = g_sharedFn->Common_ServerString(g_hornDamageBones[0].parentName);
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_BLADETRAIL;
	}
	memcpy(obj->aiObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->aiObj->hasLastDmg = true;

	obj->aiObj->combatRange = 512.0f;
	if (!AI_NonAnglingAnim(obj) && obj->aiObj->enemy && AI_FacingEnemy(obj, obj->aiObj->enemy))
	{
		if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			ObjHorn_CanDodge(obj) &&
			ObjPlayer_InDodgeableAttack(obj->aiObj->enemy) &&
			ObjHorn_DodgeEnemy(obj, obj->aiObj->enemy))
		{ //dodge
			obj->aiObj->noMoveTime = g_curTime+100;
		}
		else if (obj->aiObj->enemy->net.index < MAX_NET_CLIENTS &&
			obj->generalFlag <= 0 &&
			ObjHorn_CanAct(obj) &&
			ObjPlayer_InVulnerableState(obj->aiObj->enemy) &&
			obj->aiObj->distToEnemy < 1200.0f)
		{ //lunge when they're vulnerable
			AI_StartAnim(obj, ObjHorn_PickLungeAnim(obj), true);
			obj->debounceLungeTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%8000));
			obj->debounceLungeDuration = g_curTime + Util_LocalDelay(obj, 800);
			obj->debounceFastLunge = g_curTime + Util_LocalDelay(obj, 1000);
			obj->debounceFastLook = g_curTime + Util_LocalDelay(obj, 300);
		}
		else
		{
			bool shouldCloseIn = AI_ShouldCloseIn(obj);
			/*
			if (obj->generalFlag <= -1 && ObjHorn_CanAct(obj))
			{ //hard guys will be vicious
				obj->aiObj->attackTime = 0;
			}
			*/
			if (obj->aiObj->attackTime < g_curTime && shouldCloseIn && ObjHorn_CanAct(obj))
			{
				obj->aiObj->combatRange = 128.0f;
				float closeAttackDist = (obj->generalFlag <= -1) ? 500.0f : 350.0f;
				if (obj->aiObj->distToEnemy < closeAttackDist)
				{
					int windAnim = (obj->generalFlag <= -1) ? HORNANIM_WINDUP2 : HORNANIM_WINDUP1;
					AI_StartAnim(obj, windAnim, true);
					if (obj->generalFlag <= -1)
					{
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(1500+rand()%2000));
					}
					else
					{
						obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					}
					obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
				}
				else if (obj->generalFlag <= 0 && obj->debounceLungeTime < g_curTime &&
					obj->aiObj->distToEnemy < 2000.0f /*&& (obj->aiObj->distToEnemy > 850.0f)*/ &&
					AI_CanReachPoint(obj, obj->aiObj->goalPos, obj->aiObj->enemy))
				{
					obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 90.0f);
					AI_StartAnim(obj, ObjHorn_PickLungeAnim(obj), true);
					obj->aiObj->attackTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%4000));
					obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 1000);
					obj->debounceLungeDuration = g_curTime + Util_LocalDelay(obj, 800);
					obj->debounceLungeTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%8000));
				}
			}
			else
			{
				float closeDist = (obj->generalFlag <= 0) ? 200.0f : 100.0f;
				float r = (shouldCloseIn) ? closeDist : 500.0f;
				obj->aiObj->combatRange = g_ai.playerCloseDist+r;
			}
		}
	}
}

//transition to the secon round swing?
static bool ObjHorn_GoIntoRoundSwing(gameObject_t *obj)
{
	if (!obj->aiObj->enemy || !obj->aiObj->enemy->inuse)
	{
		return false;
	}
	if (obj->generalFlag > 1)
	{
		return false;
	}

	if (obj->generalFlag == 1)
	{
		if (rand()%10 >= 7)
		{
			return false;
		}
	}
	else
	{
		if (rand()%20 >= 18)
		{
			return false;
		}
	}

	return true;
}

//pick which anim to be in
void ObjHorn_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim == HORNANIM_WINDUP1 || obj->curAnim == HORNANIM_WINDUP2)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
			AI_StartAnim(obj, HORNANIM_SWING1, true);
		}
	}
	else if (obj->curAnim == HORNANIM_SWING1 && obj->net.frame == 196 && ObjHorn_GoIntoRoundSwing(obj))
	{
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 90.0f);
		AI_StartAnim(obj, HORNANIM_ROUNDSWING, true);
	}
	else if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->debounceLungeDuration < g_curTime)
			{
				AI_StartAnim(obj, HORNANIM_LUNGE_END, true);
				obj->atkLastSeq--; //hack to prevent double lunge damage
			}
		}
	}
	else if ((obj->curAnim == HORNANIM_DODGE_R || obj->curAnim == HORNANIM_DODGE_L) &&
		obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		if (obj->aiObj->enemy && obj->aiObj->distToEnemy < 2000.0f && rand()%10 <= 5)
		{ //lunge counter
			AI_StartAnim(obj, ObjHorn_PickLungeAnim(obj), true);
			obj->debounceLungeTime = g_curTime + Util_LocalDelay(obj, (serverTime_t)(2000+rand()%8000));
			obj->debounceLungeDuration = g_curTime + Util_LocalDelay(obj, 800);
			obj->debounceFastLunge = g_curTime + Util_LocalDelay(obj, 1000);
			obj->debounceFastLook = g_curTime + Util_LocalDelay(obj, 300);
		}
		else
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjHorn_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HORNANIM_ROUNDSWING:
		if (obj->net.frame == 205)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
		}
		obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 300);
		break;
	case HORNANIM_LUNGE:
	case HORNANIM_LUNGE2:
	case HORNANIM_LUNGE_END:
		if (obj->net.frame == 213)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
			ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
		}
		obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 300);
		break;
	case HORNANIM_BLOCK:
		if (obj->net.frame == 227)
		{
			ObjSound_CreateFromIndex(obj->net.pos, g_soundPunch[rand()%NUM_SOUNDS_PUNCH], 1.0f, -1);
		}
		break;
	default:
		break;
	}
}

//touch
void ObjHorn_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	if (!col || col->containsSolid)
	{
		return;
	}

	if (obj->curAnim == HORNANIM_LUNGE || obj->curAnim == HORNANIM_LUNGE2)
	{
		if (other && other->hurtable && Util_ValidEnemies(obj, other))
		{
			if (Util_SequenceDamage(obj, other))
			{
				float ang[3];
				Math_VecToAngles((float *)col->endNormal, ang);
				Util_SetSequenceDamage(obj, other);
				float impactPos[3];
				Math_VecCopy((float *)col->endPos, impactPos);
				impactPos[2] += obj->net.mins[2]*0.5f;
				ObjSound_CreateFromIndex(impactPos, g_soundHitLight[rand()%NUM_SOUNDS_HITLIGHT], 1.0f, -1);
				ObjParticles_Create("melee/impacten", impactPos, ang, -1);
				collObj_t nCol = *col;
				Math_VecCopy(impactPos, nCol.endPos);
				int dmg = AI_LevelDamage(obj, 100);
				Util_DamageObject(obj, other, dmg, &nCol);
			}
		}
		float groundPl[3] = {0.0f, 0.0f, 1.0f};
		float groundDot = 0.4f;
		float gd = Math_DotProduct(groundPl, col->endNormal);
		if (gd <= groundDot)
		{
			AI_StartAnim(obj, HORNANIM_LUNGE_END, true);
			obj->atkLastSeq--; //hack to prevent double lunge damage
		}
	}

	AI_GenericTouch(obj, other, col);
}

//skilled guys can block
bool ObjHorn_AttackBlock(gameObject_t *obj, gameObject_t *hurter, int dmg, collObj_t *col,
							float *impactPos, float *impactAng)
{
	if (obj->generalFlag >= 1)
	{
		return false;
	}

	if (hurter->atkType != ATKTYPE_NORMAL && hurter->atkType != ATKTYPE_BULLET)
	{
		return false;
	}

	if (obj->generalFlag >= 0)
	{
		if (rand()%10 < 5)
		{
			return false;
		}
	}

	/*
	if (rand()%10 < 2)
	{
		return false;
	}
	*/

	if (!ObjHorn_CanAct(obj))
	{
		if (obj->curAnim != HUMANIM_GETUP_BACK &&
			obj->curAnim != HUMANIM_GETUP_FRONT &&
			obj->curAnim != HORNANIM_BLOCK)
		{ //make an exception to allow blocking while getting up and in a block
			if (!(obj->generalFlag <= -1 &&
				obj->curAnim >= HUMANIM_PAIN_HIGH1 && obj->curAnim <= HUMANIM_PAIN_LOW2 &&
				rand()%10 < 2))
			{ //small chance of blocking in pain if highly skilled
				return false;
			}
		}
	}

	if (obj->generalFlag <= -1 &&
		obj->curAnim != HORNANIM_BLOCK &&
		!ObjHorn_CanAct(obj))
	{ //always block in getups/pain
		obj->net.ang[YAW] = Math_BlendAngleLinear(obj->net.ang[YAW], obj->aiObj->lookYaw, 360.0f);
	}
	else
	{
		float fwd[3];
		Math_AngleVectors(obj->net.ang, fwd, 0, 0);
		float d[3];
		Math_VecSub(impactPos, obj->net.pos, d);
		d[2] = 0.0f;
		Math_VecNorm(d);
		float dp = Math_DotProduct(fwd, d);
		if (dp <= 0.0f)
		{
			return false;
		}
	}

	ObjParticles_Create("melee/impact", impactPos, impactAng, -1);
	ObjSound_Create(impactPos, "assets/sound/cb/hitmetal.wav", 1.0f, -1);
	AI_StartAnim(obj, HORNANIM_BLOCK, true);
	obj->aiObj->noMoveTime = g_curTime + Util_LocalDelay(obj, 300);
	return true;
}

//spawn
void ObjHorn_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	static int dmgSounds[3];
	dmgSounds[0] = g_sharedFn->Common_ServerString("$assets/sound/cb/slash01.wav");
	dmgSounds[1] = g_sharedFn->Common_ServerString("$assets/sound/cb/slash02.wav");
	dmgSounds[2] = g_sharedFn->Common_ServerString("$assets/sound/cb/slash03.wav");

	AI_GenericSpawn(obj, b, args, numArgs);

	obj->touch = ObjHorn_Touch;
	obj->animframetick = ObjHorn_FrameTick;

	const char *elvalue = Common_GetValForKey(b, "elvalue");
	if (elvalue && elvalue[0])
	{
		obj->generalFlag = atoi(elvalue);
	}

	if (obj->generalFlag == -1)
	{
		obj->aiObj->makoValue = 3;
		obj->net.aiDescIndex = AIDESC_HORN2;
	}
	else
	{
		obj->aiObj->makoValue = 2;
		obj->net.aiDescIndex = AIDESC_HORN1;
	}

	obj->aiObj->dropChances[INVITEM_POTION] = 128;
	if (obj->generalFlag == 0)
	{
		obj->aiObj->dropChances[INVITEM_ETHER] = 20;
	}
	else if (obj->generalFlag == -1)
	{
		obj->aiObj->dropChances[INVITEM_ETHER] = 30;
		obj->aiObj->dropChances[INVITEM_PHOENIXDOWN] = 10;
	}
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->dmgEffect = "melee/impactslash";
	obj->aiObj->dmgSounds = dmgSounds;
	obj->aiObj->numDmgSounds = 3;
	if (obj->generalFlag == -1)
	{
		obj->aiObj->moveSpeed = 350.0f;
	}
	else if (obj->generalFlag == 0)
	{
		obj->aiObj->moveSpeed = 270.0f;
	}
	else
	{
		obj->aiObj->moveSpeed = 200.0f;
	}
	obj->animhandler = ObjHorn_PickAnim;
	obj->animTable = g_hornAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjHorn_Think;
	obj->attackblock = ObjHorn_AttackBlock;
}
