/*
=============================================================================
Module Information
------------------
Name:			obj_choco.cpp
Author:			Rich Whitehouse
Description:	server logic object: chocobo
=============================================================================
*/

#include "main.h"
#include "ai.h"

typedef enum
{
	CHOCOANIM_CIN_IDLE_1 = NUM_HUMAN_ANIMS,
	NUM_CHOCO_ANIMS
} chocoAnims_e;

gameObject_t *g_chocoSpots[MAX_GAME_OBJECTS];
static int g_numChocoSpots = 0;
//roam spot spawn
void ObjChocoSpot_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	obj->localFlags |= LFL_NONET;
	g_chocoSpots[g_numChocoSpots] = obj;
	g_numChocoSpots++;
}

static gameAnim_t g_chocoAnims[NUM_CHOCO_ANIMS] =
{
	{1, 6, 518.0f, true},			//HUMANIM_IDLE
	{7, 12, 168.0f, true},			//HUMANIM_WALK
	{7, 12, 118.0f, true},			//HUMANIM_RUNSLOW
	{7, 12, 68.0f, true},			//HUMANIM_RUN
	{0, 2, 68.0f, false},			//HUMANIM_JUMP
	{0, 2, 68.0f, false},			//HUMANIM_FALL
	{0, 2, 68.0f, false},			//HUMANIM_LAND
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_HIGH1
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_HIGH2
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_LOW1
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_LOW2
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_AIR
	{223, 231, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{37, 40, 118.0f, false},		//HUMANIM_GETUP_BACK
	{37, 40, 118.0f, false},		//HUMANIM_GETUP_FRONT
	{211, 211, 68.0f, false},		//HUMANIM_FLYBACK
	{211, 211, 68.0f, false},		//HUMANIM_FLYBACK2
	{223, 231, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{19, 22, 68.0f, false},			//HUMANIM_FALL_LAND
	{19, 22, 68.0f, false},			//HUMANIM_POPUP_LAND
	{0, 19, 1568.0f, false},		//HUMANIM_DEATH
	{1, 6, 68.0f, true},			//CHOCOANIM_CIN_IDLE_1
};

static scriptableAnim_t g_chocoScriptAnims[] =
{
	DEF_SCRIPT_ANIM(HUMANIM_IDLE),
	{0, NULL}
};

//think
void ObjChoco_Think(gameObject_t *obj, float timeMod)
{
	if (!AI_ShouldThink(obj))
	{
		return;
	}

	if (obj->debounce2 > g_curTime)
	{ //breedings
		if (obj->debounce3 < g_curTime)
		{
			float ang[3] = {0.0f, 0.0f, 0.0f};
			float p[3];
			Math_VecCopy(obj->net.pos, p);
			p[2] += 400.0f;
			ObjParticles_Create("lovelove", p, ang, -1);
			obj->debounce3 = g_curTime + 600;
		}
	}

	if (obj->curAnim >= CHOCOANIM_CIN_IDLE_1)
	{
		Util_AnimateObject(obj, timeMod);
		return;
	}

	if (obj->curAnim == HUMANIM_WALK ||
		obj->curAnim == HUMANIM_RUNSLOW ||
		obj->curAnim == HUMANIM_RUN)
	{
		if (obj->net.frame >= 1 && obj->net.frame <= 6)
		{ //hack because i'm too lazy to make more choco idle frames
			obj->animTime = 0;
		}
	}

	if (!obj->aiObj->enemy || !obj->aiObj->enemy->inuse)
	{
		for (int j = 0; j < MAX_NET_CLIENTS; j++)
		{
			gameObject_t *other = &g_gameObjects[j];
			if (other->inuse && other->plObj)
			{
				float d[3];
				Math_VecSub(other->net.pos, obj->net.pos, d);
				if (other->health > 0 && !GVar_GetInt("intriggersequence") && Math_VecLen(d) < 512.0f)
				{ //don't trigger from jump while script says it's busy
					other->plObj->jumpUseObj = obj->net.index;
					other->plObj->canJumpUse = g_curTime + Util_LocalDelay(other, 500);
					if (other->plObj->jumpUseTime >= g_curTime)
					{
						ObjSound_Create(obj->net.pos, "assets/sound/other/c_wark.wav", 1.0f, -1);
						obj->aiObj->enemy = other;
						ObjPlayer_MountRide(other, obj);
						other->plObj->jumpUseTime = 0;
						other->plObj->canJumpUse = 0;
						if (obj->str1)
						{
							Util_RunScript("obj_chocscr", obj->str1);
						}
						break;
					}
				}
			}
		}
	}

	gameObject_t *rider = NULL;
	obj->aiObj->fly = false;
	if (obj->aiObj->enemy)
	{ //got a rider
		if (!obj->aiObj->enemy->inuse || !obj->aiObj->enemy->plObj)
		{
			obj->aiObj->enemy = NULL;
		}
		else
		{
			if (obj->generalFlag == 4)
			{
				obj->onGround = true;
				obj->aiObj->fly = true;
			}
			rider = obj->aiObj->enemy;
			float moveVec[3];
			float move, rightMove;
			rider->onGround = obj->onGround;
			ObjPlayer_GetMoveVec(rider, &g_gameObjects[CAM_TRACK_NUM], timeMod, moveVec, move, rightMove,
				obj->aiObj->fly);

			if (obj->aiObj->fly)
			{
				obj->net.vel[0] += moveVec[0]*0.5f;
				obj->net.vel[1] += moveVec[1]*0.5f;
				if (obj->net.pos[2] < 34000.0f)
				{
					obj->net.vel[2] += moveVec[2]*0.5f;
				}
			}
			else
			{
				obj->net.vel[0] += moveVec[0]*1.5f;
				obj->net.vel[1] += moveVec[1]*1.5f;
				obj->net.vel[2] += moveVec[2]*1.5f;
			}

			float moveAngTol = obj->onGround ? 1.0f : 100.0f;
			float velCheck[3] = {obj->net.vel[0], obj->net.vel[1], 0.0f};
			float velCheckLen = Math_VecLen(velCheck);
			if ((move || rightMove) && velCheckLen > moveAngTol)
			{
				float velBasedTurn = (velCheckLen > 64.0f) ? 0.5f : 0.05f;
				float angleBlendScale = velBasedTurn;
				float velAng[3];
				Math_VecToAngles(obj->net.vel, velAng);
				obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], velAng[YAW], timeMod*angleBlendScale);
				obj->aiObj->lookYaw = obj->net.ang[YAW];
			}
		}
		AI_GenericThink(obj, timeMod);
	}
	else
	{
		if (!obj->aiObj->enemy && obj->debounce < g_curTime)
		{
			if (obj->debounce2 > g_curTime)
			{ //love love time
				int *objIdx = (int *)_alloca(sizeof(int)*g_gameObjectSlots);
				int numObj = 0;
				for (int i = MAX_NET_CLIENTS; i < g_gameObjectSlots; i++)
				{
					gameObject_t *other = &g_gameObjects[i];
					if (!other->inuse || !other->aiObj ||
						other->net.aiDescIndex != AIDESC_CHOCO ||
						obj == other || other->debounce2 < g_curTime)
					{
						continue;
					}
					objIdx[numObj] = i;
					numObj++;
				}
				if (numObj > 0)
				{
					obj->aiObj->goalObj = &g_gameObjects[objIdx[rand()%numObj]];
					obj->debounce = g_curTime + 1000 + rand()%30000;
				}
			}
			else if (g_numChocoSpots > 1)
			{
				obj->aiObj->goalObj = g_chocoSpots[rand()%g_numChocoSpots];
				obj->debounce = g_curTime + 1000 + rand()%30000;
			}
		}

		float lookPos[3];
		bool changeLook = false;
		if (obj->debounce2 > g_curTime)
		{
			obj->aiObj->goalRange = 700.0f;
		}
		else
		{
			obj->aiObj->goalRange = obj->aiObj->moveSpeed;
		}
		if (Math_VecLen(obj->net.vel) > 32.0f)
		{ //look in the direction i'm headed
			Math_VecAdd(obj->net.pos, obj->net.vel, lookPos);
			changeLook = true;
		}
		if (obj->aiObj->goalObj && !obj->aiObj->goalObj->inuse)
		{
			obj->aiObj->goalObj = NULL;
		}
		if (obj->aiObj->goalObj)
		{
			Math_VecCopy(obj->aiObj->goalObj->net.pos, obj->aiObj->goalPos);
		}
		if (changeLook)
		{
			float a[3];
			Math_VecSub(lookPos, obj->net.pos, a);
			Math_VecToAngles(a, a);
			obj->aiObj->lookYaw = a[YAW];
			obj->aiObj->lookPitch = (obj->aiObj->fly) ? a[PITCH] : 0.0f;
		}

		AI_GenericThink(obj, timeMod);
		AI_GetToGoal(obj, timeMod);
	}

	if (rider)
	{
		obj->canPush = true;
		gameObject_t *ride = obj;
		if (ride->net.frame >= 7 && ride->net.frame <= 12)
		{ //match the frame and lerp time directly
			int anim = ObjPlayer_GetChocoAnim(1);
			AI_StartAnim(rider, anim, false);
			gameAnim_t *matchAnim = rider->animTable+anim;
			rider->net.frame = matchAnim->startFrame+(ride->net.frame-7);
			rider->net.lerpDuration = ride->net.lerpDuration;
			rider->animStartTime = ride->animStartTime;
			rider->animTime = ride->animTime;
		}
		else if (ride->net.frame <= 6)
		{
			int anim = ObjPlayer_GetChocoAnim(0);
			AI_StartAnim(rider, anim, false);
			gameAnim_t *matchAnim = rider->animTable+anim;
			rider->net.frame = matchAnim->startFrame;
			rider->net.lerpDuration = ride->net.lerpDuration;
			rider->animStartTime = ride->animStartTime;
			rider->animTime = ride->animTime;
		}
		else
		{
			int anim = ObjPlayer_GetChocoAnim(0);
			AI_StartAnim(rider, anim, false);
		}

		Math_VecCopy(obj->net.pos, rider->net.pos);
		Math_VecCopy(obj->net.ang, rider->net.ang);
		rider->net.vel[0] = 0.0f;
		rider->net.vel[1] = 0.0f;
		rider->net.vel[2] = 0.0f;
		LServ_UpdateRClip(rider);

		if (rider->plObj->jumpUseTime >= g_curTime)
		{
			if (ObjPlayer_UnmountRide(rider, false))
			{
				if (rand()%10 < 5)
				{
					ObjSound_Create(obj->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
				}
				else
				{
					ObjSound_Create(obj->net.pos, "assets/sound/other/c_kweh2.wav", 1.0f, -1);
				}
				obj->aiObj->enemy = NULL;
				if (obj->str2)
				{
					Util_RunScript("obj_chocscr", obj->str2);
				}
			}
			rider->plObj->jumpUseTime = 0;
		}
	}
	else
	{
		obj->canPush = false;
	}
}

//pick which anim to be in
void ObjChoco_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= CHOCOANIM_CIN_IDLE_1)
	{ //script will break out
		return;
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//frame tick
void ObjChoco_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	switch (obj->curAnim)
	{
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 8 || obj->net.frame == 11) && obj->onGround)
		{
			if (obj->aiObj->fly)
			{
				ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[rand()%NUM_SOUNDS_DEEPBLOW], 1.0f, -1);
			}
			else
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/chocostep.wav", 1.0f, -1);
			}
		}
		break;
	default:
		break;
	}
}

//spawn
void ObjChoco_Spawn(gameObject_t *obj, BYTE *b, const objArgs_t *args, int numArgs)
{
	g_sharedFn->Common_ServerString("&&lovelove");
	g_sharedFn->Common_ServerString("$assets/music/Kweh.ogg");
	g_sharedFn->Common_ServerString("$assets/sound/other/c_kweh1.wav");
	g_sharedFn->Common_ServerString("$assets/sound/other/c_kweh2.wav");
	g_sharedFn->Common_ServerString("$assets/sound/other/c_wark.wav");
	g_sharedFn->Common_ServerString("$assets/sound/cb/chocostep.wav");

	const char *chocoType = Common_GetValForKey(b, "chocotype");
	obj->generalFlag = 0;
	if (chocoType && chocoType[0])
	{
		obj->generalFlag = atoi(chocoType);
	}

	int gvVal = 1;

	if (obj->generalFlag == 1)
	{
		gvVal = GVar_GetInt("choco_cblue");
		obj->net.renderEffects2 |= FXFL2_CHOCOBLUE;
		g_sharedFn->Common_ServerString("^assets/choco/blue_body");
		g_sharedFn->Common_ServerString("^assets/choco/blue_feather");
	}
	else if (obj->generalFlag == 2)
	{
		gvVal = GVar_GetInt("choco_cgreen");
		obj->net.renderEffects2 |= FXFL2_CHOCOGREEN;
		g_sharedFn->Common_ServerString("^assets/choco/green_body");
		g_sharedFn->Common_ServerString("^assets/choco/green_feather");
	}
	else if (obj->generalFlag == 3)
	{
		gvVal = GVar_GetInt("choco_cred");
		obj->net.renderEffects2 |= FXFL2_CHOCORED;
		g_sharedFn->Common_ServerString("^assets/choco/red_body");
		g_sharedFn->Common_ServerString("^assets/choco/red_feather");
	}
	else if (obj->generalFlag == 4)
	{
		gvVal = GVar_GetInt("choco_cpink");
		obj->net.renderEffects2 |= FXFL2_CHOCOPINK;
		g_sharedFn->Common_ServerString("^assets/choco/pink_body");
		g_sharedFn->Common_ServerString("^assets/choco/pink_feather");
	}

	if (!gvVal)
	{ //not yet available
		obj->think = ObjGeneral_RemoveThink;
		obj->thinkTime = g_curTime;
		return;
	}

	for (int i = 0; i < numArgs; i++)
	{
		const objArgs_t *arg = args+i;
		if (!stricmp(arg->key, "onmount"))
		{
			obj->str1 = arg->val;
		}
		else if (!stricmp(arg->key, "onunmount"))
		{
			obj->str2 = arg->val;
		}
	}

	AI_GenericSpawn(obj, b, args, numArgs);
	obj->death = AI_ImmortalDeath;
	obj->localFlags &= ~LFL_ENEMY;
	obj->groundHugger = true;

	obj->debounce = g_curTime + 5000 + rand()%10000;

	obj->net.aiDescIndex = AIDESC_CHOCO;
	obj->aiObj->maxHealth = obj->health;

	obj->aiObj->moveSpeed = 80.0f;
	obj->animhandler = ObjChoco_PickAnim;
	obj->animframetick = ObjChoco_FrameTick;
	obj->animTable = g_chocoAnims;
	obj->scriptAnims = g_chocoScriptAnims;
	obj->curAnim = HUMANIM_IDLE;
	obj->think = ObjChoco_Think;
}

//show the love, baby
static void ObjChoco_ShowLove(gameObject_t **chocos, int numChocos, int typeA, int typeB)
{
	for (int i = 0; i < numChocos; i++)
	{
		gameObject_t *choco = chocos[i];
		if (typeA == -2)
		{
			choco->debounce = 0;
			choco->debounce2 = g_curTime + 10000;
			ObjSound_Create(choco->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
			ObjSound_Create(choco->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
		}
		else if (choco->generalFlag == typeA)
		{
			choco->debounce = 0;
			choco->debounce2 = g_curTime + 10000;
			typeA = -1;
			ObjSound_Create(choco->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
			ObjSound_Create(choco->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
		}
		else if (choco->generalFlag == typeB)
		{
			choco->debounce = 0;
			choco->debounce2 = g_curTime + 10000;
			typeB = -1;
			ObjSound_Create(choco->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
			ObjSound_Create(choco->net.pos, "assets/sound/other/c_kweh1.wav", 1.0f, -1);
		}
	}
}

//feed the chocobos
bool ObjChoco_FeedChocos(gameObject_t *feeder, const invItemDef_t *greens)
{
	gameObject_t *fedChocos[16];
	int numFedChocos = 0;
	int gvBlue = GVar_GetInt("choco_cblue");
	int gvGreen = GVar_GetInt("choco_cgreen");
	int gvRed = GVar_GetInt("choco_cred");
	int gvPink = GVar_GetInt("choco_cpink");
	int numYellow = 0;
	int numBlue = 0;
	int numGreen = 0;
	int numRed = 0;
	int numPink = 0;
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		gameObject_t *obj = &g_gameObjects[i];
		if (!obj->inuse || !obj->aiObj ||
			obj->net.aiDescIndex != AIDESC_CHOCO)
		{
			continue;
		}
		fedChocos[numFedChocos] = obj;
		numFedChocos++;
		if (numFedChocos >= 16)
		{
			break;
		}
		switch (obj->generalFlag)
		{
		case 0:
			numYellow++;
			break;
		case 1:
			numBlue++;
			break;
		case 2:
			numGreen++;
			break;
		case 3:
			numRed++;
			break;
		case 4:
			numPink++;
			break;
		}
	}

	if (numFedChocos <= 0)
	{
		return false;
	}

	if (numYellow >= 2 && !gvBlue && greens->itemValue >= 1)
	{ //blue baby choco!
		if (rand()%10 < 7)
		{
			ObjChoco_ShowLove(fedChocos, numFedChocos, 0, 0);
			GVar_SetInt("choco_cblue", 1);
			Util_StatusMessage("That meal seems to have started something.");
		}
		else
		{
			Util_StatusMessage("They enjoyed the meal, but she looks tired.");
		}
	}
	else if (numBlue >= 1 && numYellow >= 1 && !gvGreen && greens->itemValue >= 2)
	{ //green baby choco!
		if (rand()%10 < 6)
		{
			ObjChoco_ShowLove(fedChocos, numFedChocos, 0, 1);
			GVar_SetInt("choco_cgreen", 1);
			Util_StatusMessage("It looks like she's in the mood.");
		}
		else
		{
			Util_StatusMessage("No luck this time. What's in this stuff?");
		}
	}
	else if (numBlue >= 1 && numGreen >= 1 && !gvRed && greens->itemValue >= 3)
	{ //red baby choco!
		if (rand()%10 < 8)
		{
			ObjChoco_ShowLove(fedChocos, numFedChocos, 1, 2);
			GVar_SetInt("choco_cred", 1);
			Util_StatusMessage("There's no better way to end a fine meal.");
		}
		else
		{
			Util_StatusMessage("It'll take more than just one expensive date.");
		}
	}
	else if (numRed >= 1 && numBlue >= 1 && numGreen >= 1 && numYellow >= 1 &&
		!gvPink && greens->itemValue >= 4)
	{ //pink baby choco~~~
		ObjChoco_ShowLove(fedChocos, numFedChocos, -2, -2);
		GVar_SetInt("choco_cpink", 1);
		Util_StatusMessage("They're all acting very frisky!");
	}
	else
	{
		Util_StatusMessage("Mmm, tasty.");
	}

	for (int i = 0; i < numFedChocos; i++)
	{
		gameObject_t *obj = fedChocos[i];
		ObjSound_Create(obj->net.pos, "assets/sound/other/c_wark.wav", 1.0f, -1);
		if (rand()%5 > greens->itemValue)
		{ //item chance failed
			continue;
		}

		float p[3];
		Math_VecCopy(obj->net.pos, p);
		p[2] += 400.0f;
		gameObject_t *item = LServ_ObjectFromName("obj_item_drop", p, obj->net.ang, NULL, 0);
		switch (obj->generalFlag)
		{
		case 0:
			if (rand()%10 < 5)
			{
				item->makoCount = INVITEM_LOCOWEED;
			}
			else
			{
				item->makoCount = INVITEM_HIPOTION;
			}
			break;
		case 1:
			if (rand()%10 < 7)
			{
				item->makoCount = INVITEM_HIPOTION;
			}
			else
			{
				item->makoCount = INVITEM_STARDUST;
			}
			break;
		case 2:
			if (rand()%10 < 8)
			{
				item->makoCount = INVITEM_CHOCONECTAR;
			}
			else
			{
				item->makoCount = INVITEM_XPOTION;
			}
			break;
		case 3:
			if (rand()%10 < 9)
			{
				item->makoCount = INVITEM_HELLFANG;
			}
			else
			{
				item->makoCount = INVITEM_PHOENIXDOWN;
			}
			break;
		case 4:
			if (rand()%10 < 5)
			{
				item->makoCount = INVITEM_PHOENIXDOWN;
			}
			else
			{
				item->makoCount = INVITEM_ELIXER;
			}
			break;
		default:
			item->makoCount = INVITEM_POTION;
			break;
		}
	}

	return true;
}
