/*
=============================================================================
Module Information
------------------
Name:			obj_player.cpp
Author:			Rich Whitehouse
Description:	server logic object: player
=============================================================================
*/

#include "main.h"
#include "ai.h"

float g_initialFPSPos[3] = {0.0f, 0.0f, 0.0f};
float g_initialFPSAng[3] = {0.0f, 210.0f, 0.0f}; //hax

bool ObjPlayer_InChain(gameObject_t *obj);
void ObjPlayer_ProgressChain(gameObject_t *obj);

typedef enum
{
	PLANIM_WALLJUMP = NUM_HUMAN_ANIMS,
	PLANIM_DIVE_BOUNCE,
	PLANIM_GETUP_FLIP,
	PLANIM_DODGE_LEFT,
	PLANIM_DODGE_RIGHT,
	PLANIM_BACKFLIP,
	PLANIM_LUNGE,
	PLANIM_TACKLE,
	PLANIM_STRAFE_LEFT,
	PLANIM_STRAFE_RIGHT,
	PLANIM_TARG_FWD,
	PLANIM_TARG_BACK,
	PLANIM_PUNCH_R1,
	PLANIM_PUNCH_R2,
	PLANIM_C1_COMBO1,
	PLANIM_C1_COMBO2,
	PLANIM_C1_COMBO3,
	PLANIM_PUNCH_L1,
	PLANIM_PUNCH_UPPERCUT,
	PLANIM_PUNCH_UPPERCUTSHORT,
	PLANIM_PUNCH_AIRCRUSH,
	PLANIM_PUNCH_AIRCRUSHSUPER,
	PLANIM_PUNCH_AIRCRUSH_LAND,
	PLANIM_PUNCH_AIRCRUSH_LANDSUPER,
	PLANIM_DIVE_LAND,
	PLANIM_SWORD_1,
	PLANIM_SWORD_2,
	PLANIM_SWORD_3,
	PLANIM_SWORD_4,
	PLANIM_SWORD_AIR,
	PLANIM_SWORD_AIR_LAND,
	PLANIM_ITEM_USE,
	PLANIM_KICK_R1,
	PLANIM_KICK_R2,
	PLANIM_KICK_R3,
	PLANIM_KICK_R4,
	PLANIM_KICK_BACKFLIP,
	PLANIM_KICK_AIR_FLIP,
	PLANIM_KICK_DIVE,
	PLANIM_KICK_AIR1,
	PLANIM_KICK_AIR2,
	PLANIM_KICK_AIR3,
	PLANIM_KICK_AIR4,
	PLANIM_LUNGE_FINISH,
	PLANIM_TACKLE_FINISH,
	PLANIM_AIR_RECOVER,
	PLANIM_CHOCO_SIT,
	PLANIM_CHOCO_RIDE,
	PLANIM_RISE,
	PLANIM_SEPH_STABBED,
	PLANIM_SEPH_BEATDOWN,
	PLANIM_SEPH_AERITHPUNCH,
	PLANIM_SEPH_BEATDOWNLOOP,
	PLANIM_DEATH,
	PLANIM_DEATH_LAND,
	PLANIM_DEATH_MP,
	PLANIM_DEATH_LAND_MP,
	PLANIM_CIN_IDLE_1,
	PLANIM_CIN_COMBATREADY,
	PLANIM_CIN_SITTING,
	PLANIM_CIN_GETUP,
	PLANIM_CIN_CALLOUT,
	PLANIM_CIN_CALLDONE,
	PLANIM_CIN_TALKAERITH,
	PLANIM_CIN_KNEELDOWN,
	NUM_PLAYER_ANIMS
} playerAnims_e;

static scriptableAnim_t g_plScriptAnims[] =
{
	DEF_SCRIPT_ANIM(HUMANIM_IDLE),
	DEF_SCRIPT_ANIM(PLANIM_PUNCH_AIRCRUSH_LAND),
	DEF_SCRIPT_ANIM(PLANIM_CIN_IDLE_1),
	DEF_SCRIPT_ANIM(PLANIM_CIN_COMBATREADY),
	DEF_SCRIPT_ANIM(PLANIM_CIN_SITTING),
	DEF_SCRIPT_ANIM(PLANIM_CIN_GETUP),
	DEF_SCRIPT_ANIM(PLANIM_CIN_CALLOUT),
	DEF_SCRIPT_ANIM(PLANIM_CIN_CALLDONE),
	DEF_SCRIPT_ANIM(PLANIM_CIN_TALKAERITH),
	DEF_SCRIPT_ANIM(PLANIM_CIN_KNEELDOWN),
	{0, NULL}
};

static gameAnim_t g_playerAnims[NUM_PLAYER_ANIMS] =
{
	{0, 13, 68.0f, true},			//HUMANIM_IDLE
	{446, 451, 168.0f, true},		//HUMANIM_WALK
	{446, 451, 118.0f, true},		//HUMANIM_RUNSLOW
	{446, 451, 88.0f, true},		//HUMANIM_RUN
	{85, 86, 68.0f, false},			//HUMANIM_JUMP
	{452, 452, 68.0f, false},		//HUMANIM_FALL
	{88, 89, 68.0f, false},			//HUMANIM_LAND
	{190, 197, 68.0f, false},		//HUMANIM_PAIN_HIGH1
	{553, 559, 68.0f, false},		//HUMANIM_PAIN_HIGH2
	{210, 217, 68.0f, false},		//HUMANIM_PAIN_LOW1
	{210, 217, 68.0f, false},		//HUMANIM_PAIN_LOW2
	{575, 577, 68.0f, false},		//HUMANIM_PAIN_AIR
	{572, 577, 68.0f, false},		//HUMANIM_PAIN_POPUP
	{589, 594, 168.0f, false},		//HUMANIM_GETUP_BACK
	{595, 599, 168.0f, false},		//HUMANIM_GETUP_FRONT
	{581, 581, 68.0f, false},		//HUMANIM_FLYBACK
	{581, 581, 68.0f, false},		//HUMANIM_FLYBACK2
	{582, 585, 68.0f, false},		//HUMANIM_HIT_FALLFORWARD
	{586, 588, 68.0f, false},		//HUMANIM_FALL_LAND
	{578, 580, 68.0f, false},		//HUMANIM_POPUP_LAND
	{0, 5, 68.0f, false},			//HUMANIM_DEATH
	{457, 462, 68.0f, false},		//PLANIM_WALLJUMP
	{511, 515, 68.0f, false},		//PLANIM_DIVE_BOUNCE
	{511, 515, 68.0f, false},		//PLANIM_GETUP_FLIP
	{495, 500, 68.0f, false},		//PLANIM_DODGE_LEFT
	{501, 506, 68.0f, false},		//PLANIM_DODGE_RIGHT
	{507, 515, 68.0f, false},		//PLANIM_BACKFLIP
	{516, 517, 68.0f, false},		//PLANIM_LUNGE
	{619, 619, 68.0f, false},		//PLANIM_TACKLE
	{477, 481, 118.0f, true},		//PLANIM_STRAFE_LEFT
	{482, 486, 118.0f, true},		//PLANIM_STRAFE_RIGHT
	{487, 490, 118.0f, true},		//PLANIM_TARG_FWD
	{491, 494, 118.0f, true},		//PLANIM_TARG_BACK
	{336, 341, 68.0f, false},		//PLANIM_PUNCH_R1
	{134, 143, 68.0f, false},		//PLANIM_PUNCH_R2
	{138, 143, 68.0f, false},		//PLANIM_C1_COMBO1
	{394, 401, 68.0f, false},		//PLANIM_C1_COMBO2
	{371, 375, 68.0f, false},		//PLANIM_C1_COMBO3
	{370, 375, 68.0f, false},		//PLANIM_PUNCH_L1
	{467, 476, 68.0f, false},		//PLANIM_PUNCH_UPPERCUT
	{468, 472, 118.0f, false},		//PLANIM_PUNCH_UPPERCUTSHORT
	{560, 563, 118.0f, false},		//PLANIM_PUNCH_AIRCRUSH
	{560, 563, 118.0f, false},		//PLANIM_PUNCH_AIRCRUSHSUPER
	{564, 571, 168.0f, false},		//PLANIM_PUNCH_AIRCRUSH_LAND
	{564, 571, 118.0f, false},		//PLANIM_PUNCH_AIRCRUSH_LANDSUPER
	{568, 571, 118.0f, false},		//PLANIM_DIVE_LAND
	{684, 691, 118.0f, false},		//PLANIM_SWORD_1
	{692, 697, 98.0f, false},		//PLANIM_SWORD_2
	{697, 702, 68.0f, false},		//PLANIM_SWORD_3
	{703, 708, 118.0f, false},		//PLANIM_SWORD_4
	{709, 711, 118.0f, false},		//PLANIM_SWORD_AIR
	{711, 714, 138.0f, false},		//PLANIM_SWORD_AIR_LAND
	{120, 131, 118.0f, false},		//PLANIM_ITEM_USE
	{343, 350, 68.0f, false},		//PLANIM_KICK_R1
	{355, 367, 68.0f, false},		//PLANIM_KICK_R2
	{375, 383, 68.0f, false},		//PLANIM_KICK_R3
	{389, 401, 68.0f, false},		//PLANIM_KICK_R4
	{600, 608, 68.0f, false},		//PLANIM_KICK_BACKFLIP
	{609, 614, 68.0f, false},		//PLANIM_KICK_AIR_FLIP
	{616, 618, 68.0f, false},		//PLANIM_KICK_DIVE
	{539, 546, 68.0f, false},		//PLANIM_KICK_AIR1
	{547, 552, 68.0f, false},		//PLANIM_KICK_AIR2
	{526, 532, 68.0f, false},		//PLANIM_KICK_AIR3
	{533, 538, 68.0f, false},		//PLANIM_KICK_AIR4
	{518, 525, 68.0f, false},		//PLANIM_LUNGE_FINISH
	{384, 389, 68.0f, false},		//PLANIM_TACKLE_FINISH
	{451, 452, 218.0f, false},		//PLANIM_AIR_RECOVER
	{677, 677, 68.0f, true},		//PLANIM_CHOCO_SIT
	{678, 683, 68.0f, true},		//PLANIM_CHOCO_RIDE
	{620, 632, 68.0f, false},		//PLANIM_RISE
	{715, 724, 218.0f, false},		//PLANIM_SEPH_STABBED
	{725, 770, 118.0f, false},		//PLANIM_SEPH_BEATDOWN
	{770, 784, 168.0f, false},		//PLANIM_SEPH_AERITHPUNCH
	{762, 769, 68.0f, false},		//PLANIM_SEPH_BEATDOWNLOOP
	{218, 222, 68.0f, false},		//PLANIM_DEATH
	{223, 225, 68.0f, false},		//PLANIM_DEATH_LAND
	{218, 222, 68.0f, false},		//PLANIM_DEATH_MP
	{223, 225, 68.0f, false},		//PLANIM_DEATH_LAND_MP
	{655, 656, 1068.0f, true},		//PLANIM_CIN_IDLE_1
	{0, 13, 68.0f, true},			//PLANIM_CIN_COMBATREADY
	{657, 657, 68.0f, false},		//PLANIM_CIN_SITTING
	{658, 670, 168.0f, false},		//PLANIM_CIN_GETUP
	{671, 672, 368.0f, false},		//PLANIM_CIN_CALLOUT
	{672, 674, 268.0f, false},		//PLANIM_CIN_CALLDONE
	{675, 675, 468.0f, false},		//PLANIM_CIN_TALKAERITH
	{785, 786, 1068.0f, false}		//PLANIM_CIN_KNEELDOWN
};

typedef enum
{
	PLCHAIN_NONE,
	PLCHAIN_PUNCH_R1,
	PLCHAIN_PUNCH_R1_2,
	PLCHAIN_PUNCH_R2,
	PLCHAIN_PUNCH_L1,
	PLCHAIN_PUNCH_L1_2,
	PLCHAIN_PUNCH_UPPERCUT,
	PLCHAIN_PUNCH_UPPERCUTSHORT,
	PLCHAIN_PUNCH_AIRCRUSH,
	PLCHAIN_PUNCH_AIRCRUSHSUPER,
	PLCHAIN_SWORD_1,
	PLCHAIN_SWORD_2,
	PLCHAIN_SWORD_3,
	PLCHAIN_SWORD_4,
	PLCHAIN_SWORD_AIR,
	PLCHAIN_KICK_R1,
	PLCHAIN_KICK_R2,
	PLCHAIN_KICK_R3,
	PLCHAIN_KICK_R4,
	PLCHAIN_KICK_BACKFLIP,
	PLCHAIN_KICK_AIR_FLIP,
	PLCHAIN_KICK_DIVE,
	PLCHAIN_KICK_AIR1,
	PLCHAIN_KICK_AIR2,
	PLCHAIN_LUNGE_FINISH,
	PLCHAIN_TACKLE_FINISH,
	PLCHAIN_COMBO_1,
	PLCHAIN_COMBO_2,
	PLCHAIN_COMBO_3,
	PLCHAIN_COMBO_4,
	NUM_PLAYER_CHAINS
} playerChains_e;

static playerChain_t g_playerChains[NUM_PLAYER_CHAINS] =
{
	{HUMANIM_IDLE, 0, -2, -2, 0, -2 -2, 0, 0, 0, 0},											//PLCHAIN_NONE
	{PLANIM_PUNCH_R1, PLCHAIN_PUNCH_L1, -2, -2, 0, -2, -2, 1, 0, 0, 0},		//PLCHAIN_PUNCH_R1
	{PLANIM_PUNCH_R1, PLCHAIN_PUNCH_L1_2, -2, -2, 0, -2, -2, 1, 0, 0, 0},					//PLCHAIN_PUNCH_R1_2
	{PLANIM_PUNCH_R2, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_PUNCH_R2
	{PLANIM_PUNCH_L1, PLCHAIN_PUNCH_R1_2, PLCHAIN_PUNCH_UPPERCUT, PLCHAIN_PUNCH_R2, PLCHAIN_COMBO_1, PLCHAIN_KICK_R3, PLCHAIN_KICK_R4, 2, PLCHAIN_SWORD_2, PLCHAIN_SWORD_4, -2},		//PLCHAIN_PUNCH_L1
	{PLANIM_PUNCH_L1, PLCHAIN_PUNCH_R2, PLCHAIN_PUNCH_UPPERCUT, -2, 0, -2, PLCHAIN_KICK_R4, 2, PLCHAIN_SWORD_3, PLCHAIN_SWORD_4, -2},						//PLCHAIN_PUNCH_L1_2
	{PLANIM_PUNCH_UPPERCUT, 0, -2, -2, PLCHAIN_KICK_AIR1, -2, -2, 4, 0, 0, 0},										//PLCHAIN_PUNCH_UPPERCUT
	{PLANIM_PUNCH_UPPERCUTSHORT, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_PUNCH_UPPERCUTSHORT
	{PLANIM_PUNCH_AIRCRUSH, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_PUNCH_AIRCRUSH
	{PLANIM_PUNCH_AIRCRUSHSUPER, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_PUNCH_AIRCRUSHSUPER
	{PLANIM_SWORD_1, 0, -2, -2, 0, -2, -2, 4, PLCHAIN_SWORD_2, -2, -2},	//PLCHAIN_SWORD_1
	{PLANIM_SWORD_2, 0, -2, -2, 0, -2, -2, 1, PLCHAIN_SWORD_3, PLCHAIN_SWORD_4, -2,},	//PLCHAIN_SWORD_2
	{PLANIM_SWORD_3, 0, -2, -2, 0, -2, -2, 0, 0, -2, -2},	//PLCHAIN_SWORD_3
	{PLANIM_SWORD_4, 0, -2, -2, 0, -2, -2, 0, 0, -2, -2},	//PLCHAIN_SWORD_4
	{PLANIM_SWORD_AIR, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},	//PLCHAIN_SWORD_AIR
	{PLANIM_KICK_R1, PLCHAIN_PUNCH_L1, -2, -2, PLCHAIN_KICK_R3, -2, -2, 3, 0, 0, 0},							//PLCHAIN_KICK_R1
	{PLANIM_KICK_R2, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_KICK_R2
	{PLANIM_KICK_R3, 0, PLCHAIN_PUNCH_UPPERCUTSHORT, -2, PLCHAIN_KICK_R2, -2, PLCHAIN_KICK_R4, 3, PLCHAIN_SWORD_2, -2, -2},					//PLCHAIN_KICK_R3
	{PLANIM_KICK_R4, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_KICK_R4
	{PLANIM_KICK_BACKFLIP, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_KICK_BACKFLIP
	{PLANIM_KICK_AIR_FLIP, PLCHAIN_PUNCH_AIRCRUSHSUPER, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_KICK_AIR_FLIP
	{PLANIM_KICK_DIVE, 0, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_KICK_DIVE
	{PLANIM_KICK_AIR1, 0, -2, -2, PLCHAIN_KICK_AIR2, -2, -2, 2, 0, 0, 0},										//PLCHAIN_KICK_AIR1
	{PLANIM_KICK_AIR2, 0, -2, -2, /*PLCHAIN_KICK_AIR1*/0, -2, PLCHAIN_KICK_AIR_FLIP, 2, 0, 0, 0},										//PLCHAIN_KICK_AIR2
	{PLANIM_LUNGE_FINISH, PLCHAIN_PUNCH_R1, -2, -2, 0, -2, -2, 0, 0, 0, 0},										//PLCHAIN_LUNGE_FINISH
	{PLANIM_TACKLE_FINISH, PLCHAIN_PUNCH_L1, -2, -2, 0, -2, -2, 3, 0, 0, 0},										//PLCHAIN_TACKLE_FINISH
	{PLANIM_KICK_R3, PLCHAIN_COMBO_2, -2, -2, PLCHAIN_KICK_R2, -2, PLCHAIN_KICK_R4, 3, 0, 0, 0},					//PLCHAIN_COMBO_1
	{PLANIM_C1_COMBO1, 0, -2, -2, PLCHAIN_COMBO_3, -2, -2, 1, 0, 0, 0},					//PLCHAIN_COMBO_2
	{PLANIM_C1_COMBO2, PLCHAIN_COMBO_4, -2, -2, 0, -2, -2, 1, 0, 0, 0},					//PLCHAIN_COMBO_3
	{PLANIM_C1_COMBO3, PLCHAIN_PUNCH_R2, PLCHAIN_PUNCH_UPPERCUT, -2, 0, -2, -2, 1, PLCHAIN_SWORD_2, -2, -2}					//PLCHAIN_COMBO_4
};

#define DEFAULT_DMG_AMOUNT_SMALL	5//10
#define DEFAULT_DMG_AMOUNT_MEDIUM	20//30
#define DEFAULT_DMG_AMOUNT_HEAVY	45//55
#define DEFAULT_DMG_RADIUS	128.0f

static damageBone_t g_plDamageBones[NUM_DAMAGE_BONES] =
{
	{"b32", "b31", {0.0f, 0.0f, -80.0f}, DEFAULT_DMG_RADIUS},		//DMGBONE_FIST_R
	{"b27", "b26", {0.0f, 0.0f, -80.0f}, DEFAULT_DMG_RADIUS},		//DMGBONE_FIST_L
	{"b22", "b20", {0.0f, 0.0f, 0.0f}, DEFAULT_DMG_RADIUS},			//DMGBONE_FOOT_R
	{"b15", "b13", {0.0f, 0.0f, 0.0f}, DEFAULT_DMG_RADIUS},			//DMGBONE_FOOT_L
	{"b32", "b31", {0.0f, 0.0f, -90.0f}, 170.0f},					//DMGBONE_FIST_R_BIG
	{"", "", {0.0f, -1.0f, 0.0f}, 0.0f}								//DMGBONE_SWORD
};

typedef struct chainDamage_s
{
	int					dmgBone;
	int					startFrame;
	int					endFrame;
	int					dmg;
} chainDamage_t;

static chainDamage_t g_chainDamage[NUM_PLAYER_CHAINS] =
{
	{NULL, 0, 0, 0},													//PLCHAIN_NONE
	{DMGBONE_FIST_R, 338, 341, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_PUNCH_R1
	{DMGBONE_FIST_R, 338, 341, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_PUNCH_R1_2
	{DMGBONE_FIST_R, 138, 140, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_PUNCH_R2
	{DMGBONE_FIST_L, 371, 373, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_PUNCH_L1
	{DMGBONE_FIST_L, 371, 373, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_PUNCH_L1_2
	{DMGBONE_FIST_R, 471, 473, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_PUNCH_UPPERCUT
	{DMGBONE_FIST_R, 471, 473, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_PUNCH_UPPERCUTSHORT
	{DMGBONE_FIST_R, 563, 563, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_PUNCH_AIRCRUSH
	{DMGBONE_FIST_R, 563, 563, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_PUNCH_AIRCRUSHSUPER
	{DMGBONE_SWORD, 685, 688, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_SWORD_1
	{DMGBONE_SWORD, 693, 696, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_SWORD_2
	{DMGBONE_SWORD, 698, 699, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_SWORD_3
	{DMGBONE_SWORD, 704, 705, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_SWORD_4
	{DMGBONE_SWORD, 710, 711, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_SWORD_AIR
	{DMGBONE_FOOT_R, 345, 347, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_KICK_R1
	{DMGBONE_FOOT_R, 361, 363, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_KICK_R2
	{DMGBONE_FOOT_R, 377, 380, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_KICK_R3
	{DMGBONE_FOOT_R, 396, 398, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_KICK_R4
	{DMGBONE_FOOT_R, 601, 606, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_KICK_BACKFLIP
	{DMGBONE_FOOT_R, 612, 614, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_KICK_AIR_FLIP
	{DMGBONE_FOOT_R, 617, 618, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_KICK_DIVE
	{DMGBONE_FOOT_R, 541, 543, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_KICK_AIR1
	{DMGBONE_FOOT_L, 549, 550, DEFAULT_DMG_AMOUNT_SMALL},				//PLCHAIN_KICK_AIR2
	{DMGBONE_FIST_R_BIG, 518, 520, DEFAULT_DMG_AMOUNT_MEDIUM},			//PLCHAIN_LUNGE_FINISH
	{DMGBONE_FIST_R_BIG, 385, 387, DEFAULT_DMG_AMOUNT_MEDIUM},			//PLCHAIN_TACKLE_FINISH
	{DMGBONE_FOOT_R, 377, 380, DEFAULT_DMG_AMOUNT_MEDIUM},				//PLCHAIN_COMBO_1
	{DMGBONE_FIST_R, 138, 140, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_COMBO_2
	{DMGBONE_FOOT_R, 396, 398, DEFAULT_DMG_AMOUNT_HEAVY},				//PLCHAIN_COMBO_3
	{DMGBONE_FIST_L, 371, 373, DEFAULT_DMG_AMOUNT_MEDIUM}				//PLCHAIN_COMBO_4
};

//true if ability is present
bool ObjPlayer_Ability(gameObject_t *obj, int ability)
{
	if (obj->plObj->plData.abilities & (1<<ability))
	{
		return true;
	}
	return false;
}

//strength-modified damage
int ObjPlayer_StrMod(gameObject_t *obj, int dmg)
{
	if (!obj || !obj->inuse || obj->net.index >= MAX_NET_CLIENTS || !obj->plObj)
	{
		return dmg;
	}

	if (dmg <= 0)
	{
		return dmg;
	}

	float strf = (float)obj->plObj->plData.statStr;
	const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
	if (item && item->itemBehavior == INVITEM_BEHAVIOR_MODSTR)
	{
		strf += (float)item->itemValue;
	}
	if (dmg >= 45)
	{
		return (dmg-10)+(int)(strf*2.0f);
	}
	else if (dmg >= 20)
	{
		return (dmg-5)+(int)(strf*1.5f);
	}
	else if (dmg >= 5)
	{
		return dmg+(int)strf;
	}

	strf *= 0.15f;
	return dmg+(int)strf;
}

//defense-modified damage
int ObjPlayer_DefMod(gameObject_t *obj, int dmg, gameObject_t *attacker)
{
	if (dmg <= 0)
	{
		return dmg;
	}
	/*
	dmg -= (int)obj->plObj->plData.statDef;
	*/
	float def = (float)obj->plObj->plData.statDef;
	const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
	if (item && item->itemBehavior == INVITEM_BEHAVIOR_MODDEF)
	{
		def += (float)item->itemValue;
		if (def > 255.0f)
		{
			def = 255.0f;
		}
	}
	if (obj != attacker && attacker && attacker->inuse && attacker->plObj)
	{ //another player
		def *= 0.75f;
		if (def < 1.0f)
		{
			def = 1.0f;
		}
	}
	float f = (float)dmg * (1.0f-(def/255.0f));
	dmg = (int)f;
	if (dmg < 1)
	{
		dmg = 1;
	}
	return dmg;
}

//are they weaksauce?
bool ObjPlayer_IsWeakSauce(gameObject_t *obj)
{
	if (!obj || !obj->inuse || !obj->plObj)
	{
		return false;
	}

	if (!g_ai.weakAI)
	{
		return false;
	}

	float f = (float)obj->health / (float)obj->plObj->plData.maxHealth;
	if (f < 0.6f)
	{
		return true;
	}
	return false;
}

//more novice stuff, see if it's ok for enemy to attack
bool ObjPlayer_OkayToAttack(gameObject_t *enemy, gameObject_t *obj)
{
	if (!enemy->aiObj || enemy->aiObj->aiLevel >= 15)
	{
		return true;
	}

	if (g_ai.weakAI && ObjPlayer_IsWeakSauce(obj))
	{
		if (ObjPlayer_IsUsingItem(obj))
		{ //back off during item use
			g_ai.playerLastHitIndex = -1;
			g_ai.playerLastHitTime = g_curTime;
			return false;
		}
		if (g_ai.playerLastHitIndex != obj->net.index && (g_curTime-g_ai.playerLastHitTime) < 1000)
		{ //if it's less than a second since the player was beating up on someone else, leave them alone
			return false;
		}
	}
	return true;
}

//in seph anim?
bool ObjPlayer_InSephAnim(gameObject_t *obj)
{
	if (!obj || !obj->inuse || !obj->plObj)
	{
		return false;
	}
	if (obj->curAnim < PLANIM_SEPH_STABBED || obj->curAnim > PLANIM_SEPH_BEATDOWNLOOP)
	{
		return false;
	}
	return true;
}

//get proper anim
int ObjPlayer_GetSephAnim(int a)
{
	int anim = -1;
	switch (a)
	{
	case 0:
		anim = PLANIM_SEPH_STABBED;
		break;
	case 1:
		anim = PLANIM_SEPH_BEATDOWN;
		break;
	default:
		break;
	}
	return anim;
}

//get proper anim
int ObjPlayer_GetChocoAnim(int a)
{
	int anim = -1;
	switch (a)
	{
	case 0:
		anim = PLANIM_CHOCO_SIT;
		break;
	case 1:
		anim = PLANIM_CHOCO_RIDE;
		break;
	default:
		break;
	}
	return anim;
}

//check damage for chain attacks
void ObjPlayer_CheckChainDamage(gameObject_t *obj, float timeMod)
{
	dmgBoneLine_t dmgPos[NUM_DAMAGE_BONES];
	AI_TransformDamageBones(obj, g_plDamageBones, dmgPos, NUM_DAMAGE_BONES);
	if (obj->plObj->hasLastDmg && ObjPlayer_InChain(obj))
	{ //do actual chain damage checks
		gameAnim_t *curAnim = obj->animTable+obj->curAnim;
		chainDamage_t *cd = &g_chainDamage[obj->plObj->chainNum];
		int startFrame = (cd->startFrame == -1) ? curAnim->startFrame : cd->startFrame;
		int endFrame = (cd->endFrame == -1) ? curAnim->endFrame : cd->endFrame;
		if (obj->net.frame >= startFrame && obj->net.frame <= endFrame)
		{
			AI_RunDamageBone(obj, &g_plDamageBones[cd->dmgBone], &obj->plObj->lastDmg[cd->dmgBone],
				&dmgPos[cd->dmgBone], cd->dmg);
		}
	}

	memcpy(obj->plObj->lastDmg, dmgPos, sizeof(dmgPos));
	obj->plObj->hasLastDmg = true;
}

//put relevant stats into spare net fields
void ObjPlayer_EncodeNetData(gameObject_t *obj)
{
	assert(obj->plObj);
	assert(obj->health < (1<<12));
	assert(obj->plObj->makoCharges < (1<<4));
	//assert(obj->plObj->makoStash < (1<<17));
	assert(obj->plObj->chargeMeter < (1<<12));
	assert(obj->plObj->equippedItem < (1<<8));

	int h = (obj->health < 0) ? 0 : obj->health;
	obj->net.lives = h|(obj->plObj->chargeMeter<<12)|(obj->plObj->makoCharges<<24);
	obj->net.plAmmo = obj->plObj->makoStash;
	obj->net.spare1 = obj->plObj->equippedItem;
	if (obj->plObj->waitOnInput == BUTTON_JUMP+1)
	{
		obj->net.spare1 |= (1<<12);
	}
	if (obj->plObj->canJumpUse >= g_curTime)
	{
		obj->net.spare1 |= (1<<13);
	}
}

//update states
void ObjPlayer_UpdatePlayerStates(gameObject_t *obj)
{
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (!other->inuse || !other->plObj)
		{
			continue;
		}
		if (memcmp(&other->plObj->plData, &obj->plObj->lastPlData[i], sizeof(other->plObj->plData)) != 0)
		{ //need to send a delta off to the client
			int msgSize = sizeof(other->plObj->plData)+1024;
			WORD *msgOut = (WORD *)_alloca(msgSize);
			*msgOut = other->net.index; //first word for the event type is the desired client index
			BYTE *deltaOut = (BYTE *)(msgOut+2);
			int deltaSize = msgSize-4;

			int realDeltaSize = g_sharedFn->Net_CreateDelta(&obj->plObj->lastPlData[i], &other->plObj->plData, sizeof(other->plObj->plData),
				deltaOut, deltaSize);
			memcpy(&obj->plObj->lastPlData[i], &other->plObj->plData, sizeof(other->plObj->plData));
			assert(realDeltaSize > 0);
		
			*(msgOut+1) = realDeltaSize; //second word for the event type is the delta size

			g_sharedFn->Net_SendEventType(CLEV_PLAYERDATA, msgOut, realDeltaSize+4, obj->net.index);
		}
	}
}

//display enemy health
void ObjPlayer_HealthDisplay(gameObject_t *obj, gameObject_t *other)
{
	if (!other || !other->aiObj)
	{
		return;
	}

	if (obj->plObj->targetObj && obj->plObj->targetObj != other)
	{ //don't change the meter to non-targetted things
		return;
	}

	int *val = (int *)_alloca(sizeof(int)*3);
	val[0] = (other->health >= 0) ? other->health : 0;
	val[1] = other->aiObj->maxHealth;
	val[2] = other->net.aiDescIndex;
	g_sharedFn->Net_SendEventType(CLEV_ENEMYHEALTH, val, sizeof(int)*3, obj->net.index);
}

//get equipped item
const invItemDef_t *ObjPlayer_GetEquippedItem(gameObject_t *obj)
{
	if (obj->plObj->equippedItem < 0)
	{
		return NULL;
	}
	playerInvItem_t *plItem = &obj->plObj->plData.inventory[obj->plObj->equippedItem];
	const invItemDef_t *item = &g_invItems[plItem->itemIndex];
	if (plItem->itemQuantity <= 0)
	{
		return NULL;
	}
	return item;
}

//give item
void ObjPlayer_GiveItem(gameObject_t *obj, int itemIndex)
{
	bool gave = Shared_GiveItem(&obj->plObj->plData, itemIndex);
	if (gave)
	{ //tell the client
		g_sharedFn->Net_SendEventType(CLEV_ITEMPICKUP, &itemIndex, sizeof(itemIndex), obj->net.index);
	}
}

//use it
bool ObjPlayer_UseItem(gameObject_t *obj, const invItemDef_t *item)
{
	playerInvItem_t *plItem = &obj->plObj->plData.inventory[obj->plObj->equippedItem];
	if (plItem->itemIndex != (int)(item-g_invItems))
	{
		return false;
	}
	modelMatrix_t boneMat;
	float fxPos[3], fxAng[3] = {0.0f, 0.0f, 0.0f};
	float p[3] = {0.0f, 0.0f, -135.0f};
	if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b32", &boneMat))
	{
		Math_TransformPointByMatrix(&boneMat, p, fxPos);
	}
	else
	{
		Math_VecCopy(obj->net.pos, fxPos);
	}

	switch (item->itemBehavior)
	{
	case INVITEM_BEHAVIOR_ADDHEALTH:
		ObjSound_Create(obj->net.pos, "assets/sound/items/usepotion.wav", 1.0f, -1);
		obj->health += item->itemValue;
		if (obj->health > obj->plObj->plData.maxHealth)
		{
			obj->health = obj->plObj->plData.maxHealth;
		}
		break;
	case INVITEM_BEHAVIOR_MMTS:
		if (g_magicMode && g_magicMode != 2)
		{
			g_magicMode = 2;
			Util_RunScript("dmanscr", "choco_dollarman");
		}
		ObjSound_Create(obj->net.pos, "assets/sound/other/mmts.wav", 1.0f, -1);
		obj->health += item->itemValue;
		if (obj->health > obj->plObj->plData.maxHealth)
		{
			obj->health = obj->plObj->plData.maxHealth;
		}
		break;
	case INVITEM_BEHAVIOR_GREENS:
		if (!ObjChoco_FeedChocos(obj, item))
		{
			Util_StatusMessage("No one seems interested...");
			return false;
		}
		break;
	case INVITEM_BEHAVIOR_ADDCHARGE:
		ObjSound_Create(obj->net.pos, "assets/sound/items/usepotion.wav", 1.0f, -1);
		obj->plObj->makoCharges++;
		if (obj->plObj->makoCharges > obj->plObj->plData.maxMakoCharges)
		{
			obj->plObj->makoCharges = obj->plObj->plData.maxMakoCharges;
		}
		break;
	case INVITEM_BEHAVIOR_ADDMAKO:
		ObjSound_Create(obj->net.pos, "assets/sound/cb/lstream.wav", 1.0f, -1);
		ObjPlayer_GiveMako(obj, item->itemValue);
		break;
	case INVITEM_BEHAVIOR_FULLHEALTHCHARGE:
		ObjSound_Create(obj->net.pos, "assets/sound/items/usepotion.wav", 1.0f, -1);
		obj->plObj->makoCharges = obj->plObj->plData.maxMakoCharges;		
		obj->health = obj->plObj->plData.maxHealth;
		break;
	case INVITEM_BEHAVIOR_PHOENIX:
		if (obj->animNeverInterrupt)
		{
			return false;
		}
		obj->hurtable = 0;
		obj->health = 1;
		//ObjSound_Create(obj->net.pos, "assets/sound/cb/pldeath.wav", 1.0f, -1);
		AI_StartAnim(obj, PLANIM_RISE, false);
		obj->animNeverInterrupt = true;
		break;
	case INVITEM_BEHAVIOR_RADIUSDMG:
		Util_RadiusDamage(obj, item->itemValue, (float)item->itemValue*4.0f, fxPos);
		ObjSound_Create(fxPos, "assets/sound/cb/boom.wav", 1.0f, -1);
		ObjSound_Create(fxPos, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
		ObjSound_Create(fxPos, "assets/sound/cb/boom.wav", 1.0f, -1);
		ObjSound_Create(fxPos, "assets/sound/cb/meteorsmash.wav", 1.0f, -1);
		break;
	case INVITEM_BEHAVIOR_DRAIN:
		PlItem_DrainEnemies(obj, fxPos, 2048.0f, item->itemValue);
		break;
	case INVITEM_BEHAVIOR_VORTEX:
		PlItem_CreateVortex(obj, fxPos, 20000, (float)item->itemValue);
		break;
	case INVITEM_BEHAVIOR_QUAKE:
		PlItem_Quake(obj, fxPos, 2048.0f, item->itemValue);
		break;
	case INVITEM_BEHAVIOR_HASTEUSE:
		obj->plObj->hasteTime = g_curTime+item->itemValue;
		ObjSound_Create(obj->net.pos, "assets/sound/cin/fast.wav", 1.0f, -1);
		break;
	case INVITEM_BEHAVIOR_CONFUSE:
		PlItem_Confuse(obj, fxPos, 2048.0f, item->itemValue);
		break;
	case INVITEM_BEHAVIOR_MAKOFIRE:
		{
			if ((g_timeType != TIMETYPE_NORMAL && g_timeType != TIMETYPE_TONBERRY) || obj->plObj->fistOfFlames || obj->plObj->explosiveBlows ||
				obj->plObj->healingAura)
			{
				return false;
			}
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeflame.wav", 1.0f, -1);
			obj->plObj->fistOfFlames = g_curTime+item->itemValue;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 0.4f;
			c[2] = 0.1f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
		break;
	case INVITEM_BEHAVIOR_MAKOAURA:
		{
			if ((g_timeType != TIMETYPE_NORMAL && g_timeType != TIMETYPE_TONBERRY) || obj->plObj->fistOfFlames || obj->plObj->explosiveBlows ||
				obj->plObj->healingAura)
			{
				return false;
			}
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeaura.wav", 1.0f, -1);
			obj->plObj->explosiveBlows = g_curTime+item->itemValue;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 1.0f;
			c[2] = 0.5f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
		break;
	case INVITEM_BEHAVIOR_MAKOTIME:
		{
			if ((g_timeType != TIMETYPE_NORMAL && g_timeType != TIMETYPE_TONBERRY) || obj->plObj->fistOfFlames || obj->plObj->explosiveBlows ||
				obj->plObj->healingAura)
			{
				return false;
			}
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargetime.wav", 1.0f, -1);
			g_timeType = TIMETYPE_TIMEATTACK;
			obj->plObj->timeAttackTime = g_curTime+item->itemValue;
		}
		break;
	case INVITEM_BEHAVIOR_GUN:
		break;
	case INVITEM_BEHAVIOR_SWORD:
		break;
	case INVITEM_BEHAVIOR_SPECIAL:
		break;
	case INVITEM_BEHAVIOR_NOPAIN:
		break;
	case INVITEM_BEHAVIOR_VIEWHEALTH:
		break;
	case INVITEM_BEHAVIOR_HEALSTEP:
		break;
	case INVITEM_BEHAVIOR_FINALHEAVEN:
		break;
	case INVITEM_BEHAVIOR_FLASHLIGHT:
		break;
	case INVITEM_BEHAVIOR_MODSTR:
	case INVITEM_BEHAVIOR_MODDEF:
	case INVITEM_BEHAVIOR_MODDEX:
	case INVITEM_BEHAVIOR_MODLUCK:
		break;
	default:
		break;
	}

	if (item->useFX && item->useFX[0] && item->useFX[0] != '*')
	{
		ObjParticles_Create(item->useFX, fxPos, fxAng, -1);
	}

	if (item->itemBehavior != INVITEM_BEHAVIOR_MMTS)
	{
		plItem->itemQuantity--;
		if (plItem->itemQuantity == 0)
		{
			plItem->itemIndex = 0;
		}
	}
	return true;
}

//get gravity factor
float ObjPlayer_GetGravity(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;
	if (obj->net.frame < curAnim->startFrame || obj->net.frame > curAnim->endFrame)
	{
		return AI_GetGravity(obj, timeMod);
	}

	switch (obj->curAnim)
	{
	case PLANIM_PUNCH_UPPERCUT:
		if (obj->net.frame >= 472)
		{
			return 20.0f;
		}
		break;
	case PLANIM_SWORD_4:
		if (obj->net.frame >= 704)
		{
			return 20.0f;
		}
		break;
	case PLANIM_KICK_AIR1:
		if (obj->net.frame >= 541 && obj->net.frame <= 542)
		{
			return 20.0f;
		}
		break;
	case PLANIM_KICK_AIR2:
		if (obj->net.frame >= 549 && obj->net.frame <= 550)
		{
			return 20.0f;
		}
		break;
	case PLANIM_KICK_AIR_FLIP:
		if (obj->net.frame <= 612)
		{
			return 30.0f;
		}
		break;
	case PLANIM_PUNCH_AIRCRUSH:
	case PLANIM_PUNCH_AIRCRUSHSUPER:
		if (obj->net.frame < 563)
		{
			return 0.0f;
		}
		break;
	case PLANIM_SWORD_AIR:
		if (obj->net.frame <= 709)
		{
			return 0.0f;
		}
		break;
	default:
		break;
	}
	return AI_GetGravity(obj, timeMod);
}

//set the attack type
void ObjPlayer_SetAttackType(gameObject_t *obj)
{
	obj->atkType = ATKTYPE_NORMAL;
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;
	if (obj->net.frame < curAnim->startFrame || obj->net.frame > curAnim->endFrame)
	{
		return;
	}

	switch (obj->curAnim)
	{
	case PLANIM_PUNCH_UPPERCUT:
	case PLANIM_SWORD_4:
		obj->atkType = ATKTYPE_POPUP;
		break;
	case PLANIM_PUNCH_UPPERCUTSHORT:
		obj->atkType = ATKTYPE_POPUP3;
		break;
	case PLANIM_KICK_BACKFLIP:
		obj->atkType = ATKTYPE_POPUP2;
		break;
	case PLANIM_KICK_AIR1:
	case PLANIM_KICK_AIR2:
		obj->atkType = ATKTYPE_INAIR;
		break;
	case PLANIM_LUNGE_FINISH:
	case PLANIM_SWORD_3:
		obj->atkType = ATKTYPE_KNOCKBACK;
		break;
	case PLANIM_PUNCH_R2:
	case PLANIM_KICK_R4:
	case PLANIM_KICK_AIR_FLIP:
	case PLANIM_SWORD_AIR:
		obj->atkType = ATKTYPE_SMACKDOWN;
		break;
	case PLANIM_KICK_R2:
		obj->atkType = ATKTYPE_KNOCKBACK2;
		break;
	default:
		break;
	}
}

//take a step
static void ObjPlayer_Footstep(gameObject_t *obj, float timeMod)
{
	const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
	if (item && item->itemBehavior == INVITEM_BEHAVIOR_HEALSTEP)
	{
		obj->health += item->itemValue;
		if (obj->health > obj->plObj->plData.maxHealth)
		{
			obj->health = obj->plObj->plData.maxHealth;
		}
	}
	int snd = g_soundFootsteps[rand()%NUM_SOUNDS_FOOTSTEPS];
	ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
}

//frame tick
void ObjPlayer_FrameTick(gameObject_t *obj, float timeMod, int oldFrame)
{
	if (ObjPlayer_InChain(obj) && obj->curAnim != PLANIM_PUNCH_AIRCRUSH && obj->curAnim != PLANIM_PUNCH_AIRCRUSHSUPER &&
		obj->curAnim != PLANIM_SWORD_AIR)
	{
		gameAnim_t *curAnim = obj->animTable+obj->curAnim;
		chainDamage_t *cd = &g_chainDamage[obj->plObj->chainNum];
		int startFrame = (cd->startFrame > curAnim->startFrame) ? cd->startFrame : curAnim->startFrame;
		if (obj->net.frame == startFrame)
		{ //on the first frame of possible damage
			int snd;
			if (obj->curAnim >= PLANIM_SWORD_1 && obj->curAnim <= PLANIM_SWORD_AIR_LAND)
			{
				if (obj->plObj->sword && obj->plObj->sword->swordObj &&
					obj->plObj->sword->swordObj->numSwingSounds > 0)
				{
					gameObject_t *sword = obj->plObj->sword;
					snd = sword->swordObj->swingSounds[rand()%sword->swordObj->numSwingSounds];
				}
				else
				{
					snd = g_soundPunch[rand()%NUM_SOUNDS_PUNCH];
				}
			}
			else if ((obj->curAnim < PLANIM_KICK_R1 && obj->curAnim != PLANIM_C1_COMBO2) || cd->dmg < 30)
			{
				snd = g_soundPunch[rand()%NUM_SOUNDS_PUNCH];
			}
			else
			{
				snd = g_soundKick[rand()%NUM_SOUNDS_KICK];
				if (obj->curAnim == PLANIM_KICK_BACKFLIP)
				{
					ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
				}
			}
			ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
			if (obj->plObj->fistOfFlames)
			{
				if (rand()%10 < 5)
				{
					ObjSound_Create(obj->net.pos, "assets/sound/cb/swingflame.wav", 1.0f, -1);
				}
				else
				{
					ObjSound_Create(obj->net.pos, "assets/sound/cb/swingflame02.wav", 1.0f, -1);
				}
			}
			if (obj->plObj->explosiveBlows)
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/swingaura.wav", 1.0f, -1);
			}
		}
	}

	switch (obj->curAnim)
	{
	case HUMANIM_JUMP:
		if ((!obj->onGround || obj->net.frame == 86) && obj->net.frame != oldFrame && obj->net.vel[2] < 200.0f)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
			if (obj->plObj->clButtons[BUTTON_JUMP])
			{
				obj->net.vel[2] = 1200.0f;
			}
			else
			{
				obj->net.vel[2] = 700.0f;
			}
		}
		break;
	case PLANIM_PUNCH_UPPERCUT:
		if (obj->net.frame == 472)
		{
			obj->net.vel[2] += 1000.0f;
		}
		break;
	case PLANIM_SWORD_4:
		if (obj->net.frame == 704)
		{
			obj->net.vel[2] += 1000.0f;
		}
		break;
	case PLANIM_KICK_AIR1:
		if (obj->net.frame == 541)
		{
			obj->net.vel[0] *= 0.5f;
			obj->net.vel[1] *= 0.5f;
			obj->net.vel[2] = 200.0f;
		}
		break;
	case PLANIM_KICK_AIR2:
		if (obj->net.frame == 549)
		{
			obj->net.vel[0] *= 0.5f;
			obj->net.vel[1] *= 0.5f;
			obj->net.vel[2] = 200.0f;
		}
		break;
	case PLANIM_PUNCH_AIRCRUSH:
	case PLANIM_PUNCH_AIRCRUSHSUPER:
		if (obj->net.frame == 560)
		{
			obj->net.vel[0] = 0.0f;
			obj->net.vel[1] = 0.0f;
			obj->net.vel[2] = 200.0f;
		}
		else if (obj->net.frame == 563 && obj->net.frame != oldFrame)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
			if (obj->curAnim == PLANIM_PUNCH_AIRCRUSHSUPER)
			{
				obj->net.vel[2] = -1300.0f;
			}
			else
			{
				obj->net.vel[2] = -1000.0f;
			}
		}
		break;
	case PLANIM_SWORD_AIR:
		if (obj->net.frame == 709)
		{
			obj->net.vel[0] = 0.0f;
			obj->net.vel[1] = 0.0f;
			obj->net.vel[2] = 200.0f;
		}
		else if (obj->net.frame == 710 && obj->net.frame != oldFrame)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
			obj->net.vel[2] = -1000.0f;
		}
		break;
	case PLANIM_KICK_AIR_FLIP:
		if (obj->net.frame == 610)
		{
			obj->net.vel[2] = 0.0f;
		}
		break;
	case PLANIM_KICK_BACKFLIP:
		if (obj->net.frame >= 602 && obj->onGround)
		{
			float fwd[3];
			Math_AngleVectors(obj->net.ang, fwd, 0, 0);
			obj->net.vel[0] = -fwd[0]*400.0f;
			obj->net.vel[1] = -fwd[1]*400.0f;
			obj->net.vel[2] = 1000.0f;
		}
		break;
	case HUMANIM_WALK:
	case HUMANIM_RUNSLOW:
	case HUMANIM_RUN:
		if ((obj->net.frame == 448 || obj->net.frame == 451) && obj->onGround)
		{
			ObjPlayer_Footstep(obj, timeMod);
		}
		break;
	case PLANIM_STRAFE_LEFT:
		if ((obj->net.frame == 480 || obj->net.frame == 477) && obj->onGround)
		{
			ObjPlayer_Footstep(obj, timeMod);
		}
		break;
	case PLANIM_STRAFE_RIGHT:
		if ((obj->net.frame == 482 || obj->net.frame == 485) && obj->onGround)
		{
			ObjPlayer_Footstep(obj, timeMod);
		}
		break;
	case PLANIM_TARG_FWD:
		if ((obj->net.frame == 488 || obj->net.frame == 490) && obj->onGround)
		{
			ObjPlayer_Footstep(obj, timeMod);
		}
		break;
	case PLANIM_TARG_BACK:
		if ((obj->net.frame == 491 || obj->net.frame == 493) && obj->onGround)
		{
			ObjPlayer_Footstep(obj, timeMod);
		}
		break;
	case PLANIM_ITEM_USE:
		if (obj->net.frame == 122)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/items/rifle.wav", 1.0f, -1);
		}
		if (obj->net.frame == 128)
		{
			const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
			if (item && item->usable)
			{
				ObjPlayer_UseItem(obj, item);
			}
		}
		break;
	case PLANIM_RISE:
		if (obj->net.frame == 624)
		{
			obj->health = obj->plObj->plData.maxHealth/4;
			ObjSound_Create(obj->net.pos, "assets/sound/cb/jesus.wav", 1.0f, -1);
			g_actionTime = g_curTime+500;
			obj->net.renderEffects |= FXFL_CONTRAST2;
		}
		//625 == head hit back
		//629 == risen highest
		//630 == snap back
		break;
	case PLANIM_DEATH:
	case PLANIM_DEATH_LAND:
		if (obj->net.frame != oldFrame)
		{
			if (obj->net.frame == 220 || obj->net.frame == 222)
			{
				ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[2], 1.0f, -1);
			}
			if (obj->net.frame == 222)
			{
				g_sharedFn->Common_MapChange(NULL, 6000);
			}
			if (obj->net.frame == 224)
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/objland.wav", 1.0f, -1);
			}
		}
		break;
	case PLANIM_DEATH_MP:
	case PLANIM_DEATH_LAND_MP:
		if (obj->net.frame != oldFrame)
		{
			if (obj->net.frame == 220 || obj->net.frame == 222)
			{
				ObjSound_CreateFromIndex(obj->net.pos, g_soundDeepBlow[2], 1.0f, -1);
			}
			if (obj->net.frame == 224)
			{
				ObjSound_Create(obj->net.pos, "assets/sound/cb/objland.wav", 1.0f, -1);
			}
		}
		break;
	case PLANIM_SEPH_AERITHPUNCH:
		if (obj->net.frame == 770)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeaura.wav", 1.0f, -1);
		}
		break;
	default:
		break;
	}
}

//custom movement animations
bool ObjPlayer_IdleAnim(gameObject_t *obj, float timeMod)
{
	if (!obj->plObj->clButtons[BUTTON_EVENT7] || !obj->onGround)
	{ //only override the anims when in target-locked mode
		return false;
	}

	float velLen = Math_VecLen(obj->net.vel);
	if (velLen <= 5.0f)
	{ //not moving enough
		return false;
	}

	float fwd[3], right[3];
	float nVel[3];
	Math_AngleVectors(obj->net.ang, fwd, right, 0);
	Math_VecCopy(obj->net.vel, nVel);
	Math_VecNorm(nVel);

	float dpfwd = Math_DotProduct(fwd, nVel);
	float dpright = Math_DotProduct(right, nVel);

	if (fabsf(dpright) > fabsf(dpfwd))
	{ //left/right
		if (dpright >= 0.0f)
		{
			AI_StartAnim(obj, PLANIM_STRAFE_RIGHT, false);
		}
		else
		{
			AI_StartAnim(obj, PLANIM_STRAFE_LEFT, false);
		}
	}
	else
	{ //forward/back
		if (dpfwd >= 0.0f)
		{
			AI_StartAnim(obj, PLANIM_TARG_FWD, false);
		}
		else
		{
			AI_StartAnim(obj, PLANIM_TARG_BACK, false);
		}
	}

	return true;
}

//pick which anim to be in
void ObjPlayer_PickAnim(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;

	if (obj->curAnim >= PLANIM_CIN_IDLE_1)
	{ //script will break me out
		return;
	}
	else if (obj->plObj->ride)
	{ //chocobo
		return;
	}
	else if (obj->curAnim == PLANIM_AIR_RECOVER)
	{
		if (obj->onGround)
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
		else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_FALL, true);
		}
	}
	else if (obj->curAnim == PLANIM_WALLJUMP || obj->curAnim == PLANIM_BACKFLIP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->onGround)
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_FALL, true);
			}
		}
	}
	else if (obj->curAnim == PLANIM_DIVE_BOUNCE || obj->curAnim == PLANIM_KICK_AIR_FLIP)
	{
		if (obj->onGround &&
			(obj->curAnim != PLANIM_DIVE_BOUNCE || obj->net.frame >= 512 && obj->net.frame <= 515))
		{
			obj->plObj->chainNum = 0;
			obj->plObj->nextChain = 0;
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
		else if (obj->curAnim == PLANIM_KICK_AIR_FLIP)
		{
			ObjPlayer_ProgressChain(obj);
		}
		else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			obj->plObj->chainNum = 0;
			obj->plObj->nextChain = 0;
			AI_StartAnim(obj, PLANIM_AIR_RECOVER, true);
		}
	}
	else if (obj->curAnim == PLANIM_GETUP_FLIP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->onGround)
			{
				AI_StartAnim(obj, HUMANIM_IDLE, true);
			}
			else
			{
				AI_StartAnim(obj, PLANIM_AIR_RECOVER, true);
			}
		}
	}
	else if (obj->curAnim == PLANIM_LUNGE || obj->curAnim == PLANIM_TACKLE)
	{
		if (obj->curAnim == PLANIM_LUNGE && obj->plObj->clButtons[BUTTON_EVENT1] && ObjPlayer_Ability(obj, PLABIL_TACKLE))
		{ //go into tackle
			AI_StartAnim(obj, PLANIM_TACKLE, true);
		}
		else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (!obj->onGround || obj->plObj->lungeTime < g_curTime)
			{
				obj->plObj->lungeTime = 0;
				obj->plObj->chainNum = (obj->curAnim == PLANIM_TACKLE) ? PLCHAIN_TACKLE_FINISH : PLCHAIN_LUNGE_FINISH;
				obj->plObj->nextChain = 0;
				obj->atkLastSeq++;

				playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
				AI_StartAnim(obj, ch->anim, true);
			}
		}
	}
	else if (obj->curAnim == PLANIM_KICK_AIR1 ||
		obj->curAnim == PLANIM_KICK_AIR2)
	{
		if (!ObjPlayer_InChain(obj) || obj->onGround)
		{
			AI_StartAnim(obj, HUMANIM_LAND, true);
		}
		else
		{
			ObjPlayer_ProgressChain(obj);
		}
	}
	else if (obj->curAnim == PLANIM_DEATH)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart && obj->onGround)
		{
			obj->animNeverInterrupt = false;
			AI_StartAnim(obj, PLANIM_DEATH_LAND, true);
			obj->animNeverInterrupt = true;
		}
	}
	else if (obj->curAnim == PLANIM_DEATH_MP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart && obj->onGround)
		{
			AI_StartAnim(obj, PLANIM_DEATH_LAND_MP, true);
		}
	}
	else if (obj->curAnim == PLANIM_RISE)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (g_actionTime)
			{
				g_actionTime = 1;
			}
			obj->net.renderEffects &= ~FXFL_CONTRAST2;
			obj->hurtable = 1;
			obj->animNeverInterrupt = false;
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->curAnim == PLANIM_SEPH_STABBED)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_GETUP_FRONT, true);
		}
	}
	else if (obj->curAnim == PLANIM_SEPH_BEATDOWN)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, PLANIM_SEPH_BEATDOWNLOOP, true);
		}
	}
	else if (obj->curAnim == PLANIM_SEPH_BEATDOWNLOOP)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			if (obj->dualAnimCnt <= 0)
			{
				AI_StartAnim(obj, PLANIM_SEPH_AERITHPUNCH, true);
			}
			else
			{
				obj->dualAnimCnt--;
				AI_StartAnim(obj, PLANIM_SEPH_BEATDOWNLOOP, true);
			}
		}
	}
	else if (obj->curAnim == PLANIM_SEPH_AERITHPUNCH)
	{
		if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->curAnim == PLANIM_DEATH_LAND)
	{
		return;
	}
	else if (obj->curAnim == PLANIM_DEATH_LAND_MP)
	{
		return;
	}
	else if (ObjPlayer_InChain(obj))
	{
		ObjPlayer_ProgressChain(obj);
	}
	else if (obj->curAnim >= PLANIM_STRAFE_LEFT && obj->curAnim <= PLANIM_TARG_BACK)
	{
		if (!ObjPlayer_IdleAnim(obj, timeMod))
		{
			AI_StartAnim(obj, HUMANIM_IDLE, true);
		}
	}
	else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
	{
		AI_StartAnim(obj, HUMANIM_IDLE, true);
	}
}

//write game data
void ObjPlayer_WriteGameData(gameObject_t *obj, cntStream_t *st)
{
	if (!obj->inuse || !obj->plObj)
	{
		Stream_WriteBool(st, false);
		return;
	}
	Stream_WriteBool(st, true);

	Stream_WriteInt(st, obj->health);
	Stream_WriteInt(st, obj->plObj->makoCharges);
	Stream_WriteInt(st, obj->plObj->makoStash);
	Stream_WriteInt(st, obj->plObj->equippedItem);
	Stream_WriteBytes(st, &obj->plObj->plData, sizeof(obj->plObj->plData));
}

//read game data
void ObjPlayer_ReadGameData(gameObject_t *obj, cntStream_t *st)
{
	bool valid = Stream_ReadBool(st);
	if (!valid)
	{
		return;
	}

	if (obj && !obj->plObj)
	{
		obj->plObj = (playerObject_t *)g_sharedFn->Common_RCMalloc(sizeof(playerObject_t));
		memset(obj->plObj, 0, sizeof(playerObject_t));
	}

	gameObject_t *lObj;
	playerObject_t *lPlObj;
	gameObject_t ta;
	playerObject_t tb;
	if (!obj)
	{
		lObj = &ta;
		lPlObj = &tb;
	}
	else
	{
		lObj = obj;
		lPlObj = obj->plObj;
	}
	lObj->health = Stream_ReadInt(st);
	lPlObj->makoCharges = Stream_ReadInt(st);
	lPlObj->makoStash = Stream_ReadInt(st);
	lPlObj->equippedItem = Stream_ReadInt(st);
	lPlObj->desiredItem = lPlObj->equippedItem;
	Stream_ReadBytes(st, &lPlObj->plData, sizeof(lPlObj->plData));

	//actually, just always start at full health
	if (obj)
	{
		obj->health = obj->plObj->plData.maxHealth;
		obj->plObj->makoCharges = obj->plObj->plData.maxMakoCharges;
	}
}

//called when inflicting damage
void ObjPlayer_OnHurt(gameObject_t *obj, gameObject_t *victim, int dmg, const collObj_t *col)
{
	if (obj->inuse && obj->plObj && obj->plObj->sword && obj->plObj->sword->inuse &&
		obj->plObj->sword->generalFlag3 && obj->plObj->sword->swordTime == g_curTime)
	{ //drain sword
		int d = dmg/8;
		if (d < 1)
		{
			d = 1;
		}
		obj->health += d;
		if (obj->health > obj->plObj->plData.maxHealth)
		{
			obj->health = obj->plObj->plData.maxHealth;
		}
		//obj->plObj->sword->net.renderEffects2 |= FXFL2_DRAINAURA;
	}
}

//client joined, create its map object.
void LServ_InitPlayerObj(int clientNum, const char *clientInfo)
{
	gameObject_t *pl = &g_gameObjects[clientNum];

	bool noClSave = false;
	if (!pl->plObj)
	{
		pl->plObj = (playerObject_t *)g_sharedFn->Common_RCMalloc(sizeof(playerObject_t));
		memset(pl->plObj, 0, sizeof(playerObject_t));
		noClSave = true;
	}

	pl->targetName = "__the_player";

	pl->net.spare1 = 255;

	pl->plObj->moveSpeedScale = 1.0f;

	if (noClSave || clientNum == 0)
	{
		if (noClSave || !GVar_GetInt("plhasstats"))
		{
			pl->plObj->chargeMeter = 0;
			pl->plObj->plData.maxHealth = 250;
			pl->plObj->plData.maxMakoCharges = 0;
			pl->plObj->makoCharges = 0;
			pl->plObj->makoStash = 0;
			pl->plObj->plData.statStr = 10;
			pl->plObj->plData.statDef = 10;
			pl->plObj->plData.statDex = 10;
			pl->plObj->plData.statLuck = 5;

			//you get 1 of nothing
			pl->plObj->plData.inventory[0].itemIndex = 0;
			pl->plObj->plData.inventory[0].itemQuantity = 1;
			pl->plObj->equippedItem = 0;
			pl->plObj->desiredItem = 0;

			GVar_SetInt("plhasstats", 1);
		}
	}

	strcpy(g_clientInfo[clientNum].infoStr, clientInfo);
	for (int i = 0; i < MAX_NET_CLIENTS; i++)
	{
		gameObject_t *otherPl = &g_gameObjects[i];
		if (i == clientNum || !otherPl->inuse)
		{
			continue;
		}
		if (!stricmp(g_clientInfo[i].infoStr, g_clientInfo[clientNum].infoStr))
		{ //name is taken
			g_clientInfo[clientNum].infoStr[20] = 0;
			strcat(g_clientInfo[clientNum].infoStr, "_2");
		}
	}
	g_sharedFn->Net_SendEventType(CLEV_UPDATEHIGHSCORE, &g_musicHighScore, sizeof(g_musicHighScore), clientNum);

	pl->animTable = g_playerAnims;
	pl->numAnims = NUM_PLAYER_ANIMS;
	pl->scriptAnims = g_plScriptAnims;
	pl->curAnim = HUMANIM_IDLE;
	pl->animhandler = ObjPlayer_PickAnim;
	pl->animframetick = ObjPlayer_FrameTick;
	pl->animidleoverride = ObjPlayer_IdleAnim;

	pl->canPush = true;

	pl->net.owner = MAP_OBJ_NUM;
	pl->net.staticIndex = -1;
	pl->inuse = INUSE_ACTIVE;
	pl->net.index = clientNum;
	pl->creationCount = g_creationCount;
	g_creationCount++;
	pl->localTimeScale = 1.0f;
	pl->net.type = OBJ_TYPE_MODEL;
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];
	if (cam->inuse)
	{
		pl->net.pos[0] = cam->net.pos[0]-2048.0f;
		pl->net.pos[1] = cam->net.pos[1];
		pl->net.pos[2] = cam->net.pos[2]-4096.0f;

		if (clientNum == 0)
		{
			cam->net.ang[0] = g_initialFPSAng[0];
			cam->net.ang[1] = g_initialFPSAng[1];
			cam->net.ang[2] = g_initialFPSAng[2];
		}
	}
	else
	{
		pl->net.pos[0] = -2048.0f;
		pl->net.pos[1] = 0.0f;
		pl->net.pos[2] = DEFAULT_PLR_Z-2048.0f;
	}

	pl->health = pl->plObj->plData.maxHealth;
	pl->hurtable = 1;

	pl->plObj->playerSpawnSeq = true;

	pl->net.ang[0] = 0.0f;
	pl->net.ang[1] = 0.0f;
	pl->net.ang[2] = 0.0f;

	pl->net.mins[0] = -100.0f;
	pl->net.mins[1] = -100.0f;
	pl->net.mins[2] = 0.0f;
	pl->net.maxs[0] = 100.0f;
	pl->net.maxs[1] = 100.0f;
	pl->net.maxs[2] = 430.0f;
	pl->radius = 200.0f;

	pl->net.plAmmo = 0;

	pl->net.solid = 1;

	pl->touch = ObjPlayer_Touch;
	pl->pain = AI_GenericPain;
	pl->death = ObjPlayer_Death;
	pl->think = ObjPlayer_Think;
	pl->thinkTime = 0;
	pl->onhurt = ObjPlayer_OnHurt;

	pl->localAngles = true;

	pl->net.entNameIndex = g_sharedFn->Common_ServerString("obj_player");

	if (clientNum > 0)
	{
		pl->net.strIndex = g_sharedFn->Common_ServerString("&assets/models/tifa2.rdm");
	}
	else
	{
		pl->net.strIndex = g_sharedFn->Common_ServerString("&assets/models/tifa.rdm");
	}
	pl->net.strIndexB = g_sharedFn->Common_ServerString("@assets/models/tifa.rda");

	pl->rcColModel = g_sharedFn->Coll_RegisterModelInstance("assets/models/tifa.rdm");
	if (pl->rcColModel)
	{
		pl->rcColModel->radius = pl->radius;
		pl->rcColModel->clipFlags = CLIPFL_BOXMOVE;
		Math_VecCopy(pl->net.mins, pl->rcColModel->mins);
		Math_VecCopy(pl->net.maxs, pl->rcColModel->maxs);
		Math_VecCopy(pl->net.pos, pl->rcColModel->pos);
		Math_VecCopy(pl->net.ang, pl->rcColModel->ang);
		Math_VecCopy(pl->net.modelScale, pl->rcColModel->modelScale);
		pl->rcColModel->gameOwner = pl->net.index;
		pl->rcColModel->solid = pl->net.solid;
		pl->rcColModel->frame = pl->net.frame;
		g_sharedFn->Coll_UpdateModel(pl->rcColModel, "assets/models/tifa.rda");
	}
}

//should others dodge me?
bool ObjPlayer_InDodgeableAttack(gameObject_t *obj)
{
	if (obj->curAnim != PLANIM_LUNGE &&
		obj->curAnim != PLANIM_TACKLE &&
		obj->curAnim != PLANIM_KICK_DIVE)
	{
		return false;
	}
	if (Math_VecLen(obj->net.vel) < 50.0f)
	{
		return false;
	}
	return true;
}

//am i open to attack?
bool ObjPlayer_InVulnerableState(gameObject_t *obj)
{
	if (obj->net.index >= MAX_NET_CLIENTS || !obj->plObj)
	{
		return false;
	}
	if (obj->curAnim == PLANIM_PUNCH_AIRCRUSH_LAND ||
		obj->curAnim == PLANIM_PUNCH_AIRCRUSH_LANDSUPER ||
		obj->curAnim == PLANIM_DIVE_LAND ||
		obj->curAnim == PLANIM_ITEM_USE ||
		obj->curAnim == PLANIM_SWORD_AIR_LAND)
	{
		return true;
	}
	return false;
}

//using item?
bool ObjPlayer_IsUsingItem(gameObject_t *obj)
{
	if (!obj || !obj->inuse || obj->net.index >= MAX_NET_CLIENTS || !obj->plObj)
	{
		return false;
	}
	if (obj->curAnim == PLANIM_ITEM_USE)
	{
		return true;
	}
	return false;
}

//did the client tap a button?
bool ObjPlayer_ButtonTapped(gameObject_t *obj, int button)
{
	bool hit = false;
	int lastCheck = -1;
	if (obj->plObj->clButtons[button])
	{
		hit = true;
		lastCheck = obj->plObj->numButtonsBuffered-1;
	}
	else
	{
		for (int i = 0; i < obj->plObj->numButtonsBuffered; i++)
		{
			if (obj->plObj->clButtonsBackBuffer[i][button])
			{
				hit = true;
				lastCheck = i-1;
				break;
			}
		}
	}

	return (hit && lastCheck >= 0 && !obj->plObj->clButtonsBackBuffer[lastCheck][button]);
}

//player touch
void ObjPlayer_Touch(gameObject_t *obj, gameObject_t *other, const collObj_t *col)
{
	if ((obj->curAnim == PLANIM_DODGE_LEFT ||
		obj->curAnim == PLANIM_DODGE_RIGHT) && other && other->hurtable)
	{ //push it away
		float otherCenter[3], plCenter[3], d[3];
		Util_GameObjectCenter(other, otherCenter);
		Util_GameObjectCenter(obj, plCenter);
		Math_VecSub(otherCenter, plCenter, d);
		other->net.vel[0] += d[0]*2.0f;
		other->net.vel[1] += d[1]*2.0f;
	}

	if (obj->curAnim == PLANIM_LUNGE || obj->curAnim == PLANIM_TACKLE)
	{
		if (other && other->hurtable && Util_ValidEnemies(obj, other))
		{
			if (Util_SequenceDamage(obj, other))
			{
				float ang[3];
				Math_VecToAngles((float *)col->endNormal, ang);
				Util_SetSequenceDamage(obj, other);

				modelMatrix_t boneMat;
				float impactPos[3];
				if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b27", &boneMat))
				{
					Math_VecCopy(boneMat.o, impactPos);
				}
				else
				{
					Math_VecCopy((float *)col->endPos, impactPos);
				}
				collObj_t nCol = *col;
				Math_VecCopy(impactPos, nCol.endPos);
				int hitDmg = ObjPlayer_StrMod(obj, 5);
				if (other && (!other->attackblock || !other->attackblock(other, obj, hitDmg, &nCol, impactPos, ang)))
				{
					ObjSound_CreateFromIndex(impactPos, g_soundHitLight[rand()%NUM_SOUNDS_HITLIGHT], 1.0f, -1);
					ObjParticles_Create("melee/impact", impactPos, ang, -1);
					Util_DamageObject(obj, other, hitDmg, &nCol);
				}
				obj->plObj->lungeTime = 0;
			}
		}

		float groundPl[3] = {0.0f, 0.0f, 1.0f};
		float groundDot = 0.7f;
		float dp = Math_DotProduct(groundPl, col->endNormal);
		if (dp < groundDot)
		{
			float nVel[3];
			Math_VecCopy(obj->net.vel, nVel);
			Math_VecNorm(nVel);
			float d = Math_DotProduct(nVel, col->endNormal);
			if (d < -0.35f)
			{ //hitting against the surface
				if (obj->plObj->lungeTime)
				{
					obj->net.vel[0] = 0.0f;
					obj->net.vel[1] = 0.0f;
					obj->plObj->lungeTime = 0;
					if (obj->curAnim == PLANIM_TACKLE && other && other->hurtable)
					{ //turn toward them and pull them in
						float d[3];
						Math_VecSub(other->net.pos, obj->net.pos, d);
						Math_VecNorm(d);
						float ang[3];
						Math_VecToAngles(d, ang);
						obj->net.ang[YAW] = ang[YAW];
						float f = 200.0f;
						other->net.vel[0] -= d[0]*f;
						other->net.vel[1] -= d[1]*f;
						other->net.vel[2] -= d[2]*f;
					}
				}
			}
		}
	}
	else if (obj->curAnim == PLANIM_KICK_BACKFLIP)
	{
		/*
		if (!obj->onGround && obj->net.vel[2] <= 0.0f && obj->net.frame < 605)
		{
			obj->plObj->chainNum = 0;
			obj->plObj->nextChain = 0;
			AI_StartAnim(obj, HUMANIM_IDLE, false);
		}
		*/
		if (other && other->hurtable && Util_ValidEnemies(obj, other))
		{
			if (Util_SequenceDamage(obj, other))
			{
				float ang[3];
				Math_VecToAngles((float *)col->endNormal, ang);
				Util_SetSequenceDamage(obj, other);

				float impactPos[3];
				Math_VecCopy((float *)col->endPos, impactPos);
				impactPos[2] = other->net.pos[2]+other->net.mins[2];
				ObjSound_CreateFromIndex(impactPos, g_soundHitHeavy[rand()%NUM_SOUNDS_HITHEAVY], 1.0f, -1);
				ObjParticles_Create("melee/impact", impactPos, ang, -1);
				Util_DamageObject(obj, other, ObjPlayer_StrMod(obj, 40), col);
			}
			if (other->net.pos[2] > obj->net.pos[2]+32.0f)
			{
				float push[3];
				Math_VecSub(other->net.pos, obj->net.pos, push);
				Math_VecNorm(push);
				float pushf = 400.0f;
				other->net.vel[0] = push[0]*pushf;
				other->net.vel[1] = push[1]*pushf;
				other->net.vel[2] = 800.0f;
			}
		}
	}
	else if (obj->curAnim == PLANIM_KICK_DIVE && obj->net.frame >= 617 && obj->net.frame <= 618)
	{
		if (other && other->hurtable && Util_ValidEnemies(obj, other))
		{
			if (Util_SequenceDamage(obj, other))
			{
				float ang[3];
				Math_VecToAngles((float *)col->endNormal, ang);
				Util_SetSequenceDamage(obj, other);

				float impactPos[3];
				Math_VecCopy((float *)col->endPos, impactPos);
				ObjSound_CreateFromIndex(impactPos, g_soundHitLight[rand()%NUM_SOUNDS_HITLIGHT], 1.0f, -1);
				ObjParticles_Create("melee/impact", impactPos, ang, -1);
				int dmg = (!other->onGround) ? 30 : 5; //hurts more when you get kicked out of the air
				Util_DamageObject(obj, other, ObjPlayer_StrMod(obj, dmg), col);
			}
		}
		float groundPl[3] = {0.0f, 0.0f, 1.0f};
		float groundDot = 0.7f;
		float dp = Math_DotProduct(groundPl, col->endNormal);
		if (dp < groundDot || (other && other->hurtable))
		{
			float push[3];
			if (other && other->hurtable)
			{
				Math_VecSub(obj->net.pos, other->net.pos, push);
				push[2] = 0.0f;
			}
			else
			{
				push[0] = col->endNormal[0];
				push[1] = col->endNormal[1];
				push[2] = 0.0f;
			}
			Math_VecNorm(push);
			if (other && other->hurtable && !other->onGround)
			{ //bouncing off of another guy in the air
				obj->net.vel[0] = push[0]*1000.0f;
				obj->net.vel[1] = push[1]*1000.0f;
				obj->net.vel[2] = 700.0f;
				other->net.vel[0] += -push[0]*400.0f;
				other->net.vel[1] += -push[1]*400.0f;
			}
			else
			{
				obj->net.vel[0] = push[0]*fabsf(obj->net.vel[0]);
				obj->net.vel[1] = push[1]*fabsf(obj->net.vel[1]);
				obj->net.vel[2] = 200.0f;
			}
			AI_StartAnim(obj, PLANIM_DIVE_BOUNCE, false);
			ObjSound_Create(obj->net.pos, "assets/sound/cb/walljump01.wav", 1.0f, -1);
			ObjSound_Create(obj->net.pos, "assets/sound/cb/hitbounce.wav", 1.0f, -1);
		}
		else
		{
			AI_StartAnim(obj, PLANIM_DIVE_LAND, false);
		}
	}
	else if (obj->curAnim == PLANIM_PUNCH_AIRCRUSH || obj->curAnim == PLANIM_PUNCH_AIRCRUSHSUPER ||
		obj->curAnim == PLANIM_SWORD_AIR)
	{
		gameAnim_t *curAnim = obj->animTable+obj->curAnim;
		if ((obj->net.frame == curAnim->endFrame && obj->rcColModel->blendFrame == curAnim->endFrame) ||
			other->hurtable)
		{
			float groundPl[3] = {0.0f, 0.0f, 1.0f};
			float groundDot = 0.0f;
			float gd = Math_DotProduct(groundPl, col->endNormal);
			if (gd > groundDot)
			{
				modelMatrix_t boneMat;
				Math_VecCopy((float *)col->endPos, obj->rcColModel->pos);
				g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);
				if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b32", &boneMat))
				{
					float ang[3], p[3];
					float ofs[3] = {0.0f, 0.0f, -80.0f};
					Math_VecToAngles(col->endNormal, ang);
					Math_TransformPointByMatrix(&boneMat, ofs, p);
					if (obj->curAnim == PLANIM_PUNCH_AIRCRUSHSUPER)
					{
						ObjParticles_Create("impacts/aircrushsuper", p, ang, -1);
						ObjSound_Create(p, "assets/sound/cb/groundsmash.wav", 1.0f, -1);
						ObjSound_Create(p, "assets/sound/cb/groundsmash.wav", 1.0f, -1);
						Util_RadiusDamage(obj, ObjPlayer_StrMod(obj, 100), 1400.0f, (float *)col->endPos);
					}
					else if (obj->curAnim == PLANIM_SWORD_AIR)
					{
						if (obj->plObj && obj->plObj->sword &&
							obj->plObj->sword->swordObj && obj->plObj->sword->swordObj->numImpactFX > 0)
						{
							gameObject_t *sword = obj->plObj->sword;
							int fxIdx = sword->swordObj->impactFX[rand()%sword->swordObj->numImpactFX];
							ObjParticles_CreateFromIndex(fxIdx, p, obj->net.ang, -1);
						}
						else
						{
							ObjParticles_Create("impacts/airsword", p, obj->net.ang, -1);
						}

						if (obj->plObj && obj->plObj->sword &&
							obj->plObj->sword->swordObj && obj->plObj->sword->swordObj->numImpactSounds > 0)
						{
							gameObject_t *sword = obj->plObj->sword;
							int snd = sword->swordObj->impactSounds[rand()%sword->swordObj->numImpactSounds];
							ObjSound_CreateFromIndex(p, snd, 1.0f, -1);
						}
						else
						{
							ObjSound_Create(p, "assets/sound/cb/swdhit.wav", 1.0f, -1);
						}
						Util_RadiusDamage(obj, ObjPlayer_StrMod(obj, 200), 500.0f, (float *)col->endPos);
					}
					else
					{
						ObjParticles_Create("impacts/aircrush", p, ang, -1);
						ObjSound_Create(p, "assets/sound/cb/groundsmash.wav", 1.0f, -1);
						Util_RadiusDamage(obj, ObjPlayer_StrMod(obj, 70), 1000.0f, (float *)col->endPos);
					}
				}
				Math_VecCopy(obj->net.pos, obj->rcColModel->pos);
				g_sharedFn->Coll_UpdateModel(obj->rcColModel, NULL);

				obj->plObj->chainNum = 0;
				obj->plObj->nextChain = 0;
				if (obj->curAnim == PLANIM_PUNCH_AIRCRUSHSUPER)
				{
					AI_StartAnim(obj, PLANIM_PUNCH_AIRCRUSH_LANDSUPER, false);
				}
				else if (obj->curAnim == PLANIM_SWORD_AIR)
				{
					AI_StartAnim(obj, PLANIM_SWORD_AIR_LAND, false);
				}
				else
				{
					AI_StartAnim(obj, PLANIM_PUNCH_AIRCRUSH_LAND, false);
				}
			}
		}
	}
	AI_GenericTouch(obj, other, col);
}

//player die
void ObjPlayer_Death(gameObject_t *obj, gameObject_t *killer, int dmg)
{
	obj->isOnFire = 0;
	obj->plObj->fistOfFlames = 0;
	obj->net.renderEffects &= ~FXFL_FLAMES;
	obj->plObj->explosiveBlows = 0;
	obj->net.renderEffects &= ~FXFL_AURA2;
	obj->net.renderEffects &= ~FXFL_BONEAURA;
	obj->plObj->healingAura = 0;
	obj->net.renderEffects &= ~FXFL_HEALINGAURA;
	obj->meteorAttack = 0;
	obj->net.renderEffects &= ~FXFL_METEORSTREAM;
	if (g_timeType == TIMETYPE_TIMEATTACK)
	{
		g_timeType = TIMETYPE_NORMAL;
	}
	obj->plObj->timeAttackTime = 0;

	const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
	if (dmg < 9999 && item && item->itemBehavior == INVITEM_BEHAVIOR_PHOENIX && ObjPlayer_UseItem(obj, item))
	{
		float killerCenter[3], plCenter[3], d[3];
		Util_GameObjectCenter(killer, killerCenter);
		Util_GameObjectCenter(obj, plCenter);
		Math_VecSub(plCenter, killerCenter, d);
		Math_VecNorm(d);
		obj->net.vel[0] += d[0]*600.0f;
		obj->net.vel[1] += d[1]*600.0f;
		return;
	}

	obj->hurtable = 0;

	if (g_mpVersusMode)
	{
		float killerCenter[3], plCenter[3], d[3];
		Util_GameObjectCenter(killer, killerCenter);
		Util_GameObjectCenter(obj, plCenter);
		Math_VecSub(plCenter, killerCenter, d);
		Math_VecNorm(d);
		obj->net.vel[0] += d[0]*800.0f;
		obj->net.vel[1] += d[1]*800.0f;
		obj->net.vel[2] = 400.0f;

		AI_StartAnim(obj, PLANIM_DEATH_MP, false);
		ObjSound_Create(obj->net.pos, "assets/sound/cb/pldeath.wav", 1.0f, -1);
		g_timeType = TIMETYPE_DEATH;
		//obj->net.solid = 0;
		//LServ_UpdateRClip(obj);
		return;
	}

	LServ_ChangeMusic("$$");
	g_timeType = TIMETYPE_DEATH;
	obj->animNeverInterrupt = false;
	AI_StartAnim(obj, PLANIM_DEATH, false);
	obj->animNeverInterrupt = true;
	float killerCenter[3], plCenter[3], d[3];
	Util_GameObjectCenter(killer, killerCenter);
	Util_GameObjectCenter(obj, plCenter);
	Math_VecSub(plCenter, killerCenter, d);
	Math_VecNorm(d);
	obj->net.vel[0] += d[0]*400.0f;
	obj->net.vel[1] += d[1]*400.0f;
	obj->net.vel[2] = 600.0f;
	ObjSound_Create(obj->net.pos, "assets/sound/cb/pldeath.wav", 1.0f, -1);
}

float g_worldMins[3], g_worldMaxs[3];

//check analog input movement
void ObjPlayer_AnalogMove(gameObject_t *obj, float &move, float &moveRight,
							  const float moveSpeed, const float timeMod)
{
	if (!obj->plObj->hasClAnalog)
	{
		return;
	}

	float moveDir[2];
	for (int i = 0; i < 2; i++)
	{
		const float minMove = 0.2f;
		moveDir[i] = (float)((int)obj->plObj->clAnalog[i] - 32767)/32767.0f;
		if (fabsf(moveDir[i]) < minMove)
		{
			moveDir[i] = 0.0f;
		}
	}

	if (moveDir[0] != 0.0f)
	{
		moveRight = moveDir[0]*moveSpeed;
	}
	if (moveDir[1] != 0.0f)
	{
		move = -moveDir[1]*moveSpeed;
	}
}

//check analog input turning
void ObjPlayer_AnalogLook(gameObject_t *obj, const float timeMod, float *outAng)
{
	if (!obj->plObj->hasClAnalog)
	{
		return;
	}

	float moveDir[2];
	for (int i = 2; i < 4; i++)
	{
		const float minMove = 0.2f;
		moveDir[i-2] = (float)((int)obj->plObj->clAnalog[i] - 32767)/32767.0f;
		if (fabsf(moveDir[i-2]) < minMove)
		{
			moveDir[i-2] = 0.0f;
		}
	}

	outAng[0] += moveDir[1]*(8.0f*timeMod);
	if (outAng[0] > 90.0f)
	{
		outAng[0] = 90.0f;
	}
	else if (outAng[0] < -90.0f)
	{
		outAng[0] = -90.0f;
	}
	outAng[1] -= moveDir[0]*(8.0f*timeMod);
}

//turn
void ObjPlayer_AdjustTurnAngles(gameObject_t *obj, float timeMod, float *outAng)
{
	float turnRate = 5.0f; //10.0f
	float pitchTurnRate = 2.5f; //5.0f
	int turnAxis = YAW;
	if (obj->plObj->clButtons[BUTTON_EVENT2])
	{ //turn fetus
		outAng[turnAxis] += turnRate*timeMod;		
	}
	if (obj->plObj->clButtons[BUTTON_EVENT3])
	{ //turn fetus
		outAng[turnAxis] -= turnRate*timeMod;		
	}

	if (obj->plObj->clButtons[BUTTON_EVENT5])
	{ //fps mode - look down
		outAng[0] += pitchTurnRate*timeMod;
		if (outAng[0] > 90.0f)
		{
			outAng[0] = 90.0f;
		}
	}
	if (obj->plObj->clButtons[BUTTON_EVENT4])
	{ //fps mode - look up
		outAng[0] -= pitchTurnRate*timeMod;
		if (outAng[0] < -90.0f)
		{
			outAng[0] = -90.0f;
		}
	}

	if (obj->plObj->hasLastClAngles)
	{
		for (int n = 0; n < 3; n++)
		{
			outAng[n] += (obj->plObj->clAngles[n]-obj->plObj->lastClAngles[n]);
			obj->plObj->lastClAngles[n] = obj->plObj->clAngles[n];
		}
		if (outAng[0] > 90.0f)
		{
			outAng[0] = 90.0f;
		}
		else if (outAng[0] < -90.0f)
		{
			outAng[0] = -90.0f;
		}
	}

	ObjPlayer_AnalogLook(obj, timeMod, outAng);
}

//can the player start a new action?
bool ObjPlayer_CanTakeAction(gameObject_t *obj)
{
	if (obj->curAnim <= HUMANIM_FALL || obj->curAnim == HUMANIM_LAND)
	{
		return true;
	}
	if (obj->curAnim >= PLANIM_STRAFE_LEFT && obj->curAnim <= PLANIM_TARG_BACK)
	{
		return true;
	}

	return false;
}

//is the player in an attack chain?
bool ObjPlayer_InChain(gameObject_t *obj)
{
	if (obj->plObj->chainNum > 0 && obj->curAnim >= PLANIM_PUNCH_R1)
	{
		return true;
	}
	return false;
}

//is the player transitioning to or from a chain?
bool ObjPlayer_InChainTransition(gameObject_t *obj)
{
	if (!ObjPlayer_InChain(obj))
	{
		return false;
	}

	gameAnim_t *curAnim = obj->animTable+obj->curAnim;
	playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];

	if (ch->anim == PLANIM_SWORD_1 && obj->curAnim == ch->anim &&
		obj->net.frame >= curAnim->endFrame-ch->chainFrames && obj->net.frame < curAnim->endFrame)
	{ //special case
		return false;
	}

	if (obj->curAnim != ch->anim ||
		obj->net.frame <= curAnim->startFrame ||
		obj->net.frame >= curAnim->endFrame-ch->chainFrames)
	{
		return true;
	}

	return false;
}

//go into the next part of a chain attack
void ObjPlayer_ProgressChain(gameObject_t *obj)
{
	if (!obj->plObj->chainNum)
	{ //not in a chain
		AI_StartAnim(obj, HUMANIM_IDLE, false);
		return;
	}

	if (obj->plObj->nextChain == PLCHAIN_KICK_AIR1 ||
		obj->plObj->nextChain == PLCHAIN_KICK_AIR2 ||
		obj->plObj->nextChain == PLCHAIN_KICK_AIR_FLIP ||
		obj->plObj->nextChain == PLCHAIN_KICK_DIVE ||
		obj->plObj->nextChain == PLCHAIN_PUNCH_AIRCRUSH ||
		obj->plObj->nextChain == PLCHAIN_PUNCH_AIRCRUSHSUPER ||
		obj->plObj->nextChain == PLCHAIN_SWORD_AIR)
	{
		if (obj->onGround)
		{
			obj->plObj->nextChain = 0;
		}
	}

	gameAnim_t *curAnim = obj->animTable+obj->curAnim;
	playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
	if (obj->curAnim == ch->anim &&
		obj->net.frame >= curAnim->startFrame &&
		obj->net.frame <= curAnim->endFrame &&
		obj->net.frame >= curAnim->endFrame-ch->chainFrames)
	{
		if (obj->plObj->nextChain > 0)
		{
			obj->plObj->chainNum = obj->plObj->nextChain;
			obj->plObj->nextChain = 0;
			obj->atkLastSeq++;

			playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
			AI_StartAnim(obj, ch->anim, true);
		}
		else if (obj->net.frame == curAnim->endFrame && !obj->animRestart)
		{ //didn't chain by the last frame, too late
			if (obj->plObj->chainNum != PLCHAIN_PUNCH_AIRCRUSH &&
				obj->plObj->chainNum != PLCHAIN_PUNCH_AIRCRUSHSUPER &&
				obj->plObj->chainNum != PLCHAIN_KICK_DIVE &&
				obj->plObj->chainNum != PLCHAIN_SWORD_AIR)
			{ //special-case for air attacks
				if (obj->plObj->chainNum == PLCHAIN_KICK_AIR_FLIP && !obj->onGround)
				{
					AI_StartAnim(obj, PLANIM_AIR_RECOVER, true);
				}
				else
				{
					AI_StartAnim(obj, HUMANIM_IDLE, false);
				}
				obj->plObj->chainNum = 0;
				obj->plObj->nextChain = 0;
			}
		}
	}
}

//allow moving?
bool ObjPlayer_NonMovingAnim(gameObject_t *obj)
{
	if (AI_NonMovingAnim(obj))
	{
		return true;
	}

	if (obj->curAnim == PLANIM_PUNCH_AIRCRUSH_LAND ||
		obj->curAnim == PLANIM_PUNCH_AIRCRUSH_LANDSUPER ||
		obj->curAnim == PLANIM_SWORD_AIR_LAND ||
		obj->curAnim == PLANIM_DIVE_LAND ||
		obj->curAnim == PLANIM_ITEM_USE ||
		obj->curAnim == PLANIM_RISE ||
		obj->curAnim == PLANIM_SEPH_STABBED ||
		obj->curAnim == PLANIM_SEPH_BEATDOWN ||
		obj->curAnim == PLANIM_SEPH_AERITHPUNCH ||
		obj->curAnim == PLANIM_SEPH_BEATDOWNLOOP ||
		obj->curAnim == PLANIM_DEATH ||
		obj->curAnim == PLANIM_DEATH_LAND ||
		obj->curAnim == PLANIM_DEATH_MP ||
		obj->curAnim == PLANIM_DEATH_LAND_MP ||
		obj->curAnim >= PLANIM_CIN_IDLE_1)
	{
		return true;
	}

	if (obj->plObj->ride)
	{
		return true;
	}

	return false;
}

//chain attacks
void ObjPlayer_ChainInput(gameObject_t *obj, const invItemDef_t *eqItem, float *moveVec)
{
	if (obj->curAnim == HUMANIM_JUMP || obj->curAnim == PLANIM_WALLJUMP || obj->curAnim == PLANIM_DIVE_BOUNCE ||
		obj->curAnim == PLANIM_GETUP_FLIP ||
		obj->curAnim == PLANIM_BACKFLIP || obj->curAnim == PLANIM_LUNGE || obj->curAnim == PLANIM_TACKLE ||
		obj->curAnim == PLANIM_AIR_RECOVER /*|| obj->plObj->gunFireTime >= g_curTime*/)
	{
		return;
	}

	if (ObjPlayer_NonMovingAnim(obj))
	{
		obj->plObj->chainNum = 0;
		obj->plObj->nextChain = 0;
		return;
	}

	if (ObjPlayer_InChain(obj) && obj->plObj->nextChain)
	{ //already have a planned chain
		return;
	}

	float turnVec[3];
	Math_AngleVectors(obj->net.ang, turnVec, 0, 0);
	float d = obj->plObj->clButtons[BUTTON_EVENT7] ? Math_DotProduct(moveVec, turnVec) : 0.0f;

	if (ObjPlayer_ButtonTapped(obj, BUTTON_ACTION))
	{ //punch
		if (ObjPlayer_InChain(obj))
		{
			playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
			int whichChain;
			if (d < -0.7f)
			{
				whichChain = ch->chainPunchBack;
			}
			else if (d > 0.7f)
			{
				whichChain = ch->chainPunchFwd;
			}
			else
			{
				whichChain = ch->chainPunch;
			}

			if (whichChain == -2)
			{
				whichChain = ch->chainPunch;
			}

			if (whichChain == PLCHAIN_PUNCH_AIRCRUSH || whichChain == PLCHAIN_PUNCH_AIRCRUSHSUPER)
			{
				if (!ObjPlayer_Ability(obj, PLABIL_GROUNDCRUSH))
				{ //don't chain into aircrush if i don't have it
					whichChain = 0;
				}
			}

			if (whichChain > 0)
			{ //valid chain for next move
				obj->plObj->nextChain = whichChain;
			}
			else
			{ //messed up, can't chain now
				obj->plObj->nextChain = -1;
			}
		}
		else
		{
			if (obj->onGround && d > 0.7f && ObjPlayer_Ability(obj, PLABIL_THRUST))
			{ //special-case - lunge
				obj->plObj->lungeTime = g_curTime + Util_LocalDelay(obj, 500);
				ObjSound_Create(obj->net.pos, "assets/sound/cb/lunge.wav", 1.0f, -1);
				AI_StartAnim(obj, PLANIM_LUNGE, true);
			}
			else
			{
				if (obj->onGround)
				{
					obj->plObj->chainNum = PLCHAIN_PUNCH_R1;
				}
				else
				{
					if (ObjPlayer_Ability(obj, PLABIL_GROUNDCRUSH))
					{
						obj->plObj->chainNum = PLCHAIN_PUNCH_AIRCRUSH;
					}
					else
					{
						if (Math_VecLen(moveVec) > 0.0f && !obj->plObj->clButtons[BUTTON_EVENT7])
						{
							float ang[3];
							Math_VecToAngles(moveVec, ang);
							obj->net.ang[PITCH] = 0.0f;
							obj->net.ang[YAW] = ang[YAW];
							obj->net.ang[ROLL] = 0.0f;
						}
						float fwd[3];
						Math_AngleVectors(obj->net.ang, fwd, 0, 0);
						fwd[2] -= 1.0f;
						Math_VecNorm(fwd);
						obj->net.vel[0] = fwd[0]*800.0f;
						obj->net.vel[1] = fwd[1]*800.0f;
						obj->net.vel[2] = fwd[2]*800.0f;
						obj->plObj->chainNum = PLCHAIN_KICK_DIVE;
					}
				}
				obj->plObj->nextChain = 0;
				obj->atkLastSeq++;

				playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
				AI_StartAnim(obj, ch->anim, true);
			}
		}
	}
	else if (ObjPlayer_ButtonTapped(obj, BUTTON_EVENT1))
	{ //kick
		if (ObjPlayer_InChain(obj))
		{
			playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
			int whichChain;
			if (d < -0.7f)
			{
				whichChain = ch->chainKickBack;
			}
			else if (d > 0.7f)
			{
				whichChain = ch->chainKickFwd;
			}
			else
			{
				whichChain = ch->chainKick;
			}

			if (whichChain == PLCHAIN_KICK_AIR_FLIP && !ObjPlayer_Ability(obj, PLABIL_METEORKICK))
			{ //cannot make this chain without the meteor kick ability
				whichChain = -2;
			}

			if (whichChain == -2)
			{
				whichChain = ch->chainKick;
			}

			if ((whichChain == PLCHAIN_COMBO_1 || whichChain == PLCHAIN_COMBO_2) &&
				!ObjPlayer_Ability(obj, PLABIL_FLURRY))
			{ //cannot make this chain without the flurry ability
				whichChain = 0;
			}

			if (whichChain > 0)
			{ //valid chain for next move
				obj->plObj->nextChain = whichChain;
			}
			else
			{ //messed up, can't chain now
				obj->plObj->nextChain = -1;
			}
		}
		else
		{
			if (obj->onGround)
			{
				if (d < -0.7f && ObjPlayer_Ability(obj, PLABIL_FLIPKICK))
				{
					obj->plObj->chainNum = PLCHAIN_KICK_BACKFLIP;
				}
				else
				{
					obj->plObj->chainNum = PLCHAIN_KICK_R1;
				}
			}
			else
			{
				if (moveVec[0] == 0.0f &&
					moveVec[1] == 0.0f &&
					moveVec[2] == 0.0f)
				{
					obj->plObj->chainNum = PLCHAIN_KICK_AIR1;
				}
				else
				{
					obj->plObj->chainNum = PLCHAIN_KICK_DIVE;
					if (!obj->plObj->clButtons[BUTTON_EVENT7])
					{
						float ang[3];
						Math_VecToAngles(moveVec, ang);
						obj->net.ang[PITCH] = 0.0f;
						obj->net.ang[YAW] = ang[YAW];
						obj->net.ang[ROLL] = 0.0f;
					}
					float fwd[3];
					Math_AngleVectors(obj->net.ang, fwd, 0, 0);
					fwd[2] -= 1.0f;
					Math_VecNorm(fwd);
					obj->net.vel[0] = fwd[0]*800.0f;
					obj->net.vel[1] = fwd[1]*800.0f;
					obj->net.vel[2] = fwd[2]*800.0f;
				}
			}
			obj->plObj->nextChain = 0;
			obj->atkLastSeq++;

			playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
			AI_StartAnim(obj, ch->anim, true);
		}
	}
	else if (eqItem && eqItem->itemBehavior == INVITEM_BEHAVIOR_SWORD &&
		ObjPlayer_ButtonTapped(obj, BUTTON_EVENT9))
	{ //slash
		if (ObjPlayer_InChain(obj))
		{
			playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
			int whichChain;
			if (d < -0.7f)
			{
				whichChain = ch->chainSwordBack;
			}
			else if (d > 0.7f)
			{
				whichChain = ch->chainSwordFwd;
			}
			else
			{
				whichChain = ch->chainSword;
			}

			if (whichChain == -2)
			{
				whichChain = ch->chainSword;
			}

			if (whichChain > 0)
			{ //valid chain for next move
				obj->plObj->nextChain = whichChain;
			}
			else
			{ //messed up, can't chain now
				obj->plObj->nextChain = -1;
			}
		}
		else
		{
			if (obj->onGround)
			{
				obj->plObj->chainNum = PLCHAIN_SWORD_1;
			}
			else
			{
				obj->plObj->chainNum = PLCHAIN_SWORD_AIR;
			}
			obj->plObj->nextChain = 0;
			obj->atkLastSeq++;

			playerChain_t *ch = &g_playerChains[obj->plObj->chainNum];
			AI_StartAnim(obj, ch->anim, true);
		}
	}
}

//is the player in an animation that dictates the angles should continue adjusting?
bool ObjPlayer_AnglingAnim(gameObject_t *obj)
{
	if (obj->curAnim == PLANIM_WALLJUMP)
	{
		return true;
	}
	return false;
}

//do not adjust yaw during these anims
bool ObjPlayer_NonAnglingAnim(gameObject_t *obj)
{
	if (AI_NonAnglingAnim(obj))
	{
		return true;
	}

	if (obj->curAnim == PLANIM_DODGE_LEFT || obj->curAnim == PLANIM_DODGE_RIGHT ||
		obj->curAnim == PLANIM_BACKFLIP || obj->curAnim == PLANIM_LUNGE || obj->curAnim == PLANIM_TACKLE ||
		obj->curAnim == PLANIM_LUNGE_FINISH || obj->curAnim == PLANIM_TACKLE_FINISH || obj->curAnim == PLANIM_PUNCH_AIRCRUSH ||
		obj->curAnim == PLANIM_SWORD_AIR || obj->curAnim == PLANIM_SWORD_AIR_LAND ||
		obj->curAnim == PLANIM_PUNCH_AIRCRUSHSUPER || obj->curAnim == PLANIM_PUNCH_AIRCRUSH_LANDSUPER ||
		obj->curAnim == PLANIM_PUNCH_AIRCRUSH_LAND || obj->curAnim == PLANIM_KICK_BACKFLIP ||
		obj->curAnim == PLANIM_KICK_DIVE || obj->curAnim == PLANIM_RISE || obj->curAnim == PLANIM_SEPH_STABBED ||
		obj->curAnim == PLANIM_SEPH_AERITHPUNCH || obj->curAnim == PLANIM_DEATH_MP || obj->curAnim == PLANIM_DEATH_LAND_MP ||
		obj->curAnim == PLANIM_DEATH || obj->curAnim == PLANIM_SEPH_BEATDOWN || obj->curAnim == PLANIM_SEPH_BEATDOWNLOOP ||
		obj->curAnim == PLANIM_DIVE_LAND || obj->curAnim == PLANIM_ITEM_USE || obj->curAnim == PLANIM_DEATH_LAND ||
		obj->curAnim == PLANIM_DIVE_BOUNCE || obj->curAnim == PLANIM_GETUP_FLIP)
	{
		return true;
	}

	return false;
}

//see if a target is valid
bool ObjPlayer_TargetValid(gameObject_t *obj, gameObject_t *target)
{
	if (obj == target)
	{
		return false;
	}
	if (obj->health <= 0 || !obj->hurtable)
	{
		return false;
	}

	if (!target || !target->inuse)
	{
		return false;
	}

	if (!target->hurtable || target->health <= 0)
	{
		return false;
	}

	if (!Util_ValidEnemies(obj, target))
	{
		return false;
	}

	float d[3];
	float targCenter[3], plCenter[3];
	Util_GameObjectCenter(target, targCenter);
	Util_GameObjectCenter(obj, plCenter);

	//check distance
	Math_VecSub(targCenter, plCenter, d);
	float dist = Math_VecLen(d);
	if (dist > 12000.0f)//65536.0f)
	{
		return false;
	}

	if (target->aiObj && target->aiObj->nonActive && dist > 3500.0f)
	{
		return false;
	}

	//check tree visibility
	if (!g_sharedFn->Coll_GeneralVisibilityWithBounds(targCenter, plCenter, target->radius, target->spawnMins, target->spawnMaxs))
	{
		return false;
	}

	//finally resort to a line trace
	collObj_t col;
	collOpt_t opt;
	memset(&opt, 0, sizeof(opt));
	opt.ignoreBoxModels = true;
	g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, plCenter, targCenter, 0.0f, NULL, &opt);
	if (col.hit && col.hitObjectIndex != target->net.index)
	{
		return false;
	}

	return true;
}

//look for target objects
gameObject_t *ObjPlayer_FindTarget(gameObject_t *obj)
{
	float bestDist;
	gameObject_t *best = NULL;
	float lookDir[3];
	Math_AngleVectors(obj->net.ang, lookDir, 0, 0);
	for (int i = 0; i < g_gameObjectSlots; i++)
	{
		gameObject_t *other = &g_gameObjects[i];
		if (ObjPlayer_TargetValid(obj, other))
		{
			float d[3];
			float targCenter[3], plCenter[3];
			Util_GameObjectCenter(other, targCenter);
			Util_GameObjectCenter(obj, plCenter);
			Math_VecSub(targCenter, plCenter, d);
			float dist = Math_VecNorm(d);
			float dp = Math_DotProduct(lookDir, d);
			if (dp < -0.75f)
			{
				dp = -0.75f;
			}
			//factor distance by the angle to the thing
			dp = 1.0f + (1.0f-dp);
			dist *= dp;

			if (!best || dist < bestDist)
			{
				best = other;
				bestDist = dist;
			}
		}
	}

	return best;
}

//some animations/frames will adjust velocity
bool ObjPlayer_ApplyAnimationVelocity(gameObject_t *obj, float timeMod)
{
	float velFwd = 0.0f;
	float velRight = 0.0f;

	switch (obj->curAnim)
	{
	case PLANIM_DODGE_LEFT:
		if (obj->onGround)
		{
			if (obj->net.frame >= 496 && obj->net.frame <= 497)
			{
				velRight = -2.5f;
			}
			else
			{
				velRight = -1.5f;
			}
		}
		break;
	case PLANIM_DODGE_RIGHT:
		if (obj->onGround)
		{
			if (obj->net.frame >= 502 && obj->net.frame <= 503)
			{
				velRight = 2.5f;
			}
			else
			{
				velRight = 1.5f;
			}
		}
		break;
	case PLANIM_LUNGE:
	case PLANIM_TACKLE:
		if (obj->onGround && obj->plObj->lungeTime)
		{
			velFwd = 3.0f;
		}
		break;
	case PLANIM_LUNGE_FINISH:
		if (obj->onGround && obj->net.frame == 518)
		{
			velFwd = 2.5f;
		}
		break;
	case PLANIM_KICK_DIVE:
		velFwd = 0.5f;
		break;
	default:
		break;
	}

	if (velFwd != 0.0f || velRight != 0.0f)
	{
		const float animVelFactor = 200.0f*timeMod;
		velFwd *= animVelFactor;
		velRight *= animVelFactor;

		float fwd[3], right[3];
		Math_AngleVectors(obj->net.ang, fwd, right, 0);
		obj->net.vel[0] += (fwd[0]*velFwd + right[0]*velRight);
		obj->net.vel[1] += (fwd[1]*velFwd + right[1]*velRight);
		obj->net.vel[2] += (fwd[2]*velFwd + right[2]*velRight);
		return true;
	}

	return AI_ApplyAnimationVelocity(obj, timeMod);
}

//set render effects for animations
void ObjPlayer_ApplyAnimationEffects(gameObject_t *obj, float timeMod)
{
	gameAnim_t *curAnim = obj->animTable+obj->curAnim;
	int numFrames = curAnim->endFrame-curAnim->startFrame;
	float animPerc = (!numFrames || obj->net.frame < curAnim->startFrame || obj->net.frame > curAnim->endFrame) ? 0.0f : (float)(obj->net.frame-curAnim->startFrame)/(float)numFrames;

	AI_ApplyAnimationEffects(obj, timeMod);

	obj->net.renderEffects &= ~FXFL_ITEMSPARKLE;

	switch (obj->curAnim)
	{
	case PLANIM_ITEM_USE:
		if (obj->net.frame >= 124 && obj->net.frame <= 126)
		{
			obj->net.renderEffects |= FXFL_ITEMSPARKLE;
		}
		break;
	case PLANIM_DODGE_LEFT:
	case PLANIM_DODGE_RIGHT:
		if (animPerc >= 0.2f && animPerc <= 0.8f)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL2;
		}
		break;
	case PLANIM_LUNGE:
	case PLANIM_TACKLE:
		obj->net.renderEffects |= FXFL_MODELTRAIL;
		break;
	case PLANIM_PUNCH_UPPERCUT:
		if (animPerc > 0.0f && obj->net.frame >= 472)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL;
		}
		break;
	case PLANIM_SWORD_4:
		if (animPerc > 0.0f && obj->net.frame >= 704)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL;
		}
		break;
	case PLANIM_KICK_BACKFLIP:
		if (obj->net.vel[2] > 200.0f)
		{
			obj->net.renderEffects |= FXFL_MODELTRAIL2;
		}
		break;
	case PLANIM_PUNCH_AIRCRUSH:
	case PLANIM_PUNCH_AIRCRUSHSUPER:
	case PLANIM_SWORD_AIR:
		obj->net.renderEffects |= FXFL_MODELTRAIL2;
		break;
	default:
		break;
	}
}

//fire gun
void ObjPlayer_FireGun(gameObject_t *obj)
{
	float pos[3] = {0.0f, -50.0f, -245.0f};
	float ang[3] = {0.0f, 0.0f, -1.0f};
	modelMatrix_t boneMat;
	if (g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b32", &boneMat))
	{
		float t[3];
		Math_VecCopy(pos, t);
		Math_TransformPointByMatrix(&boneMat, t, pos);

		if (obj->plObj->targetObj && obj->plObj->targetObj->inuse)
		{ //fire at them
			float c[3];
			Util_GameObjectCenter(obj->plObj->targetObj, c);
			Math_VecSub(c, boneMat.o, ang);

			//add some spread
			float maxSpread = 0.03f;
			float r;
			r = (float)(rand()%16384)/16384.0f;
			float spreadX = -maxSpread + r*maxSpread*2.0f;
			r = (float)(rand()%16384)/16384.0f;
			float spreadY =  -maxSpread + r*maxSpread*2.0f;
			Math_VecNorm(ang);
			float right[3], up[3];
			Math_AxisForNormal(ang, 0, right, up);
			Math_VecScale(right, spreadX);
			Math_VecScale(up, spreadY);
			Math_VecAdd(ang, right, ang);
			Math_VecAdd(ang, up, ang);
		}
		else
		{ //fire where the barrel is pointed
			Math_VecCopy(ang, t);
			Math_TransformPointByMatrix(&boneMat, t, ang);
			Math_VecSub(ang, boneMat.o, ang);
		}
	}
	Math_VecNorm(ang);

	int atk = obj->atkType;
	obj->atkType = ATKTYPE_BULLET;
	bulletDesc_t plBullet =
	{
		ObjPlayer_StrMod(obj, 4),
		"melee/impactslash",
		"assets/sound/cb/bulflesh.wav",
		"bulletspark",
		"assets/sound/cb/bullet.wav"
	};
	Util_FireBullet(obj, pos, ang, &plBullet);
	obj->atkType = atk;

	if (!obj->onGround && obj->net.vel[2] < 0.0f)
	{
		obj->net.vel[2] += 300.0f;
		if (obj->net.vel[2] > 100.0f)
		{
			obj->net.vel[2] = 100.0f;
		}
	}

	Math_VecToAngles(ang, ang);
	ObjParticles_Create("playerfire", pos, ang, -1);
	ObjSound_Create(pos, "assets/sound/cb/glock.wav", 1.0f, -1);
}

//manage item use
void ObjPlayer_CheckItemUse(gameObject_t *obj, float timeMod)
{
	if (obj->plObj->equippedItem <= 0)
	{
		return;
	}

	if (obj->curAnim == HUMANIM_JUMP || obj->curAnim == PLANIM_WALLJUMP ||
		obj->curAnim == PLANIM_BACKFLIP || obj->curAnim == PLANIM_AIR_RECOVER || obj->curAnim == PLANIM_LUNGE || obj->curAnim == PLANIM_TACKLE ||
		obj->curAnim == PLANIM_DIVE_BOUNCE || obj->curAnim == PLANIM_GETUP_FLIP || ObjPlayer_NonMovingAnim(obj) || ObjPlayer_InChain(obj))
	{
		return;
	}
	
	if (!ObjPlayer_ButtonTapped(obj, BUTTON_EVENT9))
	{
		return;
	}

	if (obj->plObj->noItemTime > g_curTime)
	{
		return;
	}

	const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
	if (!item || !item->usable)
	{
		if (item && item->itemBehavior == INVITEM_BEHAVIOR_GUN)
		{
			ObjPlayer_FireGun(obj);
			obj->plObj->gunFireTime = g_curTime+Util_LocalDelay(obj, 500);
			if (obj->plObj->gunFirePose > 0.9f && obj->onGround)
			{
				obj->plObj->gunFirePose -= 0.5f;
			}
		}
		return;
	}
	AI_StartAnim(obj, PLANIM_ITEM_USE, true);
}

//manage charge
void ObjPlayer_CheckCharge(gameObject_t *obj, float timeMod)
{
	if (obj->plObj->chargeDrainTime > g_curTime)
	{
		return;
	}

	obj->plObj->chargeDrainTime = g_curTime + Util_LocalDelay(obj, 50);
	if (obj->plObj->chargeMeter == 0)
	{
		obj->plObj->chargeDrain = 0.0f;
	}
	else
	{
		float cap = 64.0f;
		float f = 0.1f + ((float)obj->plObj->chargeMeter/4095.0f)*1.25f;
		obj->plObj->chargeDrain += f;//timeMod*f;
		if (obj->plObj->chargeDrain > cap)
		{
			obj->plObj->chargeDrain = cap;
		}
	}

	if (obj->plObj->chargeDrain >= 1.0f)
	{
		int c = (int)obj->plObj->chargeDrain;
		obj->plObj->chargeMeter -= c;
		if (obj->plObj->chargeMeter < 0)
		{
			obj->plObj->chargeMeter = 0;
		}
	}

	if (obj->plObj->makoCharges < obj->plObj->plData.maxMakoCharges && obj->health > 0)
	{
		if (g_timeType == TIMETYPE_TIMEATTACK)
		{
			obj->plObj->chargeToMakoPower += obj->plObj->chargeMeter*0.1f;
		}
		else
		{
			obj->plObj->chargeToMakoPower += obj->plObj->chargeMeter;
		}
		if (obj->plObj->chargeToMakoPower >= 200000)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/lstreamint.wav", 1.0f, -1);
			obj->plObj->makoCharges++;
			obj->plObj->chargeToMakoPower -= 200000;
		}
	}
	else
	{
		obj->plObj->chargeToMakoPower = 0;
	}
}

//check charge attacks
int ObjPlayer_CheckChargeAttack(gameObject_t *obj, float timeMod)
{
	bool hadAura = !!(obj->net.renderEffects & FXFL_AURA);
	obj->net.renderEffects &= ~FXFL_AURA;
	if (!ObjPlayer_Ability(obj, PLABIL_MAKOFLAME) &&
		!ObjPlayer_Ability(obj, PLABIL_MAKOTIME) &&
		!ObjPlayer_Ability(obj, PLABIL_MAKOHEAL) &&
		!ObjPlayer_Ability(obj, PLABIL_MAKOAURA))
	{ //no charge abilities
		return -1;
	}
	if ((g_timeType != TIMETYPE_NORMAL && g_timeType != TIMETYPE_TONBERRY && !g_mpVersusMode) || obj->plObj->fistOfFlames || obj->plObj->explosiveBlows ||
		obj->plObj->healingAura || ObjPlayer_NonMovingAnim(obj) || obj->plObj->makoCharges <= 0)
	{
		return -1;
	}

	int clearInput = -1;
	if (obj->plObj->clButtons[BUTTON_EVENT8])
	{
		obj->net.renderEffects |= FXFL_AURA;
		if (!hadAura)
		{
			float c[3];
			float ang[3] = {0.0f, 0.0f, 0.0f};
			Util_GameObjectCenter(obj, c);
			c[2] += 100.0f;
			ObjParticles_Create("general/makochargeup", c, ang, -1);
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeup.wav", 1.0f, -1);
		}
		if (ObjPlayer_ButtonTapped(obj, BUTTON_JUMP) && ObjPlayer_Ability(obj, PLABIL_MAKOTIME))
		{ //time attack
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargetime.wav", 1.0f, -1);
			g_timeType = TIMETYPE_TIMEATTACK;
			obj->plObj->timeAttackTime = ObjPlayer_Ability(obj, PLABIL_CHARGEFORCE) ? g_curTime+4000 : g_curTime+2000;
			clearInput = BUTTON_JUMP;
			obj->plObj->makoCharges--;
		}
		else if (ObjPlayer_ButtonTapped(obj, BUTTON_ACTION) && ObjPlayer_Ability(obj, PLABIL_MAKOFLAME))
		{ //fist of flames
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeflame.wav", 1.0f, -1);
			obj->plObj->fistOfFlames = ObjPlayer_Ability(obj, PLABIL_CHARGEFORCE) ? g_curTime+20000 : g_curTime+10000;
			clearInput = BUTTON_ACTION;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 0.4f;
			c[2] = 0.1f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
			obj->plObj->makoCharges--;
		}
		else if (ObjPlayer_ButtonTapped(obj, BUTTON_EVENT1) && ObjPlayer_Ability(obj, PLABIL_MAKOAURA))
		{ //explosive blows
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeaura.wav", 1.0f, -1);
			obj->plObj->explosiveBlows = ObjPlayer_Ability(obj, PLABIL_CHARGEFORCE) ? g_curTime+20000 : g_curTime+10000;
			clearInput = BUTTON_EVENT1;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 1.0f;
			c[2] = 0.5f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
			obj->plObj->makoCharges--;
		}
		else if (ObjPlayer_ButtonTapped(obj, BUTTON_EVENT9) && ObjPlayer_Ability(obj, PLABIL_MAKOHEAL))
		{ //healing aura
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeheal.wav", 1.0f, -1);
			obj->plObj->healingAura = ObjPlayer_Ability(obj, PLABIL_CHARGEFORCE) ? g_curTime+20000 : g_curTime+10000;
			obj->plObj->healingTime = g_curTime+500;
			clearInput = BUTTON_EVENT9;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 0.25f;
			c[1] = 1.0f;
			c[2] = 0.25f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
			obj->plObj->makoCharges--;
		}
	}

	if (clearInput != -1)
	{
		obj->plObj->clButtons[clearInput] = 0;
		for (int i = 0; i < obj->plObj->numButtonsBuffered; i++)
		{
			obj->plObj->clButtonsBackBuffer[i][clearInput] = 0;
		}
	}

	return clearInput;
}

//effects
void ObjPlayer_CheckEffects(gameObject_t *obj, float timeMod)
{
	AI_CheckEffects(obj, timeMod);

	if (obj->plObj->fistOfFlames)
	{
		obj->net.renderEffects |= FXFL_FLAMES;
		if (obj->plObj->fistOfFlames < g_curTime)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeoff.wav", 1.0f, -1);
			obj->net.renderEffects &= ~FXFL_FLAMES;
			obj->plObj->fistOfFlames = 0;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 0.4f;
			c[2] = 0.1f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
	}

	if (obj->plObj->explosiveBlows)
	{
		obj->net.renderEffects |= FXFL_AURA2;
		if (ObjPlayer_InChain(obj))
		{
			obj->net.strIndexC = g_sharedFn->Common_ServerString(g_plDamageBones[g_chainDamage[obj->plObj->chainNum].dmgBone].boneName);
			obj->net.renderEffects |= FXFL_BONEAURA;
		}
		else
		{
			obj->net.renderEffects &= ~FXFL_BONEAURA;
		}

		if (obj->plObj->explosiveBlows < g_curTime)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeoff.wav", 1.0f, -1);
			obj->net.renderEffects &= ~FXFL_AURA2;
			obj->net.renderEffects &= ~FXFL_BONEAURA;
			obj->plObj->explosiveBlows = 0;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 1.0f;
			c[1] = 1.0f;
			c[2] = 0.5f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
	}

	if (obj->plObj->healingAura)
	{
		if (obj->plObj->healingTime < g_curTime)
		{
			obj->health += 50;
			if (obj->health > obj->plObj->plData.maxHealth)
			{
				obj->health = obj->plObj->plData.maxHealth;
			}
			ObjSound_Create(obj->net.pos, "assets/sound/cb/healstep.wav", 1.0f, -1);
			obj->plObj->healingTime = g_curTime + Util_LocalDelay(obj, 1000);
		}

		obj->net.renderEffects |= FXFL_HEALINGAURA;
		if (obj->plObj->healingAura < g_curTime)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeoff.wav", 1.0f, -1);
			obj->net.renderEffects &= ~FXFL_HEALINGAURA;
			obj->plObj->healingAura = 0;
			int *t = (int *)_alloca(sizeof(int)+sizeof(float)*3);
			*t = 600;
			float *c = (float *)(t+1);
			c[0] = 0.25f;
			c[1] = 1.0f;
			c[2] = 0.25f;
			g_sharedFn->Net_SendEventType(CLEV_TRIGGERFLASH, t, sizeof(int)+sizeof(float)*3, -1);
		}
	}
}

//update gun
void ObjPlayer_UpdateGun(gameObject_t *obj, gameObject_t *gun, float timeMod)
{
	Math_VecCopy(obj->net.pos, gun->net.pos);
	gun->net.pos[2] += 100.0f;
}

//update gun pose
void ObjPlayer_UpdateGunPose(gameObject_t *obj, float timeMod)
{
	if (ObjPlayer_InChain(obj) || obj->curAnim == PLANIM_WALLJUMP || obj->curAnim == PLANIM_DIVE_BOUNCE ||
		obj->curAnim == PLANIM_GETUP_FLIP ||
		obj->curAnim == PLANIM_BACKFLIP || obj->curAnim == PLANIM_LUNGE || obj->curAnim == PLANIM_TACKLE ||
		obj->curAnim == PLANIM_AIR_RECOVER)
	{
		obj->plObj->gunFirePose = 0.0f;
	}
	float lerpScale = (obj->plObj->gunFireTime >= g_curTime) ? 0.5f*timeMod : 0.1f*timeMod;
	float desiredPose = (obj->plObj->gunFireTime >= g_curTime) ? 1.0f : 0.0f;
	obj->plObj->gunFirePose += (desiredPose-obj->plObj->gunFirePose)*lerpScale;
	if (obj->plObj->gunFirePose < 0.01f)
	{
		obj->plObj->gunFirePose = 0.0f;
	}
	else if (obj->plObj->gunFirePose > 0.99f)
	{
		obj->plObj->gunFirePose = 1.0f;
	}

	if (obj->plObj->targetObj && obj->plObj->targetObj->inuse)
	{
		float targCenter[3], plCenter[3];
		Util_GameObjectCenter(obj->plObj->targetObj, targCenter);
		Util_GameObjectCenter(obj, plCenter);
		plCenter[2] += 100.0f;
		targCenter[2] += 32.0f;
		float d[3];
		Math_VecSub(targCenter, plCenter, d);
		Math_VecToAngles(d, d);
		d[0] = Math_AngleMod(Math_AngleMod(d[0])-Math_AngleMod(obj->net.ang[0]));
		d[1] = Math_AngleMod(Math_AngleMod(d[1])-Math_AngleMod(obj->net.ang[1]));
		d[2] = Math_AngleMod(Math_AngleMod(d[2])-Math_AngleMod(obj->net.ang[2]));
		for (int i = 0; i < 2; i++)
		{
			if (d[i] > 180.0f)
			{
				d[i] -= 360.0f;
			}
			if (d[i] > 60.0f)
			{
				d[i] = 60.0f;
			}
			else if (d[i] < -60.0f)
			{
				d[i] = -60.0f;
			}
			obj->net.boltOffsets[i] = -d[i];
		}
	}
	else
	{
		obj->net.boltOffsets[0] = 0.0f;
		obj->net.boltOffsets[1] = 0.0f;
	}

	obj->net.boltOffsets[2] = obj->plObj->gunFirePose;
	modelMatrix_t worldBoneMat, mat;
	if (!g_sharedFn->Coll_GetModelBoneMatrix(obj->rcColModel, "b0", &worldBoneMat))
	{
		return;
	}
	modelMatrix_t identityMat ={	{1.0f, 0.0f, 0.0f},
									{0.0f, 1.0f, 0.0f},
									{0.0f, 0.0f, 1.0f},
									{0.0f, 0.0f, 0.0f}	};
	Math_VecCopy(worldBoneMat.o, mat.o);
	Math_TransformPointByMatrix(&worldBoneMat, identityMat.x3, mat.x1);
	Math_VecSub(mat.x1, worldBoneMat.o, mat.x1);
	Math_TransformPointByMatrix(&worldBoneMat, identityMat.x1, mat.x2);
	Math_VecSub(mat.x2, worldBoneMat.o, mat.x2);
	Math_TransformPointByMatrix(&worldBoneMat, identityMat.x2, mat.x3);
	Math_VecSub(mat.x3, worldBoneMat.o, mat.x3);
	Math_VecNorm(mat.x1);
	Math_VecNorm(mat.x2);
	Math_VecNorm(mat.x3);
	Math_RotateMatrix(&mat, -90.0f, 1.0f, 0.0f, 0.0f);
	Math_RotateMatrix(&mat, 90.0f, 0.0f, 1.0f, 0.0f);

	Math_RotateMatrix(&mat, obj->net.boltOffsets[0], 1.0f, 0.0f, 0.0f);
	Math_RotateMatrix(&mat, obj->net.boltOffsets[1], 0.0f, 1.0f, 0.0f);

	Math_VecScale(mat.x1, 0.6f);
	Math_VecScale(mat.x2, 0.6f);
	Math_VecScale(mat.x3, 0.6f);
	g_sharedFn->Coll_SetBoneOverride(obj->rcColModel, "b31", &mat, BONEOFS_ABSOLUTE, obj->net.boltOffsets[2], NULL, 0);
	mat = identityMat;
	g_sharedFn->Coll_SetBoneOverride(obj->rcColModel, "b32", &mat, BONEOFS_PARENTABS, obj->net.boltOffsets[2], NULL, 0);
}

//mount ride
void ObjPlayer_MountRide(gameObject_t *obj, gameObject_t *ride)
{
	if (obj->plObj->ride)
	{ //already mounted
		return;
	}

	obj->plObj->ride = ride;
	obj->net.solid = 0;
	obj->hurtable = 0;
	LServ_UpdateRClip(obj);
}

//unmount ride
bool ObjPlayer_UnmountRide(gameObject_t *obj, bool force)
{
	collObj_t col;
	bool colValid = false;
	if (!force)
	{
		colValid = true;
		float testPos[3];
		float testDist = 400.0;
		Math_VecCopy(obj->net.pos, testPos);
		testPos[2] = obj->net.pos[2]+1.0f;
		testPos[0] = obj->net.pos[0];
		testPos[1] = obj->net.pos[1]+testDist;
		g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
		if (col.hit || col.containsSolid)
		{
			testPos[0] = obj->net.pos[0];
			testPos[1] = obj->net.pos[1]-testDist;
			g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
			if (col.hit || col.containsSolid)
			{
				testPos[0] = obj->net.pos[0]+testDist;
				testPos[1] = obj->net.pos[1];
				g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
				if (col.hit || col.containsSolid)
				{
					testPos[0] = obj->net.pos[0]-testDist;
					testPos[1] = obj->net.pos[1];
					g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
					if (col.hit || col.containsSolid)
					{
						return false;
					}
				}
			}
		}
	}

	if (colValid)
	{
		Math_VecCopy(col.endPos, obj->net.pos);
	}
	obj->net.solid = 1;
	obj->hurtable = 1;
	obj->plObj->ride = NULL;
	obj->animStartTime = 0;
	obj->animTime = 0;
	obj->net.lerpDuration = 0;
	AI_StartAnim(obj, HUMANIM_IDLE, true);
	LServ_UpdateRClip(obj);
	return true;
}

//get move vec
void ObjPlayer_GetMoveVec(gameObject_t *obj, gameObject_t *cam, float timeMod, float *moveVec,
						  float &move, float &rightMove, bool isFlying)
{
	float moveSpeed = 400.0f*obj->plObj->moveSpeedScale;//48.0f;
	float fwd[3], right[3];
	move = 0.0f;
	rightMove = 0.0f;

	if (ObjPlayer_InChainTransition(obj))
	{
		moveSpeed = 80.0f*obj->plObj->moveSpeedScale;
	}
	else if (obj->plObj->targetObj || obj->plObj->clButtons[BUTTON_EVENT7])
	{ //slow down movement with a target
		moveSpeed = 160.0f*obj->plObj->moveSpeedScale;
	}

	if (!obj->onGround)
	{
		moveSpeed *= 0.085f;
	}

	if (obj->plObj->clButtons[BUTTON_UP])
	{ //up
		move = moveSpeed;
	}
	else if (obj->plObj->clButtons[BUTTON_DOWN])
	{ //down
		move = -moveSpeed;
	}
	if (obj->plObj->clButtons[BUTTON_RIGHT])
	{ //right
		rightMove = moveSpeed;
	}
	else if (obj->plObj->clButtons[BUTTON_LEFT])
	{ //left
		rightMove = -moveSpeed;
	}

	if (move && rightMove)
	{
		move *= 0.75f;
		rightMove *= 0.75f;
	}

	ObjPlayer_AnalogMove(obj, move, rightMove, moveSpeed, timeMod);

	//ObjPlayer_AdjustTurnAngles(obj, timeMod, obj->net.ang);

	//compensate for time
	move *= timeMod;
	rightMove *= timeMod;
	float moveAng[3];
	moveAng[0] = (isFlying) ? cam->net.ang[0] : 0.0f;
	moveAng[1] = cam->net.ang[1];
	moveAng[2] = 0.0f;

	Math_AngleVectors(moveAng, fwd, right, 0);
	moveVec[0] = (fwd[0]*move)+(right[0]*rightMove);
	moveVec[1] = (fwd[1]*move)+(right[1]*rightMove);
	moveVec[2] = (fwd[2]*move)+(right[2]*rightMove);
}

//update sword
void ObjPlayer_UpdateSword(gameObject_t *obj, const invItemDef_t *eqItem, float timeMod)
{
	if (eqItem && eqItem->itemBehavior == INVITEM_BEHAVIOR_SWORD && !g_cinema)
	{
		if (obj->plObj->sword && obj->plObj->swordItem != eqItem)
		{
			LServ_FreeObject(obj->plObj->sword);
			obj->plObj->sword = NULL;
			obj->plObj->swordItem = NULL;
		}

		if (!obj->plObj->sword)
		{
			char *swordName = (eqItem->useFX[0] == '*') ? (&eqItem->useFX[1]) : "obj_masamune";
			obj->plObj->sword = LServ_ObjectFromName(swordName, obj->net.pos, obj->net.ang, 0, 0);
			if (obj->plObj->sword)
			{
				gameObject_t *sword = obj->plObj->sword;
				sword->net.owner = obj->net.index;
				obj->plObj->swordItem = eqItem;
			}
		}

		if (obj->plObj->sword)
		{
			bool holster = true;
			if (obj->net.frame >= 685 && obj->net.frame <= 712)
			{
				holster = false;
			}

			gameObject_t *sword = obj->plObj->sword;
			char *boneName = (holster) ? "b1" : "b32";
			float ofs[3];
			if (holster)
			{
				if (sword->net.renderEffects & FXFL_BLADETRAIL)
				{
					if (sword->swordObj && sword->swordObj->numHolsterSounds > 0)
					{
						int snd = sword->swordObj->holsterSounds[rand()%sword->swordObj->numHolsterSounds];
						ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
					}
					else
					{
						ObjSound_Create(obj->net.pos, "assets/sound/cb/swdre.wav", 1.0f, -1);
					}
					sword->net.renderEffects &= ~FXFL_BLADETRAIL;
					//sword->net.renderEffects2 &= ~FXFL2_BLADETRAIL2;
				}
				sword->net.frame = 1;
				ofs[0] = -80.0f;
				ofs[1] = 5.0f;
				ofs[2] = -120.0f;
			}
			else
			{
				if (!(sword->net.renderEffects & FXFL_BLADETRAIL))
				{
					if (sword->swordObj && sword->swordObj->numUnholsterSounds > 0)
					{
						int snd = sword->swordObj->unholsterSounds[rand()%sword->swordObj->numUnholsterSounds];
						ObjSound_CreateFromIndex(obj->net.pos, snd, 1.0f, -1);
					}
					else
					{
						ObjSound_Create(obj->net.pos, "assets/sound/cb/swdun.wav", 1.0f, -1);
					}
					sword->net.renderEffects |= FXFL_BLADETRAIL;
					if (sword->generalFlag2)
					{ //ein-sof
						sword->net.renderEffects2 |= FXFL2_BLADETRAIL2;
					}
					else
					{
						sword->net.renderEffects2 &= ~FXFL2_BLADETRAIL2;
					}
					if (sword->generalFlag3)
					{ //drain sword
						sword->net.renderEffects2 |= FXFL2_BLADETRAIL3;
					}
					else
					{
						sword->net.renderEffects2 &= ~FXFL2_BLADETRAIL3;
					}
				}
				sword->net.effectLen = -(sword->spareVec[0]+sword->spareVec[1]);
				sword->net.strIndexD = 0;

				if (obj->curAnim == PLANIM_SWORD_AIR || obj->curAnim == PLANIM_SWORD_AIR_LAND)
				{
					sword->net.frame = 3;
				}
				else
				{
					sword->net.frame = 0;
				}
				ofs[0] = 0.0f;
				ofs[1] = 0.0f;
				ofs[2] = -90.0f;
			}
			sword->net.boltOffsets[0] = ofs[0];
			sword->net.boltOffsets[1] = ofs[1];
			sword->net.boltOffsets[2] = ofs[2];
			sword->net.strIndexC = g_sharedFn->Common_ServerString(boneName);
			sword->net.renderEffects |= FXFL_BOLTED1;
			modelMatrix_t mat;
			if (Util_GetObjBolt(obj, boneName, &mat, ofs))
			{
				Math_VecCopy(mat.o, sword->net.pos);
				Math_MatToAngles2(sword->net.ang, &mat);
			}
		}
	}
	else if (obj->plObj->sword)
	{
		LServ_FreeObject(obj->plObj->sword);
		obj->plObj->sword = NULL;
	}
}

//update flashlight
void ObjPlayer_UpdateFlashlight(gameObject_t *obj, const invItemDef_t *eqItem, float timeMod)
{
	if (obj->health > 0 && eqItem && eqItem->itemBehavior == INVITEM_BEHAVIOR_FLASHLIGHT &&
		!g_cinema)
	{
		if (obj->plObj->flashlight && obj->plObj->flashlightItem != eqItem)
		{
			LServ_FreeObject(obj->plObj->flashlight);
			obj->plObj->flashlight = NULL;
			obj->plObj->flashlightItem = NULL;
		}

		if (!obj->plObj->flashlight)
		{
			char *flashlightName = (eqItem->useFX[0] == '*') ? (&eqItem->useFX[1]) : "obj_flashlight";
			obj->plObj->flashlight = LServ_ObjectFromName(flashlightName, obj->net.pos, obj->net.ang, 0, 0);
			if (obj->plObj->flashlight)
			{
				gameObject_t *flashlight = obj->plObj->flashlight;
				flashlight->net.owner = obj->net.index;
				flashlight->net.strIndexC = g_sharedFn->Common_ServerString("b1");
				flashlight->net.boltOffsets[0] = 0.0f;
				flashlight->net.boltOffsets[1] = 0.0f;
				flashlight->net.boltOffsets[2] = 0.0f;
				flashlight->net.renderEffects |= FXFL_BOLTED1;
				flashlight->net.renderEffects2 |= FXFL2_FLASHLIGHT;
			}
		}
		if (obj->plObj->flashlight)
		{
			Math_VecCopy(obj->net.pos, obj->plObj->flashlight->net.pos);
			obj->plObj->flashlight->net.pos[2] += 100.0f;
		}
	}
	else if (obj->plObj->flashlight)
	{
		LServ_FreeObject(obj->plObj->flashlight);
		obj->plObj->flashlight = NULL;
	}
}

//teleport to first client
void ObjPlayer_TeleportToPlZero(gameObject_t *obj)
{
	gameObject_t *plf = &g_gameObjects[0];
	if (plf->inuse && plf->plObj)
	{
		collObj_t col;
		float testPos[3];
		float testDist = 250.0;
		Math_VecCopy(plf->net.pos, testPos);
		testPos[2] = plf->net.pos[2]+1.0f;
		testPos[0] = plf->net.pos[0];
		testPos[1] = plf->net.pos[1]+testDist;
		g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
		if (col.hit || col.containsSolid)
		{
			testPos[0] = plf->net.pos[0];
			testPos[1] = plf->net.pos[1]-testDist;
			g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
			if (col.hit || col.containsSolid)
			{
				testPos[0] = plf->net.pos[0]+testDist;
				testPos[1] = plf->net.pos[1];
				g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
				if (col.hit || col.containsSolid)
				{
					testPos[0] = plf->net.pos[0]-testDist;
					testPos[1] = plf->net.pos[1];
					g_sharedFn->Coll_MovementTranslation(obj->rcColModel, &col, testPos, testPos, NULL, NULL);
					if (col.hit || col.containsSolid)
					{ //complete failure, just go inside of them
						Math_VecCopy(plf->net.pos, testPos);
					}
				}
			}
		}

		Math_VecCopy(testPos, obj->net.pos);
		Math_VecCopy(plf->safePos, obj->safePos);
		obj->plObj->camSeeTime = g_curTime;
		obj->plObj->camTeleWarn = false;
	}
}

//frame function
void ObjPlayer_Think(gameObject_t *obj, float timeMod)
{
	gameObject_t *cam = &g_gameObjects[CAM_TRACK_NUM];

	if (obj->plObj->desiredItem != obj->plObj->equippedItem)
	{
		if (obj->curAnim != PLANIM_ITEM_USE || (obj->net.frame <= 131 && obj->net.frame > 128))
		{
			obj->plObj->equippedItem = obj->plObj->desiredItem;
		}
	}

	if (g_runningMultiplayer)
	{
		obj->net.renderEffects2 &= ~FXFL2_PLICON;
		if (!g_cinema)
		{
			collObj_t camCol;
			collOpt_t camOpt;
			memset(&camOpt, 0, sizeof(camOpt));
			camOpt.ignoreBoxModels = true;
			camOpt.ignoreTexNum = 1;
			char *t = "textures/ava/noclip";
			camOpt.ignoreTexList = &t;
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &camCol, obj->net.pos, cam->net.pos, 0.0f, NULL, &camOpt);
			if (camCol.hit)
			{ //can't see camera, put an icon up over me
				obj->net.renderEffects2 |= FXFL2_PLICON;
			}
			else
			{
				obj->plObj->camSeeTime = g_curTime;
				obj->plObj->camTeleWarn = false;
			}
		}
		else
		{
			obj->plObj->camSeeTime = g_curTime;
			obj->plObj->camTeleWarn = false;
		}

		if (obj->net.index > 0)
		{
			gameObject_t *plf = &g_gameObjects[0];
			if (plf->inuse && plf->plObj)
			{
				if (g_curTime > obj->plObj->camSeeTime)
				{
					if ((g_curTime-obj->plObj->camSeeTime) > 5000)
					{ //copy over to player 0
						ObjPlayer_TeleportToPlZero(obj);
					}
					else if ((g_curTime-obj->plObj->camSeeTime) > 3000 &&
						!obj->plObj->camTeleWarn)
					{
						Util_StatusMessageToClient(obj->net.index, "You will be teleported in 2 seconds.");
						obj->plObj->camTeleWarn = true;
					}
				}
			}
		}
	}

	if (g_magicMode == 1 && obj->plObj->magicHealTime < g_curTime)
	{
		int h = (int)(10.0f*timeMod);
		if (h < 1)
		{
			h = 1;
		}
		obj->health += h;
		if (obj->health > obj->plObj->plData.maxHealth)
		{
			obj->health = obj->plObj->plData.maxHealth;
		}
	}

	const invItemDef_t *eqItem = ObjPlayer_GetEquippedItem(obj);
	if (eqItem && eqItem->itemBehavior == INVITEM_BEHAVIOR_GUN && !g_cinema)
	{
		if (obj->plObj->gun && obj->plObj->gunItem != eqItem)
		{
			LServ_FreeObject(obj->plObj->gun);
			obj->plObj->gun = NULL;
			obj->plObj->gunItem = NULL;
		}

		if (!obj->plObj->gun)
		{
			char *gunName = (eqItem->useFX[0] == '*') ? (&eqItem->useFX[1]) : "obj_glock";
			obj->plObj->gun = LServ_ObjectFromName(gunName, obj->net.pos, obj->net.ang, 0, 0);
			if (obj->plObj->gun)
			{
				gameObject_t *gun = obj->plObj->gun;
				gun->net.owner = obj->net.index;
				gun->net.strIndexC = g_sharedFn->Common_ServerString("b32");
				gun->net.boltOffsets[0] = 0.0f;
				gun->net.boltOffsets[1] = 0.0f;
				gun->net.boltOffsets[2] = -100.0f;
				gun->net.renderEffects |= FXFL_BOLTED1;
			}
		}
		if (obj->plObj->gun)
		{
			ObjPlayer_UpdateGunPose(obj, timeMod);
		}
	}
	else if (obj->plObj->gun)
	{
		obj->plObj->gunFirePose = 0.0f;
		obj->plObj->gunFireTime = 0;
		LServ_FreeObject(obj->plObj->gun);
		obj->plObj->gun = NULL;
		g_sharedFn->Coll_SetBoneOverride(obj->rcColModel, "b31", NULL, BONEOFS_ABSOLUTE, 0.0f, NULL, 0);
		g_sharedFn->Coll_SetBoneOverride(obj->rcColModel, "b32", NULL, BONEOFS_PARENTABS, 0.0f, NULL, 0);
	}

	if (obj->plObj->gunFirePose > 0.0f)
	{
		obj->net.renderEffects |= FXFL_GUNARM;
	}
	else
	{
		obj->net.renderEffects &= ~FXFL_GUNARM;
	}

	ObjPlayer_UpdateFlashlight(obj, eqItem, timeMod);

	if (obj->curAnim == PLANIM_SEPH_AERITHPUNCH &&
		obj->net.frame <= 776)
	{
		obj->net.renderEffects2 |= FXFL2_AURAFIST;
	}
	else
	{
		obj->net.renderEffects2 &= ~FXFL2_AURAFIST; 
	}

	if (obj->curAnim == PLANIM_SEPH_AERITHPUNCH &&
		obj->net.frame >= 777 && obj->net.frame <= 784)
	{
		if (!obj->plObj->aerith)
		{
			obj->plObj->aerith = LServ_ObjectFromName("obj_aerithshell", obj->net.pos, obj->net.ang, 0, 0);
		}
		gameObject_t *myAerith = obj->plObj->aerith;
		if (myAerith)
		{
			myAerith->net.frame = 492;
			Math_VecCopy(obj->net.pos, myAerith->net.pos);
			Math_VecCopy(obj->net.ang, myAerith->net.ang);
			//myAerith->net.renderEffects2 |= FXFL2_SKIPDEPTH;
		}
	}
	else if (obj->plObj->aerith)
	{
		if (!obj->plObj->aerith->debounce)
		{
			gameObject_t *myAerith = obj->plObj->aerith;
			myAerith->net.renderEffects |= FXFL_DEATH;
			myAerith->net.renderEffects2 |= FXFL2_DEATH4;
			myAerith->debounce = g_curTime+1000;
		}
		else if (obj->plObj->aerith->debounce < g_curTime)
		{
			obj->plObj->aerith->think = ObjGeneral_RemoveThink;
			obj->plObj->aerith->thinkTime = g_curTime;
			obj->plObj->aerith = NULL;
		}
	}

	if (g_cinema == 1)
	{
		ObjPlayer_UpdateSword(obj, eqItem, timeMod);
		if (ObjPlayer_ButtonTapped(obj, BUTTON_JUMP) ||
			ObjPlayer_ButtonTapped(obj, BUTTON_CHARMENU))
		{
			static serverTime_t cinMsgTime = 0;
			if (cinMsgTime < g_curTime)
			{
				Util_StatusMessage("Use *$button_k$ to fast-forward.");
				cinMsgTime = g_curTime+4000;
			}
		}
		obj->plObj->numButtonsBuffered = 0; //processed all input
		return;
	}

	if (eqItem && (eqItem->itemBehavior == INVITEM_BEHAVIOR_HASTE || eqItem->itemBehavior == INVITEM_BEHAVIOR_FINALHEAVEN))
	{
		if (obj->plObj->hasteTime < (g_curTime + 10))
		{
			obj->plObj->hasteTime = g_curTime + 10;
		}
	}

	if (obj->plObj->waitOnInput > 0)
	{ //script input
		if (ObjPlayer_ButtonTapped(obj, obj->plObj->waitOnInput-1))
		{
			obj->plObj->waitOnInput = -1;
		}
	}

	if (cam->inuse)
	{
		if (obj->plObj->playerSpawnSeq)
		{
			obj->net.pos[0] = g_initialFPSPos[0];
			obj->net.pos[1] = g_initialFPSPos[1];
			obj->net.pos[2] = g_initialFPSPos[2];
			obj->net.ang[0] = g_initialFPSAng[0];
			obj->net.ang[1] = g_initialFPSAng[1];
			obj->net.ang[2] = g_initialFPSAng[2];
			Math_VecCopy(obj->net.pos, obj->safePos);
			g_sharedFn->Coll_GetWorldBounds(g_worldMins, g_worldMaxs);
			obj->plObj->playerSpawnSeq = false;
			Phys_PutOnGround(obj);
		}

		//terrible hackery to prevent player from falling out of world
		float plAbsMins[3], plAbsMaxs[3];
		for (int k = 0; k < 3; k++)
		{
			plAbsMins[k] = obj->net.pos[k]+obj->net.mins[k];
			plAbsMaxs[k] = obj->net.pos[k]+obj->net.maxs[k];
		}
		if (!Math_BoxesOverlap(plAbsMins, plAbsMaxs, g_worldMins, g_worldMaxs))
		{
			obj->net.pos[0] = g_initialFPSPos[0];
			obj->net.pos[1] = g_initialFPSPos[1];
			obj->net.pos[2] = g_initialFPSPos[2];
			obj->net.ang[0] = g_initialFPSAng[0];
			obj->net.ang[1] = g_initialFPSAng[1];
			obj->net.ang[2] = g_initialFPSAng[2];
			Math_VecCopy(obj->net.pos, obj->safePos);
		}
	}

	int fillInput = ObjPlayer_CheckChargeAttack(obj, timeMod);

	ObjPlayer_CheckEffects(obj, timeMod);

	if (obj->plObj->clButtons[BUTTON_EVENT7] && obj->health > 0)
	{ //find a target
		if (!ObjPlayer_TargetValid(obj, obj->plObj->targetObj))
		{
			if (obj->plObj->targetObj)
			{
				obj->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(obj);
				obj->plObj->targetObj = NULL;
			}
		}

		if (!obj->plObj->targetObj/* && (!ObjPlayer_InChain(obj) || ObjPlayer_InChainTransition(obj)) && !ObjPlayer_NonMovingAnim(obj)*/)
		{ //find a new target
			obj->plObj->targetObj = ObjPlayer_FindTarget(obj);
			if (obj->plObj->targetObj)
			{
				if (eqItem && eqItem->itemBehavior == INVITEM_BEHAVIOR_VIEWHEALTH)
				{
					ObjPlayer_HealthDisplay(obj, obj->plObj->targetObj);
				}
				obj->plObj->targetObj->net.renderEffects |= ObjPlayer_TargetBit(obj);
			}
		}
		else
		{
			if (obj->plObj->healthUpdateTime < g_curTime &&
				eqItem && eqItem->itemBehavior == INVITEM_BEHAVIOR_VIEWHEALTH)
			{
				ObjPlayer_HealthDisplay(obj, obj->plObj->targetObj);
				obj->plObj->healthUpdateTime = g_curTime+500;
			}
		}
	}
	else
	{
		if (obj->plObj->targetObj)
		{
			obj->plObj->targetObj->net.renderEffects &= ~ObjPlayer_TargetBit(obj);
			obj->plObj->targetObj = NULL;
		}
	}

	float moveVec[3];
	float move, rightMove;
	if (obj->plObj->ride)
	{
		moveVec[0] = 0.0f;
		moveVec[1] = 0.0f;
		moveVec[2] = 0.0f;
		move = 0.0f;
		rightMove = 0.0f;
	}
	else
	{
		ObjPlayer_GetMoveVec(obj, cam, timeMod, moveVec, move, rightMove, false);
	}

	ObjPlayer_ApplyAnimationEffects(obj, timeMod);

	if (!ObjPlayer_ApplyAnimationVelocity(obj, timeMod) && !ObjPlayer_NonMovingAnim(obj))
	{
		if (obj->plObj->clButtons[BUTTON_EVENT7] && ObjPlayer_InChainTransition(obj))
		{
			if (obj->onGround && obj->plObj->targetObj)
			{
				float targCenter[3], plCenter[3];
				Util_GameObjectCenter(obj->plObj->targetObj, targCenter);
				Util_GameObjectCenter(obj, plCenter);

				float fwd[3];
				Math_VecSub(targCenter, plCenter, fwd);
				Math_VecNorm(fwd);
				float l = 80.0f;
				obj->net.vel[0] += fwd[0]*l;
				obj->net.vel[1] += fwd[1]*l;
				obj->net.vel[2] += fwd[2]*l;
			}
		}
		else if (!ObjPlayer_InChain(obj) || ObjPlayer_InChainTransition(obj))
		{
			obj->net.vel[0] += moveVec[0];
			obj->net.vel[1] += moveVec[1];
			obj->net.vel[2] += moveVec[2];
		}
	}

	Math_VecNorm(moveVec);
	ObjPlayer_ChainInput(obj, eqItem, moveVec);

	if (obj->plObj->canJumpUse >= g_curTime || obj->plObj->ride)
	{
		if (ObjPlayer_ButtonTapped(obj, BUTTON_JUMP))
		{
			obj->plObj->jumpUseTime = g_curTime + Util_LocalDelay(obj, 500);
		}
	}
	else if (obj->onGround)
	{
		float fwd[3], right[3];
		if (ObjPlayer_CanTakeAction(obj) && ObjPlayer_ButtonTapped(obj, BUTTON_JUMP))
		{
			if (obj->plObj->clButtons[BUTTON_EVENT7])
			{ //try to do a dodge instead of a jump
				if (Math_VecLen(moveVec) <= 0.0f)
				{
					AI_StartAnim(obj, HUMANIM_JUMP, true);
					obj->net.vel[0] *= 0.25f;
					obj->net.vel[1] *= 0.25f;
				}
				else
				{
					Math_AngleVectors(obj->net.ang, fwd, right, 0);

					float dpfwd = Math_DotProduct(fwd, moveVec);
					float dpright = Math_DotProduct(right, moveVec);
					if (fabsf(dpright) > fabsf(dpfwd)*0.5f) //bias toward rolls
					{ //left/right
						if (dpright >= 0.0f)
						{
							AI_StartAnim(obj, PLANIM_DODGE_RIGHT, true);
						}
						else
						{
							AI_StartAnim(obj, PLANIM_DODGE_LEFT, true);
						}
						ObjSound_Create(obj->net.pos, "assets/sound/cb/roll01.wav", 1.0f, -1);
					}
					else
					{ //forward/back
						if (dpfwd >= 0.0f)
						{ //regular jump
							AI_StartAnim(obj, HUMANIM_JUMP, true);
							obj->net.vel[0] *= 0.25f;
							obj->net.vel[1] *= 0.25f;
						}
						else
						{
							AI_StartAnim(obj, PLANIM_BACKFLIP, true);
							obj->net.vel[0] = -fwd[0]*800.0f;
							obj->net.vel[1] = -fwd[1]*800.0f;
							obj->net.vel[2] = 1000.0f;
							ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
						}
					}
				}
			}
			else
			{
				AI_StartAnim(obj, HUMANIM_JUMP, true);
				obj->net.vel[0] *= 0.25f;
				obj->net.vel[1] *= 0.25f;
			}
		}
		else if (ObjPlayer_Ability(obj, PLABIL_GETUP) &&
			obj->plObj->clButtons[BUTTON_JUMP] &&
			(obj->curAnim == HUMANIM_GETUP_FRONT || obj->curAnim == HUMANIM_GETUP_BACK ||
			obj->curAnim == HUMANIM_FALL_LAND || obj->curAnim == HUMANIM_POPUP_LAND))
		{ //quick getups
			Math_AngleVectors(obj->net.ang, 0, right, 0);
			float dpright = Math_DotProduct(right, moveVec);
			if (dpright == 0.0f)
			{
				AI_StartAnim(obj, PLANIM_GETUP_FLIP, true);
				ObjSound_Create(obj->net.pos, "assets/sound/cb/jump01.wav", 1.0f, -1);
				obj->net.vel[2] = 1000.0f;
			}
			else if (dpright >= 0.0f)
			{
				AI_StartAnim(obj, PLANIM_DODGE_RIGHT, true);
				ObjSound_Create(obj->net.pos, "assets/sound/cb/roll01.wav", 1.0f, -1);
			}
			else
			{
				AI_StartAnim(obj, PLANIM_DODGE_LEFT, true);
				ObjSound_Create(obj->net.pos, "assets/sound/cb/roll01.wav", 1.0f, -1);
			}
		}
	}
	else
	{
		if (ObjPlayer_Ability(obj, PLABIL_WALLJUMP) && obj->curAnim == HUMANIM_FALL && ObjPlayer_CanTakeAction(obj) && ObjPlayer_ButtonTapped(obj, BUTTON_JUMP))
		{ //try a walljump
			float traceSrc[3], traceDst[3];
			float wallJumpTraceDist = obj->radius+48.0f;//32.0f;
			Math_VecCopy(obj->net.pos, traceSrc);
			traceSrc[2] += 100.0f;
			traceDst[0] = traceSrc[0] + moveVec[0]*wallJumpTraceDist;
			traceDst[1] = traceSrc[1] + moveVec[1]*wallJumpTraceDist;
			traceDst[2] = traceSrc[2] + moveVec[2]*wallJumpTraceDist;
			collObj_t col;
			collOpt_t opt;
			memset(&opt, 0, sizeof(opt));
			opt.ignoreBoxModels = true;
			g_sharedFn->Coll_RadiusTranslation(obj->rcColModel, &col, traceSrc, traceDst, 16.0f, NULL, &opt);
			if (col.hit)
			{
				float wallDot = Math_DotProduct(moveVec, col.endNormal);
				if (wallDot < -0.8f)
				{ //do it
					float wallJumpForce = 500.0f;
					float wallJumpForceZ = 1000.0f;
					obj->net.vel[0] = col.endNormal[0]*wallJumpForce;
					obj->net.vel[1] = col.endNormal[1]*wallJumpForce;
					obj->net.vel[2] = wallJumpForceZ;
					AI_StartAnim(obj, PLANIM_WALLJUMP, true);
					ObjSound_Create(obj->net.pos, "assets/sound/cb/walljump01.wav", 1.0f, -1);
					col.endPos[2] += 128.0f;
					float ang[3], p[3];
					Math_VecToAngles(col.endNormal, ang);
					Math_VecMA(col.endPos, 8.0f, col.endNormal, p);
					ObjParticles_Create("impacts/wallhop", p, ang, -1);
				}
			}
		}
	}

	//update model angles
	float animAng[3];
	if (AI_AnglesForAnim(obj, animAng))
	{
		float angleBlendScale = 0.7f;
		obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], animAng[PITCH], timeMod*angleBlendScale);
		obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], animAng[YAW], timeMod*angleBlendScale);
		obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], animAng[ROLL], timeMod*angleBlendScale);
	}
	else if (!ObjPlayer_NonAnglingAnim(obj))
	{
		if (obj->plObj->targetObj && !ObjPlayer_AnglingAnim(obj))
		{ //turn toward the target
			float targCenter[3], plCenter[3], targVec[3], targAng[3];
			float angleBlendScale = 0.7f;
			Util_GameObjectCenter(obj->plObj->targetObj, targCenter);
			Util_GameObjectCenter(obj, plCenter);
			Math_VecSub(targCenter, plCenter, targVec);
			Math_VecToAngles(targVec, targAng);
			obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], targAng[YAW], timeMod*angleBlendScale);
		}
		else if (!obj->plObj->clButtons[BUTTON_EVENT7] || ObjPlayer_AnglingAnim(obj))
		{ //not holding target, or in an angle-overriding anim
			float moveAngTol = obj->onGround ? 1.0f : 100.0f;
			float velCheck[3] = {obj->net.vel[0], obj->net.vel[1], 0.0f};
			float velCheckLen = Math_VecLen(velCheck);
			if ((move || rightMove || obj->curAnim == PLANIM_WALLJUMP) && velCheckLen > moveAngTol)
			{
				float velBasedTurn = (velCheckLen > 64.0f) ? 0.5f : 0.05f;
				float angleBlendScale = (obj->curAnim == PLANIM_WALLJUMP) ? 0.8f : velBasedTurn;
				float velAng[3];
				Math_VecToAngles(obj->net.vel, velAng);
				obj->net.ang[YAW] = Math_BlendAngle(obj->net.ang[YAW], velAng[YAW], timeMod*angleBlendScale);
			}
		}
	}

	float idealPitch = 0.0f;
	float idealRoll = 0.0f;
	/*
	switch (obj->curAnim)
	{
	case PLANIM_DODGE_LEFT:
		idealRoll = -30.0f;
		break;
	case PLANIM_DODGE_RIGHT:
		idealRoll = 30.0f;
		break;
	default:
		break;
	}
	*/
	obj->net.ang[PITCH] = Math_BlendAngle(obj->net.ang[PITCH], idealPitch, timeMod*0.3f);
	obj->net.ang[ROLL] = Math_BlendAngle(obj->net.ang[ROLL], idealRoll, timeMod*0.3f);

	AI_PickAnim(obj, timeMod);
	Util_AnimateObject(obj, timeMod);

	ObjPlayer_SetAttackType(obj);
	ObjPlayer_CheckChainDamage(obj, timeMod);

	if (obj->plObj->noClip)
	{
		float moveAngNC[3];
		moveAngNC[0] = cam->net.ang[0];
		moveAngNC[1] = cam->net.ang[1];
		moveAngNC[2] = 0.0f;

		float fwdNC[3], rightNC[3];
		Math_AngleVectors(moveAngNC, fwdNC, rightNC, 0);
		float moveVecNC[3];
		moveVecNC[0] = obj->net.pos[0] + (fwdNC[0]*move)+(rightNC[0]*rightMove);
		moveVecNC[1] = obj->net.pos[1] + (fwdNC[1]*move)+(rightNC[1]*rightMove);
		moveVecNC[2] = obj->net.pos[2] + (fwdNC[2]*move)+(rightNC[2]*rightMove);
		obj->net.vel[0] = 0.0f;
		obj->net.vel[1] = 0.0f;
		obj->net.vel[2] = 0.0f;
		Phys_LinearMove(obj, moveVecNC, Math_Max2(fabsf(move), fabsf(rightMove)));
	}
	else
	{
		if (!obj->plObj->playerSpawnSeq && !obj->plObj->ride)
		{ //translation safety checking
			Util_SetSafeSpot(obj);
			Phys_ApplyObjectPhysics(obj, timeMod, obj->radius, ObjPlayer_GetGravity(obj, timeMod), 0.0f);
		}
	}
	LServ_UpdateRClip(obj);

	ObjPlayer_CheckItemUse(obj, timeMod);
	ObjPlayer_CheckCharge(obj, timeMod);

	if (obj->plObj->timeAttackTime && obj->plObj->timeAttackTime < g_curTime)
	{
		if (g_timeType == TIMETYPE_TIMEATTACK)
		{
			ObjSound_Create(obj->net.pos, "assets/sound/cb/chargeoff.wav", 1.0f, -1);
			g_timeType = TIMETYPE_NORMAL;
		}
		obj->plObj->timeAttackTime = 0;
	}

	if (obj->plObj->gun)
	{
		ObjPlayer_UpdateGun(obj, obj->plObj->gun, timeMod);
	}

	ObjPlayer_EncodeNetData(obj);

	if (fillInput != -1)
	{
		obj->plObj->clButtons[fillInput] = 1;
	}

	obj->plObj->numButtonsBuffered = 0; //processed all input

	//update local timescale
	if (g_timeType == TIMETYPE_TIMEATTACK)
	{
		obj->localTimeScale = 4.0f;
	}
	else
	{
		obj->localTimeScale = 1.0f;
	}
	if (obj->plObj->hasteTime > g_curTime)
	{
		obj->localTimeScale *= 2.0f;
	}

	ObjPlayer_UpdateSword(obj, eqItem, timeMod);
}

//give some dropped mako
void ObjPlayer_GiveMako(gameObject_t *obj, int makoAmount)
{
	obj->plObj->makoStash += makoAmount;
	if (obj->plObj->makoStash > 999999)
	{
		obj->plObj->makoStash = 999999;
	}
}

//charge
void ObjPlayer_ChargeUp(gameObject_t *obj, gameObject_t *other, int dmg)
{
	if (obj == other)
	{
		return;
	}
	float dex = (float)obj->plObj->plData.statDex;
	const invItemDef_t *item = ObjPlayer_GetEquippedItem(obj);
	if (item && item->itemBehavior == INVITEM_BEHAVIOR_MODDEX)
	{
		dex += (float)item->itemValue;
	}
	float smod = dex / 20.0f;
	if (smod > 1.0f)
	{
		float f = smod-1.0f;
		smod = 1.0f+f*0.1f;//0.25f;
	}
	float df = (float)dmg;
	df *= 0.5f;
	if (df > 50.0f)
	{
		df = 50.0f;
	}
	if (g_timeType == TIMETYPE_TIMEATTACK)
	{
		smod *= 0.1f;
	}
	int chg = (int)(df*smod);
	if (chg < 1)
	{
		chg = 1;
	}
	obj->plObj->chargeDrain = 0.0f;
	obj->plObj->chargeMeter += chg;
	if (obj->plObj->chargeMeter > 4000)
	{
		obj->plObj->chargeMeter = 4000;
	}
}

//charge down
void ObjPlayer_ChargeDown(gameObject_t *obj, gameObject_t *other, int dmg)
{
	obj->plObj->chargeMeter -= dmg*8;
	if (obj->plObj->chargeMeter < 0)
	{
		obj->plObj->chargeMeter = 0;
	}
}

//return bit used for target fx by this client
int ObjPlayer_TargetBit(gameObject_t *obj)
{
	if (obj->net.index > 0)
	{
		return FXFL_TARGET2;
	}
	return FXFL_TARGET;
}
