// Copyright (C) 1999-2000 Id Software, Inc.
//
// q_math.c -- stateless support routines that are included in each code module
#include "q_shared.h"


vec3_t	vec3_origin = {0,0,0};
vec3_t	axisDefault[3] = { { 1, 0, 0 }, { 0, 1, 0 }, { 0, 0, 1 } };


vec4_t		colorBlack	= {0, 0, 0, 1};
vec4_t		colorRed	= {1, 0, 0, 1};
vec4_t		colorGreen	= {0, 1, 0, 1};
vec4_t		colorBlue	= {0, 0, 1, 1};
vec4_t		colorYellow	= {1, 1, 0, 1};
vec4_t		colorMagenta= {1, 0, 1, 1};
vec4_t		colorCyan	= {0, 1, 1, 1};
vec4_t		colorWhite	= {1, 1, 1, 1};
vec4_t		colorLtGrey	= {0.75, 0.75, 0.75, 1};
vec4_t		colorMdGrey	= {0.5, 0.5, 0.5, 1};
vec4_t		colorDkGrey	= {0.25, 0.25, 0.25, 1};

vec4_t		colorLtBlue	= {0.367f, 0.261f, 0.722f, 1};
vec4_t		colorDkBlue	= {0.199f, 0.0f,   0.398f, 1};

vec4_t	g_color_table[8] =
	{
	{0.0, 0.0, 0.0, 1.0},
	{1.0, 0.0, 0.0, 1.0},
	{0.0, 1.0, 0.0, 1.0},
	{1.0, 1.0, 0.0, 1.0},
	{0.0, 0.0, 1.0, 1.0},
	{0.0, 1.0, 1.0, 1.0},
	{1.0, 0.0, 1.0, 1.0},
	{1.0, 1.0, 1.0, 1.0},
	};


vec3_t	bytedirs[NUMVERTEXNORMALS] =
{
{-0.525731f, 0.000000f, 0.850651f}, {-0.442863f, 0.238856f, 0.864188f}, 
{-0.295242f, 0.000000f, 0.955423f}, {-0.309017f, 0.500000f, 0.809017f}, 
{-0.162460f, 0.262866f, 0.951056f}, {0.000000f, 0.000000f, 1.000000f}, 
{0.000000f, 0.850651f, 0.525731f}, {-0.147621f, 0.716567f, 0.681718f}, 
{0.147621f, 0.716567f, 0.681718f}, {0.000000f, 0.525731f, 0.850651f}, 
{0.309017f, 0.500000f, 0.809017f}, {0.525731f, 0.000000f, 0.850651f}, 
{0.295242f, 0.000000f, 0.955423f}, {0.442863f, 0.238856f, 0.864188f}, 
{0.162460f, 0.262866f, 0.951056f}, {-0.681718f, 0.147621f, 0.716567f}, 
{-0.809017f, 0.309017f, 0.500000f},{-0.587785f, 0.425325f, 0.688191f}, 
{-0.850651f, 0.525731f, 0.000000f},{-0.864188f, 0.442863f, 0.238856f}, 
{-0.716567f, 0.681718f, 0.147621f},{-0.688191f, 0.587785f, 0.425325f}, 
{-0.500000f, 0.809017f, 0.309017f}, {-0.238856f, 0.864188f, 0.442863f}, 
{-0.425325f, 0.688191f, 0.587785f}, {-0.716567f, 0.681718f, -0.147621f}, 
{-0.500000f, 0.809017f, -0.309017f}, {-0.525731f, 0.850651f, 0.000000f}, 
{0.000000f, 0.850651f, -0.525731f}, {-0.238856f, 0.864188f, -0.442863f}, 
{0.000000f, 0.955423f, -0.295242f}, {-0.262866f, 0.951056f, -0.162460f}, 
{0.000000f, 1.000000f, 0.000000f}, {0.000000f, 0.955423f, 0.295242f}, 
{-0.262866f, 0.951056f, 0.162460f}, {0.238856f, 0.864188f, 0.442863f}, 
{0.262866f, 0.951056f, 0.162460f}, {0.500000f, 0.809017f, 0.309017f}, 
{0.238856f, 0.864188f, -0.442863f},{0.262866f, 0.951056f, -0.162460f}, 
{0.500000f, 0.809017f, -0.309017f},{0.850651f, 0.525731f, 0.000000f}, 
{0.716567f, 0.681718f, 0.147621f}, {0.716567f, 0.681718f, -0.147621f}, 
{0.525731f, 0.850651f, 0.000000f}, {0.425325f, 0.688191f, 0.587785f}, 
{0.864188f, 0.442863f, 0.238856f}, {0.688191f, 0.587785f, 0.425325f}, 
{0.809017f, 0.309017f, 0.500000f}, {0.681718f, 0.147621f, 0.716567f}, 
{0.587785f, 0.425325f, 0.688191f}, {0.955423f, 0.295242f, 0.000000f}, 
{1.000000f, 0.000000f, 0.000000f}, {0.951056f, 0.162460f, 0.262866f}, 
{0.850651f, -0.525731f, 0.000000f},{0.955423f, -0.295242f, 0.000000f}, 
{0.864188f, -0.442863f, 0.238856f}, {0.951056f, -0.162460f, 0.262866f}, 
{0.809017f, -0.309017f, 0.500000f}, {0.681718f, -0.147621f, 0.716567f}, 
{0.850651f, 0.000000f, 0.525731f}, {0.864188f, 0.442863f, -0.238856f}, 
{0.809017f, 0.309017f, -0.500000f}, {0.951056f, 0.162460f, -0.262866f}, 
{0.525731f, 0.000000f, -0.850651f}, {0.681718f, 0.147621f, -0.716567f}, 
{0.681718f, -0.147621f, -0.716567f},{0.850651f, 0.000000f, -0.525731f}, 
{0.809017f, -0.309017f, -0.500000f}, {0.864188f, -0.442863f, -0.238856f}, 
{0.951056f, -0.162460f, -0.262866f}, {0.147621f, 0.716567f, -0.681718f}, 
{0.309017f, 0.500000f, -0.809017f}, {0.425325f, 0.688191f, -0.587785f}, 
{0.442863f, 0.238856f, -0.864188f}, {0.587785f, 0.425325f, -0.688191f}, 
{0.688191f, 0.587785f, -0.425325f}, {-0.147621f, 0.716567f, -0.681718f}, 
{-0.309017f, 0.500000f, -0.809017f}, {0.000000f, 0.525731f, -0.850651f}, 
{-0.525731f, 0.000000f, -0.850651f}, {-0.442863f, 0.238856f, -0.864188f}, 
{-0.295242f, 0.000000f, -0.955423f}, {-0.162460f, 0.262866f, -0.951056f}, 
{0.000000f, 0.000000f, -1.000000f}, {0.295242f, 0.000000f, -0.955423f}, 
{0.162460f, 0.262866f, -0.951056f}, {-0.442863f, -0.238856f, -0.864188f}, 
{-0.309017f, -0.500000f, -0.809017f}, {-0.162460f, -0.262866f, -0.951056f}, 
{0.000000f, -0.850651f, -0.525731f}, {-0.147621f, -0.716567f, -0.681718f}, 
{0.147621f, -0.716567f, -0.681718f}, {0.000000f, -0.525731f, -0.850651f}, 
{0.309017f, -0.500000f, -0.809017f}, {0.442863f, -0.238856f, -0.864188f}, 
{0.162460f, -0.262866f, -0.951056f}, {0.238856f, -0.864188f, -0.442863f}, 
{0.500000f, -0.809017f, -0.309017f}, {0.425325f, -0.688191f, -0.587785f}, 
{0.716567f, -0.681718f, -0.147621f}, {0.688191f, -0.587785f, -0.425325f}, 
{0.587785f, -0.425325f, -0.688191f}, {0.000000f, -0.955423f, -0.295242f}, 
{0.000000f, -1.000000f, 0.000000f}, {0.262866f, -0.951056f, -0.162460f}, 
{0.000000f, -0.850651f, 0.525731f}, {0.000000f, -0.955423f, 0.295242f}, 
{0.238856f, -0.864188f, 0.442863f}, {0.262866f, -0.951056f, 0.162460f}, 
{0.500000f, -0.809017f, 0.309017f}, {0.716567f, -0.681718f, 0.147621f}, 
{0.525731f, -0.850651f, 0.000000f}, {-0.238856f, -0.864188f, -0.442863f}, 
{-0.500000f, -0.809017f, -0.309017f}, {-0.262866f, -0.951056f, -0.162460f}, 
{-0.850651f, -0.525731f, 0.000000f}, {-0.716567f, -0.681718f, -0.147621f}, 
{-0.716567f, -0.681718f, 0.147621f}, {-0.525731f, -0.850651f, 0.000000f}, 
{-0.500000f, -0.809017f, 0.309017f}, {-0.238856f, -0.864188f, 0.442863f}, 
{-0.262866f, -0.951056f, 0.162460f}, {-0.864188f, -0.442863f, 0.238856f}, 
{-0.809017f, -0.309017f, 0.500000f}, {-0.688191f, -0.587785f, 0.425325f}, 
{-0.681718f, -0.147621f, 0.716567f}, {-0.442863f, -0.238856f, 0.864188f}, 
{-0.587785f, -0.425325f, 0.688191f}, {-0.309017f, -0.500000f, 0.809017f}, 
{-0.147621f, -0.716567f, 0.681718f}, {-0.425325f, -0.688191f, 0.587785f}, 
{-0.162460f, -0.262866f, 0.951056f}, {0.442863f, -0.238856f, 0.864188f}, 
{0.162460f, -0.262866f, 0.951056f}, {0.309017f, -0.500000f, 0.809017f}, 
{0.147621f, -0.716567f, 0.681718f}, {0.000000f, -0.525731f, 0.850651f}, 
{0.425325f, -0.688191f, 0.587785f}, {0.587785f, -0.425325f, 0.688191f}, 
{0.688191f, -0.587785f, 0.425325f}, {-0.955423f, 0.295242f, 0.000000f}, 
{-0.951056f, 0.162460f, 0.262866f}, {-1.000000f, 0.000000f, 0.000000f}, 
{-0.850651f, 0.000000f, 0.525731f}, {-0.955423f, -0.295242f, 0.000000f}, 
{-0.951056f, -0.162460f, 0.262866f}, {-0.864188f, 0.442863f, -0.238856f}, 
{-0.951056f, 0.162460f, -0.262866f}, {-0.809017f, 0.309017f, -0.500000f}, 
{-0.864188f, -0.442863f, -0.238856f}, {-0.951056f, -0.162460f, -0.262866f}, 
{-0.809017f, -0.309017f, -0.500000f}, {-0.681718f, 0.147621f, -0.716567f}, 
{-0.681718f, -0.147621f, -0.716567f}, {-0.850651f, 0.000000f, -0.525731f}, 
{-0.688191f, 0.587785f, -0.425325f}, {-0.587785f, 0.425325f, -0.688191f}, 
{-0.425325f, 0.688191f, -0.587785f}, {-0.425325f, -0.688191f, -0.587785f}, 
{-0.587785f, -0.425325f, -0.688191f}, {-0.688191f, -0.587785f, -0.425325f}
};

//==============================================================

int		Q_rand( int *seed ) {
	*seed = (69069 * *seed + 1);
	return *seed;
}

float	Q_random( int *seed ) {
	return ( Q_rand( seed ) & 0xffff ) / (float)0x10000;
}

float	Q_crandom( int *seed ) {
	return 2.0 * ( Q_random( seed ) - 0.5 );
}

#ifdef __LCC__

int VectorCompare( const vec3_t v1, const vec3_t v2 ) {
	if (v1[0] != v2[0] || v1[1] != v2[1] || v1[2] != v2[2]) {
		return 0;
	}			
	return 1;
}

vec_t VectorLength( const vec3_t v ) {
	return (vec_t)sqrt (v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}

vec_t VectorLengthSquared( const vec3_t v ) {
	return (v[0]*v[0] + v[1]*v[1] + v[2]*v[2]);
}

vec_t Distance( const vec3_t p1, const vec3_t p2 ) {
	vec3_t	v;

	VectorSubtract (p2, p1, v);
	return VectorLength( v );
}

vec_t DistanceSquared( const vec3_t p1, const vec3_t p2 ) {
	vec3_t	v;

	VectorSubtract (p2, p1, v);
	return v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
}

// fast vector normalize routine that does not check to make sure
// that length != 0, nor does it return length, uses rsqrt approximation
void VectorNormalizeFast( vec3_t v )
{
	float ilength;

	ilength = Q_rsqrt( DotProduct( v, v ) );

	v[0] *= ilength;
	v[1] *= ilength;
	v[2] *= ilength;
}

void VectorInverse( vec3_t v ){
	v[0] = -v[0];
	v[1] = -v[1];
	v[2] = -v[2];
}

void CrossProduct( const vec3_t v1, const vec3_t v2, vec3_t cross ) {
	cross[0] = v1[1]*v2[2] - v1[2]*v2[1];
	cross[1] = v1[2]*v2[0] - v1[0]*v2[2];
	cross[2] = v1[0]*v2[1] - v1[1]*v2[0];
}
#endif

//=======================================================

signed char ClampChar( int i ) {
	if ( i < -128 ) {
		return -128;
	}
	if ( i > 127 ) {
		return 127;
	}
	return i;
}

signed short ClampShort( int i ) {
	if ( i < -32768 ) {
		return -32768;
	}
	if ( i > 0x7fff ) {
		return 0x7fff;
	}
	return i;
}


// this isn't a real cheap function to call!
int DirToByte( vec3_t dir ) {
	int		i, best;
	float	d, bestd;

	if ( !dir ) {
		return 0;
	}

	bestd = 0;
	best = 0;
	for (i=0 ; i<NUMVERTEXNORMALS ; i++)
	{
		d = DotProduct (dir, bytedirs[i]);
		if (d > bestd)
		{
			bestd = d;
			best = i;
		}
	}

	return best;
}

void ByteToDir( int b, vec3_t dir ) {
	if ( b < 0 || b >= NUMVERTEXNORMALS ) {
		VectorCopy( vec3_origin, dir );
		return;
	}
	VectorCopy (bytedirs[b], dir);
}


unsigned ColorBytes3 (float r, float g, float b) {
	unsigned	i;

	( (byte *)&i )[0] = r * 255;
	( (byte *)&i )[1] = g * 255;
	( (byte *)&i )[2] = b * 255;

	return i;
}

unsigned ColorBytes4 (float r, float g, float b, float a) {
	unsigned	i;

	( (byte *)&i )[0] = r * 255;
	( (byte *)&i )[1] = g * 255;
	( (byte *)&i )[2] = b * 255;
	( (byte *)&i )[3] = a * 255;

	return i;
}

float NormalizeColor( const vec3_t in, vec3_t out ) {
	float	max;
	
	max = in[0];
	if ( in[1] > max ) {
		max = in[1];
	}
	if ( in[2] > max ) {
		max = in[2];
	}

	if ( !max ) {
		VectorClear( out );
	} else {
		out[0] = in[0] / max;
		out[1] = in[1] / max;
		out[2] = in[2] / max;
	}
	return max;
}


/*
=====================
PlaneFromPoints

Returns false if the triangle is degenrate.
The normal will point out of the clock for clockwise ordered points
=====================
*/
qboolean PlaneFromPoints( vec4_t plane, const vec3_t a, const vec3_t b, const vec3_t c ) {
	vec3_t	d1, d2;

	VectorSubtract( b, a, d1 );
	VectorSubtract( c, a, d2 );
	CrossProduct( d2, d1, plane );
	if ( VectorNormalize( plane ) == 0 ) {
		return qfalse;
	}

	plane[3] = DotProduct( a, plane );
	return qtrue;
}

/*
===============
RotatePointAroundVector

This is not implemented very well...
===============
*/
void RotatePointAroundVector( vec3_t dst, const vec3_t dir, const vec3_t point,
							 float degrees ) {
	float	m[3][3];
	float	im[3][3];
	float	zrot[3][3];
	float	tmpmat[3][3];
	float	rot[3][3];
	int	i;
	vec3_t vr, vup, vf;
	float	rad;

	vf[0] = dir[0];
	vf[1] = dir[1];
	vf[2] = dir[2];

	PerpendicularVector( vr, dir );
	CrossProduct( vr, vf, vup );

	m[0][0] = vr[0];
	m[1][0] = vr[1];
	m[2][0] = vr[2];

	m[0][1] = vup[0];
	m[1][1] = vup[1];
	m[2][1] = vup[2];

	m[0][2] = vf[0];
	m[1][2] = vf[1];
	m[2][2] = vf[2];

	memcpy( im, m, sizeof( im ) );

	im[0][1] = m[1][0];
	im[0][2] = m[2][0];
	im[1][0] = m[0][1];
	im[1][2] = m[2][1];
	im[2][0] = m[0][2];
	im[2][1] = m[1][2];

	memset( zrot, 0, sizeof( zrot ) );
	zrot[0][0] = zrot[1][1] = zrot[2][2] = 1.0F;

	rad = DEG2RAD( degrees );
	zrot[0][0] = cos( rad );
	zrot[0][1] = sin( rad );
	zrot[1][0] = -sin( rad );
	zrot[1][1] = cos( rad );

	MatrixMultiply( m, zrot, tmpmat );
	MatrixMultiply( tmpmat, im, rot );

	for ( i = 0; i < 3; i++ ) {
		dst[i] = rot[i][0] * point[0] + rot[i][1] * point[1] + rot[i][2] * point[2];
	}
}

/*
===============
RotateAroundDirection
===============
*/
void RotateAroundDirection( vec3_t axis[3], float yaw ) {

	// create an arbitrary axis[1] 
	PerpendicularVector( axis[1], axis[0] );

	// rotate it around axis[0] by yaw
	if ( yaw ) {
		vec3_t	temp;

		VectorCopy( axis[1], temp );
		RotatePointAroundVector( axis[1], axis[0], temp, yaw );
	}

	// cross to get axis[2]
	CrossProduct( axis[0], axis[1], axis[2] );
}



void vectoangles( const vec3_t value1, vec3_t angles ) {
	float	forward;
	float	yaw, pitch;
	
	if ( value1[1] == 0 && value1[0] == 0 ) {
		yaw = 0;
		if ( value1[2] > 0 ) {
			pitch = 90;
		}
		else {
			pitch = 270;
		}
	}
	else {
		if ( value1[0] ) {
			yaw = ( atan2 ( value1[1], value1[0] ) * 180 / M_PI );
		}
		else if ( value1[1] > 0 ) {
			yaw = 90;
		}
		else {
			yaw = 270;
		}
		if ( yaw < 0 ) {
			yaw += 360;
		}

		forward = sqrt ( value1[0]*value1[0] + value1[1]*value1[1] );
		pitch = ( atan2(value1[2], forward) * 180 / M_PI );
		if ( pitch < 0 ) {
			pitch += 360;
		}
	}

	angles[PITCH] = -pitch;
	angles[YAW] = yaw;
	angles[ROLL] = 0;
}


/*
=================
AnglesToAxis
=================
*/
void AnglesToAxis( const vec3_t angles, vec3_t axis[3] ) {
	vec3_t	right;

	// angle vectors returns "right" instead of "y axis"
	AngleVectors( angles, axis[0], right, axis[2] );
	VectorSubtract( vec3_origin, right, axis[1] );
}

void AxisClear( vec3_t axis[3] ) {
	axis[0][0] = 1;
	axis[0][1] = 0;
	axis[0][2] = 0;
	axis[1][0] = 0;
	axis[1][1] = 1;
	axis[1][2] = 0;
	axis[2][0] = 0;
	axis[2][1] = 0;
	axis[2][2] = 1;
}

void AxisCopy( vec3_t in[3], vec3_t out[3] ) {
	VectorCopy( in[0], out[0] );
	VectorCopy( in[1], out[1] );
	VectorCopy( in[2], out[2] );
}

void AxisScale( vec3_t in[3], float scale, vec3_t out[3] ) {
   int i;

   if( (scale == 1) || (scale <= 0) )
      return;

   for(i = 0; i < 3; i++)
      VectorScale(in[i], scale, out[i]);

   return;
}

void ProjectPointOnPlane( vec3_t dst, const vec3_t p, const vec3_t normal )
{
	float d;
	vec3_t n;
	float inv_denom;

	inv_denom =  DotProduct( normal, normal );
#ifndef Q3_VM
	assert( Q_fabs(inv_denom) != 0.0f ); // bk010122 - zero vectors get here
#endif
	inv_denom = 1.0f / inv_denom;

	d = DotProduct( normal, p ) * inv_denom;

	n[0] = normal[0] * inv_denom;
	n[1] = normal[1] * inv_denom;
	n[2] = normal[2] * inv_denom;

	dst[0] = p[0] - d * n[0];
	dst[1] = p[1] - d * n[1];
	dst[2] = p[2] - d * n[2];
}

/*
================
MakeNormalVectors

Given a normalized forward vector, create two
other perpendicular vectors
================
*/
void MakeNormalVectors( const vec3_t forward, vec3_t right, vec3_t up) {
	float		d;

	// this rotate and negate guarantees a vector
	// not colinear with the original
	right[1] = -forward[0];
	right[2] = forward[1];
	right[0] = forward[2];

	d = DotProduct (right, forward);
	VectorMA (right, -d, forward, right);
	VectorNormalize (right);
	CrossProduct (right, forward, up);
}


void VectorRotate( vec3_t in, vec3_t matrix[3], vec3_t out )
{
	out[0] = DotProduct( in, matrix[0] );
	out[1] = DotProduct( in, matrix[1] );
	out[2] = DotProduct( in, matrix[2] );
}

//============================================================================

#if !idppc
/*
** float q_rsqrt( float number )
*/
float Q_rsqrt( float number )
{
	long i;
	float x2, y;
	const float threehalfs = 1.5F;

	x2 = number * 0.5F;
	y  = number;
	i  = * ( long * ) &y;						// evil floating point bit level hacking
	i  = 0x5f3759df - ( i >> 1 );               // what the fuck?
	y  = * ( float * ) &i;
	y  = y * ( threehalfs - ( x2 * y * y ) );   // 1st iteration
//	y  = y * ( threehalfs - ( x2 * y * y ) );   // 2nd iteration, this can be removed

#ifndef Q3_VM
#ifdef __linux__
	assert( !isnan(y) ); // bk010122 - FPE?
#endif
#endif
	return y;
}

float Q_fabs( float f ) {
	int tmp = * ( int * ) &f;
	tmp &= 0x7FFFFFFF;
	return * ( float * ) &tmp;
}
#endif

//============================================================

/*
===============
LerpAngle

===============
*/
float LerpAngle (float from, float to, float frac) {
	float	a;

	if ( to - from > 180 ) {
		to -= 360;
	}
	if ( to - from < -180 ) {
		to += 360;
	}
	a = from + frac * (to - from);

	return a;
}


/*
=================
AngleSubtract

Always returns a value from -180 to 180
=================
*/
float	AngleSubtract( float a1, float a2 ) {
	float	a;

	a = a1 - a2;
	assert(fabs(a) < 3600);
	while ( a > 180 ) {
		a -= 360;
	}
	while ( a < -180 ) {
		a += 360;
	}
	return a;
}


void AnglesSubtract( vec3_t v1, vec3_t v2, vec3_t v3 ) {
	v3[0] = AngleSubtract( v1[0], v2[0] );
	v3[1] = AngleSubtract( v1[1], v2[1] );
	v3[2] = AngleSubtract( v1[2], v2[2] );
}


float	AngleMod(float a) {
	a = (360.0/65536) * ((int)(a*(65536/360.0)) & 65535);
	return a;
}


/*
=================
AngleNormalize360

returns angle normalized to the range [0 <= angle < 360]
=================
*/
float AngleNormalize360 ( float angle ) {
	return (360.0 / 65536) * ((int)(angle * (65536 / 360.0)) & 65535);
}


/*
=================
AngleNormalize180

returns angle normalized to the range [-180 < angle <= 180]
=================
*/
float AngleNormalize180 ( float angle ) {
	angle = AngleNormalize360( angle );
	if ( angle > 180.0 ) {
		angle -= 360.0;
	}
	return angle;
}


/*
=================
AngleDelta

returns the normalized delta from angle1 to angle2
=================
*/
float AngleDelta ( float angle1, float angle2 ) {
	return AngleNormalize180( angle1 - angle2 );
}


//============================================================


/*
=================
SetPlaneSignbits
=================
*/
void SetPlaneSignbits (cplane_t *out) {
	int	bits, j;

	// for fast box on planeside test
	bits = 0;
	for (j=0 ; j<3 ; j++) {
		if (out->normal[j] < 0) {
			bits |= 1<<j;
		}
	}
	out->signbits = bits;
}


/*
==================
BoxOnPlaneSide

Returns 1, 2, or 1 + 2

// this is the slow, general version
int BoxOnPlaneSide2 (vec3_t emins, vec3_t emaxs, struct cplane_s *p)
{
	int		i;
	float	dist1, dist2;
	int		sides;
	vec3_t	corners[2];

	for (i=0 ; i<3 ; i++)
	{
		if (p->normal[i] < 0)
		{
			corners[0][i] = emins[i];
			corners[1][i] = emaxs[i];
		}
		else
		{
			corners[1][i] = emins[i];
			corners[0][i] = emaxs[i];
		}
	}
	dist1 = DotProduct (p->normal, corners[0]) - p->dist;
	dist2 = DotProduct (p->normal, corners[1]) - p->dist;
	sides = 0;
	if (dist1 >= 0)
		sides = 1;
	if (dist2 < 0)
		sides |= 2;

	return sides;
}

==================
*/
#if !( (defined __linux__ || __FreeBSD__) && (defined __i386__) && (!defined C_ONLY)) // rb010123

#if defined __LCC__ || defined C_ONLY || !id386

int BoxOnPlaneSide (vec3_t emins, vec3_t emaxs, struct cplane_s *p)
{
	float	dist1, dist2;
	int		sides;

// fast axial cases
	if (p->type < 3)
	{
		if (p->dist <= emins[p->type])
			return 1;
		if (p->dist >= emaxs[p->type])
			return 2;
		return 3;
	}

// general case
	switch (p->signbits)
	{
	case 0:
		dist1 = p->normal[0]*emaxs[0] + p->normal[1]*emaxs[1] + p->normal[2]*emaxs[2];
		dist2 = p->normal[0]*emins[0] + p->normal[1]*emins[1] + p->normal[2]*emins[2];
		break;
	case 1:
		dist1 = p->normal[0]*emins[0] + p->normal[1]*emaxs[1] + p->normal[2]*emaxs[2];
		dist2 = p->normal[0]*emaxs[0] + p->normal[1]*emins[1] + p->normal[2]*emins[2];
		break;
	case 2:
		dist1 = p->normal[0]*emaxs[0] + p->normal[1]*emins[1] + p->normal[2]*emaxs[2];
		dist2 = p->normal[0]*emins[0] + p->normal[1]*emaxs[1] + p->normal[2]*emins[2];
		break;
	case 3:
		dist1 = p->normal[0]*emins[0] + p->normal[1]*emins[1] + p->normal[2]*emaxs[2];
		dist2 = p->normal[0]*emaxs[0] + p->normal[1]*emaxs[1] + p->normal[2]*emins[2];
		break;
	case 4:
		dist1 = p->normal[0]*emaxs[0] + p->normal[1]*emaxs[1] + p->normal[2]*emins[2];
		dist2 = p->normal[0]*emins[0] + p->normal[1]*emins[1] + p->normal[2]*emaxs[2];
		break;
	case 5:
		dist1 = p->normal[0]*emins[0] + p->normal[1]*emaxs[1] + p->normal[2]*emins[2];
		dist2 = p->normal[0]*emaxs[0] + p->normal[1]*emins[1] + p->normal[2]*emaxs[2];
		break;
	case 6:
		dist1 = p->normal[0]*emaxs[0] + p->normal[1]*emins[1] + p->normal[2]*emins[2];
		dist2 = p->normal[0]*emins[0] + p->normal[1]*emaxs[1] + p->normal[2]*emaxs[2];
		break;
	case 7:
		dist1 = p->normal[0]*emins[0] + p->normal[1]*emins[1] + p->normal[2]*emins[2];
		dist2 = p->normal[0]*emaxs[0] + p->normal[1]*emaxs[1] + p->normal[2]*emaxs[2];
		break;
	default:
		dist1 = dist2 = 0;		// shut up compiler
		break;
	}

	sides = 0;
	if (dist1 >= p->dist)
		sides = 1;
	if (dist2 < p->dist)
		sides |= 2;

	return sides;
}
#else
#pragma warning( disable: 4035 )

__declspec( naked ) int BoxOnPlaneSide (vec3_t emins, vec3_t emaxs, struct cplane_s *p)
{
	static int bops_initialized;
	static int Ljmptab[8];

	__asm {

		push ebx
			
		cmp bops_initialized, 1
		je  initialized
		mov bops_initialized, 1
		
		mov Ljmptab[0*4], offset Lcase0
		mov Ljmptab[1*4], offset Lcase1
		mov Ljmptab[2*4], offset Lcase2
		mov Ljmptab[3*4], offset Lcase3
		mov Ljmptab[4*4], offset Lcase4
		mov Ljmptab[5*4], offset Lcase5
		mov Ljmptab[6*4], offset Lcase6
		mov Ljmptab[7*4], offset Lcase7
			
initialized:

		mov edx,dword ptr[4+12+esp]
		mov ecx,dword ptr[4+4+esp]
		xor eax,eax
		mov ebx,dword ptr[4+8+esp]
		mov al,byte ptr[17+edx]
		cmp al,8
		jge Lerror
		fld dword ptr[0+edx]
		fld st(0)
		jmp dword ptr[Ljmptab+eax*4]
Lcase0:
		fmul dword ptr[ebx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ebx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ebx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ecx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase1:
		fmul dword ptr[ecx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ebx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ebx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ecx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase2:
		fmul dword ptr[ebx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ecx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ebx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ecx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase3:
		fmul dword ptr[ecx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ecx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ebx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ecx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase4:
		fmul dword ptr[ebx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ebx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ecx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ebx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase5:
		fmul dword ptr[ecx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ebx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ecx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ebx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase6:
		fmul dword ptr[ebx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ecx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ecx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ecx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ebx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
		jmp LSetSides
Lcase7:
		fmul dword ptr[ecx]
		fld dword ptr[0+4+edx]
		fxch st(2)
		fmul dword ptr[ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[4+ecx]
		fld dword ptr[0+8+edx]
		fxch st(2)
		fmul dword ptr[4+ebx]
		fxch st(2)
		fld st(0)
		fmul dword ptr[8+ecx]
		fxch st(5)
		faddp st(3),st(0)
		fmul dword ptr[8+ebx]
		fxch st(1)
		faddp st(3),st(0)
		fxch st(3)
		faddp st(2),st(0)
LSetSides:
		faddp st(2),st(0)
		fcomp dword ptr[12+edx]
		xor ecx,ecx
		fnstsw ax
		fcomp dword ptr[12+edx]
		and ah,1
		xor ah,1
		add cl,ah
		fnstsw ax
		and ah,1
		add ah,ah
		add cl,ah
		pop ebx
		mov eax,ecx
		ret
Lerror:
		int 3
	}
}
#pragma warning( default: 4035 )

#endif
#endif

/*
=================
RadiusFromBounds
=================
*/
float RadiusFromBounds( const vec3_t mins, const vec3_t maxs ) {
	int		i;
	vec3_t	corner;
	float	a, b;

	for (i=0 ; i<3 ; i++) {
		a = fabs( mins[i] );
		b = fabs( maxs[i] );
		corner[i] = a > b ? a : b;
	}

	return VectorLength (corner);
}


void ClearBounds( vec3_t mins, vec3_t maxs ) {
	mins[0] = mins[1] = mins[2] = 99999;
	maxs[0] = maxs[1] = maxs[2] = -99999;
}

void AddPointToBounds( const vec3_t v, vec3_t mins, vec3_t maxs ) {
	if ( v[0] < mins[0] ) {
		mins[0] = v[0];
	}
	if ( v[0] > maxs[0]) {
		maxs[0] = v[0];
	}

	if ( v[1] < mins[1] ) {
		mins[1] = v[1];
	}
	if ( v[1] > maxs[1]) {
		maxs[1] = v[1];
	}

	if ( v[2] < mins[2] ) {
		mins[2] = v[2];
	}
	if ( v[2] > maxs[2]) {
		maxs[2] = v[2];
	}
}


vec_t VectorNormalize( vec3_t v ) {
	float	length, ilength;

	length = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
	length = sqrt (length);

	if ( length ) {
		ilength = 1/length;
		v[0] *= ilength;
		v[1] *= ilength;
		v[2] *= ilength;
	}
		
	return length;
}

vec_t VectorNormalize2( const vec3_t v, vec3_t out) {
	float	length, ilength;

	length = v[0]*v[0] + v[1]*v[1] + v[2]*v[2];
	length = sqrt (length);

	if (length)
	{
#ifndef Q3_VM // bk0101022 - FPE related
//	  assert( ((Q_fabs(v[0])!=0.0f) || (Q_fabs(v[1])!=0.0f) || (Q_fabs(v[2])!=0.0f)) );
#endif
		ilength = 1/length;
		out[0] = v[0]*ilength;
		out[1] = v[1]*ilength;
		out[2] = v[2]*ilength;
	} else {
#ifndef Q3_VM // bk0101022 - FPE related
//	  assert( ((Q_fabs(v[0])==0.0f) && (Q_fabs(v[1])==0.0f) && (Q_fabs(v[2])==0.0f)) );
#endif
		VectorClear( out );
	}
		
	return length;

}

void _VectorMA( const vec3_t veca, float scale, const vec3_t vecb, vec3_t vecc) {
	vecc[0] = veca[0] + scale*vecb[0];
	vecc[1] = veca[1] + scale*vecb[1];
	vecc[2] = veca[2] + scale*vecb[2];
}


vec_t _DotProduct( const vec3_t v1, const vec3_t v2 ) {
	return v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2];
}

void _VectorSubtract( const vec3_t veca, const vec3_t vecb, vec3_t out ) {
	out[0] = veca[0]-vecb[0];
	out[1] = veca[1]-vecb[1];
	out[2] = veca[2]-vecb[2];
}

void _VectorAdd( const vec3_t veca, const vec3_t vecb, vec3_t out ) {
	out[0] = veca[0]+vecb[0];
	out[1] = veca[1]+vecb[1];
	out[2] = veca[2]+vecb[2];
}

void _VectorCopy( const vec3_t in, vec3_t out ) {
	out[0] = in[0];
	out[1] = in[1];
	out[2] = in[2];
}

void _VectorScale( const vec3_t in, vec_t scale, vec3_t out ) {
	out[0] = in[0]*scale;
	out[1] = in[1]*scale;
	out[2] = in[2]*scale;
}

void Vector4Scale( const vec4_t in, vec_t scale, vec4_t out ) {
	out[0] = in[0]*scale;
	out[1] = in[1]*scale;
	out[2] = in[2]*scale;
	out[3] = in[3]*scale;
}


int Q_log2( int val ) {
	int answer;

	answer = 0;
	while ( ( val>>=1 ) != 0 ) {
		answer++;
	}
	return answer;
}



/*
=================
PlaneTypeForNormal
=================
*/
/*
int	PlaneTypeForNormal (vec3_t normal) {
	if ( normal[0] == 1.0 )
		return PLANE_X;
	if ( normal[1] == 1.0 )
		return PLANE_Y;
	if ( normal[2] == 1.0 )
		return PLANE_Z;
	
	return PLANE_NON_AXIAL;
}
*/


/*
================
MatrixMultiply
================
*/
void MatrixMultiply(float in1[3][3], float in2[3][3], float out[3][3]) {
	out[0][0] = in1[0][0] * in2[0][0] + in1[0][1] * in2[1][0] +
				in1[0][2] * in2[2][0];
	out[0][1] = in1[0][0] * in2[0][1] + in1[0][1] * in2[1][1] +
				in1[0][2] * in2[2][1];
	out[0][2] = in1[0][0] * in2[0][2] + in1[0][1] * in2[1][2] +
				in1[0][2] * in2[2][2];
	out[1][0] = in1[1][0] * in2[0][0] + in1[1][1] * in2[1][0] +
				in1[1][2] * in2[2][0];
	out[1][1] = in1[1][0] * in2[0][1] + in1[1][1] * in2[1][1] +
				in1[1][2] * in2[2][1];
	out[1][2] = in1[1][0] * in2[0][2] + in1[1][1] * in2[1][2] +
				in1[1][2] * in2[2][2];
	out[2][0] = in1[2][0] * in2[0][0] + in1[2][1] * in2[1][0] +
				in1[2][2] * in2[2][0];
	out[2][1] = in1[2][0] * in2[0][1] + in1[2][1] * in2[1][1] +
				in1[2][2] * in2[2][1];
	out[2][2] = in1[2][0] * in2[0][2] + in1[2][1] * in2[1][2] +
				in1[2][2] * in2[2][2];
}

// adnan
void AxisClear4(vec4_t Rin[4]) {
	int i, j;
	for(i = 0; i < 4; i++) {
		for(j = 0; j < 4; j++) {
			Rin[i][j] = 0;
		}
	}
}

void AxisCopy4(vec4_t Rin[4], vec4_t Rout[4]) {
	int i, j;
	for(i = 0; i < 4; i++) {
		for(j = 0; j < 4; j++) {
			Rout[i][j] = Rin[i][j];
		}
	}
}

void CartesianToHomogenousMatrix(vec3_t Rin[4], vec4_t Rout[4]) {
	int i, j;
	
	AxisClear4(Rout);
	
	for(i = 0; i < 3; i++) {
		for(j = 0; j < 3; j++) {
			Rout[i][j] = Rin[i][j];
		}
	}

	Rout[3][3] = 1;
}

void CartesianToHomogenousVector(vec3_t vin, vec4_t vout) {
	vout[0] = vin[0];
	vout[1] = vin[1];
	vout[2] = vin[2];
	vout[3] = 1;
}

void HomogenousToCartesianVector(vec4_t vin, vec3_t vout) {
	vout[0] = vin[0]/vin[3];
	vout[1] = vin[1]/vin[3];
	vout[2] = vin[2]/vin[3];
}

void MatrixMultiply4(vec4_t in1[4], vec4_t in2[4], vec4_t out[4]) {
	out[0][0] = in1[0][0] * in2[0][0] + in1[0][1] * in2[1][0] +
				in1[0][2] * in2[2][0] + in1[0][3] * in2[3][0];
	out[0][1] = in1[0][0] * in2[0][1] + in1[0][1] * in2[1][1] +
				in1[0][2] * in2[2][1] + in1[0][3] * in2[3][1];
	out[0][2] = in1[0][0] * in2[0][2] + in1[0][1] * in2[1][2] +
				in1[0][2] * in2[2][2] + in1[0][3] * in2[3][2];
	out[0][3] = in1[0][0] * in2[0][3] + in1[0][1] * in2[1][3] +
				in1[0][2] * in2[2][3] + in1[0][3] * in2[3][3];
	out[1][0] = in1[1][0] * in2[0][0] + in1[1][1] * in2[1][0] +
				in1[1][2] * in2[2][0] + in1[1][3] * in2[3][0];
	out[1][1] = in1[1][0] * in2[0][1] + in1[1][1] * in2[1][1] +
				in1[1][2] * in2[2][1] + in1[1][3] * in2[3][1];
	out[1][2] = in1[1][0] * in2[0][2] + in1[1][1] * in2[1][2] +
				in1[1][2] * in2[2][2] + in1[1][3] * in2[3][2];
	out[1][3] = in1[1][0] * in2[0][3] + in1[1][1] * in2[1][3] +
				in1[1][2] * in2[2][3] + in1[1][3] * in2[3][3];
	out[2][0] = in1[2][0] * in2[0][0] + in1[2][1] * in2[1][0] +
				in1[2][2] * in2[2][0] + in1[2][3] * in2[3][0];
	out[2][1] = in1[2][0] * in2[0][1] + in1[2][1] * in2[1][1] +
				in1[2][2] * in2[2][1] + in1[2][3] * in2[3][1];
	out[2][2] = in1[2][0] * in2[0][2] + in1[2][1] * in2[1][2] +
				in1[2][2] * in2[2][2] + in1[2][3] * in2[3][2];
	out[2][3] = in1[2][0] * in2[0][3] + in1[2][1] * in2[1][3] +
				in1[2][2] * in2[2][3] + in1[2][3] * in2[3][3];

	out[3][0] = in1[3][0] * in2[0][0] + in1[3][1] * in2[1][0] +
				in1[3][2] * in2[2][0] + in1[3][3] * in2[3][0];
	out[3][1] = in1[2][0] * in2[0][1] + in1[3][1] * in2[1][1] +
				in1[3][2] * in2[2][1] + in1[3][3] * in2[3][1];
	out[3][2] = in1[2][0] * in2[0][2] + in1[3][1] * in2[1][2] +
				in1[3][2] * in2[2][2] + in1[3][3] * in2[3][2];
	out[3][3] = in1[2][0] * in2[0][3] + in1[3][1] * in2[1][3] +
				in1[3][2] * in2[2][3] + in1[3][3] * in2[3][3];
}

void VectorTransform4(vec4_t T[4], vec4_t vin, vec4_t vout) {
	vout[0] = T[0][0] * vin[0] + T[0][1] * vin[1] + T[0][2] * vin[2] + T[0][3] * vin[3];
	vout[1] = T[1][0] * vin[0] + T[1][1] * vin[1] + T[1][2] * vin[2] + T[1][3] * vin[3];
	vout[2] = T[2][0] * vin[0] + T[2][1] * vin[1] + T[2][2] * vin[2] + T[2][3] * vin[3];
	vout[3] = T[3][0] * vin[0] + T[3][1] * vin[1] + T[3][2] * vin[2] + T[3][3] * vin[3];
}

void TransposeMatrix4(vec4_t Rin[4], vec4_t Rout[4]) {
	int i, j;

	for(i = 0; i < 4; i++) {
		for(j = 0; j < 4; j++) {
			Rout[j][i] = Rin[i][j];
		}
	}
}

void InverseTranslationMatrix4(vec4_t point, vec4_t R[4]) {
	int i, j;

	// clear out R
	for(i = 0; i < 4; i++) {
		for(j = 0; j < 4; j++) {
			if(i == j)
				R[i][j] = 1;
			else
				R[i][j] = 0;
		}
	}

	// set the translation values
	R[0][3] = -point[0];
	R[1][3] = -point[1];
	R[2][3] = -point[2];
}

void PerspectiveProjectionMatrix4(float fov_y, float aspect, float znear, float zfar, vec4_t R[4]) {
	int i, j;
	float f = 1/((float)tan(fov_y/2.0f));

	// clear out R
	for(i = 0; i < 4; i++) {
		for(j = 0; j < 4; j++) {
			R[i][j] = 0;
		}
	}

	// make the matrix
	R[0][0] = f / aspect;
	R[1][1] = f;
	R[2][2] = (zfar + znear) / (znear - zfar);
	R[2][3] = (2 * zfar * znear) / (znear - zfar);
	R[3][2] = -1;
}

// adnan
void RotationMatrixAroundDir(vec3_t dir, float radians, float r[3][3]) {
   float mag;
   float ux, uy, uz; // direction cosines...
   int i, j;
   
   radians *= -1;
   mag = VectorLength(dir);

   for(i = 0; i < 3; i++) {
      for(j = 0; j < 3; j++) {
         if(i == j)
            r[i][j] = 1;
         else
            r[i][j] = 0;
      }
   }

   if(mag <= 0.0001)
      return;

   ux = dir[0] / mag;
   uy = dir[1] / mag;
   uz = dir[2] / mag;

   r[0][0] = (ux * ux) * (1 - cos(radians)) + cos(radians);
   r[0][1] = (ux * uy) * (1 - cos(radians)) - (uz * sin(radians));
   r[0][2] = (ux * uz) * (1 - cos(radians)) + (uy * sin(radians));
   r[1][0] = (ux * uy) * (1 - cos(radians)) + (uz * sin(radians));
   r[1][1] = (uy * uy) * (1 - cos(radians)) + cos(radians);
   r[1][2] = (uy * uz) * (1 - cos(radians)) - (ux * sin(radians));
   r[2][0] = (ux * uz) * (1 - cos(radians)) - (uy * sin(radians));
   r[2][1] = (uy * uz) * (1 - cos(radians)) + (ux * sin(radians));
   r[2][2] = (uz * uz) * (1 - cos(radians)) + cos(radians);

}

void AngleVectors( const vec3_t angles, vec3_t forward, vec3_t right, vec3_t up) {
	float		angle;
	static float		sr, sp, sy, cr, cp, cy;
	// static to help MS compiler fp bugs

	angle = angles[YAW] * (M_PI*2 / 360);
	sy = sin(angle);
	cy = cos(angle);
	angle = angles[PITCH] * (M_PI*2 / 360);
	sp = sin(angle);
	cp = cos(angle);
	angle = angles[ROLL] * (M_PI*2 / 360);
	sr = sin(angle);
	cr = cos(angle);

	if (forward)
	{
		forward[0] = cp*cy;
		forward[1] = cp*sy;
		forward[2] = -sp;
	}
	if (right)
	{
		right[0] = (-1*sr*sp*cy+-1*cr*-sy);
		right[1] = (-1*sr*sp*sy+-1*cr*cy);
		right[2] = -1*sr*cp;
	}
	if (up)
	{
		up[0] = (cr*sp*cy+-sr*-sy);
		up[1] = (cr*sp*sy+-sr*cy);
		up[2] = cr*cp;
	}
}

/*
** assumes "src" is normalized
*/
void PerpendicularVector( vec3_t dst, const vec3_t src )
{
	int	pos;
	int i;
	float minelem = 1.0F;
	vec3_t tempvec;

	/*
	** find the smallest magnitude axially aligned vector
	*/
	for ( pos = 0, i = 0; i < 3; i++ )
	{
		if ( fabs( src[i] ) < minelem )
		{
			pos = i;
			minelem = fabs( src[i] );
		}
	}
	tempvec[0] = tempvec[1] = tempvec[2] = 0.0F;
	tempvec[pos] = 1.0F;

	/*
	** project the point onto the plane defined by src
	*/
	ProjectPointOnPlane( dst, tempvec, src );

	/*
	** normalize the result
	*/
	VectorNormalize( dst );
}

// This is the VC libc version of rand() without multiple seeds per thread or 12 levels
// of subroutine calls.
// Both calls have been designed to minimise the inherent number of float <--> int 
// conversions and the additional math required to get the desired value.
// eg the typical tint = (rand() * 255) / 32768
// becomes tint = irand(0, 255)

static unsigned long	holdrand = 0x89abcdef;

void Rand_Init(int seed)
{
	holdrand = seed;
}

// Returns a float min <= x < max (exclusive; will get max - 0.00001; but never max)

float flrand(float min, float max)
{
	float	result;

	assert((max - min) < 32768);

	holdrand = (holdrand * 214013L) + 2531011L;
	result = (float)(holdrand >> 17);						// 0 - 32767 range
	result = ((result * (max - min)) / 32768.0F) + min;

	return(result);
}

// Returns an integer min <= x <= max (ie inclusive)

int irand(int min, int max)
{
	int		result;

	assert((max - min) < 32768);

	max++;
	holdrand = (holdrand * 214013L) + 2531011L;
	result = holdrand >> 17;
	result = ((result * (max - min)) >> 15) + min;
	return(result);
}

float powf ( float x, int y )
{
	float r = x;
	for ( y--; y>0; y-- )
		r = r * r;
	return r;
}

// adnan
///////////////////
// begin quat stuff
///////////////////
void Quaternion(vec3_t dir, double rads, quat_t *q) {
   float sr = sin( rads / 2 );
   float mag = VectorLength(dir);
   q->w = cos( rads / 2 );
   q->x = dir[0]/mag * sr;
   q->y = dir[1]/mag * sr;
   q->z = dir[2]/mag * sr;
}

int QuaternionIsEmpty(quat_t *q) {
	return (q->x*q->x < 0.001) && (q->y*q->y < 0.001) && (q->z*q->z < 0.001) && (q->w*q->w < 0.001); 
}

void QuaternionIdentity(quat_t *q) {
   q->w = 1;
   q->x = q->y = q->z = 0;
}

void QuaternionConjugate(quat_t a, quat_t *q) {
   QuaternionNormalize(&a);
   QuaternionCopy(a, q);
   q->x *= -1;
   q->y *= -1;
   q->z *= -1;
}

void QuaternionInverse(quat_t a, quat_t *q) {
	quat_t aconj, qtemp;
	float mag;

	QuaternionConjugate(a, &aconj);

	mag = QuaternionMagnitude(a);
	mag = mag * mag;

	q->x = aconj.x / mag;
	q->y = aconj.y / mag;
	q->z = aconj.z / mag;
	q->w = aconj.w / mag;
}

void QuaternionCopy(quat_t a, quat_t *b) {
   b->w = a.w;
   b->x = a.x;
   b->y = a.y;
   b->z = a.z;
}

void QuaternionSlerp(quat_t a, quat_t b, int t_start, int t_duration, int t, quat_t *q) {
	quat_t qtemp1, qtemp2;
	float delta_t, sinphi, phi, scale;
	
	// t
	delta_t = (float)(t - t_start)/(float)t_duration;

	// a^-1
	//QuaternionConjugate(a, &qtemp1);
	QuaternionInverse(a, &qtemp1);

	// b.a^-1
	QuaternionMultiply(&qtemp1, &b, &qtemp2);

   sinphi = sqrt(qtemp2.x * qtemp2.x + qtemp2.y * qtemp2.y + qtemp2.z * qtemp2.z);
	
	if(sinphi < 0.0001) {
		QuaternionCopy(a, q);
		return;
	}
	
	// inverse the sin and return the power
	//phi = ((qtemp2.w < _R(0)) ? -t : t) * asin(sinphi);
	phi = delta_t * asin(sinphi);
	scale = sin(phi) / sinphi;

	qtemp1.w = cos(phi);
	qtemp1.x = qtemp2.x * scale;
	qtemp1.y = qtemp2.y * scale;
	qtemp1.z = qtemp2.z * scale;

	QuaternionMultiply(&qtemp1, &a, q);

	return;
}

// doesnt work at all. looks like i made up my own shit.
void QuaternionLerp(quat_t a, quat_t b, int t_start, int t_duration, int t, quat_t *q) {
	quat_t qtemp1, qtemp2;
	float delta_t, x, y, z, w;

	//QuaternionNormalize(&a);
	//QuaternionNormalize(&b);
	
	Com_Printf("a: %f,%f,%f,%f\n", a.x, a.y, a.z, a.w);

	delta_t = (float)(t - t_start)/(float)t_duration;

	// q = (b.a^-1)^t . a

	/*
	// a^-1
	QuaternionConjugate(a, &qtemp1);

	// b.a^-1
	QuaternionMultiply(&qtemp1, &b, &qtemp2);

	// (b.a^-1)^t
	qtemp2.x *= delta_t;
	qtemp2.y *= delta_t;
	qtemp2.z *= delta_t;
	qtemp2.w *= delta_t;

	// (b.a^-1)^t . a
	QuaternionMultiply(&qtemp2, &a, q);
	*/

	// a^-1
	//QuaternionConjugate(a, &qtemp1);
	QuaternionInverse(a, &qtemp1);

	// b.a^-1
	QuaternionMultiply(&qtemp1, &b, &qtemp2);

	// (b.a^-1)^t
	//Com_Printf("q1: %f,%f,%f,%f\n", qtemp2.x, qtemp2.y, qtemp2.z, qtemp2.w);

	qtemp2.x *= delta_t;
	qtemp2.y *= delta_t;
	qtemp2.z *= delta_t;
	qtemp2.w *= delta_t;

	Com_Printf("q2: %f,%f,%f,%f\n", qtemp2.x, qtemp2.y, qtemp2.z, qtemp2.w);

	// (b.a^-1)^t . a
	QuaternionMultiply(&a, &qtemp2, &qtemp1);

	QuaternionCopy(qtemp1, q);

	Com_Printf("q: %f,%f,%f,%f\n", q->x, q->y, q->z, q->w);
}

void QuaternionFromAngles(vec3_t angles, quat_t *q) {
	float pitch, yaw, roll;
	vec3_t dir;
	quat_t qr, qy, qp, qtemp;
	/*
	// if we dont need to do a rotation
	if( ((deltaAngles[PITCH]*deltaAngles[PITCH]) < 0.01)
		&& ((deltaAngles[YAW]*deltaAngles[YAW]) < 0.01)
		&& ((deltaAngles[ROLL]*deltaAngles[ROLL]) < 0.01) )
	{
		QuaternionCopy(q_previous, &(cg_quats[cent->currentState.number]));

		
		//if(cent->currentState.number == cg.predictedPlayerEntity.currentState.number) {
		//	QuaternionCopy(q_previous, &(cg.Q));
		//}
		
	}
	*/

	// convert delta angles to radians
	pitch = DEG2RAD(angles[PITCH]);
	yaw = DEG2RAD(angles[YAW]);
	roll = DEG2RAD(angles[ROLL]);

	// pulling the stick left or right causes a banked yaw, combination roll and yaw
	//roll -= DEG2RAD(deltaAngles[YAW] * 0.25f);

	// generate roll quat
	VectorSet(dir, 1, 0, 0);
	Quaternion(dir, roll, &qr);
	QuaternionNormalize(&qr);

	// generate pitch quat
	VectorSet(dir, 0, 1, 0);
	Quaternion(dir, pitch, &qp);
	QuaternionNormalize(&qp);

	// generate yaw quat
	VectorSet(dir, 0, 0, 1);
	Quaternion(dir, yaw, &qy);
	QuaternionNormalize(&qy);

	// concatenate delta rotations
	QuaternionMultiply(&qr, &qp, &qtemp);
	QuaternionMultiply(&qtemp, &qy, q);

	QuaternionNormalize(q);	
}

void QuaternionFromTieFloats(float *tieFloats, quat_t *q) {
   q->w = tieFloats[Q_W];
   q->x = tieFloats[Q_X];
   q->y = tieFloats[Q_Y];
   q->z = tieFloats[Q_Z];
}

void QuaternionToTieFloats(float *tieFloats, quat_t *q) {
   tieFloats[Q_W] = q->w;
   tieFloats[Q_X] = q->x;
   tieFloats[Q_Y] = q->y;
   tieFloats[Q_Z] = q->z;
}

void QuaternionToApos(trajectory_t *apos, quat_t *q) {
   apos->trBase[0] = q->x;
   apos->trBase[1] = q->y;
   apos->trBase[2] = q->z;
   apos->trDelta[0] = q->w;
   //apos->trDelta[1] = ps->tieFloats[Q_RESET_BGAME];
   //apos->trDelta[2] = ps->tieFloats[Q_RESET_CGAME];
}

void QuaternionFromApos(trajectory_t *apos, quat_t *q) {
   q->x = apos->trBase[0];
   q->y = apos->trBase[1];
   q->z = apos->trBase[2];
   q->w = apos->trDelta[0];
}

void AnglesFromTieFloats(float *tieFloats, vec3_t *angles) {
   *angles[PITCH] = tieFloats[PREV_ANGLES_PITCH];
   *angles[YAW] = tieFloats[PREV_ANGLES_YAW];
   *angles[ROLL] = tieFloats[PREV_ANGLES_ROLL];
}

void AnglesToTieFloats(float *tieFloats, vec3_t *angles) {
   tieFloats[PREV_ANGLES_PITCH] = *angles[PITCH];
   tieFloats[PREV_ANGLES_YAW] = *angles[YAW];
   tieFloats[PREV_ANGLES_ROLL] = *angles[ROLL];
}

void QuaternionFromString(char *string, quat_t *q) {
   QuaternionIdentity(q);
   sscanf(string, "%f,%f,%f,%f", &(q->x), &(q->y), &(q->z), &(q->w)); 
}

void AnglesToQuat (const vec3_t angles, quat_t *q)
{
	vec3_t a;
	float cr, cp, cy, sr, sp, sy, cpcy, spsy;

	/*
   a[PITCH] = (M_PI/360.0) * angles[PITCH];
	a[YAW] = (M_PI/360.0) * angles[YAW];
	a[ROLL] = (M_PI/360.0) * angles[ROLL];
   */

   a[PITCH] = angles[PITCH] / 2.0f;
	a[YAW] = angles[YAW] / 2.0f;
	a[ROLL] = angles[ROLL] / 2.0f;
	
   cr = cos(a[ROLL]);
	cp = cos(a[PITCH]);
	cy = cos(a[YAW]);
   
	sr = sin(a[ROLL]);
	sp = sin(a[PITCH]);
	sy = sin(a[YAW]);

	cpcy = cp * cy;
	spsy = sp * sy;
   q->w = cr * cpcy + sr * spsy; // w
	q->x = sr * cpcy - cr * spsy; // x
	q->y = cr * sp * cy + sr * cp * sy; // y
	q->z = cr * cp * sy - sr * sp * cy; // z
}

void QuatToAxis(quat_t q, vec3_t axis[3])
{
	/*
   float wx, wy, wz, xx, yy, yz, xy, xz, zz, x2, y2, z2;

	x2 = q.x + q.x;
	y2 = q.y + q.y;
	z2 = q.z + q.z;
	xx = q.x * x2;
	xy = q.x * y2;
	xz = q.x * z2;
	yy = q.y * y2;
	yz = q.y * z2;
	zz = q.z * z2;
	wx = q.w * x2;
	wy = q.w * y2;
	wz = q.w * z2;

	axis[0][0] = 1.0 - (yy + zz);
	axis[1][0] = xy - wz;
	axis[2][0] = xz + wy;

	axis[0][1] = xy + wz;
	axis[1][1] = 1.0 - (xx + zz);
	axis[2][1] = yz - wx;

	axis[0][2] = xz - wy;
	axis[1][2] = yz + wx;
	axis[2][2] = 1.0 - (xx + yy);
   */

   float w, x, y, z, xx, yy, zz;

   QuaternionNormalize(&q);

	w = q.w;
	x = q.x;
	y = q.y;
	z = q.z;

	xx = x * x;
	yy = y * y;
	zz = z * z;

	axis[0][0] = 1.0 - 2.0 * ( yy + zz );
	axis[1][0] = 2.0 * ( x * y + w * z );
	axis[2][0] = 2.0 * ( x * z - w * y );

	axis[0][1] = 2.0 * ( x * y - w * z );
	axis[1][1] = 1.0 - 2.0 * ( xx + zz );
	axis[2][1] = 2.0 * ( y * z + w * x );

	axis[0][2] = 2.0 * ( x * z + w * y );
	axis[1][2] = 2.0 * ( y * z - w * x );
	axis[2][2] = 1.0 - 2.0 * ( xx + yy );
}

void QuatMul(const quat_t q1, const quat_t q2, quat_t *res)
{
	float A, B, C, D, E, F, G, H;

	A = (q1.w + q1.x)*(q2.w + q2.x);
	B = (q1.z - q1.y)*(q2.y - q2.z);
	C = (q1.w - q1.x)*(q2.y + q2.z);
	D = (q1.y + q1.z)*(q2.w - q2.x);
	E = (q1.x + q1.z)*(q2.x + q2.y);
	F = (q1.x - q1.z)*(q2.x - q2.y);
	G = (q1.w + q1.y)*(q2.w - q2.z);
	H = (q1.w - q1.y)*(q2.w + q2.z);

	res->w = B + (H - E - F + G)*0.5;
	res->x = A - (E + F + G + H)*0.5;
	res->y = C + (E - F + G - H)*0.5;
	res->z = D + (E - F - G + H)*0.5;
}


float QuaternionMagnitude( quat_t q )
{
	/* fp precision isn't really all that precise &gt;_&lt; */
	return (float)sqrt( (double)(q.w*q.w + q.x*q.x + q.y*q.y + q.z*q.z) );
}

float QuaternionNormalize( quat_t *q )
{
	float mag;

	mag = QuaternionMagnitude( *q );

	if ( mag == 0.0f )
		return 0;

	q->x /= mag;
	q->y /= mag;
	q->z /= mag;
	q->w /= mag;

	return mag;
}

float QuaternionMultiply( quat_t *a, quat_t *b, quat_t *r )
{
	/* Anybody come up with something faster ? */

	r->w =	      b.w * a.w
						- b.x * a.x
						- b.y * a.y
						- b.z * a.z;

	r->x =	      b.w * a.x
						+ b.x * a.w
						+ b.y * a.z
						- b.z * a.y;

	r->y =	      b.w * a.y
						- b.x * a.z
						+ b.y * a.w
						+ b.z * a.x;

	r->z =	      b.w * a.z
						+ b.x * a.y
						- b.y * a.x
						+ b.z * a.w;

	//return QuaternionNormalize( r );
	return 1;
}


float QuaternionToAxis( quat_t q, vec3_t axis[3])
{
   /*
   axis[0][0] = 1 - ( 2*q.z*q.z + 2*q.y*q.y );
	axis[0][2] = 2*q.x*q.x - 2*q.y*q.w;
	axis[0][1] = 2*q.x*q.y + 2*q.z*q.w;

   axis[2][0] = 2*q.x*q.z + 2*q.y*q.z;
   axis[2][2] = 1 - ( 2*q.x*q.x + 2*q.y*q.y);
   axis[2][1] = 2*q.z*q.y - 2*q.x*q.w;
	
   axis[1][0] = 2*q.x*q.y - 2*q.z*q.w;
   axis[1][2] = 2*q.z*q.y + 2*q.x*q.w;
   axis[1][1] = 1 - ( 2*q.x*q.x + 2*q.z*q.z );
   */
   
   float wx, wy, wz, xx, yy, yz, xy, xz, zz, x2, y2, z2;

	x2 = q.x + q.x;
	y2 = q.y + q.y;
	z2 = q.z + q.z;
	xx = q.x * x2;
	xy = q.x * y2;
	xz = q.x * z2;
	yy = q.y * y2;
	yz = q.y * z2;
	zz = q.z * z2;
	wx = q.w * x2;
	wy = q.w * y2;
	wz = q.w * z2;

	axis[0][0] = 1.0 - (yy + zz);
	axis[1][0] = xy - wz;
	axis[2][0] = xz + wy;

	axis[0][1] = xy + wz;
	axis[1][1] = 1.0 - (xx + zz);
	axis[2][1] = yz - wx;

	axis[0][2] = xz - wy;
	axis[1][2] = yz + wx;
	axis[2][2] = 1.0 - (xx + yy);
   
}

/*
  quat to euler helpers
*/

/*
#define R11(psi, theta, phi) (cos(theta) * cos(phi));
#define R12(psi, theta, phi) ((sin(psi) * sin(theta) * cos(phi)) - ((cos(psi) * sin(phi))));
#define R13(psi, theta, phi) (((cos(psi) * sin(theta) * cos(phi)) + ((sin(psi) * sin(phi))));
#define R22(psi, theta, phi) (((sin(psi) * sin(theta) * sin(phi)) + ((cos(psi) * cos(phi))));
#define R23(psi, theta, phi) (((cos(psi) * sin(theta) * sin(phi)) - ((sin(psi) * cos(phi))));
#define R33(psi, theta, phi) (cos(psi) * cos(theta));
*/

static double R11(float psi, float theta, float phi) {
   return (cos(theta) * cos(phi));
}

static double R12(float psi, float theta, float phi) {
   return ((sin(psi) * sin(theta) * cos(phi)) - ((cos(psi) * sin(phi))));
}

static double R13(float psi, float theta, float phi) {
   return (cos(psi) * sin(theta) * cos(phi)) + ((sin(psi) * sin(phi)));
}

static double R22(float psi, float theta, float phi) {
   return (sin(psi) * sin(theta) * sin(phi)) + ((cos(psi) * cos(phi)));
}

static double R23(float psi, float theta, float phi) {
   return (cos(psi) * sin(theta) * sin(phi)) - ((sin(psi) * cos(phi)));
}

static double R33(float psi, float theta, float phi) {
   return (cos(psi) * cos(theta));
}

static int areSimilar(double a, double b, double threshHold) {
   double diff = 0;
   diff = sqrt( (a - b)*(a - b) );
   
   if(diff >= threshHold)
      return 0;
   else
      return 1;
}

int MatrixToEuler(vec3_t m[3], vec3_t *angles) {
   // triplets of psi, theta, phi that can describe the current rotation matrix
   vec3_t triplets[8];
   
   int i;
   int final_triplet_found = 0;

   // fill the triplets

   // theta:
   triplets[0][1] = -1 * asin(m[2][0]);
   triplets[1][1] = triplets[0][1];
   triplets[2][1] = triplets[0][1];
   triplets[3][1] = triplets[0][1];
   triplets[4][1] = M_PI - triplets[0][1];
   triplets[5][1] = triplets[4][1];
   triplets[6][1] = triplets[4][1];
   triplets[7][1] = triplets[4][1];

   // psi:
   triplets[0][0] = asin(m[2][1] / cos(triplets[0][1]));
   triplets[1][0] = M_PI - triplets[0][0];
   triplets[2][0] = triplets[0][0];
   triplets[3][0] = triplets[1][0];
   triplets[4][0] = asin(m[2][1] / cos(triplets[4][1]));
   triplets[5][0] = M_PI - triplets[4][0];
   triplets[6][0] = triplets[4][0];
   triplets[7][0] = triplets[5][0];

   // phi:
   triplets[0][2] = asin(m[1][0] / cos(triplets[0][1]));
   triplets[1][2] = triplets[0][2];
   triplets[2][2] = M_PI - asin(m[1][0] / cos(triplets[0][1]));
   triplets[3][2] = triplets[2][2];
   triplets[4][2] = asin(m[1][0] / cos(triplets[4][1]));
   triplets[5][2] = triplets[4][2];
   triplets[6][2] = M_PI - asin(m[1][0] / cos(triplets[4][1]));
   triplets[7][2] = triplets[6][2];

   // compare triplet products using macros to m[i][j] values
   for(i = 0; i < 8; i++) {
      /*
      float diff = 0;
      
      // R[1][1] == m[0][0]
      diff = R11(triplets[i][0], triplets[i][1], triplets[i][2]) - m[0][0];
      diff *= diff;
      diff = sqrt(diff);

      if(diff >= 0.01)
         continue;
      
      // R[1][2] == m[0][1]
      diff = R12(triplets[i][0], triplets[i][1], triplets[i][2]) - m[0][1];
      diff *= diff;
      diff = sqrt(diff);

      if(diff >= 0.01)
         continue;

      // R[1][3] == m[0][2]
      diff = R13(triplets[i][0], triplets[i][1], triplets[i][2]) - m[0][2];
      diff *= diff;
      diff = sqrt(diff);

      if(diff >= 0.01)
         continue;

      // R[2][2] == m[1][1]
      diff = R22(triplets[i][0], triplets[i][1], triplets[i][2]) - m[1][1];
      diff *= diff;
      diff = sqrt(diff);

      if(diff >= 0.01)
         continue;

      // R[2][3] == m[1][2]
      diff = R23(triplets[i][0], triplets[i][1], triplets[i][2]) - m[1][2];
      diff *= diff;
      diff = sqrt(diff);

      if(diff >= 0.01)
         continue;

      // R[3][3] == m[2][2]
      diff = R33(triplets[i][0], triplets[i][1], triplets[i][2]) - m[2][2];
      diff *= diff;
      diff = sqrt(diff);

      if(diff >= 0.01)
         continue;

      */

      if( !areSimilar( R11(triplets[i][0], triplets[i][1], triplets[i][2]), m[0][0], 0.01 ) )
         continue;
      if( !areSimilar(R12(triplets[i][0], triplets[i][1], triplets[i][2]), m[0][1], 0.01 ) )
         continue;
      if( !areSimilar(R13(triplets[i][0], triplets[i][1], triplets[i][2]), m[0][2], 0.01 ) )
         continue;
      if( !areSimilar(R22(triplets[i][0], triplets[i][1], triplets[i][2]), m[1][1], 0.01 ) )
         continue;
      if( !areSimilar(R23(triplets[i][0], triplets[i][1], triplets[i][2]), m[1][2], 0.01 ) )
         continue;
      if( !areSimilar(R33(triplets[i][0], triplets[i][1], triplets[i][2]), m[2][2], 0.01 ) )
         continue;

      // if we got here, we have a match
      final_triplet_found = 1;

      // no need to continue
      break;
   }

   if(final_triplet_found) {
      *angles[0] = triplets[i][0];
      *angles[1] = triplets[i][1];
      *angles[2] = triplets[i][2];

      return 1;
   } else {
      return 0;
   }
}

void QuaternionToEuler( quat_t q, float *p, float *y, float *r) {

   //tan(yaw) = 2(q1q2+q4q3) / (q42 + q12 - q22 - q32) 
   //*y = atan( 2 * (q.x*q.y + q.w*q.z) / (q.w * q.w + q.x * q.x - q.y * q.y - q.z * q.z) );
   //*y = atan( -2 * (q.x*q.z + q.w*q.y) / (q.w * q.w + q.x * q.x - q.y * q.y - q.z * q.z) );
   //sin(pitch) = -2(q1q3-q4q2) 
   *p = -1 * ( acos( -2 * (q.x*q.z - q.w*q.y) ) - (M_PI / 2) );
   //*p = -1 * ( acos( 2 * (q.x*q.y - q.w*q.z) ) - (M_PI / 2) );
   //tan(roll)  =  2(q4q1+q2q3) / (q42 - q12 - q22 + q32)  
   //*r = atan( 2 * (q.w*q.x + q.y*q.z) / (q.w * q.w - q.x * q.x - q.y * q.y + q.z * q.z) );
   //*r = atan( -2 * (q.w*q.x + q.y*q.z) / (q.w * q.w - q.x * q.x + q.y * q.y - q.z * q.z) );
   

	/*
   double sint,cost,sinv,cosv,sinf,cosf,ex,ey,ez;

	sint = (2*q.w*q.y)-(2*q.x*q.z);
	cost = sqrt(1 - (sint * sint));
	if (cost != 0.0){
		sinv = ((2*q.y*q.z)+(2*q.w*q.x))/cost;
		cosv = (1-(2*q.x*q.x)-(2*q.y*q.y))/cost;
		sinf = (1-(2*q.x*q.x)-(2*q.y*q.y))/cost;
		cosf = (1-(2*q.y*q.y)-(2*q.z*q.z))/cost;
	}else {
		sinv = (2*q.w*q.x)-(2*q.y*q.z);
		cosv = 1-(2*q.x*q.x)-(2*q.z*q.z);
		sinf = 0.0;
		cosf = 1.0;
	}

	*y = atan2(sinv,cosv);
	*p = atan2(sint,cost);
	*r = atan2(sinf,cosf);  //range -pi +pi
   */
}

void QuatToAngles (const quat_t q, vec3_t a)
{
	quat_t q2;
	q2.w = q.w*q.w;
	q2.x = q.x*q.x;
	q2.y = q.y*q.y;
	q2.z = q.z*q.z;
	a[ROLL] = (180.0/M_PI)*atan2 (2*(q.y*q.z + q.x*q.w) , (-q2.x - q2.y + q2.z + q2.w));
	a[PITCH] = (180.0/M_PI)*asin (-2*(q.x*q.z - q.y*q.w));
	a[YAW] = (180.0/M_PI)*atan2 (2*(q.x*q.y + q.z*q.w) , (q2.x - q2.y - q2.z + q2.w));
}
