/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// r_surf.c: surface-related refresh code

#include "quakedef.h"
#include "bsp_render.h"
#include "gl_mirror.h"

extern	qboolean gl_mtexable;

int			lightmap_textures;

unsigned	blocklights[BLOCK_WIDTH*BLOCK_HEIGHT*3]; // Tomaz - Lit Support

int			active_lightmaps;

glpoly_t	*lightmap_polys[MAX_LIGHTMAPS];
qboolean	lightmap_modified[MAX_LIGHTMAPS];
glRect_t	lightmap_rectchange[MAX_LIGHTMAPS];
byte		lightmaps[4*MAX_LIGHTMAPS*BLOCK_WIDTH*BLOCK_HEIGHT];

int			allocated[MAX_LIGHTMAPS][BLOCK_WIDTH];

msurface_t  *skychain		= NULL;
msurface_t  *waterchain		= NULL;

/*
===============
R_AddDynamicLights
===============
*/
void R_AddDynamicLights (msurface_t *surf)
{
	int			lnum;
	int			sd, td;
	float		dist, rad, minlight;
	vec3_t		impact, local;
	int			s, t;
	int			i;
	int			smax, tmax;
	mtexinfo_t	*tex;
	// Tomaz - Lit Support Begin
	float		cred, cgreen, cblue, brightness;
	unsigned	*bl;
	// Tomaz - Lit Support End

	smax = (surf->extents[0]>>4)+1;
	tmax = (surf->extents[1]>>4)+1;
	tex = surf->texinfo;

	for (lnum=0 ; lnum<MAX_DLIGHTS ; lnum++)
	{
		if ( !(surf->dlightbits & (1<<lnum) ) )
			continue;		// not lit by this light

		rad = cl_dlights[lnum].radius;
		dist = DotProduct (cl_dlights[lnum].origin, surf->plane->normal) -
				surf->plane->dist;
		rad -= fabs(dist);
		minlight = cl_dlights[lnum].minlight;
		if (rad < minlight)
			continue;
		minlight = rad - minlight;

		for (i=0 ; i<3 ; i++)
		{
			impact[i] = cl_dlights[lnum].origin[i] -
					surf->plane->normal[i]*dist;
		}

		local[0] = DotProduct (impact, tex->vecs[0]) + tex->vecs[0][3];
		local[1] = DotProduct (impact, tex->vecs[1]) + tex->vecs[1][3];

		local[0] -= surf->texturemins[0];
		local[1] -= surf->texturemins[1];

		// Tomaz - Lit Support Begin
		bl		= blocklights;
		cred	= cl_dlights[lnum].color[0] * 256.0f;
		cgreen	= cl_dlights[lnum].color[1] * 256.0f;
		cblue	= cl_dlights[lnum].color[2] * 256.0f;
		// Tomaz - Lit Support End	

		for (t = 0 ; t<tmax ; t++)
		{
			td = local[1] - t*16;
			if (td < 0)
				td = -td;
			for (s=0 ; s<smax ; s++)
			{
				sd = local[0] - s*16;
				if (sd < 0)
					sd = -sd;
				if (sd > td)
					dist = sd + (td>>1);
				else
					dist = td + (sd>>1);
				if (dist < minlight)
				// Tomaz - Lit Support Begin
				{
					brightness = rad - dist;
					bl[0] += (int) (brightness * cred);
					bl[1] += (int) (brightness * cgreen);
					bl[2] += (int) (brightness * cblue);
				}
				bl += 3;
				// Tomaz - Lit Support End
			}
		}
	}
}

/*
===============
R_BuildLightMap

Combine and scale multiple lightmaps into the 8.8 format in blocklights
===============
*/
void R_BuildLightMap (msurface_t *surf, byte *dest, int stride)
{
	int			smax, tmax;
	int			t;
	int			i, j, size;
	byte		*lightmap;
	unsigned	scale;
	int			maps;
	unsigned	*bl;

	surf->cached_dlight = (surf->dlightframe == r_framecount);

	smax = (surf->extents[0]>>4)+1;
	tmax = (surf->extents[1]>>4)+1;
	size = smax*tmax;
	lightmap = surf->samples;

// set to full bright if no light data
	if (!cl.worldmodel->lightdata)
	{
		// Tomaz - Lit Support Begin
		bl = blocklights;
		for (i=0 ; i<size ; i++)
		{
			bl[0]	= 255*256;
			bl[1]	= 255*256;
			bl[2]	= 255*256;
			bl		+= 3;
		}
		// Tomaz - Lit Support End
		goto store;
	}

// clear to no light
	// Tomaz - Lit Support Begin
	bl = blocklights;
	for (i=0 ; i<size ; i++)
	{
		bl[0]	= 0;
		bl[1]	= 0;
		bl[2]	= 0;
		bl		+= 3;
	}
	// Tomaz - Lit Support End

// add all the lightmaps
	if (lightmap)
		for (maps = 0 ; maps < MAXLIGHTMAPS && surf->styles[maps] != 255 ;
			 maps++)
		{
			scale = d_lightstylevalue[surf->styles[maps]];
			surf->cached_light[maps] = scale;	// 8.8 fraction
			// Tomaz - Lit Support Begin
			bl = blocklights;
			for (i=0 ; i<size ; i++)
			{
				bl[0]		+= lightmap[0] * scale;
				bl[1]		+= lightmap[1] * scale;
				bl[2]		+= lightmap[2] * scale;

				bl			+= 3;
				lightmap	+= 3;
			}
			// Tomaz - Lit Support End
		}

// add all the dynamic lights
	if (surf->dlightframe == r_framecount)
		R_AddDynamicLights (surf);

// bound, invert, and shift
store:
	stride -= (smax<<2);
	bl = blocklights;
	for (i=0 ; i<tmax ; i++, dest += stride)
	{
		for (j=0 ; j<smax ; j++)
		{
			// Tomaz - Lit Support Begin
			t = bl[0] >> 7;if (t > 255) t = 255;dest[0] = t;
			t = bl[1] >> 7;if (t > 255) t = 255;dest[1] = t;
			t = bl[2] >> 7;if (t > 255) t = 255;dest[2] = t;
			dest[3] = 255;

			bl		+= 3;
			dest	+= 4;
			// Tomaz - Lit Support End
		}
	}
}

/*
===============
R_TextureAnimation

Returns the proper texture for a given time and base texture
===============
*/
texture_t *R_TextureAnimation (texture_t *base)
{
	int		reletive;
	int		count;

	if (currententity->frame)
	{
		if (base->alternate_anims)
			base = base->alternate_anims;
	}
	
	if (!base->anim_total)
		return base;

	reletive = (int)(cl.time*10) % base->anim_total;

	count = 0;	
	while (base->anim_min > reletive || base->anim_max <= reletive)
	{
		base = base->anim_next;
		if (!base)
			Sys_Error ("R_TextureAnimation: broken cycle");
		if (++count > 100)
			Sys_Error ("R_TextureAnimation: infinite cycle");
	}

	return base;
}


/*
=============================================================

	BRUSH MODELS

=============================================================
*/

int oldtexture;
extern	int		solidskytexture;
extern	int		alphaskytexture;
extern	float	speedscale;		// for top sky and bottom sky

lpMTexFUNC qglMTexCoord2fSGIS_ARB = NULL;
lpSelTexFUNC qglSelectTextureSGIS_ARB = NULL;

#include "gl_rscript.h"

void R_DrawBrushMTexScript (msurface_t *s);

/*
================
R_DrawLinePolys
================
*/
void R_DrawLinePolys (msurface_t *s)
{
	int			i;
	float		*v;
	glpoly_t	*p;

	p = s->polys;
	v = p->verts[0];

	glDisable(GL_TEXTURE_2D);
	glBegin(GL_LINE_LOOP);

	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
		glVertex3fv (v);

	glEnd ();
	glEnable(GL_TEXTURE_2D);
}

/*
====================
R_DrawBrushMTex
====================
*/
int		causticstexture[32];	// Tomaz - Underwater Caustics
void R_DrawBrushMTex (msurface_t *s)
{
	glpoly_t	*p;
	float		*v;
	int			i;
	texture_t	*t;
	glRect_t	*theRect;

	p = s->polys;

	t = R_TextureAnimation (s->texinfo->texture);

	if(t->rs)
	{
		R_DrawBrushMTexScript (s);
		
		return;
	}
	glBindTexture (GL_TEXTURE_2D, t->gl_texturenum);

	qglSelectTextureSGIS_ARB(TEXTURE1_SGIS_ARB);
	glEnable(GL_TEXTURE_2D);

	glBindTexture (GL_TEXTURE_2D, lightmap_textures + s->lightmaptexturenum);
	i = s->lightmaptexturenum;
	if (lightmap_modified[i])
	{
		lightmap_modified[i] = false;
		theRect = &lightmap_rectchange[i];
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t, BLOCK_WIDTH, theRect->h, GL_RGBA, GL_UNSIGNED_BYTE, lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*4);
		theRect->l = BLOCK_WIDTH;
		theRect->t = BLOCK_HEIGHT;
		theRect->h = 0;
		theRect->w = 0;
	}

	glBegin(GL_POLYGON);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		qglMTexCoord2fSGIS_ARB (TEXTURE0_SGIS_ARB, v[3], v[4]);
		qglMTexCoord2fSGIS_ARB (TEXTURE1_SGIS_ARB, v[5], v[6]);
		glVertex3fv (v);

	}
	glEnd ();

	glDisable(GL_TEXTURE_2D);
	qglSelectTextureSGIS_ARB(TEXTURE0_SGIS_ARB);

	if(gl_caustics.value)
		if (s->flags & SURF_UNDERWATER)
			EmitCausticsPolys(s);

	if (t->fullbrights != -1 && gl_fbr.value)
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);	
		glBlendFunc(GL_SRC_COLOR, GL_ONE);

		glBindTexture (GL_TEXTURE_2D, t->fullbrights);

		p = s->polys;
		v = p->verts[0];

		glBegin (GL_POLYGON);

		for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
		{
			glTexCoord2f (v[3], v[4]);
			glVertex3fv (v);
		}

		glEnd ();

		glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	}
}

int RS_AnimTexture(int rs);

/*
====================
R_DrawBrushMTexScript
====================
*/
void R_DrawBrushMTexScript (msurface_t *s)
{
	glpoly_t	*p;
	float		*v, vt[3], os, ot;
	int			i, rs;
	texture_t	*t;
	glRect_t	*theRect;
	qboolean	stage;
	float		txm, tym;
	vec3_t		nv;

	p = s->polys;

	t = R_TextureAnimation (s->texinfo->texture);

	stage = true;

	rs = t->rs;

	while (stage)
	{
		if (!rscripts[rs].texexist)
			glBindTexture (GL_TEXTURE_2D, t->gl_texturenum);
		else if (!rscripts[rs].useanim)
			glBindTexture (GL_TEXTURE_2D, rscripts[rs].texnum);
		else
			glBindTexture (GL_TEXTURE_2D, RS_AnimTexture(rs));
		
		if (gl_envmap.value)
		{
			glDepthMask (false);
		}
		
		qglSelectTextureSGIS_ARB(TEXTURE1_SGIS_ARB);
		glEnable(GL_TEXTURE_2D);

		glBindTexture (GL_TEXTURE_2D, lightmap_textures + s->lightmaptexturenum);
		i = s->lightmaptexturenum;
		if (lightmap_modified[i])
		{
			lightmap_modified[i] = false;
			theRect = &lightmap_rectchange[i];
			glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t, BLOCK_WIDTH, theRect->h, GL_RGBA, GL_UNSIGNED_BYTE, lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*4);
			theRect->l = BLOCK_WIDTH;
			theRect->t = BLOCK_HEIGHT;
			theRect->h = 0;
			theRect->w = 0;
		}
		

		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

		glBegin(GL_POLYGON);
		v = p->verts[0];
		for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
		{
			txm		= 0; 
			tym		= 0;
			nv[0]	= v[0];
			nv[1]	= v[1];
			nv[2]	= v[2];
			vt[0]	= v[3];
			vt[1]	= v[4];

			if (rscripts[rs].usescroll) 
			{
				txm = realtime*rscripts[rs].scroll.xspeed;
				while (txm > 1 && (1-txm) > 0) txm=1-txm;
				while (txm < 0 && (1+txm) > 1) txm=1+txm;
				tym = realtime*rscripts[rs].scroll.yspeed;
				while (tym > 1 && (1-tym) > 0) tym=1-tym;
				while (tym < 0 && (1+tym) > 1) tym=1+tym;
			}
			if (rscripts[rs].useturb) 
			{
				float power, movediv;

				power	= rscripts[rs].turb.power * 0.05;	// Tomaz - Speed
				movediv = rscripts[rs].turb.movediv;
				os		= v[3]; 
				ot		= v[4];
				vt[0]	= os + sin((os * 0.1 + realtime) * power) * sin((ot * 0.1 + realtime)) / movediv;
				vt[1]	= ot + sin((ot * 0.1 + realtime) * power) * sin((os * 0.1 + realtime)) / movediv;
			} 

			if (rscripts[rs].usevturb)
			{
				float power;

				power	= rscripts[rs].vturb.power;
				nv[0]	= v[0] + sin(v[1] * 0.1 + realtime) * sin(v[2] * 0.1 + realtime) * power;
				nv[1]	= v[1] + sin(v[0] * 0.1 + realtime) * sin(v[2] * 0.1 + realtime) * power;
				nv[2]	= v[2];
			}

			qglMTexCoord2fSGIS_ARB (TEXTURE0_SGIS_ARB, vt[0]+txm, vt[1]+tym);
			qglMTexCoord2fSGIS_ARB (TEXTURE1_SGIS_ARB, v[5], v[6]);

			glVertex3fv (nv);
		}
		
		glEnd ();

		glDisable(GL_TEXTURE_2D);
		qglSelectTextureSGIS_ARB(TEXTURE0_SGIS_ARB);

		if(gl_envmap.value)
		{
			glDepthMask (true);

			if (rscripts[rs].flags.envmap)
			{
				EmitEnvMapPolys(s);
			}
		}
	
		rs = rscripts[rs].nextstage;

		if (!rs)
			stage = false;
	}
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

}

/*
================
R_DrawBrushNoMTex
================
*/
void R_DrawBrushNoMTex (msurface_t *s)
{
	texture_t	*t;
	int			i;
	float		*v;
	glpoly_t	*p;
	int			j;
	glRect_t	*theRect;

	p = s->polys;

	t = R_TextureAnimation (s->texinfo->texture);
	glBindTexture (GL_TEXTURE_2D, t->gl_texturenum);

	glBegin (GL_POLYGON);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		glTexCoord2f (v[3], v[4]);
		glVertex3fv (v);
	}
	glEnd ();

	glBindTexture (GL_TEXTURE_2D, lightmap_textures + s->lightmaptexturenum);
	i = s->lightmaptexturenum;
	if (lightmap_modified[i])
	{
		lightmap_modified[i] = false;
		theRect = &lightmap_rectchange[i];
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t, BLOCK_WIDTH, theRect->h, GL_RGBA, GL_UNSIGNED_BYTE, lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*4);
		theRect->l = BLOCK_WIDTH;
		theRect->t = BLOCK_HEIGHT;
		theRect->h = 0;
		theRect->w = 0;
	}

	glBlendFunc(GL_ZERO, GL_SRC_COLOR);

	glBegin (GL_POLYGON);
	v = p->verts[0];
	for (j=0 ; j<p->numverts ; j++, v+= VERTEXSIZE)
	{
		glTexCoord2f (v[5], v[6]);
		glVertex3fv (v);
	}
	glEnd ();

	if (t->fullbrights != -1 && gl_fbr.value)
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
		glBlendFunc(GL_SRC_COLOR, GL_ONE);

		glBindTexture (GL_TEXTURE_2D, t->fullbrights);

		p = s->polys;
		v = p->verts[0];

		glBegin (GL_POLYGON);

		for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
		{
			glTexCoord2f (v[3], v[4]);
			glVertex3fv (v);
		}

		glEnd ();

		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	}

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

extern qboolean hl_map;

/*
================
R_DrawBrushMTexTrans
================
*/
void R_DrawBrushMTexTrans (msurface_t *s, float alpha)
{
	glpoly_t	*p;
	float		*v;
	int			i;
	texture_t	*t;
	glRect_t	*theRect;

	glColor4f (1,1,1,alpha);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	if (hl_map)
		glEnable(GL_ALPHA_TEST);

	p = s->polys;

	t = R_TextureAnimation (s->texinfo->texture);

	glBindTexture (GL_TEXTURE_2D, t->gl_texturenum);

	qglSelectTextureSGIS_ARB(TEXTURE1_SGIS_ARB);
	glEnable(GL_TEXTURE_2D);
	glBindTexture (GL_TEXTURE_2D, lightmap_textures + s->lightmaptexturenum);

	i = s->lightmaptexturenum;
	if (lightmap_modified[i])
	{
		lightmap_modified[i] = false;
		theRect = &lightmap_rectchange[i];
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t, BLOCK_WIDTH, theRect->h, GL_RGBA, GL_UNSIGNED_BYTE, lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*4);
		theRect->l = BLOCK_WIDTH;
		theRect->t = BLOCK_HEIGHT;
		theRect->h = 0;
		theRect->w = 0;
	}

	glBegin(GL_POLYGON);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		qglMTexCoord2fSGIS_ARB (TEXTURE0_SGIS_ARB, v[3], v[4]);
		qglMTexCoord2fSGIS_ARB (TEXTURE1_SGIS_ARB, v[5], v[6]);
		glVertex3fv (v);
	}
	glEnd ();

	glDisable(GL_TEXTURE_2D);
	qglSelectTextureSGIS_ARB(TEXTURE0_SGIS_ARB);

	if (hl_map)
		glDisable(GL_ALPHA_TEST);

	glColor4f (1,1,1,1);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
}

/*
================
R_DrawBrushNoMTexTrans
================
*/
void R_DrawBrushNoMTexTrans (msurface_t *s, float alpha)
{
	texture_t	*t;
	int			i;
	float		*v;
	glpoly_t	*p;
	glRect_t	*theRect;

	glColor4f (1,1,1,alpha);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	p = s->polys;

	t = R_TextureAnimation (s->texinfo->texture);
	glBindTexture (GL_TEXTURE_2D, t->gl_texturenum);
	glBegin (GL_POLYGON);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		glTexCoord2f (v[3], v[4]);
		glVertex3fv (v);
	}
	glEnd ();
	glBindTexture (GL_TEXTURE_2D, lightmap_textures + s->lightmaptexturenum);
	i = s->lightmaptexturenum;
	if (lightmap_modified[i])
	{
		lightmap_modified[i] = false;
		theRect = &lightmap_rectchange[i];
		glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t, BLOCK_WIDTH, theRect->h, GL_RGBA, GL_UNSIGNED_BYTE, lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*4);
		theRect->l = BLOCK_WIDTH;
		theRect->t = BLOCK_HEIGHT;
		theRect->h = 0;
		theRect->w = 0;
	}

	glBlendFunc(GL_ZERO, GL_SRC_COLOR);

	glBegin (GL_POLYGON);
	v = p->verts[0];
	for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
	{
		glTexCoord2f (v[5], v[6]);
		glVertex3fv (v);
	}
	glEnd ();

	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f (1,1,1,1);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
}

/*
================
R_RenderDynamicLightmaps
Multitexture
================
*/
void R_RenderDynamicLightmaps (msurface_t *s)
{
	byte		*base;
	int			maps;
	glRect_t    *theRect;
	int			smax, tmax;

	c_brush_polys++;

	if (s->flags & ( SURF_DRAWSKY | SURF_DRAWTURB ) )
		return;
		
	s->polys->chain = lightmap_polys[s->lightmaptexturenum];
	lightmap_polys[s->lightmaptexturenum] = s->polys;

	// check for lightmap modification
	for (maps = 0 ; maps < MAXLIGHTMAPS && s->styles[maps] != 255 ;
		 maps++)
		if (d_lightstylevalue[s->styles[maps]] != s->cached_light[maps])
			goto dynamic;

	if (s->dlightframe == r_framecount	// dynamic this frame
		|| s->cached_dlight)			// dynamic previously
	{
dynamic:
		if (r_dynamic.value)
		{
			lightmap_modified[s->lightmaptexturenum] = true;
			theRect = &lightmap_rectchange[s->lightmaptexturenum];

			if (s->light_t < theRect->t) 
			{
				if (theRect->h)
					theRect->h += theRect->t - s->light_t;

				theRect->t = s->light_t;
			}

			if (s->light_s < theRect->l) 
			{
				if (theRect->w)
					theRect->w += theRect->l - s->light_s;

				theRect->l = s->light_s;
			}

			smax = (s->extents[0]>>4)+1;
			tmax = (s->extents[1]>>4)+1;

			if ((theRect->w + theRect->l) < (s->light_s + smax))
				theRect->w = (s->light_s-theRect->l)+smax;

			if ((theRect->h + theRect->t) < (s->light_t + tmax))
				theRect->h = (s->light_t-theRect->t)+tmax;

			base = lightmaps + s->lightmaptexturenum*4*BLOCK_WIDTH*BLOCK_HEIGHT;
			base += s->light_t * BLOCK_WIDTH * 4 + s->light_s * 4;

			R_BuildLightMap (s, base, BLOCK_WIDTH*4);
		}
	}
}

/*
================
R_DrawWaterSurfaces
================
*/
void R_DrawWaterSurfaces (void)
{
	msurface_t	*s;

	if (!waterchain)
		return;

    glLoadMatrixf (r_world_matrix);

	if (r_wateralpha.value < 1.0) 
	{
		glColor4f (1,1,1,r_wateralpha.value);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	}

	for ( s = waterchain ; s ; s=s->texturechain) 
	{
		glBindTexture (GL_TEXTURE_2D, s->texinfo->texture->gl_texturenum);
		EmitWaterPolys (s);
	}

	waterchain = NULL;

	if (r_wateralpha.value < 1.0) 
	{
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
		glColor4f (1,1,1,1);
	}
}

float	r_world_matrix[16];
extern qboolean wireframe;

/*
=================
R_SetupBrushPolys
=================
*/
void R_SetupBrushPolys (entity_t *e)
{
	int			k;
	vec3_t		mins, maxs;
	int			i;
	msurface_t	*psurf;
	float		dot;
	mplane_t	*pplane;
	model_t		*clmodel;
	qboolean	rotated;

	clmodel = e->model;

	if (e->angles[0] || e->angles[1] || e->angles[2])
	{
		rotated = true;
		for (i=0 ; i<3 ; i++)
		{
			mins[i] = e->origin[i] - clmodel->radius;
			maxs[i] = e->origin[i] + clmodel->radius;
		}
	}
	else
	{
		rotated = false;
		VectorAdd (e->origin, clmodel->mins, mins);
		VectorAdd (e->origin, clmodel->maxs, maxs);
	}

	if (R_CullBox (mins, maxs))
		return;

	memset (lightmap_polys, 0, sizeof(lightmap_polys));

	VectorSubtract (r_refdef.vieworg, e->origin, modelorg);

	if (rotated)
	{
		vec3_t	temp;
		vec3_t	forward, right, up;

		VectorCopy (modelorg, temp);
		AngleVectors (e->angles, forward, right, up);
		modelorg[0] = DotProduct (temp, forward);
		modelorg[1] = -DotProduct (temp, right);
		modelorg[2] = DotProduct (temp, up);
	}

	psurf = &clmodel->surfaces[clmodel->firstmodelsurface];

// calculate dynamic lighting for bmodel if it's not an
// instanced model
	if (clmodel->firstmodelsurface != 0)
	{
		for (k=0 ; k<MAX_DLIGHTS ; k++)
		{
			if ((cl_dlights[k].die < cl.time) ||
				(!cl_dlights[k].radius))
				continue;

			R_MarkLights (&cl_dlights[k], 1<<k, clmodel->nodes + clmodel->hulls[0].firstclipnode);
		}
	}

    glPushMatrix ();
	e->angles[0] = -e->angles[0];	// stupid quake bug
	R_BlendedRotateForEntity (e, 0);
	e->angles[0] = -e->angles[0];	// stupid quake bug

	//
	// draw texture
	//
	for (i=0 ; i<clmodel->nummodelsurfaces ; i++, psurf++)
	{
	// find which side of the node we are on
		pplane = psurf->plane;

		dot = DotProduct (modelorg, pplane->normal) - pplane->dist;

	// draw the polygon
		if (((psurf->flags & SURF_PLANEBACK) && (dot < -BACKFACE_EPSILON)) ||
			(!(psurf->flags & SURF_PLANEBACK) && (dot > BACKFACE_EPSILON)))
		{
			if (wireframe)
			{
				R_DrawLinePolys (psurf);
				continue;
			}

			if (gl_wireonly.value)
			{
				R_DrawLinePolys (psurf);
				continue;
			}

			R_RenderDynamicLightmaps(psurf);

			if (gl_mtexable)
			{
				if ((e->alpha != 1) || (psurf->texinfo->texture->transparent))
					R_DrawBrushMTexTrans (psurf, e->alpha);
				else
					R_DrawBrushMTex (psurf);

			}
			else
			{
				if ((e->alpha != 1) || (psurf->texinfo->texture->transparent))
				{
					R_DrawBrushNoMTexTrans (psurf, e->alpha);
				}
				else
				{
					R_DrawBrushNoMTex (psurf);
				}
			}
			if (gl_showpolys.value)
			{
				R_DrawLinePolys (psurf);
			}
		}
	}

	glPopMatrix ();
}

/*
=============================================================

	WORLD MODEL

=============================================================
*/

/*
================
R_RecursiveWorldNode
================
*/
void R_RecursiveWorldNode (mnode_t *node)
{
	int			c, side;
	mplane_t	*plane;
	msurface_t	*surf, **mark;
	mleaf_t		*pleaf;
	double		dot;

	if (node->contents == CONTENTS_SOLID)
		return;

	if (node->visframe != r_visframecount)
		return;
	if (R_CullBox (node->minmaxs, node->minmaxs+3))
		return;

// if a leaf node, draw stuff
	if (node->contents < 0)
	{
		pleaf = (mleaf_t *)node;

		mark = pleaf->firstmarksurface;
		c = pleaf->nummarksurfaces;

		if (c)
		{
			do
			{
				(*mark)->visframe = r_framecount;
				mark++;
			} while (--c);
		}

	// deal with model fragments in this leaf
		if (pleaf->efrags)
			R_StoreEfrags (&pleaf->efrags);

		return;
	}

// node is just a decision point, so go down the apropriate sides

// find which side of the node we are on
	plane = node->plane;

	switch (plane->type)
	{
	case PLANE_X:
		dot = modelorg[0] - plane->dist;
		break;
	case PLANE_Y:
		dot = modelorg[1] - plane->dist;
		break;
	case PLANE_Z:
		dot = modelorg[2] - plane->dist;
		break;
	default:
		dot = DotProduct (modelorg, plane->normal) - plane->dist;
		break;
	}

	if (dot >= 0)
		side = 0;
	else
		side = 1;

// recurse down the children, front side first
	R_RecursiveWorldNode (node->children[side]);

// draw stuff
	c = node->numsurfaces;

	if (c)
	{
		surf = cl.worldmodel->surfaces + node->firstsurface;

		if (dot < 0 -BACKFACE_EPSILON)
			side = SURF_PLANEBACK;
		else if (dot > BACKFACE_EPSILON)
			side = 0;
		{
			for ( ; c ; c--, surf++)
			{
				if (surf->visframe != r_framecount)
					continue;

				// don't backface underwater surfaces, because they warp
				if (!(surf->flags & SURF_UNDERWATER) && ( (dot < 0) ^ !!(surf->flags & SURF_PLANEBACK)) )
					continue;		// wrong side

				if (wireframe)
				{
					R_DrawLinePolys (surf);
					continue;
				}

				if (gl_wireonly.value)
				{
					R_DrawLinePolys (surf);
					continue;
				}

				if (surf->flags & SURF_DRAWSKY) 
				{
					surf->texturechain = skychain;
					skychain = surf;
				} 
					
				else if (surf->flags & SURF_DRAWTURB) 
				{
					surf->texturechain = waterchain;
					waterchain = surf;
				}
// MIRRORS!!
				else if (r_mirroralpha.value < 1.0 && !mirror_render && surf->flags & SURF_MIRROR)
				{
					mirror = true;
					surf->texturechain = mirrorchain;
					mirrorchain = surf;
					continue;
				}
// END
				else if (gl_mtexable)
				{
					R_DrawBrushMTex (surf);
				}

				else
				{
					R_DrawBrushNoMTex (surf);
				}

				if (gl_showpolys.value)
				{
					R_DrawLinePolys (surf);
				}

				R_RenderDynamicLightmaps(surf);
			}
		}
	}

// recurse down the back side
	R_RecursiveWorldNode (node->children[!side]);
}

extern char skyname[];

/*
=============
R_DrawWorld
=============
*/

void R_DrawWorld (void)
{
	entity_t	ent;

	memset (&ent, 0, sizeof(ent));
	ent.model = cl.worldmodel;

	VectorCopy (r_refdef.vieworg, modelorg);

	currententity = &ent;

	memset (lightmap_polys, 0, sizeof(lightmap_polys));

	R_RecursiveWorldNode (cl.worldmodel->nodes);

	if (gl_wireframe.value)
	{
		wireframe = true;
		glDisable (GL_DEPTH_TEST);
		R_RecursiveWorldNode (cl.worldmodel->nodes);
		glEnable (GL_DEPTH_TEST);
		wireframe = false;
	}

	if (skychain) 
	{
		if (skyname[0])
		{
			R_DrawSkyBox ();
		}
		else
		{
			R_DrawSky(skychain);
		}
	skychain = NULL;
	}
}	


/*
===============
R_MarkLeaves
===============
*/
void R_MarkLeaves (void)
{
	byte	*vis;
	mnode_t	*node;
	int		i;
	byte	solid[4096];

	if (r_oldviewleaf == r_viewleaf && !r_novis.value)
		return;
	
	if (mirror)
	{
		return;
	}

	r_visframecount++;
	r_oldviewleaf = r_viewleaf;

	if (r_novis.value)
	{
		vis = solid;
		memset (solid, 0xff, (cl.worldmodel->numleafs+7)>>3);
	}
	else
		vis = Mod_LeafPVS (r_viewleaf, cl.worldmodel);
		
	for (i=0 ; i<cl.worldmodel->numleafs ; i++)
	{
		if (vis[i>>3] & (1<<(i&7)))
		{
			node = (mnode_t *)&cl.worldmodel->leafs[i+1];
			do
			{
				if (node->visframe == r_visframecount)
					break;
				node->visframe = r_visframecount;
				node = node->parent;
			} while (node);
		}
	}
}



/*
=============================================================================

  LIGHTMAP ALLOCATION

=============================================================================
*/

// returns a texture number and the position inside it
int AllocBlock (int w, int h, int *x, int *y)
{
	int		i, j;
	int		best, best2;
	int		texnum;

	for (texnum=0 ; texnum<MAX_LIGHTMAPS ; texnum++)
	{
		best = BLOCK_HEIGHT;

		for (i=0 ; i<BLOCK_WIDTH-w ; i++)
		{
			best2 = 0;

			for (j=0 ; j<w ; j++)
			{
				if (allocated[texnum][i+j] >= best)
					break;
				if (allocated[texnum][i+j] > best2)
					best2 = allocated[texnum][i+j];
			}
			if (j == w)
			{	// this is a valid spot
				*x = i;
				*y = best = best2;
			}
		}

		if (best + h > BLOCK_HEIGHT)
			continue;

		for (i=0 ; i<w ; i++)
			allocated[texnum][*x + i] = best + h;

		return texnum;
	}

	Host_Error ("AllocBlock: full, unable to find room for %i by %i lightmap", w, h);
	return 0;
}


mvertex_t	*r_pcurrentvertbase;
model_t		*currentmodel;

int	nColinElim;

/*
================
BuildSurfaceDisplayList
================
*/
void BuildSurfaceDisplayList (msurface_t *fa)
{
	int			i, lindex, lnumverts;
	medge_t		*pedges, *r_pedge;
	int			vertpage;
	float		*vec;
	float		s, t;
	glpoly_t	*poly;

// reconstruct the polygon
	fa->visframe = 0;
	pedges = currentmodel->edges;
	lnumverts = fa->numedges;
	vertpage = 0;

	//
	// draw texture
	//
	poly = Hunk_Alloc (sizeof(glpoly_t) + (lnumverts-4) * VERTEXSIZE*sizeof(float));
	poly->next = fa->polys;
	poly->flags = fa->flags;
	fa->polys = poly;
	poly->numverts = lnumverts;

	for (i=0 ; i<lnumverts ; i++)
	{
		lindex = currentmodel->surfedges[fa->firstedge + i];

		if (lindex > 0)
		{
			r_pedge = &pedges[lindex];
			vec = r_pcurrentvertbase[r_pedge->v[0]].position;
		}
		else
		{
			r_pedge = &pedges[-lindex];
			vec = r_pcurrentvertbase[r_pedge->v[1]].position;
		}
		s = DotProduct (vec, fa->texinfo->vecs[0]) + fa->texinfo->vecs[0][3];
		s /= fa->texinfo->texture->width;

		t = DotProduct (vec, fa->texinfo->vecs[1]) + fa->texinfo->vecs[1][3];
		t /= fa->texinfo->texture->height;

		VectorCopy (vec, poly->verts[i]);
		poly->verts[i][3] = s;
		poly->verts[i][4] = t;

		//
		// lightmap texture coordinates
		//
		s = DotProduct (vec, fa->texinfo->vecs[0]) + fa->texinfo->vecs[0][3];
		s -= fa->texturemins[0];
		s += fa->light_s*16;
		s += 8;
		s /= BLOCK_WIDTH*16; //fa->texinfo->texture->width;

		t = DotProduct (vec, fa->texinfo->vecs[1]) + fa->texinfo->vecs[1][3];
		t -= fa->texturemins[1];
		t += fa->light_t*16;
		t += 8;
		t /= BLOCK_HEIGHT*16; //fa->texinfo->texture->height;

		poly->verts[i][5] = s;
		poly->verts[i][6] = t;
	}

	//
	// remove co-linear points - Ed
	//
	if (!gl_keeptjunctions.value && !(fa->flags & SURF_UNDERWATER) )
	{
		for (i = 0 ; i < lnumverts ; ++i)
		{
			vec3_t v1, v2;
			float *prev, *this, *next;

			prev = poly->verts[(i + lnumverts - 1) % lnumverts];
			this = poly->verts[i];
			next = poly->verts[(i + 1) % lnumverts];

			VectorSubtract( this, prev, v1 );
			VectorNormalize( v1 );
			VectorSubtract( next, prev, v2 );
			VectorNormalize( v2 );

			// skip co-linear points
			#define COLINEAR_EPSILON 0.001
			if ((fabs( v1[0] - v2[0] ) <= COLINEAR_EPSILON) &&
				(fabs( v1[1] - v2[1] ) <= COLINEAR_EPSILON) && 
				(fabs( v1[2] - v2[2] ) <= COLINEAR_EPSILON))
			{
				int j;
				for (j = i + 1; j < lnumverts; ++j)
				{
					int k;
					for (k = 0; k < VERTEXSIZE; ++k)
						poly->verts[j - 1][k] = poly->verts[j][k];
				}
				--lnumverts;
				++nColinElim;
				// retry next vertex next time, which is now current vertex
				--i;
			}
		}
	}
	poly->numverts = lnumverts;

}

/*
========================
GL_CreateSurfaceLightmap
========================
*/
void GL_CreateSurfaceLightmap (msurface_t *surf)
{
	int		smax, tmax;
	byte	*base;

	if (surf->flags & (SURF_DRAWSKY|SURF_DRAWTURB))
		return;

	smax = (surf->extents[0]>>4)+1;
	tmax = (surf->extents[1]>>4)+1;

	surf->lightmaptexturenum = AllocBlock (smax, tmax, &surf->light_s, &surf->light_t);
	base = lightmaps + surf->lightmaptexturenum*4*BLOCK_WIDTH*BLOCK_HEIGHT;
	base += (surf->light_t * BLOCK_WIDTH + surf->light_s) * 4;
	R_BuildLightMap (surf, base, BLOCK_WIDTH*4);
}


/*
==================
GL_BuildLightmaps

Builds the lightmap texture
with all the surfaces from all brush models
==================
*/
void GL_BuildLightmaps (void)
{
	int		i, j;
	model_t	*m;

	memset (allocated, 0, sizeof(allocated));

	r_framecount = 1;		// no dlightcache

	if (!lightmap_textures)
	{
		lightmap_textures = texture_extension_number;
		texture_extension_number += MAX_LIGHTMAPS;
	}

	for (j=1 ; j<MAX_MODELS ; j++)
	{
		m = cl.model_precache[j];
		if (!m)
			break;
		if (m->name[0] == '*')
			continue;
		r_pcurrentvertbase = m->vertexes;
		currentmodel = m;
		for (i=0 ; i<m->numsurfaces ; i++)
		{
			GL_CreateSurfaceLightmap (m->surfaces + i);
			if ( m->surfaces[i].flags & SURF_DRAWTURB )
				continue;
			if ( m->surfaces[i].flags & SURF_DRAWSKY )
				continue;
			BuildSurfaceDisplayList (m->surfaces + i);
		}
	}
 	if (gl_mtexable)
		qglSelectTextureSGIS_ARB(TEXTURE1_SGIS_ARB);

	//
	// upload all lightmaps that were filled
	//
	for (i=0 ; i<MAX_LIGHTMAPS ; i++)
	{
		if (!allocated[i][0])
			break;		// no more used
		lightmap_modified[i] = false;
		lightmap_rectchange[i].l = BLOCK_WIDTH;
		lightmap_rectchange[i].t = BLOCK_HEIGHT;
		lightmap_rectchange[i].w = 0;
		lightmap_rectchange[i].h = 0;
		glBindTexture (GL_TEXTURE_2D, lightmap_textures + i);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexImage2D (GL_TEXTURE_2D, 0, 4, BLOCK_WIDTH, BLOCK_HEIGHT, 0, GL_RGBA, GL_UNSIGNED_BYTE, lightmaps+i*BLOCK_WIDTH*BLOCK_HEIGHT*4);
	}

 	if (gl_mtexable)
		qglSelectTextureSGIS_ARB(TEXTURE0_SGIS_ARB);
}
