/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// cvar.c -- dynamic variable tracking

#include "quakedef.h"


cvar_t *Cvar_Set2 (char *var_name, char *value, qboolean force);//Q2 set support

cvar_t	*cvar_vars;
char	*cvar_null_string = "";

// Q2!
#define	CVAR_ARCHIVE	1	// set to cause it to be saved to vars.rc
#define	CVAR_USERINFO	2	// added to userinfo  when changed
#define	CVAR_SERVERINFO	4	// added to serverinfo when changed
#define	CVAR_NOSET		8	// don't allow change from console at all,
							// but can be set from the command line
#define	CVAR_LATCH		16	// save changes until server restart

#define CVAR_SERVER     4
#define CVAR_USERDEF    16  // Data archive as new


// Q2!



/*
============
Cvar_FindVar
============
*/
cvar_t *Cvar_FindVar (char *var_name)
{
	cvar_t	*var;
	
	for (var=cvar_vars ; var ; var=var->next)
		if (!Q_strcmp (var_name, var->name))
			return var;

	return NULL;
}

/*
============
Cvar_VariableValue
============
*/
float	Cvar_VariableValue (char *var_name)
{
	cvar_t	*var;
	
	var = Cvar_FindVar (var_name);
	if (!var)
		return 0;
	return Q_atof (var->string);
}


/*
============
Cvar_VariableString
============
*/
char *Cvar_VariableString (char *var_name)
{
	cvar_t *var;
	
	var = Cvar_FindVar (var_name);
	if (!var)
		return cvar_null_string;
	return var->string;
}


/*
============
Cvar_CompleteVariable
============
*/
char *Cvar_CompleteVariable (char *partial)
{
	cvar_t		*cvar;
	int			len;
	
	len = Q_strlen(partial);
	
	if (!len)
		return NULL;
		
// check functions
	for (cvar=cvar_vars ; cvar ; cvar=cvar->next)
		if (!Q_strncmp (partial,cvar->name, len))
			return cvar->name;

	return NULL;
}

// Tomaz - Enhanced Complete Command Begin
/*
============
Cvar_CompleteCountPossible
============
*/
int Cvar_CompleteCountPossible (char *partial)
{
	cvar_t	*cvar;
	int	len;
	int	h;

	h=0;

	len = Q_strlen(partial);
	
	if (!len)
		return 0;
		
	// Loop through the cvars and count all partial matches
	for (cvar=cvar_vars ; cvar ; cvar=cvar->next)
		if (!Q_strncmp (partial,cvar->name, len))
			h++;
	return h;
}

/*
============
Cvar_CompletePrintPossible
============
*/
void Cvar_CompletePrintPossible (char *partial)
{
	cvar_t	*cvar;
	int	len;
	int	lpos;
	int	out;
	int	con_linewidth;
	char	sout[32];
	char	lout[2048];

	len = Q_strlen(partial);
	lpos = 0;
	Q_strcpy(lout,"");

	// Determine the width of the console
	con_linewidth = (vid.width >> 3) - 3;

	// Loop through the cvars and print all matches
	for (cvar=cvar_vars ; cvar ; cvar=cvar->next)
		if (!Q_strncmp (partial,cvar->name, len))
		{
			Q_strcpy(sout, cvar->name);

			out = Q_strlen(sout);
			lpos += out;

			// Pad with spaces
			for (out; out<20; out++)		
			{
				if (lpos < con_linewidth)
					Q_strcat (sout, " ");
				
				lpos++;
			}

			Q_strcat (lout, sout);

			if (lpos > con_linewidth - 24)
				for  (lpos; lpos < con_linewidth; lpos++)
					Q_strcat(lout, " ");

			if (lpos >= con_linewidth)
				lpos = 0;
		}
	Con_Printf ("%s\n\n", lout);
}
// Tomaz - Enhanced Complete Command End

/*
============
Cvar_Set
============
*/
void Cvar_Set (char *var_name, char *value)
{
	cvar_t	*var;
	qboolean changed;

	
	var = Cvar_FindVar (var_name);
	if (!var)
	{	// there is an error in C code if this happens
		Con_Printf ("Cvar_Set: variable %s not found\n", var_name);
		return;
	}

	changed = Q_strcmp(var->string, value);
	
	Z_Free (var->string);	// free the old value string
	
	var->string = Z_Malloc (Q_strlen(value)+1);
	Q_strcpy (var->string, value);
	var->value = Q_atof (var->string);
	if (var->server && changed)
	{
		if (sv.active)
			SV_BroadcastPrintf ("\"%s\" changed to \"%s\"\n", var->name, var->string);
	}
	if ((var->value != 0) && (var == &deathmatch))
		Cvar_Set ("coop", "0");

	if ((var->value != 0) && (var == &coop))
		Cvar_Set ("deathmatch", "0");
}

/*
============
Cvar_SetValue
============
*/
void Cvar_SetValue (char *var_name, float value)
{
	char	val[32];
	
	sprintf (val, "%f",value);
	Cvar_Set (var_name, val);
}


/*
============
Cvar_RegisterVariable

Adds a freestanding variable to the variable list.
============
*/
void Cvar_RegisterVariable (cvar_t *variable)
{
	char	*oldstr;
	
// first check to see if it has allready been defined
	if (Cvar_FindVar (variable->name))
	{
		Con_Printf ("Can't register variable %s, allready defined\n", variable->name);
		return;
	}
	
// check for overlap with a command
	if (Cmd_Exists (variable->name))
	{
		Con_Printf ("Cvar_RegisterVariable: %s is a command\n", variable->name);
		return;
	}
		
// copy the value off, because future sets will Z_Free it
	oldstr = variable->string;
	variable->string = Z_Malloc (Q_strlen(variable->string)+1);	
	Q_strcpy (variable->string, oldstr);
	variable->value = Q_atof (variable->string);
	
// link the variable in
	variable->next = cvar_vars;
	cvar_vars = variable;
}

/*
============
Cvar_Command

Handles variable inspection and changing from the console
============
*/
qboolean	Cvar_Command (void)
{
	cvar_t			*v;

// check variables
	v = Cvar_FindVar (Cmd_Argv(0));
	if (!v)
		return false;
		
// perform a variable print or set
	if (Cmd_Argc() == 1)
	{
		Con_Printf ("\"%s\" is \"%s\"\n", v->name, v->string);
		return true;
	}

	Cvar_Set (v->name, Cmd_Argv(1));
	return true;
}


/*
============
Cvar_WriteVariables

Writes lines containing "set variable value" for all variables
with the archive flag set to true.
============
*/
void Cvar_WriteVariables (FILE *f)
{
	cvar_t	*var;
	
	for (var = cvar_vars ; var ; var = var->next)
	{
		if (var->archive)
				fprintf (f, "%s \"%s\"\n", var->name, var->string);
		if (var->userdef )
				fprintf (f, "set %s \"%s\"\n", var->name, var->string);
	}

}

// Q2! cvarlist

/*
============
Cvar_List_f

============
*/
void Cvar_List_f (void)
{
	cvar_t	*var;
	int		i;

	i = 0;
	for (var = cvar_vars ; var ; var = var->next, i++)
	{
		if (var->archive)
			Con_Printf ("*");
		else
			Con_Printf (" ");
		
		/*
		if (var->flags & CVAR_USERINFO)
			Com_Printf ("U");
		else
			Com_Printf (" ");
		*/
		
		if (var->server)
			Con_Printf ("S");
		else
			Con_Printf (" ");

		if (var->userdef)
			Con_Printf ("U");
		else
			Con_Printf (" ");

		/*
		if (!var->value)
			Com_Printf ("-");
		else if (var->flags & CVAR_LATCH)
			Com_Printf ("L");
		else
			Com_Printf (" ");
		*/
		Con_Printf (" %s \"%s\"\n", var->name, var->string);
	}
	Con_Printf ("%i cvars\n", i);
}

// Q2!  cvarlist

// Q2!


char *CopyString (char *in);

/*
============
Cvar_Get

If the variable already exists, the value will not be set
The flags will be or'ed in if the variable exists.
============
*/
void Cvar_Flags ( cvar_t * var, int flags )
{
	if (flags & CVAR_SERVER )
		var->server = true;

	if (flags & CVAR_ARCHIVE )
		var->archive = true;

	if (flags & CVAR_USERDEF )
		var->userdef = true;

}

static qboolean Cvar_InfoValidate (char *s)
{
	if (strstr (s, "\\"))
		return false;
	if (strstr (s, "\""))
		return false;
	if (strstr (s, ";"))
		return false;
	return true;
}


cvar_t *Cvar_Get (char *var_name, char *var_value, int flags)
{
	cvar_t	*var;
	
	/*
	if (flags & (CVAR_USERINFO | CVAR_SERVERINFO))
	{
		if (!Cvar_InfoValidate (var_name))
		{
			Com_Printf("invalid info cvar name\n");
			return NULL;
		}
	}
	*/

	var = Cvar_FindVar (var_name);
	if (var)
	{
		if (flags & CVAR_USERDEF)
			flags -=  CVAR_USERDEF;
		Cvar_Flags(var,flags);
		return var;
	}
	
	if (!var_value)
		return NULL;

	if (flags & (CVAR_USERINFO | CVAR_SERVERINFO))
	{
		if (!Cvar_InfoValidate (var_value))
		{
			Con_Printf("invalid info cvar value\n");
			return NULL;
		}
	}

	var = Z_Malloc (sizeof(*var));
	var->name = CopyString (var_name);
	var->string = CopyString (var_value);
	//var->modified = true;
	var->value = atof (var->string);

	// link the variable in
	var->next = cvar_vars;
	cvar_vars = var;

	Cvar_Flags(var,flags);

	return var;
}




cvar_t *Cvar_FullSet (char *var_name, char *value, int flags)
{
	cvar_t	*var;
	
	var = Cvar_FindVar (var_name);
	if (!var)
	{	// create it
		return Cvar_Get (var_name, value, flags);
	}

	//var->modified = true;
	Cvar_Flags(var,flags);
	/*
	if (flags & CVAR_SERVER )
		var->server = true;

	if (flags & CVAR_ARCHIVE )
		var->archive = true;
	*/
	Z_Free (var->string);	// free the old value string
	
	var->string = CopyString(value);
	var->value = atof (var->string);

	return var;
}

/*
============
Cvar_Set2
============
*/
cvar_t *Cvar_Set2 (char *var_name, char *value, qboolean force)
{
	cvar_t	*var;

	var = Cvar_FindVar (var_name);
	if (!var)
	{	// create it
		var = Cvar_Get (var_name, value, CVAR_USERDEF);
		var->userdef = true;
		return var;
	}

	if (var->server | var->archive)
	{
		if (!Cvar_InfoValidate (value))
		{
			Con_Printf("invalid info cvar value\n");
			return var;
		}
	}

	if (!force)
	{
		/*if (var->flags & CVAR_NOSET)
		{
			Con_Printf ("%s is write protected.\n", var_name);
			return var;
		}*/

		/*
		if (var->flags & CVAR_LATCH)
		{
			if (var->latched_string)
			{
				if (strcmp(value, var->latched_string) == 0)
					return var;
				Z_Free (var->latched_string);
			}
			else
			{
				if (strcmp(value, var->string) == 0)
					return var;
			}

			if (Com_ServerState())
			{
				Con_Printf ("%s will be changed for next game.\n", var_name);
				var->latched_string = CopyString(value);
			}
			else
			{
				var->string = CopyString(value);
				var->value = atof (var->string);
				if (!strcmp(var->name, "game"))
				{
					FS_SetGamedir (var->string);
					FS_ExecAutoexec ();
				}
			}
			return var;
		}
		*/
	}
	else
	{
		/*
		if (var->latched_string)
		{
			Z_Free (var->latched_string);
			var->latched_string = NULL;
		}
		*/
	}

	if (!strcmp(value, var->string))
		return var;		// not changed

	//var->modified = true;

	//if (var->flags & CVAR_USERINFO)
	//	userinfo_modified = true;	// transmit at next oportunity
	
	Z_Free (var->string);	// free the old value string
	
	var->string = CopyString(value);
	var->value = atof (var->string);

	return var;
}



void Cvar_Set_f (void)
{
	int		c;
	int		flags;

	c = Cmd_Argc();
	if (c != 3 && c != 4)
	{
		Con_Printf ("usage: set <variable> <value> [u / s]\n");
		return;
	}

	if (c == 4)
	{
		if (!strcmp(Cmd_Argv(3), "a"))
			flags = CVAR_ARCHIVE;
		else if (!strcmp(Cmd_Argv(3), "s"))
			flags = CVAR_SERVER;
		else
		{
			Con_Printf ("flags can only be 'a' or 's'\n");
			return;
		}
		Cvar_FullSet (Cmd_Argv(1), Cmd_Argv(2), flags);
	}
	else
		Cvar_Set2 (Cmd_Argv(1), Cmd_Argv(2), true);
}



void Cvar_Inc_f(void) {
	cvar_t	*var;
	char * cvarname;
	char newval[1000];
	int		c, val;
	

	c = Cmd_Argc();
	cvarname =Cmd_Argv(1);

	if (c != 2 )
	{
		Con_Printf ("usage: inc <variable>\n");
		return;
	}

	var = Cvar_FindVar (cvarname);

	if (!var) 
	{
		Con_Printf ("'%s' not found\n", cvarname);
		return;
	}

	val = var->value + 1;

	sprintf(newval,"%d",val);

	Cvar_FullSet ( cvarname, newval,0);
} 


void Cvar_Dec_f(void) {
	cvar_t	*var;
	char * cvarname;
	char newval[1000];
	int		c, val;
	

	c = Cmd_Argc();
	cvarname =Cmd_Argv(1);

	if (c != 2 )
	{
		Con_Printf ("usage: dec <variable>\n");
		return;
	}

	var = Cvar_FindVar (cvarname);

	if (!var) 
	{
		Con_Printf ("'%s' not found\n", cvarname);
		return;
	}

	if (var->value)
		val = var->value - 1;

	sprintf(newval,"%d",val);

	Cvar_FullSet ( cvarname, newval,0);
} 



void Cvar_Mov_f(void) {
	cvar_t	*var1;
	cvar_t	*var2;

	char * cvardst;
	char * cvarsrc;

	char newval[1000];
	int		c, val;
	

	c = Cmd_Argc();
	cvardst =Cmd_Argv(1);
	cvarsrc =Cmd_Argv(2);

	if (c != 3 )
	{
		Con_Printf ("usage: mov <dst> <src>\n");
		return;
	}

	var1 = Cvar_FindVar (cvardst);
	var2 = Cvar_FindVar (cvarsrc);

	if (!var1) 
	{
		Con_Printf ("'%s' or not found\n", cvardst);
		return;
	}

	if (!var2) 
	{
		Con_Printf ("'%s' or not found\n", cvarsrc);
		return;
	}


	val = var2->value;

	sprintf(newval,"%d",val);

	Cvar_FullSet ( cvardst, newval,0);
} 



/*
============
Cvar_Init

Reads in all archived cvars
============
*/
void Cvar_Init (void)
{
	Cmd_AddCommand ("set", Cvar_Set_f);
	Cmd_AddCommand ("dec", Cvar_Dec_f);
	Cmd_AddCommand ("mov", Cvar_Mov_f);
	Cmd_AddCommand ("inc", Cvar_Inc_f);
	//Cmd_AddCommand ("addtofile", Cvar_AddToFile_f);
	Cmd_AddCommand ("cvarlist", Cvar_List_f);
}


// Q2!