/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// cvar.c -- dynamic variable tracking

#include "quakedef.h"

cvar_t	*cvar_vars;
char	*cvar_null_string = "";

// 2000-01-13 Enhanced cvar handling by Maddes  start
/*
============
Cvar_Display
============
*/
void Cvar_Display (cvar_t *cvar)
{
	char	val[32];
	int	i;

	Con_Printf ("\"%s\" is \"%s\"", cvar->name, cvar->string);
	if (cvar->readonly)
	{
		Con_Printf (" (read-only)");
	}
	else if (cvar->checkvalue)
	{
		if (cvar->checkvalue == 3)	// boolean
		{
			Con_Printf (" (bool: 0/1)");
		}
		else if (cvar->checkvalue == 2)	// integer
		{
			Con_Printf (" (int: %d-%d)", (int)cvar->minvalue, (int)cvar->maxvalue);
		}
		else
		{
			char	val[32];
			int	i;

			if (cvar->minvalue == (int)cvar->minvalue)
			{
				sprintf (val, "%d", (int)cvar->minvalue);
			}
			else
			{
				sprintf (val, "%1f", cvar->minvalue);
				for (i=Q_strlen(val)-1 ; i>0 && val[i]=='0' && val[i-1]!='.' ; i--)
				{
					val[i] = 0;
				}
			}
			Con_Printf (" (float: %s-", val);

			if (cvar->maxvalue == (int)cvar->maxvalue)
			{
				sprintf (val, "%d", (int)cvar->maxvalue);
			}
			else
			{
				sprintf (val, "%1f", cvar->maxvalue);
				for (i=Q_strlen(val)-1 ; i>0 && val[i]=='0' && val[i-1]!='.' ; i--)
				{
					val[i] = 0;
				}
			}
			Con_Printf ("%s)", val);
		}
	}
	Con_Printf ("\n");
}
// 2000-01-13 Enhanced cvar handling by Maddes  end

/*
============
Cvar_FindVar
============
*/
cvar_t *Cvar_FindVar (char *var_name)
{
	cvar_t	*var;

	for (var=cvar_vars ; var ; var=var->next)
		if (!Q_strcmp (var_name, var->name))
			return var;

	return NULL;
}

/*
============
Cvar_VariableValue
============
*/
float	Cvar_VariableValue (char *var_name)
{
	cvar_t	*var;

	var = Cvar_FindVar (var_name);
	if (!var)
		return 0;
	return Q_atof (var->string);
}


/*
============
Cvar_VariableString
============
*/
char *Cvar_VariableString (char *var_name)
{
	cvar_t *var;

	var = Cvar_FindVar (var_name);
	if (!var)
		return cvar_null_string;
	return var->string;
}


/*
============
Cvar_CompleteVariable
============
*/
char *Cvar_CompleteVariable (char *partial)
{
	cvar_t		*cvar;
	int			len;

	len = Q_strlen(partial);

	if (!len)
		return NULL;

// check functions
	for (cvar=cvar_vars ; cvar ; cvar=cvar->next)
		if (!Q_strncmp (partial,cvar->name, len))
			return cvar->name;

	return NULL;
}


/*
============
Cvar_Set
============
*/
void Cvar_Set (char *var_name, char *value)
{
	cvar_t	*var;
	qboolean changed;
// 2000-01-13 Enhanced cvar handling by Maddes  start
	char	val[32];
	float	newvalue;
	int	i;
// 2000-01-13 Enhanced cvar handling by Maddes  end

	var = Cvar_FindVar (var_name);
	if (!var)
	{	// there is an error in C code if this happens
		Con_Printf ("Cvar_Set: variable %s not found\n", var_name);
		return;
	}

// 2000-01-13 Enhanced cvar handling by Maddes  start
	if (var->readonly)
	{
		Con_Printf ("Cvar_Set: variable %s is read-only\n", var->name);
		return;
	}

	if (var->checkvalue)
	{
		newvalue = Q_atof (value);

		if (var->checkvalue == 3)	// boolean
		{
			if (newvalue)
			{
				newvalue = 1;
			}
			else
			{
				newvalue = 0;
			}
		}
		else
		{
			if (var->checkvalue == 2)	// integer
			{
				newvalue = (int)newvalue;
			}

			// check limits of newvalue
			if (newvalue < var->minvalue)
			{
				newvalue = var->minvalue;
			}
			if (newvalue > var->maxvalue)
			{
				newvalue = var->maxvalue;
			}
		}

		if (newvalue == (int)newvalue)
		{
			sprintf (val, "%d", (int)newvalue);
		}
		else
		{
			sprintf (val, "%1f", newvalue);
			for (i=Q_strlen(val)-1 ; i>0 && val[i]=='0' && val[i-1]!='.' ; i--)
			{
				val[i] = 0;
			}
		}

		changed = Q_strcmp(var->string, val);
	}
	else
// 2000-01-13 Enhanced cvar handling by Maddes  end
		changed = Q_strcmp(var->string, value);

// 2000-01-13 Enhanced cvar handling by Maddes  start
	if (!changed)	// nothing changed, nothing to do
	{
		return;
	}
// 2000-01-13 Enhanced cvar handling by Maddes  end

	Z_Free (var->string);	// free the old value string

// 2000-01-13 Enhanced cvar handling by Maddes  start
	if (var->checkvalue)
	{
		var->string = Z_Malloc (Q_strlen(val)+1);
		Q_strcpy (var->string, val);
	}
	else
	{
// 2000-01-13 Enhanced cvar handling by Maddes  end
		var->string = Z_Malloc (Q_strlen(value)+1);
		Q_strcpy (var->string, value);
	}	// 2000-01-13 Enhanced cvar handling by Maddes

	var->value = Q_atof (var->string);

	if (var->server && changed)
	{
		if (sv.active)
			SV_BroadcastPrintf ("\"%s\" changed to \"%s\"\n", var->name, var->string);
	}

// 1999-09-06 deathmatch/coop not at the same time fix by Maddes  start
//	if ( (var->value != 0) && (!Q_strcmp (var->name, deathmatch.name)) )
	if ( (var->value != 0) && (var == &deathmatch) )
	{
		Cvar_Set ("coop", "0");
	}

//	if ( (var->value != 0) && (!Q_strcmp (var->name, coop.name)) )
	if ( (var->value != 0) && (var == &coop) )
	{
		Cvar_Set ("deathmatch", "0");
	}
// 1999-09-06 deathmatch/coop not at the same time fix by Maddes  end
}

/*
============
Cvar_SetValue
============
*/
void Cvar_SetValue (char *var_name, float value)
{
	char	val[32];
// 1999-09-07 weird cvar zeros fix by Maddes  start
	int	i;

	if (value == (int)value)
	{
		sprintf (val, "%d", (int)value);
	}
	else
	{
//		sprintf (val, "%f",value);
		sprintf (val, "%1f", value);
		for (i=Q_strlen(val)-1 ; i>0 && val[i]=='0' && val[i-1]!='.' ; i--)
		{
			val[i] = 0;
		}
	}
// 1999-09-07 weird cvar zeros fix by Maddes  end
	Cvar_Set (var_name, val);
}


/*
============
Cvar_RegisterVariable

Adds a freestanding variable to the variable list.
============
*/
void Cvar_RegisterVariable (cvar_t *variable)
{
	char	*oldstr;

// first check to see if it has already been defined
	if (Cvar_FindVar (variable->name))
	{
		Con_Printf ("Can't register variable %s, already defined\n", variable->name);
		return;
	}

// check for overlap with a command
	if (Cmd_Exists (variable->name))
	{
		Con_Printf ("Cvar_RegisterVariable: %s is a command\n", variable->name);
		return;
	}

// copy the value off, because future sets will Z_Free it
	oldstr = variable->string;
	variable->string = Z_Malloc (Q_strlen(variable->string)+1);
	Q_strcpy (variable->string, oldstr);
	variable->value = Q_atof (variable->string);

// link the variable in
	variable->next = cvar_vars;
	cvar_vars = variable;
}

/*
============
Cvar_Command

Handles variable inspection and changing from the console
============
*/
qboolean	Cvar_Command (void)
{
	cvar_t			*v;

// check variables
	v = Cvar_FindVar (Cmd_Argv(0));
	if (!v)
		return false;

// perform a variable print or set
	if (Cmd_Argc() == 1)
	{
// 2000-01-13 Enhanced cvar handling by Maddes  start
//		Con_Printf ("\"%s\" is \"%s\"\n", v->name, v->string);
		Cvar_Display (v);
// 2000-01-13 Enhanced cvar handling by Maddes  end
		return true;
	}

	Cvar_Set (v->name, Cmd_Argv(1));
	return true;
}


/*
============
Cvar_WriteVariables

Writes lines containing "set variable value" for all variables
with the archive flag set to true.
============
*/
void Cvar_WriteVariables (FILE *f)
{
	cvar_t	*var;

	for (var = cvar_vars ; var ; var = var->next)
		if (var->archive)
			fprintf (f, "%s \"%s\"\n", var->name, var->string);
}

// 2000-01-09 CvarList command by Maddes  start
/*
=========
Cvar_List
=========
*/
void Cvar_List_f (void)
{
	cvar_t		*cvar;
	char 		*partial;
	int		len;
	int		count;

	if (Cmd_Argc() > 1)
	{
		partial = Cmd_Argv (1);
		len = Q_strlen(partial);
	}
	else
	{
		partial = NULL;
		len = 0;
	}

	count=0;
	for (cvar=cvar_vars ; cvar ; cvar=cvar->next)
	{
		if (partial && Q_strncmp (partial,cvar->name, len))
		{
			continue;
		}
// 2000-01-13 Enhanced cvar handling by Maddes  start
		Cvar_Display (cvar);
// 2000-01-13 Enhanced cvar handling by Maddes  end
		count++;
	}

	Con_Printf ("%i cvar(s)", count);
	if (partial)
	{
		Con_Printf (" beginning with \"%s\"", partial);
	}
	Con_Printf ("\n");
}
// 2000-01-09 CvarList command by Maddes  end
