/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// cl_parse.c  -- parse a message received from the server

#include "quakedef.h"

char *svc_strings[] =
{
	"svc_bad",
	"svc_nop",
	"svc_disconnect",
	"svc_updatestat",
	"svc_version",		// [long] server version
	"svc_setview",		// [short] entity number
	"svc_localsound",			//jft <see code>
	"svc_sound",			// <see code>
	"svc_time",			// [float] server time
	"svc_print",			// [string] null terminated string
	"svc_stufftext",		// [string] stuffed into client's console buffer
						// the string should be \n terminated
	"svc_setangle",		// [vec3] set the view angle to this absolute value

	"svc_serverinfo",		// [long] version
						// [string] signon string
						// [string]..[0]model cache [string]...[0]sounds cache
						// [string]..[0]item cache
	"svc_lightstyle",		// [byte] [string]
	"svc_updatename",		// [byte] [string]
	"svc_updatefrags",	// [byte] [short]
	"svc_clientdata",		// <shortbits + data>
	"svc_stopsound",		// <see code>
	"svc_updateskins",	// [byte] [byte]
//	"svc_updatebodies",	// [byte] [byte]
	"svc_particle",		// [vec3] <variable>
	"svc_damage",			// [byte] impact [byte] blood [vec3] from

	"svc_spawnstatic",
	"OBSOLETE svc_spawnbinary",
	"svc_spawnbaseline",

	"svc_temp_entity",		// <variable>
	"svc_setpause",
	"svc_signonnum",
	"svc_centerprint",
	"svc_killedmonster",
	"svc_foundsecret",
	"svc_spawnstaticsound",
	"svc_intermission",
	"svc_finale",			// [string] music [string] text
	"svc_cdtrack",			// [byte] track [byte] looptrack
	"svc_sellscreen",
	"svc_cutscene"
};

//=============================================================================

/*
===============
CL_EntityNum

This error checks and tracks the total number of entities
===============
*/
entity_t	*CL_EntityNum (int num)
{
	if (num >= cl.num_entities)
	{
		if (num >= MAX_EDICTS)
			Host_Error ("CL_EntityNum: %i is an invalid number",num);
		while (cl.num_entities<=num)
		{
			cl_entities[cl.num_entities].colormap = vid.colormap;
			cl.num_entities++;
		}
	}

	return &cl_entities[num];
}

//jft
/*
==================
CL_ParseLocalSoundPacket
==================
*/
void CL_ParseLocalSoundPacket(void)
{
	int 	volume;
	int 	field_mask;
	int 	sound_num;


	field_mask = MSG_ReadByte();

	if (field_mask & SND_VOLUME)
		volume = MSG_ReadByte ();
	else
	    volume = DEFAULT_SOUND_PACKET_VOLUME;

	sound_num = MSG_ReadByte ();

	//qbism//jf 01-10-30
	if (field_mask & SND_MORE1) sound_num += 256;
	if (field_mask & SND_MORE2) sound_num += 512;
	S_LocalSound (cl.sound_precache[sound_num]->name);
}



/*
==================
CL_ParseStartSoundPacket
==================
*/
void CL_ParseStartSoundPacket(void)
{
	vec3_t  pos;
	int 	channel, ent;
	int 	sound_num;
	int 	volume;
	int 	field_mask;
	float 	attenuation;
	int		i;

	field_mask = MSG_ReadByte();

	if (field_mask & SND_VOLUME)
		volume = MSG_ReadByte ();
	else
	    volume = DEFAULT_SOUND_PACKET_VOLUME;

	if (field_mask & SND_ATTENUATION)
		attenuation = MSG_ReadByte () / 64.0;
	else
		attenuation = DEFAULT_SOUND_PACKET_ATTENUATION;

	channel = MSG_ReadShort ();
	sound_num = MSG_ReadByte ();

	//qbism//jf 01-10-30
	if (field_mask & SND_MORE1) sound_num += 256;
	if (field_mask & SND_MORE2) sound_num += 512;

	ent = channel >> 3;
	channel &= 7;

	if (ent > MAX_EDICTS)
		Host_Error ("CL_ParseStartSoundPacket: ent = %i", ent);

	for (i=0 ; i<3 ; i++)
		pos[i] = MSG_ReadCoord ();

	S_StartSound (ent, channel, cl.sound_precache[sound_num], pos, volume/255.0, attenuation);
}

/*
==================
CL_KeepaliveMessage

When the client is taking a long time to load stuff, send keepalive messages
so the server doesn't disconnect.
==================
*/
void CL_KeepaliveMessage (void)
{
	float	time;
	static float lastmsg;
	int		ret;
	sizebuf_t	old;
	byte		olddata[8192];

	if (sv.active)
		return;		// no need if server is local

	if (cls.demoplayback)
		return;

	// read messages from server, should just be nops
	old = net_message;
	memcpy (olddata, net_message.data, net_message.cursize);

	do
	    {
		ret = CL_GetMessage ();
		switch (ret)
		{
		default:
			Host_Error ("CL_KeepaliveMessage: CL_GetMessage failed");
		case 0:
			break;	// nothing waiting

		case 1:
			Host_Error ("CL_KeepaliveMessage: received a message");
			break;
		case 2:
			if (MSG_ReadByte() != svc_nop)
				Host_Error ("CL_KeepaliveMessage: datagram wasn't a nop");
			break;
		}
	}
	while (ret);

	net_message = old;
	memcpy (net_message.data, olddata, net_message.cursize);

	// check time
	time = Sys_FloatTime ();
	if (time - lastmsg < 5)
		return;
	lastmsg = time;

	// write out a nop
	Con_Printf ("--> client to server keepalive\n");

	MSG_WriteByte (&cls.message, clc_nop);
	NET_SendMessage (cls.netcon, &cls.message);
	SZ_Clear (&cls.message);
}

//qbism//00-10-15 from DP
/*extern qboolean isworldmodel;
extern char skyname[];
extern float fog_density;
extern float fog_red;
extern float fog_green;
extern float fog_blue;
extern void R_SetSkyBox (char *sky);
extern void FOG_clear();
extern cvar_t r_farclip;
*/
//qbism//00-10-15 function from DP
/*
void CL_ParseEntityLump(char *entdata)
{
char *data;
char key[128], value[1024];
char wadname[128];
int i, j, k;
//	FOG_clear(); // LordHavoc: no fog until set

//	skyname[0] = 0; // LordHavoc: no enviroment mapped sky until set

//	r_farclip.value = 6144; // LordHavoc: default farclip distance

data = entdata;
if (!data)
return;
data = COM_Parse(data);
if (!data)
return; // valid exit

if (com_token[0] != '{')
return; // error

while (1)
{
data = COM_Parse(data);
if (!data)
return; // error

if (com_token[0] == '}')
return; // since we're just parsing the first ent (worldspawn), exit

strcpy(key, com_token);
while (key[strlen(key)-1] == ' ') // remove trailing spaces
key[strlen(key)-1] = 0;
data = COM_Parse(data);
if (!data)
return; // error

strcpy(value, com_token);
if (!strcmp("sky", key))
R_SetSkyBox(value);
else if (!strcmp("skyname", key)) // non-standard, introduced by QuakeForge... sigh.
R_SetSkyBox(value);
else if (!strcmp("qlsky", key)) // non-standard, introduced by QuakeLives (EEK)
R_SetSkyBox(value);
else if (!strcmp("farclip", key))
{
r_farclip.value = atof(value);
if (r_farclip.value < 64)
r_farclip.value = 64;
}
else if (!strcmp("fog", key))
{
scanf(value, "%f %f %f %f", &fog_density, &fog_red, &fog_green, &fog_blue);
j = 0;
}
else if (!strcmp("fog_density", key))
fog_density = atof(value);
else if (!strcmp("fog_red", key))
fog_red = atof(value);
else if (!strcmp("fog_green", key))
fog_green = atof(value);
else if (!strcmp("fog_blue", key))
fog_blue = atof(value);
else if (!strcmp("wad", key)) // for HalfLife maps
{
j = 0;
for (i = 0;i < 128;i++)
if (value[i] != ';' && value[i] != '\\' && value[i] != '/' && value[i] != ':')
break;
if (value[i])
{
for (;i < 128;i++)
{
// ignore path - the \\ check is for HalfLife... stupid windoze 'programmers'...
if (value[i] == '\\' || value[i] == '/' || value[i] == ':')
j = i+1;
else if (value[i] == ';' || value[i] == 0)
{
k = value[i];
value[i] = 0;
strcpy(wadname, "textures/");
strcat(wadname, &value[j]);
W_LoadTextureWadFile (wadname, FALSE);
j = i+1;
if (!k)
break;
}
}
}
}
}
}

*/

/*
==================
CL_ParseServerInfo
==================
*/
void CL_ParseServerInfo (void)
{
	char	*str;
	int		i;
	int		nummodels, numsounds;
	char	model_precache[MAX_MODELS][MAX_QPATH];
	char	sound_precache[MAX_SOUNDS][MAX_QPATH];

	Con_DPrintf ("Serverinfo packet received.\n"); //qbism//00-10-15 typical for all Con_DPrintf

	//
// wipe the client_state_t struct
//
	CL_ClearState ();

	// parse protocol version number
	i = MSG_ReadLong ();
	if (i != PROTOCOL_VERSION)
	{
		Con_Printf ("Server returned version %i, not %i", i, PROTOCOL_VERSION);
		return;
	}

	// parse maxclients
	cl.maxclients = MSG_ReadByte ();
	if (cl.maxclients < 1 || cl.maxclients > MAX_SCOREBOARD)
	{
		Con_Printf("Bad maxclients (%u) from server\n", cl.maxclients);
		return;
	}
	cl.scores = Hunk_AllocName (cl.maxclients*sizeof(*cl.scores), "scores");
	cl.teamscores = Hunk_AllocName (14 * sizeof(*cl.teamscores), "teamscor"); //qbism// 2000-03-01 JPG - for teamscore status bar

	// parse gametype
	cl.gametype = MSG_ReadByte ();

	// parse signon message
	str = MSG_ReadString ();
	strncpy (cl.levelname, str, sizeof(cl.levelname)-1);

	// seperate the printfs so the server message can have a color
	Con_Printf("\n\n\35\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\36\37\n\n");
	Con_Printf ("%c%s\n", 2, str);

	//
// first we go through and touch all of the precache data that still
// happens to be in the cache, so precaching something else doesn't
// needlessly purge it
//

	// precache models
	memset (cl.model_precache, 0, sizeof(cl.model_precache));
	for (nummodels=1 ; ; nummodels++)
	{
		str = MSG_ReadString ();
		if (!str[0])
			break;
		if (nummodels==MAX_MODELS)
		{
			Con_Printf ("Server sent too many model precaches\n");
			return;
		}
		strcpy (model_precache[nummodels], str);
		Mod_TouchModel (str);
	}

	// precache sounds
	memset (cl.sound_precache, 0, sizeof(cl.sound_precache));
	for (numsounds=1 ; ; numsounds++)
	{
		str = MSG_ReadString ();
		if (!str[0])
			break;
		if (numsounds==MAX_SOUNDS)
		{
			Con_Printf ("Server sent too many sound precaches\n");
			return;
		}
		strcpy (sound_precache[numsounds], str);
		S_TouchSound (str);
	}

	//
// now we try to load everything else until a cache allocation fails
//

	for (i=1 ; i<nummodels ; i++)
	{
		//	isworldmodel = i == 1; // qbism//00-10-15 LordHavoc: first model is the world model

		cl.model_precache[i] = Mod_ForName (model_precache[i], false);
		if (cl.model_precache[i] == NULL)
		{
			Con_Printf("Model %s not found\n", model_precache[i]);
			return;
		}
		CL_KeepaliveMessage ();
	}

	S_BeginPrecaching ();
	for (i=1 ; i<numsounds ; i++)
	{
		cl.sound_precache[i] = S_PrecacheSound (sound_precache[i]);
		CL_KeepaliveMessage ();
	}
	S_EndPrecaching ();


	// local state
	cl_entities[0].model = cl.worldmodel = cl.model_precache[1];

	R_NewMap ();

	Hunk_Check ();		// make sure nothing is hurt

	noclip_anglehack = false;		// noclip is turned off at start

}


/*
==================
CL_ParseUpdate

Parse an entity update message from the server
If an entities model or origin changes from frame to frame, it must be
relinked.  Other attributes can change without relinking.
==================
*/
int	bitcounts[16];

void CL_ParseUpdate (int bits)
{
	int			i;
	int 		xtend=0; //qbism//jf 01-10-30

	model_t		*model;
	int			modnum;
	qboolean	forcelink;
	entity_t	*ent;
	int			num;
	int			skin;


	if (cls.signon == SIGNONS - 1)
	{	// first update is the final signon stage

		cls.signon = SIGNONS;
		CL_SignonReply ();
	}

	if (bits & U_MOREBITS)
		bits |= (MSG_ReadByte()<<8);
	if (bits & U_EXTEND1)
	{
		bits |= MSG_ReadByte() << 16;
		if (bits & U_EXTEND2)
			bits |= MSG_ReadByte() << 24;
	}

	if (bits & U_LONGENTITY)
		num = MSG_ReadShort ();
	else
		num = MSG_ReadByte ();

	ent = CL_EntityNum (num);

	// qbism//00-10-15 begin from DP- code cleanup
//for (i=0 ; i<16 ; i++)
//if (bits&(1<<i))
//	bitcounts[i]++;

	forcelink = ent->msgtime != cl.mtime[1]; // no previous frame to lerp from

	ent->msgtime = cl.mtime[0];
	// qbism//00-10-15 end from DP

	if (bits & U_MODEL)
	{
		modnum = MSG_ReadByte ();
		if (modnum >= MAX_MODELS)
			Host_Error ("CL_ParseModel: bad modnum");
	}
	else modnum = ent->baseline.modelindex;

	model = cl.model_precache[modnum];
	if (model != ent->model)
	{
		ent->model = model;
		// automatic animation (torches, etc) can be either all together
	// or randomized
		if (model)
		{
			if (model->synctype == ST_RAND)
				ent->syncbase = (float)(rand()&0x7fff) / 0x7fff;
			else
			    ent->syncbase = 0.0;
		}
		else
			forcelink = true;	// hack to make null model players work

		//qbism//jf 00-10-15 player MD2

		/* jft no need to translate skin if all MD2
		if (num > 0 && num <= cl.maxclients)
		{
			if(model && model->aliastype == ALIASTYPE_MD2)
			{
				//q3q anything special for md2...
			}
			else R_TranslatePlayerSkin (num - 1);
		}
	*/
	}

	if (bits & U_FRAME)
		ent->frame = MSG_ReadByte ();
	else
		ent->frame = ent->baseline.frame;

	if (bits & U_COLORMAP)
		i = MSG_ReadByte();
	else
		i = ent->baseline.colormap;
	if (!i)
		ent->colormap = vid.colormap;
	else
	{
		if (i > cl.maxclients)
			Sys_Error ("i >= cl.maxclients");
		ent->colormap = cl.scores[i-1].translations;
	}


	//qbism//jf 00-10-15 MD2 stuff doesn't break mdl
		if (bits & U_SKIN)
		skin = MSG_ReadByte();
	else
		skin = ent->baseline.skin;
		ent->skinnum = skin;


	//qbism//jf 01-10-30
if (bits & U_LEGSFRAME)
		ent->legsframe = MSG_ReadByte ();
	else
		ent->legsframe = ent->baseline.legsframe;

	if (bits & U_LEGSX)
	{
		xtend = MSG_ReadByte ();
		if (xtend & X_MODEL)
		{
			modnum = MSG_ReadByte ();
			if (modnum >= MAX_MODELS) Host_Error ("CL_ParseModel: bad modnum");
		}
		else modnum = ent->baseline.legsmodelindex;
		if (modnum) model = cl.model_precache[modnum];
		else model= NULL;
		if (model != ent->legsmodel)
		{
			ent->legsmodel = model;
		}


		if (xtend & X_SKIN)
			skin = MSG_ReadByte();
		else
			skin = ent->baseline.legsskin;
		ent->legsskinnum = skin;
	}
	else xtend = 0;

	if (xtend & X_ANGLES1)
		ent->legsangles[0] = MSG_ReadAngle();
	else ent->legsangles[0]= ent->legsangles1[0]= ent->legsangles2[0]= 0;

	if (xtend & X_ANGLES2)
		ent->legsangles[1] = MSG_ReadAngle();
	else ent->legsangles[1]= ent->legsangles1[1]= ent->legsangles2[1]= 0;

	if (xtend & X_ANGLES3)
		ent->legsangles[2] = MSG_ReadAngle();
	else ent->legsangles[2]= ent->legsangles1[2]= ent->legsangles2[2]= 0;

	xtend = 0;

	if (bits & U_TORSOFRAME)
		ent->torsoframe = MSG_ReadByte ();
	else
		ent->torsoframe = ent->baseline.torsoframe;

	if (bits & U_TORSOX){
		xtend = MSG_ReadByte ();

		if (xtend & X_MODEL)
		{
			modnum = MSG_ReadByte ();
			if (modnum >= MAX_MODELS) Host_Error ("CL_ParseModel: bad modnum");
		}
		else modnum = ent->baseline.torsomodelindex;
		if (modnum) model = cl.model_precache[modnum];
		else model= NULL;

		if (model != ent->torsomodel)
		{
			ent->torsomodel = model;
		}

		if (xtend & X_SKIN)
			skin = MSG_ReadByte();
		else
			skin = ent->baseline.torsoskin;
			ent->torsoskinnum = skin;

	}
	else xtend = 0;
	if (xtend & X_ANGLES1)
		ent->torsoangles[0] = MSG_ReadAngle();
	else ent->torsoangles[0]= ent->torsoangles1[0]= ent->torsoangles2[0]= 0;

	if (xtend & X_ANGLES2)
		ent->torsoangles[1] = MSG_ReadAngle();
	else ent->torsoangles[1]= ent->torsoangles1[1]= ent->torsoangles2[1]= 0;

	if (xtend & X_ANGLES3)
		ent->torsoangles[2] = MSG_ReadAngle();
	else ent->torsoangles[2]= ent->torsoangles1[2]= ent->torsoangles2[2]= 0;

	xtend = 0;

	if (bits & U_VWEPFRAME)
		ent->vwepframe = MSG_ReadByte ();
	else
		ent->vwepframe = ent->baseline.vwepframe;

	if (bits & U_VWEPX)
	{
		xtend = MSG_ReadByte ();

		if (xtend & X_MODEL)
		{
			modnum = MSG_ReadByte ();
			if (modnum >= MAX_MODELS) Host_Error ("CL_ParseModel: bad modnum");
		}
		else modnum = ent->baseline.vwepmodelindex;
		if (modnum) model = cl.model_precache[modnum];
		else model= NULL;

		if (model != ent->vwepmodel)
		{
			ent->vwepmodel = model;
		}

		if (xtend & X_SKIN)
			skin = MSG_ReadByte();
		else
			skin = ent->baseline.vwepskin;
			ent->vwepskinnum = skin;
	}

	else xtend = 0;
	if (xtend & X_ANGLES1)
		ent->vwepangles[0] = MSG_ReadAngle();
	else ent->vwepangles[0]= ent->vwepangles1[0]= ent->vwepangles2[0]= 0;

	if (xtend & X_ANGLES2)
		ent->vwepangles[1] = MSG_ReadAngle();
	else ent->vwepangles[1]= ent->vwepangles1[1]= ent->vwepangles2[1]= 0;

	if (xtend & X_ANGLES3)
		ent->vwepangles[2] = MSG_ReadAngle();
	else ent->vwepangles[2]= ent->vwepangles1[2]= ent->vwepangles2[2]= 0;



	if (bits & U_ALPHA) ent->alpha = MSG_ReadByte();
	else ent->alpha = ent->baseline.alpha;

	if (bits & U_LEGSALPHA) ent->legsalpha = MSG_ReadByte();
	else ent->legsalpha = ent->baseline.legsalpha;

	if (bits & U_TORSOALPHA) ent->torsoalpha = MSG_ReadByte();
	else ent->torsoalpha = ent->baseline.torsoalpha;

	if (bits & U_VWEPALPHA) ent->vwepalpha = MSG_ReadByte();
	else ent->vwepalpha = ent->baseline.vwepalpha;


	if (bits & U_EFFECTS)
		ent->effects = MSG_ReadByte();
	else
		ent->effects = ent->baseline.effects;

	if (bits & U_EFFECTSTRAIL)
		ent->effectstrail = MSG_ReadByte();
	else
		ent->effectstrail = ent->baseline.effectstrail;

	// shift the known values for interpolation
	VectorCopy (ent->msg_origins[0], ent->msg_origins[1]);
	VectorCopy (ent->msg_angles[0], ent->msg_angles[1]);

	if (bits & U_ORIGIN1)
		ent->msg_origins[0][0] = MSG_ReadCoord ();
	else
		ent->msg_origins[0][0] = ent->baseline.origin[0];
	if (bits & U_ANGLE1)
		ent->msg_angles[0][0] = MSG_ReadAngle();
	else
		ent->msg_angles[0][0] = ent->baseline.angles[0];

	if (bits & U_ORIGIN2)
		ent->msg_origins[0][1] = MSG_ReadCoord ();
	else
		ent->msg_origins[0][1] = ent->baseline.origin[1];
	if (bits & U_ANGLE2)
		ent->msg_angles[0][1] = MSG_ReadAngle();
	else
		ent->msg_angles[0][1] = ent->baseline.angles[1];

	if (bits & U_ORIGIN3)
		ent->msg_origins[0][2] = MSG_ReadCoord ();
	else
		ent->msg_origins[0][2] = ent->baseline.origin[2];
	if (bits & U_ANGLE3)
		ent->msg_angles[0][2] = MSG_ReadAngle();
	else
		ent->msg_angles[0][2] = ent->baseline.angles[2];

	if ( bits & U_NOLERP )
		ent->forcelink = true;

	if ( forcelink )
	{	// didn't have an update last message

		VectorCopy (ent->msg_origins[0], ent->msg_origins[1]);
		VectorCopy (ent->msg_origins[0], ent->origin);
		VectorCopy (ent->msg_angles[0], ent->msg_angles[1]);
		VectorCopy (ent->msg_angles[0], ent->angles);

		ent->forcelink = true;
	}
}

/*
==================
CL_ParseBaseline
==================
*/
void CL_ParseBaseline (entity_t *ent)
{
	int			i;

	ent->baseline.modelindex = MSG_ReadByte ();
	ent->baseline.frame = MSG_ReadByte ();
	ent->baseline.colormap = MSG_ReadByte();
	ent->baseline.skin = MSG_ReadByte();
	ent->baseline.legsmodelindex = 0;
	ent->baseline.torsomodelindex = 0;
	ent->baseline.vwepmodelindex = 0;

	/*	//q3q
	ent->baseline.legsmodelindex = MSG_ReadByte ();
	ent->baseline.legsframe = MSG_ReadByte ();
	ent->baseline.legsskin = MSG_ReadByte();
	ent->baseline.legsangles[0] = MSG_ReadAngle ();
	ent->baseline.legsangles[1] = MSG_ReadAngle ();
	ent->baseline.legsangles[2] = MSG_ReadAngle ();

	ent->baseline.torsomodelindex = MSG_ReadByte ();
	ent->baseline.torsoframe = MSG_ReadByte ();
	ent->baseline.torsoskin = MSG_ReadByte();

	ent->baseline.vwepmodelindex = MSG_ReadByte ();
	ent->baseline.vwepframe = MSG_ReadByte ();
	ent->baseline.vwepskin = MSG_ReadByte();
	*/
	for (i=0 ; i<3 ; i++)
	{
		ent->baseline.origin[i] = MSG_ReadCoord ();
		ent->baseline.angles[i] = MSG_ReadAngle ();
	}
}

/*
==================
CL_ParseClientdata

Server information pertaining to this client only
==================
*/
void CL_ParseClientdata (int bits)
{
	int		i, j;

	if (bits & SU_VIEWHEIGHT)
		cl.viewheight = MSG_ReadChar ();
	else
	    cl.viewheight = DEFAULT_VIEWHEIGHT;

	if (bits & SU_IDEALPITCH)
		cl.idealpitch = MSG_ReadChar ();
	else
	    cl.idealpitch = 0;

	VectorCopy (cl.mvelocity[0], cl.mvelocity[1]);
	for (i=0 ; i<3 ; i++)
	{
		if (bits & (SU_PUNCH1<<i) )
			cl.punchangle[i] = MSG_ReadChar();
		else
		    cl.punchangle[i] = 0;
		if (bits & (SU_VELOCITY1<<i) )
			cl.mvelocity[0][i] = MSG_ReadChar()*16;
		else
			cl.mvelocity[0][i] = 0;
	}

	// [always sent]	if (bits & SU_ITEMS)
		i = MSG_ReadLong ();

	if (cl.items != i)
	{	// set flash times

		Sbar_Changed ();
		for (j=0 ; j<32 ; j++)
			if ( (i & (1<<j)) && !(cl.items & (1<<j)))
				cl.item_gettime[j] = cl.time;
		cl.items = i;
	}

	cl.onground = (bits & SU_ONGROUND) != 0;
	cl.inwater = (bits & SU_INWATER) != 0;

	if (bits & SU_WEAPONFRAME)
		cl.stats[STAT_WEAPONFRAME] = MSG_ReadByte ();
	else
	    cl.stats[STAT_WEAPONFRAME] = 0;

	if (bits & SU_ARMOR)
		i = MSG_ReadByte ();
	else
		i = 0;
	if (cl.stats[STAT_ARMOR] != i)
	{
		cl.stats[STAT_ARMOR] = i;
		Sbar_Changed ();
	}

	if (bits & SU_WEAPON)
		i = MSG_ReadByte ();
	else
		i = 0;
	if (cl.stats[STAT_WEAPON] != i)
	{
		cl.stats[STAT_WEAPON] = i;
		Sbar_Changed ();
	}

	i = MSG_ReadShort ();
	if (cl.stats[STAT_HEALTH] != i)
	{
		cl.stats[STAT_HEALTH] = i;
		Sbar_Changed ();
	}

	i = MSG_ReadByte ();
	if (cl.stats[STAT_AMMO] != i)
	{
		cl.stats[STAT_AMMO] = i;
		Sbar_Changed ();
	}

	for (i=0 ; i<4 ; i++)
	{
		j = MSG_ReadByte ();
		if (cl.stats[STAT_SHELLS+i] != j)
		{
			cl.stats[STAT_SHELLS+i] = j;
			Sbar_Changed ();
		}
	}

	i = MSG_ReadByte ();

	if (standard_quake)
	{
		if (cl.stats[STAT_ACTIVEWEAPON] != i)
		{
			cl.stats[STAT_ACTIVEWEAPON] = i;
			Sbar_Changed ();
		}
	}
	else
	{
		if (cl.stats[STAT_ACTIVEWEAPON] != (1<<i))
		{
			cl.stats[STAT_ACTIVEWEAPON] = (1<<i);
			Sbar_Changed ();
		}
	}
}

/*
=====================
CL_NewTranslation
=====================
*/

//jft could do something simpler now.
void CL_NewTranslation (int slot)
{
	int		i, j;
	int		top, bottom;
	byte	*dest, *source;
	model_t	*model; //qbism//jf 00-10-15 MD2

	if (slot > cl.maxclients)
		Sys_Error ("CL_NewTranslation: slot > cl.maxclients");
	dest = cl.scores[slot].translations;
	source = vid.colormap;
	memcpy (dest, vid.colormap, sizeof(cl.scores[slot].translations));
	top = cl.scores[slot].skins & 0xf0;
	bottom = (cl.scores[slot].skins &15)<<4;

#ifdef GLQUAKE

	//qbism//jf 00-10-15 player MD2
currententity = &cl_entities[1+slot];
	model = currententity->model;
//jft	if (model && model->aliastype == ALIASTYPE_MD2)
//	{
		//qbism//jf 00-10-15 player MD2 switch skin based on shirt color
//			currententity->skinnum = (cl.scores[slot].colors & 15)-1;
//	}
//	else R_TranslatePlayerSkin (slot);
#endif

	for (i=0 ; i<VID_GRADES ; i++, dest += 256, source+=256)
	{
		if (top < 128)	// the artists made some backwards ranges.  sigh.
			memcpy (dest + TOP_RANGE, source + top, 16);
		else
		    for (j=0 ; j<16 ; j++)
			dest[TOP_RANGE+j] = source[top+15-j];

		if (bottom < 128)
			memcpy (dest + BOTTOM_RANGE, source + bottom, 16);
		else
		    for (j=0 ; j<16 ; j++)
			dest[BOTTOM_RANGE+j] = source[bottom+15-j];
	}
}


/*
=====================
CL_ParseStatic
=====================
*/
void CL_ParseStatic (void)
{
	entity_t *ent;
	int		i;

	i = cl.num_statics;
	if (i >= MAX_STATIC_ENTITIES)
		Host_Error ("Too many static entities");
	ent = &cl_static_entities[i];
	cl.num_statics++;
	CL_ParseBaseline (ent);

	// copy it to the current state
	ent->model = cl.model_precache[ent->baseline.modelindex];
	ent->frame = ent->baseline.frame;
	ent->colormap = vid.colormap;
	ent->skinnum = ent->baseline.skin;
	ent->alpha = 255; //qbism//jf 01-10-30
	ent->baseline.legsmodelindex = 0;
	ent->baseline.torsomodelindex = 0;
	ent->baseline.vwepmodelindex = 0;

	//q3q
/*
	ent->effects = ent->baseline.effects;

	ent->legsmodel = cl.model_precache[ent->baseline.legsmodelindex];
	ent->legsframe = ent->baseline.legsframe;
	ent->legsskinnum = ent->baseline.legsskin;
	VectorCopy (ent->baseline.legsangles, ent->legsangles);

	ent->torsomodel = cl.model_precache[ent->baseline.torsomodelindex];
	ent->torsoframe = ent->baseline.torsoframe;
	ent->torsoskinnum = ent->baseline.torsoskin;

	ent->vwepmodel = cl.model_precache[ent->baseline.vwepmodelindex];
	ent->vwepframe = ent->baseline.vwepframe;
	ent->vwepskinnum = ent->baseline.vwepskin;
	*/
	VectorCopy (ent->baseline.origin, ent->origin);
	VectorCopy (ent->baseline.angles, ent->angles);
	R_AddEfrags (ent);
}

/*
===================
CL_ParseStaticSound
===================
*/
void CL_ParseStaticSound (void)
{
	vec3_t		org;
	int			sound_num, vol, atten;
	int			i;

	for (i=0 ; i<3 ; i++)
		org[i] = MSG_ReadCoord ();
	sound_num = MSG_ReadShort (); //qbism//jf 01-10-30 was readbyte
	vol = MSG_ReadByte ();
	atten = MSG_ReadByte ();

	S_StaticSound (cl.sound_precache[sound_num], org, vol, atten);
}

//qbism// 2000-03-01 JPG - added this
int MSG_ReadBytePQ (void)
{
	return MSG_ReadByte() * 16 + MSG_ReadByte() - 272;
}

//qbism// 2000-03-01 JPG - added this
int MSG_ReadShortPQ (void)
{
	return MSG_ReadBytePQ() * 256 + MSG_ReadBytePQ();
}

/* JPG - added this function for ProQuake messages
=======================
CL_ParseProQuakeMessage
=======================
*/
void CL_ParseProQuakeMessage (void)
{
	int cmd, i, j;
	int team, frags, shirt, ping;

	MSG_ReadByte();
	cmd = MSG_ReadByte();

	switch (cmd)
	{
	case pqc_new_team:
		Sbar_Changed ();
		team = MSG_ReadByte() - 16;
		if (team < 0 || team > 16)
			Host_Error ("CL_ParseProQuakeMessage: pqc_new_team invalid team");
		shirt = MSG_ReadByte() - 16;
		cl.teamgame = true;
		cl.teamscores[team].skins = 16 * shirt + team;
		cl.teamscores[team].frags = 0;
		Con_Printf("pqc_new_team %d %d\n", team, shirt);
		break;

	case pqc_erase_team:
		Sbar_Changed ();
		team = MSG_ReadByte() - 16;
		if (team < 0 || team > 13)
			Host_Error ("CL_ParseProQuakeMessage: pqc_erase_team invalid team");
		cl.teamscores[team].skins = 0;
		//Con_Printf("pqc_erase_team %d\n", team);
		break;

	case pqc_team_frags:
		Sbar_Changed ();
		team = MSG_ReadByte() - 16;
		if (team < 0 || team > 13)
			Host_Error ("CL_ParseProQuakeMessage: pqc_team_frags invalid team");
		frags = MSG_ReadShortPQ();
		;
		if (frags & 32768)
			frags = frags - 65536;
		cl.teamscores[team].frags = frags;
		//Con_Printf("pqc_team_frags %d %d\n", team, frags);
		break;

	case pqc_match_time:
		Sbar_Changed ();
		cl.minutes = MSG_ReadBytePQ();
		cl.seconds = MSG_ReadBytePQ();
		cl.last_match_time = cl.time;
		//Con_Printf("pqc_match_time %d %d\n", cl.minutes, cl.seconds);
		break;

	case pqc_match_reset:
		Sbar_Changed ();
		for (i = 0 ; i < 14 ; i++)
			cl.teamscores[i].skins = 0;
		//Con_Printf("pqc_match_reset\n");
		break;

	case pqc_ping_times:
		while (ping = MSG_ReadShortPQ())
		{
			if ((ping / 4096) >= cl.maxclients)
				Host_Error ("CL_ParseProQuakeMessage: pqc_ping_times > MAX_SCOREBOARD");
			cl.scores[ping / 4096].ping = ping & 4095;
		}
		cl.last_ping_time = cl.time;
		/*
		Con_Printf("pqc_ping_times ");
		for (i = 0 ; i < cl.maxclients ; i++)
		Con_Printf("%4d ", cl.scores[i].ping);
		Con_Printf("\n");
		*/
		break;
	}
}

extern cvar_t pq_scoreboard_pings; //qbism// 2000-03-01 JPG - need this for CL_ParseProQuakeString

/* JPG - on a svc_print, check to see if the string contains useful information
======================
CL_ParseProQuakeString
======================
*/
void CL_ParseProQuakeString (char *string)
{
	static int checkping = -1;
	int ping, i;
	char *s, *s2;
	;

	// check for a ping time
	s = string;
	if (checkping >= 0)
	{
		while (*s == ' ')
			s++;
		ping = 0;
		while (*s >= '0' && *s <= '9')
			ping = 10 * ping + *s++ - '0';
		if ((*s++ == ' ') && *s && (s2 = strchr(s, '\n')))
		{
			*s2 = 0;
			if (strcmp(s, "unconnected"))
			{
				for ( ; checkping < cl.maxclients ; checkping++)
				{

					if (!strcmp(cl.scores[checkping].name, s))
					{
						cl.scores[checkping].ping = ping > 9999 ? 9999 : ping;
						if (cl.sbar_ping)
							*string = 0;
						break;
					}
				}
			}
			*s2 = '\n';
			if (checkping == cl.maxclients)
				checkping = -1;
		}
		else
		    checkping = -1;
		cl.sbar_ping = cl.sbar_ping && (checkping >= 0);
	}

	// check for match time
	if (!strncmp(string, "Match ends in ", 14))
	{
		s = string + 14;
		if ((*s != 'T') && strchr(s, 'm'))
		{
			sscanf(s, "%d", &cl.minutes);
			cl.seconds = 0;
			cl.last_match_time = cl.time;
		}
	}
	else if (!strcmp(string, "Match paused\n"))
		cl.match_pause_time = cl.time;
	else if (!strcmp(string, "Match unpaused\n"))
	{
		cl.last_match_time += (cl.time - cl.match_pause_time);
		cl.match_pause_time = 0;
	}
	else if (!strcmp(string, "The match is over\n") || !strncmp(string, "Match begins in", 15))
		cl.minutes = 255;
	else if (checkping < 0)
	{
		s = string;
		i = 0;
		while (*s >= '0' && *s <= '9')
			i = 10 * i + *s++ - '0';
		if (!strcmp(s, " minutes remaining\n"))
		{
			cl.minutes = i;
			cl.seconds = 0;
			cl.last_match_time = cl.time;
		}
	}

	if (!strcmp(string, "Client ping times:\n") && pq_scoreboard_pings.value)
	{
		cl.last_ping_time = cl.time;
		checkping = 0;
		if (cl.sbar_ping)
			*string = 0;
	}
}

#define SHOWNET(x) if(cl_shownet.value==2)Con_Printf ("%3i:%s\n", msg_readcount-1, x);

/*
=====================
CL_ParseServerMessage
=====================
*/
void CL_ParseServerMessage (void)
{
	int			cmd;
	int			i;
	char		*s;

	//
// if recording demos, copy the message out
//
	if (cl_shownet.value == 1)
		Con_Printf ("%i ",net_message.cursize);
	else if (cl_shownet.value == 2)
		Con_Printf ("------------------\n");

	cl.onground = false;	// unless the server says otherwise

	//
// parse the message
//
	MSG_BeginReading ();

	while (1)
	{
		if (msg_badread)
			Host_Error ("CL_ParseServerMessage: Bad server message");

		cmd = MSG_ReadByte ();

		if (cmd == -1)
		{
			SHOWNET("END OF MESSAGE");
			return;		// end of message

		}

		// if the high bit of the command byte is set, it is a fast update
		if (cmd & 128)
		{
			SHOWNET("fast update");
			CL_ParseUpdate (cmd&127);
			continue;
		}

		SHOWNET(svc_strings[cmd]);

		// other commands
		switch (cmd)
		{
		default:
			Host_Error ("CL_ParseServerMessage: Illegible server message\n");
			break;

		case svc_nop:
			//			Con_Printf ("svc_nop\n");
			break;

		case svc_time:
			cl.mtime[1] = cl.mtime[0];
			cl.mtime[0] = MSG_ReadFloat ();
			break;

		case svc_clientdata:
			i = MSG_ReadShort ();
			CL_ParseClientdata (i);
			break;

		case svc_version:
			i = MSG_ReadLong ();
			if (i != PROTOCOL_VERSION)
				Host_Error ("CL_ParseServerMessage: Server is protocol %i instead of %i\n", i, PROTOCOL_VERSION);
			break;

		case svc_disconnect:
			Host_EndGame ("Server disconnected\n");

		case svc_print:
			//qbism// 2000-03-01 JPG - check to see if the message contains useful information
			s = MSG_ReadString();
			CL_ParseProQuakeString(s);
			Con_Printf ("%s", s);
			break;

		case svc_centerprint:
			SCR_CenterPrint (MSG_ReadString ());
			break;

		case svc_stufftext:
			//qbism// 2000-03-01 JPG - check for ProQuake message
			if (MSG_PeekByte() == MOD_PROQUAKE)
				CL_ParseProQuakeMessage();
			// Still want to add text, even on ProQuake messages.  This guarantees compatibility;
			// unrecognized messages will essentially be ignored but there will be no parse errors
			Cbuf_AddText (MSG_ReadString ());
			break;

		case svc_damage:
			V_ParseDamage ();
			break;

		case svc_serverinfo:
			CL_ParseServerInfo ();
			vid.recalc_refdef = true;	// leave intermission full screen

			break;

		case svc_setangle:
			for (i=0 ; i<3 ; i++)
				cl.viewangles[i] = MSG_ReadAngle ();
			break;


		case svc_setview:
			cl.viewentity = MSG_ReadShort ();
			break;

		case svc_lightstyle:
			i = MSG_ReadByte ();
			if (i >= MAX_LIGHTSTYLES)
				Sys_Error ("svc_lightstyle > MAX_LIGHTSTYLES");
			Q_strcpy (cl_lightstyle[i].map,  MSG_ReadString());
			cl_lightstyle[i].length = Q_strlen(cl_lightstyle[i].map);
			break;

		//jft
		case svc_localsound:
			CL_ParseLocalSoundPacket();
			break;

		case svc_sound:
			CL_ParseStartSoundPacket();
			break;

		case svc_stopsound:
			i = MSG_ReadShort();
			S_StopSound(i>>3, i&7);
			break;

		case svc_updatename:
			Sbar_Changed ();
			i = MSG_ReadByte ();
			if (i >= cl.maxclients)
				Host_Error ("CL_ParseServerMessage: svc_updatename > MAX_SCOREBOARD");
			strcpy (cl.scores[i].name, MSG_ReadString ());
			break;

		case svc_updatefrags:
			Sbar_Changed ();
			i = MSG_ReadByte ();
			if (i >= cl.maxclients)
				Host_Error ("CL_ParseServerMessage: svc_updatefrags > MAX_SCOREBOARD");
			cl.scores[i].frags = MSG_ReadShort ();
			break;

		case svc_updateskins:
			Sbar_Changed ();
			i = MSG_ReadByte ();
			if (i >= cl.maxclients)
				Host_Error ("CL_ParseServerMessage: svc_updateskins > MAX_SCOREBOARD");
			cl.scores[i].skins = MSG_ReadByte ();
			CL_NewTranslation (i);
			break;
/*		case svc_updatebodies:
			//jft FUTURE- display body type in some way- Sbar_Changed ();
			i = MSG_ReadByte ();
			if (i >= cl.maxclients)
				Host_Error ("CL_ParseServerMessage: svc_updatebodies > MAX_SCOREBOARD");
			cl.scores[i].bodies = MSG_ReadByte ();
			break;
*/
		case svc_particle:
			R_ParseParticleEffect ();
			break;

		case svc_spawnbaseline:
			i = MSG_ReadShort ();
			// must use CL_EntityNum() to force cl.num_entities up
			CL_ParseBaseline (CL_EntityNum(i));
			break;
		case svc_spawnstatic:
			CL_ParseStatic ();
			break;
		case svc_temp_entity:
			CL_ParseTEnt ();
			break;

		case svc_setpause:

			cl.paused = MSG_ReadByte ();

			if (cl.paused)
			{
				CDAudio_Pause ();
			}
			else
			    {
				CDAudio_Resume ();
			}
			break;

		case svc_signonnum:
			i = MSG_ReadByte ();
			if (i <= cls.signon)
				Host_Error ("Received signon %i when at %i", i, cls.signon);
			cls.signon = i;
			CL_SignonReply ();
			break;

		case svc_killedmonster:
			cl.stats[STAT_MONSTERS]++;
			break;

		case svc_foundsecret:
			cl.stats[STAT_SECRETS]++;
			break;

		case svc_updatestat:
			i = MSG_ReadByte ();
			if (i < 0 || i >= MAX_CL_STATS)
				Sys_Error ("svc_updatestat: %i is invalid", i);
			cl.stats[i] = MSG_ReadLong ();
			;
			break;

		case svc_spawnstaticsound:
			CL_ParseStaticSound ();
			break;

		case svc_cdtrack:
			cl.cdtrack = MSG_ReadByte ();
			cl.looptrack = MSG_ReadByte ();
			if ( (cls.demoplayback || cls.demorecording) && (cls.forcetrack != -1) )
				CDAudio_Play ((byte)cls.forcetrack, true);
			else
			    CDAudio_Play ((byte)cl.cdtrack, true);
			break;

		case svc_intermission:
			cl.intermission = 1;
			cl.completed_time = cl.time;
			vid.recalc_refdef = true;	// go to full screen

			break;

		case svc_finale:
			cl.intermission = 2;
			cl.completed_time = cl.time;
			vid.recalc_refdef = true;	// go to full screen

			SCR_CenterPrint (MSG_ReadString ());
			break;

		case svc_cutscene:
			cl.intermission = 3;
			cl.completed_time = cl.time;
			vid.recalc_refdef = true;	// go to full screen

			SCR_CenterPrint (MSG_ReadString ());
			break;

		case svc_sellscreen:
			Cmd_ExecuteString ("help", src_command);
			break;

			// qbism//00-10-15 LordHavoc: extra worldspawn fields (fog, sky, farclip)
/*		case svc_skybox:
			R_SetSkyBox(MSG_ReadString());
			break;
			case svc_farclip:
			r_farclip.value = MSG_ReadCoord();
			break;
			case svc_fog:
			if (MSG_ReadByte())
			{
			fog_density = MSG_ReadShort() * (1.0f / 4096.0f);
			fog_red = MSG_ReadByte() * (1.0 / 255.0);
			fog_green = MSG_ReadByte() * (1.0 / 255.0);
			fog_blue = MSG_ReadByte() * (1.0 / 255.0);
			}
			else
			fog_density = 0.0f;
			break; */
		}
	}
}


