/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
/*
RScript loading, parsing, and rendering.

Code syntax is similar to that of shaders
No reference to any shader material was used whilst
making this code, hence the weak and buggy state of it :)
*/

#include "quakedef.h"
#include "gl_shaders.h"

int		caustics_stage;

typedef struct glRect_s {
	unsigned char l,t,w,h;
} glRect_t;

typedef struct
{
	int		texnum;
	float	sl, tl, sh, th;
} glpic_t;

#define	MAX_LIGHTMAPS	1024 // 64
#define	BLOCK_WIDTH		128
#define	BLOCK_HEIGHT	128

extern	int			lightmap_bytes;
extern	int			lightmap_textures;
extern	glpoly_t	*lightmap_polys[MAX_LIGHTMAPS];
extern	qboolean	lightmap_modified[MAX_LIGHTMAPS];
extern	glRect_t	lightmap_rectchange[MAX_LIGHTMAPS];
extern	byte		lightmaps[4*MAX_LIGHTMAPS*BLOCK_WIDTH*BLOCK_HEIGHT];

int RS_AnimTexture(int rs)
{
	double rt;

	if (host_time < rscripts[rs].anim.lasttime)
		rscripts[rs].anim.lasttime=0;
	rt = host_time - rscripts[rs].anim.lasttime;
	if (rt < rscripts[rs].flags.animtime)
		return rscripts[rs].anim.texnum[rscripts[rs].anim.current];
	if (rt > rscripts[rs].flags.animtime)
	{
		rscripts[rs].anim.current+=(rt / rscripts[rs].flags.animtime);
		while (rscripts[rs].anim.current >= rscripts[rs].anim.num)
			rscripts[rs].anim.current = rscripts[rs].anim.current - rscripts[rs].anim.num;
		rscripts[rs].anim.lasttime+=rscripts[rs].flags.animtime;
	}
	return rscripts[rs].anim.texnum[rscripts[rs].anim.current];
}

float MakeMapXCoord(float x, int rs)
{
	float txm;

	if (rs > MAX_RS) rs=0;
	if (!rs) return x;
	txm=0;
	if (rscripts[rs].usescroll) {
		txm=realtime*rscripts[rs].scroll.xspeed;
		while (txm > 1 && (1-txm) > 0) txm=1-txm;
		while (txm < 0 && (1+txm) > 1) txm=1+txm;
	}
	if (rscripts[rs].useturb) {
		float power, movediv;

		power = rscripts[rs].turb.power / 20;
		movediv = rscripts[rs].turb.movediv;
		x += sin((x*0.1+realtime) * power) * sin((x*0.1+realtime))/movediv;
	}
	x+=txm; return x*rscripts[rs].texscale;
}

float MakeMapYCoord(float y, int rs)
{
	float tym;

	if (rs > MAX_RS) rs=0;
	if (!rs) return y;
	tym=0;
	if (rscripts[rs].usescroll) {
		tym=realtime*rscripts[rs].scroll.yspeed;
		while (tym > 1 && (1-tym) > 0) tym=1-tym;
		while (tym < 0 && (1+tym) > 1) tym=1+tym;
	}
	if (rscripts[rs].useturb) {
		float power, movediv;

		movediv = rscripts[rs].turb.movediv;
		power = rscripts[rs].turb.power / 20;
		y += sin((y*0.1+realtime) * power) * sin((y*0.1+realtime))/movediv;
	}
	y+=tym; return y*rscripts[rs].texscale;
}

void RS_DrawPic (int x, int y, qpic_t *pic)
{
	byte			*dest, *source;
	unsigned short	*pusdest;
	int				v, u, rs;
	glpic_t			*gl;
	BOOL			stage;
	float			tx,ty;

	gl = (glpic_t *)pic->data;
	glColor4f (1,1,1,1);
	stage = true;
	rs = pic->rs;

	while (stage)
	{
		glColor4f(1,1,1,1);
		if (rscripts[rs].flags.blendfunc)
		{
			glEnable(GL_BLEND);
			glColor4f(1,1,1,rscripts[rs].flags.alpha);
		} else
			glDisable(GL_BLEND);
		if (rscripts[rs].useblendmode)
			glBlendFunc(rscripts[rs].blendmode.srcmode, rscripts[rs].blendmode.dstmode);
		if (rscripts[rs].flags.alphafunc) {
			glAlphaFunc(GL_GEQUAL, 0.05f);
			glEnable(GL_ALPHA_TEST);
		}
		else
			glDisable(GL_ALPHA_TEST);
		if (rscripts[rs].flags.envmap) {
			glTexGenf(GL_S,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
			glTexGenf(GL_T,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
			glEnable(GL_TEXTURE_GEN_S);
			glEnable(GL_TEXTURE_GEN_T);
		} else {
			glDisable(GL_TEXTURE_GEN_S);
			glDisable(GL_TEXTURE_GEN_T);
		}
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

		if (rscripts[rs].useanim)
			GL_Bind (RS_AnimTexture(rs));
		else if (rscripts[rs].texnum)
			GL_Bind (rscripts[rs].texnum);
		else
			GL_Bind (gl->texnum);
			
		glBegin (GL_QUADS);

		tx = gl->sl; ty = gl->tl;
		tx=MakeMapXCoord(tx,rs);
		ty=MakeMapYCoord(ty,rs);
		glTexCoord2f (tx, ty);
		glVertex2f (x, y);

		tx = gl->sh; ty = gl->tl;
		tx=MakeMapXCoord(tx,rs);
		ty=MakeMapYCoord(ty,rs);
		glTexCoord2f (tx, ty);
		glVertex2f (x+pic->width, y);

		tx = gl->sh; ty = gl->th;
		tx=MakeMapXCoord(tx,rs);
		ty=MakeMapYCoord(ty,rs);
		glTexCoord2f (tx, ty);
		glVertex2f (x+pic->width, y+pic->height);

		tx = gl->sl; ty = gl->th;
		tx=MakeMapXCoord(tx,rs);
		ty=MakeMapYCoord(ty,rs);
		glTexCoord2f (tx, ty);
		glVertex2f (x, y+pic->height);

		glEnd ();
		glColor4f(1,1,1,1);
		glDisable(GL_BLEND);
		glDisable(GL_ALPHA_TEST);
		glDisable(GL_TEXTURE_GEN_S);
		glDisable(GL_TEXTURE_GEN_T);

		if (rscripts[rs].nextstage)
			rs = rscripts[rs].nextstage;
		else
			stage = false;
	}
	glAlphaFunc(GL_GEQUAL, 0.05f);
	glEnable(GL_ALPHA_TEST);
}

void DrawPolygon (msurface_t *s)
{
	glpoly_t	*p;
	float		*v, vt[3], os, ot;
	texture_t	*t;
	vec3_t		nv, dir;
	int			i, rs;
	float		tx,ty;
	glRect_t	*theRect;
	float		txm, tym;
	BOOL		stage, draw_caustics;

	p = s->polys;
	t = s->texinfo->texture;
	rs = t->rs;
	stage=true;
	draw_caustics=false;
	
	while (stage)
	{
		// Binds world to texture env 0
		if (!rscripts[rs].flags.nolightmap && !rscripts[rs].flags.drawsky)
		{
				GL_SelectTexture(TEXTURE0_SGIS);
				if (rs > 0 && !rscripts[rs].useanim)
					GL_Bind (rscripts[rs].texnum);
				else if (rs > 0 && rscripts[rs].useanim)
					GL_Bind (RS_AnimTexture(rs));
				else
					GL_Bind (t->gl_texturenum);
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
				// Binds lightmap to texenv 1
				GL_EnableMultitexture(); // Same as SelectTexture (TEXTURE1)
				GL_Bind (lightmap_textures + s->lightmaptexturenum);
				i = s->lightmaptexturenum;
				if (lightmap_modified[i])
				{
					lightmap_modified[i] = false;
					theRect = &lightmap_rectchange[i];
					glTexSubImage2D(GL_TEXTURE_2D, 0, 0, theRect->t, 
						BLOCK_WIDTH, theRect->h, gl_lightmap_format, GL_UNSIGNED_BYTE,
						lightmaps+(i* BLOCK_HEIGHT + theRect->t) *BLOCK_WIDTH*lightmap_bytes);
					theRect->l = BLOCK_WIDTH;
					theRect->t = BLOCK_HEIGHT;
					theRect->h = 0;
					theRect->w = 0;
				}
			glColor4f(1,1,1,1);
			GL_SelectTexture(TEXTURE0_SGIS);
			if (rscripts[rs].flags.blendfunc)
			{
				glEnable(GL_BLEND);
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
			}
			else
				glDisable(GL_BLEND);
			if (rscripts[rs].useblendmode)
				glBlendFunc(rscripts[rs].blendmode.srcmode, rscripts[rs].blendmode.dstmode);

			if (rscripts[rs].flags.alphafunc) {
				glAlphaFunc(GL_GEQUAL, 0.05f);
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
				glEnable(GL_ALPHA_TEST);
			}
			else
				glDisable(GL_ALPHA_TEST);
			glColor4f(1,1,1,rscripts[rs].flags.alpha);

			if (rscripts[rs].flags.envmap) {
				glTexGenf(GL_S,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
				glTexGenf(GL_T,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
				glEnable(GL_TEXTURE_GEN_S);
				glEnable(GL_TEXTURE_GEN_T);
			} else {
				glDisable(GL_TEXTURE_GEN_S);
				glDisable(GL_TEXTURE_GEN_T);
			}
			GL_EnableMultitexture();
			if (!rs) {
				if (t->transparent) {
					glEnable(GL_ALPHA_TEST);
					glAlphaFunc(GL_GEQUAL, 0.05f);
					glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
				}
			}
			
			glBegin (GL_POLYGON);
			v = p->verts[0];
			for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
			{
				txm=0; tym=0;
				if (rscripts[rs].usescroll) {
					txm=realtime*rscripts[rs].scroll.xspeed;
					while (txm > 1 && (1-txm) > 0) txm=1-txm;
					while (txm < 0 && (1+txm) > 1) txm=1+txm;
					tym=realtime*rscripts[rs].scroll.yspeed;
					while (tym > 1 && (1-tym) > 0) tym=1-tym;
					while (tym < 0 && (1+tym) > 1) tym=1+tym;
				}
				if (rscripts[rs].useturb) {
					float power, movediv;

					power = rscripts[rs].turb.power / 20;
					movediv = rscripts[rs].turb.movediv;
					os=v[3]; ot = v[4];
					vt[0] = os + sin((os*0.1+realtime) * power) * sin((ot*0.1+realtime))/movediv;
					vt[1] = ot + sin((ot*0.1+realtime) * power) * sin((os*0.1+realtime))/movediv;
				} else {
					vt[0] = v[3];
					vt[1] = v[4];
				}
				if (rscripts[rs].usevturb)
				{
					float power;

					power = rscripts[rs].vturb.power;
					nv[0] = v[0] + power*sin(v[1]*0.05+realtime)*sin(v[2]*0.05+realtime);
					nv[1] = v[1] + power*sin(v[0]*0.05+realtime)*sin(v[2]*0.05+realtime);
					nv[2] = v[2];
				} else {
					nv[0] = v[0];
					nv[1] = v[1];
					nv[2] = v[2];
				}

				qglMTexCoord2fSGIS (TEXTURE0_SGIS, vt[0]+txm, vt[1]+tym);
				qglMTexCoord2fSGIS (TEXTURE1_SGIS, v[5], v[6]);

				glVertex3fv (nv);
			}
			glEnd ();
			GL_DisableMultitexture();
		} else if (rscripts[rs].flags.nolightmap && !rscripts[rs].flags.drawsky) {
			GL_DisableMultitexture();
			if (rs > 0 && !rscripts[rs].useanim)
				GL_Bind (rscripts[rs].texnum);
			else if (rs > 0 && rscripts[rs].useanim)
				GL_Bind (RS_AnimTexture(rs));
			else
				GL_Bind (t->gl_texturenum);
			glColor4f(1,1,1,1);
			if (rscripts[rs].flags.blendfunc)
			{
				glEnable(GL_BLEND);
				glColor4f(1,1,1,rscripts[rs].flags.alpha);
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
			} else
				glDisable(GL_BLEND);
			if (rscripts[rs].useblendmode)
				glBlendFunc(rscripts[rs].blendmode.srcmode, rscripts[rs].blendmode.dstmode);

			if (rscripts[rs].flags.alphafunc) {
				glAlphaFunc(GL_GEQUAL, 0.05f);
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
				glEnable(GL_ALPHA_TEST);
			}
			else
				glDisable(GL_ALPHA_TEST);
			if (rscripts[rs].flags.envmap) {
				glTexGenf(GL_S,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
				glTexGenf(GL_T,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
				glEnable(GL_TEXTURE_GEN_S);
				glEnable(GL_TEXTURE_GEN_T);
			} else {
				glDisable(GL_TEXTURE_GEN_S);
				glDisable(GL_TEXTURE_GEN_T);
			}

			glBegin (GL_POLYGON);
			v = p->verts[0];
			for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
			{
				txm=0; tym=0;
				if (rscripts[rs].usescroll) {
					txm=realtime*rscripts[rs].scroll.xspeed;
					while (txm > 1 && (1-txm) > 0) txm=1-txm;
					while (txm < 0 && (1+txm) > 1) txm=1+txm;
					tym=realtime*rscripts[rs].scroll.yspeed;
					while (tym > 1 && (1-tym) > 0) tym=1-tym;
					while (tym < 0 && (1+tym) > 1) tym=1+tym;
				}
				if (rscripts[rs].useturb) {
					float power, movediv;

					power = rscripts[rs].turb.power / 20;
					movediv = rscripts[rs].turb.movediv;
					os=v[3]; ot = v[4];
					vt[0] = os + sin((os*0.1+realtime) * power) * sin((ot*0.1+realtime))/movediv;
					vt[1] = ot + sin((ot*0.1+realtime) * power) * sin((os*0.1+realtime))/movediv;
				} else {
					vt[0] = v[3];
					vt[1] = v[4];
				}
				if (rscripts[rs].usevturb)
				{
					float power;

					power = rscripts[rs].vturb.power;
					nv[0] = v[0] + power*sin(v[1]*0.05+realtime)*sin(v[2]*0.05+realtime);
					nv[1] = v[1] + power*sin(v[0]*0.05+realtime)*sin(v[2]*0.05+realtime);
					nv[2] = v[2];
				} else {
					nv[0] = v[0];
					nv[1] = v[1];
					nv[2] = v[2];
				}

				glTexCoord2f (vt[0]+txm, vt[1]+tym);
				glVertex3fv (nv);
			}
			glEnd ();
		} else if (rscripts[rs].flags.drawsky) {
			R_DrawSkyChain(s);
		}
		glColor4f(1,1,1,1);
		glDisable(GL_BLEND);
		glDisable(GL_ALPHA_TEST);
		glDisable(GL_TEXTURE_GEN_S);
		glDisable(GL_TEXTURE_GEN_T);
		GL_EnableMultitexture();
		rs=rscripts[rs].nextstage;
		if (!rs && (p->flags & SURF_UNDERWATER) && !draw_caustics) {
			draw_caustics=true;
			rs=caustics_stage;
		} else if (!rs)
			stage = false;
	}

	// Show BSP cuts
	if (gl_showcuts.value)
	{
		GL_DisableMultitexture();
		glDisable(GL_TEXTURE_2D);
		glLineWidth(1.5f);
		glBegin(GL_LINE_LOOP);
		glColor4f(1,1,1,1);
		v = p->verts[0];
		for (i=0 ; i<p->numverts ; i++, v+= VERTEXSIZE)
			glVertex3fv (v);
		glEnd ();
		glEnable(GL_TEXTURE_2D);
	}
}

void FinishScripts(int i)
{
	int c;

	if (!strcmp(rscripts[i].scriptname,""))
		return;
	if (rscripts[i].nextname)
		rscripts[i].nextstage=GetRSForName(rscripts[i].nextname);
	if (!strcmp(rscripts[i].texname,""))
		strcpy(rscripts[i].texname,rscripts[i].scriptname);
	if (rscripts[i].anim.num) {
		for (c=0;c<rscripts[i].anim.num;c++)
			rscripts[i].anim.texnum[c] = loadtextureimage((6000 + i*MAX_ANIM_FRAMES)+c,rscripts[i].anim.name[c].name,false,0,0);
	}
	rscripts[i].texnum = loadtextureimage(5000+i,rscripts[i].texname,false,0,0);
}

int GetRSForName(char name[56])
{
	int i;

	for (i=0;i<MAX_RS;i++)
	{
		if (!_stricmp(name,rscripts[i].scriptname))
		{
			FinishScripts(i);
			return i;
		}
	}
	return 0;
}

float GetBlendForName(char *blend)
{
	if (!strcmp(blend,""))
		return 0;
	if (!_stricmp(blend,"GL_ZERO"))
		return GL_ZERO;
	if (!_stricmp(blend,"GL_ONE"))
		return GL_ONE;
	if (!_stricmp(blend,"GL_DST_COLOR"))
		return GL_DST_COLOR;
	if (!_stricmp(blend,"GL_ONE_MINUS_DST_COLOR"))
		return GL_ONE_MINUS_DST_COLOR;
	if (!_stricmp(blend,"GL_SRC_ALPHA"))
		return GL_SRC_ALPHA;
	if (!_stricmp(blend,"GL_ONE_MINUS_SRC_ALPHA"))
		return GL_ONE_MINUS_SRC_ALPHA;
	if (!_stricmp(blend,"GL_DST_ALPHA"))
		return GL_DST_ALPHA;
	if (!_stricmp(blend,"GL_ONE_MINUS_DST_ALPHA"))
		return GL_ONE_MINUS_DST_ALPHA;
	if (!_stricmp(blend,"GL_SRC_ALPHA_SATURATE"))
		return GL_SRC_ALPHA_SATURATE;
	if (!_stricmp(blend,"GL_SRC_COLOR"))
		return GL_SRC_COLOR;
	if (!_stricmp(blend,"GL_ONE_MINUS_SRC_COLOR"))
		return GL_ONE_MINUS_SRC_COLOR;
	return 0;
}

BOOL torf(float i) { return (i == 1) ? true : false; }

void InitRenderScripts()
{
	FILE		*f;
	char		*ch, *sp1,*sp2, *output;
	float		fp1,fp2,fp3,fp4;
	int			inscript, num,i;

	inscript=0; num=1; output=malloc(56);
	ch=malloc(1024); sp1=malloc(1024); sp2=malloc(1024);
	if (!COM_FOpenFile("textures.rs",&f))
		return;

	do {
		fscanf(f,"%s",ch);
		if (inscript) {
			if (!_stricmp(ch,"turb")) { // turb effect
				fscanf(f,"%f",&fp1); fscanf(f,"%f",&fp2);
				rscripts[num].turb.power = fp1;
				rscripts[num].turb.movediv = fp2;
				rscripts[num].useturb=true;
			} else if (!_stricmp(ch,"turbvert")) { // vertex turb effect
				fscanf(f,"%f",&fp1); fscanf(f,"%f",&fp2); fscanf(f,"%f",&fp3); fscanf(f,"%f",&fp4);
				rscripts[num].vturb.power = fp1;
				rscripts[num].usevturb=true;
			} else if (!_stricmp(ch,"scroll")) { // scrolling texture
				fscanf(f,"%f",&fp1); fscanf(f,"%f",&fp2);
				rscripts[num].scroll.xspeed = fp1;
				rscripts[num].scroll.yspeed = fp2;
				rscripts[num].usescroll=true;
			} else if (!_stricmp(ch,"blend")) { // blendmodes
				fscanf(f,"%s",sp1); fscanf(f,"%s",sp2);
				rscripts[num].blendmode.srcmode = GetBlendForName(sp1);
				rscripts[num].blendmode.dstmode = GetBlendForName(sp2);
				rscripts[num].useblendmode=true;
			} else if (!_stricmp(ch,"stage")) { // next stage
				fscanf(f,"%s",sp1);
				strcpy(rscripts[num].nextname, sp1);
			} else if (!_stricmp(ch,"map")) { // texture
				fscanf(f,"%s",sp1);
				strcpy(rscripts[num].texname, sp1);
			} else if (!_stricmp(ch,"anim")) { // anim map
				fscanf(f,"%f",&fp1);
				rscripts[num].anim.num = fp1;
				for (i=0; i<fp1; i++)
				{
					if (i > MAX_ANIM_FRAMES-1)
						continue;
					fscanf(f,"%s",sp1);
					strcpy(rscripts[num].anim.name[i].name,sp1);
				}
				rscripts[num].useanim=true;
			} else if (!_stricmp(ch,"set")) { // set texture flags
				fscanf(f,"%s",sp1); fscanf(f,"%f",&fp1);
				if (!_stricmp(sp1, "alpha")) // alpha amount
					rscripts[num].flags.alpha=fp1;
				else if (!_stricmp(sp1, "blendfunc")) // use blendfunc?
					rscripts[num].flags.blendfunc=torf(fp1);
				else if (!_stricmp(sp1, "alphafunc")) // use alpha test?
					rscripts[num].flags.alphafunc=torf(fp1);
				else if (!_stricmp(sp1, "nolightmap")) // skip lightmap?
					rscripts[num].flags.nolightmap=torf(fp1);
				else if (!_stricmp(sp1, "texscale")) // texture scaling
					rscripts[num].texscale=fp1;
				else if (!_stricmp(sp1, "envmap")) // environmental mapping?
					rscripts[num].flags.envmap=torf(fp1);
				else if (!_stricmp(sp1, "sky")) // draw sky here?
					rscripts[num].flags.drawsky=torf(fp1);
				else if (!_stricmp(sp1, "animtime")) // animation timing (ms)
					rscripts[num].flags.animtime=fp1;
			} else if (!_stricmp(ch,"}")) {
				inscript=0;
				num++;
			}	
		} else {
			if (_stricmp(ch,"{"))
			{
				strcpy(rscripts[num].scriptname,ch);
			} else {
				rscripts[num].flags.alpha=1;
				rscripts[num].texscale=1;
				rscripts[num].flags.animtime=100;
				inscript=1;
			}
		}
	} while (!feof(f));
	fclose(f);
	caustics_stage = GetRSForName("underwater_caustics");
}
