/* main.c - Main file to test the bezier surface function
 * Programmed by Kilbert <kilbert@inside3d.com>
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef WIN32
#include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <math.h>
#include "bezier.h"

bezier_t bez; // the bezier curve
vec3d normals[BEZIER_RES_Y][BEZIER_RES_X+1][2];

float pitch, yaw, roll; // viewer's angles
float orgx, orgy, orgz; // viewer's position

// Setup OpenGL enviroment
void SetupGL()
{
   glClearColor(0.0f, 0.0f, 0.0f, 1.0f);
   glClearDepth(1.0f);

   glEnable(GL_DEPTH_TEST);
   glDepthFunc(GL_LEQUAL);

#ifndef LINES
   // setup lighting and materials
	{
      float ambient[] = {0.0f, 0.0f, 0.0f, 1.0f};
      float position[] = {5.0f, 10.0f, -5.0f, 1.0f};
      float mat_diffuse[] = {0.0f, 0.3f, 0.6f, 1.0f};
      float mat_specular[] = {1.0f, 1.0f, 1.0f, 1.0f};
      float mat_shininess[] = {25.0f};
      
		glEnable(GL_LIGHTING);
		//glEnable(GL_COLOR_MATERIAL);
		glEnable(GL_LIGHT0);
      glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);

		glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, 1);
		glShadeModel(GL_SMOOTH);
		glLightfv(GL_LIGHT0, GL_POSITION, position);

      glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
      glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
      glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
      glMaterialfv(GL_BACK, GL_DIFFUSE, mat_diffuse);
      glMaterialfv(GL_BACK, GL_SPECULAR, mat_specular);
      glMaterialfv(GL_BACK, GL_SHININESS, mat_shininess);
	}
#endif

   // set FOV
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   gluPerspective(90.0f, 640.0f/480.0f, 0.01f, 4096.0f);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}

// Setup matrixes to cam. position
void MoveToCamera()
{
   glLoadIdentity();
   glRotatef(-pitch, 1.0f, 0.0f, 0.0f);
   glRotatef(-yaw, 0.0f, 1.0f, 0.0f);
   glRotatef(-roll, 0.0f, 0.0f, 1.0f);
   glTranslatef(-orgx, -orgy, -orgz);
}

// Calculates a normal given 3 points and normalizes it
void CalcNormal(vec3d p1, vec3d p2, vec3d p3, vec3d normal)
{
	vec3d v1, v2, temp;
	float div;

	v1[0] = p1[0] - p2[0];
	v1[1] = p1[1] - p2[1];
	v1[2] = p1[2] - p2[2];

	v2[0] = p3[0] - p2[0];
	v2[1] = p3[1] - p2[1];
	v2[2] = p3[2] - p2[2];

	temp[0] = v1[1]*v2[2] - v1[2]*v2[1];
	temp[1] = -(v1[0]*v2[2] - v1[2]*v2[0]);
	temp[2] = v1[0]*v2[1] - v1[1]*v2[0];

	div = (float)sqrt(temp[0]*temp[0] + temp[1]*temp[1] + temp[2]*temp[2]);
	if(div != 0)
	{
		temp[0] /= div;
		temp[1] /= div;
		temp[2] /= div;
	}
	else
	{
		temp[0] = 0.0f;
		temp[1] = 0.0f;
		temp[2] = 0.0f;
	}

	normal[0] = temp[0];
	normal[1] = temp[1];
	normal[2] = temp[2];
}

void CalcNormals()
{
	int x, y;

	for(y = 0; y < BEZIER_RES_Y; y++)
	{
		for(x = 0; x <= BEZIER_RES_X; x++)
		{
			CalcNormal(bez.verts[y][x], bez.verts[y][x+1], bez.verts[y+1][x],
				normals[y][x][0]);
			CalcNormal(bez.verts[y][x+1], bez.verts[y+1][x+1], bez.verts[y+1][x],
				normals[y][x][1]);
		}
	}
}

// Draw the scene
void Display(void)
{
   int x, y;
   vec3d normal;
   
   glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
   glPushMatrix();

   MoveToCamera();

   // Draw the curve
#ifdef LINES
   // Use a wireframe
   glColor3f(0.0f, 0.5f, 1.0f);
   glBegin(GL_LINES);
      for(y = 0; y < BEZIER_RES_Y; y++)
      {
         for(x = 0; x <= BEZIER_RES_X; x++)
         {
            glVertex3fv(bez.verts[y][x]);
            glVertex3fv(bez.verts[y][x+1]);
            glVertex3fv(bez.verts[y][x]);
            glVertex3fv(bez.verts[y+1][x]);
         }
         glVertex3fv(bez.verts[y][x]);
         glVertex3fv(bez.verts[y+1][x]);
      }
      for(x = 0; x < BEZIER_RES_X; x++)
      {
         glVertex3fv(bez.verts[BEZIER_RES_Y][x]);
         glVertex3fv(bez.verts[BEZIER_RES_Y][x+1]);
      }
   glEnd();
#else
   // or have it solid. Noraml calculation could be better though
   for(y = 0; y < BEZIER_RES_Y; y++)
   {
      glBegin(GL_TRIANGLE_STRIP); // also works w/ a triangle strip
      for(x = 0; x <= BEZIER_RES_X; x++)
      {
         glNormal3fv(normals[y][x][0]);
         glVertex3fv(bez.verts[y][x]);
         glNormal3fv(normals[y][x][1]);
         glVertex3fv(bez.verts[y+1][x]);
      }
      glEnd();
   }
#endif
   
   glDisable(GL_LIGHTING);
   glPointSize(3.0f);
   glBegin(GL_POINTS);
   for(y = 0; y < 9; y++)
   {
	   float c = (float)y/8.0f;
	   glColor3f(c, 1.0f - c, 0.0f);
	   glVertex3fv(bez.ctrl_pts[y]);
   }
   glEnd();
   glEnable(GL_LIGHTING);

   glPopMatrix();
   glutSwapBuffers();
   glutPostRedisplay();
}

// degrees to radian
#define DEG2RAD(a) (3.14159265359f*a/180.0f)
void Key(unsigned char key, int x, int y)
{
   if(key == 27) // exit
      exit(0);
	else if(key == 'j') // turn left
	{
		yaw += 5;
		if(yaw >= 360)
			yaw -= 360.0f;
		else if(yaw < 0)
			yaw += 360.0f;
	}
	else if(key == 'l') // turn right
	{
		yaw += -5;
		if(yaw >= 360)
			yaw -= 360.0f;
		else if(yaw < 0)
			yaw += 360.0f;
	}
	else if(key == 'a') // look up
	{
		pitch += 5;
		if(pitch >= 360)
			pitch -= 360.0f;
		else if(pitch < 0)
			pitch += 360.0f;
	}
	else if(key == 'z') // look down
	{
		pitch += -5;
		if(pitch >= 360)
			pitch -= 360.0f;
		else if(pitch < 0)
			pitch += 360.0f;
	}
	else if(key == 'i') // move forward
	{
		orgx += (float)sin(DEG2RAD(yaw)) * -5.0f;
		orgz += (float)cos(DEG2RAD(yaw)) * -5.0f;
	}
	else if(key == 'k') // move backward
	{
		orgx += (float)sin(DEG2RAD(yaw)) * 5.0f;
		orgz += (float)cos(DEG2RAD(yaw)) * 5.0f;
	}
	else if(key == 'u') // move down
		orgy += -5;
	else if(key == 'o') // move up
		orgy += 5;

   glutPostRedisplay();
}

void Reshape(int w, int h)
{
   glViewport(0, 0, w, h);
}

int main(int argc, char *argv[])
{
   FILE *f = stdin; // where the points are coming from
   int i;

   // see if the file is from the cmd line
   if(argc > 1)
   {
      for(i = 0; i < argc-1; i++)
      {
         if(!strcmp(argv[i], "-points"))
         {
            f = fopen(argv[i+1], "rt");

            // error opening file, set f back to stdin
            if(!f)
            {
               printf("Error opening \"%s\", using STDIN\n", argv[i+1]);
               f = stdin;
            }
         }
      }
   }

   // read in the verts
   for(i = 0; i < 9; i++)
   {
      if(f == stdin) // prompt user
         printf("Cords for pt %i: ", i);
      fscanf(f, "%f %f %f", &bez.ctrl_pts[i][0], &bez.ctrl_pts[i][1],
            &bez.ctrl_pts[i][2]);
   }

   // close f if !stdin
   if(f != stdin)
      fclose(f);

   // calculate the curve
   BezierCalc(&bez);
	CalcNormals();

   // Open up a window
   glutInitWindowSize(640, 480);
   glutInit(&argc, argv);
   glutInitDisplayMode(GLUT_RGB|GLUT_DOUBLE|GLUT_DEPTH);
   glutCreateWindow("Bezier Surfaces by Kilbert <kilbert@inside3d.com>");

   // Setup OpenGL
   SetupGL();

   // Tell GLUT about our handlers
   glutDisplayFunc(Display);
   glutKeyboardFunc(Key);
   glutReshapeFunc(Reshape);

   // Start GLUT's main loop
   glutMainLoop();
   
   return 0;
}
