/* vid_win.cpp - Video and input driver for Windows
 */
#include <windows.h>
#include <GL/gl.h>
#include "resource.h"
#include "system.h"
#include "util.h"
#include "console.h"
#include "concmd.h"
#include "convar.h"
#include "engine.h"
#include "video.h"

extern HINSTANCE sys_instance;

Variable<int> vid_width("vid_width", 640, true), vid_height("vid_height", 480, true);
Variable<int> vid_bpp("vid_bpp", 16, true), vid_zdepth("vid_zdepth", 16, true);
Variable<int> vid_stencil("vid_stencil", 0, true);
Variable<bool> vid_fullscreen("vid_fullscreen", false, true);
ReadOnlyVariable<std::string> gl_vendor("gl_vendor", "");
ReadOnlyVariable<std::string> gl_renderer("gl_renderer", "");
ReadOnlyVariable<std::string> gl_extensions("gl_extensions", "");
ReadOnlyVariable<std::string> gl_version("gl_version", "");

bool vid_initted = false;
HWND vid_window = NULL;
HDC vid_dc = NULL;
HGLRC vid_gl_rc;

bool Vid_OpenWindow();
bool Vid_CloseWindow();

COMMAND(vid_restart)
{
	if(!vid_initted)
	{
		console << "Video not initted..." << std::endl;
		return;
	}

	console << "Restarting video...";
	Vid_CloseWindow();
	if(!Vid_OpenWindow())
	{
		vid_width.reset();
		vid_height.reset();
		vid_bpp.reset();
		vid_zdepth.reset();
		vid_stencil.reset();
		vid_fullscreen.reset();
		if(!Vid_OpenWindow())
		{
			Sys_Hide(false);
			console << "unable to open video mode" << std::endl;
		}
	}

	console << "mode set" << std::endl;
}

bool Vid_SetupPixelFormat()
{
	PIXELFORMATDESCRIPTOR pfd;

	memset(&pfd, 0, sizeof(pfd));
	pfd.nSize         = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion      = 1;
	pfd.dwFlags       = PFD_SUPPORT_OPENGL | PFD_DRAW_TO_WINDOW | PFD_DOUBLEBUFFER;
	pfd.dwLayerMask   = PFD_MAIN_PLANE;
	pfd.iPixelType    = PFD_TYPE_RGBA;
	pfd.cColorBits    = *vid_bpp; //bpp;
	pfd.cDepthBits    = *vid_zdepth; //zbuffer;
	pfd.cAccumBits    = 0;
	pfd.cStencilBits  = *vid_stencil; //stencil;

	int pixel_format = ChoosePixelFormat(vid_dc, &pfd);
	if(!pixel_format)
	{
		console << "ChoosePixelFormat failed" << std::endl;
		return false;
	}

	memset(&pfd, 0, sizeof(pfd));
	DescribePixelFormat(vid_dc, pixel_format, sizeof(PIXELFORMATDESCRIPTOR), &pfd);

	SetPixelFormat(vid_dc, pixel_format, &pfd);
	return true;
}

int CALLBACK Vid_WndProc(HWND wnd, unsigned int msg, WPARAM wparam, LPARAM lparam)
{
	if(msg == WM_CREATE)
	{
		vid_dc = GetDC(wnd);
		if(!Vid_SetupPixelFormat())
			Vid_Close();

		vid_gl_rc = wglCreateContext(vid_dc);
		wglMakeCurrent(vid_dc, vid_gl_rc);
		SetCursor(NULL);
		ShowCursor(FALSE);
	}
	else if(msg == WM_CLOSE)
	{
		if(vid_gl_rc)
			wglDeleteContext(vid_gl_rc);
		if(vid_dc)
			ReleaseDC(wnd, vid_dc);
		vid_gl_rc = NULL;
		vid_dc = NULL;
	}
	else if(msg == WM_DESTROY)
	{
		if(vid_gl_rc)
			wglDeleteContext(vid_gl_rc);
		if(vid_dc)
			ReleaseDC(wnd, vid_dc);
		vid_gl_rc = NULL;
		vid_dc = NULL;
		vid_window = NULL; // window has already been destroyed
		Vid_CloseWindow();
	}
	else if(msg == WM_KEYDOWN)
	{
		if(wparam == VK_RETURN)
		{
			Sys_Hide(!Sys_IsHidden());
			if(!Sys_IsHidden())
				ShowCursor(true);
			else
				ShowCursor(false);
		}
		else if(wparam == VK_ESCAPE)
			Vid_Close();
		return true;
	}
	else if(msg == WM_PAINT)
	{
		PAINTSTRUCT paint;
		HDC dc = BeginPaint(wnd, &paint);
		EndPaint(wnd, &paint);
		return true;
	}

	return DefWindowProc(wnd, msg, wparam, lparam);
};

bool Vid_ChangeRes()
{
	DEVMODE mode;

	for(int i = 0; EnumDisplaySettings(NULL, i, &mode); i++)
	{
		if(mode.dmPelsWidth == *vid_width && mode.dmPelsHeight == *vid_height &&
			mode.dmBitsPerPel == *vid_bpp)
		{
			if(ChangeDisplaySettings(&mode, CDS_FULLSCREEN) == DISP_CHANGE_SUCCESSFUL)
				return true;
		}
	}
	return false;
}

bool Vid_OpenWindow()
{
	if(vid_window)
		Vid_CloseWindow();

	WNDCLASS wnd_class;
	wnd_class.style = 0;
	wnd_class.lpfnWndProc = (WNDPROC)Vid_WndProc;
	wnd_class.cbClsExtra = 0;
	wnd_class.cbWndExtra = 0;
	wnd_class.hInstance = sys_instance;
	wnd_class.hIcon = LoadIcon(sys_instance, MAKEINTRESOURCE(IDI_ICON));
	wnd_class.hCursor = NULL;
	wnd_class.hbrBackground = (HBRUSH)COLOR_BACKGROUND;
	wnd_class.lpszMenuName = NULL;
	wnd_class.lpszClassName = eng_name->c_str();

	if(!RegisterClass(&wnd_class))
		return false;

	if(*vid_fullscreen)
	{
		if(Vid_ChangeRes())
			vid_window = CreateWindow(eng_name->c_str(), eng_name->c_str(), WS_POPUP, 0, 0, *vid_width, *vid_height, NULL, NULL, sys_instance, NULL);
		else
			console << "Error changing resolution. " << std::cout;
	}
	else
	{
		int w = GetSystemMetrics(SM_CXDLGFRAME)*2 + *vid_width;
		int h = GetSystemMetrics(SM_CYCAPTION) + GetSystemMetrics(SM_CXDLGFRAME)*2 + *vid_height;

		vid_window = CreateWindow(eng_name->c_str(), eng_name->c_str(),
			 WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,
			CW_USEDEFAULT, CW_USEDEFAULT,
			w, h, NULL, NULL, sys_instance, NULL);
	}

	if(!vid_window)
	{
		console << "Error creating window" << std::endl;
		return false;
	}

	ShowWindow(vid_window, SW_SHOW);
	UpdateWindow(vid_window);

	Sys_Hide();

	SetForegroundWindow(vid_window);
	SetFocus(vid_window);

	return true;
}

bool Vid_CloseWindow()
{
	if(vid_initted)
	{
		if(vid_window)
		{
			DestroyWindow(vid_window);
			vid_window = NULL;
			UnregisterClass(eng_name->c_str(), sys_instance);
		}
		ChangeDisplaySettings(NULL, 0);
		ShowCursor(true);
		Sys_Hide(false);
		return true;
	}
	else
		return false;
}

void Vid_Swap()
{
	if(vid_initted && vid_dc)
		SwapBuffers(vid_dc);
}

void Vid_Update()
{
	// system takes care of this in windows
}

bool Vid_Initted()
{
	return vid_initted;
}

bool Vid_Init()
{
	if(Vid_OpenWindow())
	{
		(*gl_renderer) = (const char *)glGetString(GL_RENDERER);
		(*gl_vendor) = (const char *)glGetString(GL_VENDOR);
		(*gl_extensions) = (const char *)glGetString(GL_EXTENSIONS);
		(*gl_version) = (const char *)glGetString(GL_VERSION);
		vid_initted = true;
		return true;
	}
	else
		return false;
}

bool Vid_Close()
{
	if(vid_initted)
	{
		Vid_CloseWindow();
		(*gl_renderer) = "";
		(*gl_vendor) = "";
		(*gl_extensions) = "";
		(*gl_version) = "";
		vid_initted = false;
		return true;
	}
	else
		return false;
}