/* vfs.cpp - Virtual File System Implmentation
 */
#include <iostream>
#include <fstream>
#ifdef _WIN32
#include <streambuf>
#else
#include <streambuf.h>
#endif
#include <string>
#include <stdio.h>
#include "system.h"
#include "console.h"
#include "concmd.h"
#include "convar.h"
#include "util.h"
#include "vfs.h"

using std::streambuf;
using std::string;

ConstVariable<string> basepath("basepath", "./data", false);
ConstVariable<string> gamepath("gamepath", "./data", false);
bool vfs_initted = false;

void vfs_istream::open(const char *path, std::ios::openmode mode)
{
	string s(gamepath);
	s += "/";
	s += path;
	m_buf.open(s.c_str(), mode);
	if(!m_buf.is_open())
	{
		s = basepath;
		s += "/";
		s += path;
		m_buf.open(s.c_str(), mode);
	}
}

void vfs_ostream::open(const char *path, std::ios::openmode mode)
{
	string s(gamepath);
	s += "/";
	s += path;
	m_buf.open(s.c_str(), mode);
	if(!m_buf.is_open())
	{
		s = basepath;
		s += "/";
		s += path;
		m_buf.open(s.c_str(), mode);
	}
}

void vfs_stream::open(const char *path, std::ios::openmode mode)
{
	string s(gamepath);
	s += "/";
	s += path;
	m_buf.open(s.c_str(), mode);
	if(!m_buf.is_open())
	{
		s = basepath;
		s += "/";
		s += path;
		m_buf.open(s.c_str(), mode);
	}
}

// ABC that implements the interface to the various
// archive formats(file directory, zip file, etc.)
class VFS_BaseArchive
{
public:
	VFS_BaseArchive(const char *description)
		: m_description(NULL), m_next(NULL)
	{
		if(!description)
			return;

		m_description = Util_StrCopy(description);
		VFS_BaseArchive *temp = archives();

		while(temp)
		{
			if(!temp->m_next)
				break;
			else
				temp = temp->m_next;
		}
		temp->m_next = this;
	}

	virtual ~VFS_BaseArchive()
	{
		VFS_BaseArchive *temp = archives();
		while(temp)
		{
			if(temp->next() == m_next)
			{
				temp->m_next = m_next;
				break;
			}
			else
				temp = temp->next();
		}

		delete[] m_description;
	}

	const char *getDescription() { return m_description; }
	VFS_BaseArchive *next() { return m_next; }

	virtual bool fileExists(const char *path) { return false; }
	virtual streambuf *open(const char *path, std::ios::open_mode) { return NULL; }
	virtual bool close(streambuf *str) { return false; }

	static VFS_BaseArchive &findArchive(const char *path)
	{
		VFS_BaseArchive *temp = archives();
		while(temp)
		{
			if(temp->fileExists(path))
				return *temp;
			else
				temp = temp->next();
		}

		return *archives();
	}

	static VFS_BaseArchive *archives()
	{
		static VFS_BaseArchive root(NULL);
		return &root;
	}

private:
	char *m_description;
	VFS_BaseArchive *m_next;
};

class VFS_FileArchive: public VFS_BaseArchive
{
public:
	VFS_FileArchive(const char *base_path)
		: VFS_BaseArchive("provides access to normal files"),
		  m_base_path(Util_StrCopy(base_path)) { }
	~VFS_FileArchive() { delete[] m_base_path; }

	bool fileExists(const char *path)
	{
		char *temp = new char[strlen(m_base_path)+strlen(path)+2];
		sprintf(temp, "%s/%s", m_base_path, path);
		std::ifstream f(temp);
		delete[] temp;
		if(f)
			return true;
		else
			return false;
	}

	streambuf *open(const char *path, std::ios::open_mode mode)
	{
		char *temp = new char[strlen(m_base_path)+strlen(path)+2];
		sprintf(temp, "%s/%s", m_base_path, path);
		std::filebuf *ret = new std::filebuf;
		ret->open(temp, mode);
		delete[] temp;
		return ret;
	}

	bool close(streambuf *str)
	{
		delete str; // check in an array to see if this archive did open that file
		return true;
	}

private:
	char *m_base_path; // check for an ending '/' and remove if it is there
};

VFS_FileArchive vfs_file_archive(basepath->c_str());

bool VFS_Init()
{
	console << "File System Init:" << std::endl;
	VFS_AddPath(basepath->c_str());

	console << "   archive formats:" << std::endl;
	VFS_BaseArchive *root = VFS_BaseArchive::archives();
	while(root)
	{
		if(root->getDescription())
			console << "      " << root->getDescription() << std::endl;
		root = root->next();
	}

	return false;
}

bool VFS_Close()
{
	console << "Closing file system..." << std::endl;
	return false;
}

bool VFS_AddPath(const char *path)
{
	if(vfs_initted)
		console << "VFS: added \"" << path << "\"" << std::endl;

	return false;
}
