/* vector.h - Vector class
 */
#ifndef _VECTOR_H_

#include <math.h>

typedef float scalar_t;
extern const scalar_t epsilon;

class Vector3d
{
public:
	Vector3d(scalar_t x = 0.0, scalar_t y = 0.0, scalar_t z = 0.0);
	Vector3d(const scalar_t *array);
	Vector3d(const Vector3d &v);

	scalar_t *get() { return m_data; }
	const scalar_t *get() const { return m_data; }

	void set(scalar_t x = 0.0, scalar_t y = 0.0, scalar_t z = 0.0);
	void set(const scalar_t *array);
	void set(const Vector3d &v);

	scalar_t magnitude() const;
	scalar_t normalize();

	Vector3d add(const Vector3d &v) const;
	Vector3d sub(const Vector3d &v) const;
	Vector3d mul(const Vector3d &v) const;
	Vector3d div(const Vector3d &v) const;
	scalar_t dot(const Vector3d &v) const;
	Vector3d cross(const Vector3d &v) const;

	Vector3d mul(const scalar_t &s) const;
	Vector3d div(const scalar_t &s) const;

	Vector3d &operator=(const Vector3d &v) { set(v); return *this; }
	Vector3d &operator=(const scalar_t *array) { set(array); return *this; }

	Vector3d operator+(const Vector3d &v) const
		{ return add(v); }
	Vector3d operator-(const Vector3d &v) const
		{ return sub(v); }
	Vector3d operator*(const Vector3d &v) const
		{ return mul(v); }
	Vector3d operator/(const Vector3d &v) const
		{ return div(v); }
	Vector3d operator%(const Vector3d &v) const // cross product
		{ return cross(v); }

	Vector3d operator*(const scalar_t s) const
		{ return mul(s); }
	Vector3d operator/(const scalar_t s) const
		{ return div(s); }

	Vector3d &operator+=(const Vector3d &v)
		{ *this = *this + v; return *this; }
	Vector3d &operator-=(const Vector3d &v)
		{ *this = *this - v; return *this; }
	Vector3d &operator*=(const Vector3d &v)
		{ *this = *this * v; return *this; }
	Vector3d &operator/=(const Vector3d &v)
		{ *this = *this / v; return *this; }
	Vector3d &operator%=(const Vector3d &v)
		{ *this = *this % v; return *this; }

	Vector3d &operator*=(const scalar_t &s)
		{ *this = *this * s; return *this; }
	Vector3d &operator/=(const scalar_t &s)
		{ *this = *this / s; return *this; }

	scalar_t &operator[](int i)
		{ if(i >= 0 && i < 3) return m_data[i]; else return m_data[i % 3]; }

	scalar_t operator[](int i) const
		{ if(i >= 0 && i < 3) return m_data[i]; else return m_data[i % 3]; }

	bool operator==(const Vector3d &v) const
	{
		for(int i = 0; i < 3; i++)
		{
			if(fabs(m_data[i]-v[i]) <= epsilon)
				return false;
		}
		return true;
	}

	bool operator!=(const Vector3d &v) const { return !(*this == v); }
	bool operator>(const Vector3d &v) const
		{ if(magnitude() > v.magnitude()) return true; else return false; }
	bool operator<(const Vector3d &v) const
		{ if(magnitude() < v.magnitude()) return true; else return false; }
	bool operator>=(const Vector3d &v) const
		{ if(magnitude() >= v.magnitude()) return true; else return false; }
	bool operator<=(const Vector3d &v) const
		{ if(magnitude() <= v.magnitude()) return true; else return false; }

private:
	scalar_t m_data[4]; // makes sizeof(*this) == 16, should help alignment?
};

inline Vector3d operator-(const Vector3d &v)
{ return Vector3d(-v[0], -v[1], -v[2]); }
inline Vector3d operator+(const Vector3d &v)
{ return v; }

inline Vector3d operator+(const scalar_t s, const Vector3d &v)
{ return Vector3d(s+v[0], s+v[1], s+v[2]); }
inline Vector3d operator-(const scalar_t s, const Vector3d &v)
{ return Vector3d(s-v[0], s-v[1], s-v[2]); }
inline Vector3d operator*(const scalar_t s, const Vector3d &v)
{ return Vector3d(s*v[0], s*v[1], s*v[2]); }

#endif