/* timer.cpp - A stopwatch type class that uses seconds
 */

/* need to customize for each OS, so a higher resolution then 1 second
 * can be achieved
 */
#if defined(_WIN32)
#include <windows.h>
#include <mmsystem.h>
#elif defined(LINUX)
#include <sys/time.h>
#else
#include <time.h>
#endif
#include "timer.h"

Timer::Timer(bool start_now /* = false */)
   : m_stopped(true), m_systime(0.0), m_time(0.0)
{
	start();
   if(!start_now)
      stop();
}

void Timer::start()
{
   m_systime = sys_time();
   m_stopped = false;
}

double Timer::stop()
{
   double cur_time = sys_time();
   m_time += cur_time - m_systime;
   m_systime = cur_time;
   m_stopped = true;
   return m_time;
}

double Timer::reset()
{
   double ret = time();
   m_systime = m_time = 0.0;
   return ret;
}

double Timer::time()
{
   double cur_time = sys_time();
   if(!m_stopped)
      m_time += cur_time - m_systime;
   m_systime = cur_time;
   return m_time;
}

double Timer::sys_time()
{
#if defined(_WIN32)
	double time = (double)timeGetTime()/1000.0;
	if(m_systime > time) // timeGetTime wrapped around
		return m_systime+time; // handle it better
	else
		return time;
#elif defined(LINUX)
   struct timeval tp;
   struct timezone tzp;
   gettimeofday(&tp, &tzp);
   return ((double)tp.tv_sec + (double)tp.tv_usec/1000000.0);
#else
	return ((double)clock()/(double)CLOCKS_PER_SEC);
#endif
}
