/* net.cpp - Nework main file */
#include <winsock2.h>
#include "net.h"
#include "util.h"
#include "console.h"
#include "concmd.h"

Net_UDPSocket::Packet::Packet()
	: m_hostname(NULL), m_data(NULL), m_port(0), m_len(0)
{
}

Net_UDPSocket::Packet::Packet(const char *dest_name, int dest_port, const char *data, int len)
	: m_hostname(NULL), m_data(NULL), m_port(0), m_len(0)
{
	setData(data, len);
	setHostname(dest_name);
	setPort(dest_port);
}

Net_UDPSocket::Packet::Packet(Packet &msg)
	: m_hostname(NULL), m_data(NULL), m_port(0), m_len(0)
{
	set(msg);
}

Net_UDPSocket::Packet::~Packet()
{
	delete[] m_data;
	delete[] m_hostname;
}

bool Net_UDPSocket::Packet::set(Packet &msg)
{
	if(!setData(msg.getData(), msg.getLength()))
		return false;
	setHostname(msg.getHostname());
	setPort(msg.getPort());
	return true;
}

bool Net_UDPSocket::Packet::setData(const char *data, int len)
{
	delete[] m_data;
	m_data = new char[len];
	memcpy(m_data, data, len);
	m_len = len;
	return true;
}

bool Net_UDPSocket::Packet::setLength(int len)
{
	if(len == m_len)
		return true;

	char *temp = new char[len];
	if(m_len < len)
		memcpy(temp, m_data, m_len);
	else
		memcpy(temp, m_data, len);

	delete[] m_data;
	m_data = temp;
	m_len = len;
	return true;
}

void Net_UDPSocket::Packet::setHostname(const char *name)
{
	delete[] m_hostname;
	m_hostname = Util_StrCopy(name);
}

void Net_UDPSocket::Packet::setPort(int port)
{
	m_port = port;
}

Net_UDPSocket::Net_UDPSocket()
	: m_open(false), m_port(0)
{
}

Net_UDPSocket::Net_UDPSocket(int port)
	: m_open(false), m_port(0)
{
	open(port);
}

Net_UDPSocket::~Net_UDPSocket()
{
	close();
}

bool Net_UDPSocket::open(int port)
{
	close();

	m_socket = socket(PF_INET, SOCK_DGRAM, 0);
	if(m_socket == -1)
		return false;
	else
		m_open = true;

	sockaddr_in local_name;
	local_name.sin_family = AF_INET;
	local_name.sin_port = port;
	local_name.sin_addr.s_addr = htonl(INADDR_ANY);
	if(bind(m_socket, (sockaddr *)&local_name, sizeof(local_name)) < 0)
	{
		close();
		return false;
	}
	return true;
}

bool Net_UDPSocket::close()
{
	if(m_open)
	{
		closesocket(m_socket);
		return true;
	}
	else
		return false;
}

bool Net_UDPSocket::msgPending()
{
	if(!m_open)
		return false;
	else
	{
		fd_set fdset;
		FD_ZERO(&fdset);
		FD_SET(m_socket, &fdset);
		timeval timeout = { 0, 0 }; // no time out
		if(select(1, &fdset, NULL, NULL, &timeout) == INVALID_SOCKET)
			return false;

		if(FD_ISSET(m_socket, &fdset))
			return true;
		else
			return false;
	}
}

bool Net_UDPSocket::send(Net_UDPSocket::Packet &msg)
{
	if(!m_open)
		return false;

	sockaddr_in dest;
	dest.sin_family = AF_INET;
	dest.sin_port = msg.getPort();

	hostent *hostinfo = gethostbyname(msg.getHostname());
	if(!hostinfo)
		return false;

	dest.sin_addr = *(in_addr *)hostinfo->h_addr;

	if(sendto(m_socket, msg.getData(), msg.getLength(), 0, (sockaddr *)&dest, sizeof(dest)) == SOCKET_ERROR)
		return false;
	else
		return true;
}

bool Net_UDPSocket::receive(Net_UDPSocket::Packet &msg)
{
	if(!m_open)
		return false;

	unsigned long data_len = 0;
	ioctlsocket(m_socket, FIONREAD, &data_len);
	msg.setLength(data_len);

	sockaddr_in addr;
	int addr_len = sizeof(addr);

	if(recvfrom(m_socket, msg.getData(), msg.getLength(), 0, (sockaddr *)&addr, &addr_len) == SOCKET_ERROR)
		return false;

	if(addr.sin_family == AF_INET) // probably best to leave this check
	{
		msg.setPort(addr.sin_port);
		msg.setHostname(inet_ntoa(addr.sin_addr));
	}
	else
	{
		msg.setPort(0);
		msg.setHostname("");
	}

	return true;
}

bool Net_Init()
{
	WSADATA version_info;
	if(WSAStartup(MAKEWORD(2, 2), &version_info) == 0)
	{
		if(LOBYTE(version_info.wVersion) == 2 &&
			HIBYTE(version_info.wVersion) == 2)
		{
			console << "Winsock initialized" << std::endl;
			return true;
		}
	}

	console << "Failed to initialize Winsock" << std::endl;

	return false;
}

Net_UDPSocket *sock = NULL;

bool Net_Close()
{
	console << "Closing Winsock..." << std::endl;
	delete sock;
	WSACleanup();
	return true;
}

COMMAND(n_open)
{
	if(args.count() > 1)
	{
		delete sock;
		sock = new Net_UDPSocket(atoi(args[1]));
		console << "socket open" << std::endl;
	}
}

COMMAND(n_send)
{
	if(args.count() > 3)
	{
		Net_UDPSocket::Packet packet(args[1], atoi(args[2]), args[3], strlen(args[3]) + 1);
		if(sock)
		{
			if(sock->send(packet))
				console << "Sent " << strlen(args[3]) + 1 << " bytes" << std::endl;
			else
				console << "Error sending packet" << std::endl;
		}
	}
}

void Net_PollTestSock()
{
	if(sock && sock->msgPending())
	{
		Net_UDPSocket::Packet packet;
		if(sock->receive(packet))
		{
			console << "Packet from " << packet.getHostname() << ":" << packet.getPort() << std::endl;
			console << packet.getLength() << " bytes received" << std::endl;
			console << packet.getData() << std::endl;
		}
		else
			console << "Error receiving(" << WSAGetLastError() << ")" << std::endl;
	}
}