/* convar.h - Console variable header
 */
#ifndef _CONVAR_H_
#define _CONVAR_H_

#include "console.h"
#include "concmd.h"

/* Variable types needed:
 *		Normal variable(set in code and on console)
 *		Readonly variable(set in code and from command line)
 *		Function variable(executes a function when set)
 *		User variable(normal variable, managed by user, inaccessable to code)
 * All variable classes should overide the operators that will
 * make its data accesible like a pointer.
 */
class BaseVariable: public Function // manages all the variable types
{
public:
	BaseVariable(const char *name, const char *type, bool write_out = false);
	~BaseVariable();

	bool writeOut() { return m_write_out; }
	virtual std::ostream &write(std::ostream &str) { return str; } // avoids problems with templates later on

	BaseVariable *nextVariable() { return m_next_var; }

	virtual void get(std::ostream &str) { }
	virtual bool set(const char *value) { return false; }
	virtual void reset() { }

	static BaseVariable *variables();
	static bool setVariable(const char *name, const char *value);
	static bool resetVariable(const char *name);
	static BaseVariable *findVariable(const char *name);
	static bool writeOutVars(const char *path);

private:
	bool m_write_out;
	BaseVariable *m_next_var;
};

// Normal variable
template<class T> class Variable: public BaseVariable
{
public:
	typedef T data_type;

	Variable(const char *name, const data_type &value, bool write_out = true)
		: BaseVariable(name, "Variable", write_out), m_value(value), m_def_value(value)
	{
		// nothing
	}

	std::ostream &write(std::ostream &str) { return str << "set " << getName() << " \"" << m_value << "\"" << std::endl; }

	virtual void run(Arguments &args)
	{
		if(args.count() > 1)
			set(args[1]);
		else
			console << args[0] << ": " << getValue() << "    default: " << getDefValue() << std::endl;
	}

	virtual void get(std::ostream &str)
	{
		str << m_value;
	}

	virtual bool set(const char *value)
	{
		if(!value)
			return false;

		// FIXME: using strstream prevents values from having any whitespace in them
		std::istrstream str(value);
		data_type temp;
		str >> temp;
		return set(temp);
	}

	virtual bool set(const data_type &value)
	{
		setValue(value);
		return true;
	}

	virtual void reset()
	{
		setValue(getDefValue());
	}

	operator data_type() { return m_value; }
	data_type *operator->() { return &m_value; }
	data_type &operator*() { return m_value; }
	data_type &operator=(const data_type &data) { return m_value = data; }

	operator const data_type() const { return m_value; }
	const data_type *operator->() const { return &m_value; }
	const data_type &operator*() const { return m_value; }

protected:
	void setValue(const data_type &value)
	{
		m_value = value;
	}
	data_type &getValue() { return m_value; }
	data_type &getDefValue() { return m_def_value; }

private:
	data_type m_value, m_def_value;
};

// Special type of variable that can't be set from the console
template<class T> class ConstVariable: public Variable<T>
{
public:
	ConstVariable(const char *name, const data_type &value, bool write_out = true)
		: Variable<T>(name, value, write_out)
	{
		// nothing
	}

	std::ostream &write(std::ostream &str) { return str << "set " << getName() << " \"" << getValue() << "\"" << std::endl; }

	void run(Arguments &args)
	{
		if(args.count() > 1)
		{
			if(!console.initted())
				set(args[1]);
			else
				console << args[0] << " is readonly" << std::endl;
		}
		else
			console << args[0] << ": " << getValue() << "    default: " << getDefValue() << std::endl;
	}

	void get(std::ostream &str) { str << getValue(); }

	bool set(const char *value)
	{
		if(!value || console.initted())
			return false;
		else
		{
			std::istrstream str(value);
			data_type temp;
			str >> temp;
			return set(temp);
		}
	}

	bool set(const data_type &value)
	{
		setValue(value);
		return true;
	}

	void reset()
	{
		// no reset for this variable
	}
};

template<class T> class ReadOnlyVariable: public BaseVariable
{
public:
	typedef T data_type;

	ReadOnlyVariable(const char *name, const data_type &value)
		: BaseVariable(name, "ReadOnlyVariable", false), m_value(value) { }

	void run(Arguments &args)
	{
		if(args.count() > 1)
			console << args[0] << " is readonly" << std::endl;
		else
			console << args[0] << ": " << m_value << std::endl;
	}

	void get(std::ostream &str) { str << m_value; }

	bool set(const char *value)
	{
		return false;
	}

	bool set(const data_type &value)
	{
		return false;
	}

	void reset()
	{
		// no reset for this variable
	}

	operator data_type() { return m_value; }
	data_type *operator->() { return &m_value; }
	data_type &operator*() { return m_value; }
	data_type &operator=(const data_type &data) { return m_value = data; }

	operator const data_type() const { return m_value; }
	const data_type *operator->() const { return &m_value; }
	const data_type &operator*() const { return m_value; }

private:
	data_type m_value;
};

template<class T> class FuncVariable: public Variable<T>
{
public:
	typedef void (*function)();

	FuncVariable(const char *name, const data_type &value, function func, bool write_out)
		: Variable<T>(name, value, write_out), m_func(func) { }
	~FuncVariable() { }

	virtual void run(Arguments &args)
	{
		if(args.count() > 1)
			set(args[1]);
		else
			console << args[0] << ": " << getValue() << "    default: " << getDefValue() << std::endl;
	}

	virtual bool set(const char *value)
	{
		if(!value)
			return false;

		std::istrstream str(value);
		data_type temp;
		str >> temp;
		return set(temp);
	}

	virtual bool set(const data_type &value)
	{
		setValue(value);
		m_func();
		return true;
	}

	virtual void reset()
	{
		setValue(getDefValue());
		m_func();
	}

private:
	function m_func;
};

#endif
