/* concmd.cpp - Console command and variable class implementations
 */
#include <string.h>
#include <strstream>
#include "util.h"
#include "tokenizer.h"
#include "console.h"
#include "concmd.h"
#include "convar.h"

/*
 * Arguments class
 */
Arguments::Arguments(const char *str)
	: m_count(0), m_tokens(NULL), m_str(NULL)
{
	m_str = Util_StrCopy(str);
	tokenize();
}

Arguments::Arguments(int c, char **args)
	: m_count(c), m_str(NULL)
{
	m_tokens = new char*[m_count];
	for(int i = 0; i < m_count; i++)
		m_tokens[i] = Util_StrCopy(args[i]);
}

Arguments::~Arguments()
{
	for(int i = m_count; i > 0; i--)
		delete[] m_tokens[i-1];
	delete[] m_tokens;
	delete[] m_str;
}

const char *Arguments::arg(int i)
{
	if(i >= 0 && i < m_count)
		return m_tokens[i];
	else
		return NULL;
}

bool Arguments::arg(const char *a)
{
	for(int i = 0; i < m_count; i++)
	{
#ifdef _WIN32
		if(!stricmp(a, m_tokens[i]))
#else
		if(!strcasecmp(a, m_tokens[i]))
#endif
			return true;
	}
	return false;
}

bool Arguments::tokenize()
{
	if(!m_str)
		return false;

	// split m_str into tokens
	SimpleTokenizer t(m_str);
	for(m_count = 0; t.next(); m_count++);
	t.reset();

	m_tokens = new char *[m_count];
	for(int i = 0; i < m_count; i++)
	{
		const char *token = t.next();
		if(token[0] == '$') // replace with a variable or skip
		{
			BaseVariable *var = BaseVariable::findVariable(&token[1]);
			if(var)
			{
				std::strstream str;
				var->get(str);
				char temp_tok[256];
				str.getline(temp_tok, 256);
				m_tokens[i] = Util_StrCopy(temp_tok);
			}
			else
			{
				// skip this var, wastes 4 bytes of memory, but who cares
				m_count--;
				i--;
				continue;
			}
		}
		else
			m_tokens[i] = Util_StrCopy(token);
	}
	return true;
}

Function::Function(const char *name, const char *type)
	: m_next(NULL), m_name(NULL), m_type(NULL)
{
	m_type = Util_StrCopy(type);

	if(!name) // empty function, do nothing. usually the root function
		return;
	m_name = Util_StrCopy(name);

	Function *f = findFunction(name);
	if(!f)// can't duplicate functions
	{
		f = root();
		while(f && f->m_next)
			f = f->m_next;
		f->m_next = this;
		m_next = NULL;
	}
	else
		console << "Name collision: " << name << " already exists" << std::endl;
}

Function::~Function()
{
	Function *f = root();
	while(f)
	{
		if(f->m_next == this)
		{
			f->m_next = m_next;
			break;
		}
		else
			f = f->m_next;
	}

	delete[] m_name;
	delete[] m_type;
}

void Function::run(Arguments &args)
{
	// do nothing
}

Function *Function::root()
{
	static Function functions(NULL, "null");
	return &functions;
}

Function *Function::findFunction(const char *name)
{
	Function *f = root();
	while(f)
	{
		if(!f->getName() ||
#ifdef _WIN32
			stricmp(name, f->getName())
#else
			strcasecmp(name, f->getName())
#endif
			)
			f = f->m_next;
		else
			return f;
	}

	return NULL;
}

bool Function::execute(Arguments &args)
{
	Function *f = findFunction(args[0]);
	if(!f)
		return false;
	else
	{
		f->run(args);
		return true;
	}
}

/*
 * Command
 */
Command::Command(const char *name, func_t function)
: Function(name, "Command"), m_function(function)
{
}

Command::~Command()
{
}

void Command::run(Arguments &args)
{
	if(m_function)
		m_function(args);
}

// Commands dealing with commands
COMMAND(ls_cmds)
{
	int count = 0;
	Function *f = Function::root();
	while(f)
	{
		if(!stricmp(f->getType(), "Command") && f->getName())
		{
			console << f->getName() << std::endl;
			count++;
		}
		f = f->next();
	}
	console << count << " total commands" << std::endl;
}