#include "g_local.h"

// some ease-of-use defines
#define LAST_WAYPOINT num_waypoints-1
#define CURRENT_WAYPOINT num_waypoints
#define BOT_SEARCH_TIME 3

void wSetMoveinfo (edict_t *bot, vec3_t pos1, vec3_t pos2, vec3_t pos3)
{
	// set the bot's waypoint definitions
	VectorCopy(pos1, bot->point1);
	VectorCopy(pos2, bot->point2);
	VectorCopy(pos3, bot->point3);
}

// wNearAnotherPoint - checks ALL points to see if we are near one we've
// already planted - to prevent duplication
qboolean wNearAnotherPoint (edict_t *bot)
{
	int points = LAST_WAYPOINT;
	int i;
	vec3_t result;
	float distance = 0;

	if (!G_EntExists(bot))
		return true;

	if (!G_ClientInGame(bot))
		return true; // no points while dead

	// i=num_waypoints, keep i on the list, subtract one each time
	for (i = points ; i >= 0 ; i--)
	{
		distance = 0;
		VectorClear(result);

		// if its visible, to the bots origin and close, we dont need a new one
		if (wVisible(bot->s.origin, waypoints[i]->location))
		{
			VectorSubtract(waypoints[i]->location, bot->s.origin, result);
			distance = VectorLength(result);

			// within acceptable limits - return true
			if (distance < 500)
			{
				#ifdef SHOW_ADV_LOGIC
					gi.dprintf("Near waypoint %d\n", i);
				#endif
				return true;
			}
		}
	}

	return false;
}

// wSpawnPoint - checks distance and spawns if necessary
void wSpawnPoint (edict_t *bot)
{
	if (!G_EntExists(bot))
		return;
	
	if (!G_ClientInGame(bot))
		return;

	if (bot->deadflag)
		return; // no waypoints while dead

	if (num_waypoints == 0)
	{
		#ifdef SHOW_ADV_LOGIC
			gi.dprintf("First waypoint, place without check\n");
		#endif
		wCreateWaypoint(bot->s.origin, bot);
	}
	else
	{
		if (!wNearAnotherPoint(bot))
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Not near another point, place new.\n");
			#endif
			wCreateWaypoint(bot->s.origin, bot);
		}
		else
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Near another point, do not place\n");
			#endif
		}
	}
}

// wWaypointPriority - called to index waypoint's priority, based upon items
int wWaypointPriority (vec3_t location)
{
	edict_t *ent=NULL;
	int priority = 0;
	float radius=1000;
	
	while ((ent=findradius(ent, location, radius)) != NULL) 
	{
		if (!ent || !ent->item) continue;
		// PSY: updated 2-8: use wVisible, lower priority of rooms with unreachable
		// items.
		if (!wVisible(location, ent->s.origin)) continue;
		if (gi.pointcontents(ent->s.origin) & (CONTENTS_LAVA|CONTENTS_SLIME)) continue;

		// add to priority based upon item type
		if (ent->item->flags & IT_HEALTH)
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Health, priority plus 2\n");
			#endif
			priority += 2;
		}
		if (ent->item->flags & IT_ARMOR)
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Armor, priority plus 3\n");
			#endif
			priority += 3;
		}
		if (ent->item->flags & IT_WEAPON)
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Weapon, priority plus 5\n");
			#endif
			priority += 5;
		}
		if (ent->item->flags & IT_AMMO)
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Ammo, priority plus 4\n");
			#endif
			priority += 4;
		}
		if (ent->item->flags & IT_PACK)
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Pack, priority plus 2\n");
			#endif
			priority += 2;
		}
		if (ent->item->flags & IT_POWERUP)
		{
			#ifdef SHOW_ADV_LOGIC
				gi.dprintf("Powerup, priority plus 2\n");
			#endif
			priority += 2;
		}
	}

	return priority;
}

// Thanks WarZone! Prevents memory foul-ups
waypoint_t *GetNextWaypoint (void)
{
	int i = nextwaypoint;

	while ((waypoints[i]->inuse) && (i < MAX_WAYPOINTS))
		i++;
	
	if (waypoints[i]->inuse)
		return NULL;

	nextwaypoint = i+1>MAX_WAYPOINTS?MAX_WAYPOINTS:i+1;

	return waypoints[i];
}

// wCreateWaypoint - creats one waypoint, called OFTEN
void wCreateWaypoint (vec3_t location, edict_t *bot)
{
	if (!G_EntExists(bot))
		return;
	
	if (!G_ClientInGame(bot))
		return;

	if (CURRENT_WAYPOINT == MAX_WAYPOINTS)
	{
		#ifdef SHOW_LOGIC
			gi.dprintf("Max waypoints reached, do not plant\n");
		#endif
		return;
	}

	if (waypoints[CURRENT_WAYPOINT] == NULL)
		return; // out of memory or related f*ck up

	waypoints[CURRENT_WAYPOINT] = GetNextWaypoint();
	
	if (waypoints[CURRENT_WAYPOINT] == NULL)
		return; // test again after GetNextWaypoint()

	// we're good to go, so mark this one as inuse
	waypoints[CURRENT_WAYPOINT]->inuse = true;

	VectorCopy(location, waypoints[CURRENT_WAYPOINT]->location);
	
	waypoints[CURRENT_WAYPOINT]->priority = wWaypointPriority(location);

	num_waypoints++; // add to counter

	#ifdef SHOW_LOGIC
		gi.dprintf("Waypoint %d, at %s, with %d priority\n", LAST_WAYPOINT, 
			vtos(waypoints[LAST_WAYPOINT]->location), 
			waypoints[LAST_WAYPOINT]->priority);
	#endif
}

// StartWaypointSystem - called upon connect of bot
void StartWaypointSystem (edict_t *bot)
{
	if (!G_EntExists(bot))
		return;

	if (!G_ClientInGame(bot))
		return;

	// memory allocation should not be done here, its done dynamically inside
	// wCreateWaypoint (my mistake in the original release, that's what was
	// screwing it up)

	#ifdef SHOW_LOGIC
		gi.dprintf("Starting waypoint system\n");
	#endif

	wSpawnPoint(bot);
}

// wEnemySearch - handles all the movetogoal stuff for waypoints going back
// to find one thats visible to their enemy. NOTE: waypoints[CURRENT_WAYPOINT]
// has NOT been created yet.
void wEnemySearch (edict_t *bot)
{
	int points = LAST_WAYPOINT;
	int i, j, k; // three for loops

	if (!G_EntExists(bot))
		return;

	if (!G_ClientInGame(bot))
		return;

	if (!G_EntExists(bot->enemy))
		return;

	if (!G_ClientInGame(bot->enemy))
		return; // might be dead or something

	if (bot->enemy == NULL)
		return; // nah. i'm not paranoid.

	if ((level.time - bot->lastsearch) < BOT_SEARCH_TIME)
		return; // dont search too often, cuts down on memory usage.

	bot->lastsearch = level.time;

	if (visible(bot, bot->enemy)) // we've already seen him, exit quietly
	{
		#ifdef SHOW_LOGIC
			gi.dprintf("Enemy is already visible\n");
		#endif

		wSetMoveinfo(bot, bot->enemy->s.origin, vec3_origin, vec3_origin);

		return;
	}

	// i=num_waypoints, keep i on the list, subtract one each time
	for (i = points ; i >= 0 ; i--)
	{
		if (wVisible(bot->enemy->s.origin, waypoints[i]->location))
		{
			if (wVisible(bot->s.origin, waypoints[i]->location))
			{
				#ifdef SHOW_LOGIC
					gi.dprintf("One waypoint path complete\n");
				#endif

				// we can see this waypoint, and the enemy can see this waypoint
				// save it off to the movetogoal
				wSetMoveinfo(bot, waypoints[i]->location, vec3_origin, vec3_origin);

				// get outta here we're done
				return;
			}

			// the enemy can be seen from this point, but the bot cant see
			// this point. advance search back to search for a point which
			// can see the point which can see the enemy =)
			for (j = points ; j >= 0 ; j--)
			{
				if (wVisible(waypoints[i]->location, waypoints[j]->location))
				{
					if (wVisible(bot->s.origin, waypoints[j]->location))
					{
						#ifdef SHOW_LOGIC
							gi.dprintf("Two waypoint path complete\n");
						#endif

						// this waypoint->the first waypoint->the enemy is complete
						// go to the "i" waypoint second
						wSetMoveinfo(bot, waypoints[j]->location, waypoints[i]->location, vec3_origin);
					}
					else
					{
						// try a three point search, i.e.:
						//
						// ------- ------------------
						// | bot | |   X------enemy |
						// | |   |-|   | ------------
						// | |         | |
						// | X---------X |
						// ---------------
						for (k = points ; k >= 0 ; k--)
						{
							if (wVisible(waypoints[j]->location, waypoints[k]->location))
							{
								if (wVisible(bot->s.origin, waypoints[k]->location))
								{
									#ifdef SHOW_LOGIC
										gi.dprintf("Three waypoint path complete\n");
									#endif

									// this waypoint->->the second waypoint->the first waypoint->the enemy is complete
									wSetMoveinfo(bot, waypoints[k]->location, waypoints[j]->location, waypoints[i]->location);
								}
							}
						}
					}
				}
			} // for j
		}
	} // for i
}

// wItemSearch
void wItemSearch (edict_t *bot)
{
	int points = LAST_WAYPOINT;
	int i; // for loop
	vec3_t best_point;
	int best_priority = 0;

	if (!G_EntExists(bot))
		return;

	if (!G_ClientInGame(bot))
		return;

	if (bot->itemwant == NULL)
		return; // how in hell could we get HERE? oh well.

	VectorClear(best_point);

	// i=num_waypoints, keep i on the list, subtract one each time
	for (i = points ; i >= 0 ; i--)
	{
		if ((waypoints[i]->priority > best_priority) && wVisible(bot->s.origin, waypoints[i]->location))
		{
			VectorCopy(waypoints[i]->location, best_point);
			best_priority = waypoints[i]->priority;
		}
	}

	if (VectorCompare(vec3_origin, best_point))
		wSetMoveinfo(bot, vec3_origin, vec3_origin, vec3_origin);
	else
		wSetMoveinfo(bot, best_point, vec3_origin, vec3_origin);
}

// memsets/clears the entire list, done upon initgame
void wAllocatePoints (void)
{
	int i;
	qboolean failed;

	// allocate and clear points
	for (i = 0 ; i < MAX_WAYPOINTS ; i++)
	{
		waypoints[i] = gi.TagMalloc(sizeof(waypoints[i]), TAG_LEVEL);

		if (waypoints[i] == NULL)
		{
			if (!failed)
			{
				failed = true;
				gi.dprintf("Waypoint allocation failed on waypoint %d.\nNote that exceeding this value of waypoints will crash.\n");
			}
			continue;
		}

		waypoints[i]->inuse = false;
		VectorCopy(vec3_origin, waypoints[i]->location);
		waypoints[i]->priority = 0;
	}
}

// clears points, called upon level load
void wClearPoints (void)
{
	int i;

	for (i = 0 ; i < MAX_WAYPOINTS ; i++)
	{
		if (waypoints[i] == NULL)
			continue;

		waypoints[i]->inuse = false;
		VectorCopy(vec3_origin, waypoints[i]->location);
		waypoints[i]->priority = 0;
	}
}

