#include "g_local.h"

// can bot jump in the up direction?
qboolean bCanJump(edict_t *bot) 
{
	vec3_t point;
	vec3_t mins;
	trace_t tr;

	// PSY: modified - cut down on calls to tv();
	VectorSet(mins, -16, -16, -16);
	
	// Can't jump from crouched position.
	if (bot->maxs[2]==4) 
		return false;
	
	// Already in the air?
	if (!bot->groundentity) 
		return false;
	
	// Timer still active?
	if (level.time < bot->last_jump + 1.0) 
		return false;
	
	// Any obstructions 2X above bot's head?
	VectorCopy(bot->s.origin, point);
	point[2] += bot->viewheight*2; // 2X height
	
	tr = gi.trace(bot->s.origin, mins, bot->maxs, point, bot, MASK_PLAYERSOLID);
	
	if (tr.fraction == 1.0)
		return true;

	return false;
}

// any obstructions in direction dir for length dist?
qboolean bCanMove(edict_t *bot, float length, int dir) 
{
	vec3_t forward, right, up, end, end2;
	vec3_t mins;
	vec3_t finmins;
	vec3_t finmaxs;
	trace_t tr;
	float len;
	vec3_t start;
	vec3_t lmins;
	vec3_t lmaxs;

	VectorCopy(bot->s.origin, start);
	
	// PSY: modified again, no calls to tv()
	VectorSet(mins, -16, -16, -16);
	VectorSet(finmins, -8, -8, -8);
	VectorSet(finmaxs, 8, 8, 8);
	VectorSet(lmins, bot->mins[0], bot->mins[1], -250);
	VectorSet(lmaxs, bot->maxs[0], bot->maxs[1], 250);

	// Ensure length properly bounded [24..96]
	len = (length<steplen) ? steplen: (length>jumplen) ? jumplen:length;
	
	AngleVectors(bot->s.angles, forward, right, up);
	
	switch (dir) 
	{
	case bLEFT: 
		VectorMA(bot->s.origin,-len, right, end); 
		break;
	case bRIGHT:
		VectorMA(bot->s.origin, len, right, end); 
		break;
	case bUP: 
		VectorMA(bot->s.origin, len, up, end); 
		break;
	case bFORWARD:
		VectorMA(bot->s.origin, len, forward, end); 
		break;
	case bBACKWARD:
		VectorMA(bot->s.origin, -len, forward, end); 
	}
	
	// Trace BBOX which is slightly shortened in -Z direction
	tr=gi.trace(bot->s.origin, mins, bot->maxs, end, bot, MASK_SHOT);
	
	// Any obstructions in this direction?
	if (tr.fraction < 1.0) 
		return false;

	// Trace down to see what we'll eventually step on..
	VectorMA(end, -1000, up, end2);

	// Use a slightly smaller BBOX for this second trace...
	tr=gi.trace(end, finmins, finmaxs, end2, bot, MASK_SOLID|MASK_WATER);
	
	// TRUE only if trace NOT hit lava/slime (everything else okay).
	return (!(tr.contents & (CONTENTS_LAVA|CONTENTS_SLIME)));
}

// turn bot to direction dir
void bTurn(edict_t *bot, vec3_t cmd_angles, int dir) 
{
	vec3_t v, end, forward, right;
	
	// Split current facing direction into components
	AngleVectors(bot->s.angles, forward, right, NULL);
	
	switch (dir) 
	{
	case bLEFT:
		VectorMA(bot->s.origin,-100, right, end); 
		break;
	case bRIGHT:
		VectorMA(bot->s.origin, 100, right, end); 
		break;
	case bBACKWARD:
		VectorMA(bot->s.origin,-100, forward, end); 
		break;
	case bUP:
		VectorCopy(bot->s.origin, end); 
		break;
	case bFORWARD:
		VectorMA(bot->s.origin, 100, forward, end); 
	}
	
	VectorSubtract(end, bot->s.origin, v);
	
	// Turn bot to face new dir!
	vectoangles(v, cmd_angles);
}

// turn and walk forward
void bTurnWalk(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles, int dir) 
{
	// Face this direction
	bTurn(bot,cmd_angles,dir);
	
	// Turning will face you forward!
	cmd->forwardmove=100;
}

// walk/strafe in direction dir
void bWalk(edict_t *bot, usercmd_t *cmd, int dir) 
{
	// Start walking movement.
	switch (dir) 
	{
    case bRIGHT:   
		cmd->sidemove=100; 
		break;
    case bLEFT:    
		cmd->sidemove=-100; 
		break;
    case bUP:      
		cmd->forwardmove=50; 
		break;
    case bFORWARD: 
		cmd->forwardmove=100; 
		break;
    case bBACKWARD:
		cmd->forwardmove=-100; 
	}
}

// turn and run in direction dir
qboolean bTurnRun(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles, int dir) 
{
	// Face this direction
	bTurn(bot,cmd_angles,dir);
	
	// Turning will face you forward!
	cmd->forwardmove=300;
	
	return true; // Always true;
}

// run/strafe-run
qboolean bRun(edict_t *bot, usercmd_t *cmd, int dir) 
{
	// Start running movements
	switch (dir) 
	{
    case bRIGHT:   
		cmd->sidemove=200; 
		break;
    case bLEFT:    
		cmd->sidemove=-200; 
		break;
    case bUP:      
		cmd->forwardmove=50; 
		break;
    case bFORWARD: 
		cmd->forwardmove=300; 
		break;
    case bBACKWARD:
		cmd->forwardmove=-300; 
	}
	
	return true; // Always true!
}

// turn and jump
qboolean bTurnJump(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles, int dir) 
{
	vec3_t up;
	
	// Quickly test some things first!
	if (!bCanJump(bot)) return false;
	
	// Any obstructions ahead or lava/slime at projected landing spot?
	if (!bCanMove(bot, jumplen, dir)) return false;
	
	// Turn in this direction too!
	bTurn(bot,cmd_angles,dir);
	
	// Jump NEEDS up velocity component.
	AngleVectors(bot->s.angles, NULL, NULL, up);
	VectorClear(bot->velocity);
	VectorMA(bot->velocity, 200, up, bot->velocity);
	cmd->upmove=200;
	
	// Turning will face you forward.
	cmd->forwardmove=300;
	
	bot->last_jump = level.time;
	
	return true;
} 

// did we jump?
qboolean bJump(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles, int dir) 
{
	vec3_t up;
	
	// Quickly test some things first!
	if (!bCanJump(bot)) return false;
	
	// Any obstructions ahead or lava/slime at projected landing spot?
	if (!bCanMove(bot, jumplen, dir)) return false;
	
	// Jump NEEDS up velocity component.
	AngleVectors(bot->s.angles, NULL, NULL, up);
	VectorClear(bot->velocity);
	VectorMA(bot->velocity, 200, up, bot->velocity);
	cmd->upmove=200;
	
	bot->last_jump = level.time;
	
	// Fast forward motion
	return bRun(bot,cmd,dir);
}


// duck
void bDuck(edict_t *bot) 
{
	bot->maxs[2]=4;
}

// climb the ladder
void bClimbLadder(edict_t  *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	// If we can jump forward then must be at top of ladder!
	if (bJump(bot, cmd, cmd_angles, bFORWARD))
		bot->on_ladder=false;
	else
		// Else, continue climbing ladder..
		bJump(bot, cmd, cmd_angles, bUP);
}

// swim for it
void bSwim(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	if (bJump(bot, cmd, cmd_angles,bUP))
		return;

	if (bTurnJump(bot,cmd,cmd_angles,bFORWARD))
		return;
	
	if (bTurnJump(bot,cmd,cmd_angles,bLEFT))
		return;
	
	if (bTurnJump(bot,cmd,cmd_angles,bRIGHT))
		return;
	
	if (bTurnJump(bot,cmd,cmd_angles,bBACKWARD))
		return;

	// just swim up
	bJump(bot, cmd, cmd_angles,bUP);
}

// platform riding routine
void bRidePlatform(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	// Is this platform still moving??
	if ((bot->groundentity->moveinfo.state == STATE_UP)
	|| (bot->groundentity->moveinfo.state == STATE_DOWN)) 
	{
		bot->pausetime = level.time + 0.3;
		return; 
	}

	bot->pausetime=level.time; // Turn OFF

	// Should be already facing forward!
	if (!bJump(bot,cmd,cmd_angles,bFORWARD))
		if (bCanMove(bot,steplen,bFORWARD)) 
		{
			bWalk(bot,cmd,bFORWARD); 
			return; 
		}

	if (!bJump(bot,cmd,cmd_angles,bLEFT))
		if (bCanMove(bot,steplen,bLEFT)) 
		{
			bTurnWalk(bot,cmd,cmd_angles,bLEFT);
			return; 
		}

	if (!bJump(bot,cmd,cmd_angles,bRIGHT))
		if (bCanMove(bot,steplen,bRIGHT)) 
		{
			bTurnWalk(bot,cmd,cmd_angles,bRIGHT);
			return; 
		}

	// What!! Something must be blocking me!!
	if (!bJump(bot,cmd,cmd_angles,bBACKWARD))
		if (bCanMove(bot,steplen,bRIGHT)) 
		{
			bTurnWalk(bot,cmd,cmd_angles,bBACKWARD);
			return; 
		}
	
	// just walk forward
	bWalk(bot,cmd,bFORWARD);
}

// get the item
qboolean bGetItem(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	vec3_t v;
	float dist;
	
	// No longer visible?  What happened?
	if (!visible2(bot, bot->itemwant)) 
	{
		bot->itemwant=NULL;
		return false; 
	}
	
	VectorSubtract(bot->itemwant->s.origin, bot->s.origin, v);
	dist=VectorLength(v);
	vectoangles(v, cmd_angles);
	
	// Can we jump the last bit toward this item.
	if (dist<32 && infront(bot,bot->itemwant))
		if (bJump(bot,cmd,cmd_angles,bFORWARD))
			return true;
		
	// Okay, let's get moving!
	if (bCanMove(bot,steplen,bFORWARD))
		return bRun(bot, cmd, bFORWARD);
		
	if (bCanMove(bot,steplen,bLEFT))
		return bRun(bot, cmd, bLEFT);
		
	if (bCanMove(bot,steplen,bRIGHT))
		return bRun(bot, cmd, bRIGHT);
					
	// Okay, Forget this item!
	bot->itemwant=NULL;
				
	// Turn around and head in reverse dir!
	if (bCanMove(bot,steplen,bBACKWARD))
		return bTurnRun(bot, cmd, cmd_angles, bBACKWARD);

	// just run forward
	return bRun(bot, cmd, bFORWARD);
}

// dodge fire routine - great piece of work
qboolean bDodgeFire(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	vec3_t forward, end;
	trace_t tr;
	
	// Check for incoming flak and take evasive action!!
	AngleVectors(bot->s.angles, forward, NULL, NULL);
	VectorMA(bot->s.origin, 1000, forward, end);
	tr=gi.trace(bot->s.origin, bot->mins, bot->maxs, end, bot, MASK_SHOT);
	
	// If there is incoming then move or duck!
	if (tr.ent && tr.ent->s.effects & (EF_BLASTER|EF_ROCKET|EF_GRENADE|EF_HYPERBLASTER|EF_BFG)) {
		// Quick jump check right away!
		if (bCanJump(bot)) 
		{
			if (bJump(bot,cmd,cmd_angles,bLEFT))
				return true;
			if (bJump(bot,cmd,cmd_angles,bRIGHT))
				return true; 
		}
		
		// Otherwise, step left/right
		if (bCanMove(bot,steplen,bLEFT))
			return bRun(bot, cmd, bLEFT);
		
		if (bCanMove(bot,steplen,bRIGHT))
			return bRun(bot, cmd, bRIGHT);
		
		// Lastly, if can't jump UP then DUCK!!
		if (!bJump(bot,cmd,cmd_angles,bUP)) 
		{
			bDuck(bot); return true; 
		}
	}
	
	return false;
}

void bFace(edict_t *bot, edict_t *target, vec3_t angles)
{
	vec3_t dir, start, end;

	VectorCopy(target->s.origin, start);
	VectorCopy(bot->s.origin, end);
    VectorSubtract(start, end, dir);
    vectoangles(dir, angles);

	angles[YAW] += (crandom() * 20); // a little randomization for fire
	// -20 to +20 will get you off the player's bbox about 10% of the time, about
	// equal to an above average Q2 dm'er w/ no lag
}

// select a new direction of travel
void bNewChaseDir(edict_t *bot, edict_t *goal, float dist, vec3_t cmd_angles) 
{
	float d[3],deltax, deltay;
	float yaw, olddir, reversedir;
	
	olddir=anglemod((int)(bot->ideal_yaw/45)*45);
	reversedir=anglemod(olddir-180);
	
	deltax=goal->s.origin[0]-bot->s.origin[0];
	deltay=goal->s.origin[1]-bot->s.origin[1];
	
	if (deltax>10)
		d[1]=0;
	else if (deltax<-10)
		d[1]=180;
	else
		d[1]=-99;
	
	if (deltay<-10)
		d[2]=270;
	else if (deltay>10)
		d[2]=90;
	else
		d[2]=-99;
	
	// Try direct route first!
	if (d[1]!=-99 && d[2]!=-99) 
	{
		if (d[1]==0)
			yaw=(d[2]==90)?45:315;
		else
			yaw=(d[2]==90)?135:215;
		if (yaw != reversedir)
			// Can we step in yaw*dist direction?
			if (bCanStep(bot, yaw, dist, cmd_angles))
				return; 
	}
	
	// Rotate the vector coordinates around...
	if (((rand()&3)&1) || (abs(deltay)>abs(deltax))) 
	{
		yaw=d[1]; d[1]=d[2]; d[2]=yaw; 
	}
	
	// Let's try the d[1] direction?
	if (yaw!=-99 && yaw!=reversedir)
		// Can we step in d[1]*dist direction?
		if (bCanStep(bot, yaw, dist, cmd_angles))
			return;
		
	// Let's try the d[2] direction?
	yaw=d[2];
	if (yaw!=-99 && yaw!=reversedir)
		// Can we step in d[2]*dist direction?
		if (bCanStep(bot, yaw, dist, cmd_angles))
			return;
			
	// Search from 0..315 in 45 deg increments?
	if (rand()&1) {
		for (yaw=0; yaw<=315; yaw+=45)
			if (yaw!=reversedir)
				// Can we step in yaw*dist direction?
				if (bCanStep(bot, yaw, dist, cmd_angles))
					return; }
	else
		// Search from 315..0 in -45 deg increments?
		for (yaw=315; yaw>=0; yaw-=45)
			if (yaw!=reversedir)
				// Can we step in yaw*dist direction?
				if (bCanStep(bot, yaw, dist, cmd_angles))
					return;
						
	// Can we step in the reverse direction?
	if (bCanStep(bot, reversedir, dist, cmd_angles))
		return;
}

// move bot 1 step to enemy, through the search waypoints
void bMoveToEnemy(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	float dist=0, yaw;
	vec3_t v1, v2, forward;
	
	if (!bot->enemy)
		return;

	// PSY: save cycles: only search when we need it
	if ((VectorCompare(bot->point1, vec3_origin)) /*if we have no point1*/
		&& !visible(bot, bot->enemy) /*AND we arent visible to our enemy*/)
		wEnemySearch(bot); // no points, so search
	else if (visible(bot, bot->enemy)) // if we're visible, why use the points?
	{
		VectorSubtract(bot->s.origin, bot->enemy->s.origin, v2);
		dist = VectorLength(v2);
		VectorSubtract(bot->enemy->s.origin, bot->s.origin, v1);
		yaw=vectoyaw(v1);
					
		if (!bCanStep(bot, yaw, dist, cmd_angles))
			bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
		else {
			AngleVectors(cmd_angles, forward, NULL, NULL);
			VectorMA(bot->velocity, 80, forward, bot->velocity);
			cmd->forwardmove=300; }
	}

	if (!VectorCompare(bot->point1, vec3_origin))
	{
		if (bDistance(bot->s.origin, bot->point1) < 15)
		{
			// we're close, go to point2
			if (!VectorCompare(bot->point2, vec3_origin))
			{
				if (bDistance(bot->s.origin, bot->point2) < 15)
				{
					// we're close, go to point3
					if (!VectorCompare(bot->point3, vec3_origin))
					{
						VectorSubtract(bot->s.origin, bot->point3, v2);
						dist = VectorLength(v2);
						VectorSubtract(bot->point3, bot->s.origin, v1);
						yaw=vectoyaw(v1);
					
						if (!bCanStep(bot, yaw, dist, cmd_angles))
							bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
						else {
							AngleVectors(cmd_angles, forward, NULL, NULL);
							VectorMA(bot->velocity, 80, forward, bot->velocity);
							cmd->forwardmove=300; }
						
						return;
					}
					else
						bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
				}

				VectorSubtract(bot->s.origin, bot->point2, v2);
				dist = VectorLength(v2);
				VectorSubtract(bot->point2, bot->s.origin, v1);
				yaw=vectoyaw(v1);
			
				if (!bCanStep(bot, yaw, dist, cmd_angles))
					bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
				else {
					AngleVectors(cmd_angles, forward, NULL, NULL);
					VectorMA(bot->velocity, 80, forward, bot->velocity);
					cmd->forwardmove=300; }
				
				return;
			}
			else
				bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
		}

		VectorSubtract(bot->s.origin, bot->point1, v2);
		dist = VectorLength(v2);
		VectorSubtract(bot->point1, bot->s.origin, v1);
		yaw=vectoyaw(v1);
	
		if (!bCanStep(bot, yaw, dist, cmd_angles))
			bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
		else {
			AngleVectors(cmd_angles, forward, NULL, NULL);
			VectorMA(bot->velocity, 80, forward, bot->velocity);
			cmd->forwardmove=300; }
		
		return;
	}
	else
	{
		VectorSubtract(bot->s.origin, bot->enemy->s.origin, v2);
		dist = VectorLength(v2);
		VectorSubtract(bot->enemy->s.origin, bot->s.origin, v1);
		yaw=vectoyaw(v1);
	
		if (!bCanStep(bot, yaw, dist, cmd_angles))
			bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
		else {
			AngleVectors(cmd_angles, forward, NULL, NULL);
			VectorMA(bot->velocity, 80, forward, bot->velocity);
			cmd->forwardmove=300; }
	}
}

// move bot 1 step to item waypoint, but we only need to go to point1 (see
// inside of wItemSearch)
void bMoveToItem(edict_t *bot, usercmd_t *cmd, vec3_t cmd_angles) 
{
	float dist=0, yaw;
	vec3_t v1, v2, forward;
	
	// set up point1 vector
	wItemSearch(bot);

	// note: we just stand here if there's NO waypoints to go to
	if (!VectorCompare(bot->point1, vec3_origin))
	{
		VectorSubtract(bot->s.origin, bot->point1, v2);
		dist = VectorLength(v2);
		VectorSubtract(bot->point1, bot->s.origin, v1);
		yaw=vectoyaw(v1);
	
		if (!bCanStep(bot, yaw, dist, cmd_angles))
			bNewChaseDir(bot, bot->enemy, dist, cmd_angles);
		else {
			AngleVectors(cmd_angles, forward, NULL, NULL);
			VectorMA(bot->velocity, 80, forward, bot->velocity);
			cmd->forwardmove=300; }
		
		return;
	}
}

// move bot 1 step to goal
void bMoveToGoal(edict_t *bot, edict_t *goal, usercmd_t *cmd, vec3_t cmd_angles) 
{
	float dist, yaw;
	vec3_t v1, v2, forward;
	
	if (!goal) return;
	
	VectorSubtract(bot->s.origin, goal->s.origin, v2);
	dist = VectorLength(v2);
	VectorSubtract(goal->s.origin, bot->s.origin, v1);
	yaw=vectoyaw(v1);
	
	if (!bCanStep(bot, yaw, dist, cmd_angles))
		bNewChaseDir(bot, goal, dist, cmd_angles);
	else {
		AngleVectors(cmd_angles, forward, NULL, NULL);
		VectorMA(bot->velocity, 80, forward, bot->velocity);
		cmd->forwardmove=300; }
}

// rotate cmd_angles to ideal_yaw
void bChangeYaw(edict_t *bot, vec3_t cmd_angles) 
{
	float ideal, current;
	float move, speed;
	
	current = anglemod(cmd_angles[YAW]);
	ideal = bot->ideal_yaw;
	
	if (current == ideal) return;
	
	move = ideal - current;
	speed = bot->yaw_speed;
	
	if (ideal > current) {
		if (move >= 180)
			move = move - 360;}
	else {
		if (move <= -180)
			move = move + 360; }
	if (move > 0) {
		if (move > speed)
			move = speed; }
	else {
		if (move < -speed)
			move = -speed; }
	
	cmd_angles[YAW] = anglemod(current + move);
}
