#include "g_local.h"

extern gitem_armor_t jacketarmor_info;
extern gitem_armor_t combatarmor_info;
extern gitem_armor_t bodyarmor_info;

//========================================================
// Returns the WEAPON inventory index for the given weapon.
//========================================================
int WeapIndex(gitem_t *weapon) 
{
	return ITEM_INDEX(weapon);
}

//========================================================
// Returns the AMMO inventory index for the given weapon.
//========================================================
int AmmoIndex(gitem_t *weapon) 
{
	if (weapon==item_shotgun)         return ITEM_INDEX(item_shells);
	if (weapon==item_supershotgun)    return ITEM_INDEX(item_shells);
	if (weapon==item_machinegun)      return ITEM_INDEX(item_bullets);
	if (weapon==item_chaingun)        return ITEM_INDEX(item_bullets);
	if (weapon==item_handgrenade)     return ITEM_INDEX(item_grenades);
	if (weapon==item_grenadelauncher) return ITEM_INDEX(item_grenades);
	if (weapon==item_rocketlauncher)  return ITEM_INDEX(item_rockets);
	if (weapon==item_railgun)         return ITEM_INDEX(item_slugs);
	if (weapon==item_hyperblaster)    return ITEM_INDEX(item_cells);
	if (weapon==item_bfg)             return ITEM_INDEX(item_cells);
	
	// PSY: modified: set this to zero, check for it in all the
	// search functions, the bot was actually prioritizing shells while on blaster
	return 0;
}

//========================================================
//============ AMMO/WEAPONS IN INVENTORY? ================
//========================================================

//========================================================
// TRUE if inventory FULL of given ammo item.
//========================================================
qboolean AlreadyAtMaxAmmo(edict_t *ent, gitem_t *ammo) 
{
	// PSY: modified: if it's a pack (which returns true out of IsAmmo, see how this function
	// is used down below) then grab it if we need it!
	if (ammo->flags & IT_PACK)
	{
		if ((ent->client->pers.max_bullets < 250) && ammo == item_bandolier)
			// it's a bandolier, and we need it, so get it!
			return false;
		else if ((ent->client->pers.max_bullets < 300) && ammo == item_pack)
			// it's an ammo pack and we need it so get it!
			return false;
		return true;
	}

	if ((int)dmflags->value & DF_INFINITE_AMMO)
		return true;
	
	if (ammo == item_shells)
		return (ent->client->pers.inventory[ITEM_INDEX(item_shells)] >= ent->client->pers.max_shells);
	if (ammo == item_bullets)
		return (ent->client->pers.inventory[ITEM_INDEX(item_bullets)] >= ent->client->pers.max_bullets);
	if (ammo == item_grenades)
		return (ent->client->pers.inventory[ITEM_INDEX(item_grenades)] >= ent->client->pers.max_grenades);
	if (ammo == item_rockets)
		return (ent->client->pers.inventory[ITEM_INDEX(item_rockets)] >= ent->client->pers.max_rockets);
	if (ammo == item_slugs)
		return (ent->client->pers.inventory[ITEM_INDEX(item_slugs)] >= ent->client->pers.max_slugs);
	if (ammo == item_cells)
		return (ent->client->pers.inventory[ITEM_INDEX(item_cells)] >= ent->client->pers.max_cells);

	return false;
}

//========================================================
// TRUE if bot is already FULL of armor item.
//========================================================
qboolean AlreadyAtMaxArmor(edict_t *bot, gitem_t *newarmor) 
{
	int old_armor_index;
	gitem_armor_t *oldinfo;
	gitem_armor_t *newinfo;
	
	// These are additive to existing armor!
	// ALWAYS pick up these types!
	// PSY: modified - if its a power shield/screen and we already have
	// a power shield/screen, then why get another one (we can't use 2 at once)
	// also check for the fact that a shield is better than a screen.
	if (newarmor == item_armorshard)
		return false;
	if (newarmor == item_powershield || newarmor == item_powerscreen)
	{
		if (newarmor == item_powershield)
		{
			if (bot->client->pers.inventory[ITEM_INDEX(item_powershield)] > 0)
				return true; // already have one
			if (bot->client->pers.inventory[ITEM_INDEX(item_powerscreen)] > 0)
				return false; // a power shield is better than a power screen
		}
		else // if (newarmor == item_powerscreen)
		{
			if (bot->client->pers.inventory[ITEM_INDEX(item_powershield)] > 0)
				return true; // dont get the screen, the shield is better
			if (bot->client->pers.inventory[ITEM_INDEX(item_powerscreen)] > 0)
				return true; // already have a screen
		}
	}
	
	// See if bot wearing armor now!
	old_armor_index=ArmorIndex(bot);
	
	// No current armor? Then, pick it up!
	if (!old_armor_index) return false;
	
	// -----------------------------------------------
	// See gitem_armor_t structs at top of g_items.c
	//
	// Armor is ADDITIVE !!
	// Check ceilings on various armor types..
	// -----------------------------------------------
	
	// What kind of armor is bot currently wearing?
	if (old_armor_index == jacket_armor_index)
		oldinfo=&jacketarmor_info;
	else if (old_armor_index == combat_armor_index)
		oldinfo=&combatarmor_info;
	else
		oldinfo=&bodyarmor_info;
	
	// What kind of armor is new item?
	newinfo=(gitem_armor_t *)newarmor->info;
	
	// If new armor of better protection type then grab it!
	if (newinfo->normal_protection > oldinfo->normal_protection)
		return false;
	
	// New armor must be same or lesser type than previous armor...
	// Check if already at max allowed armor for prev armor.
	return (bot->client->pers.inventory[old_armor_index] >= oldinfo->max_count);
}

//========================================================
// TRUE if self has weapon for the given ammo item.
//========================================================
qboolean HasWeaponForAmmo(edict_t *self, gitem_t *ammo) 
{
	if (ammo==item_shells)
		return ((HasItemInInventory(self,item_supershotgun)
		|| (HasItemInInventory(self,item_shotgun))));
	if (ammo==item_rockets)
		return (HasItemInInventory(self,item_rocketlauncher));
	if (ammo==item_cells)
		return ((HasItemInInventory(self,item_hyperblaster)
		|| (HasItemInInventory(self,item_bfg))));
	if (ammo==item_bullets)
		return ((HasItemInInventory(self,item_chaingun)
		|| (HasItemInInventory(self,item_machinegun))));
	if (ammo==item_slugs)
		return (HasItemInInventory(self,item_railgun));
	if (ammo==item_grenades)
		return ((HasItemInInventory(self,item_grenadelauncher)
        || (HasItemInInventory(self,item_grenades))));
	
	return false;
}

//========================================================
// TRUE if self has sufficient ammo in inventory for weapon.
//========================================================
qboolean HasAmmoForWeapon(edict_t *self, gitem_t *weapon) 
{
	if ((int)dmflags->value & DF_INFINITE_AMMO)
		return true;
	
	if (weapon==item_blaster) return true;
	
	if (weapon==item_bfg)
		return (self->client->pers.inventory[AmmoIndex(item_bfg)]>=50);
	
	if (weapon==item_supershotgun)
		return (self->client->pers.inventory[AmmoIndex(item_supershotgun)]>=2);
	
	//PSY: modified: the blaster now returns AmmoIndex=0 so check for that here to be safe
	if (AmmoIndex(weapon) == 0)
		return true;

	return (self->client->pers.inventory[AmmoIndex(weapon)]);
}

//========================================================
// TRUE if self has Item in inventory - PSY: modified - we
// used to have two functions here - weaponininventory and
// special in inventory - not necessary, they did the same thing.
//========================================================
qboolean HasItemInInventory(edict_t *self, gitem_t *item) 
{
	return (self->client->pers.inventory[ITEM_INDEX(item)]);
}

//========================================================
//=============== ITEM IDENTIFICATION ====================
//========================================================

//========================================================
// TRUE if item is ammo
//========================================================
qboolean IsAmmo(gitem_t *item) 
{
	return (item->flags & IT_AMMO|IT_PACK);
}

//========================================================
// TRUE if item is armor or 'armor related'
//========================================================
qboolean IsArmor(gitem_t *item) 
{
	return (item->flags & IT_ARMOR);
}

//========================================================
// TRUE if item is health
//========================================================
qboolean IsHealth(gitem_t *item) 
{
	return (item->flags & IT_HEALTH);
}

//========================================================
// TRUE if item is weapon
//========================================================
qboolean IsWeapon(gitem_t *item) 
{
	return (item->flags & IT_WEAPON);
}

//========================================================
// TRUE if item is a 'PowerUp' item
//========================================================
qboolean IsPowerup(gitem_t *item) 
{
	return (item->flags & IT_POWERUP);
}

//========================================================
// TRUE if item is a 'Key' item
//========================================================
qboolean IsKey(gitem_t *item) 
{
	return (item->flags & IT_KEY);
}

//========================================================
//================= WEAPON SELECTION =====================
//========================================================

//========================================================
// TRUE if Self switched to that weapon
//========================================================
qboolean CanSwitchToThisWeapon(edict_t *self, gitem_t *weapon) 
{
	if (HasItemInInventory(self,weapon))
		if (HasAmmoForWeapon(self,weapon)) 
		{
			self->client->newweapon=weapon;
			
			// PSY: modified - blaster now returns AmmoIndex=0 so check for that,
			// and set ammo_index to 0 if necessary.
			if (AmmoIndex(weapon) == 0)
				self->client->ammo_index = 0;
			else
				self->client->ammo_index=AmmoIndex(weapon);

			if (weapon!=self->client->pers.weapon)
				self->weap_delay = level.time + 3;
			
			ChangeWeapon(self);
			return true; 
		}
		
		return false;
}

//========================================================
// Switch to the first CLOSE weapon you can switch to! - PSY: modified,
// we'd rather use a blaster than a GL at less than 160 quake units.
//=======================================================
void GetCloseWeapon(edict_t *self) {
	
	if (CanSwitchToThisWeapon(self,item_chaingun))       return;
	if (CanSwitchToThisWeapon(self,item_supershotgun))   return;
	if (CanSwitchToThisWeapon(self,item_hyperblaster))   return;
	if (CanSwitchToThisWeapon(self,item_machinegun))     return;
	if (CanSwitchToThisWeapon(self,item_grenades))       return;
	if (CanSwitchToThisWeapon(self,item_shotgun))        return;
	if (CanSwitchToThisWeapon(self,item_railgun))        return;
	
	CanSwitchToThisWeapon(self,item_blaster);
}

//========================================================
// Switch to the first FAR weapon you can switch to! - PSY: modified:
// we'd rather use a blaster than hand grenades or a super shotgun
// at > 700 units
//======================================================
void GetFarWeapon(edict_t *self) {
	
	if (CanSwitchToThisWeapon(self,item_railgun))        return;
	if (CanSwitchToThisWeapon(self,item_rocketlauncher)) return;
	if (CanSwitchToThisWeapon(self,item_hyperblaster))   return;
	if (CanSwitchToThisWeapon(self,item_machinegun))     return;
	if (CanSwitchToThisWeapon(self,item_bfg))            return;
	if (CanSwitchToThisWeapon(self,item_shotgun))        return;
	if (CanSwitchToThisWeapon(self,item_chaingun))       return;
	
	CanSwitchToThisWeapon(self,item_blaster);
}

//========================================================
// Switch to the first BEST weapon you can switch to!
//========================================================
void GetBestWeapon(edict_t *self) {
	
	if (CanSwitchToThisWeapon(self,item_bfg))            return;
	if (CanSwitchToThisWeapon(self,item_railgun))        return;
	if (CanSwitchToThisWeapon(self,item_hyperblaster))   return;
	if (CanSwitchToThisWeapon(self,item_rocketlauncher)) return;
	if (CanSwitchToThisWeapon(self,item_chaingun))       return;
	if (CanSwitchToThisWeapon(self,item_supershotgun))   return;
	if (CanSwitchToThisWeapon(self,item_grenadelauncher))return;
	if (CanSwitchToThisWeapon(self,item_machinegun))     return;
	if (CanSwitchToThisWeapon(self,item_shotgun))        return;
	if (CanSwitchToThisWeapon(self,item_grenades))       return;
	
	CanSwitchToThisWeapon(self,item_blaster);
}

//=============================================================
// TRUE if the item is something we should go after.. PSY: modified-
// use our new HasItemInInventory function instead of two functions
// that do exactly the same thing
//=============================================================
qboolean bItemHasPriority(edict_t *bot, gitem_t *item) 
{
	if (!item) return false;
	
	if (item->flags & IT_HEALTH)
		return (bot->health < bot->max_health);
	if (item->flags & IT_ARMOR)
		return (!AlreadyAtMaxArmor(bot,item));
	if (item->flags & IT_WEAPON)
		return (!HasItemInInventory(bot,item));
	if (item->flags & IT_AMMO)
		return ((!AlreadyAtMaxAmmo(bot,item)) && HasWeaponForAmmo(bot, item));
	if (item->flags & IT_PACK)
		return true;
	if (item->flags & IT_POWERUP)
		return (!HasItemInInventory(bot,item));
	if (item->flags & IT_KEY)
		return (!HasItemInInventory(bot,item));
	
	return false;
}

//===========================================================
// Would bot leave a fragfest for this particular item?
//===========================================================
qboolean bNeedItem(edict_t *bot) 
{
	gitem_t *item;
	vec3_t v;
	
	// Little bit of safeguarding... PSY: modified - check here for SVF_NOCLIENT of
	// respawning items
	if (!bot->itemwant || !bot->itemwant->item || (bot->itemwant->svflags & SVF_NOCLIENT))
		return 0;
	
	// Is item a bit too far away to get right now??? PSY: modified: let's look up
	// to 500 units, its not that far (bot used to be a bodyguard after all)
	VectorSubtract(bot->s.origin, bot->itemwant->s.origin, v);
	if (VectorLength(v) > 500) return 0;
	
	item = bot->itemwant->item;
	
	if (IsWeapon(item))
	{
		if (CanSwitchToThisWeapon(bot,item))
			return 0; //already got this weapon!
		else
			// this new weapon would be better than my current weapon!!
			return (WeapIndex(bot->client->pers.weapon) < WeapIndex(item));
	}
		
	if (IsAmmo(item))
	{
		if (HasWeaponForAmmo(bot,item) && !AlreadyAtMaxAmmo(bot,item))
			return true; // PSY: modified: we always need ammo dammit!
		else
			return 0;
	}
		
	//PSY: modified - get health if we're less than 25, irrespective of count
	if (IsHealth(item))
		return (bot->health < 25);
			
	if (IsArmor(item))
		return (!AlreadyAtMaxArmor(bot,item));

	return 0;
}

//===========================================================
// Returns closest health item to bot's location - PSY:
// created this function to mimic behavior of an actual player -
// you look for health if less than 30 or so (see bAI();)
//===========================================================
edict_t *bSeekHealth(edict_t *bot) 
{
	edict_t *ent=NULL, *best_ent=NULL;
	float this_dist=0, best_dist=8192;
	float radius=1000;
	
	while ((ent=findradius(ent, bot->s.origin, radius)) != NULL) 
	{
		if (!ent || !ent->item) continue;
		if (ent->svflags & SVF_NOCLIENT) continue;
		if (ent->solid==SOLID_NOT) continue;
		if (!visible(bot, ent) || !visible2(bot, ent)) continue;
		if (!(ent->item->flags & IT_HEALTH)) continue; // not health
		// Ignore items dropped in lava/slime!
		if (gi.pointcontents(ent->s.origin) & (CONTENTS_LAVA|CONTENTS_SLIME)) continue;
		
		this_dist=abs(G_Distance(bot, ent));
		
		if (this_dist < best_dist) {
			best_dist = this_dist;
			best_ent = ent; } 
	}
			
	return best_ent;
}

//===========================================================
// Returns closest 'priority' item to bot's location
//===========================================================
edict_t *bSeekItem(edict_t *bot) 
{
	edict_t *ent=NULL, *best_ent=NULL;
	float this_dist=0, best_dist=8192;
	float radius=1000;
	
	// Already got an itemwant?  Can we use the current item?
	if (bot->itemwant && bot->itemwant->inuse && bot->itemwant->solid !=SOLID_NOT)
		if (visible(bot, bot->itemwant) && visible2(bot, bot->itemwant)) 
			return bot->itemwant; // PSY: we dont wanna switch items too much - causes
		// bot jerking betewen targets bug
			
	while ((ent=findradius(ent, bot->s.origin, radius)) != NULL) 
	{
		if (!ent || !ent->item) 
			continue;
		if (ent->svflags & SVF_NOCLIENT) 
			continue;
		if (ent->solid==SOLID_NOT) 
			continue;
		if (!visible(bot, ent) || !visible2(bot, ent)) 
			continue;
		if (!bItemHasPriority(bot, ent->item)) 
			continue;
		// Ignore items dropped in lava/slime!
		if (gi.pointcontents(ent->s.origin) & (CONTENTS_LAVA|CONTENTS_SLIME)) 
			continue;
		if (bot->client->pers.weapon == FindItem("Blaster"))
			if ((!(ent->item->flags & IT_WEAPON)) && (!(ent->item->flags & IT_AMMO)))
				continue; // PSY: real player behavior: only get guns/ammo till we 
			// have something other than blaster

		this_dist=abs(G_Distance(bot, ent));
		
		if (this_dist < best_dist) 
		{
			best_dist = this_dist;
			best_ent = ent; 
		} 
	}
			
	return best_ent;
}

