// g_weapon.c

#include "g_local.h"
#include "m_player.h"

static byte		is_silenced;
void fire_flash (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage);
void fire_homing (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage);
void fire_cluster (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage);
void fire_mega (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, float damage_radius, int radius_damage);

static void P_ProjectSource (gclient_t *client, vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result)
{
	vec3_t	_distance;
	
	if (!client)
		return;

	VectorCopy (distance, _distance);
	if (client->pers.hand == LEFT_HANDED)
		_distance[1] *= -1;
	else if (client->pers.hand == CENTER_HANDED)
		_distance[1] = 0;
	G_ProjectSource (point, _distance, forward, right, result);
}

void N_ProjectSource (gclient_t *client, vec3_t point, vec3_t distance, vec3_t forward, vec3_t right, vec3_t result, side side)
{
	vec3_t	_distance;
	
	if (!client)
		return;

	VectorCopy (distance, _distance);
	if (side == LEFT)
		_distance[1] *= -1;
	G_ProjectSource (point, _distance, forward, right, result);
}

side SideToggle(side aside)
{
	if (aside == LEFT)
		return RIGHT;
	else if (aside == RIGHT)
		return LEFT;
	else
		gi.dprintf("a side was neither left nor right!\n");
	return RIGHT;
}

/*
===============
PlayerNoise

Each player can have two noise objects associated with it:
a personal noise (jumping, pain, weapon firing), and a weapon
target noise (bullet wall impacts)
===============
*/
void PlayerNoise(edict_t *who, vec3_t where, int type)
{
	if (!who->client)
		return;

	if (type == PNOISE_WEAPON && who->client->silencer_shots)
		who->client->silencer_shots--;
}

qboolean Pickup_Weapon (edict_t *ent, edict_t *other)
{
	int			index;
	gitem_t		*ammo;

	if (!ent->client)
		return false;

	index = ITEM_INDEX(ent->item);

	other->client->pers.inventory[index]++;

	// give them some ammo with it
	ammo = FindItem (ent->item->ammo);
	if ( (int)dmflags->value & DF_INFINITE_AMMO )
		Add_Ammo (other, ammo, 1000);
	else
		Add_Ammo (other, ammo, ammo->quantity);

	if ((int)(dmflags->value) & DF_WEAPONS_STAY)
		ent->flags |= FL_RESPAWN;
	else
		SetRespawn (ent, 30);

	if (other->client->pers.weapon != ent->item && 
		(other->client->pers.inventory[index] == 1) &&
		( other->client->pers.weapon == FindItem("laser level 1") ) )
		other->client->newweapon = ent->item;

	return true;
}


/*
===============
ChangeWeapon

The old weapon has been dropped all the way, so make the new one
current
===============
*/
void ChangeWeapon (edict_t *ent)
{
	int i;

	if (!ent->client)
		return;

	ent->client->pers.lastweapon = ent->client->pers.weapon;
	ent->client->pers.weapon = ent->client->newweapon;
	ent->client->newweapon = NULL;

	// set visible model
	if (ent->s.modelindex == 255) {
		if (ent->client->pers.weapon)
			i = ((ent->client->pers.weapon->weapmodel & 0xff) << 8);
		else
			i = 0;
		ent->s.skinnum = (ent - g_edicts - 1) | i;
	}

	if (ent->client->pers.weapon && ent->client->pers.weapon->ammo)
		ent->client->ammo_index = ITEM_INDEX(FindItem(ent->client->pers.weapon->ammo));
	else
		ent->client->ammo_index = 0;

	if (!ent->client->pers.weapon)
	{	// dead
		ent->client->ps.gunindex = 0;
		return;
	}

	ent->client->weaponstate = WEAPON_ACTIVATING;
	ent->client->ps.gunframe = 0;
	ent->client->ps.gunindex = 0;
}

/*
=================
NoAmmoWeaponChange
=================
*/
void NoAmmoWeaponChange (edict_t *ent)
{
	if (!ent->client)
		return;

	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("rockets"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("cluster missiles"))] )
	{
		ent->client->newweapon = FindItem ("cluster missiles");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("rockets"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("homing missiles"))] )
	{
		ent->client->newweapon = FindItem ("homing missiles");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("rockets"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("flash missiles"))] )
	{
		ent->client->newweapon = FindItem ("flash missiles");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("rockets"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("concussion missiles"))] )
	{
		ent->client->newweapon = FindItem ("concussion missiles");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("bullets"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("gauss cannon"))] )
	{
		ent->client->newweapon = FindItem ("gauss cannon");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("bullets"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("vulcan"))] )
	{
		ent->client->newweapon = FindItem ("vulcan");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("cells"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("laser level 3"))] )
	{
		ent->client->newweapon = FindItem ("laser level 3");
		return;
	}
	if ( ent->client->pers.inventory[ITEM_INDEX(FindItem("cells"))]
		&&  ent->client->pers.inventory[ITEM_INDEX(FindItem("laser level 2"))] )
	{
		ent->client->newweapon = FindItem ("laser level 2");
		return;
	}
	ent->client->newweapon = FindItem ("laser level 1");
}

/*
=================
Think_Weapon

Called by ClientBeginServerFrame and ClientThink
=================
*/
void Think_Weapon (edict_t *ent)
{
	// if just died, put the weapon away
	if (ent->health < 1)
	{
		ent->client->newweapon = NULL;
		ChangeWeapon (ent);
	}

	if (!ent->client)
		return;

	// call active weapon think routine
	if (ent->client->pers.weapon && ent->client->pers.weapon->weaponthink)
	{
		ent->client->pers.weapon->weaponthink (ent);
	}
}


/*
================
Use_Weapon

Make the weapon ready if there is ammo
================
*/
void Use_Weapon (edict_t *ent, gitem_t *item)
{
	int			ammo_index;
	gitem_t		*ammo_item;

	if (!ent->client)
		return;

	// see if we're already using it
	if (item == ent->client->pers.weapon)
		return;

	if (item->ammo && !g_select_empty->value && !(item->flags & IT_AMMO))
	{
		ammo_item = FindItem(item->ammo);
		ammo_index = ITEM_INDEX(ammo_item);

		if (!ent->client->pers.inventory[ammo_index])
		{
			gi.cprintf (ent, PRINT_HIGH, "No %s for %s.\n", ammo_item->pickup_name, item->pickup_name);
			return;
		}

		if (ent->client->pers.inventory[ammo_index] < item->quantity)
		{
			gi.cprintf (ent, PRINT_HIGH, "Not enough %s for %s.\n", ammo_item->pickup_name, item->pickup_name);
			return;
		}
	}

	// change to this weapon when down
	ent->client->newweapon = item;
}

/*
================
Weapon_Generic

A generic function to handle the basics of weapon thinking
================
*/
#define FRAME_FIRE_FIRST		(FRAME_ACTIVATE_LAST + 1)
#define FRAME_IDLE_FIRST		(FRAME_FIRE_LAST + 1)
#define FRAME_DEACTIVATE_FIRST	(FRAME_IDLE_LAST + 1)

void Weapon_Generic (edict_t *ent, int FRAME_ACTIVATE_LAST, int FRAME_FIRE_LAST, int FRAME_IDLE_LAST, int FRAME_DEACTIVATE_LAST, int *pause_frames, int *fire_frames, void (*fire)(edict_t *ent))
{
	int		n;

	if (!ent->client)
		return;

	if(ent->deadflag || ent->s.modelindex != 255) // VWep animations screw up corpses
	{
		return;
	}

	if (ent->client->weaponstate == WEAPON_DROPPING)
	{
		if (ent->client->ps.gunframe == FRAME_DEACTIVATE_LAST)
		{
			ChangeWeapon (ent);
			return;
		}

		ent->client->ps.gunframe++;
		return;
	}

	if (ent->client->weaponstate == WEAPON_ACTIVATING)
	{
		if (ent->client->ps.gunframe == FRAME_ACTIVATE_LAST)
		{
			ent->client->weaponstate = WEAPON_READY;
			ent->client->ps.gunframe = FRAME_IDLE_FIRST;
			return;
		}

		ent->client->ps.gunframe++;
		return;
	}

	if ((ent->client->newweapon) && (ent->client->weaponstate != WEAPON_FIRING))
	{
		ent->client->weaponstate = WEAPON_DROPPING;
		ent->client->ps.gunframe = FRAME_DEACTIVATE_FIRST;

		return;
	}

	if (ent->client->weaponstate == WEAPON_READY)
	{
		if ( ((ent->client->latched_buttons|ent->client->buttons) & BUTTON_ATTACK) )
		{
			ent->client->latched_buttons &= ~BUTTON_ATTACK;
			if ((!ent->client->ammo_index) || 
				( ent->client->pers.inventory[ent->client->ammo_index] >= ent->client->pers.weapon->quantity))
			{
				ent->client->ps.gunframe = FRAME_FIRE_FIRST;
				ent->client->weaponstate = WEAPON_FIRING;

				// start the animation
				ent->client->anim_priority = ANIM_ATTACK;
				ent->s.frame = FRAME_attack1-1;
				ent->client->anim_end = FRAME_attack8;
			}
			else
			{
				if (level.time >= ent->pain_debounce_time)
				{
					gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
					ent->pain_debounce_time = level.time + 1;
				}
				NoAmmoWeaponChange (ent);
			}
		}
		else
		{
			if (ent->client->ps.gunframe == FRAME_IDLE_LAST)
			{
				ent->client->ps.gunframe = FRAME_IDLE_FIRST;
				return;
			}

			if (pause_frames)
			{
				for (n = 0; pause_frames[n]; n++)
				{
					if (ent->client->ps.gunframe == pause_frames[n])
					{
						if (rand()&15)
							return;
					}
				}
			}

			ent->client->ps.gunframe++;
			return;
		}
	}

	if (ent->client->weaponstate == WEAPON_FIRING)
	{
		for (n = 0; fire_frames[n]; n++)
		{
			if (ent->client->ps.gunframe == fire_frames[n])
			{
				fire (ent);
				break;
			}
		}

		if (!fire_frames[n])
			ent->client->ps.gunframe++;

		if (ent->client->ps.gunframe == FRAME_IDLE_FIRST+1)
			ent->client->weaponstate = WEAPON_READY;
	}
}

/*
======================================================================

MISSILES (6-0)

======================================================================
*/

void Weapon_Concussion_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 120;
	damage_radius = 120;

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);
	fire_rocket (ent, start, forward, damage, 850, damage_radius, radius_damage);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/concussion.wav"), 1, ATTN_NORM, 0);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Concussion (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {5, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 5, 8, 10, pause_frames, fire_frames, Weapon_Concussion_Fire);
}

void Weapon_Flash_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 50;
	damage_radius = 50;

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);
	fire_flash (ent, start, forward, damage, 850, damage_radius, radius_damage);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/concussion.wav"), 1, ATTN_NORM, 0);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Flash (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {5, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 5, 8, 10, pause_frames, fire_frames, Weapon_Flash_Fire);
}

void Weapon_Homing_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 150;
	damage_radius = 150;

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);
	fire_homing (ent, start, forward, damage, 750, damage_radius, radius_damage);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/concussion.wav"), 1, ATTN_NORM, 0);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Homing (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {5, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 5, 8, 10, pause_frames, fire_frames, Weapon_Homing_Fire);
}

void Weapon_Cluster_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	damage = 100 + (int)(random() * 20.0);
	radius_damage = 150;
	damage_radius = 150;

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);
	fire_cluster (ent, start, forward, damage, 750, damage_radius, radius_damage);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/concussion.wav"), 1, ATTN_NORM, 0);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Cluster (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {5, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 5, 8, 10, pause_frames, fire_frames, Weapon_Cluster_Fire);
}

void Weapon_Mega_Fire (edict_t *ent)
{
	vec3_t	offset, start;
	vec3_t	forward, right;
	int		damage;
	float	damage_radius;
	int		radius_damage;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	damage = 200;
	radius_damage = 400;
	damage_radius = 1000;

	AngleVectors (ent->client->v_angle, forward, right, NULL);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	VectorSet(offset, 8, 8, ent->viewheight-8);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);
	fire_mega (ent, start, forward, damage, 450, damage_radius, radius_damage);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/mega.wav"), 1, ATTN_NORM, 0);

	ent->client->ps.gunframe++;

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]-=75;
}

void Weapon_Mega (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {5, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 20, 22, 24, pause_frames, fire_frames, Weapon_Mega_Fire);
}


/*
======================================================================

LASER WEAPONRY (1-3)

======================================================================
*/
// Laser Beam Color Codes
#define Laser_Red 0xf2f2f0f0 // bright red
#define Laser_Green 0xd0d1d2d3 // bright green
#define Laser_Blue 0xf3f3f1f1 // bright blue
#define Laser_Yellow 0xdcdddedf // bright yellow
#define Laser_YellowS 0xe0e1e2e3 // yellow strobe
#define Laser_DkPurple 0x80818283 // dark purple
#define Laser_LtBlue 0x70717273 // light blue
#define Laser_Green2 0x90919293 // different green
#define Laser_Purple 0xb0b1b2b3 // purple
#define Laser_Red2 0x40414243 // different red
#define Laser_Orange 0xe2e5e3e6 // orange
#define Laser_Mix 0xd0f1d3f3 // mixture
#define Laser_RedBlue 0xf2f3f0f1 // inner = red, outer = blue
#define Laser_BlueRed 0xf3f2f1f0 // inner = blue, outer = red
#define Laser_GreenY 0xdad0dcd2 // inner = green, outer = yellow
#define Laser_YellowG 0xd0dad2dc // inner = yellow, outer = green

void laser1_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t movdir;
	int		mod;

	if (!plane)
		VectorCopy(vec3_origin, movdir);
	else
		VectorCopy(plane->normal, movdir);

	if (other == self->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (self);
		return;
	}

	if (self->owner->client)
		PlayerNoise(self->owner, self->s.origin, PNOISE_IMPACT);

	if (other->takedamage)
	{
		mod = MOD_LASER;
		T_Damage (other, self, self->owner, self->velocity, self->s.origin, plane->normal, self->dmg, 1, DAMAGE_ENERGY, mod);
	}
	else
	{
		gi.sound (self, CHAN_VOICE, gi.soundindex ("weapons/lashit.wav"), 1, ATTN_NORM, 0);
		G_Spawn_Splash(TE_LASER_SPARKS, 16, Laser_Yellow, self->s.origin, movdir, self->s.origin);
	}

	G_FreeEdict (self);
}


void fire_laser1 (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, int effect)
{
	edict_t	*bolt;
	trace_t	tr;

	if (!self->client)
		return;

	VectorNormalize (dir);
    bolt = G_Spawn();
    bolt->owner = self;
    bolt->classname= "laser1";
    bolt->movetype = MOVETYPE_FLYMISSILE;
	bolt->clipmask = MASK_SHOT;
	bolt->solid = SOLID_BBOX;
    bolt->svflags &= ~SVF_NOCLIENT;
    bolt->s.renderfx = RF_BEAM|RF_FULLBRIGHT;
    bolt->s.modelindex = 1;
    bolt->spawnflags = 16; // Store Length here..
    bolt->s.skinnum = Laser_Yellow;
    bolt->s.sound = gi.soundindex ("world/laser.wav");
    bolt->s.frame=2;
	VectorSet (bolt->mins, -8, -8, -8);
	VectorSet (bolt->maxs, 8, 8, 8);
    bolt->dmg = damage;
    VectorClear(bolt->s.angles);
    VectorCopy(dir, bolt->movedir);
	VectorScale (dir, speed, bolt->velocity);
    VectorCopy(start, bolt->s.origin);
    VectorCopy(start, bolt->s.old_origin);
	bolt->touch = laser1_touch;
	bolt->nextthink = level.time + 2;
	bolt->think = G_FreeEdict;
    gi.linkentity(bolt);

	tr = gi.trace (self->s.origin, NULL, NULL, bolt->s.origin, bolt, MASK_SHOT);
	if (tr.fraction < 1.0)
	{
		VectorMA (bolt->s.origin, -10, dir, bolt->s.origin);
		bolt->touch (bolt, tr.ent, NULL, NULL);
	}
}

void Laser1_Fire (edict_t *ent, vec3_t g_offset, int damage, int effect)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	AngleVectors (ent->client->v_angle, forward, right, NULL);
	VectorSet(offset, 24, 8, ent->viewheight-8);
	VectorAdd (offset, g_offset, offset);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	fire_laser1 (ent, start, forward, damage, 2000, effect);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/laser01.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);
}

void Weapon_Laser1_Fire (edict_t *ent)
{
	int		damage;

	if (!ent->client)
		return;

	damage = 15;
	Laser1_Fire (ent, vec3_origin, damage, 0);
	ent->client->ps.gunframe++;
}

void Weapon_Laser1 (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {4, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 4, 7, 9, pause_frames, fire_frames, Weapon_Laser1_Fire);
}

void laser2_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t movdir;
	int		mod;

	if (!plane)
		VectorCopy(vec3_origin, movdir);
	else
		VectorCopy(plane->normal, movdir);

	if (other == self->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (self);
		return;
	}

	if (self->owner->client)
		PlayerNoise(self->owner, self->s.origin, PNOISE_IMPACT);

	if (other->takedamage)
	{
		mod = MOD_LASER;
		T_Damage (other, self, self->owner, self->velocity, self->s.origin, plane->normal, self->dmg, 1, DAMAGE_ENERGY, mod);
	}
	else
	{
		gi.sound (self, CHAN_VOICE, gi.soundindex ("weapons/lashit.wav"), 1, ATTN_NORM, 0);
		G_Spawn_Splash(TE_LASER_SPARKS, 16, Laser_Green, self->s.origin, movdir, self->s.origin);
	}

	G_FreeEdict (self);
}


void fire_laser2 (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, int effect)
{
	edict_t	*bolt;
	trace_t	tr;

	if (!self->client)
		return;

	VectorNormalize (dir);
    bolt = G_Spawn();
    bolt->owner = self;
    bolt->classname= "laser2";
    bolt->movetype = MOVETYPE_FLYMISSILE;
	bolt->clipmask = MASK_SHOT;
	bolt->solid = SOLID_BBOX;
    bolt->svflags &= ~SVF_NOCLIENT;
    bolt->s.renderfx = RF_BEAM|RF_FULLBRIGHT;
    bolt->s.modelindex = 1;
    bolt->spawnflags = 16; // Store Length here..
    bolt->s.skinnum = Laser_Green;
    bolt->s.sound = gi.soundindex ("world/laser.wav");
    bolt->s.frame=2;
	VectorSet (bolt->mins, -8, -8, -8);
	VectorSet (bolt->maxs, 8, 8, 8);
    bolt->dmg = damage;
    VectorClear(bolt->s.angles);
    VectorCopy(dir, bolt->movedir);
	VectorScale (dir, speed, bolt->velocity);
    VectorCopy(start, bolt->s.origin);
    VectorCopy(start, bolt->s.old_origin);
	bolt->touch = laser2_touch;
	bolt->nextthink = level.time + 10;
	bolt->think = G_FreeEdict;
    gi.linkentity(bolt);

	tr = gi.trace (self->s.origin, NULL, NULL, bolt->s.origin, bolt, MASK_SHOT);
	if (tr.fraction < 1.0)
	{
		VectorMA (bolt->s.origin, -10, dir, bolt->s.origin);
		bolt->touch (bolt, tr.ent, NULL, NULL);
	}
}

void Laser2_Fire (edict_t *ent, vec3_t g_offset, int damage, int effect)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	AngleVectors (ent->client->v_angle, forward, right, NULL);
	VectorSet(offset, 24, 8, ent->viewheight-8);
	VectorAdd (offset, g_offset, offset);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	fire_laser2 (ent, start, forward, damage, 2000, effect);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/laser02.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Laser2_Fire (edict_t *ent)
{
	int		damage;

	if (!ent->client)
		return;

	damage = 20;
	Laser2_Fire (ent, vec3_origin, damage, 0);
	ent->client->ps.gunframe++;
}

void Weapon_Laser2 (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {4, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 4, 7, 9, pause_frames, fire_frames, Weapon_Laser2_Fire);
}

void laser3_touch (edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf)
{
	vec3_t movdir;
	int		mod;

	if (!plane)
		VectorCopy(vec3_origin, movdir);
	else
		VectorCopy(plane->normal, movdir);

	if (other == self->owner)
		return;

	if (surf && (surf->flags & SURF_SKY))
	{
		G_FreeEdict (self);
		return;
	}

	if (self->owner->client)
		PlayerNoise(self->owner, self->s.origin, PNOISE_IMPACT);

	if (other->takedamage)
	{
		mod = MOD_LASER;
		T_Damage (other, self, self->owner, self->velocity, self->s.origin, plane->normal, self->dmg, 1, DAMAGE_ENERGY, mod);
	}
	else
	{
		gi.sound (self, CHAN_VOICE, gi.soundindex ("weapons/lashit.wav"), 1, ATTN_NORM, 0);
		G_Spawn_Splash(TE_LASER_SPARKS, 16, Laser_Red, self->s.origin, movdir, self->s.origin);
	}

	G_FreeEdict (self);
}


void fire_laser3 (edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, int effect)
{
	edict_t	*bolt;
	trace_t	tr;

	if (!self->client)
		return;

	VectorNormalize (dir);
    bolt = G_Spawn();
    bolt->owner = self;
    bolt->classname= "laser3";
    bolt->movetype = MOVETYPE_FLYMISSILE;
	bolt->clipmask = MASK_SHOT;
	bolt->solid = SOLID_BBOX;
    bolt->svflags &= ~SVF_NOCLIENT;
    bolt->s.renderfx = RF_BEAM|RF_FULLBRIGHT;
    bolt->s.modelindex = 1;
    bolt->spawnflags = 16; // Store Length here..
    bolt->s.skinnum = Laser_Red;
    bolt->s.sound = gi.soundindex ("world/laser.wav");
    bolt->s.frame=2;
	VectorSet (bolt->mins, -8, -8, -8);
	VectorSet (bolt->maxs, 8, 8, 8);
    bolt->dmg = damage;
    VectorClear(bolt->s.angles);
    VectorCopy(dir, bolt->movedir);
	VectorScale (dir, speed, bolt->velocity);
    VectorCopy(start, bolt->s.origin);
    VectorCopy(start, bolt->s.old_origin);
	bolt->touch = laser3_touch;
	bolt->nextthink = level.time + 10;
	bolt->think = G_FreeEdict;
    gi.linkentity(bolt);

	tr = gi.trace (self->s.origin, NULL, NULL, bolt->s.origin, bolt, MASK_SHOT);
	if (tr.fraction < 1.0)
	{
		VectorMA (bolt->s.origin, -10, dir, bolt->s.origin);
		bolt->touch (bolt, tr.ent, NULL, NULL);
	}
}

void Laser3_Fire (edict_t *ent, vec3_t g_offset, int damage, int effect)
{
	vec3_t	forward, right;
	vec3_t	start;
	vec3_t	offset;
	static side aside=RIGHT;

	if (!ent->client)
		return;

	aside = SideToggle (aside);

	AngleVectors (ent->client->v_angle, forward, right, NULL);
	VectorSet(offset, 24, 8, ent->viewheight-8);
	VectorAdd (offset, g_offset, offset);
	N_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start, aside);

	VectorScale (forward, -2, ent->client->kick_origin);
	ent->client->kick_angles[0] = -1;

	fire_laser3 (ent, start, forward, damage, 2000, effect);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/laser03.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}

void Weapon_Laser3_Fire (edict_t *ent)
{
	int		damage;

	if (!ent->client)
		return;

	damage = 25;
	Laser3_Fire (ent, vec3_origin, damage, 0);
	ent->client->ps.gunframe++;
}

void Weapon_Laser3 (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {4, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 4, 7, 9, pause_frames, fire_frames, Weapon_Laser3_Fire);
}


/*
======================================================================

VULCAN CANNON / GAUSS CANNON

======================================================================
*/
void Vulcan_Fire (edict_t *ent)
{
	int			i;
	vec3_t		start;
	vec3_t		forward, right, up;
	float		r, u;
	vec3_t		offset;
	int			damage;
	int			kick = 2;

	if (!ent->client)
		return;

	damage = 6;

	if (!(ent->client->buttons & BUTTON_ATTACK))
	{
		ent->client->ps.gunframe = 5;
		ent->client->weapon_sound = 0;
		return;
	}
	else if ((ent->client->ps.gunframe == 4) && (ent->client->buttons & BUTTON_ATTACK)
		&& ent->client->pers.inventory[ent->client->ammo_index])
	{
		ent->client->ps.gunframe = 3;
	}
	else
	{
		ent->client->ps.gunframe++;
	}

	ent->client->anim_priority = ANIM_ATTACK;
	ent->s.frame = FRAME_attack1;
	ent->client->anim_end = FRAME_attack8;

	if (!ent->client->pers.inventory[ent->client->ammo_index])
	{
		if (level.time >= ent->pain_debounce_time)
		{
			gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
			ent->pain_debounce_time = level.time + 1;
		}
		NoAmmoWeaponChange (ent);
		return;
	}

	for (i=0 ; i<3 ; i++)
	{
		ent->client->kick_origin[i] = crandom() * 0.35;
		ent->client->kick_angles[i] = crandom() * 0.7;
	}

	// get start / end positions
	AngleVectors (ent->client->v_angle, forward, right, up);
	r = crandom()*3;
	u = crandom()*3;
	VectorSet(offset, 0, r, u + ent->viewheight-8);
	// i can use a normal p_source here, its not a projectile
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	fire_bullet (ent, start, forward, damage, kick, DEFAULT_BULLET_HSPREAD, DEFAULT_BULLET_VSPREAD, MOD_VULCAN);

	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/vulcan.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}


void Weapon_Vulcan (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {3, 4, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 4, 7, 9, pause_frames, fire_frames, Vulcan_Fire);
}


void Gauss_Fire (edict_t *ent)
{
	int			i;
	vec3_t		start;
	vec3_t		forward, right, up;
	float		r, u;
	vec3_t		offset;
	int			damage;
	int			kick = 2;

	if (!ent->client)
		return;

	damage = 12;

	if (!(ent->client->buttons & BUTTON_ATTACK))
	{
		ent->client->ps.gunframe = 5;
		ent->client->weapon_sound = 0;
		return;
	}
	else if ((ent->client->ps.gunframe == 4) && (ent->client->buttons & BUTTON_ATTACK)
		&& ent->client->pers.inventory[ent->client->ammo_index])
	{
		ent->client->ps.gunframe = 3;
	}
	else
	{
		ent->client->ps.gunframe++;
	}

	ent->client->anim_priority = ANIM_ATTACK;
	ent->s.frame = FRAME_attack1;
	ent->client->anim_end = FRAME_attack8;

	if (!ent->client->pers.inventory[ent->client->ammo_index])
	{
		if (level.time >= ent->pain_debounce_time)
		{
			gi.sound(ent, CHAN_VOICE, gi.soundindex("weapons/noammo.wav"), 1, ATTN_NORM, 0);
			ent->pain_debounce_time = level.time + 1;
		}
		NoAmmoWeaponChange (ent);
		return;
	}

	for (i=0 ; i<3 ; i++)
	{
		ent->client->kick_origin[i] = crandom() * 0.35;
		ent->client->kick_angles[i] = crandom() * 0.7;
	}

	// get start / end positions
	AngleVectors (ent->client->v_angle, forward, right, up);
	r = crandom()*3;
	u = crandom()*3;
	VectorSet(offset, 0, r, u + ent->viewheight-8);
	// i can use a normal p_source here, its not a projectile
	P_ProjectSource (ent->client, ent->s.origin, offset, forward, right, start);

	fire_bullet (ent, start, forward, damage, kick, DEFAULT_BULLET_HSPREAD, DEFAULT_BULLET_VSPREAD, MOD_GAUSS);
	
	// send muzzle flash
	gi.sound (ent, CHAN_VOICE, gi.soundindex ("craft/gauss.wav"), 1, ATTN_NORM, 0);

	PlayerNoise(ent, start, PNOISE_WEAPON);

	if (! ( (int)dmflags->value & DF_INFINITE_AMMO ) )
		ent->client->pers.inventory[ent->client->ammo_index]--;
}


void Weapon_Gauss (edict_t *ent)
{
	static int	pause_frames[]	= {0};
	static int	fire_frames[]	= {3, 4, 0};

	if (!ent->client)
		return;

	Weapon_Generic (ent, 2, 4, 7, 9, pause_frames, fire_frames, Gauss_Fire);
}
