#include "defines.h"
#include "m_player.h"

//======================================================
// Ghost keeps updating on who best player is to follow.
//======================================================
void Ghost_Think(edict_t *Ghost) {
	vec3_t up, start;
	edict_t *fPlayer=NULL;
	
	// Unlink Ghost if intermission..
	if (level.intermissiontime >= level.time) {
		gi.unlinkentity(Ghost);
		return; }
	
	// Check on scores every 1 sec.
	if (Ghost->delay<=level.time) {
		fPlayer=BestScoreEnt();
		Ghost->delay =level.time + 1.0; }
	
	// Move Ghost above fPlayer (high score player).
	if ((fPlayer!=NULL) && G_ClientInGame(fPlayer)) {
		Ghost->svflags &= ~SVF_NOCLIENT; // Turn ON Ghost
		AngleVectors(fPlayer->s.angles, NULL, NULL, up);
		VectorMA(fPlayer->s.origin, 60, up, start);
		VectorCopy(start, Ghost->s.origin); // Move Ghost
		gi.linkentity(Ghost); } // Must update Ghost!
	else
		// No fPlayer to follow so..
		// turn OFF Ghost temporarily
		Ghost->svflags &= SVF_NOCLIENT;
	
	// Keep checking every frame.
	Ghost->nextthink = level.time + 0.1;
}

//==================================================
// Create Ghost Entity
//==================================================
void Create_Ghost(void) {
	edict_t *Ghost;
	
	Ghost=G_Spawn();
	Ghost->owner=NULL;
	Ghost->s.effects=EF_TELEPORTER; // Particle effect.
	Ghost->model="";
	Ghost->takedamage=DAMAGE_NO;
	Ghost->movetype=MOVETYPE_NONE;
	Ghost->solid=SOLID_NOT;
	Ghost->s.modelindex = 0;
	Ghost->s.modelindex2 = 0;
	VectorClear(Ghost->mins);
	VectorClear(Ghost->maxs);
	Ghost->delay=0;
	Ghost->think=Ghost_Think;
	Ghost->nextthink = 1.0; // Start in 1 second.
	gi.linkentity(Ghost);
}

//=========================================================
//=========== Insure FOV is within limits =================
//=========================================================
void Check_FOV(edict_t *ent, char *userinfo) {
	
	if (CVAR_DEATHMATCH && (int)dmflags->value & DF_FIXED_FOV)
		ent->client->ps.fov=90;
	else {
		ent->client->ps.fov=atoi(Info_ValueForKey(userinfo, "fov"));
		if (ent->client->ps.fov < 30)
			ent->client->ps.fov=90;
		else if (ent->client->ps.fov > 160)
			ent->client->ps.fov=160; } // Max FOV for deathmatch mode.
}

//============================================================
int PlayerSort(void const *a, void const *b) {
	int anum, bnum;
	
	anum=*(int *)a;
	bnum=*(int *)b;
	
	anum=ga.clients[anum].ps.stats[STAT_FRAGS];
	bnum=ga.clients[bnum].ps.stats[STAT_FRAGS];
	
	if (anum < bnum) return -1;
	if (anum > bnum) return 1;
	
	return 0;
}

//=========================================================
void SP_FixCoopSpots(edict_t *self) {
	edict_t *spot=NULL;
	vec3_t d;
	
	while (1) {
		spot=G_Find(spot, FOFS(classname), "info_player_start");
		if (!spot) return;
		if (!spot->targetname) continue;
		VectorSubtract(self->s.origin, spot->s.origin, d);
		if (VectorLength(d) < 384) {
			if ((!self->targetname) || Q_stricmp(self->targetname, spot->targetname))
				self->targetname=spot->targetname;
			return; } }
}

//=========================================================
// now if that one wasn't ugly enough for you then try this
// one on for size some maps don't have any coop spots at all,
// so we need to create them where they should have been
//=========================================================
void SP_CreateCoopSpots(edict_t *self) {
	edict_t *spot;
	
	if (!Q_stricmp(level.mapname, "security")) {
		spot=G_Spawn();
		spot->classname="info_player_coop";
		spot->s.origin[0]=188-64;
		spot->s.origin[1]= -164;
		spot->s.origin[2]=80;
		spot->targetname="jail3";
		spot->s.angles[1]=90;
		spot=G_Spawn();
		spot->classname="info_player_coop";
		spot->s.origin[0]=188+64;
		spot->s.origin[1]= -164;
		spot->s.origin[2]=80;
		spot->targetname="jail3";
		spot->s.angles[1]=90;
		spot=G_Spawn();
		spot->classname="info_player_coop";
		spot->s.origin[0]=188+128;
		spot->s.origin[1]= -164;
		spot->s.origin[2]=80;
		spot->targetname="jail3";
		spot->s.angles[1]=90;
		return; }
}

//=========================================================
void SP_info_player_start(edict_t *self) {
	
	if (!CVAR_COOP) return;
	
	if (!Q_stricmp(level.mapname, "security")) {
		self->think=SP_CreateCoopSpots;
		self->nextthink=level.time+FRAMETIME; }
}

void SP_misc_teleporter_dest(edict_t *self);

//=========================================================
void SP_info_player_deathmatch(edict_t *self) {
	
	if (!CVAR_DEATHMATCH) {
		G_FreeEdict(self);
		return; }
	
	SP_misc_teleporter_dest(self);
}

//=========================================================
void SP_info_player_coop(edict_t *self) {
	
	if (!CVAR_COOP) {
		G_FreeEdict(self);
		return; }
	
	if ((!Q_stricmp(level.mapname, "jail2"))   ||
		(!Q_stricmp(level.mapname, "jail4"))   ||
		(!Q_stricmp(level.mapname, "mine1"))   ||
		(!Q_stricmp(level.mapname, "mine2"))   ||
		(!Q_stricmp(level.mapname, "mine3"))   ||
		(!Q_stricmp(level.mapname, "mine4"))   ||
		(!Q_stricmp(level.mapname, "lab"))     ||
		(!Q_stricmp(level.mapname, "boss1"))   ||
		(!Q_stricmp(level.mapname, "fact3"))   ||
		(!Q_stricmp(level.mapname, "biggun"))  ||
		(!Q_stricmp(level.mapname, "space"))   ||
		(!Q_stricmp(level.mapname, "command")) ||
		(!Q_stricmp(level.mapname, "power2")) ||
		(!Q_stricmp(level.mapname, "strike"))) {
		// invoke one of our gross, ugly, disgusting hacks
		self->think=SP_FixCoopSpots;
		self->nextthink=level.time+FRAMETIME; }
}

//=========================================================
qboolean IsFemale(edict_t *ent) {
	char *info;
	if (!G_EntExists(ent)) return false;
	info=Info_ValueForKey(ent->client->pers.userinfo, "gender");
	return (info[0] == 'f' || info[0] == 'F');
}

//=========================================================
qboolean IsNeutral(edict_t *ent) {
	char *info;
	if (!G_EntExists(ent)) return false;
	info=Info_ValueForKey(ent->client->pers.userinfo, "gender");
	return (info[0] != 'f' && info[0] != 'F' && info[0] != 'm' && info[0] != 'M');
}

//======================================================
// Frag award and Obits for Victims of Monster attacks!
// Attacker=Monster
//======================================================
qboolean Monster_Obits(edict_t *victim, edict_t *attacker) {
	char *message1="";
	char *message2="";
	
	if (victim==attacker->activator) return false;
	
	// No obits in single-player mode.
	if (!MonstersInUse)
		if (!CVAR_DEATHMATCH)
			return false;
		
		// Make sure both attacker and victim still in game!
		if (!(G_EntExists(attacker->activator) && G_EntExists(victim)))
			return false;
		
		message1="was killed by";
		
		// What type of monster was this?
		switch (attacker->mtype) {
		case M_BERSERK:
			message2="'s Berserker";
			break;
		case M_BOSS2:
			message2="'s Boss";
			break;
		case M_SOLDIERSS:
			message2="'s SoldierSS";
			break;
		case M_JORG:
			message2="'s Jorg";
			break;
		case M_BRAIN:
			message2="'s Brain";
			break;
		case M_CHICK:
			message2="'s Chick";
			break;
		case M_FLIPPER:
			message2="'s Shark";
			break;
		case M_FLOATER:
			message2="'s Floater";
			break;
		case M_FLYER:
			message2="'s Flyer";
			break;
		case M_INSANE:
			message2="'s Insane";
			break;
		case M_GLADIATOR:
			message2="'s Gladiator";
			break;
		case M_HOVER:
			message2="'s Icarus";
			break;
		case M_INFANTRY:
			message2="'s Infantry";
			break;
		case M_SOLDIERLT:
			message2="'s SoldierLT";
			break;
		case M_SOLDIER:
			message2="'s Soldier";
			break;
		case M_MEDIC:
			message2="'s Medic";
			break;
		case M_MUTANT:
			message2="'s Mutant";
			break;
		case M_PARASITE:
			message2="'s Parasite";
			break;
		case M_TANK:
			message2="'s Tank";
			break;
		case M_MAKRON:
			message2="'s Makron";
			break;
		case M_GUNNER:
			message2="'s Gunner";
			break;
		case M_SUPERTANK:
			message2="'s Supertank";
			break;
		default:
			return false;
		} // end switch
		
		// Print the obituary message..
		gi_bprintf(PRINT_MEDIUM,"%s %s %s %s\n",
			victim->client->pers.netname,
			message1,
			attacker->activator->client->pers.netname,
			message2);
		
		// Give monster owner this frag!!
		attacker->activator->client->resp.score++;
		
		return true;
}

//=========================================================
void ClientObituary(edict_t *self, edict_t *inflictor, edict_t *attacker) {
	int mod;
	char *message=NULL;
	char *message2="";
	qboolean ff=false;
	
	// If no victim then no obit!
	if (!G_EntExists(self)) return;
	
	// Make sure both attacker and victim still in game!
	if ((deathmatch->value) && (G_EntExists(attacker)) && (G_EntExists(self)))
		// Was the attacker a Flyer?
		if (!Q_stricmp(attacker->classname, "XFlyer"))
			// Flyer's rider still exists?
			if (G_EntExists(attacker->rider)) {
				// Assign Flyer's kills to Flyer's Owner's score!!
				gi_bprintf(PRINT_MEDIUM,"%s was nailed by %s's Flyer\n",self->client->pers.netname,attacker->rider->client->pers.netname);
				attacker->rider->client->resp.score += 1;
				return; }
			
			if (Monster_Obits(self, attacker)) return;
			
			if (CVAR_COOP && attacker->client)
				meansOfDeath |= MOD_FRIENDLY_FIRE;
			
			if (CVAR_DEATHMATCH || CVAR_COOP) {
				ff=meansOfDeath & MOD_FRIENDLY_FIRE;
				mod=meansOfDeath & ~MOD_FRIENDLY_FIRE;
				
				switch (mod) {
				case MOD_SUICIDE:
					message="suicides";
					break;
				case MOD_FALLING:
					message="cratered";
					break;
				case MOD_CRUSH:
					message="was squished";
					break;
				case MOD_WATER:
					message="sank like a rock";
					break;
				case MOD_SLIME:
					message="melted";
					break;
				case MOD_LAVA:
					message="does a back flip into the lava";
					break;
				case MOD_EXPLOSIVE:
				case MOD_BARREL:
					message="blew up";
					break;
				case MOD_EXIT:
					message="found a way out";
					break;
				case MOD_TARGET_LASER:
					message="saw the light";
					break;
				case MOD_TARGET_BLASTER:
					message="got blasted";
					break;
				case MOD_BOMB:
				case MOD_SPLASH:
				case MOD_TRIGGER_HURT:
					message="was in the wrong place";
					break;
				case MOD_HGRENADE:
					message="caught a homing grenade";
					break;
				case MOD_RAILBOMB:
					message="was railbombed";
					break;
				case MOD_ZYLON_GAS:
					message="inhaled Zylon gas";
					break;
				case MOD_HANGED:
					message="was strung up";
					break;
				case MOD_DEATHPAK:
					message="picked up a deathpak";
					break;
				case MOD_LASERDRONE:
					message="was railed by the drone";
					break;
				case MOD_CHAMBER:
					message="was tortured to death";
					break;
				case MOD_LASERSWEEP:
					message="cut in half by lasersweep";
					break;
				case MOD_BOOBYTRAP:
					message="was killed by a booby trap";
					break;
				case MOD_BATON:
					message="was killed by a baton";
					break;
				case MOD_DEPTHCHARGE:
					message="got obliterated";
					break;
				case MOD_CLUSTER_BOMBS:
					message="caught some Cluster Bombs.";
					break;
				case MOD_ROCKET_BOMBS:
					message="ate some Rocket Bombs.";
					break;
				case MOD_BFG_NUKE:
					message="bought a BFG Nuke.";
					break;
				case MOD_LASERSTRIKE:
					message="was killed by a LaserStrike";
					break;
				case MOD_RAILSTRIKE:
					message="was killed by a RailStrike";
					break;
				} // end switch
				
				if (attacker == self)
					switch (mod) {
				case MOD_HELD_GRENADE:
					message="tried to put the pin back in";
					break;
				case MOD_HG_SPLASH:
				case MOD_G_SPLASH:
					if (IsNeutral(self))
						message="tripped on its own grenade";
					else if (IsFemale(self))
						message="tripped on her own grenade";
					else
						message="tripped on his own grenade";
					break;
				case MOD_R_SPLASH:
					if (IsNeutral(self))
						message="blew itself up";
					else if (IsFemale(self))
						message="blew herself up";
					else
						message="blew himself up";
					break;
				case MOD_BFG_BLAST:
					message="should have used a smaller gun";
					break;
				case MOD_HGRENADE:
					message="caught a homing grenade";
					break;
				case MOD_RAILBOMB:
					if (IsFemale(self))
						message="got riveted by her railbomb";
					else
						message="got riveted by his railbomb";
					break;
				case MOD_ZYLON_GAS:
					message="inhaled Zylon gas";
					break;
				case MOD_HANGED:
					message="was hanged";
					break;
				case MOD_DEATHPAK:
					message="picked up a deathpak";
					break;
				case MOD_LASERDRONE:
					message="was railed by a LaserDrone";
					break;
				case MOD_CHAMBER:
					message="died under torture";
					break;
				case MOD_LASERSWEEP:
					message="was halved by a lasersweep";
					break;
				case MOD_BOOBYTRAP:
					if (IsFemale(self))
						message="caught her booby trap";
					else
						message="caught his booby trap";
					break;
				case MOD_BATON:
					message="drilled by the baton";
					break;
				case MOD_DECOY:
					if (IsFemale(self))
						message="bought her own Decoy!";
					else
						message="bought his own Decoy!";
					break;
				case MOD_DEPTHCHARGE:
					message="got depth charged";
					break;
				case MOD_CLUSTER_BOMBS:
					if (IsFemale(self))
						message="caught her own Cluster Bombs.";
					else
						message="caught his own Cluster Bombs.";
					break;
				case MOD_ROCKET_BOMBS:
					if (IsFemale(self))
						message="caught her own Rocket Bombs.";
					else
						message="caught his own Rocket Bombs.";
					break;
				case MOD_BFG_NUKE:
					if (IsFemale(self))
						message="caught her own BFG Nuke.";
					else
						message="caught his own BFG Nuke.";
					break;
				case MOD_LASERSTRIKE:
					message="got struck by a LaserStrike";
					break;
				case MOD_RAILSTRIKE:
					message="got struck by a RailStrike";
					break;
				default:
					if (IsNeutral(self))
						message="killed itself";
					else if (IsFemale(self))
						message="killed herself";
					else
						message="killed himself";
					break;
        } // end switch
		
		if (message) {
			gi_bprintf(PRINT_MEDIUM, "%s %s.\n", self->client->pers.netname, message);
			if (CVAR_DEATHMATCH)
				self->client->resp.score--;
			self->enemy=NULL;
			return; }
		
		self->enemy=attacker;
		
		if (G_EntExists(attacker)) {
			switch (mod) {
			case MOD_BLASTER:
				message="was blasted by";
				break;
			case MOD_SHOTGUN:
				message="was gunned down by";
				break;
			case MOD_SSHOTGUN:
				message="was blown away by";
				message2="'s super shotgun";
				break;
			case MOD_MACHINEGUN:
				message="was machinegunned by";
				break;
			case MOD_CHAINGUN:
				message="was cut in half by";
				message2="'s chaingun";
				break;
			case MOD_GRENADE:
				message="was popped by";
				message2="'s grenade";
				break;
			case MOD_G_SPLASH:
				message="was shredded by";
				message2="'s shrapnel";
				break;
			case MOD_ROCKET:
				message="ate";
				message2="'s rocket";
				break;
			case MOD_R_SPLASH:
				message="almost dodged";
				message2="'s rocket";
				break;
			case MOD_HYPERBLASTER:
				message="was melted by";
				message2="'s hyperblaster";
				break;
			case MOD_RAILGUN:
				message="was railed by";
				break;
			case MOD_BFG_LASER:
				message="saw the pretty lights from";
				message2="'s BFG";
				break;
			case MOD_BFG_BLAST:
				message="was disintegrated by";
				message2="'s BFG blast";
				break;
			case MOD_BFG_EFFECT:
				message="couldn't hide from";
				message2="'s BFG";
				break;
			case MOD_HANDGRENADE:
				message="caught";
				message2="'s handgrenade";
				break;
			case MOD_HG_SPLASH:
				message="didn't see";
				message2="'s handgrenade";
				break;
			case MOD_DECOY:
				message="was blown to pieces by";
				message2="'s Decoy";
				break;
			case MOD_HELD_GRENADE:
				message="feels";
				message2="'s pain";
				break;
			case MOD_BOOBYTRAP:
				message="was killed by";
				message2="'s booby trap";
				break;
			case MOD_BATON:
				message="was killed by";
				message2="'s baton";
				break;
			case MOD_DEATHPAK:
				message="picked up";
				message2="'s deathpak";
				break;
			case MOD_HANGED:
				message="was hanged by";
				message2="'s rope";
				break;
			case MOD_RAILBOMB:
				message="was riveted by";
				message2="'s railbomb";
				break;
			case MOD_ZYLON_GAS:
				message="was ovecome by";
				message2="'s grenade gas";
				break;
			case MOD_LASERDRONE:
				message="was railed by";
				message2="'s drone";
				break;
			case MOD_CHAMBER:
				message="died in";
				message2="'s torture chamber";
				break;
			case MOD_LASERSWEEP:
				message="was cut in half by";
				message2="'s lasersweep";
				break;
			case MOD_HGRENADE:
				message="caught";
				message2="'s homing grenade";
				break;
			case MOD_TELEFRAG:
				message="tried to invade";
				message2="'s personal space";
				break;
			case MOD_DEPTHCHARGE:
				message="was killed by";
				message2="'s depth charge";
				break;
			case MOD_CLUSTER_BOMBS:
				message="was shredded by";
				message2="'s Cluster bombs";
				break;
			case MOD_ROCKET_BOMBS:
				message="swallowed";
				message2="'s Rocket bombs";
				break;
			case MOD_BFG_NUKE:
				message="was obliterated by";
				message2="'s Nuke airstrike";
				break;
			case MOD_LASERSTRIKE:
				message="was blasted by";
				message2="'s LaserStrike";
				break;
			case MOD_RAILSTRIKE:
				message="was toasted by";
				message2="'s RailStrike";
				break;
        } // end switch
		
		if (message) {
			gi_bprintf(PRINT_MEDIUM,"%s %s %s%s\n", self->client->pers.netname, message, attacker->client->pers.netname, message2);
			if (CVAR_DEATHMATCH) {
				if (ff)
					attacker->client->resp.score--;
				else
					attacker->client->resp.score++; }
			return; }
      } // endif
    } // endif
	
	gi_bprintf(PRINT_MEDIUM,"%s died.\n", self->client->pers.netname);
	if (CVAR_DEATHMATCH)
		self->client->resp.score--;
}

//=========================================================
void Touch_Item(edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf);

//=========================================================
void TossClientWeapon(edict_t *self) {
	gitem_t *item;
	edict_t *drop;
	qboolean quad;
	float spread;
	
	if (!CVAR_DEATHMATCH) return;
	
	item=self->client->pers.weapon;
	if (!self->client->pers.inventory[self->client->ammo_index])
		item=NULL;
	if (item && (!Q_stricmp(item->pickup_name, "Blaster")))
		item=NULL;
	
	if (!((int)(dmflags->value) & DF_QUAD_DROP))
		quad=false;
	else
		quad = (self->client->quad_framenum >(level.framenum+10));
	
	spread=(item && quad)?22.5:0.0;
	
	if (item) {
		self->client->v_angle[YAW] -= spread;
		drop=Drop_Item(self, item);
		self->client->v_angle[YAW] += spread;
		drop->spawnflags=DROPPED_PLAYER_ITEM; }
	
	if (quad) {
		self->client->v_angle[YAW] += spread;
		drop=Drop_Item(self, item_quad);
		self->client->v_angle[YAW] -= spread;
		drop->spawnflags |= DROPPED_PLAYER_ITEM;
		drop->touch=Touch_Item;
		drop->nextthink=level.time+(self->client->quad_framenum-level.framenum)*FRAMETIME;
		drop->think=G_FreeEdict; }
}

//=========================================================
void LookAtKiller(edict_t *self, edict_t *inflictor, edict_t *attacker) {
	vec3_t dir;
	
	if (attacker && attacker != world && attacker != self)
		VectorSubtract(attacker->s.origin, self->s.origin, dir);
	else if (inflictor && inflictor != world && inflictor != self)
		VectorSubtract(inflictor->s.origin, self->s.origin, dir);
	else {
		self->client->killer_yaw=self->s.angles[YAW];
		return; }
	
	if (dir[0])
		self->client->killer_yaw=180/M_PI*atan2(dir[1], dir[0]);
	else {
		self->client->killer_yaw=0;
		if (dir[1] > 0)
			self->client->killer_yaw=90;
		else if (dir[1] < 0)
			self->client->killer_yaw=-90; }
	
	if (self->client->killer_yaw < 0)
		self->client->killer_yaw += 360;
}

//=========================================================
// Dummy player_pain function
//=========================================================
void player_pain(edict_t *self, edict_t *other, float kick, int damage)
{}

//=========================================================
// Dummy player_touch function
//=========================================================
void player_touch(edict_t *ent, edict_t *other, cplane_t *plane, csurface_t *surf)
{}

//=========================================================
// Dummy player_use function
//=========================================================
void player_use(edict_t *button, edict_t *other, edict_t *activator)
{}

//=========================================================
void player_die(edict_t *self, edict_t *inflictor, edict_t *attacker, int damage, vec3_t point) {
	int n;
	static int i;
	
	// if no-one died, then exit..
	if (!G_EntExists(self)) return;
	
	VectorClear(self->avelocity);
	
	self->takedamage=DAMAGE_YES;
	self->movetype=MOVETYPE_TOSS;
	
	self->s.modelindex2=0;  // remove linked weapon model
	
	self->s.angles[0]=0;
	self->s.angles[2]=0;
	
	self->s.sound=0;
	self->client->weapon_sound=0;
	
	self->maxs[2]= -8;
	
	self->svflags |= SVF_DEADMONSTER;
	
	if (self->deadflag==DEAD_NO) {
		self->client->respawn_time=level.time+1.0;
		LookAtKiller(self, inflictor, attacker);
		self->client->ps.pmove.pm_type=PM_DEAD;
		ClientObituary(self, inflictor, attacker);
		TossClientWeapon(self);
		if (CVAR_DEATHMATCH)
			Cmd_Help_f(self); // show scores
		// clear inventory
		// this is kind of ugly, but it's how we want to handle keys in coop
		for (n=0; n < ga.num_items; n++) {
			if (CVAR_COOP && itemlist[n].flags & IT_KEY)
				self->client->resp.coop_respawn.inventory[n]=self->client->pers.inventory[n];
			self->client->pers.inventory[n]=0; } }
	
	// remove powerups
	self->client->quad_framenum=0;
	self->client->invincible_framenum=0;
	self->client->breather_framenum=0;
	self->client->enviro_framenum=0;
	self->client->cloak_framenum=0;
	self->client->is_cloaked=false;
	self->flags &= ~FL_POWER_ARMOR;
	
	if (self->health < -40) {
		gi.sound(self, CHAN_BODY, gi.soundindex("misc/udeath.wav"), 1, ATTN_NORM, 0);
		for (n= 0; n < 4; n++)
			ThrowGib(self, "models/objects/gibs/sm_meat/tris.md2", damage, GIB_ORGANIC);
		ThrowClientHead(self, damage);
		self->takedamage=DAMAGE_NO; }
	else
		// normal death
		if (self->deadflag==DEAD_NO) {
			i = (i+1)%3;
			// start a death animation
			self->client->anim_priority=ANIM_DEATH;
			if (self->client->ps.pmove.pm_flags & PMF_DUCKED){
				self->s.frame=FRAME_crdeath1-1;
				self->client->anim_end=FRAME_crdeath5; }
			else
				switch (i) {
          case 0: self->s.frame=FRAME_death101-1;
			  self->client->anim_end=FRAME_death106;
			  break;
          case 1: self->s.frame=FRAME_death201-1;
			  self->client->anim_end=FRAME_death206;
			  break;
          case 2: self->s.frame=FRAME_death301-1;
			  self->client->anim_end=FRAME_death308;
			} // end switch
			gi.sound(self, CHAN_VOICE, gi.soundindex(va("*death%i.wav",(rand()%4)+1)), 1, ATTN_NORM, 0); }
		
		self->deadflag=DEAD_DEAD;
		
		self->touch_debounce_time=0;
		self->client->id_on=OFF;
		self->client->railbomb=OFF;
		self->client->baton=OFF;
		self->client->chamber=OFF;
		self->client->laserdrone=OFF;
		self->client->Tgrenade=OFF;
		self->client->is_trapped=OFF;
		
		gi.linkentity(self);
}

//=========================================================
void InitClientPersistant(gclient_t *client) {
	gitem_t *item;
	
	memset(&client->pers, 0, sizeof(client->pers));
	
	item=item_blaster;
	
	client->pers.selected_item=ITEM_INDEX(item);
	client->pers.inventory[client->pers.selected_item]=1;
	client->pers.weapon=item;
	client->pers.health       = 100;
	client->pers.max_health   = 100;
	client->pers.max_bullets  = 200;
	client->pers.max_shells   = 100;
	client->pers.max_rockets  =  50;
	client->pers.max_grenades =  50;
	client->pers.max_cells    = 200;
	client->pers.max_slugs    =  50;
	
	client->hookstate=HOOK_OFF;   // default to Off
	
	client->cloak_framenum=0;   // default to OFF
	client->is_cloaked=false;   // default to NO Cloaking.
	client->id_on=OFF;          // Player Identification: OFF.
	client->railbomb=OFF;
	client->zylon_grenade=OFF;  // default to OFF
	client->baton=OFF;          // Turn OFF.
	client->chamber=OFF;        // default to OFF.
	client->laserdrone=OFF;     // Turn OFF.
	client->Tgrenade=OFF;       // default to OFF
	client->is_trapped=OFF;
	
	client->pers.connected=true;
}

//=========================================================
void InitClientResp(gclient_t *client) {
	memset(&client->resp, 0, sizeof(client->resp));
	client->resp.enterframe=level.framenum;
	client->resp.coop_respawn=client->pers;
}

//=========================================================
void SaveClientData(void) {
	int i;
	edict_t *ent;
	
	for (i=0; i<ga.maxclients; i++) {
		ent=&g_edicts[1+i];
		if (!G_EntExists(ent)) continue;
		ga.clients[i].pers.health=ent->health;
		ga.clients[i].pers.max_health=ent->max_health;
		ga.clients[i].pers.savedFlags = (ent->flags & (FL_GODMODE|FL_NOTARGET|FL_POWER_ARMOR));
		if (CVAR_COOP)
			ga.clients[i].pers.score=ent->client->resp.score; }
}

//=========================================================
void FetchClientEntData(edict_t *ent) {
	ent->health=ent->client->pers.health;
	ent->max_health=ent->client->pers.max_health;
	ent->flags |= ent->client->pers.savedFlags;
	if (CVAR_COOP)
		ent->client->resp.score=ent->client->pers.score;
}

//=========================================================
void InitBodyQue(void) {
	int i;
	edict_t *ent;
	
	level.body_que=0;
	for (i=0; i<BODY_QUEUE_SIZE; i++) {
		ent=G_Spawn();
		ent->classname="bodyque"; }
}

//===================================================
//======= RANDOM EARTHQUAKE GENERATOR CODE ==========
//===================================================

//===================================================
void Earthquake_Think(edict_t *quake) {
	int i;
	edict_t *ent;
	
	// Make rumbling sounds..
	if (quake->wait < level.time) {
		gi.positioned_sound(quake->s.origin, quake, CHAN_AUTO, gi.soundindex("world/quake.wav"), 1.0, ATTN_NONE, 0);
		quake->wait = level.time + 0.5; }
	
	// Shake around all clients in ga..
	for(i=0;i < ga.maxclients;i++) {
		ent=g_edicts+i+1;
		if (!G_ClientInGame(ent)) continue;
		if (!ent->groundentity) continue; // Is ent currently airborne??
		ent->groundentity = NULL; // If not, then keep'em on the ground..
		ent->velocity[0] += crandom()*250;
		ent->velocity[1] += crandom()*150;
		ent->velocity[2] = 250*(100.0/ent->mass); }
	
	// Time to stop shaking yet..
	if (level.time < quake->delay)
		quake->nextthink = level.time + 0.1;
	else {
		// Random Earthquakes every 2..10 minutes
		quake->nextthink = level.time + (2*60) + (60*(rand()%8));
		// Duration of shaking lasts 2..10 seconds..
		quake->delay=quake->nextthink + 2 + 10*(rand()%8); }
}

//===================================================
void Init_Earthquake_Generator(void) {
	edict_t *quake;
	static int i=0;
	
	// First time thru..
	if (i!=0) return;
	
	i=1; // Set so only init ONCE!
	
	quake = G_Spawn();
	quake->classname = "EarthQuake";
	VectorClear(quake->s.origin);
	quake->svflags |= SVF_NOCLIENT;
	quake->wait=0;
	quake->think = Earthquake_Think;
	
	// Random Earthquakes every 2..10 minutes
	quake->nextthink = level.time + (2*60) + (60*(rand()%8));
	
	// Duration of shaking lasts 2..10 seconds..
	quake->delay=quake->nextthink + 2 + 10*(rand()%8);
	
	gi.linkentity(quake);
}

//=========================================================
void PutClientInServer(edict_t *ent) {
	vec3_t mins, maxs;
	int i, clientnum;
	vec3_t spawn_origin, spawn_angles;
	client_persistant_t saved;
	client_respawn_t resp;
	char userinfo[MAX_INFO_STRING];
	
	VectorSet(mins,-16, -16, -24);
	VectorSet(maxs, 16,  16,  32);
	
	// Start the Earthquake Generator
	Init_Earthquake_Generator();
	
	// find a spawn point
	SelectSpawnPoint(ent, spawn_origin, spawn_angles);
	
	clientnum=ent-g_edicts-1;
	
	// deathmatch wipes most client data every spawn
	if (CVAR_DEATHMATCH) {
		resp=ent->client->resp;
		memcpy(userinfo, ent->client->pers.userinfo, sizeof(userinfo));
		InitClientPersistant(ent->client);
		ClientUserinfoChanged(ent, userinfo); }
	else if (CVAR_COOP) {
		resp=ent->client->resp;
		memcpy(userinfo, ent->client->pers.userinfo, sizeof(userinfo));
		resp.coop_respawn.game_helpchanged=ent->client->pers.game_helpchanged;
		resp.coop_respawn.helpchanged=ent->client->pers.helpchanged;
		ent->client->pers=resp.coop_respawn;
		ClientUserinfoChanged(ent, userinfo);
		if (resp.score > ent->client->pers.score)
			ent->client->pers.score=resp.score; }
	else
		memset(&resp, 0, sizeof(resp));
	
	// clear everything but the persistant data
	saved=ent->client->pers;
	memset(ent->client, 0, sizeof(*ent->client));
	ent->client->pers=saved;
	if (ent->client->pers.health<=0)
		InitClientPersistant(ent->client);
	ent->client->resp=resp;
	
	// copy some data from the client to the entity
	FetchClientEntData(ent);
	
	// clear entity values
	ent->groundentity=NULL;
	ent->client=&ga.clients[clientnum];
	ent->takedamage=DAMAGE_AIM;
	ent->movetype=MOVETYPE_WALK;
	ent->viewheight=22;
	ent->inuse=true;
	ent->classname="player";
	ent->mass=200;
	ent->solid=SOLID_BBOX;
	ent->deadflag=DEAD_NO;
	ent->air_finished=level.time+12;
	ent->clipmask=MASK_PLAYERSOLID;
	ent->model="players/male/tris.md2";
	ent->pain=player_pain;
	ent->die=player_die;
	ent->waterlevel=0;
	ent->watertype=0;
	ent->flags &= ~FL_NO_KNOCKBACK;
	ent->svflags &= ~SVF_DEADMONSTER;
	ent->airstrike_called=false;
	
	VectorCopy(mins, ent->mins);
	VectorCopy(maxs, ent->maxs);
	VectorClear(ent->velocity);
	
	// clear playerstate values
	memset(&ent->client->ps, 0, sizeof(ent->client->ps));
	
	ent->client->ps.pmove.origin[0]=spawn_origin[0]*8;
	ent->client->ps.pmove.origin[1]=spawn_origin[1]*8;
	ent->client->ps.pmove.origin[2]=spawn_origin[2]*8;
	
	Check_FOV(ent, userinfo);
	
	ent->client->ps.gunindex=gi.modelindex(ent->client->pers.weapon->view_model);
	
	// clear entity state values
	ent->s.effects=EF_NONE;
	ent->s.modelindex=255;    // will use the skin specified model
	ent->s.modelindex2=255;    // custom gun model
	ent->s.skinnum=ent-g_edicts-1;
	
	ent->s.frame=0;
	VectorCopy(spawn_origin, ent->s.origin);
	ent->s.origin[2]++;  // make sure off ground
	VectorCopy(ent->s.origin, ent->s.old_origin);
	
	// set the delta angle
	for (i=0; i<3; i++)
		ent->client->ps.pmove.delta_angles[i]=ANGLE2SHORT(spawn_angles[i]-ent->client->resp.cmd_angles[i]);
	
	ent->s.angles[PITCH]=0;
	ent->s.angles[YAW]=spawn_angles[YAW];
	ent->s.angles[ROLL]=0;
	VectorCopy(ent->s.angles, ent->client->ps.viewangles);
	VectorCopy(ent->s.angles, ent->client->v_angle);
	
	// spawn a spectator
	if (ent->client->pers.spectator) {
		ent->client->chase_target=NULL;
		ent->client->resp.spectator=true;
		ent->movetype=MOVETYPE_NOCLIP;
		ent->solid=SOLID_NOT;
		ent->svflags |= SVF_NOCLIENT;
		ent->client->ps.gunindex=0;
		gi.linkentity(ent);
		return; }
	else
		ent->client->resp.spectator=false;
	
	// Add 3 secs worth of invincibility during respawn!
	ent->client->invincible_framenum = level.time + 30;
	
	// Telefrag all ents at respawn location!
	Telefrag_All(ent);
	
	gi.linkentity(ent);
	
	ent->client->newweapon=ent->client->pers.weapon;
	ChangeWeapon(ent);
}

static qboolean Ghost_Spawned=false;

//=========================================================
void ClientBeginDeathmatch(edict_t *ent) {
	
	// Create Ghost on first Player connect.
	if ((sv_bestplayer->value==1.0) && !Ghost_Spawned) {
		Create_Ghost();
		Ghost_Spawned=true; }
	
	G_InitEdict(ent);
	
	InitClientResp(ent->client);
	
	// locate ent at a spawn point
	PutClientInServer(ent);
	
	// Add Player to EntList
	if (sv_bestplayer->value==1.0)
		G_AddToEntList(ent);
	
	if (level.intermissiontime)
		MoveClientToIntermission(ent);
	else
		G_MuzzleFlash((short)(ent-g_edicts), ent->s.origin, (int)MZ_LOGIN);
	
	gi_bprintf(PRINT_HIGH, "%s entered the game\n", ent->client->pers.netname);
	
	// make sure all view stuff is valid
	ClientEndServerFrame(ent);
}

//=========================================================
void ClientBegin(edict_t *ent) {
	int i;
	char timestr[20];
	struct tm *ltime;
	time_t gmtime;
	
	ent->client=ga.clients+(ent-g_edicts-1);
	
	time(&gmtime);
	ltime=localtime(&gmtime);
	Com_sprintf(timestr, 16, "%2i:%2i:%2i", ltime->tm_hour, ltime->tm_min, ltime->tm_sec);
	
	gi_centerprintf(ent, "Welcome aboard %s !!\n\nLocal Time: %s",ent->client->pers.netname,timestr);
	
	if (CVAR_DEATHMATCH) {
		ClientBeginDeathmatch(ent);
		return; }
	
	if (ent->inuse)
		for (i=0; i<3; i++)
			ent->client->ps.pmove.delta_angles[i]=ANGLE2SHORT(ent->client->ps.viewangles[i]);
		else {
			G_InitEdict(ent);
			ent->classname="player";
			InitClientResp(ent->client);
			PutClientInServer(ent); }
		
		if (level.intermissiontime)
			MoveClientToIntermission(ent);
		else
			// send effect if in a multiplayer game
			if (ga.maxclients > 1) {
				G_MuzzleFlash((short)(ent-g_edicts), ent->s.origin, (int)MZ_LOGIN);
				gi_bprintf(PRINT_HIGH, "%s entered the game\n", ent->client->pers.netname); }
			
			// make sure all view stuff is valid
			ClientEndServerFrame(ent);
			
}

//=========================================================
void ClientUserinfoChanged(edict_t *ent, char *userinfo) {
	char *s;
	int playernum;
	
	// check for malformed or illegal info strings
	if (!Info_Validate(userinfo))
		strcpy(userinfo, "\\name\\badinfo\\skin\\male/grunt");
	
	// set name
	s=Info_ValueForKey(userinfo, "name");
	strncpy(ent->client->pers.netname, s, sizeof(ent->client->pers.netname)-1);
	
	// set spectator
	s=Info_ValueForKey(userinfo, "spectator");
	
	// spectators are only supported in deathmatch
	ent->client->pers.spectator=(CVAR_DEATHMATCH && *s && Q_stricmp(s, "0"));
	
	// set skin
	s=Info_ValueForKey(userinfo, "skin");
	
	playernum=ent-g_edicts-1;
	
	// combine name and skin into a configstring
	gi.configstring(CS_PLAYERSKINS+playernum, va("%s\\%s", ent->client->pers.netname, s));
	
	Check_FOV(ent, userinfo);
	
	// handedness
	s=Info_ValueForKey(userinfo, "hand");
	if (strlen(s))
		ent->client->pers.hand=atoi(s);
	
	// save off the userinfo in case we want to check something later
	strncpy(ent->client->pers.userinfo, userinfo, sizeof(ent->client->pers.userinfo)-1);
}

//=========================================================
qboolean ClientConnect(edict_t *ent, char *userinfo) {
	char *value;
	int i, numspec;
	int clientnum;
	
	// check to see if they are on the banned IP list
	value=Info_ValueForKey(userinfo, "ip");
	if (SV_FilterPacket(value)) {
		Info_SetValueForKey(userinfo, "rejmsg", "Banned.");
		return false; }
	
	// check for a spectator
	value=Info_ValueForKey(userinfo, "spectator");
	if (CVAR_DEATHMATCH && *value && Q_stricmp(value, "0")) {
		if (*spectator_password->string &&
			Q_stricmp(spectator_password->string, "none") &&
			Q_stricmp(spectator_password->string, value)) {
			Info_SetValueForKey(userinfo, "rejmsg", "Spectator password required or incorrect.");
			return false; }
		
		// count spectators
		for (i=numspec=0; i < ga.maxclients; i++)
			if (g_edicts[i+1].inuse && g_edicts[i+1].client->pers.spectator)
				numspec++;
			if (numspec >= maxspectators->value) {
				Info_SetValueForKey(userinfo, "rejmsg", "Server spectator limit is full.");
				return false; } }
	else {
		// check for a password
		value=Info_ValueForKey(userinfo, "password");
		if (*password->string && Q_stricmp(password->string, "none") &&
			Q_stricmp(password->string, value)) {
			Info_SetValueForKey(userinfo, "rejmsg", "Password required or incorrect.");
			return false; } }
	
	if ((ent-g_edicts) > ga.maxclients) {
		Info_SetValueForKey(userinfo, "rejmsg", "Server is full (really!).");
		return false; }
	
	// What is ent's client number?
	clientnum=ent-g_edicts-1;
	
	// Assign player's ent record.
	ent=g_edicts+clientnum+1;
	
	// index player's client record
	ent->client=&ga.clients[clientnum];
	
	// if there is already a body waiting for us(a loadgame), just
	// take it, otherwise spawn one from scratch
	if (!ent->inuse) {
		// clear the respawning variables
		InitClientResp(ent->client);
		if (!ga.autosaved || !ent->client->pers.weapon)
			InitClientPersistant(ent->client); }
	
	ClientUserinfoChanged(ent, userinfo);
	
	if (ga.maxclients > 1)
		gi.dprintf("%s connected\n", ent->client->pers.netname);
	
	ent->svflags=0; // make sure we start with known default
	ent->client->pers.connected=true;
	
	return true;
}

//=========================================================
void ClientDisconnect(edict_t *ent) {
	int playernum;
	
	if (!G_EntExists(ent)) return;
	
	// Remove Player from EntList
	if (sv_bestplayer->value==1.0)
		G_RemoveFromEntList(ent);
	
	gi_bprintf(PRINT_HIGH, "%s disconnected\n", ent->client->pers.netname);
	
	G_MuzzleFlash((short)(ent-g_edicts), ent->s.origin, (int)MZ_LOGOUT);
	
	gi.unlinkentity(ent);
	ent->s.modelindex=0;
	ent->solid=SOLID_NOT;
	ent->inuse=false;
	ent->classname="disconnected";
	ent->client->pers.connected=false;
	
	playernum=ent-g_edicts-1;
	gi.configstring(CS_PLAYERSKINS+playernum, "");
}

//=========================================================
//=========================================================
//=========================================================

edict_t *pm_passent;

//=========================================================
trace_t  PM_trace(vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end) {
	if (pm_passent->health > 0)
		return gi.trace(start, mins, maxs, end, pm_passent, MASK_PLAYERSOLID);
	else
		return gi.trace(start, mins, maxs, end, pm_passent, MASK_DEADSOLID);
}

//=========================================================
unsigned CheckBlock(void *b, int c) {
	int v=0,i;
	for (i=0; i<c; i++)
		v += ((byte *)b)[i];
	return v;
}

//=========================================================
void PrintPmove(pmove_t *pm) {
	unsigned  c1, c2;
	
	c1=CheckBlock(&pm->s, sizeof(pm->s));
	c2=CheckBlock(&pm->cmd, sizeof(pm->cmd));
	Com_Printf("sv %3i:%i %i\n", pm->cmd.impulse, c1, c2);
}

//======================================================
void Player_Identification(edict_t *self) {
	int i, distance;
	edict_t *player;
	vec3_t forward;
	
	if (!(int)level.time%10) return;
	
	// Not available to dead or respawning players!
	if (!G_ClientInGame(self)) {
		self->client->id_on=0; // Turn OFF
		return; }
	
	for(i=0;i < ga.maxclients;i++) {
		player=g_edicts+i+1;
		if (!G_ClientInGame(player)) continue;
		if (self==player) continue;
		if (!G_Within_Radius(self->s.origin, player->s.origin, 500)) continue;
		if (!infront(self, player)) continue;
		VectorSubtract(self->s.origin, player->s.origin, forward);
		distance=(int)VectorLength(forward);
		gi_cprintf(self, PRINT_HIGH,"%s (%d) at %d units\n",player->client->pers.netname,player->health, distance);
    } // end for
}

//=========================================================
void ClientThink(edict_t *ent, usercmd_t *ucmd) {
	edict_t *other;
	int i, j;
	pmove_t  pm;
	
	if (!G_EntExists(ent)) return;
	
	// Player trapped in MegaHealth DeathBeam?
	if (ent->client->is_trapped) return;
	
	level.current_entity=ent;
	
	if (level.intermissiontime) {
		ent->client->ps.pmove.pm_type=PM_FREEZE;
		level.exitintermission=(level.time > level.intermissiontime+5.0 && (ucmd->buttons & BUTTON_ANY));
		return; }
	
	if (ent->client->id_on)
		Player_Identification(ent);
	
	pm_passent=ent;
	
	if (ent->client->chase_target) {
		ent->client->resp.cmd_angles[0]=SHORT2ANGLE(ucmd->angles[0]);
		ent->client->resp.cmd_angles[1]=SHORT2ANGLE(ucmd->angles[1]);
		ent->client->resp.cmd_angles[2]=SHORT2ANGLE(ucmd->angles[2]); }
	else {
		// set up for pmove
		memset(&pm, 0, sizeof(pm));
		if (ent->movetype == MOVETYPE_NOCLIP)
			ent->client->ps.pmove.pm_type=PM_SPECTATOR;
		else if (ent->s.modelindex != 255)
			ent->client->ps.pmove.pm_type=PM_GIB;
		else if (ent->deadflag==DEAD_DEAD)
			ent->client->ps.pmove.pm_type=PM_DEAD;
		else
			ent->client->ps.pmove.pm_type=PM_NORMAL;
		ent->client->ps.pmove.gravity=sv_gravity->value;
		pm.s=ent->client->ps.pmove;
		for (i=0; i<3; i++) {
			pm.s.origin[i]=ent->s.origin[i]*8;
			pm.s.velocity[i]=ent->velocity[i]*8; }
		
		pm.snapinitial=(memcmp(&ent->client->old_pmove, &pm.s, sizeof(pm.s)));
		
		pm.cmd=*ucmd;
		
		pm.trace=PM_trace;  // adds default parms
		pm.pointcontents=gi.pointcontents;
		
		// perform a pmove
		gi.Pmove(&pm);
		
		// save results of pmove
		ent->client->ps.pmove=pm.s;
		ent->client->old_pmove=pm.s;
		
		for (i=0; i<3; i++) {
			ent->s.origin[i]=pm.s.origin[i]*0.125;
			ent->velocity[i]=pm.s.velocity[i]*0.125; }
		
		VectorCopy(pm.mins, ent->mins);
		VectorCopy(pm.maxs, ent->maxs);
		
		ent->client->resp.cmd_angles[0]=SHORT2ANGLE(ucmd->angles[0]);
		ent->client->resp.cmd_angles[1]=SHORT2ANGLE(ucmd->angles[1]);
		ent->client->resp.cmd_angles[2]=SHORT2ANGLE(ucmd->angles[2]);
		
		if (ent->groundentity && !pm.groundentity && (pm.cmd.upmove >= 10) && (pm.waterlevel == 0)) {
			gi.sound(ent, CHAN_VOICE, gi.soundindex("*jump1.wav"), 1, ATTN_NORM, 0);
			PlayerNoise(ent, ent->s.origin, PNOISE_SELF); }
		
		ent->viewheight=pm.viewheight;
		ent->waterlevel=pm.waterlevel;
		ent->watertype=pm.watertype;
		ent->groundentity=pm.groundentity;
		if (pm.groundentity)
			ent->groundentity_linkcount=pm.groundentity->linkcount;
		
		if (ent->deadflag==DEAD_DEAD) {
			ent->client->ps.viewangles[ROLL]=40;
			ent->client->ps.viewangles[PITCH]= -15;
			ent->client->ps.viewangles[YAW]=ent->client->killer_yaw; }
		else {
			VectorCopy(pm.viewangles, ent->client->v_angle);
			VectorCopy(pm.viewangles, ent->client->ps.viewangles); }
		
		gi.linkentity(ent);
		
		if (ent->movetype != MOVETYPE_NOCLIP)
			G_TouchTriggers(ent);
		
		// touch other objects
		for (i=0; i<pm.numtouch; i++) {
			other=pm.touchents[i];
			for (j=0; j<i; j++)
				if (pm.touchents[j] == other)
					break;
				if (j != i) continue;  // duplicated
				if (!other->touch) continue;
				other->touch(other, ent, NULL, NULL); } }
	
	ent->client->oldbuttons=ent->client->buttons;
	ent->client->buttons=ucmd->buttons;
	ent->client->latched_buttons |= ent->client->buttons & ~ent->client->oldbuttons;
	
	// save light level for monster sighting AI
	ent->light_level=ucmd->lightlevel;
	
	// fire weapon from final position if needed
	if (ent->client->latched_buttons & BUTTON_ATTACK) {
		if (ent->client->resp.spectator) {
			ent->client->latched_buttons=BUTTON_NONE;
			if (ent->client->chase_target) {
				ent->client->chase_target=NULL;
				ent->client->ps.pmove.pm_flags &= ~PMF_NO_PREDICTION; }
			else
				GetChaseTarget(ent); }
		else if (!ent->client->weapon_thunk) {
			ent->client->weapon_thunk=true;
			Think_Weapon(ent); } }
	
	if (ent->client->resp.spectator) {
		if (ucmd->upmove >= 10) {
			if (!(ent->client->ps.pmove.pm_flags & PMF_JUMP_HELD)) {
				ent->client->ps.pmove.pm_flags |= PMF_JUMP_HELD;
				if (ent->client->chase_target)
					ChaseNext(ent);
				else
					GetChaseTarget(ent); } }
		else
			ent->client->ps.pmove.pm_flags &= ~PMF_JUMP_HELD; }
	
	// Airstrike Called AND Time to launch AirStrike
	if (G_ClientInGame(ent))
		if (ent->airstrike_called && (level.time > ent->airstrike_time))
			spawn_aircraft(ent);
		
		// update chase cam if being followed
		for (i=1; i<=maxclients->value; i++) {
			other=g_edicts+i;
			if (other->inuse && other->client->chase_target == ent)
				UpdateChaseCam(other); }
}

//=========================================================
void ClientBeginServerFrame(edict_t *ent) {
	int buttonMask;
	
	if (level.intermissiontime) return;
	
	// Flyer exists and ent is riding Flyer?
	if (ent->flyer && ent->is_riding) {
		// Move ent with Flyer's movement.
		VectorCopy(ent->flyer->s.origin, ent->s.origin);
		VectorCopy(ent->flyer->s.angles, ent->s.angles); }
	
	if (CVAR_DEATHMATCH &&
		ent->client->pers.spectator != ent->client->resp.spectator &&
		(level.time-ent->client->respawn_time) >= 5) {
		spectator_respawn(ent);
		return; }
	
	// run weapon animations if it hasn't been done by a ucmd_t
	if (!ent->client->weapon_thunk && !ent->client->resp.spectator)
		Think_Weapon(ent);
	else
		ent->client->weapon_thunk=false;
	
	if (ent->deadflag==DEAD_DEAD) {
		// wait for any button just going down
		if (level.time > ent->client->respawn_time) {
			// in deathmatch, only wait for attack button
			buttonMask=(CVAR_DEATHMATCH)?BUTTON_ATTACK:-1;
			if ((ent->client->latched_buttons & buttonMask) ||
				(CVAR_DEATHMATCH && ((int)dmflags->value & DF_FORCE_RESPAWN))) {
				respawn(ent);
				ent->client->latched_buttons=BUTTON_NONE; } }
		return; }
	
	// add player trail so monsters can follow
	if (!CVAR_DEATHMATCH)
		if (!visible(ent, PlayerTrail_LastSpot()))
			PlayerTrail_Add(ent->s.old_origin);
		
		ent->client->latched_buttons=BUTTON_NONE;
}
