#include "defines.h"

qboolean ai_checkattack(edict_t *self, float dist);

qboolean enemy_vis;
qboolean enemy_infront;
int enemy_range;
float enemy_yaw;

//=======================================================
void AI_SetSightClient(void) {
	edict_t *ent;
	int start, check;
	
	check=start=(level.sight_client == NULL)?1:level.sight_client-g_edicts;
	
	while (1) {
		check++;
		if (check > ga.maxclients)
			check=1;
		ent=&g_edicts[check];
		if (ent->inuse && ent->health > 0 && !(ent->flags & FL_NOTARGET)) {
			level.sight_client=ent;
			return; }
		if (check == start) {
			level.sight_client=NULL;
			return; } }
}

//=======================================================
void ai_move(edict_t *self, float dist) {
	M_walkmove(self, self->s.angles[YAW], dist);
}

//=======================================================
void ai_stand(edict_t *self, float dist) {
	vec3_t v;
	
	if (dist)
		M_walkmove(self, self->s.angles[YAW], dist);
	
	if (self->monsterinfo.aiflags & AI_STAND_GROUND) {
		if (self->enemy) {
			VectorSubtract(self->enemy->s.origin, self->s.origin, v);
			self->ideal_yaw=vectoyaw(v);
			if (self->s.angles[YAW] != self->ideal_yaw && self->monsterinfo.aiflags & AI_TEMP_STAND_GROUND) {
				self->monsterinfo.aiflags &= ~(AI_STAND_GROUND|AI_TEMP_STAND_GROUND);
				self->monsterinfo.run(self); }
			M_ChangeYaw(self);
			ai_checkattack(self, 0); }
		else
			FindTarget(self);
		return; }
	
	if (FindTarget(self))
		return;
	
	if (level.time > self->monsterinfo.pausetime) {
		self->monsterinfo.walk(self);
		return; }
	
	if (!(self->spawnflags & 1) && (self->monsterinfo.idle) && (level.time > self->monsterinfo.idle_time))
		if (self->monsterinfo.idle_time) {
			self->monsterinfo.idle(self);
			self->monsterinfo.idle_time=level.time+15+random()*15; }
		else
			self->monsterinfo.idle_time=level.time+random()*15;
}

//=======================================================
void ai_walk(edict_t *self, float dist) {
	
	M_MoveToGoal(self, dist);
	
	// check for noticing a player
	if (FindTarget(self))
		return;
	
	if ((self->monsterinfo.search) && (level.time > self->monsterinfo.idle_time))
		if (self->monsterinfo.idle_time) {
			self->monsterinfo.search(self);
			self->monsterinfo.idle_time=level.time+15+random()*15; }
		else
			self->monsterinfo.idle_time=level.time+random()*15;
}

//=======================================================
void ai_charge(edict_t *self, float dist) {
	vec3_t v;
	
	VectorSubtract(self->enemy->s.origin, self->s.origin, v);
	self->ideal_yaw=vectoyaw(v);
	M_ChangeYaw(self);
	
	if (dist)
		M_walkmove(self, self->s.angles[YAW], dist);
}

//=======================================================
void ai_turn(edict_t *self, float dist) {
	
	if (dist)
		M_walkmove(self, self->s.angles[YAW], dist);
	
	if (FindTarget(self))
		return;
	
	M_ChangeYaw(self);
}

//=======================================================
int range(edict_t *self, edict_t *other) {
	vec3_t v;
	float len;
	VectorSubtract(self->s.origin, other->s.origin, v);
	len=VectorLength(v);
	return (len < MELEE_DISTANCE)?RANGE_MELEE:(len < 500)?RANGE_NEAR:(len < 1000)?RANGE_MID:RANGE_FAR;
}

//=======================================================
// TRUE if other is in PVS and in direct line-of-sight.
//=======================================================
qboolean visible2(edict_t *self, edict_t *other) {
	return (gi.inPVS(self->s.origin, other->s.origin)
		&& gi.trace(self->s.origin, NULL, NULL, other->s.origin, NULL, MASK_OPAQUE).fraction == 1.0);
}

//=======================================================
qboolean visible(edict_t *self, edict_t *other) {
	vec3_t spot1, spot2;
	
	if (!gi.inPVS(self->s.origin, other->s.origin))
		return false;
	
	// Is this Flyer looking for visible targets?
	if (!Q_stricmp(self->classname, "XFlyer"))
		// Is ent currently riding this Flyer?
		if (self->rider->is_riding)
			// Set so Flyer can't see its Rider!
			if (self->rider==other)
				return false;
			
			VectorCopy(self->s.origin, spot1);
			spot1[2] += self->viewheight;
			VectorCopy(other->s.origin, spot2);
			spot2[2] += other->viewheight;
			
			return (gi.trace(spot1, zvec, zvec, spot2, self, MASK_OPAQUE).fraction == 1.0);
}

//=======================================================
qboolean infront(edict_t *self, edict_t *other) {
	vec3_t v, forward;
	AngleVectors(self->s.angles, forward, NULL, NULL);
	VectorSubtract(other->s.origin, self->s.origin, v);
	VectorNormalize(v);
	return (DotProduct(v, forward) > 0.3);
}

//=======================================================
void HuntTarget(edict_t *self) {
	vec3_t v;
	
	self->goalentity=self->enemy;
	if (self->monsterinfo.aiflags & AI_STAND_GROUND)
		self->monsterinfo.stand(self);
	else
		self->monsterinfo.run(self);
	VectorSubtract(self->enemy->s.origin, self->s.origin, v);
	self->ideal_yaw=vectoyaw(v);
	if (!(self->monsterinfo.aiflags & AI_STAND_GROUND))
		self->monsterinfo.attack_finished=level.time + 1;
}

//=======================================================
void FoundTarget(edict_t *self) {
	
	// let other monsters see this monster for a while
	if (self->enemy->client) {
		level.sight_entity=self;
		level.sight_entity_framenum=level.framenum;
		level.sight_entity->light_level=128; }
	
	self->show_hostile=level.time+1;    // wake up other monsters
	
	VectorCopy(self->enemy->s.origin, self->monsterinfo.last_sighting);
	self->monsterinfo.trail_time=level.time;
	
	if (!self->combattarget) {
		HuntTarget(self);
		return; }
	
	self->goalentity=self->movetarget=G_PickTarget(self->combattarget);
	if (!self->movetarget) {
		self->goalentity=self->movetarget=self->enemy;
		HuntTarget(self);
		gi.dprintf("%s at %s, combattarget %s not found\n", self->classname, vtos(self->s.origin), self->combattarget);
		return; }
	
	// clear out our combattarget, these are a one shot deal
	self->combattarget=NULL;
	self->monsterinfo.aiflags |= AI_COMBAT_POINT;
	
	// clear the targetname, that point is ours!
	self->movetarget->targetname=NULL;
	self->monsterinfo.pausetime=0;
	
	// run for it
	self->monsterinfo.run(self);
}

//=======================================================
qboolean FindTarget(edict_t *self) {
	edict_t *client;
	qboolean heardit;
	int r;
	vec3_t temp;
	
	if (self->monsterinfo.aiflags & AI_GOOD_GUY) {
		if (self->goalentity && self->goalentity->inuse && self->goalentity->classname)
			if (!Q_stricmp(self->goalentity->classname, "target_actor"))
				return false;
			return false; }
	
	// if we're going to a combat point, just proceed
	if (self->monsterinfo.aiflags & AI_COMBAT_POINT)
		return false;
	
	heardit=false;
	if ((level.sight_entity_framenum >=(level.framenum-1))
		&& !(self->spawnflags & 1)) {
		client=level.sight_entity;
		if (client->enemy == self->enemy)
			return false; }
	else if (level.sound_entity_framenum >=(level.framenum-1)) {
		client=level.sound_entity;
		heardit=true; }
	else if (!(self->enemy) && (level.sound2_entity_framenum >=(level.framenum-1)) && !(self->spawnflags & 1)) {
		client=level.sound2_entity;
		heardit=true; }
	else {
		client=level.sight_client;
		if (!client)
			return false; }
	
	// if the entity went away, forget it
	if (!client->inuse)
		return false;
	
	if (client == self->enemy)
		return true;
	
	if (client->client) {
		if (client->flags & FL_NOTARGET)
			return false; }
	else if (client->svflags & SVF_MONSTER) {
		if (!client->enemy)
			return false;
		if (client->enemy->flags & FL_NOTARGET)
			return false; }
	else if (heardit) {
		if (client->owner->flags & FL_NOTARGET)
			return false; }
	else
		return false;
	
	if (!heardit) {
		r=range(self, client);
		if (r == RANGE_FAR)
			return false;
		
		// is client in an spot too dark to be seen?
		if (client->light_level<=5)
			return false;
		
		if (!visible(self, client))
			return false;
		
		if (r == RANGE_NEAR) {
			if (client->show_hostile < level.time && !infront(self, client))
				return false; }
		else if (r == RANGE_MID)
			if (!infront(self, client))
				return false;
			
			self->enemy=client;
			
			if (Q_stricmp(self->enemy->classname, "player_noise")) {
				self->monsterinfo.aiflags &= ~AI_SOUND_TARGET;
				if (!self->enemy->client) {
					self->enemy=self->enemy->enemy;
					if (!self->enemy->client) {
						self->enemy=NULL;
						return false; } } }
    } // endif
	else  { // heardit
		if (self->spawnflags & 1) {
			if (!visible(self, client))
				return false; }
		else
			if (!gi.inPHS(self->s.origin, client->s.origin))
				return false;
			VectorSubtract(client->s.origin, self->s.origin, temp);
			if (VectorLength(temp) > 1000)  // too far to hear
				return false;
			
			// check area portals-if they are different and not connected then we can't hear it
			if (client->areanum != self->areanum)
				if (!gi.AreasConnected(self->areanum, client->areanum))
					return false;
				
				self->ideal_yaw=vectoyaw(temp);
				M_ChangeYaw(self);
				
				// hunt the sound for a bit; hopefully find the real player
				self->monsterinfo.aiflags |= AI_SOUND_TARGET;
				self->enemy=client;
    } // end if
	
	//
	// got one
	//
	FoundTarget(self);
	
	if (!(self->monsterinfo.aiflags & AI_SOUND_TARGET) && (self->monsterinfo.sight))
		self->monsterinfo.sight(self, self->enemy);
	
	return true;
}

//=======================================================
qboolean FacingIdeal(edict_t *self) {
	float delta;
	delta=anglemod(self->s.angles[YAW]-self->ideal_yaw);
	return !(delta > 45 && delta < 315);
}

//=======================================================
qboolean M_CheckAttack(edict_t *self) {
	vec3_t spot1, spot2;
	float chance;
	trace_t tr;
	
	if (self->enemy->health > 0) {
		// see if any entities are in the way of the shot
		VectorCopy(self->s.origin, spot1);
		spot1[2] += self->viewheight;
		VectorCopy(self->enemy->s.origin, spot2);
		spot2[2] += self->enemy->viewheight;
		tr=gi.trace(spot1, NULL, NULL, spot2, self, CONTENTS_SOLID|CONTENTS_MONSTER|CONTENTS_SLIME|CONTENTS_LAVA|CONTENTS_WINDOW);
		// do we have a clear shot?
		if (tr.ent != self->enemy)
			return false; }
	
	// melee attack
	if (enemy_range == RANGE_MELEE) {
		// don't always melee in easy mode
		if (CVAR_SKILL == SKILL_EASY && (rand()&3))
			return false;
		self->monsterinfo.attack_state=(self->monsterinfo.melee)?AS_MELEE:AS_MISSILE;
		return true; }
	
	// missile attack
	if (!self->monsterinfo.attack)
		return false;
	
	if (level.time < self->monsterinfo.attack_finished)
		return false;
	
	if (enemy_range == RANGE_FAR)
		return false;
	
	if (self->monsterinfo.aiflags & AI_STAND_GROUND)
		chance=0.4;
	else if (enemy_range == RANGE_MELEE)
		chance=0.2;
	else if (enemy_range == RANGE_NEAR)
		chance=0.1;
	else if (enemy_range == RANGE_MID)
		chance=0.02;
	else
		return false;
	
	if (CVAR_SKILL == SKILL_EASY)
		chance *= 0.5;
	else if (CVAR_SKILL >= SKILL_HARD)
		chance *= 2;
	
	if (random() < chance) {
		self->monsterinfo.attack_state=AS_MISSILE;
		self->monsterinfo.attack_finished=level.time+2*random();
		return true; }
	
	if (self->flags & FL_FLY)
		self->monsterinfo.attack_state=(random()<0.3)?AS_SLIDING:AS_STRAIGHT;
	
	return false;
}

//=======================================================
void ai_run_melee(edict_t *self) {
	
	self->ideal_yaw=enemy_yaw;
	M_ChangeYaw(self);
	
	if (FacingIdeal(self)) {
		self->monsterinfo.melee(self);
		self->monsterinfo.attack_state=AS_STRAIGHT; }
}

//=======================================================
void ai_run_missile(edict_t *self) {
	
	self->ideal_yaw=enemy_yaw;
	M_ChangeYaw(self);
	
	if (FacingIdeal(self)) {
		self->monsterinfo.attack(self);
		self->monsterinfo.attack_state=AS_STRAIGHT; }
}

//=======================================================
void ai_run_slide(edict_t *self, float distance) {
	float ofs;
	
	self->ideal_yaw=enemy_yaw;
	M_ChangeYaw(self);
	
	ofs=(self->monsterinfo.lefty)?90:-90;
	
	if (M_walkmove(self, self->ideal_yaw+ofs, distance))
		return;
	
	self->monsterinfo.lefty=1-self->monsterinfo.lefty;
	M_walkmove(self, self->ideal_yaw-ofs, distance);
}

//=======================================================
qboolean ai_checkattack(edict_t *self, float dist) {
	vec3_t temp;
	qboolean hesDeadJim;
	
	if (self->goalentity) {
		if (self->monsterinfo.aiflags & AI_COMBAT_POINT)
			return false;
		if (self->monsterinfo.aiflags & AI_SOUND_TARGET) {
			if ((level.time-self->enemy->teleport_time) > 5.0) {
				if (self->goalentity == self->enemy)
					self->goalentity=(self->movetarget)?self->movetarget:NULL;
				self->monsterinfo.aiflags &= ~AI_SOUND_TARGET;
				if (self->monsterinfo.aiflags & AI_TEMP_STAND_GROUND)
					self->monsterinfo.aiflags &= ~(AI_STAND_GROUND|AI_TEMP_STAND_GROUND); }
			else {
				self->show_hostile=level.time+1;
				return false; } } }
	
	enemy_vis=false;
	
	// see if the enemy is dead
	hesDeadJim=false;
	if ((!self->enemy) || (!self->enemy->inuse))
		hesDeadJim=true;
	else if (self->monsterinfo.aiflags & AI_MEDIC) {
		if (self->enemy->health > 0) {
			hesDeadJim=true;
			self->monsterinfo.aiflags &= ~AI_MEDIC; } }
	else {
		if (self->monsterinfo.aiflags & AI_BRUTAL)
			hesDeadJim=(self->enemy->health<=-80);
		else
			hesDeadJim=(self->enemy->health<=0); }
	
	if (hesDeadJim) {
		self->enemy=NULL;
		if (self->oldenemy && self->oldenemy->health > 0) {
			self->enemy=self->oldenemy;
			self->oldenemy=NULL;
			HuntTarget(self); }
		else {
			if (self->movetarget) {
				self->goalentity=self->movetarget;
				self->monsterinfo.walk(self); }
			else {
				self->monsterinfo.pausetime=level.time+100000000;
				self->monsterinfo.stand(self); }
			return true; } }
	
	self->show_hostile=level.time+1; // wake up other monsters
	
	// check knowledge of enemy
	enemy_vis=visible(self, self->enemy);
	if (enemy_vis) {
		self->monsterinfo.search_time=level.time+5;
		VectorCopy(self->enemy->s.origin, self->monsterinfo.last_sighting); }
	
	enemy_infront=infront(self, self->enemy);
	enemy_range=range(self, self->enemy);
	VectorSubtract(self->enemy->s.origin, self->s.origin, temp);
	enemy_yaw=vectoyaw(temp);
	
	if (self->monsterinfo.attack_state == AS_MISSILE) {
		ai_run_missile(self);
		return true; }
	
	if (self->monsterinfo.attack_state == AS_MELEE) {
		ai_run_melee(self);
		return true; }
	
	// if enemy is not currently visible, we will never attack
	if (!enemy_vis) return false;
	
	return self->monsterinfo.checkattack(self);
}

//=======================================================
void ai_run(edict_t *self, float dist) {
	edict_t *tempgoal, *save, *marker;
	qboolean qnew;
	float d1, d2, left, center, right;
	trace_t tr;
	vec3_t v, v_forward, v_right;
	vec3_t left_target, right_target;
	
	// if we're going to a combat point, just proceed
	if (self->monsterinfo.aiflags & AI_COMBAT_POINT) {
		M_MoveToGoal(self, dist);
		return; }
	
	if (self->monsterinfo.aiflags & AI_SOUND_TARGET) {
		VectorSubtract(self->s.origin, self->enemy->s.origin, v);
		if (VectorLength(v) < 64) {
			self->monsterinfo.aiflags |=(AI_STAND_GROUND|AI_TEMP_STAND_GROUND);
			self->monsterinfo.stand(self);
			return; }
		M_MoveToGoal(self, dist);
		if (!FindTarget(self))
			return; }
	
	if (ai_checkattack(self, dist))
		return;
	
	if (self->monsterinfo.attack_state == AS_SLIDING) {
		ai_run_slide(self, dist);
		return; }
	
	if (enemy_vis) {
		M_MoveToGoal(self, dist);
		self->monsterinfo.aiflags &= ~AI_LOST_SIGHT;
		VectorCopy(self->enemy->s.origin, self->monsterinfo.last_sighting);
		self->monsterinfo.trail_time=level.time;
		return; }
	
	// coop will change to another enemy if visible
	if (CVAR_COOP && FindTarget(self))
		return;
	
	if ((self->monsterinfo.search_time)
		&& (level.time >(self->monsterinfo.search_time+20))) {
		M_MoveToGoal(self, dist);
		self->monsterinfo.search_time=0;
		return; }
	
	save=self->goalentity;
	tempgoal=G_Spawn();
	self->goalentity=tempgoal;
	
	qnew=false;
	
	if (!(self->monsterinfo.aiflags & AI_LOST_SIGHT)) {
		// just lost sight of the player, decide where to go first
		self->monsterinfo.aiflags |=(AI_LOST_SIGHT|AI_PURSUIT_LAST_SEEN);
		self->monsterinfo.aiflags &= ~(AI_PURSUE_NEXT|AI_PURSUE_TEMP);
		qnew=true; }
	
	if (self->monsterinfo.aiflags & AI_PURSUE_NEXT) {
		self->monsterinfo.aiflags &= ~AI_PURSUE_NEXT;
		self->monsterinfo.search_time=level.time+5;
		if (self->monsterinfo.aiflags & AI_PURSUE_TEMP) {
			self->monsterinfo.aiflags &= ~AI_PURSUE_TEMP;
			marker=NULL;
			VectorCopy(self->monsterinfo.saved_goal, self->monsterinfo.last_sighting);
			qnew=true; }
		else if (self->monsterinfo.aiflags & AI_PURSUIT_LAST_SEEN) {
			self->monsterinfo.aiflags &= ~AI_PURSUIT_LAST_SEEN;
			marker=PlayerTrail_PickFirst(self); }
		else
			marker=PlayerTrail_PickNext(self);
		
		if (marker) {
			VectorCopy(marker->s.origin, self->monsterinfo.last_sighting);
			self->monsterinfo.trail_time=marker->timestamp;
			self->s.angles[YAW]=self->ideal_yaw=marker->s.angles[YAW];
			qnew=true; } }
	
	VectorSubtract(self->s.origin, self->monsterinfo.last_sighting, v);
	d1=VectorLength(v);
	if (d1<=dist) {
		self->monsterinfo.aiflags |= AI_PURSUE_NEXT;
		dist=d1; }
	
	VectorCopy(self->monsterinfo.last_sighting, self->goalentity->s.origin);
	
	if (qnew) {
		tr=gi.trace(self->s.origin, self->mins, self->maxs, self->monsterinfo.last_sighting, self, MASK_PLAYERSOLID);
		if (tr.fraction != 1.0) {
			VectorSubtract(self->goalentity->s.origin, self->s.origin, v);
			d1=VectorLength(v);
			center=tr.fraction;
			d2=d1*((center+1)/2);
			self->s.angles[YAW]=self->ideal_yaw=vectoyaw(v);
			AngleVectors(self->s.angles, v_forward, v_right, NULL);
			VectorSet(v, d2, -16, 0);
			G_ProjectSource(self->s.origin, v, v_forward, v_right, left_target);
			tr=gi.trace(self->s.origin, self->mins, self->maxs, left_target, self, MASK_PLAYERSOLID);
			left=tr.fraction;
			VectorSet(v, d2, 16, 0);
			G_ProjectSource(self->s.origin, v, v_forward, v_right, right_target);
			tr=gi.trace(self->s.origin, self->mins, self->maxs, right_target, self, MASK_PLAYERSOLID);
			right=tr.fraction;
			center = (d1*center)/d2;
			if (left >= center && left > right) {
				if (left < 1) {
					VectorSet(v, d2*left*0.5, -16, 0);
					G_ProjectSource(self->s.origin, v, v_forward, v_right, left_target); }
				VectorCopy(self->monsterinfo.last_sighting, self->monsterinfo.saved_goal);
				self->monsterinfo.aiflags |= AI_PURSUE_TEMP;
				VectorCopy(left_target, self->goalentity->s.origin);
				VectorCopy(left_target, self->monsterinfo.last_sighting);
				VectorSubtract(self->goalentity->s.origin, self->s.origin, v);
				self->s.angles[YAW]=self->ideal_yaw=vectoyaw(v); }
			else if (right >= center && right > left) {
				if (right < 1) {
					VectorSet(v, d2*right*0.5, 16, 0);
					G_ProjectSource(self->s.origin, v, v_forward, v_right, right_target); }
				VectorCopy(self->monsterinfo.last_sighting, self->monsterinfo.saved_goal);
				self->monsterinfo.aiflags |= AI_PURSUE_TEMP;
				VectorCopy(right_target, self->goalentity->s.origin);
				VectorCopy(right_target, self->monsterinfo.last_sighting);
				VectorSubtract(self->goalentity->s.origin, self->s.origin, v);
				self->s.angles[YAW]=self->ideal_yaw=vectoyaw(v); } } }
	
	M_MoveToGoal(self, dist);
	
	G_FreeEdict(tempgoal);
	
	if (self)
		self->goalentity=save;
}
