#include "defines.h"

//============================================================
void monster_fire_bullet(edict_t *self, vec3_t start, vec3_t dir, int damage, int kick, int hspread, int vspread, int flashtype) {
  fire_bullet(self, start, dir, damage, kick, hspread, vspread, MOD_UNKNOWN);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void monster_fire_shotgun(edict_t *self, vec3_t start, vec3_t aimdir, int damage, int kick, int hspread, int vspread, int count, int flashtype) {
  fire_shotgun(self, start, aimdir, damage, kick, hspread, vspread, count, MOD_UNKNOWN);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void monster_fire_blaster(edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, int flashtype, int effect) {
  fire_blaster(self, start, dir, damage, speed, effect, false);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void monster_fire_grenade(edict_t *self, vec3_t start, vec3_t aimdir, int damage, int speed, int flashtype) {
  fire_grenade(self, start, aimdir, damage, speed, 2.5, damage+40);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void monster_fire_chaingun(edict_t *self, vec3_t start, vec3_t dir, int damage, int kick, int hspread, int vspread) {
  fire_bullet(self, start, dir, damage, kick, hspread, vspread, MOD_CHAINGUN);
  G_MuzzleFlash2(self-g_edicts, start, (MZ_CHAINGUN1+7));
}

//============================================================
void monster_fire_rocket(edict_t *self, vec3_t start, vec3_t dir, int damage, int speed, int flashtype) {
  fire_rocket(self, start, dir, damage, speed, damage+20, damage);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void monster_fire_railgun(edict_t *self, vec3_t start, vec3_t aimdir, int damage, int kick, int flashtype) {
  fire_rail(self, start, aimdir, damage, kick);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void monster_fire_bfg(edict_t *self, vec3_t start, vec3_t aimdir, int damage, int speed, int kick, float damage_radius, int flashtype) {
  fire_bfg(self, start, aimdir, damage, speed, damage_radius);
  G_MuzzleFlash2(self-g_edicts, start, flashtype);
}

//============================================================
void M_FliesOff(edict_t *self) {
  self->s.effects &= ~EF_FLIES;
  self->s.sound=0;
}

//============================================================
void M_FliesOn(edict_t *self) {
  if (self->waterlevel) return;
  self->s.effects |= EF_FLIES;
  self->s.sound=gi.soundindex("infantry/inflies1.wav");
  self->think=M_FliesOff;
  self->nextthink=level.time+60;
}

//============================================================
void M_FlyCheck(edict_t *self) {
  if (self->waterlevel || random() > 0.5) return;
  self->think=M_FliesOn;
  self->nextthink=level.time+5+10*random();
}

//============================================================
void AttackFinished(edict_t *self, float time) {
  self->monsterinfo.attack_finished=level.time+time;
}

//============================================================
void M_CheckGround(edict_t *ent) {
vec3_t point;
trace_t tr;

  if (ent->flags & (FL_SWIM|FL_FLY)) return;

  if (ent->velocity[2] > 100) {
    ent->groundentity=NULL;
    return; }

  // if the hull point one-quarter unit
  // down is solid the entity is on ground
  point[0]=ent->s.origin[0];
  point[1]=ent->s.origin[1];
  point[2]=ent->s.origin[2]-0.25;

  tr=gi.trace(ent->s.origin, ent->mins, ent->maxs, point, ent, MASK_MONSTERSOLID);

  // check steepness
  if (tr.plane.normal[2] < 0.7 && !tr.startsolid) {
    ent->groundentity=NULL;
    return; }

  if (!tr.startsolid && !tr.allsolid) {
    VectorCopy(tr.endpos, ent->s.origin);
    ent->groundentity=tr.ent;
    ent->groundentity_linkcount=tr.ent->linkcount;
    ent->velocity[2]=0; }
}

//=================================================
// Sets the ent's waterlevel (every frame)
// 0-not in liquid at all
// 1-only with feet in liquid
// 2-swimming on surface of liquid
// 3-completely immersed in liquid
//=================================================
void M_CatagorizePosition(edict_t *ent) {
vec3_t point;

  point[0]=ent->s.origin[0];
  point[1]=ent->s.origin[1];
  point[2]=ent->s.origin[2]-25;

  // Not in water at all
  if (!(gi.pointcontents(point) & MASK_WATER)) {
    ent->waterlevel=0;
    return; }

  // Standing in water.
  point[2] += 26;
  if (!(gi.pointcontents(point) & MASK_WATER)) {
    ent->waterlevel=1;
    return; }

  // Waist deep waterlevel
  point[2] += 26;
  if (!(gi.pointcontents(point) & MASK_WATER)) {
    ent->waterlevel=2;
    return; }

  // Completely underwater
  ent->waterlevel=3;
}

//============================================================
void M_WorldEffects(edict_t *ent) {
int dmg;

  if (ent->health > 0) {
    if (!(ent->flags & FL_SWIM)) {
      if (ent->waterlevel < 3)
        ent->air_finished=level.time+12;
      else if (ent->air_finished < level.time) {
        if (ent->pain_debounce_time < level.time) {
          dmg=2+2*floor(level.time-ent->air_finished);
          if (dmg > 15) dmg=15;
          T_Damage(ent, world, world, zvec, ent->s.origin, zvec, dmg, 0, DAMAGE_NO_ARMOR, MOD_WATER);
          ent->pain_debounce_time=level.time+1; } } }
    else {
      if (ent->waterlevel > 0)
        ent->air_finished=level.time+9;
      else if (ent->air_finished < level.time) {
        if (ent->pain_debounce_time < level.time) {
          dmg=2+2*floor(level.time-ent->air_finished);
          if (dmg > 15) dmg=15;
          T_Damage(ent, world, world, zvec, ent->s.origin, zvec, dmg, 0, DAMAGE_NO_ARMOR, MOD_WATER);
          ent->pain_debounce_time=level.time+1; } } } }

  if (ent->waterlevel == 0) {
    if (ent->flags & FL_INWATER) {
      gi.sound(ent, CHAN_BODY, gi.soundindex("player/watr_out.wav"), 1, ATTN_NORM, 0);
      ent->flags &= ~FL_INWATER; }
    return; }

  if ((ent->watertype & CONTENTS_LAVA) && !(ent->flags & FL_IMMUNE_LAVA)) {
    if (ent->damage_debounce_time < level.time) {
      ent->damage_debounce_time=level.time+0.2;
      T_Damage(ent, world, world, zvec, ent->s.origin, zvec, 10*ent->waterlevel, 0, 0, MOD_LAVA); } }
  if ((ent->watertype & CONTENTS_SLIME) && !(ent->flags & FL_IMMUNE_SLIME)) {
    if (ent->damage_debounce_time < level.time) {
      ent->damage_debounce_time=level.time+1;
      T_Damage(ent, world, world, zvec, ent->s.origin, zvec, 4*ent->waterlevel, 0, 0, MOD_SLIME); } }

  if (!(ent->flags & FL_INWATER)) {
    if (!(ent->svflags & SVF_DEADMONSTER)) {
      if (ent->watertype & CONTENTS_LAVA)
        if (random()<=0.5)
          gi.sound(ent, CHAN_BODY, gi.soundindex("player/lava1.wav"), 1, ATTN_NORM, 0);
        else
          gi.sound(ent, CHAN_BODY, gi.soundindex("player/lava2.wav"), 1, ATTN_NORM, 0);
      else if (ent->watertype & CONTENTS_SLIME)
        gi.sound(ent, CHAN_BODY, gi.soundindex("player/watr_in.wav"), 1, ATTN_NORM, 0);
      else if (ent->watertype & CONTENTS_WATER)
        gi.sound(ent, CHAN_BODY, gi.soundindex("player/watr_in.wav"), 1, ATTN_NORM, 0); }

    ent->flags |= FL_INWATER;
    ent->damage_debounce_time=0; }
}

//============================================================
void M_droptofloor(edict_t *ent) {
vec3_t end;
trace_t tr;

  ent->s.origin[2]++;
  VectorCopy(ent->s.origin, end);
  end[2] -= 256;

  tr=gi.trace(ent->s.origin, ent->mins, ent->maxs, end, ent, MASK_MONSTERSOLID);

  if (tr.fraction == 1 || tr.allsolid)
    return;

  VectorCopy(tr.endpos, ent->s.origin);

  gi.linkentity(ent);
  M_CheckGround(ent);
  M_CatagorizePosition(ent);
}

//============================================================
void M_SetEffects(edict_t *ent) {

  ent->s.effects &= ~(EF_COLOR_SHELL|EF_POWERSCREEN);
  ent->s.renderfx &= ~(RF_SHELL_RED|RF_SHELL_GREEN|RF_SHELL_BLUE);

  if (ent->monsterinfo.aiflags & AI_RESURRECTING) {
    ent->s.effects |= EF_COLOR_SHELL;
    ent->s.renderfx |= RF_SHELL_RED; }

  if (ent->health<=0)
    return;

  if (ent->powerarmor_time > level.time) {
    if (ent->monsterinfo.power_armor_type == POWER_ARMOR_SCREEN)
      ent->s.effects |= EF_POWERSCREEN;
    else if (ent->monsterinfo.power_armor_type == POWER_ARMOR_SHIELD) {
      ent->s.effects |= EF_COLOR_SHELL;
      ent->s.renderfx |= RF_SHELL_GREEN; } }
}

//============================================================
void M_MoveFrame(edict_t *self) {
mmove_t *move;
int index;

  move=self->monsterinfo.currentmove;
  self->nextthink=level.time+FRAMETIME;

  if ((self->monsterinfo.nextframe)
   && (self->monsterinfo.nextframe >= move->firstframe)
   && (self->monsterinfo.nextframe<=move->lastframe)) {
    self->s.frame=self->monsterinfo.nextframe;
    self->monsterinfo.nextframe=0; }
  else {
    if (self->s.frame == move->lastframe) {
      if (move->endfunc) {
        move->endfunc(self);
        // regrab move, endfunc is very likely to change it
        move=self->monsterinfo.currentmove;
        // check for death
        if (self->svflags & SVF_DEADMONSTER)
          return; } }

    if (self->s.frame < move->firstframe || self->s.frame > move->lastframe) {
      self->monsterinfo.aiflags &= ~AI_HOLD_FRAME;
      self->s.frame=move->firstframe; }
    else {
      if (!(self->monsterinfo.aiflags & AI_HOLD_FRAME)) {
        self->s.frame++;
        if (self->s.frame > move->lastframe)
          self->s.frame=move->firstframe; } } }

  index=self->s.frame-move->firstframe;
  if (move->frame[index].aifunc)
    if (!(self->monsterinfo.aiflags & AI_HOLD_FRAME))
      move->frame[index].aifunc(self, move->frame[index].dist*self->monsterinfo.scale);
    else
      move->frame[index].aifunc(self, 0);

  if (move->frame[index].thinkfunc)
    move->frame[index].thinkfunc(self);
}

//============================================================
void monster_think(edict_t *self) {

  // If there is a Rider presently riding..
  if (self->rider && self->rider->is_riding)
    // Want Flyer to target your enemy?
    if (self->rider->target_my_enemy)
      // And the rider has an enemy
      if (self->rider->enemy)
        // Make your enemy the Flyer's enemy
        self->enemy=self->rider->enemy;

  M_MoveFrame(self);
  if (self->linkcount != self->monsterinfo.linkcount) {
    self->monsterinfo.linkcount=self->linkcount;
    M_CheckGround(self); }
  M_CatagorizePosition(self);
  M_WorldEffects(self);
  M_SetEffects(self);
}

//============================================================
void monster_use(edict_t *self, edict_t *other, edict_t *activator) {
  if (self->enemy) return;
  if (self->health<=0) return;
  if (activator->flags & FL_NOTARGET) return;
  if (!(activator->client)
   && !(activator->monsterinfo.aiflags & AI_GOOD_GUY))
    return;
  self->enemy=activator;
  FoundTarget(self);
}

//============================================================
void monster_triggered_spawn(edict_t *self) {

  self->s.origin[2]++;

  Telefrag_All(self);

  self->solid=SOLID_BBOX;
  self->movetype=MOVETYPE_STEP;
  self->svflags &= ~SVF_NOCLIENT;
  self->air_finished=level.time+12;
  gi.linkentity(self);

  monster_start_go(self);

  if (self->enemy && !(self->spawnflags & 1) && !(self->enemy->flags & FL_NOTARGET))
    FoundTarget(self);
  else
    self->enemy=NULL;
}

//============================================================
void monster_triggered_spawn_use(edict_t *self, edict_t *other, edict_t *activator) {
  self->think=monster_triggered_spawn;
  self->nextthink=level.time+FRAMETIME;
  if (activator->client)
    self->enemy=activator;
  self->use=monster_use;
}

//============================================================
void monster_triggered_start(edict_t *self) {
  self->solid=SOLID_NOT;
  self->movetype=MOVETYPE_NONE;
  self->svflags |= SVF_NOCLIENT;
  self->nextthink=0;
  self->use=monster_triggered_spawn_use;
}

//============================================================
// Generic Monster dead function (replaced similar functions).
//============================================================
void monster_dead(edict_t *monster) {
  VectorSet(monster->mins, -16, -16, -24);
  VectorSet(monster->maxs, 16, 16, -8);
  monster->movetype=MOVETYPE_TOSS;
  monster->svflags |= SVF_DEADMONSTER;
  monster->nextthink=0;
  gi.linkentity(monster);
  M_FlyCheck(monster);
}

//==================================================
qboolean monster_start(edict_t *self) {

  if ((self->spawnflags & 4) && !(self->monsterinfo.aiflags & AI_GOOD_GUY)) {
    self->spawnflags &= ~4;
    self->spawnflags |= 1; }

  if (!(self->monsterinfo.aiflags & AI_GOOD_GUY))
    level.total_monsters++;

  self->nextthink=level.time+FRAMETIME;
  self->svflags |= SVF_MONSTER;
  self->s.renderfx |= RF_FRAMELERP;
  self->takedamage=DAMAGE_AIM;
  self->air_finished=level.time+12;
  self->use=monster_use;
  self->max_health=self->health;
  self->clipmask=MASK_MONSTERSOLID;

  self->s.skinnum=0;
  self->deadflag=DEAD_NO;
  self->svflags &= ~SVF_DEADMONSTER;

  if (!self->monsterinfo.checkattack)
    self->monsterinfo.checkattack=M_CheckAttack;
  VectorCopy(self->s.origin, self->s.old_origin);

  if (st.item) {
    self->item=FindItemByClassname(st.item);
    if (!self->item)
      gi.dprintf("%s at %s has bad item: %s\n", self->classname, vtos(self->s.origin), st.item); }

  // randomize what frame they start on
  if (self->monsterinfo.currentmove)
    self->s.frame=self->monsterinfo.currentmove->firstframe+(rand() %(self->monsterinfo.currentmove->lastframe-self->monsterinfo.currentmove->firstframe+1));

  return true;
}

//============================================================
void monster_start_go(edict_t *self) {
vec3_t v;
qboolean notcombat=false;
qboolean fixup=false;
edict_t *target=NULL;

  if (self->health<=0) return;

  // check for target to combat_point and change to combattarget
  if (self->target) {
    while ((target=G_Find(target, FOFS(targetname), self->target)) != NULL)
      if (!Q_stricmp(target->classname, "point_combat")) {
        self->combattarget=self->target;
        fixup=true; }
      else
        notcombat=true;
    if (notcombat && self->combattarget)
      gi.dprintf("%s at %s has target with mixed types\n", self->classname, vtos(self->s.origin));
    if (fixup)
      self->target=NULL; }

  // validate combattarget
  if (self->combattarget) {
    target=NULL;
    while ((target=G_Find(target, FOFS(targetname), self->combattarget)) != NULL)
      if (Q_stricmp(target->classname, "point_combat"))
        gi.dprintf("%s at(%i %i %i) has a bad combattarget %s : %s at(%i %i %i)\n",
          self->classname,(int)self->s.origin[0],(int)self->s.origin[1],(int)self->s.origin[2],
          self->combattarget, target->classname,(int)target->s.origin[0],(int)target->s.origin[1],
         (int)target->s.origin[2]); }

  if (self->target) {
    self->goalentity=self->movetarget=G_PickTarget(self->target);
    if (!self->movetarget) {
      gi.dprintf("%s can't find target %s at %s\n", self->classname, self->target, vtos(self->s.origin));
      self->target=NULL;
      self->monsterinfo.pausetime=100000000;
      self->monsterinfo.stand(self); }
    else if (!Q_stricmp(self->movetarget->classname, "path_corner")) {
      VectorSubtract(self->goalentity->s.origin, self->s.origin, v);
      self->ideal_yaw=self->s.angles[YAW]=vectoyaw(v);
      self->monsterinfo.walk(self);
      self->target=NULL; }
    else {
      self->goalentity=self->movetarget=NULL;
      self->monsterinfo.pausetime=100000000;
      self->monsterinfo.stand(self); } }
  else {
    self->monsterinfo.pausetime=100000000;
    self->monsterinfo.stand(self); }

  self->think=monster_think;
  self->nextthink=level.time+FRAMETIME;
}

//============================================================
void walkmonster_start_go(edict_t *self) {

  if (!(self->spawnflags & 2) && level.time < 1) {
    M_droptofloor(self);
    if (self->groundentity)
      if (!M_walkmove(self, 0, 0))
        gi.dprintf("%s in solid at %s\n", self->classname, vtos(self->s.origin)); }

  if (!self->yaw_speed)
    self->yaw_speed=20;
  self->viewheight=25;

  monster_start_go(self);

  if (self->spawnflags & 2)
    monster_triggered_start(self);
}

//============================================================
void walkmonster_start(edict_t *self) {
  self->think=walkmonster_start_go;
  monster_start(self);
}

//============================================================
void flymonster_start_go(edict_t *self) {

  if (!M_walkmove(self, 0, 0))
    gi.dprintf("%s in solid at %s\n", self->classname, vtos(self->s.origin));

  if (!self->yaw_speed)
    self->yaw_speed=10;

  self->viewheight=25;

  monster_start_go(self);

  if (self->spawnflags & 2)
    monster_triggered_start(self);
}

//============================================================
void flymonster_start(edict_t *self) {
  self->flags |= FL_FLY;
  self->think=flymonster_start_go;
  monster_start(self);
}


//======================================================
//======== GENERIC MONSTER SPAWNING ROUTINES ===========
//======================================================

//======================================================
// Timer entity initiates the Monsters monster-Detonation.
//======================================================
void Kill_Monster(edict_t *timer) {

  // Is Monster still alive?
  if (timer->activator)
    // Maximally damage this monster (5000 dmg units ought to do it!!)
    T_Damage(timer->activator, timer, timer->owner, zvec, timer->activator->s.origin, NULL, 5000, 1, 0, MOD_SPLASH);

  G_FreeEdict(timer);
}

//=========================================================
// Spawn a Monster of type 'mtype' at 'torigin' and timer
// will monster-destruct monster in number of 'secs'.
//=========================================================
qboolean G_Spawn_Monster(edict_t *ent, vec3_t torigin, int mtype, float secs) {
edict_t *monster;
edict_t *timer;

  if (secs<=0) {
    gi_cprintf(ent, PRINT_HIGH,"G_Spawn_Monster called with 0 secs??\n");
    return false; }

  if (mtype!=M_FLIPPER)
    if (gi.pointcontents(torigin) & CONTENTS_SOLID) {
      gi_cprintf(ent,PRINT_HIGH,"Monster materialized in solid matter!\n");
      return false; }

  // Create basic entity stuff...
  monster = G_Spawn();
  monster->owner=world;
  monster->activator=ent;
  monster->mtype=mtype; // Type of monster this is (for obits)..
  VectorCopy(torigin, monster->s.origin);
  gi.linkentity(monster);

  MonstersInUse=true; // Set Global Flag!

  // -------------------------
  // Now the Monster Stuff..
  // -------------------------

  // Force skill to most difficult level- hehe!!
  gi.cvar_forceset("skill", va("%f", 3.0));

  // Create this monster type
  switch (mtype) {
    case M_BERSERK:   SP_monster_berserk(monster);
                      monster->classname = "XBerserk";break;
    case M_BOSS2:     SP_monster_boss2(monster);
                      monster->classname = "XBoss2";break;
    case M_SOLDIERSS: SP_monster_soldier_ss(monster);
                      monster->classname = "XSoldierSS";break;
    case M_JORG:      SP_monster_jorg(monster);
                      monster->classname = "XJorg";break;
    case M_BRAIN:     SP_monster_brain(monster);
                      monster->classname = "XBrain";break;
    case M_CHICK:     SP_monster_chick(monster);
                      monster->classname = "XChick";break;
    case M_FLIPPER:   SP_monster_flipper(monster);
                      monster->classname = "XShark";break;
    case M_FLOATER:   SP_monster_floater(monster);
                      monster->classname = "XFloater";break;
    case M_FLYER:     SP_monster_flyer(monster);
                      monster->classname = "XFlyer";break;
    case M_INSANE:    SP_misc_insane(monster);
                      monster->classname = "XInsane";break;
    case M_GLADIATOR: SP_monster_gladiator(monster);
                      monster->classname = "XGladiator";break;
    case M_HOVER:     SP_monster_hover(monster);
                      monster->classname = "XIcarus";break;
    case M_INFANTRY:  SP_monster_infantry(monster);
                      monster->classname = "XInfantry";break;
    case M_SOLDIERLT: SP_monster_soldier_light(monster);
                      monster->classname = "XSoldierLT";break;
    case M_SOLDIER:   SP_monster_soldier(monster);
                      monster->classname = "XSoldier";break;
    case M_MEDIC:     SP_monster_medic(monster);
                      monster->classname = "XMedic";break;
    case M_MUTANT:    SP_monster_mutant(monster);
                      monster->classname = "XMutant";break;
    case M_PARASITE:  SP_monster_parasite(monster);
                      monster->classname = "XParasite";break;
    case M_TANK:      SP_monster_tank(monster);
                      monster->classname = "XTank";break;
    case M_MAKRON:    SP_monster_makron(monster);
                      monster->classname = "XMakron";break;
    case M_GUNNER:    SP_monster_gunner(monster);
                      monster->classname = "XGunner";break;
    case M_SUPERTANK: SP_monster_supertank(monster);
                      monster->classname = "XSuperTank";
    } // end switch

  // Set ai for Brutal Mode and to start pursuit ASAP!!
  monster->monsterinfo.aiflags = AI_BRUTAL & AI_PURSUE_NEXT;

  // Play a powerup sound..
  gi.sound(monster, CHAN_VOICE, POWER1_SOUND, 1, ATTN_IDLE, 0);

  // -------------------------------------
  // Timer destroys monster after # secs.
  // -------------------------------------

  timer=G_Spawn();
  timer->owner=ent;          // Link to Owner
  timer->activator=monster;  // Link to Monster
  timer->takedamage=DAMAGE_NO;
  timer->movetype=MOVETYPE_NONE;
  timer->solid = SOLID_NOT;
  VectorClear(timer->s.origin);
  VectorClear(timer->mins);
  VectorClear(timer->maxs);
  timer->think=Kill_Monster;
  timer->nextthink=level.time + secs;
  gi.linkentity(timer);

  // Link back to Owner
  ent->goalentity=timer;

  return true; // Spawn successful
}

//====================================================
// Spawn a monster from the command line..
//====================================================
void Cmd_Monsters_f(edict_t *ent, int mtype) {
vec3_t forward, torigin;

  // Not available to dead or respawning players!
  if (!G_ClientInGame(ent)) return;

  // Already spawned? Then detonate
  if (ent->goalentity) {
    Kill_Monster(ent->goalentity);
    ent->goalentity=NULL;
    gi_centerprintf(ent, "MONSTER DESTROYED\n");
    return; }

  // See if we can project origin forward 50 units...
  AngleVectors(ent->s.angles, forward, NULL, NULL);
  VectorMA(ent->s.origin, 50, forward, torigin);

  if (G_Spawn_Monster(ent, torigin, mtype, 180))
    gi_centerprintf(ent, "MOVE AWAY NOW!!\n");
}

