// STRUCTS.H

extern int sm_meat_index;
extern int snd_fry;
extern int meansOfDeath;
extern qboolean MonstersInUse;

typedef unsigned char byte;

typedef struct gclient_s gclient_t;
typedef struct edict_s edict_t;

typedef float vec_t;
typedef vec_t vec3_t[3];

extern vec3_t monster_flash_offset[];
extern vec3_t zvec;

//===============================================
//===============================================
//         Definition of Console Vars
//===============================================
//===============================================

typedef struct cvar_s {
  char   *name;       // name of cvar variable
  char   *string;
  char   *latched_string;  // for CVAR_LATCH vars
  int     flags;
  qboolean modified;  // set each time the cvar is changed
  float   value;      // present value of the cvar variable
  struct cvar_s *next;// next ptr in linked list
} cvar_t;

extern cvar_t *g_select_empty;
extern cvar_t *dedicated;
extern cvar_t *filterban;
extern cvar_t *flood_msgs;
extern cvar_t *flood_persecond;
extern cvar_t *flood_waitdelay;
extern cvar_t *gun_x, *gun_y, *gun_z;
extern cvar_t *run_pitch;
extern cvar_t *run_roll;
extern cvar_t *bob_up;
extern cvar_t *bob_pitch;
extern cvar_t *bob_roll;

// Server-Side Variables
extern cvar_t *sv_gravity;
extern cvar_t *sv_maxvelocity;
extern cvar_t *sv_rollspeed;
extern cvar_t *sv_rollangle;
extern cvar_t *sv_cheats;
extern cvar_t *sv_maplist;
extern cvar_t *maxentities;
extern cvar_t *deathmatch;
extern cvar_t *coop;
extern cvar_t *dmflags;
extern cvar_t *skill;
extern cvar_t *fraglimit;
extern cvar_t *timelimit;
extern cvar_t *password;
extern cvar_t *spectator_password;
extern cvar_t *needpass;
extern cvar_t *maxclients;
extern cvar_t *maxspectators;
extern cvar_t *sv_bestplayer;

//===============================================
//===============================================
//    Complete Definition of World Planes
//===============================================
//===============================================

typedef struct cplane_s {
  vec3_t normal;  // normal vector
  float  dist;    // distance
  byte   type;    // for fast side tests
  byte   signbits;// signx +(signy<<1) +(signz<<1)
  byte   pad[2];  // unused
} cplane_t;

//===============================================
//===============================================
//    Complete Definition of World Models
//===============================================
//===============================================

typedef struct cmodel_s {
  vec3_t mins,      // min position relative to origin
         maxs,      // max position relative to origin
         origin;    // for sounds or lights
  int    headnode;
} cmodel_t;

//===============================================
//===============================================
//    Complete Definition of World Surfaces
//===============================================
//===============================================
typedef struct csurface_s {
  char name[16];  // String name of surface, ie "sky",
  int  flags;     // See SURF_* flag types
  int  value;    // unused
} csurface_t;

//===============================================
//===============================================
//  Record Returned as BBOX moves thru World
//===============================================
//===============================================

typedef struct {
  qboolean    allsolid;   // if TRUE, plane is not valid
  qboolean    startsolid; // if TRUE, the initial point was in a solid area
  float       fraction;   // 1.0 = Trace from start to end not blocked!!
  vec3_t      endpos;     // final position
  cplane_t    plane;      // surface normal at impact
  csurface_t *surface;    // surface hit
  int         contents;   // contents on other side of surface hit
  struct edict_s *ent;    // not set by CM_*() functions
} trace_t;

//======================================================
//======================================================
//    Information on Each Player's Present Move State
//======================================================
//======================================================
// This structure needs to be communicated bit-accurate
// from the server to the client to guarantee that
// prediction stays in sync, so no floats are used.
// if any part of the game code modifies this struct, it
// will result in a prediction error of some degree.
//======================================================
//======================================================
typedef struct {
  pmtype_t pm_type;         // see pmtype_t enumerated types, ex: PM_NORMAL
  short    origin[3];       // vector origin
  short    velocity[3];     // vector velocity
  byte     pm_flags;        // see PMF_ defined flags, ex: PMF_DUCKED
  byte     pm_time;         // each unit = 8 ms
  short    gravity;         // per-entity gravity multiplier(1=normal)
  short    delta_angles[3]; // add to command angles to get view direction
} pmove_state_t;

//===============================================
//    Only used for Entity Area Links now
//===============================================
typedef struct link_s {
  struct link_s *prev, *next;
} link_t;

//===============================================
//    Only used for Monster Frames
//===============================================
typedef struct {
  void (*aifunc)(edict_t *self, float dist);
  float dist;
  void (*thinkfunc)(edict_t *self);
} mframe_t;

typedef struct {
  int firstframe;
  int lastframe;
  mframe_t *frame;
  void (*endfunc)(edict_t *self);
} mmove_t;

typedef struct {
  mmove_t *currentmove;
  int aiflags;
  int nextframe;
  float scale;
  void (*stand)(edict_t *self);
  void (*idle)(edict_t *self);
  void (*search)(edict_t *self);
  void (*walk)(edict_t *self);
  void (*run)(edict_t *self);
  void (*dodge)(edict_t *self, edict_t *other, float eta);
  void (*attack)(edict_t *self);
  void (*melee)(edict_t *self);
  void (*sight)(edict_t *self, edict_t *other);
  qboolean (*checkattack)(edict_t *self);
  float pausetime;
  float attack_finished;
  vec3_t saved_goal;
  float search_time;
  float trail_time;
  vec3_t last_sighting;
  int attack_state;
  int lefty;
  float idle_time;
  int linkcount;
  int power_armor_type;
  int power_armor_power;
} monsterinfo_t;

//===============================================
//===============================================
//    Complete Definition of Player's Armor
//===============================================
//===============================================
typedef struct {
  int   base_count;
  int   max_count;
  float normal_protection; // Amount of protection from this armor
  float energy_protection; // Amount of energy protection..
  int   armor;             // See #defined ARMOR_ defs, ex: ARMOR_JACKET
} gitem_armor_t;

//===============================================
//===============================================
//           Local Field Types
//===============================================
//===============================================
typedef struct {
  char *name;
  int ofs;
  fieldtype_t type;
  int flags;
} field_t;

extern field_t fields[];

//===============================================
//===============================================
//    Information Local to Each Level
//===============================================
//===============================================
typedef struct {
  int framenum;  // level.framenum
  float time;    // level.time

  char level_name[MAX_QPATH];// the descriptive name(Outer Base, etc)
  char mapname[MAX_QPATH];   // the server name(base1, etc)
  char nextmap[MAX_QPATH];   // go here when fraglimit is hit

  // intermission state
  float intermissiontime;   // time the intermission was started
  char *changemap;          // filename of next map to change
  int exitintermission;

  vec3_t intermission_origin;
  vec3_t intermission_angle;

  edict_t *sight_client;

  edict_t *sight_entity;
  int     sight_entity_framenum;

  edict_t *sound_entity;
  int     sound_entity_framenum;

  edict_t *sound2_entity;
  int     sound2_entity_framenum;

  int pic_health; // health Icon for HUD

  int total_secrets; // Total secrets existing on a level
  int found_secrets; // Number of secrets found on a level

  int total_goals; // Total goals existing on a level
  int found_goals; // Number of goals found on a level

  int total_monsters;  // Total monsters existing on a level
  int killed_monsters; // Number of monsters killed on a level

  edict_t *current_entity; // entity running from G_RunFrame
  int    body_que;         // dead bodies
  int    power_cubes;      // ugly necessity for coop

} level_locals_t;

extern level_locals_t level;

//===============================================
//===============================================
//    Complete Definition of User Commands
//===============================================
//===============================================
// A Complete Record of usercmd_t is sent by the
// Client to the server each client frame
//
typedef struct usercmd_s {
  byte  msec;
  byte  buttons;  // BUTTON_ATTACK, etc.
  short angles[3];
  short forwardmove,
        sidemove,
        upmove;
  byte  impulse;    // Used now as EFFECTS
  byte  lightlevel; // light level the player is standing on
} usercmd_t;

//===============================================
//===============================================
//    Complete Definition of Movement States
//===============================================
//===============================================
typedef struct {
  pmove_state_t  s;
  usercmd_t cmd;  // command(in)
  qboolean snapinitial;  // if s has been changed outside pmove
  int     numtouch;
  struct  edict_s *touchents[MAX_TOUCH];
  vec3_t  viewangles;  // clamped
  float   viewheight;  // height above origin where eyes are found
  vec3_t  mins, maxs;  // bounding box size
  struct  edict_s *groundentity;
  int     watertype,   // See #defined CONTENTS_ Enumerated Water Color Types:
          waterlevel;  // 0 - not in liquid
                       // 1 - feet in liquid
                       // 2 - on surface of liquid
                       // 3 - surrounded by liquid

  // callbacks to test the world
  trace_t(*trace)(vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end);

  // Returns the CONTENT_ defined types at a given point
  int(*pointcontents)(vec3_t point);
} pmove_t;

//===============================================
//===============================================
//    Complete Definition of WORLD Vars
//===============================================
//===============================================
// spawn_temp_t is only used to hold entity field values that
// can be set from the editor, but aren't actualy present
// in edict_t during gameplay

typedef struct {
  char   *sky;  // filename of the sky map
  float    skyrotate;
  vec3_t   skyaxis;
  char   *nextmap;    // filename of next map to rotate
  int      lip;
  int      distance;
  int      height;
  char   *noise;
  float    pausetime;
  char   *item;
  char   *gravity;  // per-entity gravity multiplier(1=normal)
  float    minyaw;
  float    maxyaw;
  float    minpitch;
  float    maxpitch;
} spawn_temp_t;

extern spawn_temp_t st; // For SP_worldspawn()

//===============================================
//===============================================
//  Complete Definition of Entity's Movements
//===============================================
//===============================================
typedef struct {
  vec3_t  start_origin;  // starting position
  vec3_t  start_angles;  // starting angle
  vec3_t  end_origin;    // ending position
  vec3_t  end_angles;    // ending angle

  int     sound_start;
  int     sound_middle;
  int     sound_end;

  float   accel;      // acceleration of platforms
  float   speed;      // speed of misc entities, including moving buttons,
  float   decel;      // deceleration of platforms

  float   distance;

  float   wait;   // unused

  int     state; // state data

  vec3_t  dir;   // Directional vector: forward, right, up

  float    current_speed;
  float    move_speed;
  float    next_speed;
  float    remaining_distance;
  float    decel_distance;

  void   (*endfunc)(edict_t *);

} moveinfo_t;

//===============================================
//===============================================
//   Complete Definition of Inventory Items
//===============================================
//===============================================
typedef struct gitem_s {
  char *classname;  // classname of spawning item

  // Items can call these function types
  qboolean (*pickup)(struct edict_s *ent, struct edict_s *other);
  void (*use)(struct edict_s *ent, struct gitem_s *item);
  void (*drop)(struct edict_s *ent, struct gitem_s *item);
  void (*weaponthink)(struct edict_s *ent);

  char *pickup_sound;
  char *world_model;
  int   world_model_flags;
  char *view_model;
  char *icon;
  char *pickup_name; // for printing on pickup
  int   count_width; // number of digits to display by icon
  int   quantity;   // for ammo how much, for weapons how much is used per shot
  char *ammo;       // for weapons
  int   flags;      // IT_* flags
  int   weapmodel;  // weapon model index (for weapons)
  void *info;       // pointer to jacketarmor_info, combatarmor_info, bodyarmor_info
  int   tag;
  char *precaches;  // string of all models, sounds, and images this item will use
} gitem_t;

extern gitem_t itemlist[];

// Item Ammo (IT_AMMO)
gitem_t *item_shells,
       *item_cells,
       *item_rockets,
       *item_grenades,
       *item_slugs,
       *item_bullets,

// Item Weapons (IT_WEAPON)
       *item_blaster,
       *item_shotgun,
       *item_machinegun,
       *item_supershotgun,
       *item_chaingun,
       *item_handgrenade,
       *item_grenadelauncher,
       *item_rocketlauncher,
       *item_hyperblaster,
       *item_railgun,
       *item_bfg,

// Item Armor (IT_ARMOR)
       *item_jacketarmor,
       *item_combatarmor,
       *item_bodyarmor,
       *item_armorshard,
       *item_powerscreen,
       *item_powershield,

// Item Health (IT_HEALTH)
       *item_adrenaline,
       *item_health,
       *item_stimpak,
       *item_health_large,
       *item_health_mega,

// Item Powerup (IT_POWERUP)
       *item_quad,
       *item_invulnerability,
       *item_silencer,
       *item_breather,
       *item_enviro,

// Item Pak (IT_PACK)
       *item_pack,
       *item_bandolier,

// Item Keys (IT_KEY)
       *item_ancient_head,
       *key_data_cd,
       *key_power_cube,
       *key_pyramid,
       *key_data_spinner,
       *key_pass,
       *key_blue_key,
       *key_red_key,
       *key_commander_head,
       *key_airstrike_target;

//===============================================
//===============================================
// client data that stays across multiple levels
//===============================================
//===============================================
typedef struct {

  char userinfo[MAX_INFO_STRING];
  char netname[16];
  int  hand;

  qboolean connected; // a loadgame will leave valid entities that
                      // just don't have a connection yet

  // values saved and restored from edicts when changing levels
  int health;
  int max_health;
  int savedFlags;

  int selected_item;
  int inventory[MAX_ITEMS];

  // ammo capacities
  int max_bullets;
  int max_shells;
  int max_rockets;
  int max_grenades;
  int max_cells;
  int max_slugs;

  // weapon items
  gitem_t *weapon;
  gitem_t *lastweapon;

  int power_cubes;  // used for tracking the cubes in coop games
  int score;      // for calculating total unit score in coop games

  int game_helpchanged;
  int helpchanged;

  qboolean spectator; // client is a spectator
} client_persistant_t;

//===============================================
//===============================================
//   client data that stays across respawns
//===============================================
//===============================================
typedef struct {
  client_persistant_t coop_respawn;  // what to set client->pers to on a respawn
  int      enterframe;   // level.framenum the client entered the game
  int      score;        // frags, etc
  vec3_t   cmd_angles;   // angles sent over in the last command
  qboolean spectator;    // client is a spectator
} client_respawn_t;

//===============================================
// entity_state_t is the information conveyed
// from the server in an update message about
// entities that the client will need to render
// in some way
//===============================================

typedef struct entity_state_s {
  int     number;      // edict index
  vec3_t  origin;
  vec3_t  angles;
  vec3_t  old_origin;  // for lerping
  int     modelindex;
  int     modelindex2,
          modelindex3, // unused
          modelindex4; // unused
  int     frame;
  int     skinnum;
  int     effects;   // see EF_ defined flags, ex: EF_PENT
  int     renderfx;  // see RF_ defined flags, ex: RF_SHELL_GREEN
  int     solid;     // for client side prediction, 8*(bits 0-4) is x/y radius
                     // 8*(bits 5-9) is z down distance, 8(bits10-15) is z up
                     // gi.linkentitysets this properly
  int     sound;     // for looping sounds, to guarantee shutoff
  int     event;     // see EV_ defined flags, ex: EV_FOOTSTEP
                     // Events only go out for a single frame, they
                     // are automatically cleared each frame
} entity_state_t;

//===================================================================
//===================================================================
// player_state_t is the information needed in addition to pmove_state_t
// to rendered a view.  There will only be 10 player_state_t sent each second,
// but the number of pmove_state_t changes will be relative to client
// frame rates
//===================================================================
//===================================================================
typedef struct {
  pmove_state_t pmove;// for prediction
  vec3_t viewangles;  // for fixed views
  vec3_t viewoffset;  // add to pmovestate->origin
  vec3_t kick_angles; // add to view direction to get render angles
                      // set by weapon kicks, pain effects, etc
  vec3_t gunangles;   // unused
  vec3_t gunoffset;   // see gun_x,y,z.
  int    gunindex;    // index into the ents item list.
  int    gunframe;    // Number of gun animation frames.
  float  blend[4];    // rgba full screen effect
  float  fov;         // horizontal field of view
  int    rdflags;     // refdef flags

  short stats[MAX_STATS]; // status bar updates
} player_state_t;

//===============================================
//===============================================
//    Complete Definition of Every Client
//===============================================
//===============================================
struct gclient_s {

  // known to server
  player_state_t ps;  // communicated by server to clients
  int ping;

  // private to game
  client_persistant_t  pers;
  client_respawn_t  resp;
  pmove_state_t old_pmove;  // for detecting out-of-pmove changes

  qboolean  showscores;      // set layout stat
  qboolean  showinventory;    // set layout stat
  qboolean  showhelp;
  qboolean  showhelpicon;

  int ammo_index;

  int buttons;
  int oldbuttons;
  int latched_buttons;

  qboolean weapon_thunk;

  gitem_t *newweapon;

  // sum up damage over an entire frame, so
  // shotgun blasts give a single big kick
  int damage_armor;     // damage absorbed by armor
  int damage_parmor;    // damage absorbed by power armor
  int damage_blood;     // damage taken out of health
  int damage_knockback; // impact damage
  vec3_t  damage_from;  // origin of damage

  float killer_yaw;     // when dead, look at killer

  weaponstate_t weaponstate; // see WEAPON_* flags

  vec3_t kick_angles;  // weapon kicks
  vec3_t kick_origin;
  float  v_dmg_roll, v_dmg_pitch, v_dmg_time;  // damage kicks
  float  fall_time, fall_value;    // for view drop on fall
  float  damage_alpha;
  float  bonus_alpha;
  vec3_t damage_blend;
  vec3_t v_angle;      // aiming direction
  float  bobtime;      // so off-ground doesn't change it
  vec3_t oldviewangles;
  vec3_t oldvelocity;

  float  next_drown_time;
  int    old_waterlevel;
  int    breather_sound;

  int    machinegun_shots;  // for weapon raising

  // animation vars
  int      anim_end;
  int      anim_priority;
  qboolean anim_duck;
  qboolean anim_run;

  // powerup timers
  float    quad_framenum;
  float    cloak_framenum;
  float    invincible_framenum;
  float    breather_framenum;
  float    enviro_framenum;

  qboolean grenade_blew_up;
  float    grenade_time;
  int      silencer_shots;
  int      weapon_sound;

  float    pickup_msg_time;

  float    flood_locktill;  // locked from talking
  float    flood_when[10];  // when messages were said
  int      flood_whenhead;  // head pointer for when said

  float    respawn_time;    // can respawn when time > this

  edict_t *chase_target;    // player we are chasing
  qboolean update_chase;    // need to update chase info?

//------- NEW VARS ---------//
  int      id_on;           // Player Identification 1=ON, 0=OFF
  int      hookstate;       // 1=ON, 0=OFF
  int      is_cloaked;      // 1=ON, 0=OFF
  int      hgrenade;        // Homing Handgrenade 1=ON, 0=OFF
  int      zylon_grenade;   // 1=ON, 0=OFF
  int      railbomb;        // 1=ON, 0=OFF
  int      baton;           // 1=ON, 0=OFF
  int      chamber;         // 1=ON, 0=OFF
  int      laserdrone;      // 1=ON, 0=OFF
  int      Tgrenade;        // 1=ON, 0=OFF
  int      is_trapped;      // Trapped in MegaHealth Deathpak Beam
};

//===============================================
//===============================================
//     Complete Definition of Every Entity
//===============================================
//===============================================
struct edict_s {

  entity_state_t  s;  // ptr to entity_state_t record structure

  //================================

  struct gclient_s *client;  // ptr to gclient_s record structure
                              // the server expects the first part
                              // of gclient_s to be a player_state_t
                              // but the rest of it is opaque
  //================================

  qboolean  inuse;  // entity exists in the world

  //================================

  int linkcount;  // number of records in linked list

  //================================

  link_t area;    // linked to a division node or leaf

  //================================

  int num_clusters;  // if -1, use headnode instead
  int clusternums[MAX_ENT_CLUSTERS];

  //================================

  int headnode; // unused if num_clusters != -1

  //================================

  int areanum,   // area where entity currently is positioned
      areanum2;  // Unused

  //================================

  int svflags;  // SVF_NOCLIENT    - don't send to clients
                // SVF_DEADMONSTER - a dead monster
                // SVF_MONSTER     - same, a bit earlier

  //================================

  vec3_t  mins,      // min position of entity relative to origin
          maxs,      // max position of entity relative to origin
          absmin,    // absolute minimum value
          absmax,    // absolute maximum value
          size;      // size of entity(vector from mins to maxs)

  //================================
  solid_t  solid;    // See #defined SOLID_ solidity of entity types

  int      clipmask; // current state of NO_CLIP

  edict_t *owner;   // usually the entity that spawned this entity
                     //(eg bullet is 'owned' by the monster that fired it)

  //===============================================
  // DO NOT MODIFY ANYTHING ABOVE THIS, THE SERVER
  // EXPECTS THE FIELDS IN A PARTICULAR ORDER!
  //===============================================

  int      movetype; // See MOVETYPE_ entity moves types

  int      flags;    // See FL_ entity Special Characteristics Types

  char   *model;    // filename of entity's model

  float    freetime; // sv.time when the object was freed in memory

  //=========================================
  // only used locally in game, not by server
  //=========================================

  char   *message;     // message(if any) when entity is activated
  char   *classname;   // type of entity(eg 'monster_soldier')
  int     spawnflags;   // See #defined SPAWNFLAG_ Enumerated Types
  float   timestamp;    // used by various functions to mark the spawn time of an
                        // entity or the start of a change(eg a light ramp)

  float   angle;        // entity's present view angle.

  char   *target;      // type of entity being targeted
  char   *targetname;  // name of entity being targeted
  char   *killtarget;  // name of entity to remove when killed
  char   *team;        // name of team(if any)
  char   *pathtarget;  // name of entity to move towards
  char   *deathtarget; // name of entity when current entity dies
  char   *combattarget;// name of entity to attack(used once??)

  edict_t *target_ent;  // record of enemy being targeted

  //================================

  float  speed,   // current speed of entity
         accel,   // acceleration of entity
         decel;   // deceleration of entity

  //================================

  vec3_t  movedir,   // entity's current direction of movement

          pos1,      // top position for Lift?
          pos2,      // bottom position of Lift?

          velocity,  // entity's present velocity
          avelocity; // entity's present angular-velocity

  //================================

  int      mass;         // entity's mass used when calculating kickback force

  float    air_finished; // calculated time when entity's air will run out

  float    gravity;      // per entity gravity multiplier(1.0 is normal)

  //================================

  edict_t *goalentity; // currently seeking this particular entity

  edict_t *movetarget; // entity to wark towards when not got anything better to do

  //================================

  float    yaw_speed;  // speed at which yaw changes
  float    ideal_yaw;  // yaw the entity wants to face

  //================================

  float nextthink;  // time at which entity should call its think function

  // Entity's PRETHINK Function used exclusively by G_RunEntity()
  void (*prethink)(edict_t *ent);

  // Entity's THINK Function to call -
  void (*think)(edict_t *self);

  // Used exclusively by g_func.c
  void (*blocked)(edict_t *self, edict_t *other);

  // Entity's TOUCH Function to call upon touching item or another entity
  void (*touch)(edict_t *self, edict_t *other, cplane_t *plane, csurface_t *surf);

  // Entity's USE Function to call upon item's use
  void (*use)(edict_t *self, edict_t *other, edict_t *activator);

  // unused(P_DamageFeedback() calculates pain now..)
  void (*pain)(edict_t *self, edict_t *other, float kick, int damage);

  // Entity's DIE Function to call upon death
  void (*die)(edict_t *self, edict_t *inflictor, edict_t *attacker, int damage, vec3_t point);

  //================================

  float touch_debounce_time;     // prevent touching over and over again
  float pain_debounce_time;      // prevent pain sounds/anims playing to often
  float damage_debounce_time;    // prevent damaging over and over again
  float fly_sound_debounce_time; // prevent replay of fly_sound over and over
  float last_move_time;          // time to play earthquake sound again
  float id_timer;                // for player identification.

  //================================

  int   health;        // present value of entity's health
  int   max_health;    // maximum value health can obtain
  int   gib_health;    //

  //================================

  int      deadflag;      // See #defined DEAD_ Enumerated States of Deadness types

  qboolean show_hostile;  // any monsters seeing this one will inherit its enemy

  float    powerarmor_time; // time to stop showing translucent shield effects

  char   *map;           // fileame of the next map

  int      viewheight;    // height above origin where eyesight is determined

  int      takedamage;    // See Entity DAMAGE_ Enumerated Types

  int      dmg;           // amount of damage done to current enemy
  int      radius_dmg;    // amount of radius damage
  float    dmg_radius;    // area of radius damage

  int      sounds;        // type of sounds to play - used by doors, lifts, etc

  int      count;         // unused

  edict_t *chain,         // ptr to first record in linked-list of entity structs
         *enemy,         // ptr to record of current enemy under attack
         *oldenemy,      // ptr to record of previous enemy under attack
         *activator,     // ptr to record of entity causing item's activation
         *groundentity;  // ptr to record of entity which being stood upon

  int groundentity_linkcount;// Number of linked groundentity records

  edict_t *teamchain,   // pointer of first record of linked-list of team entity structs
         *teammaster;  // pointer to record team master(only 1 master record)

  edict_t *mynoise,     // spawned when player makes a noise, so monsters can find them
         *mynoise2;    // Unused

  int     noise_index,  // index of sound to play when entity is triggered
          noise_index2; // Unused

  float   volume,       // volume when playing above sound
          attenuation;  // distance from sound point of origin

  float   wait,       // Idle the entity for the WAIT period
          delay,      // time between triggering switch and effect activating
          random;     // used to add a bit of time variation to func_timer

  float   teleport_time; // time when entity teleported into level

  int     watertype,     // See Entity CONTENTS_ Enumerated Types
          waterlevel;    // depth of liquid :
                         // 0 - not in liquid
                         // 1 - feet in liquid
                         // 2 - on surface of liquid
                         // 3 - surrounded by liquid

  vec3_t  move_origin,  // the origin of the gun turret
          move_angles;  // angle of the gun turret

  int     light_level;  // player's light level, used by monsters when checking
                        // if player can be seen

  int      style;       // type of entities(eg type of health box) also used as
                        // areaportal number(something to do with separating areas)

  gitem_t *item;        // pointer to record for entity's current items

  moveinfo_t  moveinfo; // record structure of player's movements

  monsterinfo_t  monsterinfo;

  qboolean airstrike_called;   // TRUE if Airstrike called
  vec3_t   airstrike_start;    // Position of Viper Start.
  vec3_t   airstrike_target;   // Position of Crosshairs.
  float    airstrike_time;     // Timer for incoming missiles
  int      airstrike_type;     // 1=Rocket, 2=Cluster, 3=BFG Nuke
  int      mtype;              // Type of Monster: see M_* defines.. (M_HOVER, etc)
  int      zylon_timer;        // duration of zylon gas clouds
  edict_t  *pad;               // For the Torture Chamber
  edict_t  *flyer;           // Monster 'Flyer' Vehicle
  edict_t  *rider;             // Flyer's Rider..
  qboolean is_riding;          // Is ent on top of Flyer?
  qboolean target_my_enemy;    // TRUE if Flyer to target your enemy too.
};

extern edict_t *g_edicts;

//===============================================================
//===============================================================
// This structure is left intact through an entire game
// it should be initialized at dll load time, and read/written to
// the server.ssv file for savegames
//===============================================================
//===============================================================
typedef struct {

  char  helpmessage1[512];
  char  helpmessage2[512];
  int   helpchanged;    // flash F1 icon if non 0, play sound
                        // and increment only if 1, 2, or 3

  gclient_t *clients;   // [1..maxclients]

//===============================================
// can't store spawnpoint in level, because
// it would get overwritten by the savegame restore
//===============================================

  char  spawnpoint[512];  // needed for coop respawns

//===============================================
// store latched cvars here that we want to get at often
//===============================================
  int    maxclients;  // ga.maxclients
  int    maxentities; // ga.maxentities

//===============================================
// cross level triggers
//===============================================
  int    serverflags;  // ga.serverflags

//===============================================
// items
//===============================================
  int    num_items; // ga.num_items

//===============================================
  qboolean  autosaved;

} game_locals_t;

extern game_locals_t game, ga;

//===============================================
//===============================================
//    Functions provided by the main engine
//===============================================
//===============================================

typedef struct {

//===========================================================
//============== Text Management Routines ===================
//===========================================================
// examples:
//
// Broadcast to all printf(msg) with priority PRINT_HIGH
//    gi_bprintf(PRINT_HIGH, "%s overflowed\n", ENTS_NETNAME);
//
// Print only to 'ent' the printf(msg) with PRINT_HIGH priority
//    gi_cprintf(ent, PRINT_HIGH, "Your damage increased from %d ", damage);
//
// Print only to 'ent' the printf(msg) at center of screen
//    gi_centerprintf(ent, "%s", ent->message);
//
// Debug - simple printf()
//    gi.dprintf("G_PickTarget called with NULL targetname\n");
//
//===========================================================

  // Print message to all players with PRINT_ type priority
  void (*bprintf)(int printlevel, char *fmt, ...);

  // Print debug message(set debug cvar)
  void (*dprintf)(char *fmt, ...);

  // Print message with a PRINT_ type priority
  void (*cprintf)(edict_t *ent, int printlevel, char *fmt, ...);

  // Print message in centre of Client's screen
  void (*centerprintf)(edict_t *ent, char *fmt, ...);

//===========================================================
//============= Sound Management Routines ===================
//===========================================================
//
// gi.sound() generates sound to the desired entity
// Channel: a given CHAN_ which specifies sort of sound to play
// SoundIndex: the sound to play is returned by gi.soundindex
// Volume: level is between 0 and 1
// Attenuation: is ATTN_ flag and tells how far away sound can be heard
// time_ofs: is time before playing sound?(not yet found anything other than 0)
// examples:
//
// Broadcast Sound from 'ent' on 'CHAN_AUTO' using 'index'
// at volume with attenuation 'ATTN_NORM' and ??
//    gi.sound(activator, CHAN_AUTO, index, volume, ATTN_NORM, 0);
//
// Broadcast sound at 'origin' of 'ent' over 'chan' using 'index'
// at volume with attenuation 'ATTN_NORM' and ??
//    gi.positioned_sound(origin, ent, chan, index, volume, attenuation, 0);
//
//===========================================================

   void (*sound)(edict_t *ent, int channel, int soundindex, float volume, float attenuation, float timeofs);

   void (*positioned_sound)(vec3_t origin, edict_t *ent, int channel, int soundinedex, float volume, float attenuation, float timeofs);

//===========================================================
//============ Server/Client Communication ==================
//===========================================================
// Config strings is an array holding all the config strings.
// All of the current configstrings are sent to clients when
// they connect, and changes are sent to all connected clients.
//
// example:
//
// Write 'name' to 'configstring' at position 'index'
//    gi.configstring(index, name); }
//
//====================================================================

  void (*configstring)(int num, char *string);

//===========================================================
//================ Error Handling Routines ==================
//===========================================================
//
// Shuts down the game and Prints error message to stdout
//    gi.error(ERR_FATAL, "G_Spawn(): No free edicts!!");
//
//====================================================================

  // Shuts down the game and displays the error message
  void (*error)(char *fmt, ...);

//===========================================================
//======== Precached Item's Information Indexing ============
//===========================================================
// Return the index number for the sound file "misc/tele1.wav"
//    index=gi.soundindex("misc/tele1.wav");
//
// Return the model number for image 'p_quad'
//    index=gi.imageindex("p_quad")
//
// Sets the model for the 'ent'
//  gi.setmodel(ent, "models/objects/gibs/arm/tris.md2");
//
//===========================================================

  int (*modelindex)(char *name);  // Pointer to Precached Models
  int (*soundindex)(char *name);  // Pointer to Precached Sound
  int (*imageindex)(char *name);  // Pointer to Precached Icons
  void (*setmodel)(edict_t *ent, char *name); // Precached Model's Filename

//===========================================================
//=========== Collision Detection Routines ==================
//===========================================================
//
// gi.trace(point1, min, max, point2, ignore, masktype)
//
// Traces a box from point1 to point2, ignoring entity ignore,
// stopping if it hits an object of type specified in mask.
// Mask is one of the MASK_ defined types
// Vector1 and vector2 define the box which will do the tracing
// - if NULL, then a line is used instead
//
// gi.trace() returns value of type trace_t with attributes :
//   .allsolid - if true, entire trace was in a wall
//   .startsolid - if true, trace started in a wall
//   .fraction - fraction of trace completed(1.0 = totally completed)
//   .endpos - point where trace ended
//   .plane - surface normal at hitpoint (type cplane_t)
//   .surface - type surface hit (type csurface_t)
//   .contents - of point at end of trace(see gi.pointcontents for values)
//   .ent - entity hit by trace
//
// example:
//     tr=gi.trace(origin, mins, maxs, s.origin, NULL, MASK_PLAYERSOLID);
//
//===========================================================

  trace_t (*trace)(vec3_t start, vec3_t mins, vec3_t maxs, vec3_t end, edict_t *passent, int contentmask);

//===========================================================
//=== Retrieving Information about World at single point ====
//===========================================================
//
// Returns the type of CONTENT_ at a given point
//    if (gi.pointcontents(point) & MASK_WATER)
//
//==========
// Checks to see if point2 is in the Potentially Visible Set of point1.
// The potentially visible set(PVS) is a data structure used by Q2 to
// determine visibility and lines of sight.
// Note that if point2 is in the PVS of point1, then point1 will also be
// in the PVS of point2, hence the first law of optics: if I can see you,
// then you can see me.
//
// example:
//
//   if (gi.inPVS(start,end))
//
//==========
//
// Checks to see if point2 is in the Potentially Hearable Set of point1.
// The PHS is the same concept as a PVS, except it tracks sound instead
// of line of sight.
//
// example:
//
//   if (gi.inPHS(start,end))
//
//===========================================================

  int (*pointcontents)(vec3_t point);

  qboolean (*inPVS)(vec3_t p1, vec3_t p2);

  qboolean (*inPHS)(vec3_t p1, vec3_t p2);

//===========================================================
//========= Managing States of the World ====================
//===========================================================
//
// Set when a door opens or closes (to prevent sound detection
// through enclosed spaces)
//
//  gi.SetAreaPortalState(ent->style, ent->count);
//
//============
// Checks if two areas are connected - used to check whether a
// monster can hear a sound/see something or not
//
// if (gi.AreasConnected(area1, area2))
//
//===========================================================

  void (*SetAreaPortalState)(int portalnum, qboolean open);

  qboolean (*AreasConnected)(int area1, int area2);

//===========================================================
//=========== Linking Entities to World =====================
//===========================================================
// An entity will never be sent to a client or used for collision
// if it is not passed to linkentity.  If the size, position, or
// solidity changes, it must be relinked.
//
// Links entity into the world so that it is sent to the client and
// used for collision detection, etc. must be re-linked if its size,
// position or solidity changes
//
// Link the entity 'trigger' to the world
//    gi.linkentity(trigger);
//
//==========
// Stop entity from interacting with the world.
// Must be done before deleting entity/item.
//
// Unlink the entity 'ed' from the world
//    gi.unlinkentity(ed);
//
//==========
// Generates a list of all the entities in a certain box mins and maxs
//(define the box relative to point of origin)
//
//  num=gi.BoxEdicts(ent->absmin, ent->absmax, touch, MAX_EDICTS, AREA_TRIGGERS);
//
//===========================================================

  void (*linkentity)(edict_t *ent);

  void (*unlinkentity)(edict_t *ent);

  int (*BoxEdicts)(vec3_t mins, vec3_t maxs, edict_t **list, int maxcount, int areatype);

//===========================================================
//========= Update Engine on Player's Movements =============
//===========================================================
//
// Player movement code common with client prediction communicated
// to gaming engine through struct pmove_t
//
// ent->client->ps.pmove.pm_type=PM_FREEZE;
// pm_passent=ent;
// memset(&pm, 0, sizeof(pm));
// ent->client->ps.pmove.pm_type=PM_SPECTATOR;
// ent->client->ps.pmove.gravity=sv_gravity->value;
// pm.s=ent->client->ps.pmove;
//
// for (i=0 ; i<3 ; i++) {
//  pm.s.origin[i]=ENTS_S_ORIGIN[i]*8;
//  pm.s.velocity[i]=ENTS_VELOCITY[i]*8; }
//
// if (memcmp(&ent->client->old_pmove, &pm.s, sizeof(pm.s)))
//   pm.snapinitial=true;
//
// pm.cmd=*ucmd;
//
// pm.trace=PM_trace;  // adds default parms
// pm.pointcontents=gi.pointcontents;
//
// gi.Pmove(&pm); // Update Gaming Engine
//
//===========================================================

  void (*Pmove)(pmove_t *pmove);

//===========================================================
//========= Information Broadcasting Routines ===============
//===========================================================
//
// Broadcast to all entities from 'origin' to MULTI_CAST flags
//
//   gi.multicast(ENTS_S_ORIGIN, MULTICAST_PVS);
//
// Broadcast to a single entity(set reliable to true for
// important messages)/
//
//    gi.unicast(ent, true);
//
//===========================================================

  void (*multicast)(vec3_t origin, multicast_t to);

  void (*unicast)(edict_t *ent, qboolean reliable);

//===========================================================
//========== Send Individual Messages to Engine =============
//===========================================================
//
// Write directly into the engine's message buffer..
//
// These routines are important for entity creation or for
// spawning an effect into the world.  Communication is order
// specific for certain activities(see routines in g_spawn.c).
//
//===========================================================

  void (*WriteChar) (int c);
  void (*WriteByte) (int c);
  void (*WriteShort)(int c);
  void (*WriteLong) (int c);
  void (*WriteFloat)(float f);
  void (*WriteString)(char *s);
  void (*WritePosition)(vec3_t pos);
  void (*WriteDir)(vec3_t pos);
  void (*WriteAngle)(float f);

//===========================================================
//============== Memory Management Routines =================
//===========================================================
//
// allocate 'size' bytes of memory, referenced by tag
//
// examples:
//  self->message=gi.TagMalloc(CLOCK_MESSAGE_SIZE, TAG_LEVEL);
//
//  g_edicts= gi.TagMalloc(ga.maxentities * sizeof(g_edicts[0]), TAG_GAME);
//
//  game.clients=gi.TagMalloc(ga.maxclients*sizeof(game.clients[0]),TAG_GAME);
//
//  newb=gi.TagMalloc(l, TAG_LEVEL);
//
//===============
// Releases the tag back to memory(unused)
//   gi.TagFree(block);
//
//===============
// Free memory space referenced by tag
//
// examples:
//  gi.FreeTags(TAG_LEVEL);
//  gi.FreeTags(TAG_GAME);
//
//===========================================================

  void *(*TagMalloc)(int size, int tag);

  void (*TagFree)(void *block); // unused?

  void (*FreeTags)(int tag);

//===========================================================
//========= Managing Console Variable Information ===========
//===========================================================
//
// Types of CVARS are defined by the pre-defined console variable
// flags of type:
//
//  CVAR_ARCHIVE    - set to be saved to vars.rc
//  CVAR_USERINFO   - added to userinfo when changed
//  CVAR_SERVERINFO - added to serverinfo when changed
//  CVAR_NOSET      - don't allow change from console at all,
//  CVAR_LATCH      - save changes until server restart
//
//=============
//
// gi.cvar() returns the value of the specified cvar variable..
//
// examples:
//  gun_x=gi.cvar("gun_x", "0", 0);
//  gun_y=gi.cvar("gun_y", "0", 0);
//  gun_z=gi.cvar("gun_z", "0", 0);
//  run_pitch=gi.cvar("run_pitch", "0.002", 0);
//  run_roll=gi.cvar("run_roll", "0.005", 0);
//  bob_up=gi.cvar("bob_up", "0.005", 0);
//  bob_pitch=gi.cvar("bob_pitch", "0.002", 0);
//  bob_roll=gi.cvar("bob_roll", "0.002", 0);
//
//  sv_rollspeed=gi.cvar("sv_rollspeed", "200", 0);
//  sv_rollangle=gi.cvar("sv_rollangle", "2", 0);
//  sv_maxvelocity=gi.cvar("sv_maxvelocity", "2000", 0);
//  sv_gravity=gi.cvar("sv_gravity", "800", 0);
//  sv_cheats=gi.cvar("cheats", "0", CVAR_SERVERINFO | CVAR_LATCH);
//
//  dedicated=gi.cvar("dedicated", "0", CVAR_NOSET);
//
//  gi.cvar("gamename", GAMEVERSION , CVAR_SERVERINFO | CVAR_LATCH);
//  gi.cvar("gamedate", __DATE__ , CVAR_SERVERINFO | CVAR_LATCH);
//
//  maxclients=gi.cvar("maxclients", "4", CVAR_SERVERINFO | CVAR_LATCH);
//  deathmatch=gi.cvar("deathmatch", "0", CVAR_LATCH);
//  coop=gi.cvar("coop", "0", CVAR_LATCH);
//  skill=gi.cvar("skill", "1", CVAR_LATCH);
//  maxentities=gi.cvar("maxentities", "1024", CVAR_LATCH);
//
//  dmflags=gi.cvar("dmflags", "0", CVAR_SERVERINFO);
//  fraglimit=gi.cvar("fraglimit", "0", CVAR_SERVERINFO);
//  timelimit=gi.cvar("timelimit", "0", CVAR_SERVERINFO);
//  password=gi.cvar("password", "", CVAR_USERINFO);
//
//  g_select_empty=gi.cvar("g_select_empty", "0", CVAR_ARCHIVE);
//
//==============
// Sets the variable to value returns the variable just set
//
// examples:
//
//  if (!st.gravity)
//    gi.cvar_set("sv_gravity", "800");
//  else
//    gi.cvar_set("sv_gravity", st.gravity);
//
//==============
// Forces the console variable to be set(hard override)
//
// examples:
//
//  gi.cvar_forceset("skill", va("%f", skill_level));
//
//===========================================================

  cvar_t *(*cvar)(char *var_name, char *value, int flags);

  cvar_t *(*cvar_set)(char *var_name, char *value);

  cvar_t *(*cvar_forceset)(char *var_name, char *value);

//===========================================================
//============ Receiving Console Commands as input ==========
//===========================================================
//
// Parsing user commands entered at the console.
//
//=================
//
// Returns the entire string typed at console
//
//  name=gi.args(); // Arg-String
//
//=================
//
// Returns the number of individual components in args
// which are separated by SPACE
//
//  if (gi.argc() == 3) // Arg-Count
//
//=================
//
// Returns the argument at index in the args string
//
// ent->health=atoi(gi.argv(2)); // Arg-Value
//
//===========================================================

  int  (*argc)(void);   // returns the number of arguments
  char *(*argv)(int n);  // returns the argument number index
  char *(*args)(void);   // returns the next entry

//===========================================================
//
// Add commands to the console as if they were typed
//
// example:
//
//  gi.AddCommandString(command);
//
//===========================================================

  void (*AddCommandString)(char *text);

//===========================================================
//================ System Debugging Routines ================
//===========================================================
//
// Used only if 'set debuggraph 1'
//
//===========================================================

  void (*DebugGraph)(float value, int color);

} game_import_t;

extern game_import_t gi;

//===============================================
//===============================================
//   Game Engine Communication with DLL File.
//===============================================
//===============================================
// This is where you can modify the behavior of
// the Quake2.exe Gaming Engine..
//===============================================

typedef struct {

//===============================================
//====== Asking DLL for current Game Version ====
//===============================================
//
// example:
//
//  ge.apiversion=GAME_API_VERSION;
//
//===============================================

  int  apiversion;

//=============================================
//===== Game Engine initializing DLL ==========
//=============================================
//
// The init function will only be called when a game starts,
// not each time a level is loaded.  Persistant data for clients
// and the server can be allocated in init().
//
// examples:
//  ge.Init=InitGame;
//  ge.Shutdown=ShutdownGame;
//
//=============================================

  void (*Init)(void);
  void (*Shutdown)(void);

//=============================================
//========= Spawning Entity Routine ===========
//=============================================
//
// Each new level entered will cause a call to
// SpawnEntities by the Gaming Engine.
//
// examples:
//
//  ge.SpawnEntities=SpawnEntities;
//
//=============================================

  void (*SpawnEntities)(char *mapname, char *entstring, char *spawnpoint);

//=============================================
//========= System Read/Write Routines ========
//=============================================
//
// Read/Write Game is for storing persistant cross level information
// about the world state and the clients.
//
// WriteGame is called every time a level is exited.
// ReadGame is called on a loadgame.
//
// Examples:
//  ge.WriteGame=WriteGame;
//  ge.ReadGame=ReadGame;
//
// Read/Write Level is called after the default map information has been
// loaded with SpawnEntities
//
// examples:
//  ge.WriteLevel=WriteLevel;
//  ge.ReadLevel=ReadLevel;
//
//=============================================

  void (*WriteGame)(char *filename, qboolean autosave);
  void (*ReadGame)(char *filename);

  void (*WriteLevel)(char *filename);
  void (*ReadLevel)(char *filename);

//=============================================
//========== Gaming Engine Routines ===========
//=============================================
//
// Each time a client connects/disconnects from the
// server, the Gaming Engine will make the following
// calls to the DLL.
//
// examples:
//  ge.ClientThink=ClientThink;
//  ge.ClientConnect=ClientConnect;
//  ge.ClientUserinfoChanged=ClientUserinfoChanged;
//  ge.ClientDisconnect=ClientDisconnect;
//  ge.ClientBegin=ClientBegin;
//  ge.ClientCommand=ClientCommand;
//
//=============================================

  qboolean(*ClientConnect)(edict_t *ent, char *userinfo);
  void (*ClientBegin)(edict_t *ent);
  void (*ClientUserinfoChanged)(edict_t *ent, char *userinfo);
  void (*ClientDisconnect)(edict_t *ent);
  void (*ClientCommand)(edict_t *ent);
  void (*ClientThink)(edict_t *ent, usercmd_t *cmd);

//=============================================
//========= Game Animation Routine ============
//=============================================
//
// Gaming Engine uses this call to RUN THE
// GAME every frame_time(10 per second)...
//
// example:
//  ge.RunFrame=G_RunFrame;
//=============================================

  void (*RunFrame)(void);

//=============================================
//========= Server Command Routines ===========
//=============================================
//
// ServerCommand will be called when "sv <command>"
// command is typed on the console. The game can issue
// gi.argc() and gi.argv() commands to get the rest
// of the parameters.
//
// example:
//  ge.ServerCommand=ServerCommand;
//=============================================

  void (*ServerCommand)(void);

//=============================================
//========= Shared Global Variables ===========
//=============================================
//
// The Game Engine understands all the information
// about the current state of the World by storing
// the information in edict records(a linked list
// of records) which is allocated in the DLL.  The
// record definition can vary in size from one game
// to another so long as the Game Engine knows what
// the size of each edict_t record is.  The size will
// be fixed when ge->Init() is called.
//
// ge.edicts->edict_size  - size of a single edict record
// ge.edicts->num_edicts  - current num of linked edict recs
// ge.edicts->max_edicts  - maximum num of linked edict recs
//
//===============================================

  struct edict_s *edicts;
  int  edict_size;    // size of a single edict record
  int  num_edicts;    // current number of linked edict records
  int  max_edicts;    // maximum number of linked edict records

} game_export_t;

extern game_export_t ge;

