#include "defines.h"

//============================================================
edict_t *SV_TestEntityPosition(edict_t *ent) {
  if (gi.trace(ent->s.origin, ent->mins, ent->maxs, ent->s.origin, ent, (ent->clipmask)?ent->clipmask:MASK_SOLID).startsolid)
    return g_edicts;
  return NULL;
}

//============================================================
void SV_CheckVelocity(edict_t *ent) {
int i;
  for (i=0; i<3; i++)
    if (ent->velocity[i] > sv_maxvelocity->value)
      ent->velocity[i]=sv_maxvelocity->value;
    else if (ent->velocity[i] < -sv_maxvelocity->value)
      ent->velocity[i]= -sv_maxvelocity->value;
}

//============================================================
qboolean SV_RunThink(edict_t *ent) {
float thinktime;

  thinktime=ent->nextthink;
  if (thinktime<=0 || thinktime > level.time+0.001)
    return true;

  ent->nextthink=0;
  if (!ent->think)
    gi.error(ERR_FATAL,"NULL ent->think");

  ent->think(ent);

  return false;
}

//============================================================
void SV_Impact(edict_t *e1, trace_t *tr) {
edict_t *e2;

  e2=tr->ent;

  if (e1->touch && e1->solid != SOLID_NOT)
    e1->touch(e1, e2, &tr->plane, tr->surface);

  if (e2->touch && e2->solid != SOLID_NOT)
    e2->touch(e2, e1, NULL, NULL);
}

//============================================================
//============================================================
//============================================================
int ClipVelocity(vec3_t in, vec3_t normal, vec3_t out, float overbounce) {
float backoff, change;
int i, blocked=0;

  if (normal[2] > 0)
    blocked |= 1;    // floor
  if (!normal[2])
    blocked |= 2;    // step

  backoff=DotProduct(in, normal)*overbounce;

  for (i=0; i<3; i++) {
    change=normal[i]*backoff;
    out[i]=in[i]-change;
    if (out[i] > -0.1 && out[i] < 0.1)
      out[i]=0; }

  return blocked;
}

//============================================================
//============================================================

#define MAX_CLIP_PLANES  5

//============================================================
int SV_FlyMove(edict_t *ent, float time, int mask) {
edict_t *hit;
int i,j, blocked=0, bumpcount, numbumps=4, numplanes=0;
vec3_t end, dir, planes[MAX_CLIP_PLANES];
vec3_t primal_velocity, original_velocity, new_velocity;
trace_t tr;
float d,time_left;

  VectorCopy(ent->velocity, original_velocity);
  VectorCopy(ent->velocity, primal_velocity);

  time_left=time;

  ent->groundentity=NULL;
  for (bumpcount=0; bumpcount<numbumps; bumpcount++) {
    for (i=0; i<3; i++)
      end[i]=ent->s.origin[i]+time_left*ent->velocity[i];
    tr=gi.trace(ent->s.origin, ent->mins, ent->maxs, end, ent, mask);
    if (tr.allsolid) {
      // entity is trapped in another solid
      VectorCopy(zvec, ent->velocity);
      return 3; }
    if (tr.fraction > 0) {
      // actually covered some distance
      VectorCopy(tr.endpos, ent->s.origin);
      VectorCopy(ent->velocity, original_velocity);
      numplanes=0; }
    if (tr.fraction == 1.0)
       break;    // moved the entire distance
    hit=tr.ent;
    if (tr.plane.normal[2] > 0.7) {
      blocked |= 1;    // floor
      if (hit->solid == SOLID_BSP) {
        ent->groundentity=hit;
        ent->groundentity_linkcount=hit->linkcount; } }
    if (!tr.plane.normal[2])
      blocked |= 2;    // step

//
// run the impact function
//
    SV_Impact(ent, &tr);
    if (!ent->inuse)
      break;    // removed by the impact function

    time_left -= time_left*tr.fraction;

    // cliped to another plane
    if (numplanes >= MAX_CLIP_PLANES) {
      // this shouldn't really happen
      VectorCopy(zvec, ent->velocity);
      return 3; }

    VectorCopy(tr.plane.normal, planes[numplanes]);
    numplanes++;

//
// modify original_velocity so it parallels all of the clip planes
//
    for (i=0; i<numplanes; i++) {
      ClipVelocity(original_velocity, planes[i], new_velocity, 1);
      for (j=0; j<numplanes; j++)
        if ((j != i) && !VectorCompare(planes[i], planes[j]))
          if (DotProduct(new_velocity, planes[j]) < 0)
            break;  // not ok
      if (j == numplanes)
        break;
      } // end for

    if (i != numplanes)
      // go along this plane
      VectorCopy(new_velocity, ent->velocity);
    else {
      // go along the crease
      if (numplanes != 2) {
        VectorCopy(zvec, ent->velocity);
        return 7; }
      CrossProduct(planes[0], planes[1], dir);
      d=DotProduct(dir, ent->velocity);
      VectorScale(dir, d, ent->velocity); }

//
// if original velocity is against the original velocity, stop dead
// to avoid tiny occilations in sloping corners
//
    if (DotProduct(ent->velocity, primal_velocity)<=0) {
      VectorCopy(zvec, ent->velocity);
      return blocked; }
    } // endif

  return blocked;
}

//============================================================
void SV_AddGravity(edict_t *ent) {
  ent->velocity[2] -= ent->gravity*sv_gravity->value*FRAMETIME;
}

//============================================================
trace_t SV_PushEntity(edict_t *ent, vec3_t push) {
trace_t tr;
vec3_t start, end;

  VectorCopy(ent->s.origin, start);
  VectorAdd(start, push, end);

retry:
  tr=gi.trace(start, ent->mins, ent->maxs, end, ent, (ent->clipmask)?ent->clipmask:MASK_SOLID);
  VectorCopy(tr.endpos, ent->s.origin);
  gi.linkentity(ent);

  if (tr.fraction != 1.0) {
    SV_Impact(ent, &tr);
    // if the pushed entity went away and the pusher is still there
    if (!tr.ent->inuse && ent->inuse) {
      // move the pusher back and try again
      VectorCopy(start, ent->s.origin);
      gi.linkentity(ent);
      goto retry; } }

  if (ent->inuse)
    G_TouchTriggers(ent);

  return tr;
}

//============================================================
//============================================================

typedef struct {
  edict_t *ent;
  vec3_t origin;
  vec3_t angles;
  float deltayaw;
} pushed_t;
pushed_t  pushed[MAX_EDICTS], *pushed_p;

edict_t *obstacle;

//============================================================
qboolean SV_Push(edict_t *pusher, vec3_t move, vec3_t amove) {
int i, e;
edict_t *check, *block;
float temp;
pushed_t *p;
vec3_t mins, maxs, org, org2, move2, forward, right, up;

  // clamp the move to 1/8 units, so the position will
  // be accurate for client side prediction
  for (i=0; i<3; i++) {
    temp=move[i]*8.0;
    temp=(temp > 0.0)?temp+0.5:temp-0.5;
    move[i]=0.125*(int)temp; }

  // find the bounding box
  for (i=0; i<3; i++) {
    mins[i]=pusher->absmin[i]+move[i];
    maxs[i]=pusher->absmax[i]+move[i]; }

  // we need this for pushing things later
  VectorSubtract(zvec, amove, org);
  AngleVectors(org, forward, right, up);

  // save the pusher's original position
  pushed_p->ent=pusher;
  VectorCopy(pusher->s.origin, pushed_p->origin);
  VectorCopy(pusher->s.angles, pushed_p->angles);
  if (pusher->client)
    pushed_p->deltayaw=pusher->client->ps.pmove.delta_angles[YAW];
  pushed_p++;

  // move the pusher to it's final position
  VectorAdd(pusher->s.origin, move, pusher->s.origin);
  VectorAdd(pusher->s.angles, amove, pusher->s.angles);
  gi.linkentity(pusher);

  // see if any solid entities are inside the final position
  check=g_edicts+1;
  for (e=1; e < ge.num_edicts; e++, check++) {
    if (!check->inuse) continue;
    if (check->movetype == MOVETYPE_PUSH
     || check->movetype == MOVETYPE_STOP
     || check->movetype == MOVETYPE_NONE
     || check->movetype == MOVETYPE_NOCLIP) continue;
    if (!check->area.prev) continue;
    // if the entity is standing on the pusher, it will definitely be moved
    if (check->groundentity != pusher) {
      // see if the ent needs to be tested
      if (check->absmin[0] >= maxs[0]
      || check->absmin[1] >= maxs[1]
      || check->absmin[2] >= maxs[2]
      || check->absmax[0]<=mins[0]
      || check->absmax[1]<=mins[1]
      || check->absmax[2]<=mins[2])
        continue;
      // see if the ent's bbox is inside the pusher's final position
      if (!SV_TestEntityPosition(check))
        continue;  }

    if ((pusher->movetype == MOVETYPE_PUSH)
     || (check->groundentity == pusher)) {
      // move this entity
      pushed_p->ent=check;
      VectorCopy(check->s.origin, pushed_p->origin);
      VectorCopy(check->s.angles, pushed_p->angles);
      pushed_p++;

      // try moving the contacted entity
      VectorAdd(check->s.origin, move, check->s.origin);
      if (check->client)
        check->client->ps.pmove.delta_angles[YAW] += amove[YAW];

      // figure movement due to the pusher's amove
      VectorSubtract(check->s.origin, pusher->s.origin, org);
      org2[0]=DotProduct(org, forward);
      org2[1]=-DotProduct(org, right);
      org2[2]=DotProduct(org, up);
      VectorSubtract(org2, org, move2);
      VectorAdd(check->s.origin, move2, check->s.origin);

      // may have pushed them off an edge
      if (check->groundentity != pusher)
        check->groundentity=NULL;

      block=SV_TestEntityPosition(check);
      if (!block) {
        // pushed ok
        gi.linkentity(check);
        // impact?
        continue; }

      // if it is ok to leave in the old position, do it
      // this is only relevent for riding entities, not pushed
      VectorSubtract(check->s.origin, move, check->s.origin);
      block=SV_TestEntityPosition(check);
      if (!block) {
        pushed_p--;
        continue; }
      } // end for

    // save off the obstacle so we can call the block function
    obstacle=check;

    // move back any entities we already moved
    // go backwards, so if the same entity was pushed
    // twice, it goes back to the original position
    for (p=pushed_p-1; p>=pushed; p--) {
      VectorCopy(p->origin, p->ent->s.origin);
      VectorCopy(p->angles, p->ent->s.angles);
      if (p->ent->client)
        p->ent->client->ps.pmove.delta_angles[YAW]=p->deltayaw;
      gi.linkentity(p->ent); }
    return false; }

  // see if anything we moved has touched a trigger
  for (p=pushed_p-1; p>=pushed; p--)
    G_TouchTriggers(p->ent);

  return true;
}

//============================================================
void SV_Physics_Pusher(edict_t *ent) {
vec3_t move, amove;
edict_t *part, *mv;

  // if not a team captain, so movement will be handled elsewhere
  if (ent->flags & FL_TEAMSLAVE)
    return;

  // make sure all team slaves can move before commiting
  // any moves or calling any think functions
  // if the move is blocked, all moved objects will be backed out
  pushed_p=pushed;
  for (part=ent; part; part=part->teamchain) {
    if (part->velocity[0] || part->velocity[1] || part->velocity[2] ||
      part->avelocity[0] || part->avelocity[1] || part->avelocity[2]) {
      // object is moving
      VectorScale(part->velocity, FRAMETIME, move);
      VectorScale(part->avelocity, FRAMETIME, amove);
      if (!SV_Push(part, move, amove))
        break; } }  // move was blocked

  if (part) {
    // the move failed, bump all nextthink times and back out moves
    for (mv=ent; mv; mv=mv->teamchain)
      if (mv->nextthink > 0)
        mv->nextthink += FRAMETIME;
    // if the pusher has a "blocked" function, call it
    // otherwise, just stay in place until the obstacle is gone
    if (part->blocked)
      part->blocked(part, obstacle);
    } // endif
  else
    // the move succeeded, so call all think functions
    for (part=ent; part; part=part->teamchain)
      SV_RunThink(part);
}

//============================================================
void SV_Physics_None(edict_t *ent) {
  SV_RunThink(ent); // regular thinking
}

//============================================================
void SV_Physics_Noclip(edict_t *ent) {

  if (!SV_RunThink(ent)) return;

  VectorMA(ent->s.angles, FRAMETIME, ent->avelocity, ent->s.angles);
  VectorMA(ent->s.origin, FRAMETIME, ent->velocity, ent->s.origin);

  gi.linkentity(ent);
}

//============================================================
void SV_Physics_Toss(edict_t *ent) {
trace_t tr;
vec3_t move;
edict_t *slave;
qboolean wasinwater;
qboolean isinwater;
vec3_t old_origin;

  // regular thinking
  SV_RunThink(ent);

  // if not a team captain, so movement will be handled elsewhere
  if (ent->flags & FL_TEAMSLAVE)
    return;

  if (ent->velocity[2] > 0)
    ent->groundentity=NULL;

  // check for the groundentity going away
  if (ent->groundentity)
    if (!ent->groundentity->inuse)
      ent->groundentity=NULL;

  // if onground, return without moving
  if (ent->groundentity)
    return;

  VectorCopy(ent->s.origin, old_origin);

  SV_CheckVelocity(ent);

  // add gravity
  if (ent->movetype != MOVETYPE_FLY && ent->movetype != MOVETYPE_FLYMISSILE)
    SV_AddGravity(ent);

  // move angles
  VectorMA(ent->s.angles, FRAMETIME, ent->avelocity, ent->s.angles);

  // move origin
  VectorScale(ent->velocity, FRAMETIME, move);
  tr=SV_PushEntity(ent, move);

  if (!ent->inuse) return;

  if (tr.fraction != 1.0) {
     ClipVelocity(ent->velocity, tr.plane.normal, ent->velocity, (ent->movetype==MOVETYPE_BOUNCE)?1.5:1.0);
    // stop if on ground
    if (tr.plane.normal[2] > 0.7) {
      if (ent->velocity[2] < 60 || ent->movetype != MOVETYPE_BOUNCE) {
        ent->groundentity=tr.ent;
        ent->groundentity_linkcount=tr.ent->linkcount;
        VectorCopy(zvec, ent->velocity);
        VectorCopy(zvec, ent->avelocity); } } }

  // check for water transition
  wasinwater=(ent->watertype & MASK_WATER);
  ent->watertype=gi.pointcontents(ent->s.origin);
  isinwater=ent->watertype & MASK_WATER;
  ent->waterlevel=(isinwater)?1:0;

  if (!wasinwater && isinwater)
    gi.positioned_sound(old_origin, g_edicts, CHAN_AUTO, gi.soundindex("misc/h2ohit1.wav"), 1, 1, 0);
  else if (wasinwater && !isinwater)
    gi.positioned_sound(ent->s.origin, g_edicts, CHAN_AUTO, gi.soundindex("misc/h2ohit1.wav"), 1, 1, 0);

  // move teamslaves
  for (slave=ent->teamchain; slave; slave=slave->teamchain) {
    VectorCopy(ent->s.origin, slave->s.origin);
    gi.linkentity(slave); }

  // Reposition botpad's linked parts
  if (ent->pad!=NULL)
    Link_All_Together(ent);
}

//============================================================
//============================================================
//============================================================

#define sv_stopspeed    100
#define sv_friction       6
#define sv_waterfriction  1

//============================================================
void SV_AddRotationalFriction(edict_t *ent) {
int n;
float adjustment;

  VectorMA(ent->s.angles, FRAMETIME, ent->avelocity, ent->s.angles);
  adjustment=FRAMETIME*sv_stopspeed*sv_friction;
  for (n=0; n < 3; n++) {
    if (ent->avelocity[n] > 0) {
      ent->avelocity[n] -= adjustment;
      if (ent->avelocity[n] < 0)
        ent->avelocity[n]=0; }
    else {
      ent->avelocity[n] += adjustment;
      if (ent->avelocity[n] > 0)
        ent->avelocity[n]=0; } }
}

//============================================================
void SV_Physics_Step(edict_t *ent) {
qboolean wasonground, hitsound=false;
float *vel, speed, newspeed, control, friction;

  // airborn monsters should always check for ground
  if (!ent->groundentity)
    M_CheckGround(ent);

  SV_CheckVelocity(ent);

  wasonground=(ent->groundentity!=NULL);

  if (ent->avelocity[0] || ent->avelocity[1] || ent->avelocity[2])
    SV_AddRotationalFriction(ent);

  // add gravity except:
  //   flying monsters
  //   swimming monsters who are in the water
  if (!wasonground && !(ent->flags & FL_FLY))
    if (!((ent->flags & FL_SWIM) && (ent->waterlevel > 2))) {
      hitsound=(ent->velocity[2] < sv_gravity->value*-0.1);
      if (ent->waterlevel == 0)
        SV_AddGravity(ent); }

  // friction for flying monsters that have been given vertical velocity
  if ((ent->flags & FL_FLY) && (ent->velocity[2] != 0)) {
    speed=fabs(ent->velocity[2]);
    control=(speed < sv_stopspeed)?sv_stopspeed:speed;
    friction=sv_friction/3;
    newspeed=speed-(FRAMETIME*control*friction);
    newspeed=((newspeed<0)?0:newspeed)/speed;
    ent->velocity[2] *= newspeed; }

  // friction for flying monsters that have been given vertical velocity
  if ((ent->flags & FL_SWIM) && (ent->velocity[2] != 0)) {
    speed=fabs(ent->velocity[2]);
    control=(speed < sv_stopspeed)?sv_stopspeed:speed;
    newspeed=speed-(FRAMETIME*control*sv_waterfriction*ent->waterlevel);
    newspeed=((newspeed<0)?0:newspeed)/speed;
    ent->velocity[2] *= newspeed; }

  if (ent->velocity[2] || ent->velocity[1] || ent->velocity[0]) {
    // apply friction
    // let dead monsters who aren't completely onground slide
    if ((wasonground) || (ent->flags & (FL_SWIM|FL_FLY)))
      if (!(ent->health<=0.0 && !M_CheckBottom(ent))) {
        vel=ent->velocity;
        speed=sqrt(vel[0]*vel[0]+vel[1]*vel[1]);
        if (speed) {
          friction=sv_friction;
          control=(speed < sv_stopspeed)?sv_stopspeed:speed;
          newspeed=speed-FRAMETIME*control*friction;
          newspeed=((newspeed<0)?0:newspeed)/speed;
          vel[0] *= newspeed;
          vel[1] *= newspeed; } }

    SV_FlyMove(ent, FRAMETIME, (ent->svflags & SVF_MONSTER)?MASK_MONSTERSOLID:MASK_SOLID);
    gi.linkentity(ent);
    G_TouchTriggers(ent);
    if (!ent->inuse)
      return;

    if (ent->groundentity && !wasonground && hitsound)
      gi.sound(ent, CHAN_AUTO, gi.soundindex("world/land.wav"), 1, ATTN_NORM, 0);
    } // endif

  SV_RunThink(ent);
}

//======================================================
// New Move_Type BOUNCE.  When onground, do nothing.
//======================================================
void SV_Physics_Bounce(edict_t *ent) {
trace_t tr;
vec3_t move, planeVel, normalVel, temp;
edict_t *slave;
qboolean wasinwater;
qboolean isinwater;
vec3_t old_origin;
float movetime;

  SV_RunThink(ent); // regular thinking

  if (!ent->inuse) return;

  // if not a team captain, so movement will be handled elsewhere
  if (ent->flags & FL_TEAMSLAVE) return;

  if (ent->velocity[2] > 0)
    ent->groundentity = NULL;

  // check for the groundentity going away
  if (ent->groundentity && !ent->groundentity->inuse)
    ent->groundentity = NULL;

  // if onground, return without moving
  if (ent->groundentity) return;

  VectorCopy(ent->s.origin, old_origin);
  SV_CheckVelocity(ent);

  // move angles
  VectorMA(ent->s.angles, FRAMETIME, ent->avelocity, ent->s.angles);

  // add gravity
  ent->velocity[2] -= ent->gravity*sv_gravity->value * FRAMETIME;

  movetime = FRAMETIME;
  while (movetime > 0) {
    // move origin
    VectorScale(ent->velocity, movetime, move);
    // do the move (checking to see if we hit anything)
    tr = SV_PushEntity(ent, move);
    // SV_PushEntity might have called touch() functions so its possible we disappeared
    if (!ent->inuse) return;
    // we hit something so calculate rebound velocity etc.
    if (tr.fraction != 1.0) {
      // find component of velocity in direction of normal
      VectorScale(tr.plane.normal, DotProduct(tr.plane.normal, ent->velocity), normalVel);
      // find component of velocity parallel to plane of impact
      VectorSubtract(ent->velocity, normalVel, planeVel);
      // apply friction (lose 25% of velocity parallel to impact plane)
      VectorScale(planeVel, 0.75, planeVel);
      // calculate final velocity (rebound with 60% of impact velocity)
      // negative since rebound causes direction reversal
      VectorMA(planeVel, -0.6, normalVel, ent->velocity);
      // stop if on ground
      // only consider stopping if impact plane is less than 45 degrees with respect to horizontal
      // arccos(0.7) ~ 45 degrees
      if (tr.plane.normal[2] > 0.7) {
        VectorCopy(ent->velocity, temp);
        // zero out z component (vertical) since we want to consider horizontal and vertical motion separately
        temp[2] = 0;
        if ((VectorLength(temp) < 20)
         && (ent->velocity[2] < ent->gravity*sv_gravity->value*FRAMETIME*1.5)) {
          ent->groundentity = tr.ent;
          ent->groundentity_linkcount = tr.ent->linkcount;
          VectorCopy(zvec, ent->velocity);
          VectorCopy(zvec, ent->avelocity);
          break; // stopped, so no further simulation is necessary
          } // endif
        } // endif
      } // endif

    // remaining amount of FRAMETIME left to simulate
    movetime *= (1.0-tr.fraction);
    } // end while

  // check for water transition
  wasinwater = (ent->watertype & MASK_WATER);
  ent->watertype = gi.pointcontents(ent->s.origin);
  isinwater = (ent->watertype & MASK_WATER);
  ent->waterlevel=(isinwater)?1:0;

  if (!wasinwater && isinwater)
    gi.positioned_sound(old_origin, g_edicts, CHAN_AUTO, gi.soundindex("misc/h2ohit1.wav"), 1, 1, 0);
  else
  if (wasinwater && !isinwater)
    gi.positioned_sound(ent->s.origin, g_edicts, CHAN_AUTO, gi.soundindex("misc/h2ohit1.wav"), 1, 1, 0);

  // move teamslaves
  for (slave = ent->teamchain; slave; slave = slave->teamchain) {
    VectorCopy(ent->s.origin, slave->s.origin);
    gi.linkentity(slave); }

  // Reposition botpad's linked parts
  if (ent->pad!=NULL)
    Link_All_Together(ent);
}

//============================================================
void G_RunEntity(edict_t *ent) {

  if (ent->prethink)
    ent->prethink(ent);

  switch ((int)ent->movetype) {
    case MOVETYPE_PUSH:
    case MOVETYPE_STOP:   SV_Physics_Pusher(ent); break;
    case MOVETYPE_NONE:   SV_Physics_None(ent); break;
    case MOVETYPE_BOUNCE: SV_Physics_Bounce(ent); break;
    case MOVETYPE_NOCLIP: SV_Physics_Noclip(ent); break;
    case MOVETYPE_STEP:   SV_Physics_Step(ent); break;
    case MOVETYPE_TOSS:
    case MOVETYPE_FLY:
    case MOVETYPE_FLYMISSILE: SV_Physics_Toss(ent); break;
    default:
      gi.error(ERR_FATAL,"SV_Physics: bad movetype %i",(int)ent->movetype);
    } // end switch
}
