{*******************************************************}
{                     PHP4Delphi                        }
{               PHP - Delphi interface                  }
{                       v.5.0                           }
{ Author:                                               }
{ Serhiy Perevoznyk                                     }
{ serge_perevoznyk@hotmail.com                          }
{ http://users.chello.be/ws36637                        }
{*******************************************************}
{$I PHP.INC}
unit phpModules;

interface
 uses
   SyncObjs, Windows, SysUtils, Classes, Forms,
   {$IFDEF VERSION6}Variants,{$ENDIF} ZendAPI, phpAPI, phpFunctions;

type
  TOnModuleEvent = procedure(Sender : TObject; TSRMLS_DC : pointer) of object;


  TCustomPHPExtension = class(TDataModule)
  private
    FAbout : string;
    FTSRMLS : pointer;
    FOnModuleInfo : TOnModuleEvent;
    FOnModuleInit : TOnModuleEvent;
    FOnModuleShutdown : TOnModuleEvent;
    FOnRequestInit : TOnModuleEvent;
    FOnRequestShutdown : TOnModuleEvent;
    FOnActivation : TNotifyEvent;
    FOnDeactivation : TNotifyEvent;
    FModuleType : TZendModuleType;
    FVersion    : string;
    FModuleName : string;
    FFunctions  : TPHPFunctions;
    procedure SetFunctions(const Value : TPHPFunctions);
  protected
  public
    constructor Create(AOwner : TComponent); override;
    destructor  Destroy; override;
    procedure puts(str : PChar);
    procedure phpwrite(str : PChar; str_len : integer);
    procedure phpwrite_h(str : PChar; str_len : integer);
    procedure puts_h(str : PChar);
    procedure ReportError(ErrType : integer; ErrText : PChar);
    property About : string read FAbout write FAbout stored False;
    property ModuleType : TZendModuleType read FModuleType write FModuleType default mtPersistent;
    property Version    : string read FVersion write FVersion;
    property Functions  : TPHPFunctions read FFunctions write SetFunctions;
    property ModuleName : string read FModuleName write FModuleName;
    property TSRMLS : pointer read FTSRMLS;
    property OnActivation : TNotifyEvent read FOnActivation write FOnActivation;
    property OnDeactivation : TNotifyEvent read FOnDeactivation write FOnDeactivation;
    property OnModuleInit : TOnModuleEvent read FOnModuleInit write FOnModuleInit;
    property OnModuleShutdown : TOnModuleEvent read FOnModuleShutdown write FOnModuleShutdown;
    property OnRequestInit : TOnModuleEvent read FOnRequestInit write FOnRequestInit;
    property OnRequestShutdown : TOnModuleEvent read FOnRequestShutdown write FOnRequestShutdown;
    property OnModuleInfo : TOnModuleEvent read FOnModuleInfo write FOnModuleInfo;
  end;


  TPHPExtension = class(TCustomPHPExtension)
  public
    constructor Create(AOwner : TComponent); override;
  published
    property About;
    property ModuleType;
    property Version;
    property Functions;
    property ModuleName;
    property OnActivation;
    property OnDeactivation;
    property OnModuleInit;
    property OnModuleShutdown;
    property OnRequestInit;
    property OnRequestShutdown;
    property OnModuleInfo;
  end;

  TPHPApplication = class(TComponent)
  private
    FModuleInitFunction : TInitFunction;
    FModuleShutdownFunction : TInitFunction;
    FRequestInitFunction : TInitFunction;
    FRequestShutdownFunction : TInitFunction;
    FModuleInfoFunction : TZendModuleInfoFunction;
    FActiveFunctionName : PChar;
    FPHPExtensionClass: TComponentClass;
    FCriticalSection: TCriticalSection;
    FActivePHPModules: TList;
    FInactivePHPModules: TList;
    FTitle: string;
    FMaxConnections: Integer;
    FCacheConnections: Boolean;
    function GetActiveCount: Integer;
    function GetInactiveCount: Integer;
    procedure SetCacheConnections(Value: Boolean);
  protected
    function  ActivatePHPModule: TPHPExtension; dynamic;
    procedure DeactivatePHPModule(DataModule: TPHPExtension); dynamic;
    procedure DoHandleException(E: Exception); dynamic;
    procedure OnExceptionHandler(Sender: TObject; E: Exception);
    procedure HandleRequest(ht : integer; return_value : pzval; this_ptr : pzval;
      return_value_used : integer; TSRMLS_DC : pointer);
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure   Initialize; virtual;
    procedure   Run; virtual;
    procedure   CreateForm(InstanceClass: TComponentClass; var Reference);
    procedure   SetModuleInitFunction(AValue : TInitFunction);
    procedure   SetModuleShutdownFunction(AValue : TInitFunction);
    procedure   SetRequestInitFunction(AValue : TInitFunction);
    procedure   SetRequestShutdownFunction(AValue : TInitFunction);
    procedure   SetModuleInfoFunction(AValue : TZendModuleInfoFunction);
    property    Title: string read FTitle write FTitle;
    property    ActiveCount: Integer read GetActiveCount;
    property    CacheConnections: Boolean read FCacheConnections write SetCacheConnections;
    property    InactiveCount: Integer read GetInactiveCount;
    property    MaxConnections: Integer read FMaxConnections write FMaxConnections;
  end;


function get_module : Pzend_module_entry; cdecl;


var
  Application : TPHPApplication = nil;
  ModuleEntry : Tzend_module_entry;
  module_entry_table : array  of zend_function_entry;


implementation

uses
  Dialogs;

resourcestring
  SResNotFound = 'Resource %s not found';


procedure php_info_module(zend_module : Pzend_module_entry; TSRMLS_DC : pointer); cdecl;
var
  Extension : TPHPExtension;
begin
  if Assigned(Application) then
   begin
     Extension := Application.ActivatePHPModule;
     if Assigned(Extension.OnModuleInfo) then
      Extension.OnModuleInfo(Application, TSRMLS_DC);
     Application.DeactivatePHPModule(Extension);
   end;
end;

function minit (_type : integer; module_number : integer; TSRMLS_DC : pointer) : integer; cdecl;
var
  Extension : TPHPExtension;
begin
  result := SUCCESS;
  if Assigned(Application) then
   begin
     Extension := Application.ActivatePHPModule;
     if Assigned(Extension.OnModuleInit) then
      Extension.OnModuleInit(Application, TSRMLS_DC);
     Application.DeactivatePHPModule(Extension);
   end;
end;


function mshutdown (_type : integer; module_number : integer; TSRMLS_DC : pointer) : integer; cdecl;
var
  Extension : TPHPExtension;
begin
  result := SUCCESS;
  if Assigned(Application) then
   begin
     Extension := Application.ActivatePHPModule;
     if Assigned(Extension.OnModuleShutdown) then
      Extension.OnModuleShutdown(Application, TSRMLS_DC);
     Application.DeactivatePHPModule(Extension);
   end;
end;


function rinit (_type : integer; module_number : integer; TSRMLS_DC : pointer) : integer; cdecl;
var
  Extension : TPHPExtension;
begin
  Result := SUCCESS;
  if Assigned(Application) then
   begin
     Extension := Application.ActivatePHPModule;
     if Assigned(Extension.OnRequestInit) then
      Extension.OnRequestInit(Application, TSRMLS_DC);
     Application.DeactivatePHPModule(Extension);
   end;
end;

function rshutdown (_type : integer; module_number : integer; TSRMLS_DC : pointer) : integer; cdecl;
var
  Extension : TPHPExtension;
begin
  Result := SUCCESS;
  if Assigned(Application) then
   begin
     Extension := Application.ActivatePHPModule;
     if Assigned(Extension.OnRequestShutdown) then
      Extension.OnRequestShutdown(Application, TSRMLS_DC);
     Application.DeactivatePHPModule(Extension);
   end;
end;

procedure DispatchRequest(ht : integer; return_value : pzval; this_ptr : pzval;
      return_value_used : integer; TSRMLS_DC : pointer); cdecl;
begin
  ZVAL_NULL(return_value);
  if Assigned(Application) then
   try
     Application.HandleRequest(ht, return_value, this_ptr, return_value_used, TSRMLS_DC);
   except
   end;
end;



function get_module : Pzend_module_entry; cdecl;
var
  cnt : integer;
  Extension : TPHPExtension;
begin
  try
  if Assigned(Application) then
  begin
    Extension := TPHPExtension(Application.ActivatePHPModule);
    ModuleEntry.size := sizeof(Tzend_module_entry);
    ModuleEntry.zend_api := ZEND_MODULE_API_NO;
    ModuleEntry.zts := USING_ZTS;
    ModuleEntry.Name := PChar(Extension.ModuleName);
    ModuleEntry.version := PChar(Extension.Version);
    ModuleEntry.module_startup_func := @Application.FModuleInitFunction;
    ModuleEntry.module_shutdown_func := @Application.FModuleShutdownFunction;
    ModuleEntry.request_startup_func := @Application.FRequestInitFunction;
    ModuleEntry.request_shutdown_func := @Application.FRequestShutdownFunction;
    ModuleEntry.info_func := @Application.FModuleInfoFunction;
    SetLength(module_entry_table, Extension.FFunctions.Count + 1);
    for cnt := 0 to Extension.FFunctions.Count - 1 do
    begin
      module_entry_table[cnt].fname := PChar(Extension.FFunctions[cnt].FunctionName);
      module_entry_table[cnt].handler := @DispatchRequest;
      module_entry_table[cnt].func_arg_types := nil;
    end;
    module_entry_table[Extension.FFunctions.Count].fname := nil;
    module_entry_table[Extension.FFunctions.Count].handler := nil;
    module_entry_table[Extension.FFunctions.Count].func_arg_types := nil;

    ModuleEntry.functions :=  @module_entry_table[0];
    ModuleEntry._type := ORD(Extension.ModuleType) + 1;
    Application.DeactivatePHPModule(Extension);
    Result := @ModuleEntry;
  end
   else
     Result := nil;
   except
     Result := nil;
   end;
end;

{ TCustomPHPExtension }
constructor TCustomPHPExtension.Create(AOwner: TComponent);
begin
  inherited CreateNew(AOwner);
  FFunctions := TPHPFunctions.Create(Self, TPHPFunction);
  FModuleType := mtPersistent;
  FVersion := '0.0';
end;

destructor TCustomPHPExtension.Destroy;
begin
  if Assigned(OnDestroy) then
  try
    OnDestroy(Self);
  except
  end;
  FFunctions.Free;
  inherited;
end;



procedure TCustomPHPExtension.phpwrite(str: PChar; str_len: integer);
begin
  php_body_write(str, str_len, FTSRMLS);
end;

procedure TCustomPHPExtension.phpwrite_h(str: PChar; str_len: integer);
begin
  php_header_write(str, str_len, FTSRMLS);
end;

procedure TCustomPHPExtension.puts(str: PChar);
begin
  php_body_write(str, strlen(str), FTSRMLS);
end;

procedure TCustomPHPExtension.puts_h(str: PChar);
begin
  php_header_write(str, strlen(str), FTSRMLS);
end;

procedure TCustomPHPExtension.ReportError(ErrType: integer;
  ErrText: PChar);
begin
  zend_error(ErrType, ErrText);
end;

procedure TCustomPHPExtension.SetFunctions(const Value: TPHPFunctions);
begin
  FFunctions.Assign(Value);
end;

{ TPHPExtension }

constructor TPHPExtension.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  if (ClassType <> TCustomPHPExtension) and not (csDesigning in ComponentState) then
  begin
    if not InitInheritedComponent(Self, TCustomPHPExtension) then
      raise EResNotFound.CreateFmt(SResNotFound, [ClassName]);
    try
      if Assigned(OnCreate) and OldCreateOrder then OnCreate(Self);
    except
      Forms.Application.HandleException(Self);
    end;
  end;
end;

{ TPHPApplication }

procedure DoneVCLApplication;
begin
  with Forms.Application do
  begin
    if Handle <> 0 then ShowOwnedPopups(Handle, False);
    ShowHint := False;
    Destroying;
    DestroyComponents;
  end;
  with Application do
  begin
    Destroying;
    DestroyComponents;
  end;
end;

procedure DLLExitProc(Reason: Integer); register;
begin
  if Reason = DLL_PROCESS_DETACH then DoneVCLApplication;
end;

constructor TPHPApplication.Create(AOwner: TComponent);
begin
  inherited;
  FModuleInitFunction := minit;
  FModuleShutdownFunction := mshutdown;
  FRequestInitFunction := rinit;
  FRequestShutdownFunction := rshutdown;
  FModuleInfoFunction :=  php_info_module;
  FCriticalSection := TCriticalSection.Create;
  FActivePHPModules := TList.Create;
  FInactivePHPModules := TList.Create;
  FMaxConnections := 32;
  FCacheConnections := True;
  IsMultiThread := True;
  DLLProc := @DLLExitProc;
end;

destructor TPHPApplication.Destroy;
begin
  Forms.Application.OnException := nil;
  FCriticalSection.Free;
  FActivePHPModules.Free;
  FInactivePHPModules.Free;
  inherited Destroy;
end;


procedure TPHPApplication.CreateForm(InstanceClass: TComponentClass;
  var Reference);
begin
  if FPHPExtensionClass = nil then
    FPHPExtensionClass := InstanceClass
  else
   raise Exception.Create('Only one PHP extension allowed');
end;

function TPHPApplication.ActivatePHPModule: TPHPExtension;
begin
  FCriticalSection.Enter;
  try
    Result := nil;
    if (FMaxConnections > 0) and (FActivePHPModules.Count >= FMaxConnections) then
      raise Exception.Create('Too many active connections');
    if FInactivePHPModules.Count > 0 then
    begin
      Result := FInactivePHPModules[0];
      FInactivePHPModules.Delete(0);
      FActivePHPModules.Add(Result);
    end else if FPHPExtensionClass <> nil then
    begin
      TComponent(Result) := FPHPExtensionClass.Create(Self);
      FActivePHPModules.Add(Result);
      if Assigned(Result.FOnActivation) then
       Result.FOnActivation(Self);
    end else raise Exception.Create('No PHP extensions registered');
  finally
    FCriticalSection.Leave;
  end;
end;

procedure TPHPApplication.DeactivatePHPModule(DataModule: TPHPExtension);
begin
  FCriticalSection.Enter;
  try
    if Assigned(DataModule.FOnDeactivation) then
     DataModule.FOnDeactivation(Self);
    FActivePHPModules.Remove(DataModule);
    if FCacheConnections then
      FInactivePHPModules.Add(DataModule)
    else DataModule.Free;
  finally
    FCriticalSection.Leave;
  end;
end;

procedure TPHPApplication.DoHandleException(E: Exception);
begin

end;


function TPHPApplication.GetActiveCount: Integer;
begin
  FCriticalSection.Enter;
  try
    Result := FActivePHPModules.Count;
  finally
    FCriticalSection.Leave;
  end;
end;

function TPHPApplication.GetInactiveCount: Integer;
begin
  FCriticalSection.Enter;
  try
    Result := FInactivePHPModules.Count;
  finally
    FCriticalSection.Leave;
  end;
end;


procedure TPHPApplication.Initialize;
begin
  // This is a place holder
  if InitProc <> nil then TProcedure(InitProc);
end;

procedure TPHPApplication.OnExceptionHandler(Sender: TObject;
  E: Exception);
begin
  DoHandleException(E);
end;

procedure TPHPApplication.Run;
begin
  Forms.Application.OnException := OnExceptionHandler;
end;


procedure TPHPApplication.SetCacheConnections(Value: Boolean);
var
  I: Integer;
begin
  if Value <> FCacheConnections then
  begin
    FCacheConnections := Value;
    if not Value then
    begin
      FCriticalSection.Enter;
      try
        for I := 0 to FInactivePHPModules.Count - 1 do
          TPHPExtension(FInactivePHPModules[I]).Free;
        FInactivePHPModules.Clear;
      finally
        FCriticalSection.Leave;
      end;
    end;
  end;
end;



procedure TPHPApplication.HandleRequest(ht: integer;  return_value : pzval;
  this_ptr: pzval; return_value_used: integer;  TSRMLS_DC: pointer);
var
  DataModule: TPHPExtension;
  cnt : integer;
  Params : array of Pzval;
  AFunction : TPHPFunction;
  i : integer;
begin
 try
  if ( not (zend_get_parameters_ex(ht, @Params) = SUCCESS )) then
  begin
    zend_wrong_param_count(TSRMLS_DC);
    Exit;
  end;
  DataModule := TPHPExtension(ActivatePHPModule);
  if DataModule <> nil then
  try
    DataModule.FTSRMLS := TSRMLS_DC;
    FActiveFunctionName := get_active_function_name(TSRMLS_DC);
    for cnt := 0 to DataModule.FFunctions.Count - 1 do
      begin
        if SameText(DataModule.FFunctions[cnt].FunctionName, FActiveFunctionName) then
          begin
             AFunction := DataModule.FFunctions[cnt];
             if Assigned(AFunction.OnExecute) then
                begin
                  if AFunction.Parameters.Count <> ht then
                   begin
                     zend_wrong_param_count(TSRMLS_DC);
                     Exit;
                    end;
                  for i := 0 to ht - 1 do
                   begin
                     if not IsParamTypeCorrect(AFunction.Parameters[i].ParamType, Params[i]) then
                      begin
                        zend_error(E_WARNING, PChar(Format('Wrong parameter type for %s()', [get_active_function_name(TSRMLS_DC)])));
                        Exit;
                      end;
                     AFunction.Parameters[i].Value := zval2variant(Params[i]^);
                   end;
                  AFunction.OnExecute(DataModule, AFunction.Parameters, AFunction.ReturnValue, this_ptr, TSRMLS_DC);
                  variant2zval(AFunction.ReturnValue, return_value);
                end;
             break;
          end;
      end;
  finally
    DeactivatePHPModule(DataModule);
  end;
 except
 end;
end;

procedure TPHPApplication.SetModuleInitFunction(AValue: TInitFunction);
begin
  FModuleInitFunction := AValue;
end;

procedure TPHPApplication.SetModuleShutdownFunction(AValue: TInitFunction);
begin
  FModuleShutdownFunction := AValue;
end;

procedure TPHPApplication.SetRequestInitFunction(AValue: TInitFunction);
begin
  FRequestInitFunction := AValue;
end;

procedure TPHPApplication.SetRequestShutdownFunction(
  AValue: TInitFunction);
begin
  FRequestShutdownFunction := AValue;
end;

procedure TPHPApplication.SetModuleInfoFunction(
  AValue: TZendModuleInfoFunction);
begin
  FModuleInfoFunction := AValue;
end;



{ InitApplication }
procedure InitApplication;
begin
  Application := TPHPApplication.Create(nil);
end;

{ DoneApplication }
procedure DoneApplication;
begin
  try
    Application.CacheConnections := false;
    Application.Free;
    Application := nil;
  except
  end;
end;

exports
  get_module;

initialization
  InitApplication;
finalization
  DoneApplication;
end.