// ------------------------------------------------------------------------
// System       : MS-DOS
// Program      : sgrep.c
// Description  : a grep with list options (igrep revised)
//                companion for sir.exe
// Written by   : Bill Buckels
// Date Written : 1994
// Revision     : 1.1 First Release
//                2.0 Added Support For Long FileNames - Feb 2000
// ------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <io.h>
#include <string.h>
#include <dos.h>
#include <direct.h>
#include <malloc.h>

// not really stristr since we work on a copy...
char *stristr(char *buf, char *find)
{
  char scratch[512],
    str[128];
  
  strcpy(scratch, buf);
  strcpy(str, find);
  _strlwr(scratch);
  _strlwr(buf);
  return strstr(scratch, str);
}

typedef struct{
        char str[128];
    }PATSTRUCT;

PATSTRUCT *pstr;
unsigned patcnt=0;

// strip carriage returns from file lines
void nocr(char *buffer)
{
  int i;
  for(i = 0;buffer[i] != 0;i++)
  {
    if(buffer[i] == 10 || buffer[i] == 13)buffer[i] = 0;
  }
}

#define TRUE  1
#define FALSE 0

int NOCASE  = FALSE;
int QUOTE   = FALSE;
int COMMENT = FALSE;
int CCOMMENT= FALSE;
int BANNER  = TRUE;

// open fname... searches for all occurences of patterns in structure
int cgrep(char *fname)
{
  FILE *fp;
  long ctr = 1L;
  int found = 0,oldfound,i,j,quote = TRUE;
  char scratch[512], buffer[512], *wordptr;
  
  if((fp = fopen(fname, "r")) == NULL)
  {
    perror(fname);
    return 0;
  }
  
  while(fgets(buffer, 512, fp) != NULL)
  {
    nocr(buffer);
    oldfound = found;
    
    if(QUOTE == TRUE)
    {
      quote = FALSE;
      for(i = 0;buffer[i] != 0;i++)if(buffer[i] == '\'')quote = TRUE;
    }
    
    if(quote == TRUE)
    {
      // check for all occurrences of patterns
      for(i = 0;i < patcnt;i++)
      {
        strcpy(scratch, buffer);
        // check for clarion style comments
        if(COMMENT == TRUE)
        {
          for(j = 0;scratch[j] != 0;j++)
          {
            if(scratch[j] == '!')scratch[j] = 0;
          }
        }
        
        // ccomment is not perfect
        if(CCOMMENT == TRUE) 
        {
          for(j = 0;scratch[j] != 0;j++)
          {
            if(scratch[j] != ' ' && scratch[j] != '*')break;
            if(scratch[j] == '*')scratch[j] = 0;
            
          }
          for(j = 0;scratch[j] != 0;j++)
          {
            if(scratch[j] == '/' && scratch[j + 1] == '/')scratch[j] = 0;
            if(scratch[j] == '/' && scratch[j + 1] == '*')scratch[j] = 0;
            
          }
        }
        
        switch(NOCASE)
        {
          case TRUE:
          if((wordptr = stristr(scratch, (char *)&pstr[i].str[0])) != NULL)
          {
            if(found == 0)printf("%s\n", fname);
            printf("  %-10ld:", ctr);  // fixed leading blank
            puts(buffer);
            found++;
          }
          break;
          default  :
          if((wordptr = strstr(scratch, (char *)&pstr[i].str[0])) != NULL)
          {
            if(found == 0)printf("%s\n", fname);
            printf("  %-10ld:", ctr);  // fixed leading blank
            puts(buffer);
            found++;
          }
          
        }
        if(found != oldfound)i = patcnt;
      }
    }
    ctr++;
  }
  fclose(fp);
  return found;
}



int main(int argc, char **argv)
{
  FILE *patfile;
  char patbuf[128];
  char *fptr;
  int i;
  long findhandle;
  struct _finddata_t wild_card;
  
  char filename[_MAX_PATH],
    scratch[256],
    searchspec[_MAX_PATH];
  char *ptr,c;
  char path_buffer[_MAX_PATH],
    drive[_MAX_DRIVE],
    dir[_MAX_DIR],
    rootname[_MAX_FNAME],
    ext[_MAX_EXT];
  int found;
  
  if(argc < 2)
  {
    puts("SGREP(C) CopyLeft Bill Buckels 1994-2000. All Rights Reversed.");
    puts("Usage is \"SGREP [\"search spec\"] [filename(s)] -icq!s\"");
    puts("A List of Multiple Search Specs can also be Used :");
    puts("Usage is \"SGREP @[listname] [filename(s)] -i!s\"");
    puts("Switches are : i (ignore case)");
    puts("             : c (ignore c comments)");
    puts("             : q (no banner)");
    puts("             : ! (ignore clarion comments)");
    puts("             : s (match only lines with clarion strings)");
    return 1;
  }
  
  // parse the switch options
  if(argc == 4)
  {
    if(argv[3][0] == '/' || argv[3][0] == '-')
    {
      for(i = 1;argv[3][i] != 0;i++)
      {
        c = toupper(argv[3][i]);
        switch(c)
        {
          case 'I': NOCASE = TRUE;break;
          case '!': COMMENT = TRUE;break;
          case 'S': QUOTE = TRUE;break;
          case 'C': CCOMMENT = TRUE;break;
          case 'Q': BANNER = FALSE;break;
        }
      }
    }
  }
  
  if(BANNER == TRUE)
    puts("SGREP(C) CopyLeft Bill Buckels 1994-2000. All Rights Reversed.");
  
  // search all files if no filename...
  if(argc < 3)strcpy(filename, "*.*");
  else strcpy(filename, argv[2]);
  strcpy(path_buffer, filename);       // allow pathing...
  
  _splitpath(path_buffer, drive, dir, rootname, ext);
  
  // file list option
  if(argv[1][0] == '@')
  {
    fptr = (char *)&argv[1][1];
    if((patfile = fopen(fptr, "r")) == NULL)
    {
      sprintf(patbuf, "Pattern File : %s", fptr);
      perror(patbuf);
      return 1;
    }
    while(fgets(patbuf, 128, patfile) != NULL)patcnt++;
    if(patcnt == 0)
    {
      fprintf(stderr, "Pattern File : %s : Empty File.\n", fptr);
      fclose(patfile);
      return 1;
    }
    rewind(patfile);
    pstr = malloc(patcnt*sizeof(PATSTRUCT));
    for(i = 0;i < patcnt;i++)
    {
      fgets(patbuf, 128, patfile);
      nocr(patbuf);
      strcpy((char *)&pstr[i].str[0], patbuf);
    }
    fclose(patfile);
    sprintf(searchspec, "Pattern File %s\n", fptr);
  }
  // command line option
  else
  {
    // if a search spec is entered using quotes as delimiters
    // search for a string...
    // they can enclose a quote in quotes if they need to...
    if(argv[1][0] == '\"')
    {
      // allow strings enclosed with quotes...
      ptr = (char *)&argv[1][1];
      strcpy(scratch, ptr);
      found = strlen(scratch);         // trim the delimiters...
      if(found < 1 && scratch[found - 2] == '\"')scratch[found - 2] = 0;
      strcpy(searchspec, scratch);
    }
    else
    {
      strcpy(searchspec, argv[1]);     // also allow words only...
    }
    pstr = malloc(sizeof(PATSTRUCT));
    strcpy((char *)&pstr[0].str[0], searchspec);
    patcnt = 1;
  }
  
  // loop through all directory entries...
  if(BANNER == TRUE) 
  {
    printf("File spec   = %s\n", filename);
    printf("Search spec = ");
    puts(searchspec);
    puts("");
  }
  found = 0;
  
  if((findhandle = _findfirst(filename, &wild_card)) < 1l)
  {
    if(BANNER == TRUE)
      printf("%s : File not found... exiting.\n", filename);
    free(pstr);
    return 1;
  }
  if ( !(wild_card.attrib  &_A_SUBDIR)) {
    sprintf(path_buffer, "%s%s%s", drive, dir, wild_card.name);
    found += cgrep(path_buffer);
  }
  
  while(_findnext(findhandle, &wild_card) == 0)
  {
    if ( !(wild_card.attrib  &_A_SUBDIR)) {
      sprintf(path_buffer, "%s%s%s", drive, dir, wild_card.name);
      found += cgrep(path_buffer);
    }
  }
  
  _findclose(findhandle);
  free(pstr);
  if(!found)
  {
    if(BANNER == TRUE)
      puts("no files contain the searchspec...");
    return 1;
  }
  
  return 0;
}
