// ------------------------------------------------------------------------
// System       : WIN9X/NT
// Program      : bingrep.c
// Description  : simple grep for binary files
//                use upper case comparison only
// Written by   : Bill Buckels
// Date Written : 1995
// Revision     : 1.1 First Release
//                2.0 Added Support For Long FileNames - Feb 2000
// ------------------------------------------------------------------------

#include <stdlib.h>
#include <io.h>
#include <stdio.h>
#include <string.h>
#include <direct.h>
#include <malloc.h>
#include <dos.h>

int bingrep(unsigned char *fname, unsigned char *cmp, 
            unsigned char *src, int len, long cnt)
{
  FILE *fp;
  unsigned char buf[512];
  int c, found,i;
  
  if((fp = fopen(fname, "rb")) == NULL)
  {
    perror(fname);return 0;
  }
  // print a banner to say we haven't died
  // but we don't want this redirected...
  fprintf(stderr, "Reading %s...\n", fname);
  
  if(cnt < len)
  {
    fprintf(stderr, 
      "Filelength %ld is smaller than pattern length %d.\n", cnt, len);
    fclose(fp);
    return 0;
  }
  
  fread(buf, len, 1, fp);              // get the seed string
  for(i = 0;i < len;i++)
  {
    if(buf[i] > 96 && buf[i] < 123)buf[i] -= 32;
    // set to upper
  }                                    // terminate with 0
  buf[len] = 0;
  
  cnt = 0L;
  // set the count to the start of the string
  found = 0;                           // none found yet
  for(;;)
  {
    // try to do this with as little overhead as possible
    for(i = 0;i < len;i++)
    {
      if(src[i] != buf[i])break;
    }
    
    if(src[i] == buf[i])
    {
      if(found == 0)printf("%s\n", fname);
      found++;
      printf("     %0000006lX : %s\n", cnt, cmp);
    }
    
    if((c = fgetc(fp)) == EOF)break;
    if(c > 96 && c < 123)buf[len] = c - 32;
    // if alpha convert to upper
    else buf[len] = c;                 // and put at end of string
    cnt++;                             // count of lead byte
    for(i = 0;i < len;i++)buf[i] = buf[i + 1];
    // shift string left 1 byte
    buf[len] = 0;
    
    // allow an escape key press to exit every 100 bytes...
    // no point in checking every time
    if(cnt % 100)continue;
    if(!kbhit())continue;
    if((c = getch()) == 0)
    {
      getch();
    }
    else
    {
      if(c == 27)
      {
        fprintf(stderr, "Escape pressed... file search terminated.\n");
        break;
      }
    }
  }
  fclose(fp);
  
  return found;
}


int main(int argc, unsigned char **argv)
{
  int i,len;
  long findhandle;
  struct _finddata_t wild_card;
  
  unsigned char filename[_MAX_PATH],
    searchspec[128],
    src[128];
  
  unsigned char path_buffer[_MAX_PATH],
    drive[_MAX_DRIVE],
    dir[_MAX_DIR],
    rootname[_MAX_FNAME],
    ext[_MAX_EXT];
  int found;
  
  puts("BINGREP(C) CopyLeft Bill Buckels 1995-2000. All Rights Reversed.");
  
  if(argc < 2)
  {
    puts("Usage is : \"BINGREP [\"Search String\"] [FileName(s)]\"");
    puts("Purpose  : Upper Case Binary File String Match Search\n");
    return 1;
  }
  
  // search all files if no filename...
  if(argc < 3)strcpy(filename, "*.*");
  else strcpy(filename, argv[2]);
  strcpy(path_buffer, filename);       // allow pathing...
  
  _splitpath(path_buffer, drive, dir, rootname, ext);
  
  strcpy(searchspec, argv[1]);
  len = strlen(searchspec);
  if(len < 1)
  {
    printf("File spec   = %s\n", filename);
    printf("Search spec = ");puts(searchspec);
    puts("The search string is too short.");
    return 1;
  }
  if(searchspec[0] == '"' && searchspec[len - 1] == '"')
  {
    searchspec[len - 1] = 0;
    for(i = 0;i < len;i++)searchspec[i] = searchspec[i + 1];
    len = strlen(searchspec);
  }
  if(len < 1)
  {
    printf("File spec   = %s\n", filename);
    printf("Search spec = ");puts(searchspec);
    puts("The search string is too short.");
    return 1;
  }
  found = 0;
  printf("File spec   = %s\n", filename);
  printf("Search spec = ");
  puts(searchspec);
  
  if((findhandle = _findfirst(filename, &wild_card)) < 1l)
  {
    printf("%s : File not found... exiting.\n", filename);
    return 1;
  }
  
  // compare on upper case
  strcpy(src, searchspec);
  for(i = 0;src[i] != 0;i++)src[i] = toupper(searchspec[i]);
  
  if(!(wild_card.attrib  &_A_SUBDIR)) 
  {
    sprintf(path_buffer, "%s%s%s", drive, dir, wild_card.name);
    found += bingrep(path_buffer, searchspec, src, len, wild_card.size);
  }
  
  while(_findnext(findhandle, &wild_card) == 0)
  {
    if(!(wild_card.attrib  &_A_SUBDIR)) 
    {
      sprintf(path_buffer, "%s%s%s", drive, dir, wild_card.name);
      found += bingrep(path_buffer, searchspec, src, len, wild_card.size);
    }
  }

  _findclose(findhandle);
  
  if(!found)
  {
    puts("No files contain the searchspec...");
    return 1;
  }
  
  return 0;
}
