{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}



unit pFIBErrorHandler;

interface
{$I FIBPlus.inc}
 uses
  SysUtils,Classes,fib,pFIBDatabase, IB_ErrorCodes,
  ibase,IB_Intf, IB_Externals, DB;


 type

    TOptionErrorHandler=(oeException ,oeForeignKey,oeLostConnect,oeCheck);
    TKindIBError =(keNoError,keException ,keForeignKey,keLostConnect,
     keSecurity,keCheck, keOther
    );

    TOnFIBErrorEvent = procedure (Sender:TObject;ErrorValue:EFIBError;
                       KindIBError:TKindIBError;
                        var DoRaise:boolean
                       ) of object;

    TOptionsErrorHandler= set of TOptionErrorHandler;

    TpFibErrorHandler= class (TComponent)
    private
     FLastError:TKindIBError;
     FOnFIBErrorEvent:TOnFIBErrorEvent;
     FOptions:TOptionsErrorHandler;
     FExceptionNumber:integer;
     procedure DefaultOnError(Sender:TObject;ErrorValue:EFIBError;
                        var DoRaise:boolean);
    public
     constructor Create(AOwner:TComponent);override;
     destructor Destroy;override;
     procedure DoOnErrorEvent(Sender:TObject;ErrorValue:EFIBError;
                        var DoRaise:boolean); dynamic; // for internal use
     procedure DoOnLostConnect(ErrorValue:EFIBError);
     property ExceptionNumber:integer read FExceptionNumber;
     property LastError:TKindIBError read FLastError;
    published
     property OnFIBErrorEvent:TOnFIBErrorEvent read FOnFIBErrorEvent write FOnFIBErrorEvent;
     property Options:TOptionsErrorHandler read FOptions write FOptions
      default [oeException,oeLostConnect]
     ;
    end;

implementation

uses FIBConsts,StrUtil;

constructor TpFibErrorHandler.Create(AOwner:TComponent);//override;
begin
 if ErrorHandlerRegistered then
  raise Exception.Create(SFIBErrorHandlerExists);
 inherited Create(AOwner);
 RegisterErrorHandler(Self);
 Options:=[oeException,oeLostConnect];
 FLastError:=keNoError;
end;

destructor TpFibErrorHandler.Destroy;//override;
begin
 UnRegisterErrorHandler;
 inherited Destroy;
end;

function IsConnectionLost( const IBErrorCode: integer ): boolean;
begin
  Result:=(IBErrorCode = isc_shutdown )               or
          (IBErrorCode = isc_lost_db_connection )     or
          (IBErrorCode = isc_net_connect_err )        or
          (IBErrorCode = isc_net_connect_listen_err ) or
          (IBErrorCode = isc_net_event_connect_err )  or
          (IBErrorCode = isc_net_event_listen_err )   or
          (IBErrorCode = isc_net_read_err )           or
          (IBErrorCode = isc_net_write_err );
end;


procedure TpFibErrorHandler.DefaultOnError(Sender:TObject;
                        ErrorValue:EFIBError;
                        var DoRaise:boolean);
const EPrefix='EXCEPTION ';
var p:integer;
begin
  FExceptionNumber:=-1;
  FLastError:=keOther;
  with ErrorValue do
  if SQLCode=-836 then // Is Developer Exception
  begin
     FLastError:=keException;
     if oeException in Options then begin
      p:=Pos(EPrefix,UpperCase(Message));
      if p>0 then
       Message:=Copy(Message,p+10,MaxInt);
      p:=Pos(EPrefix,UpperCase(Message));
      if p>0 then
       Message:=Copy(Message,p+10,MaxInt);
      p:=Pos('.',Message);
      if p>0 then
      try
       FExceptionNumber:=StrToInt(Copy(Message,1,p-1));
       Message:=TrimCLRF(Copy(Message,p+1,MaxInt));
      except
      end;
     end;
  end   // end Developer Exception
  else
  if IsConnectionLost(IBErrorCode)  or
  ((SQLCode = -902) and (IBErrorCode = isc_network_error))
  then begin
   FLastError:=keLostConnect;
   if oeLostConnect in Options then
      DoOnLostConnect(ErrorValue);
  end 
  else
  if SQLCode=-551 then FLastError:=keSecurity
  else
  if SQLCode=-297 then begin
   FLastError:=keCheck;
   if oeCheck in Options then
   begin
    p:=Pos(#10'.',ErrorValue.Message);
    if p>0 then
     ErrorValue.Message:=TrimCLRF(Copy(ErrorValue.Message,p+2,MaxInt));
   end
  end;
end;

type THack=class(TpFIBDatabase);


procedure TpFibErrorHandler.DoOnLostConnect(ErrorValue:EFIBError);
var i:integer;
    Actions:TOnLostConnectActions;
begin
 for i:=0 to Pred(pDataBaseCount) do
 with THack(pDataBase(i)) do begin
  if not Connected then Continue;
  Actions:=laCloseConnect;
  DoOnLostConnect(pDataBase(i),ErrorValue,Actions);
 end;
end;

procedure TpFibErrorHandler.DoOnErrorEvent(
           Sender:TObject;ErrorValue:EFIBError;
              var DoRaise:boolean
           ); //dynamic;
begin
 DefaultOnError(Sender,ErrorValue,DoRaise);
 if Assigned(FOnFIBErrorEvent) then
  FOnFIBErrorEvent(Sender,ErrorValue,LastError,DoRaise);
end;



end.

