{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}

(*
 * StdFuncs -
 *   A file chock full of functions that should exist in Delphi, but
 *   dont, like "Max", "GetTempFile", "Soundex", etc...
 *)
unit StdFuncs;

{$i FIBPlus.inc}

interface

uses
 Classes, SysUtils,Db,
 {$IFDEF MSWINDOWS}
   Windows
  {$IFDEF D6+},FMTBcd, Variants {$ENDIF};
 {$ENDIF}
 {$IFDEF LINUX}
   Types,FMTBcd, Variants;
 {$ENDIF}

type
  EParserError = class(Exception);
  TCharSet = set of Char;

function ConvertFromBase(sNum: String; iBase: Integer; cDigits: String): Integer;
function ConvertToBase(iNum, iBase: Integer; cDigits: String): String;




function Max(n1, n2: Integer): Integer;
function MaxD(n1, n2: Double): Double;
function Min(n1, n2: Integer): Integer;
function MinD(n1, n2: Double): Double;
function Signum(Arg:Integer) :Integer;

function RandomString(iLength: Integer): String;
function RandomInteger(iLow, iHigh: Integer): Integer;
function Soundex(st: String): String;
function StripString(const st: String; CharsToStrip: String): String;
function ClosestWeekday(const d: TDateTime): TDateTime;
function Year(d: TDateTime): Integer;
function Month(d: TDateTime): Integer;
function DayOfYear(d: TDateTime): Integer;
function DayOfMonth(d: TDateTime): Integer;
function VarCoalesce(V1, V2: Variant): Variant;
function VarEqual(V1, V2: Variant): Boolean;
procedure WeekOfYear(d: TDateTime; var Year, Week: Integer);
function  Degree10(Degree:integer):Extended;
function  ExtPrecision(Value:Extended) :integer;
function  RoundExtend(Value: Extended;Decimals:integer): Extended;

// Comp type stuff

function  CompToStr(Value: comp): string;
function  ExtendedToStr(Value: Extended;Scale:integer): string;
function  FormatExtended(const Format: string; Value: Extended;Scale:integer): string;

function  CompWithScaleToStr(Value: comp;Scale:integer;DSep:Char): string;

{$IFDEF D5+}
overload;
function  CompWithScaleToStr(Value: comp;Scale:integer): string;overload;
{$ENDIF}
type PComp=^Comp;
//end Comp type stuff

{$IFDEF D5+}
function  CompToBCD(Value: Comp;Scale:integer; var BCD: TBcd ): Boolean;
function  BCDToExtended(BCD: TBcd; var Value: Extended): Boolean;
function  BCDToCompWithScale(BCD: TBcd; var Value: Comp;var Scale:byte): Boolean;
function  BCDToStr(BCD: TBcd): String;
function  BCDToSQLStr(BCD: TBcd): String;
function  CompareBCD(const BCD1,BCD2: TBcd): integer;
{$ENDIF}
function  StrToComp(const Value: string): Comp;
function  CompDiv(A, B: comp): Comp;
function  CompMod(A, B: comp): Comp;



function  TimeStamp(const aDate,aTime:integer):TTimeStamp;
function  CmpFullName(cmp:TComponent):string;
procedure FullClearStrings(aStrings:TStrings);

function HookTimeStampToMSecs(const TimeStamp:TTimeStamp): Comp;
function HookTimeStampToDateTime(const TimeStamp: TTimeStamp): TDateTime;

//DB rtns


function FieldOldValAsString(Field:TField;SQLFormat:boolean):string;
{$IFDEF D5+}
function BCDFieldAsSQLString(Field:TField;OldVal:boolean):variant;
function BCDFieldAsString(Field:TField;OldVal:boolean):variant;
function GetBCDFieldData(Field:TField;OldVal:boolean; var BCD:TBcd):boolean;
{$ENDIF}
{$IFNDEF D5+}
procedure FreeAndNil(var Obj);
{$ENDIF}

procedure InitFPU;


var
  TempPath: PChar;
  TempPathLength: Integer;

implementation

uses FIBConsts,StrUtil;

type THackDS=class(TDataSet);


function FieldOldValAsString(Field:TField;SQLFormat:boolean):string;
var
    OldState:TDataSetState;
begin
  OldState:=Field.DataSet.State;
  try
    THackDS(Field.DataSet).SetTempState(dsOldValue);
    Result:=Field.AsString;
  finally
    THackDS(Field.DataSet).RestoreState(OldState);
  end;
  if SQLFormat and (DecimalSeparator<>'.') then
   ReplaceStr(Result,DecimalSeparator,'.')
end;

{$IFDEF D5+}
function GetBCDFieldData(Field:TField;OldVal:boolean; var BCD:TBcd):boolean;
var
    OldState:TDataSetState;
begin
  OldState:=Field.DataSet.State;
  with THackDS(Field.DataSet) do
  try
    if OldVal and (OldState<>dsOldValue) then
     SetTempState(dsOldValue);
    Result:=GetFieldData(Field,@Bcd);
  finally
    if OldVal and (OldState<>dsOldValue) then
     RestoreState(OldState);
  end;
end;


function InternalBCDFieldAsString(Field:TField;OldVal,SQLFormat:boolean):variant;
var
    Bcd :TBcd;
begin
  if GetBCDFieldData(Field,OldVal,Bcd) then
  begin
     if SQLFormat then
      Result:=BCDToSQLStr(BCD)
     else
      Result:=BCDToStr(BCD);
  end
  else
  begin
     Result:=UnAssigned;
  end;
end;

function BCDFieldAsSQLString(Field:TField;OldVal:boolean):variant;
begin
 Result:= InternalBCDFieldAsString(Field,OldVal,true)
end;

function BCDFieldAsString(Field:TField;OldVal:boolean):variant;
begin
 Result:= InternalBCDFieldAsString(Field,OldVal,false)
end;
{$ENDIF}
//
function RoundExtend(Value: Extended;Decimals:integer): Extended;
var
    st:double;
begin
 st    :=Degree10(Decimals);
 Result:=System.Int(Value)+Round(Frac(Value)*st)/st;
end;

function TimeStamp(const aDate,aTime:integer):TTimeStamp;
begin
  with Result do  begin
    Time:=aTime;
    Date:=aDate;
  end;
end;

function HookTimeStampToMSecs(const TimeStamp:TTimeStamp): Comp;
var t:TTimeStamp;
begin
  if TimeStamp.Date=0 then begin
   t.Date:=1;
   t.Time:=TimeStamp.Time;
   Result:=TimeStampToMSecs(t)-86400000;
  end
  else
   Result:=TimeStampToMSecs(TimeStamp);
end;

function HookTimeStampToDateTime(const TimeStamp: TTimeStamp): TDateTime;
var t:TTimeStamp;
begin
 if TimeStamp.Date=0 then begin
  t.Date:=1;
  t.Time:=TimeStamp.Time;
  Result:=TimeStampToDateTime(t)-1;
 end
 else
  Result:=TimeStampToDateTime(TimeStamp)
end;
 
function CmpFullName(cmp:TComponent):string;
begin
 result:='';
 while cmp<>nil do
 with cmp do begin
  if Name<>'' then
  if Result='' then
   result:=Name
  else
   result:=Name+'.'+Result;
  cmp:=Owner;
 end;
end;

procedure FullClearStrings(aStrings:TStrings);
var
  j: Integer;
begin
 with aStrings do
 for j := 0 to Pred(aStrings.Count) do
 begin
   if Objects[j]<>nil then Objects[j].Free;
 end;
  aStrings.Clear;
end;

const
    i10:array [0..18] of comp =
     (1, 1E1, 1E2, 1E3, 1E4, 1E5, 1E6, 1E7, 1E8,1E9, 1E10,
       1E11, 1E12, 1E13, 1E14, 1E15, 1E16, 1E17, 1E18
      );

function Degree10(Degree:integer):Extended;
begin
 if Degree>=0 then
  Result:=i10[Degree]
 else
  Result:=1/i10[-Degree]
end;

function  ExtPrecision(Value:Extended) :integer;
var
  L, H, I: Integer;
  c:comp;
  a:comp;
begin
  L := 0;
  H := 18;
  a := Abs(Int(Value));
  while L <= H do
  begin
    I := (L + H) shr 1;
    C := i10[I]-a;
    if C < 0 then L := I + 1 else
    begin
      H := I - 1;
      if C = 0 then
      begin
        L := I+1;
        Break
      end;
    end;
  end;
  Result:=L
end;

// Comp type stuff

{$IFDEF D5+}
function  CompToBCD(Value: Comp;Scale:integer; var BCD: TBcd ): Boolean;
var c:Currency;
begin
  c:=Value/1E4;
  Result:= CurrToBCD(c,BCD);
  with BCD do
  begin
   if Value<0 then
    SignSpecialPlaces:=128
   else
    SignSpecialPlaces:=0;
   SignSpecialPlaces :=Scale+SignSpecialPlaces;
  end;
end;

function  BCDToCompWithScale(BCD: TBcd; var Value: Comp;var Scale:byte): Boolean;
var Sign:integer;
    c:Currency;
begin
 with BCD do
 begin
  if SignSpecialPlaces>=128 then
  begin
   Sign:=-1;
   Scale:=SignSpecialPlaces-128;
  end
  else
  begin
   Sign :=1 ;
   Scale:=SignSpecialPlaces;
  end;
  if Scale>=64 then
  begin
  // null
   Result:=true;
   Value:=0;
   Exit;
  end;
  SignSpecialPlaces:=4;
 end;
 Result:=BCDToCurr(BCD,C);
 Value :=Sign*C*Degree10(4)
end;

function  BCDToStr(BCD: TBcd): String;
var  c:Comp;
     Scale:byte;
begin
 if BCDToCompWithScale(BCD,c,Scale) then
  Result:=CompWithScaleToStr(c,Scale)
 else
  Result:=''
end;

function  BCDToSQLStr(BCD: TBcd): String;
var  c:Comp;
     Scale:byte;
begin
 if BCDToCompWithScale(BCD,c,Scale) then
  Result:=CompWithScaleToStr(c,Scale,'.')
 else
  Result:=''
end;

function  BCDToExtended(BCD: TBcd; var Value: Extended): Boolean;
var c:Comp;
    Scale:byte;
begin
 Result:= BCDToCompWithScale(BCD,c,Scale);
 if Result then
  Value :=C*Degree10(-Scale)
end;

function  CompareBCD(const BCD1,BCD2: TBcd): integer;
var e1,e2:extended;
begin
 BCDToExtended(Bcd1,e1);
 BCDToExtended(Bcd2,e2);
 if e1=e2 then
  Result:=0
 else
 if e1<e2 then
  Result:=-1
 else
  Result:=1
end;

{$ENDIF}

function StrToComp(const Value: string): comp;
var e:integer;
begin
 Val(Value, Result, E);
 if e<>0  Then raise EConvertError.CreateFmt(SFIBErrorInvalidComp, [Value]);
end;

const
    ZeroStr='000000000000000000';

function  FormatExtended(const Format: string; Value: Extended;Scale:integer): string;
var FmtE :string;
    FmtInt,FmtFrac:string;
    FracStr:string;
    IntE,FracE:comp;
    p,p1,PrFrac:integer;
    NoMinus:boolean;
begin
  if Value=0 then
  begin
   Result:=FormatFloat(Format,Value);
   Exit;
  end;
  NoMinus:=False;
  p:=Pos(';',Format);
  if p=0 then
   FmtE:=Format
  else
   if Value>0 then
    DoCopy(Format,FmtE,1,p-1)
   else
   begin
    p1:=PosInSubstr(';',Format,p+1,MaxInt);
    if p1=0 then p1:=MaxInt;
    DoCopy(Format,FmtE,p+1,p1-p-1);
    if TrimLeft(FmtE)='' then  FmtE:=Copy(Format,1,p-1) else NoMinus:=True;
   end;

  if Pos('E',FmtE)>0 then
  begin
   Result :=FormatFloat(Format,Value);
   Exit;
  end;
  
  IntE  :=Int(Value);
  if (1-Abs(Frac(Value)))<Degree10(-Scale-1) then
  begin
   FracE :=0;
   if IntE<0 then IntE:=IntE-1 else IntE:=IntE+1;
  end
  else
  FracE :=Abs(Frac(Value)*Degree10(Scale));

  p:=Pos('.',FmtE);
  if p>0 then
  begin
    FmtInt :=Copy(FmtE,1,p-1);
    FmtFrac:=Copy(FmtE,p+1,MaxInt);
    FracStr:=CompToStr(FracE);
    PrFrac :=Length(FracStr);
    if Scale>PrFrac then
     FracStr:=Copy(ZeroStr,1,Scale-PrFrac)+FracStr;

    p:=Length(FmtFrac);

    if Length(FracStr)>p then
    begin
     Insert(DecimalSeparator,FracStr,Length(FmtFrac)+1);
     FracStr:=IntToStr(Round(StrToFloat(FracStr)));
      if p>Length(FracStr) then
      begin
       FracStr:=Copy(ZeroStr,1,p-Length(FracStr))+FracStr
      end;
    end
    else
    begin
     while FmtFrac[p]='#' do Dec(p);
      if p>Length(FracStr) then
      begin
       FracStr:=FracStr+Copy(ZeroStr,1,p-Length(FracStr))
      end;
    end;
    p:=Length(FracStr);
    if (p>0) and (FmtFrac[p]='#') then
     begin
        while (FracStr[p]='0') and (FmtFrac[p]='#') do  Dec(p);
        Delete(FracStr,p+1,MaxInt);
     end;
    if (p>0) or (Pos('0',FracStr)>0) then
     Result :=FormatFloat(FmtInt,IntE)+DecimalSeparator+ FracStr
    else
     Result  :=FormatFloat(FmtInt,IntE);
  end
  else
  begin
    Result:=FormatFloat(FmtE,IntE);
  end;
  if NoMinus and (Value<0) then     Delete(Result ,1,1);
end;

function  ExtendedToStr(Value: Extended;Scale:integer): string;
var
    IntE,FracE:comp;
    FracStr :string;
    p       :integer;
begin
 Value :=RoundExtend(Value,Scale);
 IntE  :=Int(Value);
 FracE :=Abs(Frac(Value)*Degree10(Scale));
 if FracE<>0 then begin
   FracStr:=CompToStr(FracE);
   p     :=Length(FracStr);
   if Scale>p then
    FracStr:=Copy(ZeroStr,1,Scale-p)+FracStr;
   p:=Length(FracStr);
   while FracStr[p]='0' do Dec(p);
   Delete(FracStr,p+1,Length(FracStr));
   Result:=CompToStr(IntE)+DecimalSeparator+FracStr
 end
 else
  Result:=CompToStr(IntE)
end;


{$IFDEF D5+}
function  CompWithScaleToStr(Value: comp;Scale:integer): string;
begin
 Result:=CompWithScaleToStr(Value,Scale,DecimalSeparator)
end;
{$ENDIF}

function  CompWithScaleToStr(Value: comp;Scale:integer;DSep:Char): string;
var i,j:integer;
    IntStr,DecStr:string;
    sign:string[1];
begin
 if Value=0 then begin
  Result:='0';
  Exit;
 end;
 Result:= CompToStr(Value);
 if Scale>0 then
 begin
   if Result[1]='-' then
   begin
    sign:='-';
    Delete(Result,1,1);
   end
   else
    sign:='';
  j:=Length(Result)-Scale ;
  if j>0 then
  begin
    IntStr:=sign+ Copy(Result,1,j);
    DecStr:=Copy(Result,j+1,Length(Result));
  end
  else
  begin
   IntStr:=sign+'0';
   DecStr:=MakeStr('0',-j)+Result;
  end;
  Result:= IntStr+DSep+DecStr;
  i:=Length(Result);
  while Result[i] = '0' do
  begin
    Dec(i);
    if Result[i]=DSep then
    begin
     Dec(i);Break;
    end;
  end;
   Delete(Result,i+1,Length(Result));
 end;
end;

{$IFDEF MSWINDOWS}
procedure InitFPU;
var
  Default8087CW: Word;
begin
  asm
    FSTCW Default8087CW
    OR Default8087CW, 0300h
    FLDCW Default8087CW
  end;
end;

{$ELSE}
procedure InitFPU;
begin

end;
{$ENDIF}
function CompToStr(Value: comp): string;
var
  A: Extended;
  I: Integer;
  NonZero: boolean;
begin
  InitFPU;
  if ((Value<1E18) and (Value>-1E18)) then begin
   FmtStr(Result,'%.f', [Value]);
   Exit
  end
{$IFDEF D5+}
  else
  if (Value+9223372036854775807)=-1 then
  begin
   Result :='-9223372036854775808';
   Exit
  end
{$ENDIF}  
  ;
  Result := '';
  if Value < 0 then
  begin
    Result := Result + '-';
    Value  := -Value
  end;
  NonZero := false;
  for I := 18 downto 0 do
  begin
    A := Value/i10[I];
    if NonZero or (Trunc(A) > 0) then
    begin
      Result := Result + Chr(48 + Trunc(A));
      NonZero := true;
    end;
    Value := Value - Trunc(A) * i10[I];
  end;
end;

function CompDiv(A, B: comp): comp;
{$IFNDEF INT64_SUPPORT}
var
    C: comp;
{$ENDIF}
begin
  {$IFDEF INT64_SUPPORT}
    Result:= PInt64(@A)^ div PInt64(@B)^
  {$ELSE}
    if A < 0 then A := -A;
    Result := A/B;
    C := Result * B -A;
    if C > 0 then
      Result := Result - 1;
  {$ENDIF}    
end;

function CompMod(A, B: comp): comp;
{$IFNDEF INT64_SUPPORT}
var
    C: comp;
{$ENDIF}
begin
  {$IFDEF INT64_SUPPORT}
    Result:= PInt64(@A)^ mod PInt64(@B)^
  {$ELSE}
    if A < 0 then A := -A;
    Result := A/B;
    C := Result * B - A ;
    if C > 0 then
      Result := Result - 1;
    C := Result * B;
    Result := A - C;
  {$ENDIF}
end;
//
function ConvertFromBase(sNum: String; iBase: Integer; cDigits: String): Integer;
var
  i: Integer;

  function GetValue(c: Char): Integer;
  var
    i: Integer;
  begin
    result := 0;
    for i := 1 to Length(cDigits) do
      if (cDigits[i] = c) then begin
        result := i - 1;
        exit;
      end;
  end;

begin
  result := 0;
  for i := 1 to Length(sNum) do
    result := (result * iBase) + GetValue(sNum[i]);
end;

function ConvertToBase(iNum, iBase: Integer; cDigits: String): String;
var
  i, r: Integer;
  s: String;
const
  iLength = 16;
begin
  result := '';
  SetString(s, nil, iLength);
  i := 0;
  repeat
    r := iNum mod iBase;
    Inc(i);
    if (i > iLength) then
      SetString(s, PChar(s), Length(s) + iLength);
    s[i] := cDigits[r + 1];
    iNum := iNum div iBase;
  until iNum = 0;
  SetString(result, nil, i);
  for r := 1 to i do
    result[r] := s[i - r + 1];
end;



function Max(n1, n2: Integer): Integer;
begin
  if (n1 > n2) then
    result := n1
  else
    result := n2;
end;

function MaxD(n1, n2: Double): Double;
begin
  if (n1 > n2) then
    result := n1
  else
    result := n2;
end;

function Min(n1, n2: Integer): Integer;
begin
  if (n1 < n2) then
    result := n1
  else
    result := n2;
end;

function Signum(Arg:Integer) :integer;
begin
 if Arg>0 then Result:=1 else
  if Arg<0 then Result:=-1  else  Result:=0;
end;

function MinD(n1, n2: Double): Double;
begin
  if (n1 < n2) then
    result := n1
  else
    result := n2;
end;

function RandomString(iLength: Integer): String;
begin
  result := '';
  while Length(result) < iLength do
    result := result + IntToStr(RandomInteger(0, High(Integer)));
  if Length(result) > iLength then
    result := Copy(result, 1, iLength);
end;

function RandomInteger(iLow, iHigh: Integer): Integer;
begin
  result := Trunc(Random(iHigh - iLow)) + iLow;
end;

function Soundex(st: String): String;
var
  code: Char;
  i, j, len: Integer;
begin
  result := ' 0000';
  if (st = '') then exit;
  result[1] := UpCase(st[1]);
  j := 2;                   
  i := 2;
  len := Length(st);
  while (i <= len) and (j < 6) do begin
    case st[i] of
      'B', 'F', 'P', 'V', 'b', 'f', 'p', 'v' : code := '1';
      'C', 'G', 'J', 'K', 'Q', 'S', 'X', 'Z',
      'c', 'g', 'j', 'k', 'q', 's', 'x', 'z' : code := '2';
      'D', 'T', 'd', 't' :                     code := '3';
      'L', 'l' :                               code := '4';
      'M', 'N', 'm', 'n' :                     code := '5';
      'R', 'r' :                               code := '6';
    else
      code := '0';
    end; {case}

    if (code <> '0') and (code <> result[j - 1]) then begin
      result[j] := code;
      inc(j);
    end;
    inc(i);
  end;
end;

function StripString(const st: String; CharsToStrip: String): String;
var
  i: Integer;
begin
  result := '';
  for i := 1 to Length(st) do begin
    if Pos(st[i], CharsToStrip) = 0 then
      result := result + st[i];
  end;
end;

function ClosestWeekday(const d: TDateTime): TDateTime;
begin
  if (DayOfWeek(d) = 1) then
    result := d + 1
  else if (DayOfWeek(d) = 7) then
    result := d + 2
  else
    result := d;
end;

function Year(d: TDateTime): Integer;
var
  y, m, day: Word;
begin
  DecodeDate(d, y, m, day);
  result := y;
end;

function Month(d: TDateTime): Integer;
var
  yr, mn, dy: Word;
begin
  DecodeDate(d, yr, mn, dy);
  result := mn;
end;

function DayOfYear(d: TDateTime): Integer;
var
  yr, mn, dy: Word;
  b: TDateTime;
begin
  DecodeDate(d, yr, mn, dy);
  b := EncodeDate(yr, 1, 1);
  result := Trunc(d - b);
end;

function DayOfMonth(d: TDateTime): Integer;
var
  yr, mn, dy: Word;
begin
  DecodeDate(d, yr, mn, dy);
  result := dy;
end;

function VarCoalesce(V1, V2: Variant): Variant;
begin
  if (VarIsNull(V1) or VarIsEmpty(V1)) then
    result := V2
  else
    result := V1;
end;

function VarEqual(V1, V2: Variant): Boolean;
begin
  try
    result := V1 = V2;
  except
    result := (VarIsNull(V1) and VarIsNull(V2)) or
              (VarIsEmpty(V1) and VarIsEmpty(V2));
  end;
end;

procedure WeekOfYear(d: TDateTime; var Year, Week: Integer);
var
  yr, mn, dy: Word;
  dow_ybeg: Integer;
  ThisLeapYear, LastLeapYear: Boolean;
begin
  DecodeDate(d, yr, mn, dy);
  // When did the year begin?
  Year := yr;
  dow_ybeg := SysUtils.DayOfWeek(EncodeDate(yr, 1, 1));
  ThisLeapYear := IsLeapYear(yr);
  LastLeapYear := IsLeapYear(yr - 1);
  // Get the Sunday beginning this week.
  Week := (DayOfYear(d) - DayOfWeek(d) + 1);
  (*
   * If the Sunday beginning this week was last year, then
   *   if this year begins on a Wednesday or previous, then
   *     this is most certainly the first week of the year.
   *   if this year begins on a thursday or
   *     last year was a leap year and this year begins on a friday, then
   *     this week is 53 of last year.
   *   Otherwise this week is 52 of last year.
   *)
  if Week <= 0 then begin
    if (dow_ybeg <= 4) then
      Week := 1
    else if (dow_ybeg = 5) or (LastLeapYear and (dow_ybeg = 6)) then begin
      Week := 53;
      Dec(Year);
    end else begin
      Week := 52;
      Dec(Year);
    end;
  (* If the Sunday beginning this week falls in this year!!! Yeah
   *   if the year begins on a Sun, Mon, Tue or Wed then
   *     This week # is (Week + 7) div 7
   *   otherwise this week is
   *     Week div 7 + 1.
   *   if the week is > 52 then
   *     if this year began on a wed or this year is leap year and it
   *       began on a tuesday, then set the week to 53.
   *     otherwise set the week to 1 of *next* year.
   *)
  end else begin
    if (dow_ybeg <= 4) then
      Week := (Week + 6 + dow_ybeg) div 7
    else
      Week := (Week div 7) + 1;
    if Week > 52 then begin
      if (dow_ybeg = 4) or (ThisLeapYear and (dow_ybeg = 3)) then
        Week := 53
      else begin
        Week := 1;
        Inc(Year);
      end;
    end;
  end;
end;

{$IFNDEF D5+}
procedure FreeAndNil(var Obj);
var
  P: TObject;
begin
  P := TObject(Obj);
  TObject(Obj) := nil;  // clear the reference before destroying the object
  P.Free;
end;
{$ENDIF}

initialization
  Randomize;

finalization



end.



